#!/usr/bin/perl -wT
#
# $Id: hostinstall.pl,v 1.28.2.1 2005/01/17 20:10:29 ms152511 Exp $
# hostinstall.pl - configure a host for management or remove host-side management components
#
# Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#

use strict;
use Fcntl qw(:flock);
use File::Spec;
use File::Temp;
use Getopt::Std;
use IO::File;
use IO::Handle;
use IO::Select;
use POSIX qw(WEXITSTATUS);

#########
# Globals

# command line options (see main and print_usage)
my %opts;

###########
# Constants

my $VERSION           = '2.2-29';

# Control Station locations
# modified by updateIP.sh during installation or manual invocation
my @CSHostAddrs       = ( '__CONTROLSTATION__' );
my $CSHostInstallPath = 'pub/hostinstall.pl';
my $CSKeyFilePath     = 'pub/csid_rsa.pub';

# allow a minute to contact and update NIS servers
my $UserAdd_Timeout   = 60;

# user manipulation commands
my $UserAddCmd = '/usr/sbin/useradd';
my $UserDelCmd = '/usr/sbin/userdel';
my $ChownCmd   = '/bin/chown';
my $UNameCmd   = '/bin/uname';
my $RmCmd      = '/bin/rm';

my @Cmds       = ( $UserAddCmd, $UserDelCmd,
                   $ChownCmd, $UNameCmd, $RmCmd );

# password manipulation
my $ShadowFile = "/etc/shadow";


# home directory creation
my $CSUserName = 'csadmin';
my $CSHomeDir  = "/var/$CSUserName";
my $SSHDir     = "$CSHomeDir/.ssh";

# package manipulation command sets
my @PkgCmdSets = ( # RPM (SuSE, JDS, Redhat)
                   {
                     'rpm'      => '/bin/rpm'
                   },

                   # SVR4 pkgs (Solaris)
                   {
                     'pkgadd'   => '/usr/sbin/pkgadd',
                     'pkgparam' => '/usr/bin/pkgparam',
                     'pkgrm'    => '/usr/sbin/pkgrm'
                   }
                 );

# After check_commands complete, this holds a hash reference
# mapping command names to absolute pathnames.  (Copied from
# the first row of PkgCmdSets where all the commands are
# accessible.)
my $PkgCmdSet;

# RPM manipulation
my $AgentRPMName = 'base-mgmt-agent';
my $HostRPMName  = 'base-mgmt-client';
my $HostRPMPath  = "pub/pkgs/$HostRPMName-latest.i386.rpm";

# pkg manipulation

my $HostPkgName  = 'SUNWbmc';
# use a hash instead of just interpolating the architecture name into
# the package filename so we can detect an unsupported architecture;
# the hash entry will not exist
my %HostPkgPaths = ( 'sparc' => "pub/pkgs/$HostPkgName-latest.sparc.pkg.Z",
                     'i386'  => "pub/pkgs/$HostPkgName-latest.i386.pkg.Z" );

# admin file for pkgadd
# Defines overrides to allow installation of SUID binaries and
# execution of the package install script.
my $ADMIN_DOC = << 'EOD';
basedir=default
mail=
runlevel=quit
conflict=nocheck
setuid=nocheck
action=nocheck
partial=quit
instance=quit
idepend=quit
rdepend=quit
space=quit
EOD

# get_url
my @WGetCmds      = ( '/usr/bin/wget', '/usr/sfw/bin/wget' );
my @CurlCmds      = ( '/usr/bin/curl' );
my $GetURLTimeout = 60;

# management directory locations
my $MgmtBaseDir          = '/usr/mgmt';
my $PkgMgrCmd            = "$MgmtBaseDir/libexec/pkgMgr.pl";
my $HostinstallLocalPath = "$MgmtBaseDir/sbin/hostinstall.pl";

# regexps for taint clearing
my $HostnameRE   = '(?: [[:alnum:]]    # First character is alpha-numeric. (RFC 1123)
                        [[:alnum:].-]* # Remaining characters are drawn from the
                                       # alphabet (A-Z), digits (0-9), minus sign (-),
                                       # and period (.) (RCF 952)
                        [[:alnum:]]    # The last character must not be a minus sign
                                       # or period.  (RFC 952)
                    )'; # this RE also accepts IP addresses

#
# CLI Routines
#
# print_usage
# print_info
#

#
# print_usage
#
sub print_usage ()
{
    no strict;

    print { ($opts{'h'} or $opts{'?'}) ? STDOUT : STDERR } << "EOD";
hostinstall.pl - prepare a host for management by a Control Station

Usage:
    hostinstall.pl [ OPTIONS ] < MODE >

Modes:
    install
        Creates the csadmin user if not already present and
        installs support files.

    remove
        Removes the csadmin user and prevents further remote
        administration by the Control Station.

Options:
    -?
    -h  help

        Displays this usage information.

    -c  Control Station addresses

        Takes a comma-separated list of IP addresses or hostnames
        at which to attempt to contact the Control Station.
        hostinstall.pl first attempts to connect to the Control
        Station at these addresses before falling back (if necessary)
        to the addresses listed below.

    -q  quiet mode

        Suppresses all output except for critical errors.

    -y  \"yes\" mode

        Do not ask for confirmation.

This script is customized for the Control Station accessible at
@CSHostAddrs.

EOD
}

#
# print_info
#
sub print_info ($)
{
    print STDERR $_[0] unless ($opts{'q'});
}           
                
#
# Administrative User (csadmin) Setup Subroutines
#
# create_user_home
# setup_ssh_key
#

#
# create_user_home
#
# Creates the csadmin user and corresponding home directory.
# Sets the account to suppress last login messages.
#
sub create_user_home () {
    my $rc;

    # create csadmin user if it doesn't already exist
    print_info "Creating csadmin user.\n";

    my $useradd_cmdline;
    if ( get_os() eq 'Linux' ) {
        $useradd_cmdline = "$UserAddCmd -s/bin/sh -m -d$CSHomeDir -r $CSUserName 2> /dev/null";
    } else {
        $useradd_cmdline = "$UserAddCmd -s/bin/sh -m -d$CSHomeDir    $CSUserName 2> /dev/null";
    }

    my $useradd_pipe = new IO::Handle;
    # -r adds as a system account, to prevent display in lists of
    # regular users
    my $useradd_pid  = open( $useradd_pipe,
                             "$useradd_cmdline |" )
        or die "Could not open pipe for $UserAddCmd: $!";
    my $selector     = new IO::Select($useradd_pipe);

    # loop using select to be able to detect a hung useradd operation
 LOOP_USERADD_SELECT:
    while (1) {
        my @ready_fh = $selector->can_read($UserAdd_Timeout);
        if ( scalar(@ready_fh) != 1 || $ready_fh[0] != $useradd_pipe ) {
            # timed out
            kill("TERM", $useradd_pid);
            close($useradd_pipe);
            print STDERR << 'EOD';

ERROR: Could not create the csadmin user

Timed out while waiting for useradd.
This may be caused by improper NIS server configuration or other directory services.

Please correct the problem and try again.

EOD
            exit(1);
        }
        
        my $buffer;
        my $Buffer_Size = 1024;
        my $chars_read = $useradd_pipe->sysread( $buffer, $Buffer_Size );
        
        if ( $chars_read == 0 ) {
            # got EOF
            if (!close($useradd_pipe) && $!) {
                die "Failed to close pipe to useradd: $!";
            }
            # grab return code
            $rc = $?;
            last;
        }
    }
    
    die "useradd failed" if ( $rc != 0 && 
                              # useradd returns 9 if the user already exists
                              WEXITSTATUS($rc) != 9 );

    # set csadmin password to a random string to disallow interactive
    # login while still allowing RSA keypair authentication over SSH
    print_info "Randomizing csadmin login password.\n";
    my $new_passwd = random_string(8);
    change_passwd( $CSUserName, crypt_string(random_string(8)) );
    
    # create directory if necessary
    print_info "Creating ssh directory.\n";
    if ( not -d $SSHDir ) {
        $rc = mkdir( $SSHDir, '0700' );
        die "mkdir failed" if ( !$rc );
    }
    
    # suppress login messages
    print_info "Setting csadmin login to suppress login messages.\n";
    my $hushlogin_fh = new IO::File;
    $hushlogin_fh->open(">> $CSHomeDir/.hushlogin");
    die "Could not create .hushlogin file" if ( not defined $hushlogin_fh );
    $hushlogin_fh->close;
}

#
# setup_ssh_key
#
# Fetches the control station's public host key and adds it
# to the authorized_keys database.
#
sub setup_ssh_key ()
{
    # fetch public SSH key and verify URL
    print_info "Attempting to fetch control station public key.\n";
    my ($rc, $cs_public_key) = get_path_from_cs($CSKeyFilePath);

    die "Couldn't fetch the control station public SSH key" if ( $rc < 0 );
    
    # check if key is already in authorized_keys
    print_info "Checking for public key in $SSHDir/authorized_keys... ";
    my $AuthorizedKeysFH = new IO::File;

    my $key_was_present = 0;
    if ( -r "$SSHDir/authorized_keys" ) {
        $AuthorizedKeysFH->open("< $SSHDir/authorized_keys")
            or die "Could not open $SSHDir/authorized_keys for reading";

        while ( my $existing_key = $AuthorizedKeysFH->getline ) {
            if ( $existing_key eq $cs_public_key ) {
                $key_was_present = 1;
                last;
            }
        }

        $AuthorizedKeysFH->close;
    }

    # add key if it was not already present
    if ( $key_was_present ) {
        print_info "already present.\n";
    } else {
        print_info "not found.\n";
        print_info "Adding control station public key to $SSHDir/authorized_keys.\n";
        $AuthorizedKeysFH->open(">> $SSHDir/authorized_keys")
            or die "Could not open $SSHDir/authorized_keys for writing";
        print $AuthorizedKeysFH $cs_public_key;
        $AuthorizedKeysFH->close;
    }
}

#
# version checking subroutines
#
# check_rpm_file_version
# check_pkg_file_version
# compare_versions
#

#
# check_rpm_file_version
#
# Takes a pathname to an RPM package file.
# Returns the package version and release as a string.
#
sub check_rpm_file_version ($)
{
    my ( $rpm_file_path )  = @_;

    my $rpm_path = $PkgCmdSet->{'rpm'};

    # check that we have an rpm file to check
    die "check_rpm_file_version() requires an rpm pathname to check"
        if ( not defined $rpm_file_path );

    # check that rpm is executable
    die "$rpm_path is not an absolute path to an executable copy of the rpm program"
        if ( ! -x $rpm_path or
             ! File::Spec->file_name_is_absolute($rpm_path) or
             ( File::Spec->splitpath($rpm_path) )[2] ne 'rpm' );

    # protect arguments from the shell
    my $escaped_rpm_file_path = $rpm_file_path;
    sanitize_cli_args(\$escaped_rpm_file_path);
    
    my $out = `$rpm_path --queryformat '%{Version}-%{Release}' -qp $escaped_rpm_file_path 2>&1`;

    if ( $? == 0 ) {
        # installed, verify format and clear taint
        my ($version) = ( $out =~ /^((?:\d+\.)*\d+-\d+)$/ );
        return $version;
    } else {
        die "Call to rpm failed: $out";
    }
}


#
# check_pkg_file_version
#
# Takes a pathname to an uncompressed Solaris pkg, and the
# corresponding package identifies.  Returns the package
# version string.
#
sub check_pkg_file_version ($$)
{
    my ( $pkg_path, $pkg_name ) = @_;
    
    my $pkgparam_path = $PkgCmdSet->{'pkgparam'};

    # check that pkgparam is executable
    die "$pkgparam_path is not an absolute path to an executable copy of the pkgparam program"
        if ( ! -x $pkgparam_path or
             ! File::Spec->file_name_is_absolute($pkgparam_path) or
             ( File::Spec->splitpath($pkgparam_path) )[2] ne 'pkgparam' );

    # protect arguments from the shell
    sanitize_cli_args( \$pkg_path, \$pkg_name );

    my $command = "pkgparam -d $pkg_path $pkg_name VERSION 2>&1";
    my $output  = `$command`;
    chomp $output;

    die "Call to '$command' failed" if ( $? != 0 );

    return $output;
}

#
# compare_versions
#
# Flexible compare two version strings.  Breaks each string into
# integer components, separated by any number of non-numeric
# characters.  Each pair of components is compared numerically.
# The component that appears first is considered most significant.
#
# If the two version strings have an unequal number of components,
# and all of the leading components are equivalent, the longer
# version string is taken to be greater.
#
# Returns -1 if the first version string represents a smaller
# version number than the second, and 1 if the opposite is true.
# Returns 0 if the two version strings have equivalent components.
#
# Examples:
#
# compare_versions('1.2-3',  '2.4') = -1
# compare_versions( '10-3',  '7.4') =  1
# compare_versions( 'v1.2',  '6_3') = -1
# compare_versions(   '66', '66.2') = -1
#
sub compare_versions ($$)
{
    my ($a, $b) = @_;

    my @a_components = split(/\D+/, $a);
    my @b_components = split(/\D+/, $b);

    while ( scalar(@a_components) > 0 &&
            scalar(@b_components) > 0 ) {
        my $a_comp = shift @a_components;
        my $b_comp = shift @b_components;

        if ($a_comp < $b_comp) {
            return -1;
        } elsif ($a_comp > $b_comp) {
            return 1;
        }
    }

    # Ran out of components.  Check if one if longer.
    if ( scalar(@a_components) > 0 ) {
        return 1;
    } elsif ( scalar(@b_components) > 0 ) {
        return -1;
    }
    
    return 0;
}

#
# host package manipulation subroutines
#
# check_rpm
# check_pkg
# check_host_package
# 
# install_host_rpm
# install_host_pkg
# install_host_package
#
# remove_rpm
# remove_pkg
# remove_host_package
#

#
# check_rpm
#
# Takes a rpm name, and returns its version string if the rpm is
# installed.  Returns the undefined value if the package is not
# installed on the system.
#
sub check_rpm ($)
{
    my $rpm_path   = $PkgCmdSet->{'rpm'};
    my ($rpm_name) = @_;

    # check that we have an rpm name to check
    die "check_rpm() requires an rpm name to check"
        if ( not defined $rpm_name );

    # check that rpm is executable
    die "$rpm_path is not an absolute path to an executable copy of the rpm program"
        if ( ! -x $rpm_path or
             ! File::Spec->file_name_is_absolute($rpm_path) or
             ( File::Spec->splitpath($rpm_path) )[2] ne 'rpm' );    

    # protect arguments from the shell
    my $escaped_rpm_name = $rpm_name;
    sanitize_cli_args(\$escaped_rpm_name);
    
    my $out = `$rpm_path --queryformat '%{Version}-%{Release}' -q $escaped_rpm_name 2>&1`;

    if ( $? == 0 ) {
        # installed, verify format and clear taint
        my ($version) = ( $out =~ /^((?:\d+\.)*\d+-\d+)$/ );
        return $version;
    } else {
        # not installed or error
        if ( $out =~ /^package $rpm_name is not installed$/ ) {
            return undef;
        } else {
            die "Call to rpm failed";
        }
    }
}

#
# check_pkg
#
# Takes a package identifier, and returns its version string if the
# package is installed.  Returns the undefined value if the package
# is not installed on the system.
#
sub check_pkg ($)
{
    my ($pkg_name) = @_;

    my $pkgparam_path = $PkgCmdSet->{'pkgparam'};

    # check that pkgparam is executable
    die "$pkgparam_path is not an absolute path to an executable copy of the pkgparam program"
        if ( ! -x $pkgparam_path or
             ! File::Spec->file_name_is_absolute($pkgparam_path) or
             ( File::Spec->splitpath($pkgparam_path) )[2] ne 'pkgparam' );

    # protect arguments from the shell
    my $escaped_pkg_name = $pkg_name;
    sanitize_cli_args(\$escaped_pkg_name);

    my $command = "pkgparam $escaped_pkg_name VERSION 2>&1";
    my $output  = `$command`;
    chomp $output;
    
    if ( $? == 0 ) {
        return $output;
    } elsif ( $output =~ /^pkgparam: ERROR: unable to locate parameter information for "$pkg_name"$/ ) {
        return undef;
    } else {
        die "Call to '$command' failed" ;
    }
}

#
# check_host_package
#
sub check_host_package ()
{
    CASE_PKGMGR : {
        if ( exists $PkgCmdSet->{'rpm'} ) {
            return check_rpm($HostRPMName);
            
            last CASE_PKGMGR;
        }
        
        if ( exists $PkgCmdSet->{'pkgparam'} ) {
            return check_pkg($HostPkgName);
            
            last CASE_PKGMGR;
        }
    }
}

#
# install_host_rpm
#
sub install_host_rpm ()
{
    my $rpm_path = $PkgCmdSet->{'rpm'};
    my $rc;
    my $purge_when_finished = 0;

    # check that rpm is executable
    die "$rpm_path is not an absolute path to an executable copy of the rpm program"
        if ( ! -x $rpm_path or
             ! File::Spec->file_name_is_absolute($rpm_path) or
             ( File::Spec->splitpath($rpm_path) )[2] ne 'rpm' );
    
    # fetch the package
    print_info "Fetching host RPM from control station.\n";
    my $host_rpm_doc;
    ( $rc, $host_rpm_doc ) = get_path_from_cs($HostRPMPath);

    die "Couldn't fetch the host RPM"
        if ( $rc < 0 );

    # create temporary filehandle to hold package
    my ( $temp_fh, $temp_filename ) = File::Temp::tempfile( 'hostrpmXXXXXX',
                                                            'DIR'    => File::Spec->tmpdir(),
                                                            'UNLINK' => 1 );
    die "Couldn't create a temporary file for the host RPM: $!"
        if ( not defined $temp_fh );

    print $temp_fh $host_rpm_doc;
    $temp_fh->flush;
    
    # check if the old agent package is already installed
    print_info "Checking if legacy agent is installed... ";
    my $agent_rpm_version = check_rpm($AgentRPMName);
    if ( defined $agent_rpm_version ) {
        print_info "installed (version $agent_rpm_version).\n";
        if (confirm_package_removal()) {
            remove_rpm($AgentRPMName);
            $purge_when_finished = 1;
        } else {
            die "Must remove out-of-date packages to proceed";
        }
    } else {
        print_info "not installed.\n";
    }

    # check if the client package is already installed
    print_info "Checking if $HostRPMName is already installed... ";
    my $host_rpm_version = check_rpm($HostRPMName);
    if ( defined $host_rpm_version ) {

        my $rpm_file_version = check_rpm_file_version( $temp_filename );
        if ( compare_versions($rpm_file_version, $host_rpm_version) > 0 ) {
            print_info "older version ($host_rpm_version).\n";
            if (confirm_package_removal()) {
                remove_rpm($HostRPMName);
                $purge_when_finished = 1;
            } else {
                die "Must remove out-of-date packages to proceed";
            }
        } else {
            print_info "up-to-date version ($host_rpm_version).\n";
            close($temp_fh) or die "Couldn't close temporary file: $!";
            return;
        }
        
    } else {

        print_info "not installed.\n";

    }

    print_info "Installing host RPM.\n";
    $rc = system( $rpm_path, '-i', $temp_filename );
    die 'Call to rpm failed.'
        if ( $rc != 0 );
    
    close($temp_fh) or die "Couldn't close temporary file: $!";

    purge_module_packages() if ($purge_when_finished);
}

#
# install_host_pkg
#
# Copies the Solaris SVR4 pkg file from the control station
# and installs it using pkgadd.
#
sub install_host_pkg ()
{
    my $pkgadd_path  = $PkgCmdSet->{'pkgadd'};
    my $rc;
    my $purge_when_finished = 0;

    # check that commands are executable
    die "$pkgadd_path is not an absolute path to an executable copy of the pkgadd program."
        if ( ! -x $pkgadd_path or
             ! File::Spec->file_name_is_absolute($pkgadd_path) or
             ( File::Spec->splitpath($pkgadd_path) )[2] ne 'pkgadd' );

    # check host architecture
    print_info "Checking processor architecture... ";
    my $architecture = get_architecture();
    chomp $architecture;
    print_info "$architecture\n";
    
    my $host_pkg_path;
    if ( not exists $HostPkgPaths{$architecture} ) {
        die "Unknown architecture '$architecture'";
    } else {
        $host_pkg_path = $HostPkgPaths{$architecture};
    }
    

    # fetch the package
    print_info "Fetching the host pkg from the control station.\n";
    my $host_pkg_archive_doc;
    ( $rc, $host_pkg_archive_doc ) = get_path_from_cs($host_pkg_path);
    
    die "Couldn't fetch the host pkg."
        if ( $rc < 0 );
    
    # create temporary filehandle to hold package
    my ( $pkg_temp_fh, $pkg_temp_filename ) =
        File::Temp::tempfile( 'hostpkgXXXXXX',
                              'DIR'    => File::Spec->tmpdir(),
                              'UNLINK' => 1 );
    die "Couldn't create a temporary file for the host pkg: $!"
        if ( not defined $pkg_temp_fh );

    print $pkg_temp_fh $host_pkg_archive_doc;
    $pkg_temp_fh->flush;

    # decompress the package
    my $host_pkg_doc = `zcat < $pkg_temp_filename`;

    # overwrite temporary file with decompressed package
    seek( $pkg_temp_fh, 0, 0 );
    print $pkg_temp_fh $host_pkg_doc;
    $pkg_temp_fh->flush;

    # check if the client package is already installed
    print_info "Checking if $HostPkgName is already installed... ";
    my $host_pkg_version = check_pkg($HostPkgName);
    if ( defined $host_pkg_version ) {

        my $pkg_file_version = check_pkg_file_version( $pkg_temp_filename,
                                                       $HostPkgName );
        if ( compare_versions($pkg_file_version, $host_pkg_version) > 0 ) {
            print_info "older version ($host_pkg_version).\n";
            if (confirm_package_removal()) {
                remove_pkg($HostPkgName);
                $purge_when_finished = 1;
            } else {
                die "Must remove out-of-date packages to proceed";
            }
        } else {
            print_info "up-to-date version ($host_pkg_version).\n";
            close ($pkg_temp_fh)   or die "Couldn't close temporary file: $!";
            return;
        }

    } else {

        print_info "not installed.\n";

    }

    # create temporary filehandle to hold admin policy document
    my ( $admin_temp_fh, $admin_temp_filename ) =
        File::Temp::tempfile( 'adminXXXXXX',
                              'DIR'    => File::Spec->tmpdir(),
                              'UNLINK' => 1 );
    die "Couldn't create a temporary file for the host admin policy document: $!"
        if ( not defined $admin_temp_fh );
    
    print $admin_temp_fh $ADMIN_DOC;
    $admin_temp_fh->flush;
    
    my $escaped_pkg_temp_filename   = $pkg_temp_filename;
    my $escaped_admin_temp_filename = $admin_temp_filename;
    sanitize_cli_args( \$escaped_pkg_temp_filename, \$escaped_admin_temp_filename );
    
    print_info "Installing package $HostPkgName.\n";
    $rc = system( "$pkgadd_path -n -d $escaped_pkg_temp_filename " .
                  "-a $escaped_admin_temp_filename $HostPkgName " );
    die 'Call to pkgadd failed.'
        if ( $rc != 0 );
    
    close ($admin_temp_fh) or die "Couldn't close temporary file: $!";
    close ($pkg_temp_fh)   or die "Couldn't close temporary file: $!";

    purge_module_packages() if ($purge_when_finished);
}

#
# install_host_package
#
sub install_host_package ()
{
    CASE_PKGMGR : {
        if ( exists $PkgCmdSet->{'rpm'} ) {
            install_host_rpm();
            
            last CASE_PKGMGR;
        }

        if ( exists $PkgCmdSet->{'pkgadd'} ) {
            install_host_pkg();

            last CASE_PKGMGR;
        }
    }
}

#
# remove_rpm
#
sub remove_rpm ($)
{
    my $rpm_name = shift @_;
    die "No rpm name provided to remove_rpm()." if ( ! defined $rpm_name );
    my $rpm_path = $PkgCmdSet->{'rpm'};
    my $rc;

    # check that rpm is executable
    die "$rpm_path is not an absolute path to an executable copy of the rpm program."
        if ( ! -x $rpm_path or
             ! File::Spec->file_name_is_absolute($rpm_path) or
             ( File::Spec->splitpath($rpm_path) )[2] ne 'rpm' );

    print_info "Removing RPM $rpm_name.\n";

    my $escaped_rpm_name = $rpm_name;
    sanitize_cli_args(\$escaped_rpm_name);
    
    my $out = `$rpm_path -e $escaped_rpm_name 2>&1`;
    die 'Call to rpm failed: $out'
        if ( $? != 0 );
}

#
# remove_pkg
#
sub remove_pkg ($)
{
    my $pkgrm_path = $PkgCmdSet->{'pkgrm'};
    my ($pkg_name) = @_;
    my $rc;

    # check arguments
    die "remove_pkg() requires a package name to remove"
        if ( not defined $pkg_name );

    # check that pkgrm is executable
    die "$pkgrm_path is not an absolute path to an executable copy of the pkgrm program."
        if ( ! -x $pkgrm_path or
             ! File::Spec->file_name_is_absolute($pkgrm_path) or
             ( File::Spec->splitpath($pkgrm_path) )[2] ne 'pkgrm' );
    
    # check if the client package is already installed
    print_info "Removing package $pkg_name.\n";
    $rc = system( "$pkgrm_path -n $HostPkgName" );
    die 'Call to pkgrm failed.'
        if ( $rc != 0 );
}

#
# remove_host_package
#
sub remove_host_package ()
{
    CASE_PKGMGR : {
        if ( exists $PkgCmdSet->{'rpm'} ) {
            return if ( not defined check_rpm($HostRPMName) );
            remove_rpm($HostRPMName);

            last CASE_PKGMGR;
        }

        if ( exists $PkgCmdSet->{'pkgrm'} ) {
            return if ( not defined check_pkg($HostPkgName) );
            remove_pkg($HostPkgName);
            
            last CASE_PKGMGR;
        }
    }
}

#
# Basic Modes
#
# do_install
# do_remove
#

#
# do_install
#
sub do_install ()
{
    # check that essential utility commands are available
    check_commands();
    
    # download fresh hostinstall from control station and check version
    # save this copy so we can save it in the management directory
    my $hostinstall_doc = check_hostinstall_version();

    install_host_package();
    create_user_home();
    setup_ssh_key();

    # copy hostinstall to a known location
    print_info "Installing $HostinstallLocalPath.\n";
    my $hostinstall_fh;
    open( $hostinstall_fh,
          "> $HostinstallLocalPath" )
        or die "Could not open $HostinstallLocalPath: $!";
    print $hostinstall_fh $hostinstall_doc;
    close($hostinstall_fh)
        or die "Could not close $HostinstallLocalPath: $!";

    # set permissions
    print_info "Setting owner and permissions.\n";
    my $rc = system( $ChownCmd, '-R', $CSUserName, $CSHomeDir );
    die "Could not set file/directory ownership: $!" if ( $rc != 0 );
    my $files_changed = chmod( 0700, $CSHomeDir, $SSHDir );
    die "Could not set directory permissions: $!" if ( $files_changed != 2 );
    $files_changed    = chmod( 0600, "$SSHDir/authorized_keys" );
    die "Could not set file permissions: $!" if ( $files_changed != 1 );
    $files_changed    = chmod( 0700, $HostinstallLocalPath );
    die "Could not set file permissions: $!" if ( $files_changed != 1 );

    # all done
    print_info "Host is ready to be managed.\n";
}

#
# do_remove
#
sub do_remove ()
{
    # check that essential utility commands are available
    check_commands();
    
    # allow bypass on command-line
    unless ($opts{'y'}) {
        my $confirm;
        print "Ready to remove host administration support.\n";
        print "The $MgmtBaseDir and $CSHomeDir directories\n";
        print "will be removed.\n\n";
        do {
            print "Are you sure you want to proceed? [y/N] ";
            $confirm = <STDIN>;
        } while ( ( $confirm !~ /^y(?:es)?$/i ) and
                  ( $confirm !~ /^no?$/i      ) and
                  ( $confirm !~ /^$/          ) );
        
        exit 4 if ( ( $confirm =~ /^no?$/i ) or 
                    ( $confirm =~ /^$/     ) );
    }
    
    # point of no return
    
    purge_module_packages();
    remove_host_package();
    purge_mgmt_directory();

    # delete csadmin user
    print_info "Deleting csadmin user.\n";
    my $rc = system( "$UserDelCmd -r $CSUserName 2> /dev/null" );
    die "userdel failed" if ( $rc != 0 &&
                              # if the user does not exist, userdel returns
                              WEXITSTATUS($rc) != 6 &&  # shadow-utils (on Solaris, Redhat, JDS)
                              WEXITSTATUS($rc) != 22 ); # pwdutils (on SuSE)

    print_info "Host management removed.\n";
}

#
# Utility Routines
#
# check_hostinstall_version
# check_commands
# has_usable_lwp
# get_url
# get_path_from_cs
# sanitize_cli_args
# purge_module_packages
# confirm_package_removal
#

#
# check_hostinstall_version
#
# Verifies that this copy of hostinstall has been customized and
# populated with a list of control station addresses, and that it is
# the same version as the script present on the control station.
# Exits if there is a version mismatch or there is no available
# control station address.  Returns a complete copy of the server's
# copy of hostinstall as a single scalar.
#
sub check_hostinstall_version ()
{
    # download a copy of hostinstall.pl from the control station
    print_info "Checking hostinstall.pl version.\n";
    my $server_hostinstall_doc = get_path_from_cs($CSHostInstallPath);
    die "Could not connect to the control station"
        unless ( $server_hostinstall_doc );
    
    # parse file to extract version information
    my ($server_hostinstall_version) = ( $server_hostinstall_doc =~ /my \$VERSION\s+=\s+\'(.+)\';/ );

    die "Could not extract version number from the server's copy of hostinstall.pl"
        if ( not defined $server_hostinstall_version );

    if ( $server_hostinstall_version ne $VERSION ) {
        print STDERR << "EOD";
ERROR: This copy of hostinstall.pl is out of date.

This copy of hostinstall.pl is version $VERSION, and the version found
on the control station is version $server_hostinstall_version.
Download the new version of hostinstall.pl from the Control Station
and try again.

EOD
        exit 2;
    }
    print_info "Version is up-to-date.\n\n";

    return $server_hostinstall_doc;
}

#
# check_commands
#
# Checks that various shell utilities are available.  Searches for
# available package commands, and returns the first one found, if any.
# If no package manager commands are found, terminates the script.
#
# After check_commands complete, $PkgCmdSet is set to hold a hash
# reference mapping command names to absolute pathnames.  (Copied from
# the first row of PkgCmdSets where all the commands are accessible.)
#
sub check_commands ()
{
    my $cmd;
    my $cmdset;

    for $cmd (@Cmds) {
        print_info "Checking for $cmd... ";
        if ( -x $cmd ) {
            print_info "found.\n";
        } else {
            print_info "not found.\n";
            die "Could not find $cmd";
        }
    }

    # determine which package system is available
 LOOP_CMD_SET:
    for $cmdset (@PkgCmdSets) {
        for $cmd (values %$cmdset) {
            print_info "Checking for $cmd... ";
            if ( -x $cmd ) {
                print_info "found.\n";
            } else {
                print_info "not found.\n";
                next LOOP_CMD_SET;
            }
        }
        # found all commands in this command set
        $PkgCmdSet = $cmdset;
        last LOOP_CMD_SET;
    }

    die "Could not find a usable package manager."
        if ( ! defined $PkgCmdSet );
}

#
# has_usable_lwp
#
# Check whether the perl system has the Lib Web Perl (LWP) module.
# Adapted from CPAN::has_usable.
#
sub has_usable_lwp {
    my @check_procs = 
        (   
            sub {require LWP},
            sub {require LWP::Simple},
            sub {require LWP::UserAgent},
            sub {require HTTP::Request},
            sub {require URI::URL},
            );
    for my $c (0..$#check_procs) {
        my $code = $check_procs[$c];
        my $ret = eval { &$code() };
        if ($@) {
            warn "DEBUG: c[$c]\$\@[$@]ret[$ret]";
            return;
        }
    }
    return 1;
}

#
# get_url
#
# Returns an array ( $result, $doc ).
#
# On success $doc is the document located at the URL, and $result is
# 0.
#
# On failure $result is set to -1 and $doc is undefined.
#
sub get_url ($) {
    my ($url) = @_;
    my $doc;

    my $escaped_url = $url;
    sanitize_cli_args(\$escaped_url);

    # first try wget
    foreach my $wgetcmd (@WGetCmds) {
        if ( -x $wgetcmd ) {
            $doc = `$wgetcmd -q -t 1 -T $GetURLTimeout -O - $escaped_url`;
            if ( $? != 0 ) {
                return ( -1, undef );
            }
            return ( 0, $doc );
        }
    }
        
    # then try curl
    foreach my $curlcmd (@CurlCmds) {
        if ( -x $curlcmd ) {
            $doc = `$curlcmd -s -m $GetURLTimeout $escaped_url`;
            if ( $? != 0 ) {
                return ( -1, undef );
            }
            return ( 0, $doc );
        }
    }
    
    # finally fall back to LWP if available
    if (&has_usable_lwp) {
        require LWP::Simple;
        $doc = LWP::Simple::get($url);
        if ( not defined $doc ) {
            return ( -1, undef );
        }
        return ( 0, $doc );
    }
    
    # couldn't find anything useful
    warn "Could not find a method to fetch $url";
    return ( -1, undef );
}

#
# get_path_from_cs
#
# Iterates through the possible control station addresses provided on
# the command line through the -c option and in @CSHostAddrs,
# attempting to connect and download a file using get_url().  Memoizes
# the first address the is accessible for future invocations of this
# routine.
#
# Returns the same ( $result, $doc ) array as get_url.
#
# $result is 0 if *any* address was accessible, and a negative integer
# if none of the available addresses worked.
#
{
    # this layer of scoping isolates the memoized address
    my $cs_host_addr;

    sub get_path_from_cs ($) {
        my ($path) = @_;

        my ( $rc, $doc );

        if ( defined $cs_host_addr ) {
            
            # have memoized address already
            print_info "Attempting to retrieve $path from control station at $cs_host_addr... ";
            my $url = "http://$cs_host_addr/$path";
            ( $rc, $doc ) = get_url($url);

            if ( $rc < 0 ) {
                print_info "failed.\n";
            } else {
                print_info "success.\n";
            }
        
        } else {

            # no memoized result available, iterate through the possibilities

            my @cs_addrs = @CSHostAddrs;

            # check addresses provided on command line

            # Ordering is important.  We take the addresses off the
            # command line last-first with pop, and put them on the
            # front of CSHostAddrs with unshift, so that we try the
            # addresses given on the command line first, in the order
            # given.

            if (defined $opts{'c'}) {
                my @cmdline_addrs = split( /,/, $opts{'c'} );
                while (my $cmdline_addr = pop @cmdline_addrs) {
                    my ($verified_cmdline_addr) = ( $cmdline_addr =~ /^($HostnameRE)$/x );
                    if ($verified_cmdline_addr) {
                        unshift( @cs_addrs, $verified_cmdline_addr );
                    } else {
                        die "$cmdline_addr is not a valid IP address or hostname";
                    }
                }
            }

            foreach my $addr ( @cs_addrs ) {

                # make sure we have configured addresses
                if ( $addr eq "__CONTROLSTATION__" ) {
                    print STDERR << 'EOD';
ERROR: This copy of hostinstall.pl has not been configured to connect to
a Control Station.

A customized version of hostinstall.pl is generated during the
installation of the Control Station software.  To manage a host using
that copy of Control Station, you must download the customized
hostinstall.pl from the address:

http://<control station address>/pub/hostinstall.pl

and run that copy on the host you wish to manage.

EOD
                    exit 2;
                }

                print_info "Attempting to retrieve $path from control station at $addr... ";
                
                my $url = "http://$addr/$path";
                ( $rc, $doc ) = get_url($url);
                if ( $rc < 0 ) {
                    print_info "failed.\n";
                } else {
                    print_info "success.\n";
                    # memoize this address for future calls
                    $cs_host_addr = $addr;
                    last;
                }

            }

        }

        return ( $rc, $doc );
    }
}

#
# sanitize_cli_args
#
# Takes an arbitrary number of references to scalars
# which are intended to by passed into command-line programs
# via the shell.  Modifies each scalar by surrounding it in
# single quotes and escaping any embedded single quotes.
#
sub sanitize_cli_args (@)
{
    foreach my $arg_ref (@_) {
        die "sanitize_cli_args() requires references to scalars"
            unless ( ref $arg_ref eq 'SCALAR' );
        
        $$arg_ref =~ s/\'/\'\\\'\'/g;
        $$arg_ref = "'" . $$arg_ref . "'";
    }
}

#
# purge_module_packages
#
# Calls the package manager to remove all client-side module
# components that are currently registered in the package database,
# and clean up the database.  We do this before removing the client
# package and before upgrading a host previously managed by an older
# version of SCS, to avoid leaving module components lying around the
# system.
#
sub purge_module_packages ()
{
    # give up gracefully is pkgMgr is not installed
    return if ( not defined check_host_package() );

    # pkgMgr should be installed; check that it is executable
    die "Cannot run $PkgMgrCmd"
        if ( ! -x $PkgMgrCmd );

    print_info "Running $PkgMgrCmd to remove all SCS module host packages.\n";
    my $rc = system( $PkgMgrCmd, 'full', 'purge' );
    die "$PkgMgrCmd failed" if ( $rc != 0 );
}

#
# purge_mgmt_directory
#
# Deletes the $MgmtBaseDir directory and all its contents.
#
sub purge_mgmt_directory ()
{
    print_info "Removing $MgmtBaseDir.\n";
    system( $RmCmd, '-rf', $MgmtBaseDir );
}

#
# confirm_package_removal
#
# Presents a message informing the user that it is necessary to remove
# outdated client components, including all module components, to
# continue with the process of preparing the host for management.
# Then confirm_package_removal asks the user for confirmation, and
# returns the results of that confirmation to the caller as a boolean
# (1/0) value.  The prompt can be bypassed by providing the -y
# command-line switch.
#
# This routine does not actually perform any removal operations,
# leaving this for the caller.
#
sub confirm_package_removal ()
{
    # allow bypass on command-line
    if ($opts{'y'}) {

        return 1;

    } else {
        
        my $confirm;
        print << 'EOD';

SCS must remove the outdated client support agent.  Any SCS modules
installed on this host will be removed, and will have to be
reinstalled.

* Note that SCS 2.2 does not support all SCS 2.1 features on JDS
  clients.
EOD
        do {
            print "Are you sure you want to proceed? [y/N] ";
            $confirm = <STDIN>;
        } while ( ( $confirm !~ /^y(?:es)?$/i ) and
                  ( $confirm !~ /^no?$/i      ) and
                  ( $confirm !~ /^$/          ) );
        
        return ( $confirm =~ /^y(?:es)?$/i );

    }
}

#
# get_os
#
# Calls the uname(1) command to determine the gross OS type of this
# system.  Memoizes the result for future calls.
#
{
    my $os;
    sub get_os ()
    {
        if (not defined $os) {
            $os = `$UNameCmd -s`;
            chomp $os;
        }
        return $os;
    }
}

#
# get_architecture
#
# Calls the uname(1) command to determine the hardware architecture of
# this system.  Memoizes the result for future calls.
#
{
    my $arch;
    sub get_architecture ()
    {
        if (not defined $arch) {
            $arch = `$UNameCmd -p`;
            chomp $arch;
        }
        return $arch;
    }
}

#
# Password manipulation subroutines
#
# random_string
# crypt_string
# change_passwd
#

#
# random_string
#
# Returns a string of ASCII characters of a specified length.
#
sub random_string ($)
{
    my $str = '';
    $_[0] = 8 if (not defined $_[0]);
    for (my $i=0; $i<$_[0]; $i++) {
        $str .= ('A'..'Z','a'..'z','0'..'9')[rand(62)];
    }
    return $str;
}

#
# crypt_string
#
# Takes its string argument and hashes it using the UNIX crypt(3)
# algorithm, then returns the result. Uses a random salt, and is hence
# not suitable for checking passwords.
#
sub crypt_string ($)
{
    my ($str) = @_;
    die "crypt_string() requires a string argument"
        if ( not defined $str );
    my $salt  = join('', ('.','/',0..9,'A'..'Z','a'..'z')[rand 64, rand 64]);
    return crypt( $str, $salt );
}

#
# change_passwd
#
# Changes a user's password by editing the shadow password file
# /etc/shadow.  Uses perl's flock() function to reduce the possibility
# of simultaneous edits to the file.  If the username appears multiple
# times in the shadow password file, all instances are modified.
#
# Takes the username of the account to change, and the new hashed
# password.  (See crypt(3) and perlfunc(3pm).)
#
sub change_passwd ($$)
{
    my ( $username, $crypted_passwd ) = @_;

    # validate arguments
    die "change_passwd requires a valid username and crypted password"
        if ( not defined $username or not defined $crypted_passwd );
    my ($valid_username)       = ( $username       =~ m|^([./0-9A-Za-z]{2,})$|  );
    my ($valid_crypted_passwd) = ( $crypted_passwd =~ m|^([./0-9A-Za-z]{13,})$| );
    die "Invalid username provided to change_passwd"
        if ( not defined $valid_username );
    die "Invalid crypted password provided to change_passwd"
        if ( not defined $valid_crypted_passwd );

    # slurp /etc/shadow
    my $shadow_fh;
    # open for reading and writing
    open $shadow_fh, "+< $ShadowFile";
    # get a lock before we start reading
    flock( $shadow_fh, LOCK_EX )
        or die "Could not acquire a lock on $ShadowFile: $!";

    my @lines;
    while (<$shadow_fh>) {
        if (/^$valid_username:/o) {
            # replace password
            s|^$valid_username:[^:]*:|$valid_username:$valid_crypted_passwd:|;
        }
        push @lines, $_;
    }

    # overwrite file contents
    seek( $shadow_fh, 0, 0 )
        or die "Could not seek to the start of $ShadowFile: $!";
    print $shadow_fh @lines;

    flock( $shadow_fh, LOCK_UN )
        or die "Could not release lock on $ShadowFile: $!";
    close( $shadow_fh )
        or die "Could not close $ShadowFile: $!";
}

#
# main
#
{
    # process command-line arguments first so quiet mode can take effect
    getopts( 'c:qy', \%opts );
    # c : directly entered control station addresses
    # q : quiet mode
    # y : "yes" mode; don't ask for remove confirmation

    print_info "hostinstall.pl version $VERSION\n";
    if ($opts{'h'} or $opts{'?'}) {
        print_usage();
        exit 0;
    }
    
    # clear environment to protect against taint
    delete @ENV{'PATH', 'IFS', 'CDPATH', 'ENV', 'BASH_ENV'};
    # make sure we get English messages from system commands
    $ENV{'LC_ALL'} = 'C';

    my $mode = shift @ARGV;
    CASE_MODE: {
        if ( not defined $mode ) {
            print_info "No mode of operation specified.\n";
            print_usage();
            exit 1;
            
            last CASE_MODE;
        }

        if ( $mode eq 'install' ) {
            do_install();
            
            last CASE_MODE;
        }

        if ( $mode eq 'remove' ) {
            do_remove();

            last CASE_MODE;
        }

        # default
        {
            print_info "Invalid mode of operation specified.\n";
            print_usage();
            exit 3;
        }
        
    }

    exit 0;
}
