#!/usr/bin/perl -w

# Copyright (c) 2003 Sun Microsystems, Inc. All rights reserved
# SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.


#
# $Id: groupMgr.pl,v 1.15 2004/10/12 21:49:51 ms152511 Exp $
#

use strict;
use lib '/scs/lib/perl5';
use BDi18n;
use SCSDB;
use Group;
use CLIids;

sub printUsage
{
  print STDERR<<EOD;

Usage : $0 <mode> [parameters]

Modes:
------
 -l [group id]                       : List groups, or display a group's list
                                       of hosts.
 -n <name> [desc] [host id,...]      : Create a new group with the given name.
                                       Provide a null argument (i.e. "") for
                                       desc if you don't want one but you
                                       want to put host ids.
 -d <group id>                       : Delete a group.
 -m <group id> <new name> [new desc] : Modify a group.
 -a <group id> <host id,...>         : Add host(s) to a group.
 -r <group id> <host id,...>         : Remove host(s) from a group.
EOD

  CLIids::printUsage();

  exit 0;
}

#
# main
#
BDi18n::setDomain('base-mgmt-groupMgr');
BDUtil::openLog('groupMgr');

printUsage() if (!defined($ARGV[0]) || (length($ARGV[0]) != 2));

CLIids::setExitOnError();
my $mode = shift @ARGV;
$mode = substr($mode, 1, 1);

my %modeHash =
(
  'l' => \&listMode,
  'n' => \&newMode,
  'd' => \&delMode,
  'm' => \&modMode,
  'a' => \&addMode,
  'r' => \&removeMode
);

printUsage() if (!exists($modeHash{$mode}));

my $rc = &{$modeHash{$mode}}(@ARGV);

BDUtil::closeLog();
exit $rc;

#
# groupMgr_error
#
# Takes a hash ref which contains at least the key 'tag,' which is the
# resource tag for the error message to be printed.  Prints the
# interpolated error message to STDERR.
#
sub groupMgr_error ($)
{
    die 'groupMgr_error requires an argument'
        unless (defined $_[0]);
    print STDERR 'ERROR: ' . BDi18n::getMsg($_[0]) . "\n";
}

##
#
##
sub listMode
{
  if (scalar @_ == 0)
  {
    my $groupList = SCSDB::getResultHashes('select g.view_id, g.view_name, g.view_description, count(distinct a.appliance_id) as cnt from mgmt_view g left join mgmt_view_to_appliance a on g.view_id = a.view_id group by g.view_id, g.view_name, g.view_description');

    if (!$groupList)
    {
      print "No groups.\n";
      return 0;
    }

    printf "%4s  %-20s  %-30s  %s\n", 'id', 'Group Name', 'Description', 'Hosts';
    printf "%4s  %-20s  %-30s  %s\n", '--', '----------', '-----------', '-----';

    foreach my $g (@{$groupList})
    {
      printf "%4d  %-20s  %-30s  %d\n", $g->{'view_id'}, $g->{'view_name'}, $g->{'view_description'} || '', $g->{'cnt'} || 0;
    }
  }
  else
  {
    my $gId = shift @_;
    # check if group exists
    my $aIds = Group::getGroupHosts($gId);
    if (ref($aIds) ne 'ARRAY') {
        groupMgr_error(SCSDB::getErrorMessage());
        return 0;
    } elsif(scalar @$aIds == 0) {
      print "No hosts in group id = $gId\n";
      return 0;
    }
    my $appliances = Appliance::getApplianceDetail(@$aIds);
    if (!$appliances)
    {
        groupMgr_error(SCSDB::getErrorMessage());
        return 0;
    }
    printf "%4s  %-13s  %-20s\n", 'id', 'IP Address', 'Host Name';
    printf "%4s  %-13s  %-20s\n", '--', '----------', '---------';
    foreach my $a (@{$appliances})
    {
      printf "%4d  %-13s  %s\n", $a->{'appliance_id'}, $a->{'ip_address'}, $a->{'host_name'};
    }
  }
  return 0;
}

##
#
##
sub newMode
{
    my $newName = shift @_;
    my $newDesc = shift @_;
    printUsage() if (not defined $newName);
    
    my $gID = Group::addGroup( $newName, $newDesc );
    if ($gID < 0) {
        groupMgr_error(SCSDB::getErrorMessage());
        return 2;
    }
    print "New group id = " . $gID . "\n";
    if (scalar @_ > 0)
    {
        return addMode($gID, @_);
    }
}

##
#
##
sub delMode
{
    my $gID = shift @ARGV;
    printUsage() if (not defined $gID);
    
    my $rc = Group::removeGroup($gID);

    if ($rc < 0) {
        groupMgr_error(SCSDB::getErrorMessage());
        return 2;
    }
    return 0;
}

##
#
##
sub modMode
{
    my $gId = shift @_;
    my $newName = shift @_;
    my $newDesc = shift @_;
    printUsage() if (not defined $gId || not defined $newName);
    
    my $rc = Group::modifyGroupDetails( $gId, $newName, $newDesc );
    if ($rc < 0) {
        groupMgr_error(SCSDB::getErrorMessage());
        return 2;
    }
    return 0;
}

##
#
##
sub addMode
{
    my $gID = shift @_;
    my $hIDs = CLIids::getHostIds(\@_);
    printUsage() if (not defined $gID);
    
    my $rc = Group::addHostsToGroup( $gID, $hIDs );
    if ($rc < 0) {
        groupMgr_error(SCSDB::getErrorMessage());
        return 2;
    }
    print "Added hosts to group id = $gID\n";
    return 0;
}

##
#
##
sub removeMode
{
  my $gID = shift @_;
  printUsage() if (not defined $gID);

  my $rc = Group::removeHostsFromGroup( $gID, \@_ );
  if ($rc < 0) {
      groupMgr_error(SCSDB::getErrorMessage());
      return 2;
  }
  print "Removed hosts from group id = $gID\n";
  return 0;
}
