# $Id: RawAgentInterface.pm,v 1.28.2.1 2005/01/21 23:55:54 ms152511 Exp $
# RawAgentInterface.pm - low-level communication over SSH to managed hosts
#
# Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#

#
# connection manipulation:
#
# ConnCheck    ( remote_host [, remote_port [, timeout] ] )
# ConnStart    ( remote_host [, remote_port [, timeout] ] )
# ConnEnd      ( connection_hash_ref )
#
# host interaction:
#
# RunCmd       ( connection_hash_ref, command [, arguments ...] )
# SendFile     ( connection_hash_ref, local_path [, remote_path] )
# GetFile      ( connection_hash_ref, remote_path [, local_path] )
#
# hostkey management:
#
# AddPublicKey ( public_key )
#

package RawAgentInterface;

use strict;
use File::Spec;
use IO::File;
use IO::Handle;
use IO::Socket;
use IO::Select;
use IPC::Open3;
use lib '/scs/lib/perl5';
use Appliance;
use PropertyUtil;
use BDUtil;

my $Scp_Path         = PropertyUtil::getProperty('cmd.scp');
my $Ssh_Path         = PropertyUtil::getProperty('cmd.ssh');

my $Version          = '2.2';
my $Default_Username = 'csadmin';
my $Remote_Shell     = '/bin/sh';
my $Identity_Path    = PropertyUtil::getProperty('file.privatekey');
my $Known_Hosts_Path = PropertyUtil::getProperty('file.knownhosts');
my $Default_Port     = 22;
my $Default_Timeout  = 60;
my $Buffer_Size      = 1024;

my %VERIFY_CKSUM     = ( 'ONFAIL' => 1,
                         'ALWAYS' => 0 );
my $Cksum_Path       = PropertyUtil::getProperty('cmd.cksum');

#
# ConnCheck
#
# Checks only whether a TCP connection can be established to a given host/port.
#
# Returns a negative integer if some failure occurs, or zero on success.
#
sub ConnCheck ($;$$)
{
    my ($remote_host, $remote_port, $timeout) = @_;
    return -1 unless (defined $remote_host);

    if ( defined $remote_port ) {
        if ( int($remote_port) >= 1 && int($remote_port) <= 65000 ) {
            $remote_port = int($remote_port);
        } else {
            return -2;
        }
    } else {
        $remote_port = $Default_Port;
    }

    $timeout = $Default_Timeout unless ( defined $timeout && 
                                         $timeout + 0 > 0 );

    BDUtil::printDebug("Host    is $remote_host");
    BDUtil::printDebug("Port    is $remote_port");
    BDUtil::printDebug("Timeout is $timeout");

    my $conn = IO::Socket::INET->new ( PeerAddr => $remote_host,
                                       PeerPort => $remote_port,
                                       Proto    => 'tcp',
                                       Type     => SOCK_STREAM,
                                       Timeout  => $timeout );    
    unless (defined $conn) {
        BDUtil::warning("Can't connect to $remote_host");
          return -3;
      }
    
    close($conn);
    return 0;
}

#
# ConnStart
#
# Open an SSH connection to the csadmin account on the remote host.
# Returns a reference to a hash representing the connection opened.
# This hash reference needs to be passed to all other routines dealing
# with this connection.
#
# Returns a negative integer if an error occurs.
#
sub ConnStart ($;$$)
{
    my ( $remote_host, $remote_port, $timeout ) = @_;

    # validate arguments

    return -1 unless defined $remote_host;
    $remote_host = SCSUtil::getIPAddress($remote_host);
    return -1 if ( $remote_host eq '-1' );

    if ( defined $remote_port ) {
        if ( int($remote_port) >= 1 && int($remote_port) <= 65000 ) { $remote_port = int($remote_port) }
        else { return -2 };
    } else {
        $remote_port = $Default_Port;
    }

    $timeout = ( defined $timeout && ( $timeout + 0 ) > 0 ) ? $timeout : $Default_Timeout;

    # make sure host is actually managed
    my $host_id     = Appliance::getApplianceId($remote_host);
    return -2 
        if ($host_id < 0);
    my $host_status = Appliance::getApplianceStatus($host_id);
    return -2
        unless ($host_status eq 'M');

    my $remote_user = $Default_Username;

    my @ssh_cmd = ( $Ssh_Path,
                    "-i$Identity_Path",
                    # Suppress requests for host key approval, etc.
                    '-o BatchMode yes',
                    "-o GlobalKnownHostsFile $Known_Hosts_Path",
                    '-o UserKnownHostsFile ' . File::Spec->devnull(),
                    '-q',
                    "-p$remote_port",
                    "-l$remote_user",
                    $remote_host,
                    $Remote_Shell );

    my $wtr     = new IO::Handle;
    my $rdr     = new IO::Handle;

    local *DEVNULL;
    open( DEVNULL, ">", File::Spec->devnull );

    my $ssh_pid = open3( $wtr, $rdr, ">&DEVNULL", @ssh_cmd );
    $wtr->autoflush();

    my $conn_hash_ref = { 'vers' => $Version,
                          'host' => $remote_host,
                          'port' => $remote_port,
                          'user' => $remote_user,
                          'tout' => $timeout,
                          'pid'  => $ssh_pid,
                          'wtr'  => $wtr,
                          'rdr'  => $rdr };

    my ( $rc, $stdout ) = RunCmd( $conn_hash_ref, '/bin/echo', 'test' );

    if ( $rc == 0 && $stdout eq "test\n" ) {
        return $conn_hash_ref;
    } else {
        ConnEnd($conn_hash_ref);
        return -1;
    }
}

#
# ConnEnd
#
# Closes a connection opened previously with ConnStart and waits
# for the SSH process to shut down.
#
# Returns 0 on success, or -1 if the argument does not represent a
# valid connection.
#
sub ConnEnd ($)
{
    my ($conn_hash_ref) = @_;
    return -1 if ( ref($conn_hash_ref) ne 'HASH' || $conn_hash_ref->{'vers'} != $Version );
    
    if ( defined $conn_hash_ref->{'wtr'} ) {
        close $conn_hash_ref->{'wtr'};
        undef $conn_hash_ref->{'wtr'};
    }

    if ( defined $conn_hash_ref->{'rdr'} ) {
        close $conn_hash_ref->{'rdr'};
        undef $conn_hash_ref->{'rdr'};
    }

    my $pid = $conn_hash_ref->{'pid'};
    if ( defined($pid) && $pid ne '' ) {
        undef $conn_hash_ref->{'pid'};
        waitpid ( $pid, 0 )
    }

    return 0;
}

#
# RunCmd
#
# Runs the given command on the remote host on the other side
# of the provided agent connection.
#
# Returns a two-element array with the return code from the process as
# the first element and the complete standard output resulting from
# the command as the second element.  Standard error is discarded.
#
# If an error occurs, the 'return code' will be a negative integer,
# and the standard output will be the undefined value.
#
sub RunCmd ($;@)
{
    my ( $conn_hash_ref, $cmd, @args ) = @_;
    return ( -1, undef ) if ( ref($conn_hash_ref) ne 'HASH' || $conn_hash_ref->{'vers'} != $Version );
    return ( -1, undef ) if ( not defined $cmd or $cmd eq '' );

    my $rdr = $conn_hash_ref->{'rdr'};
    my $wtr = $conn_hash_ref->{'wtr'};

    # make sure the connection is alive
    return ( -1, undef ) if ( not defined $rdr || not defined $wtr );
    
    # Put command in temporary document in tmp.
    # This lets us avoid quoting problems of passing the command
    # through the session shell.

    # Generate filename based on host-side shell PID.
    
    my $cmd_filename = '/tmp/cscmd$$';

    print $wtr "cat > $cmd_filename << '__ENDOFCOMMAND__'\n";
    print $wtr "$cmd @args\n";
    print $wtr "__ENDOFCOMMAND__\n";

    # execute temporary document

    print $wtr "/bin/sh $cmd_filename ; ";
    print $wtr "echo '\n__ENDOFOUTPUT__\n'" . '$?' . "\n";

    # grab command output

    my $outputStr = '';
    my $rc;
    my $selector = new IO::Select($rdr);
    while (1) {
        my @ready_fh = $selector->can_read( $conn_hash_ref->{'tout'} );
        if ( scalar(@ready_fh) != 1 || $ready_fh[0] != $rdr ) {
            # timed out without receiving data
            return ( -2, undef );
        }

        my $buffer;
        my $chars_read = $rdr->sysread( $buffer, $Buffer_Size );

        if ( $chars_read == 0 ) {
            # EOF, so the pipe must have died
            ConnEnd($conn_hash_ref);
            return ( -2, undef );
        }

#        BDUtil::printDebug("Read: $buffer");
        $outputStr .= $buffer;
#        BDUtil::printDebug("Accumulated: \"$outputStr\"");

        # look for our distinctive ending token
        if ( $outputStr =~ s/[\n]__ENDOFOUTPUT__[\n](\d+)[\n]$// ) {
            # performed a subsitution, so the end of the output was detected
            # numeric return code is in $1
            $rc = $1;
            last;
        }
    } 

    # delete command file

    print $wtr "/bin/rm $cmd_filename\n";

    return ( int($rc), $outputStr );
}

#
# SendFile
#
# Transfers a file to the remote host using scp.  Does not actually
# use the connection provided, but do *not* depend on this behavior.
# All paths must be absolute paths, i.e. start with the '/' directory
# separator character.  The remote path is optional, and if
# unspecified, the file is copied to the remote host's temporary files
# directory (see PropertyUtil.pm).
#
# Returns a negative integer on failure or 0 on success.
#
sub SendFile  ($$$)
{
    my ( $conn_hash_ref, $local_path, $remote_path ) = @_;
    return -1 if ( ref($conn_hash_ref) ne 'HASH' || $conn_hash_ref->{'vers'} ne $Version );
    
    my $remote_host = $conn_hash_ref->{'host'};
    return -1 if ( not defined $remote_host or $remote_host eq '' );
    return -1 if ( not defined $remote_path );
    
    return -1 if ( not defined $local_path );
    # check that pathnames are absolute
    return -1 if ( not File::Spec->file_name_is_absolute($local_path) or
                   not File::Spec->file_name_is_absolute($remote_path) );
    # check that local file exists
    return -1 if ( not -f $local_path );

    my $rc = system( $Scp_Path,
                     "-i$Identity_Path",
                     '-o BatchMode yes',
                     "-o GlobalKnownHostsFile $Known_Hosts_Path",
                     '-o UserKnownHostsFile ' . File::Spec->devnull(),
                     '-q',
                     '-p', # maintain permissions
                     "-P$conn_hash_ref->{'port'}", 
                     $local_path,
                     "$conn_hash_ref->{'user'}@" . "$conn_hash_ref->{'host'}:$remote_path" );

    if ( $rc != 0 && $VERIFY_CKSUM{'ONFAIL'} ||
         $VERIFY_CKSUM{'ALWAYS'} ) {
        return ( _compare_cksum( $conn_hash_ref,
                                 $local_path, $remote_path ) == 1 ?
                 0 :
                 -1 );
    }

    return 0;
}

#
# GetFile
#
# Transfers a file from the remote host using scp.  Does not actually
# use the connection provided, but do *not* depend on this behavior.
# All paths must be absolute paths, i.e. start with the '/' directory
# separator character.  The local path is optional, and if unspecified
# the file is copied to the local temporary file directory (see
# PropertyUtil.pm).
#
# Returns -1 on failure or 0 on success.
#
sub GetFile ($$$)
{
    my ( $conn_hash_ref, $remote_path, $local_path ) = @_;
    return -1 if ( ref $conn_hash_ref ne 'HASH' || $conn_hash_ref->{'vers'} ne $Version );
    
    my $remote_host = $conn_hash_ref->{'host'};
    return -1 if ( not defined $remote_host or $remote_host eq '' );
    return -1 if ( not defined $local_path );
    
    return -1 if ( not defined $remote_path );
    # check that pathnames are absolute
    return -1 if ( not File::Spec->file_name_is_absolute($local_path) or
                   not File::Spec->file_name_is_absolute($remote_path) );

    my $rc = system( $Scp_Path,
                     "-i$Identity_Path",
                     '-o BatchMode yes', 
                     "-o GlobalKnownHostsFile $Known_Hosts_Path",
                     '-o UserKnownHostsFile ' . File::Spec->devnull(),
                     '-q',
                     '-p', # maintain permissions
                     "-P$conn_hash_ref->{'port'}", 
                     "$conn_hash_ref->{'user'}@" . "$conn_hash_ref->{'host'}:$remote_path",
                     $local_path );

    if ( $rc != 0 && $VERIFY_CKSUM{'ONFAIL'} ||
         $VERIFY_CKSUM{'ALWAYS'} ) {
        return ( _compare_cksum( $conn_hash_ref,
                                 $local_path, $remote_path ) == 1 ?
                 0 :
                 -1 );
    }

    return 0;
}

#
# AddPublicKey
#
# Adds a public key to the known hosts database for the agent.
# Takes a single argument in a string "hostname type publickey"
# as output by ssh-keyscan.
#
# Returns -1 on failure.  Returns 0 on success.
#
sub AddPublicKey ($)
{
    my ($publickey) = @_;
    my ( $host, @fields ) = split( / /, $publickey );

    BDUtil::printDebug("Entering addPublicKey host=$host fields=@fields");

    my $known_hosts_ref = _read_known_hosts_file();
    $known_hosts_ref->{$host} = \@fields;

    my $rc = _write_known_hosts_file($known_hosts_ref);

    return $rc;
}

#
# RemovePublicKey
#
# Removes a public key from the known hosts database for the agent.
# Takes a single scalar argument, the hostname (actually the IP address)
# of the host to be removed.
#
# Returns 0 on success.  Returns -2 if the hostname specified is not
# found in the known hosts database.  Returns -1 if some other error
# occurs.
#
sub RemovePublicKey ($)
{
    my ($host) = @_;
    return -4 if ( not defined $host );

    my $known_hosts_ref = _read_known_hosts_file();
    return -2 if ( not exists $known_hosts_ref->{$host} );

    delete $known_hosts_ref->{$host};
    
    my $rc = _write_known_hosts_file($known_hosts_ref);
    
    return $rc;
}

#
# Internal routines
#

#
# _compare_cksum
#
# Calculates a checksum using the cksum(1) command on the remote and
# and local hosts, and returns 1 if the two files match, and 0 zero if
# they do not.  If an error occurs, returns -1.
#
sub _compare_cksum ($$$)
{
    my ( $conn, $local_path, $remote_path ) = @_;
    my ( $rc, $output );

    # compute local checksum
    $output            = `$Cksum_Path $local_path`;
    return -1 unless (defined $output);
    my ($local_cksum)  = ( $output =~ /^(\d+)\b/ );
    return -1 unless (defined $local_cksum);

    # compute remote checksum
    ( $rc, $output )   = RunCmd( $conn, "$Cksum_Path $remote_path" );
    return -1 unless ($rc == 0);
    my ($remote_cksum) = ( $output =~ /^(\d+)\b/ );
    return -1 unless (defined $remote_cksum);
    
    return ( $local_cksum == $remote_cksum );
}

#
# _read_known_hosts_file
#
# Builds a hash of known RSA public host keys by reading from
# $Known_Hosts_Path.
#
# Returns a reference to the hash which contains entries
# in the form ( hostname, [ fields... ] ).
#
sub _read_known_hosts_file ()
{
    my %known_hosts;

    BDUtil::printDebug("Opening $Known_Hosts_Path for reading");
    if ( ! -e $Known_Hosts_Path ) {
        # no file to read
        return {};
    }
    my $known_hosts_fh = new IO::File "< $Known_Hosts_Path";
    if ( not $known_hosts_fh ) {
        BDUtil::warning("Could not open $Known_Hosts_Path for reading");
        return {};
    }

    while ( my $host_entry = $known_hosts_fh->getline ) {
        chomp $host_entry;
        my @host_entry_fields = split( / /, $host_entry );
        my $host = shift @host_entry_fields;
        my $type = $host_entry_fields[0];
        if ( $type eq 'ssh-rsa' ) {
            $known_hosts{$host} = \@host_entry_fields;
        } else {
            BDUtil::warning("Found non RSA public key in $Known_Hosts_Path for $host.");
        }
    }
    $known_hosts_fh->close;
    BDUtil::printDebug("Read " . scalar(keys(%known_hosts)) . " keys");
    
    return \%known_hosts;
}

#
# _write_known_hosts_file
#
# Writes out the contents of a hash to the known hosts database,
# overwriting any previous contents.  Takes a reference to a hash in
# ( hostname, [ type, key ] ) format.
#
# Returns 0 on success, -1 on failure.
#
sub _write_known_hosts_file ($)
{
    my ($known_hosts_ref) = @_;
    return -1 unless ( defined($known_hosts_ref) and
                       ref($known_hosts_ref) eq 'HASH' );
    
    BDUtil::printDebug("Opening $Known_Hosts_Path for writing");
    my $known_hosts_fh = new IO::File "> $Known_Hosts_Path";
    if ( not $known_hosts_fh ) {
        BDUtil::warning("Could not open $Known_Hosts_Path for writing");
          return -1;
      }
    
    my @hosts = keys %$known_hosts_ref;
    BDUtil::printDebug(scalar @hosts . " hostkeys to write");

    foreach my $host ( keys %$known_hosts_ref ) {
        my $fields_ref = $known_hosts_ref->{$host};
#        BDUtil::printDebug("Host: $host Fields: @$fields_ref");
        print $known_hosts_fh "$host @$fields_ref\n";
    }
    $known_hosts_fh->close;

    return 0;
}
