#!/bin/sh

#
# Copyright (c) 06/13/05 Sun Microsystems, Inc. All Rights Reserved.
#
# ident	"@(#)zone_create.sh	1.12	05/06/13 SMI"
#

#
# This script does the following actions
#	a. Creates a zone
#	b. Installs the zone
#	c. Sets the sysidcfg info
#	d. Boots the zone
#


#
# The return values or the exit values for all the scripts should 
# be of the following:
#
# exit_status	meaning
#	0	Command successful
#	1	Usage error
#	2 	Not a root to execute this script
#	3 	Command not found
#	4	Command failed for some reason
#

#
# Source the common commands and files script
#
SCRIPTDIR=`/usr/bin/dirname $0`
. ${SCRIPTDIR}/scm-agent-common.sh

#
# make sure this test is run as root
#
if [ "`$ID -un`" != "root" ] ; then
    $ECHO "You are not a root to execute this command\n" > "/dev/stderr"
    exit 2
fi

usage() {
    $ECHO "Usage: zone_create.sh -n <zonename> -r <zonepath> -i <IPAddress> -h <hostname> -e <Physical> -p <pool> -c <shares> -o <op> -I <inputBW> -O <outputBW> -m <maxshares>" > "/dev/stderr"

   $ECHO "
      -n <zonename>	Name of the zone to be created
      -r <zonepath>     Zone Path
      -i <IPAddress>	Zone IP Address
      -h <hostname>	Zone's Host name
      -e <Physical> 	Physical Address
      -p <pool>		Pool for the zone
      -c <shares>	Shares for this zone
      -o <operation>	how to create a zone: ready/active
      -I <inputBW>      Input Bandwidth Limit
      -O <outputBW>     Output Bandwidth Limit
      -m <maxshares>    Max Zone Shares
"
}


###############################################################################
#	Function name	: configure_zone
#	Purpose		: To configure zone
#	Arguments	: zonename [ IPAddress physical zonepath ]
#
###############################################################################

configure_zone () {

    flag=0
    zonename="$1"
    zoneip=`/usr/bin/echo $2 | /usr/bin/sed -e "s/\"//g"`
    zonepath=$3
    zonephysical=$4
    zonepool=$5
    zonecpushares=$6
    zonemaxshares=$7
    zoneautoboot=$8
    extrazoneattr="$9"

    zonelog "Configuring zone with the following attributes:"
    zonelog "------------"
    zonelog "zonename $zonename"
    zonelog "zoneip $zoneip"
    zonelog "zonepath $zonepath"
    zonelog "zonephysical $zonephysical"
    zonelog "zonepool $zonepool"
    zonelog "zonecpushares $zonecpushares"
    zonelog "zonemaxshares $zonemaxshares" 
    zonelog "zoneautoboot $zoneautoboot"
    zonelog "extrazoneattr $extrazoneattr"
    zonelog "------------"
    

    if [ -z $zonename ]; then
	zonelog "Error: Zonename is not given."
	return 1
    fi

    /usr/sbin/zoneadm list -c | /usr/bin/grep -w $zonename >/dev/null 2>&1
    if [ $? -ne 0 ]; then

	if [ -z "$zonepath" ]
	then
		zonepath=/export/$zonename
	fi

	if [ ! -z "$zoneip" ]
	then
		flag=1
	fi

	# If user inputs address,physical first predicate will be
	# applied

        # the zone.max-lwps takes care of issue described in 5028108 

	if [ $flag -ne 0 ]; then
	        zonelog "Configuring the zone with IP address and network interface."
		/usr/sbin/zonecfg -z $zonename "create;set zonepath=$zonepath;\
			add net; set address=$zoneip;\
			set physical=$zonephysical;end;\
			set pool=$zonepool;\
			set autoboot=$zoneautoboot;\
			add rctl; set name=zone.cpu-shares;\
			add value (priv=privileged,limit=$zonecpushares,action=none);end;\
			add attr; set name=zone.scm.max-shares;\
			set type=int;\
			set value=$zonemaxshares;end;\
                        add rctl; set name=zone.max-lwps;\
                        add value (priv = privileged, limit = 300, action = deny);end;\
			verify;commit" >> "$ZONE_LOGFILE" 2>&1
		if [ $? -ne 0 ]; then
	                zonelog "Error: Primary Zone Configuration failed."
			return 1
		fi
	else
	        zonelog "Configuring the zone without IP address and network interface."

		/usr/sbin/zonecfg -z $zonename "create;set zonepath=$zonepath;\
			set pool=$zonepool;\
			set autoboot=$zoneautoboot;\
			add rctl; set name=zone.cpu-shares;\
			add value (priv=privileged,limit=$zonecpushares,action=none);end;\
			add attr; set name=zone.scm.max-shares;\
			set type=int;\
			set value=$zonemaxshares;end;\
                        add rctl; set name=zone.max-lwps;\
                        add value (priv = privileged, limit = 300, action = deny);end;\
			verify;commit" >> "$ZONE_LOGFILE" 2>&1
		if [ $? -ne 0 ]; then
	                zonelog "Error: Zone Configuration failed."
			return 1
		fi
	fi
        if [ ! -z "$extrazoneattr" ]; then
                    /usr/sbin/zonecfg -z $zonename "$extrazoneattr" >> "$ZONE_LOGFILE" 2>&1
		    if [ $? -ne 0 ]; then
	                zonelog "Error: Zone Configuration failed for additional attributes."
			return 1
		    fi
        fi
    fi

    zonelog "Zone Configuration successful."
    return 0
}


###############################################################################
#	Function name	: install_zone
#	Purpose		: To install zone. 
#	Arguments	: zonename
###############################################################################

install_zone() {
	zonename=$1

	# Install the zone once configuration is created.
	/usr/sbin/zoneadm list -i | grep -w $zonename >/dev/null 2>&1
	if [ $? -ne 0 ]; then
		/usr/sbin/zoneadm -z $zonename install >>"$ZONE_LOGFILE" 2>&1
		if [ $? -ne 0 ]; then
                        zonelog "Error: Zone install failed."
			return 1
		fi
	fi

        zonelog "Zone install successful."
	return 0
}

###############################################################################
#	Function name	: activate_zone
#	Purpose		: To activate zone. i.e, boot the zone first time
#	Arguments	: zonename
###############################################################################

activate_zone() {
	zonename=$1

        zone_root_path=`zoneadm -z $zonename list -v | grep -v PATH | \
                awk -F' ' '{print $4}'`
        if [ $? -ne 0 -o -z $zone_root_path ]; then
                return 1
        fi

        sysidcfg="$zone_root_path/root/etc/sysidcfg"
        if [ ! -f "$sysidcfg" ]; then
            zonelog "$sysidcfg is not configured yet."
            return 0
        fi

	# activate the zone after installing
	/usr/sbin/zoneadm list | grep -w $zonename >/dev/null 2>&1
	if [ $? -ne 0 ]; then
		/usr/sbin/zoneadm -z $zonename boot >> "$ZONE_LOGFILE" 2>&1
		if [ $? -ne 0 ]; then
                        zonelog "Zone activation failed."
			return 1
		fi
	fi
        zonelog "Zone activation was successful."
	return 0
}


###############################################################################
#
#	Function name	: create_zone
#	Purpose		: to create a "running" state non-global zone.
#	Arguments	: zonename
#
###############################################################################

create_running_state_zone() {
	zonename=$1
	zonepath=$2
	zoneip=$3
	zonephysical=$4
	zonepool=$5
	zonecpushares=$6
	zonemaxshares=$7
        zoneautoboot=$8
        extrazoneattr="$9"

	# Attempt to create a "running" state non-global zone.
        zonelog "Creating a running state zone."
	configure_zone "$zonename" "$zoneip" "$zonepath" "$zonephysical" "$zonepool" "$zonecpushares" "$zonemaxshares" "$zoneautoboot" "$extrazoneattr"
	if [ $? -ne 0 ]; then
	        zonelog "Error while configuring the zone"
		return 1
	fi

	install_zone $zonename
	if [ $? -ne 0 ]; then
	        zonelog "Error while installing the zone"
		/usr/sbin/zonecfg -z $zonename delete -F > /dev/null 2>&1
		return 1
	fi

	activate_zone $zonename
	if [ $? -ne 0 ]; then
		cleanup_zone $zonename
	        zonelog "Error while activating the zone"
		return 1
	fi

	zonelog "Successfully created the zone."
	return 0
}

###############################################################################
#
#	Function name	: create_ready_state_zone
#	Purpose		: to create a "ready" state non-global zone.
#	Arguments	: zonename
#
###############################################################################

create_ready_state_zone() {
	zonename=$1
	zonepath=$2
	zoneip=$3
	zonephysical=$4
	zonepool=$5
	zonecpushares=$6
	zonemaxshares=$7
        zoneautoboot=$8
        extrazoneattr="$9"

	# Attempt to create a "ready" state non-global zone.
	zonelog "Creating ready state zone"
	configure_zone "$zonename" "$zoneip" "$zonepath" "$zonephysical" "$zonepool" "$zonecpushares" "$zonemaxshares" "$zoneautoboot" "$extrazoneattr"
	if [ $? -ne 0 ]; then
	        zonelog "Error while configuring the zone"
		return 1
	fi
	zonelog "Installing zone"
	install_zone $zonename
	if [ $? -ne 0 ]; then
	        zonelog "Error while installing the zone"
		/usr/sbin/zonecfg -z $zonename delete -F > /dev/null 2>&1
		return 1
	fi

	zonelog "Getting the zone in ready state"
	/usr/sbin/zoneadm -z $zonename ready > /dev/null 2>&1
	if [ $? -ne 0 ]; then
	        zonelog "Error while getting the zone in ready state"
		/usr/sbin/zoneadm -z $zonename uninstall -F > /dev/null 2>&1
		/usr/sbin/zonecfg -z $zonename delete -F > /dev/null 2>&1
		return 1
	fi


	zonelog "Successfully created a ready state zone."
	return 0
}

###############################################################################
#
#	Function name	: configure_ipqos
#	Purpose		: To create ipqos configuration file with 
#                         SCM supported parameters for the zone 
#                         and activate it if required.
#	Arguments	: $1 : Zone Name
#                         $2 : IP Address of the zone
#                         $3 : bandwidth threshold to be set as 
#                              inbound committed_rate and burst
#                         $4 : bandwidth threshold to be set as 
#                              outbound committed_rate and burst
#
###############################################################################

configure_ipqos () {
    zonename="$1"
    zoneip="$2"
    ipbw="$3"
    opbw="$4"

    zonelog "Configuring IPQoS with ipbw: $ipbw opbw: $opbw"

    if [ $ipbw -ne 0 ]; then
        $IPQOSADM -u "$zonename" "scm.zone.inputbw" "$ipbw"
    fi

    if [ $opbw -ne 0 ]; then
        $IPQOSADM -u "$zonename" "scm.zone.outputbw" "$opbw"
    fi 

    zonelog "IPQoS Configuration was successful"
    
}

###############################################################################
#
# 	Main starts here
#
###############################################################################


if [ $# -lt 2 ]; then
	usage
	exit 1
fi

ZONENAME=""
ZONEPATH=""
ZONEIPADDR=""
ZONEHOSTNAME=""
ZONEPHYSICAL=""
ZONECPUSHARES=""
ZONEPOOL=""
ZONEOP=""
ZONEIPBW=""
ZONEOPBW=""
ZONEMAXSHARES=""
ZONEAUTOBOOT=""
EXTRAZONEATTR=""

BASEDIR=`pkgparam SUNWscma BASEDIR`
IPQOSADM="$BASEDIR/SUNWsymon/modules/sbin/ipqos-adm.sh"

while [ $# -gt 0 ]
do
    case $1 in
	-n) 
	     shift
	     ZONENAME=$1
	     shift
	     ;;
	-r)
	     shift
	     ZONEPATH=$1
	     shift
	     ;;
        -i)  
	     shift
	     ZONEIPADDR=$1
	      shift
	     ;;
	-h)
	     shift
	     ZONEHOSTNAME=$1
	     shift
	     ;;
	-e)
	     shift
	     ZONEPHYSICAL=$1
	     shift
	     ;;
	-p)
	     shift
	     ZONEPOOL=$1
	     shift
	     ;;
	-c)
	     shift
	     ZONECPUSHARES=$1
	     shift
	     ;;
	-o)
	     shift
	     ZONEOP=$1
	     shift
	     ;;
	-I)
	     shift
	     ZONEIPBW=$1
	     shift
	     ;;
	-O)
	     shift
	     ZONEOPBW=$1
	     shift
	     ;;
	-m)
	     shift
	     ZONEMAXSHARES=$1
	     shift
	     ;;
	-a)
	     shift
	     ZONEAUTOBOOT=$1
	     shift
	     ;;
	-E)
	     shift
	     EXTRAZONEATTR="$1"
	     shift
	     ;;
	-*)
	     usage
	     exit 1
	     ;;
	*)
	     usage
	     exit 1
	     ;;
    esac
done

#
# If the zonename not provided, exit
#
if [ -z "$ZONENAME" ]; then
   usage
   exit 1
fi

# configure the zone log file
configure_zonelog $ZONENAME

if [ "$ZONEOP" = "ready" ]; then
    create_ready_state_zone "$ZONENAME" "$ZONEPATH" "$ZONEIPADDR" "$ZONEPHYSICAL" "$ZONEPOOL" "$ZONECPUSHARES" "$ZONEMAXSHARES" "$ZONEAUTOBOOT" "$EXTRAZONEATTR"
else
    create_running_state_zone "$ZONENAME" "$ZONEPATH" "$ZONEIPADDR" "$ZONEPHYSICAL" "$ZONEPOOL" "$ZONECPUSHARES" "$ZONEMAXSHARES" "$ZONEAUTOBOOT" "$EXTRAZONEATTR"
fi

if [ $? -ne 0 ]; then
    exit 4    
fi

# When it reaches here, the zone must be ready or booted
configure_ipqos "$ZONENAME" "$ZONEIPADDR" "$ZONEIPBW" "$ZONEOPBW"

exit 0

