#!/bin/sh

#
# Copyright (c) 06/19/05 Sun Microsystems, Inc. All Rights Reserved.
#
# ident "@(#)prj_rctl_mod.sh 1.13     05/06/19 SMI"
#


#
# The return values or the exit values for all the scripts should 
# be of the following:
# exit_status	meaning
#	0	command successfull
#	1	Usage error
#	2 	Not a root to execute this script
#	3 	Command not found
#	4	Command failed for some reason
#

#
# Source the common commands and files script
#
SCRIPTDIR=`/usr/bin/dirname $0`
. ${SCRIPTDIR}/scm-agent-common.sh

#
# Take the backup of the /etc/project file and the set
# the trap handler.
#
setTrap

#
# Make sure this script is not run simultaneously to avoid
# /etc/project file corruption
#
LOCKFILE=/var/opt/SUNWsymon/cfg/prj_rctl_mod.lock

#
# make sure this is run as root
#
if [ "`$ID -un`" != "root" ] ; then
    $ECHO "You are not a root to execute this command\n" > "/dev/stderr"
    exit 2
fi

#
# Usage message
#
usage() {
   $ECHO "Usage: prj_rctl_mod.sh -z <zonename> -n projectname -r <resset> | -w <shares> | -m <memcap>" > "/dev/stderr"
   $ECHO "
      -z <zonename>	Name of the zone
      projectname	Name of the project to be modified
      -r <resset>	Name of the resource set
      -w <shares>	CPU Shares 
      -m <memcap>	CPU Shares 
"
}


#
# Function to acquire the lock on this script.
# If someone is executing this script, a lock is
# created and no other programs can execute this
# script until the previous program releases the lock.
# This is introduced to avoid corruption of the /etc/project
# file due to simultaneous execution of this script.
#
acquireLock() {
    #
    # wait for the lock to be released
    #
    while [ 1 ]
    do
        while [ -f $LOCKFILE ]
        do
	    sleep 1
        done

        $ECHO $$ > $LOCKFILE
        chmod 444 $LOCKFILE
        pid=`$HEAD -1 $LOCKFILE`

        if [ $pid -eq $$ ]; then
	    break;
        fi
    done
}

# 
# Function to release the lock acquired on this script
# Releasing the lock would allow another process to execute
# this script.
#
releaseLock() {
    $RM -rf $LOCKFILE
}

if [ $# -lt 4 ]; then 
    usage
    exit 1
fi

NO_CHANGE=-1

GIVENRESSET=""
GIVENSHARES=$NO_CHANGE
GIVENRCAPVAL=$NO_CHANGE
GIVENMAXSHMVAL=$NO_CHANGE
PROJNAME=""
ZONENAME=""

while [ $# -gt 0 ]
do
    case $1 in
	-z) 
	     shift
	     ZONENAME=$1
	     shift
	     ;;
	-n) 
	     shift
	     PROJNAME=$1
	     shift
	     ;;
	-r)
	     shift
	     GIVENRESSET=$1
	     shift
	     ;;
	-w)
	     shift
	     GIVENSHARES=$1
	     shift
	     ;;
	-m)
	     shift
	     GIVENRCAPVAL=$1
	     shift
	     ;;
	-s)
	     shift
	     GIVENMAXSHMVAL=$1
	     shift
	     ;;
	-*)
	     usage
	     exit 1
	     ;;
	*)
	     usage
	     exit 1
	     shift
	     ;;
    esac
done

if [ -z "$PROJNAME" ]; then
    usage
    exit 1
fi

if [ $ZONENAME != "global" ]; then
    ZONECMD="$ZLOGIN $ZONENAME"
    zonepath=`/usr/sbin/zoneadm list -v | grep $ZONENAME | awk '{print $4}'`
    PROJECTFILE="$zonepath/root/etc/project"
    SCMTMPPROJECTFILE="$zonepath/root/tmp/SunMCSCM/tmpProject"
    SCMTMPDIR="$zonepath/root/tmp/SunMCSCM"
else
    ZONECMD=""
fi

PROJECT_CPUSHARES="project.cpu-shares"
PROJECT_POOL="project.pool"
PROJECT_RCAP="rcap.max-rss"
PROJECT_MAXSHM="project.max-shm-memory"

#
# Acquire the lock, so that this script is not
# run simultaneously, corrupting the /etc/project
# file.
#
acquireLock

#
# Get the line having the given project name
#
LINE=`$AWK '$0 ~ /'\^$PROJNAME:'/' $PROJECTFILE` 

if [ -z "$LINE" ]; then
    $ECHO "the given project name does not exists in the project database\n" > "/dev/stderr"
    releaseLock
    exit 4
fi

#
# Get the projectname, projectid, comment, users, groups, and resource controls
#
PROJNAME=`$ECHO $LINE | $CUT -d: -f1`
PROJID=`$ECHO $LINE | $CUT -d: -f2`
COMMENT=`$ECHO $LINE | $CUT -d: -f3`
USERS=`$ECHO $LINE | $CUT -d: -f4`
GROUPS=`$ECHO $LINE | $CUT -d: -f5`
RCTLS=`$ECHO $LINE | $CUT -d: -f6`
#
# Form the new project string 
#
RESCONTROLS=""

#
# Get the number of resource controls 
#
FIELDS=`$ECHO $RCTLS | $AWK -F';' '{ print NF }'`
count=1

#
# Traverse each of the resource controls
#

NEWRCTLSTRING=""

pool_found=0
shares_found=0
rcap_found=0
maxshm_found=0

while [ $count -le $FIELDS ]
do
    RCTLNAMEVALUE=`$ECHO $RCTLS | $CUT -d';' -f$count`
    RCTLNAME=`$ECHO $RCTLNAMEVALUE | $CUT -d'=' -f1`
    RCTLVALUE=`$ECHO $RCTLNAMEVALUE | $CUT -d'=' -f2`

    if [ "$RCTLNAME" = "$PROJECT_POOL" ]; then
	pool_found=1
	if [ -n "$GIVENRESSET" ]; then
	    RCTLSTRING=`$ECHO $RCTLNAME=$GIVENRESSET`
	else
	    RCTLSTRING=$RCTLNAMEVALUE
	fi
    elif [ "$RCTLNAME" = "$PROJECT_CPUSHARES" ]; then
 	shares_found=1
        if [ $GIVENSHARES -ne $NO_CHANGE ]; then
	    #
	    # Set the given shares here. But first, remove the ( ) around the
	    # RCTLVALUE
	    #
	    RCTLVALUE=`$ECHO $RCTLVALUE | $AWK -F'(' '{ print $2 }'`
	    RCTLVALUE=`$ECHO $RCTLVALUE | $AWK -F')' '{ print $1 }'`
	    #
	    # RCTVALUE has the format - privileged,<shares>,action - now
	    #
	    PRIV=`$ECHO $RCTLVALUE | $CUT -d',' -f1`
	    PROJSHARES=`$ECHO $RCTLVALUE | $CUT -d',' -f2`
	    ACTION=`$ECHO $RCTLVALUE | $CUT -d',' -f3`

	    #
	    # Now form the new string with the given shares value.
	    # In future, if the privilege leve and action are mentioned,
	    # we have to add them to the below string
	    #
	    RCTLSTRING=`$ECHO "$RCTLNAME=($PRIV,$GIVENSHARES,$ACTION)"`
	else
	    RCTLSTRING=$RCTLNAMEVALUE
	fi
    elif [ "$RCTLNAME" = "$PROJECT_RCAP" ]; then
	rcap_found=1
        if [ $GIVENRCAPVAL -eq $NO_CHANGE ]; then  # user doesn't want to change
	    RCTLSTRING=$RCTLNAMEVALUE
        elif [ $GIVENRCAPVAL -eq 0 ]; then  # user wants to remove
            RCTLSTRING=""
	else                                # user wants to set
	    RCTLSTRING=`$ECHO "$RCTLNAME=$GIVENRCAPVAL"`
	fi
    elif [ "$RCTLNAME" = "$PROJECT_MAXSHM" ]; then
	maxshm_found=1
        if [ $GIVENMAXSHMVAL -eq $NO_CHANGE ]; then #User doesn't want to change
	    RCTLSTRING=$RCTLNAMEVALUE
        elif [ $GIVENMAXSHMVAL -eq 0 ]; then  # User wants to delete
            RCTLSTRING=""
        else                                  # User wants to set
	    #
	    # Set the given shm mem here. But first, remove the ( ) around the
	    # RCTLVALUE
	    #
	    RCTLVALUE=`$ECHO $RCTLVALUE | $AWK -F'(' '{ print $2 }'`
	    RCTLVALUE=`$ECHO $RCTLVALUE | $AWK -F')' '{ print $1 }'`
	    #
	    # RCTVALUE has the format - privileged,<mem>,action - now
	    #
	    PRIV=`$ECHO $RCTLVALUE | $CUT -d',' -f1`
	    OLDMAXSHM=`$ECHO $RCTLVALUE | $CUT -d',' -f2`
	    ACTION=`$ECHO $RCTLVALUE | $CUT -d',' -f3`

	    #
	    # Now form the new string with the given shm-memory value.
	    # In future, if the privilege leve and action are mentioned,
	    # we have to add them to the below string
	    #
	    RCTLSTRING=`$ECHO "$RCTLNAME=($PRIV,$GIVENMAXSHMVAL,$ACTION)"`
	fi
    else
	RCTLSTRING=$RCTLNAMEVALUE
    fi

    if [ -z "$NEWRCTLSTRING" ]; then 
	NEWRCTLSTRING=$RCTLSTRING
    elif [ "$RCTLSTRING" != "" ]; then
	NEWRCTLSTRING=`$ECHO "$NEWRCTLSTRING;$RCTLSTRING"`
    fi

    count=`eval expr $count + 1`

done

#
# If the specified resource controls, like, rcap.max-rss=<rcapvalue>
# or project.pool=<poolname> or project.cpu-shares=(privileged, <shares>, deny)
# were not found in this project entry, add them to the resource controls.
# pool_found, shares_found, and rcap_found will be 0 if not found, otherwise,
# will be 1. If they are 1, donot do anything, since, it is already taken
# care above.
#
if [ -n "$GIVENRESSET" ]; then
    if [ $pool_found -eq 0 ]; then
        if [ -n "$NEWRCTLSTRING" ]; then
	    NEWRCTLSTRING=`$ECHO "$NEWRCTLSTRING;project.pool=$GIVENRESSET"`
        else
	    NEWRCTLSTRING=`$ECHO "project.pool=$GIVENRESSET"`
        fi
    fi
fi

if [ $GIVENSHARES -ne $NO_CHANGE ]; then
    if [ $shares_found -eq 0 ]; then
        if [ -n "$NEWRCTLSTRING" ]; then
	    NEWRCTLSTRING=`$ECHO "$NEWRCTLSTRING;project.cpu-shares=(privileged,$GIVENSHARES,deny)"`
        else
	    NEWRCTLSTRING=`$ECHO "project.cpu-shares=(privileged,$GIVENSHARES,deny)"`
        fi
    fi
fi

if [ $GIVENRCAPVAL -gt 0 ]; then      # This coveres both NO_CHANGE and 0 cases
    if [ $rcap_found -eq 0 ]; then
        if [ -n "$NEWRCTLSTRING" ]; then
	    NEWRCTLSTRING=`$ECHO "$NEWRCTLSTRING;rcap.max-rss=$GIVENRCAPVAL"`
        else
	    NEWRCTLSTRING=`$ECHO "rcap.max-rss=$GIVENRCAPVAL"`
        fi
    fi
fi

if [ $GIVENMAXSHMVAL -gt 0 ]; then   # This covered both NO_CHANGE and 0 cases
    if [ $maxshm_found -eq 0 ]; then
        if [ -n "$NEWRCTLSTRING" ]; then
	    NEWRCTLSTRING=`$ECHO "$NEWRCTLSTRING;project.max-shm-memory=(privileged,$GIVENMAXSHMVAL,deny)"`
        else
	    NEWRCTLSTRING=`$ECHO "project.max-shm-memory=(privileged,$GIVENMAXSHMVAL,deny)"`
        fi
    fi
fi

#
# Finally construct the project entry 
#
NEWPROJENT=`$ECHO $PROJNAME:$PROJID:$COMMENT:$USERS:$GROUPS:$NEWRCTLSTRING`


#
# Get the line number in /etc/project on which the given project name existed
# We need this number, to make sure, we place this newly modified text on the same line
# in /etc/project file.
#
PROJENTLINENO=`$AWK '$0 ~ /'\^$PROJNAME:'/ { print NR }' $PROJECTFILE`
#
# Find number of lines in the /etc/project file
#
NUMLINES=`$WC -l $PROJECTFILE | $AWK '{ print $1 }'`
HEADLINES=`eval expr $PROJENTLINENO - 1`
TAILLINES=`eval expr $NUMLINES - $PROJENTLINENO`

#
# Now, replace the /etc/project file with the new project entry
#

if [ -d "$SCMTMPDIR" ]; then
    if [ -x "$SCMTMPPROJECTFILE" ]; then
        $RM -rf "$SCMTMPPROJECTFILE"
    fi
else
    $MKDIR "$SCMTMPDIR"
fi

$HEAD -$HEADLINES $PROJECTFILE > $SCMTMPPROJECTFILE
$ECHO $NEWPROJENT >> $SCMTMPPROJECTFILE
tail -$TAILLINES $PROJECTFILE >> $SCMTMPPROJECTFILE
$CP $SCMTMPPROJECTFILE $PROJECTFILE
$RM -rf $SCMTMPPROJECTFILE
$RM -rf $SCMTMPDIR
#
# Now, if we changed the project's poolname, we have bind the processes
# running in this project to the new pool
#
if [ -n "$GIVENRESSET" ]; then
    $ZONECMD $POOLBIND -p $GIVENRESSET -i projid $PROJNAME
fi

#
# If the project cpu-shares is changed, this should affect the running
# project also
#
if [ $GIVENSHARES -ne $NO_CHANGE ]; then
    $ZONECMD $PRCTL -n project.cpu-shares -v $GIVENSHARES -r -i project $PROJNAME
fi

if [ $GIVENMAXSHMVAL -gt 0 ]; then
    $ZONECMD $PRCTL -n project.max-shm-memory -v $GIVENMAXSHMVAL -r -i project $PROJNAME
fi

releaseLock
exit 0
