#
# Copyright (c) 06/19/05 Sun Microsystems, Inc. All Rights Reserved.
#
# ident	"@(#)scm-container-d.prc	1.74	05/06/19 SMI"
#

########################################################################
# Admin Procedures (Admin managed-object)
########################################################################

########################################################################
# Procedure: trapackCancelTrap 
# Description: The sending of module-refresh-trap periodically is cancelled
#              when there is a set on the trapAck property.
#
# Input : None
# Output: The timer to send the periodic module loaded trap is cancelled
########################################################################

proc trapackCancelTrap { } {
    set traptimer [ ilookup -d "" value traptimer ]

    if { $traptimer != "" } {
       cancelOneShot $traptimer
       undefine value traptimer
    }
}

########################################################################
# Procedure: trapackSendModuleLoadedTrap
# Description: Send the module refresh trap again
# Input : None
# Output: Module Loaded trap is sent
########################################################################

proc trapackSendModuleRefreshTrap { } {
    set timer [ ilookup -d "" value traptimer ]
    if { $timer != "" } {
       undefine value traptimer
    }

    #
    # Send the module refresh trap
    #
    setTrapInfo refreshOID

    #
    # Set the timer to 5 minutes
    #
    define value traptimer [ registerOneShot 300 \
		trapackSendModuleRefreshTrap ]
}


########################################################################
# Container related procedures (Containers) 
########################################################################

########################################################################
# Procedure: 
# Description: Saves persistent information to scm-container-d.dat file. 
#  This procedure uses a callback to ensure that the rowstatus 
#  node of a new created row is set to active before saving 
#  the data. Otherwise, the new created row will always have the 
#  rowstatus value 4 ( createAndGo ) in the saved file. 
#  The callback procedure ensures that the rowstatus is set 
#  to active before saving the data to disk
#
# Input:       None
# Output:      None
########################################################################
proc saveContainerData { } {
    set rstoe [ locate iso*ContEntry.rowstatus ]
    set flag 0

    toe_send $rstoe sliceforeach idx rowvalue data {
        if { $rowvalue == 4 } {
	    set flag 1
	    break
	}
    }

    if { $flag == 1 } {
	registerOneShot 0 "toe_send [ toe_self ] saveContainerData"
    } else {
	syncSlice data
    }
}


#######################################################################
# Procedure: addProjUsersGroupsComment
# Description: Adds users/groups/comment to a project entry
#
# Input: option ( -c or -g or -u ), where
#            -c - comment
#	     -g - group
#	     -u - user(s)
#        index - Index of the project row
#        value - Value of users/groups/comment
#
# output: None. The project entry is modified with the new value
#######################################################################
proc addProjUsersGroupsComment { option index value } {

    set defaultCont [ checkForDefaultContainer $index ]

    if { $defaultCont == 1 } {
	ddl print warning "Editing on default container is not allowed"
	ddl print info "addProjUsersGroupsComment $option $index $value"
	return 0
    }

    # 
    # add the users to the project entry
    # If the user does not exist, the projmod command returns an error, 
    # and the users are not added.
    #
    set rstoe [ locate iso*ContEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {
	    #
	    # This is the result of set on the users node only 
	    # ( not part of row creation )
	    # 

	    set projectname [ toe_send [ locate ContEntry.projectName ] \
		getValue $index ]

	    set zonename [ toe_send [ locate ContEntry.zoneName ] \
		getValue $index ]

            #
            # Now call the script
            #

	    set users ""
	    set groups ""
	    set comment ""
	    set input " "

	    if { $value != "" } {
		set input $value
	    }

	    switch -exact -- $option {
	        -c { set comment $input }
		-u { set users $input }
		-g { set groups $input }
	    }

	    set result [ executeScript "prj_mod.sh" $option $zonename $projectname \
			 "" $comment $users $groups "" "" "" "" "" ]
	    if { $result == 0 } {
		ddl print error "projmod: could not update $value"
		return 0
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    refreshNupdateContainer $index
	}
    }

    return $value
}

#######################################################################
# Procedure: getCpusizeForPool
# Description: Get the cpu size of the given pool
#
# Input: pool - Poolname
# Output: cpusize of the given poolname
#
#######################################################################
proc getCpusizeForPool { pool } {

    set psetentrytoe [ locate iso*PsrSetEntry ]
    set resentrytoe [ locate iso*PoolEntry ]

    if { [ catch { toe_send $resentrytoe getRowValue \
		     psetName "$pool" } psetname ] } {
	#
	# Processor set name does not exists for this pool
	# Wrong pool definition
	#
	return 0
    }

    if { [ catch { toe_send $psetentrytoe getRowValue cpuSize "$psetname" } \
		cpusize ] } {
	#
	# The processor set is not configured properly
	#
	return 0
    }

    return $cpusize
}


#######################################################################
# Procedure: getCpuSharesForPool
# Description: Get the Total / Max cpu shares of the given pool
#
# Input: pool - Poolname
# Output: Max cpu shares of the given poolname
#
#######################################################################
proc getCpuSharesForPool { pool } {

    set resentrytoe [ locate iso*PoolEntry ]

    if { [ catch { toe_send $resentrytoe getRowValue \
		     psetName "$pool" } psetname ] } {
	#
	# Processor set name does not exists for this pool
	# Wrong pool definition
	#
	return 0
    }

    if { [ catch { toe_send $resentrytoe getRowValue maxCPUShares "$pool" } \
		cpuShares ] } {
	#
	# The processor set is not configured properly
	# Return the default value
        #
	return 100
    }

    return $cpuShares
}

#######################################################################
# Procedure: getCpuSharesForZone
# Description: Get the Total / Max cpu shares of the given zone
#
# Input: zone - zonename
# Output: Max cpu shares of the given zonename
#
#######################################################################
proc getCpuSharesForZone { zone } {

    set zoneentrytoe [ locate iso*ZoneEntry ]

    if { [ catch { toe_send $zoneentrytoe getRowValue maxZoneCPUShares "$zone" } \
		cpuShares ] } {
	#
	# The max shares are not configured properly
	# Return the default value
        #
	return 100
    }

    return $cpuShares
}

#######################################################################
# Procedure: updateCpuReservation
# Description: Update the CPU reservation for the container
#     The updating procedure is as follows:
#     a. If the sum of all the existing containers and the current 
#        specified value exceeds the max cpu shares of the resource 
#        pool -> return an error
#     b. Allocate the specified cpu shares value
# 
# Input: index - Index of the container row
#        value - New Cpu reservation in terms of shares
# Output: Success(1)/Failure(0) 
#######################################################################

proc updateCpuReservation { index value username } {


    set defaultCont [ checkForDefaultContainer $index ]

    set curruser [ checkProjAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "updateCpuReservation: Invalid access. Permission denied for user to complete this operation"
        return 0
    }

    if { $defaultCont == 1 } {
	ddl print warning "Editing on default container is not allowed"
	ddl print info "updateCpuReservation $index $value"
	return 0
    }

    set rstoe [ locate ContEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {

            if { [ sysinfo osversion ] == "5.10" } {
	        set zone [ toe_send [ locate iso*ContEntry.zoneName ] \
			getValue $index ]
                set parent $zone
	        set cpuShares [ getCpuSharesForZone $zone ]
            } else {
	        set pool [ toe_send [ locate iso*ContEntry.poolName ] \
			getValue $index ]
                set parent $pool
	        set cpuShares [ getCpuSharesForPool $pool ]
            }

	    if { $value > $cpuShares } {
	        ddl print error "The shares are greater than max pool shares"
		return 0
	    }

	    #
	    # Now, visit all the containers having this pool name
	    # and find out the target reservation alotted to them. If the
	    # current value does not fit into the size of the pool name,
	    # then return an error
	    #

	    set totalShares $value
            if { [ sysinfo osversion ] == "5.10" } {
	        set parenttoe [ locate iso*ContEntry.zoneName ]
            } else {
	        set parenttoe [ locate iso*ContEntry.poolName ]
            }
	    set cpurestoe [ locate iso*ContEntry.cpuRes ]

	    toe_send $parenttoe sliceforeach key tmpparent data {
		if { $key == $index } {
		    continue
		}

		if { $tmpparent == $parent } {
		    set cpures [ toe_send $cpurestoe getValue $key ]
	            set totalShares [ expr $totalShares + $cpures ]
		}
	    }

	    if { $totalShares > $cpuShares } {
		#
		# The calculated shares exceed the max cpu shares of the
		# pool or zone. Hence, we cannot set it.
		#
		ddl print error "udpateCpuReservation: cannot reserve the cpu"
		ddl print error "udpateCpuReservation: Not enough CPU Shares in pool or zone"
		return 0
	    }

	    #
	    # Now, the specified shares can be allocated.
	    # Try allocating it.
	    #

            set shares [ expr int ($value) ]
	    set projectname [ toe_send [ locate iso*ContEntry.projectName \
				] getValue $index ]

	    set zonename [ toe_send [ locate iso*ContEntry.zoneName \
				] getValue $index ]


            # For Solaris 8, lnode corresponding to project 
            # is updated for cpu.shares
            if { [ sysinfo osversion ] == "5.8" } {
	       set list [ split $projectname "." ]
	       lextract $list 1 username

	       set result [ executeScript "lnode_functions.sh" -c "" $username \
			"" "" "" "" "" $shares "" "" "" ] 
				
		if { $result == 5 } {
		    #
		    # The user does not exists, but the /etc/project entry exists.
		    # Continue with deleting the /etc/project entry
		    #
		    ddl print warning "updateCpuReservation: User does not exists"
		} else {
		    #
		    # if the error value is other than 5, then something wrong
		    # happened, and donot continue further
		    #
		    if { $result != 0 } {
		        ddl print error "updateCpuReservation: update CPU error"
	                return 0
		    }
		}
            }

            #
            # Now call the script 
            #
	    set result [ executeScript "prj_rctl_mod.sh" -w $zonename $projectname "" \
				"" "" "" "" $shares "" "" "" ] 
	    if { $result == 0 } {
		ddl print error "updateCpuReservation: update CPU error"
		return 0
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    refreshNupdateContainer $index
        }
    }

    return 1
}

#######################################################################
# Procedure: updatePoolName
# Description: Checks whether the new pool can accomodate 
#   current container's cpu requirements, and updates the container 
#   pool name. 
#        
# Input: index - Container row index
#        value - New pool name
# Output: Success(1)/Failure(0).  
#######################################################################

proc updatePoolName { index value username } {


    set curruser [ checkProjAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "updatePoolName: Invalid access. Permission denied for user to complete this operation"
        return 0
    }

    # The following validation is required but should be placed 
    # inside a condition that checks if this is an existing container row.
    #if { [ sysinfo osversion ] == "5.10" } {
	#ddl print warning "Changing a pool of container inside zone is not allowed"
	#ddl print info "updatePoolName $index $value"
	#return 0
    #}

    set defaultCont [ checkForDefaultContainer $index ]

    if { $defaultCont == 1 } {
	ddl print warning "Editing on default container is not allowed"
	ddl print info "updatePoolName $index $value"
	return 0
    }

    #
    # Check whether the given pool exists either for
    # container creation or for container poolname modification
    #
    set result [ checkPoolExists $value ]

    if { $result == 0 } {
	return 0
    }

    set rstoe [ locate iso*ContEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {

	    #
	    # This is a preset operation, and the getRowValue 
	    # should give you old value. The new value is in the 
	    # value argument.
	    #
	    set oldpool [ toe_send [ locate iso*ContEntry.poolName ] \
				getValue $index ]

	    if { $oldpool == $value } {
		return $value
	    }

	    #
	    # Before we update the pool, make sure that, 
            # we can accomodate the current container's cpu reservation 
	    # in the newly specified pool.
	    #

	    set pooltoe [ locate iso*ContEntry.poolName ]
	    set cputoe [ locate iso*ContEntry.cpuRes ]

	    #
	    # Get the current container's cpu reservation
	    #
	    set mincpu [ toe_send $cputoe getValue $index ]

            set totalShares $mincpu
	    set cpuShares [ getCpuSharesForPool $value ]

	    toe_send $pooltoe sliceforeach key tmpPool data {
	        if { $tmpPool == $value } {
		    set tmpcpuval [ toe_send $cputoe getValue $key ]
	            set totalShares [ expr $totalShares + $tmpcpuval ]
	        }
            }

            if { $totalShares > $cpuShares } {
	        ddl print error "udpatePoolName: cannot accomodate new cpu reservation"
		ddl print error "udpatePoolName: Not enough CPU Shares in pool"
		return 0
            }

            #
            # Now, we have found that, the cpu reservation can be 
	    # given to this container Calculate the cpu-shares for 
	    # this container
            #

            set cpushares [ expr int ($mincpu) ]
	    set projectname [ toe_send [ \
		  locate iso*ContEntry.projectName ] getValue $index ]

	    set zonename [ toe_send [ \
		  locate iso*ContEntry.zoneName ] getValue $index ]

	    set result [ executeScript "prj_rctl_mod.sh" -w $zonename $projectname "" \
				"" "" "" $value $cpushares "" "" "" ] 
	    if { $result == 0 } {
	        ddl print error "updatePoolName: Could not update the pool"
		return 0
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    refreshNupdateContainer $index
        }
    }
    return $value
}


#######################################################################
# Procedure: updateMemoryCap
# Description: updates memory cap (rcap) value for the given container
#
# Input: index - container row index
#        value - New memory cap (rcap.max-rss) value
# Output: Success(1)/Failure(0)
#######################################################################

proc updateMemoryCap { index value username } {

    set curruser [ checkProjAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "updateMemoryCap: Invalid access. Permission denied for user to complete this operation"
        return 0
    }

    set defaultCont [ checkForDefaultContainer $index ]

    if { $defaultCont == 1 } {
	ddl print warning "Editing on default container is not allowed"
	ddl print info "udpateMemoryCap $index $value"
	return 0
    }

    set rstoe [ locate ContEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {
	    set projectname [ toe_send [ \
		  locate iso*ContEntry.projectName ] getValue $index ]

	    set zonename [ toe_send [ \
		  locate iso*ContEntry.zoneName ] getValue $index ]

	    #
	    # convert the given value to Mega Bytes
	    #
	    if { $value != "" && $value != 0 } {
                set memcap [ convertMB2Bytes $value ]
	    } else {
		set memcap 0
	    }

	    set result [ executeScript "prj_rctl_mod.sh" -m $zonename $projectname "" \
				"" "" "" "" $memcap "" "" "" ] 
	    if { $result == 0 } {
		ddl print error "updateMemoryCap: Could not update rcap"
		return 0
	    }

	    #
	    # Now enforce the memory cap, by sending a SIGHUP signal to
	    # rcapd daemon. 
	    #
            set result [ enforceMemoryCap $zonename ]

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    refreshNupdateContainer $index
        }
    }

    return 1
}

#######################################################################
# Procedure: updateMaxSHMMemory
# Description: updates shared memory allocation the given container
#
# Input: index - container row index
#        value - New project.max-shm-memory value
# Output: Success(1)/Failure(0)
#######################################################################

proc updateMaxSHMMemory { index value username } {

    set curruser [ checkProjAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "updateMaxSHMMemory: Invalid access. Permission denied for user to complete this operation"
        return 0
    }

    set defaultCont [ checkForDefaultContainer $index ]

    if { $defaultCont == 1 } {
	ddl print warning "Editing on default container is not allowed"
	return 0
    }
    if { [ sysinfo osversion ] != "5.10" } {
	ddl print warning "Not applicable for solaris 9 or solaris 8"
	return 0
    }

    set rstoe [ locate ContEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {
	    set projectname [ toe_send [ \
		  locate iso*ContEntry.projectName ] getValue $index ]

	    set zonename [ toe_send [ \
		  locate iso*ContEntry.zoneName ] getValue $index ]

	    #
	    # convert the given value to Mega Bytes
	    #
	    if { $value != "" && $value != 0 } {
                set maxshm [ convertMB2Bytes $value ]
	    } else {
		set maxshm 0
	    }

	    set result [ executeScript "prj_rctl_mod.sh" -s $zonename $projectname "" \
				"" "" "" "" $maxshm "" "" "" ] 
	    if { $result == 0 } {
		ddl print error "updateMaxSHMMemory: Could not update shared memory"
		return 0
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    refreshNupdateContainer $index
        }
    }

    return 1
}

#######################################################################
# Procedure: checkPoolExists
# Description: During container creation, make sure the pool/resource 
#    set exists
# Input : pool - Pool Name
# Output: 0 if pool not found, otherwise 1
#
#######################################################################

proc checkPoolExists { pool } {
    set pooltoe [ locate iso*PoolEntry.poolName ] 

    if { [ catch { toe_send $pooltoe getRowIndex "$pool" } idx ] } {
	#
	# The psetname is not found
	#
	ddl print error "checkPoolExistence: pool $pool does not exists"
	return 0
    }

    return 1
}

#######################################################################
# Procedure: findDefaultPool
# Description: Find the default pool of the system
#
# Input: None
# Output: default pool name
#######################################################################

proc findDefaultPool { } {
    set defaulttoe [ locate iso*PoolEntry.default ]
    set pooltoe [ locate iso*PoolEntry.poolName ]
    set defaultpool ""
    
    toe_send $defaulttoe sliceforeach key default data {
	if { $default == 1 } {
	    set defaultpool [ toe_send $pooltoe getValue $key ]
	    break
	}
    }

    return $defaultpool
}

#######################################################################
# Procedure: loadContainerData
# Description: Gets the projects on the system
#   The logic here is, initially, when we load the module, it is possible 
#   that no containers existed earlier, but some (default) project might 
#   exist in /etc/project database. We have to load them as part of the 
#   initial configuration load. But, for such projects, the container
#   id, container name, the cpu/memory reservations 
#   may not be present. Under such circumstances, we make the 
#   following assumptions:
#
#      container name = project name
#      container id = project id
#      cpu reservation = 1 share ( the system default )
#       	memory cap = 0 ( entire system memory  available )
#
#   The rest of the items would remain as is, and if they are empty, 
#   they would remain empty.
#
# Input: None
# Output: list of all the containers/project with their statistics. For
#         refresh operation
#######################################################################

proc loadContainerData { } {

    set contentrytoe [ locate ContEntry ]
    set projects [ getSysProjects ]
    set result ""
    set depth [ toe_send $contentrytoe getTableDepth ]
    set rowIdx 1

    #
    # Get the rcapstat information for all the projects.
    # The format of the rcapstat result is as follows:
    # { { projid memcap virtualmem rss } [ { projid memcap virtualmem rss } ] }
    # for all the projects. The list of lists is sorted on the basis of
    # projid, for faster search.
    #
    if { [ catch { getMemStat } rcapstat ] } {
	set rcapstat {}
    }

    #
    # Find the default pool
    #
    set defaultPool [ findDefaultPool ]

    foreach project $projects {

	set details [ getContDetails $project ]

	#
	# The details we get are of the form:
	# projectname projectid, pool, shares, users, groups, comment
	# If the pool/users/groups/comment not found, "" is returned.
	# If there are no shares, then -1 is returned.
	#
	lextract $details 1 projid 2 pool 3 shares 4 memcap 5 users \
		6 groups 7 comment 8 procsRunning 9 maxshm

	#
	# From the shares, find the cpuRes
	#
	if { $pool == "" } {
	    set pool $defaultPool
	} else {
	    set poolexists [  checkPoolExists $pool ]
	    if { $poolexists == 0 } {
		set pool $defaultPool
	    }
	}

	if { $pool == "" } {
	    set poolsize 0
	    set maxShares 0
	} else {
	    set poolsize [ getCpusizeForPool $pool ]
	    set maxShares [ getCpuSharesForPool $pool ]
	}

	if { $shares == -1 } {
	    set shares 1
	}

	set cpures [ expr int ($shares) ]

	#
	# Now find the current usage. The value returned is in
        # % (percentage)
	#

	if { [ catch { getContResourceUsage $projid } cpuMemUsage ] } {
	    set cpuused 0
	    set cpuPctUsed 0.0
	    set cpuROI 0.0
	    set contMemRSS "0"
	} else {
	    lextract $cpuMemUsage 0 cpuPctUsed 1 contMemRSS
	    #
	    # The usage returned by the getContResourceUsage is
	    # in percentage and for the entire system.
	    # We need to convert it to per pool basis, in terms of
	    # actual cpus.
	    #
	    set onlinecpucount [ getOnlineCpuCount ]
	    set cpuused [ expr ( $cpuPctUsed * $onlinecpucount ) / 100 ]

            # We need to calculate return on investment
            # i.e. cpu used against cpu reserved.
            # The shares for absolute 1 CPU is maxShares / poolsize
	    if { $poolsize > 0 } {
                set onecpu [ expr $maxShares / $poolsize ]
            } else {
                set onecpu 0
            }

            # cpu shares used is
            set cpusharesused [ expr $cpuused * $onecpu ]

	    #
	    # It's possible that, the user has made the shares to 0 by
	    # editing the /etc/project entry, and that would cause 
	    # divide by zero error for the below operation.
	    #
	    if { $cpures > 0.0 } {
	        set cpuROI [ expr ( $cpusharesused * 100.0 ) / $cpures ]
	    } else {
		set cpuROI 0.0
	    }
        }

	#
	# It's time to find the values for container name, 
	# container id, etc. as described in the description above.
	#

	set rowname [ toe_send $contentrytoe getRowName $rowIdx ]

	if { $depth >= 1 && $rowIdx <= $depth } {

	    #
	    # Get the existing values 
	    #

	    set contName [ toe_send $contentrytoe getRowValue contName \
				$rowname ]
	    set contID [ toe_send $contentrytoe getRowValue contID $rowname ]
	    set matchExpr [ toe_send $contentrytoe getRowValue matchexpr \
				$rowname ]
	    set primaryUsers [ toe_send $contentrytoe getRowValue primaryUsers \
				$rowname ]
	} else {
	    #
	    # If this condition is true, then, either we have no 
	    # pre-existing containers or some projects are added on the 
	    # command line. In such cases, make the container name, 
	    # container id, and other values, as described above
	    #


	    set contID ""
	    set matchExpr ""
	    set primaryUsers ""
	    set contName ""
	}

	#
	# The default projects, system, default, user.root,
	# noproject, and group.staff have been given default
	# container names by the service layer. The following
	# logic assigns the default container name for the 
	# default projects. If the project name is none of the
	# default projects, then the container name would be
	# project name.
	#

	if { $project == "system" } {
	    set defaultContainer 1
	    set contName "System Processes"
	} elseif { $project == "default" } {
	    set defaultContainer 1
	    set contName "Default"
	} elseif { $project == "user.root" } {
	    set defaultContainer 1
	    set contName "Root User"
	} elseif { $project == "group.staff" } {
	    set defaultContainer 1
	    set contName "Users with Group Staff"
	} elseif { $project == "noproject" } {
	    set defaultContainer 1
	    set contName "Processes with No Project"
	} else {
	   set defaultContainer 0
	   if { $contName == "" } {
	       set contName $project
	   }
	}

	# 
	# Get the system memory and convert it to MB
	#
	set sysMemory [ expr [ sysinfo hwmemory ] / 1024 ]
	#
	# Get the rcapstat information from the list for this project id
	#
	set pos [ binarysearch $rcapstat $projid ]
	if { $pos == -1 } {
	    set virtualmem "0"
	    #
	    # If the rcapstat info is not available for this project, the rcap
	    # value is not specified for this container, hence, consider the RSS 
	    # size of the project collected during the /proc scanning
	    #
	    set rss $contMemRSS
	} else {
	    lextract $rcapstat $pos projrcapstat
	    lextract $projrcapstat 1 memcap 2 virtualmem 3 rss
	}
	#
	# Calculate the percentage of system memory used by this container
	#

	#
	# Now calculate the percentage value. The rss value
	# is made float to capture less than 1% of the memory. 
	#
	set memPctUsed [ expr ( $rss * 100.0 ) / $sysMemory ]
        if { [ isZero64 $memcap ] > 0 } {
	    set memROI [ expr ( $rss * 100.0 ) / $memcap ]
	} else {
	    set memROI 0.0
	}

	incr rowIdx

	lappend result $contID "$contName" $projid $project "$primaryUsers" \
		$pool $poolsize $cpures $cpuused $cpuPctUsed $cpuROI $memcap $virtualmem $rss \
		$memPctUsed $memROI "$users" "$groups" "$comment" "$matchExpr" $procsRunning $defaultContainer $maxshm
    }
    
    updateProjectFileTimestamp 
    return $result
}

#######################################################################
# Procedure: deleteUserDefaultProject
# Description: Removes the users' default project name from 
# /etc/user_attr file.
# Input : projectname, userslist
# Output: none
#
#######################################################################
proc deleteUserDefaultProject { zonename projname userlist } {
    set result [ executeScript "prj_userattr.sh" -d $zonename $projname "" "" \
			$userlist "" "" "" "" "" "" ]
}

#######################################################################
# Procedure: deleteContainer
# Description: Removes the container/project from the system
# Input : Row Index
# Output: Container is removed from the system
#
#######################################################################

proc deleteContainer { index } {

    set projnametoe [ locate iso*ContEntry.projectName ]
    set projidtoe [ locate iso*ContEntry.projectID ]

    set projname [ toe_send $projnametoe getValue $index ]
    set zonename [ toe_send [ locate iso*ContEntry.zoneName ] getValue $index ]
    set projid [ toe_send $projidtoe getValue $index ]
    set userlist [ toe_send [ locate iso*ContEntry.primaryUsers ] \
	getValue $index ]

    # deleting lnode for solaris 8
    if { [ sysinfo osversion ] == "5.8" } {
	set list [ split $projname "." ]
	if { [ catch { lextract $list 1 username } ] } {
	    set username $projname
	} 

	set result [ executeScript "lnode_functions.sh" -d "" $username "" \
				"" "" "" "" "" "" "" "" ] 
	if { $result == 6 } {
	    #
	    # This may be due to command line creation of a project entry without
	    # the corresponding lnode creation. Just continue to delete the
	    # /etc/project entry
	    #
	    ddl print warning "deleteContainer: lnode does not exists for this container"
	} elseif { $result != 0 } {
	    ddl print warning "deleteContainer: Container cannot be deleted"
	    return 0
        }
    }

    set result [ executeScript "prj_del.sh" "" $zonename $projname $projid "" "" "" \
			"" "" "" "" "" ]
    if { $result == 0 } {
	ddl print warning "deleteContainer: Container cannot be deleted"
	return 0
    }

    if { $userlist != "" } {
        #
        # delete the user's default project after the row is
        # deleted.
        #
        registerOneShot 2 "deleteUserDefaultProject $zonename $projname $userlist"
    }

    set tblObj [ locate ContEntry ]

    if { $tblObj != "" } {
        toe_send $tblObj cleanupRow $index CLEAR_PARMS
    }

    return $result
}

#######################################################################
#
# Procedure: createOrDeleteContainer
# Description: As part of the pre-validate-set operation on the rowstatus
#   node, we either create or delete the container based on the rowstatus
#   value.
#
# Input: index - Container row index
#        value - Rowstatus value to confirm container creation/deletion
# output: Success(1)/Failure(0)
#
#######################################################################
proc createOrDeleteContainer { index value rowname username } {

    set curruser [ checkProjAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "createOrDeleteContainer: Invalid access. Permission denied for user to complete this operation"
        return 0
    }


    ddl print info "entered createOrDeleteContainer: $index $value $rowname"

    if { $value == 4 || $value == 5 } {
	#
	# Create container
	#
	set result [ createContainer $index $rowname $username ]
    } else {
	if { $value == 6 } {
	    #
	    # Delete container
	    #
	    set result [ deleteContainer $index ]
	} else {
	    ddl print warning "not a valid operation on this row"
	    return 0
	}
    }

    updateProjectFileTimestamp 

    if { $result != 0 } {
        return $value
    } else {
	return 0
    }
}


########################################################################## 
# Procedure: createContainer 
# Description: Create a project entry on the system with given resource
#   controls.
# 
# Input: index - Container row index
#        rowname - Index of the new row
# Output: Success(1) - Container creation succeeded
#         Failure(0) - to fail the row creation
########################################################################## 
proc createContainer { index rowname username } {

    ddl print info "create container : $rowname"
    set indices [ split $rowname , ]
    lextract $indices 0 zonename 1 projectid
    ddl print info "zonename $zonename projectid $projectid"

    if { [ catch { ilookup resourcesetname $index } pool ] } {
	ddl print error "createContainer: resourcesetname not specified"
	return 0
    }

    if { [ catch { ilookup cpuresource $index } mincpu ] } {
	ddl print error "createContainer: CPU reservation not specified"
	return 0
    }

    if { [ catch { ilookup primaryusers $index } primaryUsers ] } {
	set primaryUsers ""
    } else {
	undefine primaryusers $index
    }

    ddl print info "undefine slices"
    undefine resourcesetname $index
    undefine cpuresource $index

    set hostresentrytoe [ locate iso*PoolEntry ]
    set psrsetentrytoe [ locate iso*PsrSetEntry ]
    set zoneentrytoe [ locate iso*ZoneEntry ]

    #
    # Get the processor set name associated with this pool
    #
    if { [ catch { toe_send $hostresentrytoe \
		getRowValue psetName "$pool" } psetname ] } {
	ddl print error "processor set not found for pool"
        return 0
    }

    #
    # Get the cpusize of the processor set associated with the given pool
    #
    if { [ catch { toe_send $psrsetentrytoe \
		getRowValue cpuSize "$psetname" } cpusize ] } {
	ddl print error "processor set has no cpus"
	return 0
    }

    #
    # Get the Max CPU shares for this pool or zone
    # based on the OS Version.
    # For S10 container, the reservation should be compared with 
    # zone max shares
    # for s9 and s8, it should be compared with pool max shares
    #
    if { [ sysinfo osversion ] == "5.10" } {
        set maxShares [ getCpuSharesForZone $zonename ]
    } else {
        set maxShares [ getCpuSharesForPool $pool ]
    }

    #
    # Check, whether the cpusize of the processor set is zero.
    #

    ddl print info "check for cpu size"
    if { $cpusize <= 0 } {
	ddl print warning "Resource Set has zero cpus - Can't Create Container"
	return 0
    }

    #
    # Now, check, if the given minimum value is greater than the 
    # maxShares of the pool or zone
    #
    if { $mincpu > $maxShares } {
	ddl print error "Given cpu value exceeds the pool max shares"
	return 0
    }

    #
    # Now, find out the minimum cpu value for all the containers 
    # having this pool or zone. If the total minimum cpu value is 
    # greater than the maxshares of the pool or zone
    # then, we cannot create a container with this value.
    #
    # For s10, parent node is zone
    # For s9 and 8, parent is pool
    #

    set totalCpus $mincpu
    if { [ sysinfo osversion ] == "5.10" } {
        set parenttoe [ locate iso*ContEntry.zoneName ]
        set parent $zonename
    } else {
        set parenttoe [ locate iso*ContEntry.poolName ]
        set parent $pool
    }
    set cpurestoe [ locate iso*ContEntry.cpuRes ]

    toe_send $parenttoe sliceforeach key tmpParent data {
	if { $key == $index } {
	    continue
	}

	if { $tmpParent == $parent } {
	    set mincpuval [ toe_send $cpurestoe getValue $key ] 
	    set totalCpus [ expr $totalCpus + $mincpuval ]
	}
    }

    if { $totalCpus > $maxShares } {
	ddl print error "Cannot allocate cpu for this container"
	return 0
    }

    #
    # Now, we have found that, the minimum cpu value can be given to 
    # this container. Calculate the cpu-shares for this container
    #

    set cpushares [ expr int ($mincpu) ]

    #
    # Get other values, like, project-id, project-name (container name), 
    # users, groups, comment, etc
    #
    set projectname [ ilookup projectname $index ]
    undefine projectname $index
    #set projectid $rowname

    if { [ catch { ilookup containerid $index } contid ] } {
	set contid $projectid
    } else {
	undefine containerid $index
    }

    #
    # Background: container name appears as project name
    # on the console.
    # Reason: Container name does not take part in project
    # creation on the system. It is maintained for the
    # service layer purposes, hence, this value was not caught
    # during the validate set operation, thinking that, by 
    # default the agent would set this value to the given value.
    # But, this was not happening, and hence it was always the
    # project name according to the logic in the loadContainerData.
    # To fix this, we need to define a slice, even though we have
    # no use of this slice. Defining the slice preserves the value
    # being set. We undefine this slice here. 
    #
    if { ! [ catch { ilookup containername $index } contName ] } {
	undefine containername $index
    }

    if { [ catch { ilookup memorycap $index } memcap ] } {
	set memcap 0
    } else {
        #
        # The memcap is specified in terms of the Mega Bytes.
        # So, convert the Mega Bytes to bytes
        #
        set memcap [ convertMB2Bytes $memcap ]
	undefine memorycap $index
    }

    if { [ sysinfo osversion ] != "5.10" } {
        set maxshm 0
    } else {

        if { [ catch { ilookup maxshmmemory $index } maxshm ] } {
	    set maxshm 0
        } else {
            #
            # The memcap is specified in terms of the Mega Bytes.
            # So, convert the Mega Bytes to bytes
            #
            set maxshm [ convertMB2Bytes $maxshm ]
	    undefine maxshmmemory $index
        }
    }

    if { ! [ catch { ilookup comment $index } comment ] } {
	undefine comment $index
    } else {
	set comment ""
    }

    if { ! [ catch { ilookup users $index } users ] } {
	undefine users $index
    } else {
	set users ""
    }

    if { ! [ catch { ilookup groups $index } groups ] } {
	undefine groups $index
    } else {
	set groups ""
    }

    #
    # Construct the resource control strings and fill in the values
    #
    ddl print info "construct resource controls"
    set resctls "project.pool=$pool;project.cpu-shares=(privileged,$cpushares,none)"

    if { [ isZero64 $memcap ] > 0 } {
        append resctls ";rcap.max-rss=$memcap"
    }

    if { [ isZero64 $maxshm ] > 0 } {
        append resctls ";project.max-shm-memory=(privileged,$maxshm,deny)"
    }

    # create lnode for container if Solaris 8
    if { [ sysinfo osversion ] == "5.8" } {
	#
	# The given projectname is the actual username of the form
	# user.<username>. Extract the username from the projectname
	# before processing it.
	#
	set list [ split $projectname "." ]
        lextract $list 1 username 

	if { ! [ checkUserExists $username ] } {
            # cannot proceed with invalid user
	    ddl print error "user $username does not exist"
	    ddl print error "Could not create container"
            return 0
        }

	set result [ executeScript "lnode_functions.sh" -c "" $username "" \
				"" "" "" "" $cpushares "" "" "" ] 
	if { $result != 0 } {
	    if { $result == 5 } {
		ddl print error "createContainer: User does not exists"
	    }

	    ddl print error "Could not create container"
	    return 0
        }
    } 

    ddl print info "executeScript for prj_ad.sh"
    set result [ executeScript "prj_add.sh" "" $zonename $projectname $projectid \
		 $comment $users $groups "" $resctls "" "" "" ]
    if { $result == 0 } {
	ddl print error "Could not create container"
	return 0
    }

    #
    # For memory cap enforcement, we need to send a signal to the rcapd
    # daemon to use the recently set value for memcap.
    #
    ddl print info "check for memcap isZero64 calling"
    if { [ isZero64 $memcap ] > 0 } {
        set result [ enforceMemoryCap $zonename ]
    }

    #
    # Now, see, if the primary users are mentioned, then we need to
    # make the project name as the default project for these users.
    #
    ddl print info " Now check for primary users "
    if { $primaryUsers != "" } {
	#
	# set the users default project immediately after the container
	# is created. 
	#
	ddl print info "createContainer: primary users = $primaryUsers"
	registerOneShot 5 "setUserDefaultProject $index $primaryUsers 1"
    }

    #
    # If matchexpr is specified, we need to move the processes
    # matching this matchexpr into this container. 
    #
    ddl print info "Now check for match expression"
    if { ! [ catch { ilookup projmatchexpr $index } matchexpr ] } {
	ddl print info "createContainer: MATCH EXPR SPECIFIED : $matchexpr"
	undefine projmatchexpr $index

	#
	# Move the processes after the container is created. 
	#
	registerOneShot 5 "moveProcsToContainer $index $rowname $matchexpr 1 $username"
    }

    ddl print info "exiting createContainer"

    return 1
}

######################################################################### 
# Procedure: checkContainerRowStatus
# Description: When we load the initial configuration, ie., the existing
#   projects/containers, the rowstatus node for these containers is not
#   set, and the value in the rowstatus node is empty, though we have some
#   default active containers. This procedure, checks to see, if there are
#   any such active containers, and sets their rowstatus value to active.
#   If we donot do this operation, when the user creates a new container or
#   edits the existing default ones, we see inconsistency failure for the
#   rowstatus node.
# 
# Input: None
# Output: the rowstatus of all the active containers is made active
######################################################################### 
proc checkContainerRowStatus { } {
    set tDepth [ toe_send [ locate ContEntry ] getTableDepth ]
    set rstoe [ locate ContEntry.rowstatus ]

    for { set index 1 } { $index <= $tDepth } { incr index } {

        if { [ toe_csend val $rstoe getValue $index ] } {
	    #
	    # The rowstatus is not set for this row
	    # Set it to active
	    #
	    toe_send $rstoe setValue $index 1
        }
    }

    #
    # as a result of changing the system configuration, like,
    # deleting a project entry in the /etc/project database,
    # the row in this table may not appear, but, the rowstatus
    # would still appear as 1, so, clear them too
    #
    while { 1 } {
	if { [ catch { toe_send $rstoe getValue $index } val ] } {
	    break
	} else {
	    #
	    # clear the rowstatus node
	    #
	    toe_send $rstoe setValue $index 6

	    incr index
	}
    }
}

#########################################################################
# Procedure: defineSliceForNewRow
# Description: Defines the given { slice index value } only for the new
#    rows
#    
# Input: slice - slice to be defined
#        index - key for the slice
#        value - value for the slice
# Output: None (slice defined with the given key,value pair
#########################################################################
proc defineSliceForNewRow { object slice index value } {
    if { $value != "" } {
        set rstoe [ locate $object.rowstatus ]

        if { [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
	    define $slice $index $value
        } 
    }
}

########################################################################## 
# procedure: collectCpuExacctInfo
# Description: Collects the extended accounting details for CPU.
# The logic for the collection process is as follows:
#	a. Every one hour, we change the exacct file name to a new filename
#	b. We collect the finished tasks from the exacct file
#	c. We collect the running tasks exacct data from kernel using getacct()
#	d. We take the difference between the current reading and previous reading
#	   to the exacct data for this hour. We always preserve the previous data.
#	   The current data becomes previous data for the next iteration.
#
# Input: None
# Output: Cpu extended accounting info for all the containers
##########################################################################
proc collectCpuExacctInfo { } {

    #
    # For the time being, disable the extended accounting
    # for solaris 8
    #
    # The following if condition is commented to disable exacct
    # for solaris 8. Bugid: 4988672 is opened on the libexacct
    # category to provide the fix on solaris 8.
    #
    #if { [ sysinfo osversion ] == "5.8" } {
    set osVer [ sysinfo osversion ]
    if { $osVer == "5.8" || $osVer == "5.10" } {
	set result ""
        set projidtoe [ locate iso*ContEntry.projectID ]
        set projidlist [ toe_send $projidtoe getValues ]

	foreach projid $projidlist {
	    append result "0.00 "
	}
	return $result
    }

    #######################################################
    # Get the current extd accounting filename
    # 1. Execute the acctadm command, and capture the output
    #    Output:
    #   Task accounting: active
    #   Task accounting file: /var/adm/exacct/task
    #   Tracked task resources: extended
    #   Untracked task resources: none
    #   Process accounting: inactive
    #   Process accounting file: none
    #   Tracked process resources: none
    #   Untracked process resources: extended,host,mstate
    #   Flow accounting: inactive
    #   Flow accounting file: none
    #   Tracked flow resources: none
    #   Untracked flow resources: extended
    # 2. Parse the output to extract the Task Accounting file:
    #######################################################
    set curfilename ""

    #
    # Run the prj_acctadm.sh script to write the output
    # to $ESDIR/cfg/scmacctadm.dat. The prj_acctadm.sh script
    # would first check, if this file exists. If so, it deletes
    # and writes the new output to this file.
    #
    global env
    set result [ executeScript "prj_acctadm.sh" "-o" "" "" "" "" "" "" \
		    "" "" "" "$env(ESDIR)/cfg/scmacctadm.dat" "" ]
    if { $result == 0 } {
       ddl print warning "acctadm command failed to write output"
    }

    if { [ catch { open "$env(ESDIR)/cfg/scmacctadm.dat" r } fd ] } {
	ddl print warning "file open error for $env(ESDIR)/cfg/scmacctadm.dat"
	set curfilename ""
    } else {

	#
	# Read the file and get the curfilename
	#
        while { [ gets $fd line ] >= 0 } {
	    set line [ string trim $line ]

	    set linelist [ split $line ":" ]
	    lextract $linelist 0 name 1 val

	    if { "$name" == "Task accounting file" } {
		#
		# The finename has a space in the beginning.
		# Trim it, otherwise, the file cannot be opened
		#
	        set curfilename [ string trimleft $val ]
	        break
	    }
        }

        #
        # Close the file
        #
        close $fd
    }

    if { "$curfilename" == "" } {
	ddl print warning "task based extended accounting is disabled"
	return
    }

    ############################################
    # Construct the new filename
    ############################################
    #
    # Construct a new filename out of the current date and time stamp
    # The date and time stamp formatted is in the form:
    # Aug111347262003 ( Aug 11 13:47:26 2003 ). We construct the file
    # name out of the above date and time stamp. The new file name
    # would look like, /var/adm/exacct/taskAug111347262003. 
    #

    #
    # Get the date and time stamp
    #
    set timestamp [ clock format [ clock seconds ] -format %b%d\%H\%M\%S\%Y ]

    #
    # Construct the filename out of the timestamp
    #
    set newfilename "/var/adm/exacct/task" 
    append newfilename $timestamp

    #############################################
    # Change the extended accounting filename to
    # new file name
    #############################################
    #
    # After we write the accounting info, now, change the exacct filename.
    # acctadm -f $filename task. From now on, the extended acct info is written
    # to this new file
    #
    set result [ executeScript "prj_acctadm.sh" "-f" "" "" "" "" "" "" \
		    "" "" "" "$newfilename" "" ]
    if { $result == 0 } {
	ddl print warning "acctadm command failed to set the new file"
    }

    ####################################################
    # Now collect the exacct info from the curfilename
    ####################################################
    set projidtoe [ locate iso*ContEntry.projectID ]
    set projidlist [ toe_send $projidtoe getValues ]

    if { [ catch { getExacctInfo $curfilename } result ] } {
	set result ""
	foreach projid $projidlist {
	    append result "0.00 "
	}
    }

    return $result
}

######################################################################### 
#							    
# Processor Set Related functions (ProcessorSets managed-object)
#							    
######################################################################### 

########################################################################## 
# Procedure: deleteProcessorSet
# Description: Deletes the processor set from the system. 
#    This is called, when the pool is deleted.
# Input : Row Index
# Output: None
########################################################################## 

proc deleteProcessorSet { index } {

    #
    # Delete the processor set from the system
    #
    set name [ toe_send [ locate PsrSetEntry.psrsetName ] \
		getValue $index ]

    deletePset $name 

    set tblObj [ locate PsrSetTable.PsrSetEntry ]

    if { $tblObj != "" } {
	toe_send $tblObj cleanupRow $index CLEAR_PARMS
    }
    return 1
}

#########################################################################
# Procedure: defineSliceForNewPsetRow
# Description: Defines a slice for new pool
#
# Input: slice - slicename
#        index - key
#        value - value
# Output: slice defined with { key value } pair
#########################################################################
proc defineSliceForNewPsetRow { slice index value } {
    set rstoe [ locate PsrSetEntry.rowstatus ]

    if { [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
	define $slice $index $value
    } 
}


########################################################################## 
# Procedure: createProcessorSet
# Description: Creates a processor set on the system 
#    The createPset command is created in pkgsrm package.
# Input : index - Row Index
#         rowname - processor set name
# Output: success - Created Processor set 
########################################################################## 

proc createProcessorSet { index rowname } {

    set name $rowname
    set minsize [ ilookup psetmincpu $index ]
    set maxsize [ ilookup psetmaxcpu $index ]
    set objectives [ ilookup psetobjectives $index ]

    undefine psetmincpu $index
    undefine psetmaxcpu $index
    undefine psetobjectives $index

    createPset $name $minsize $maxsize "$objectives"

    return 1
}

########################################################################## 
# Procedure: createOrDeletePsrset
# Description: 
# Input : 
# Output: 
########################################################################## 

proc createOrDeletePsrset { index value rowname username } {

    set curruser [ checkPoolAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "createOrDeletePsrset: Invalid access. Permission denied for user to complete this operation"
        return 0
    }

    #
    # For rowstatus value of 4 and 5 call createProcessorSet
    # For rowstatus value of 6 call delete processor set function
    # For other values, just pass

    set result 1

    if { $rowstatus == 4 || $rowstatus == 5 } {
        set result [ createProcessorSet $index $rowname ]
    } else {
	if { $rowstatus == 6 } {
	    set result [ deleteProcessorSet $index ]
	} 
    }

    if { $result != 0 } {
       return $value
    } else {
       return 0
    }
}

########################################################################## 
# Procedure: checkPsrsetRowStatus
# Description: When we load the available system  processor sets, 
#    the rows are created for each pset on the system. This creation of 
#    row is not triggered through the console or through any set operation. 
#    This happens through the initial data cascading. Hence, during this 
#    operation, the rowstatus is not set to active for these rows. 
#    This causes problems when we add/delete rows through the console or 
#    through set operations. So, the checkCommand is provided to make 
#    sure the rowstatus node is set for all the active rows. If not set, 
#    we will set it manually.
#
# Input : None
# Output: New Values
########################################################################## 

proc checkPsrsetRowStatus { } {
    set tDepth [ toe_send [ locate PsrSetEntry ] getTableDepth ]
    set rstoe [ locate PsrSetEntry.rowstatus ]

    for { set index 1 } { $index <= $tDepth } { incr index } {

        if { [ toe_csend val $rstoe getValue $index ] } {
	    #
	    # The rowstatus is not set for this row
	    # Set it to active
	    #
	    toe_send $rstoe setValue $index 1
        }
    }
}


########################################################################## 
# Host Resource Set related Functions (Pools managed-object)
########################################################################## 

########################################################################## 
# Procedure: checkPoolCpuAvail
# Description: Check whether the given cpu size is available for the new
#    pool. This is done by checking the default pool size with
#    the given CPU value. Because, the system allocates cpus to new pools
#    from the default pool of resources.
#
# Input: index - Host Resource Set new row index
#        value - New CPU value for the new pool
# Output: Succes(1) - If there is sufficient Cpu power
#         Failure(0) - there is not enough cpu
##########################################################################
proc checkPoolCpuAvail { index value } {
    set defaulttoe [ locate PoolEntry.default ]
    set cpustoe [ locate PoolEntry.cpus ]
    set tDepth [ toe_send [ locate PoolEntry ] getTableDepth ]

    #
    # When modifying the poolsize, we need to consider the current size
    # of the existing pool as part of the available CPUs. That is
    # the total cpus available is the sum of the current poolsize and
    # the default poolsize
    #
    if { $index <= $tDepth } {
	set totalcpus [ toe_send $cpustoe getValue $index ]
    } else {
	set totalcpus 0
    }

    toe_send $defaulttoe sliceforeach key default data {
	if { $default == 1 } {
	    if { $index == $key } {
		#
		# The user cannot modify the default pool 
		# of the system Hence, error out here
		ddl print warning "Default pool cannot be modified"
		return 0
	    }

	    set cpus [ toe_send $cpustoe getValue $key ] 
            set totalcpus [ expr ( $totalcpus + $cpus ) ]

	    if { $totalcpus > $value } {
		#
		# We can accomodate this host pool
		#
		return 1
	    } else {
		ddl print warning "Not enough CPU in pool"
		return 0
	    }
	}
    }

    return 1
}

##########################################################################
# procedure: updateMinPoolCpuSize
# Description: Update the cpu minsize for the current pool
# 
# Input: index - row index
#        value - new mincpu value to the  pool
# Output: Success(1) - if update succeeds
#         Failure(0) - if update fails
##########################################################################
proc updateMinPoolCpuSize { index value username } {

    set curruser [ checkPoolAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "updateMinPoolCpuSize: Invalid access. Permission denied for user to complete this operation"
        return 0
    }


    set result [ checkPoolCpuAvail $index $value ]

    if { $result == 0 } {
	return 0
    }

    if { ! [ catch { toe_send [ locate PoolEntry.rowstatus ] \
		getValue $index } rowstatus ] } {

        #
        # Pools are not supported in Solaris8, and we have simulated
        # a default pool with the entire system CPUs, and we will not
        # allow any modifications to this pool. Hence returning without
        # updation.
        #
        if { [ sysinfo osversion ] == "5.8" } {
            ddl print warning "Pool Updation is not supported in Solaris 8"
            return 0
        }

        if { $rowstatus == 1 } {

	    set poolname [ toe_send [ locate PoolEntry.poolName ] \
			getValue $index ]

	    set psetname [ toe_send [ locate PoolEntry.psetName ] \
			getValue $index ]

	    set maxsize [ toe_send [ locate PoolEntry.maxcpus ] \
			getValue $index ]

	    if { [ catch { updatePoolSize $poolname $psetname $value $maxsize } retval ] } {
	        #
	        # Could not update the pool definition
	        #
	        ddl print error "error in pool updation for min cpu"
	        return 0
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    define refreshtrap $index 1
	    updatePoolconfTimestamp 
	    refreshValue
        }
    }

    return $value
}

##########################################################################
# procedure: updateMaxPoolCpuSize
# Description: Update the cpu maxsize for the current pool
# 
# Input: index - row index
#        value - new maxcpu value to the  pool
# Output: Success(1) - if update succeeds
#         Failure(0) - if update fails
##########################################################################
proc updateMaxPoolCpuSize { index value username } {

    set curruser [ checkPoolAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "updateMaxPoolCpuSize: Invalid access. Permission denied for user to complete this operation"
        return 0
    }


    if { ! [ catch { toe_send [ locate PoolEntry.rowstatus ] \
		getValue $index } rowstatus ] } {

        #
        # Pools are not supported in Solaris8, and we have simulated
        # a default pool with the entire system CPUs, and we will not
        # allow any modifications to this pool. Hence returning without
        # updation.
        #
        if { [ sysinfo osversion ] == "5.8" } {
            ddl print warning "Pool Updation is not supported in Solaris 8"
            return 0
        }

        if { $rowstatus == 1 } {

	    set poolname [ toe_send [ locate PoolEntry.poolName ] \
			getValue $index ]

	    set psetname [ toe_send [ locate PoolEntry.psetName ] \
			getValue $index ]

	    set minsize [ toe_send [ locate PoolEntry.mincpus ] \
			getValue $index ]

	    if { [ catch { updatePoolSize $poolname $psetname $minsize $value } retval ] } {
	        #
	        # Could not update the pool definition
	        #
	        ddl print error "error in pool updation for max cpu"
	        return 0
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    define refreshtrap $index 1
	    updatePoolconfTimestamp 
	    refreshValue
        }
    }

    return $value
}

##########################################################################
# procedure: updatePoolMaxShares
# Description: Update the cpu count for the current pool
# 
# Input: index - row index
#        value - new cpu value to the  pool
# Output: Success(1) - if update succeeds
#         Failure(0) - if update fails
##########################################################################
proc updatePoolMaxShares { index value username } {

    set curruser [ checkPoolAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "updatePoolMaxShares: Invalid access. Permission denied for user to complete this operation"
        return 0
    }


    if { ! [ catch { toe_send [ locate PoolEntry.rowstatus ] \
		getValue $index } rowstatus ] } {

        #
        # Pools are not supported in Solaris8, and we have simulated
        # a default pool with the entire system CPUs, and we will not
        # allow any modifications to this pool. Hence returning without
        # updation.
        #
        if { [ sysinfo osversion ] == "5.8" } {
            ddl print warning "Pool Updation is not supported in Solaris 8"
            return 0
        }

        if { $rowstatus == 1 } {

	    set poolname [ toe_send [ locate PoolEntry.poolName ] \
			getValue $index ]

            set pooltoe [ locate iso*ContEntry.poolName ]
            set cputoe [ locate iso*ContEntry.cpuRes ]

            #
            # Now, visit all the container having this pool name
            # and find out the target reservation alotted to them.
            #

            set totalCpus 0

            toe_send $pooltoe sliceforeach key tmppool data {
                if { $tmppool == $poolname } {
                    set cpures [ toe_send $cputoe getValue $key ]
                    set totalCpus [ expr $totalCpus + $cpures ]
                }
            }

            if { $totalCpus > $value } {
                #
                # The reservation value exceeds the new maxshares of the
                # pool. Hence, we cannot set it.
                #
                ddl print error "udpatePoolMaxShares: cannot set the value $value"
                return 0
            }

            # Verification is done. Update the pool.

            global env
	    ddl print info "Executing the pool_mod.sh for $poolname and $value"
            if { [ sysinfo osversion ] == "5.10" } {
                set result [ srmShellAsync $env(ESROOT)/modules/sbin/pool_mod.sh \
                         -p $poolname -m $value ]
            } else {
                set result [ srmShell $env(ESROOT)/modules/sbin/pool_mod.sh \
                         -p $poolname -m $value ]
            }
	    if { $result != 0 } {
	        ddl print error "error in pool updation for mashares $value"
		return 0
            }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    define refreshtrap $index 1
	    updatePoolconfTimestamp 
	    refreshValue
        }
    }

    return $value
}

##########################################################################
# procedure: updatePoolScheduler
# Description: Update the default scheduler of the pool
# 
# Input: index - row index
#        value - new scheduler for the pool
# Output: Success(1) - if update succeeds
#         Failure(0) - if update fails
##########################################################################
proc updatePoolScheduler { index value username } {

    set curruser [ checkPoolAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "updatePoolScheduler: Invalid access. Permission denied for user to complete this operation"
        return 0
    }


    if { ! [ catch { toe_send [ locate PoolEntry.rowstatus ] \
		getValue $index } rowstatus ] } {

        #
        # Pools are not supported in Solaris8, and we have simulated
        # a default pool with the entire system CPUs, and we will not
        # allow any modifications to this pool. Hence returning without
        # updation.
        #
        if { [ sysinfo osversion ] == "5.8" } {
            ddl print warning "Pool Updation is not supported in Solaris 8"
            return 0
        }

        if { $rowstatus == 1 } {

	    set poolname [ toe_send [ locate PoolEntry.poolName ] \
			getValue $index ]

            # Update the pool.

            global env
	    ddl print info "Executing the pool_mod.sh for $poolname and $value"
            if { [ sysinfo osversion ] == "5.10" } {
                set result [ srmShellAsync $env(ESROOT)/modules/sbin/pool_mod.sh \
                         -p $poolname -c $value ]
            } else {
                set result [ srmShell $env(ESROOT)/modules/sbin/pool_mod.sh \
                         -p $poolname -c $value ]
            }
	    if { $result != 0 } {
	        ddl print error "error in pool updation for sched class $value"
		return 0
            }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    define refreshtrap $index 1
	    updatePoolconfTimestamp 
	    refreshValue
        }
    }

    return $value
}
##########################################################################
# procedure: updatePoolObjectives
# Description: Update the pool objectives
# 
# Input: index - row index
#        value - new objectives string
# Output: Success(1) - if update succeeds
#         Failure(0) - if update fails
##########################################################################
proc updatePoolObjectives { index value username } {

    set curruser [ checkPoolAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "updatePoolObjectives: Invalid access. Permission denied for user to complete this operation"
        return 0
    }


    if { ! [ catch { toe_send [ locate PoolEntry.rowstatus ] \
		getValue $index } rowstatus ] } {

        #
        # Pool objectives are not supported in Solaris8 and 9
        # Hence returning without updation.
        #
        if { [ sysinfo osversion ] == "5.8" } {
            ddl print warning "Pool objectives are not supported in Solaris 8"
            return 0
        }

        if { [ sysinfo osversion ] == "5.9" } {
            # Pool objectives are not applicable on Solaris 9
            # just return success.
            return " "
        }

        if { $rowstatus == 1 } {

	    set poolname [ toe_send [ locate PoolEntry.poolName ] \
			getValue $index ]

	    set psetname [ toe_send [ locate PoolEntry.psetName ] \
			getValue $index ]
            # Update the pool.

            global env
	    ddl print info "Executing the pool_mod.sh for $poolname and $value"
            if { [ sysinfo osversion ] == "5.10" } {
                set result [ srmShellAsync $env(ESROOT)/modules/sbin/pool_mod.sh \
                         -p $poolname -s "$psetname" -o "$value" ]
            } else {
                set result [ srmShell $env(ESROOT)/modules/sbin/pool_mod.sh \
                         -p $poolname -s "$psetname" -o "$value" ]
            }
	    if { $result != 0 } {
	        ddl print error "error in pool updation for objectives: $value"
		return 0
            }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
	    define refreshtrap $index 1
	    updatePoolconfTimestamp 
	    refreshValue
        }
    }

    return $value
}

##########################################################################
# Procedure: checkHostResRowStatus
# Description: When we load the available system  pools, the rows are 
#    created for each pool on the system. This creation of row is not 
#    triggered through the console or through any set operation. 
#    This happens through the initial data cascading. Hence, during this 
#    operation, the rowstatus is not set to active for these rows. 
#    This causes problems when we add/delete rows through the console or 
#    through set operations. So, the checkCommand is provided to make 
#    sure the rowstatus node is set for all the active rows. 
#    If not set, we will set it manually.
#
# Input : None
# Output: None, the rowstatus of all the pool entries is made active
##########################################################################

proc checkHostResRowStatus { } {
    set tDepth [ toe_send [ locate PoolEntry ] getTableDepth ]
    set rstoe [ locate PoolEntry.rowstatus ]

    for { set index 1 } { $index <= $tDepth } { incr index } {

        if { [ toe_csend val $rstoe getValue $index ] } {
	    #
	    # The rowstatus is not set for this row
	    # Set it to active
	    #
	    toe_send $rstoe setValue $index 1
        }
    }

    #
    # as a result of changing the system configuration, like,
    # deleting a pool, the row in this table may not appear, but, the rowstatus
    # would still appear as 1, so, clear them too
    #
    while { 1 } {
	if { [ catch { toe_send $rstoe getValue $index } val ] } {
	    break
	} else {
	    #
	    # clear the rowstatus node
	    #
	    toe_send $rstoe setValue $index 6

	    incr index
	}
    }
}


##########################################################################
# Procedure: createOrDeleteSystemPool
# Description: Create or Delete a system pool, depending on the rowstatus
#     value.
#
# Input: index - Row Index 
#        rowstatus - rowstatus value to determine create/delete operation
#        rowname - index value of the row, i.e, the poolname 
# Output: Success (1) - If the create/delete system pool succeeds
#  otherwise failure(0)
##########################################################################

proc createOrDeleteSystemPool { index rowstatus rowname username } {

    set curruser [ checkPoolAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "createOrDeleteSystemPool: Invalid access. Permission denied for user to complete this operation"
        return 0
    }

    #
    # For createAndGo and createAndWait, create the
    # pool. For destroy, delete the system pool. For the
    # unknown operation, just pass it
    #
    set result 1

    # Solaris 8 has pool_default and does not support additon/deletion
    if { [ sysinfo osversion ] == "5.8" } {
        ddl print warning "Creating/Deleting pool is not supported on Solaris 8"
        return 0
    }

    if { $rowstatus == 4 || $rowstatus == 5 } {
        set result [ createSystemPool $index $rowname ]
    } else {
	if { $rowstatus == 6 } {
	    set result [ deleteSystemPool $index ]
	} 
    }

    updatePoolconfTimestamp 

    if { $result != 0 } {
        return $rowstatus
    } else { 
	return 0
    }
}


##########################################################################
# Procedure: createSystemPool
# Description: Creates a pool in the system. Gets the psrsetname for the index.
# Input : index - new row index
#         rowname - row index value
# Output:  Success(1) - if system pool is created
#          Failure(0) - if system pool creation fails
##########################################################################

proc createSystemPool { index rowname } {

    set poolname $rowname
    set mincpus [ ilookup poolmincpus $index ]
    set maxcpus [ ilookup poolmaxcpus $index ]
    set objectives [ ilookup poolobjectives $index ]
    set schedclass [ ilookup poolschedclass $index ]
    set maxcpushares [ ilookup poolmaxcpushares $index ]
    undefine poolmincpus
    undefine poolmaxcpus
    undefine poolobjectives

    # Check if the pool name starts with a numeric character
    if { ! [ catch { expr [ string range $poolname 0 0 ] } result ] } {
	ddl print error "pool name cannot start with a numeric character"
	return 0
    }

    #
    # Find the default pool, and make sure, the given
    # CPUs can be accomodated in it. Otherwise, we
    # will not create the pool
    #
    set defpool [ findDefaultPool ]
    set defpoolsize [ getCpusizeForPool $defpool ]

    if { $mincpus >= $defpoolsize } {
	ddl print error "min poolsize should be less than default poolsize"
	return 0
    }

    if { [ catch { createPset $poolname $mincpus $maxcpus "$objectives" } result ] } {
	ddl print error "processor set cannot be created for pool"
	return 0
    }

    #
    # First create a pool
    #
    if { [ catch { createPool $poolname $schedclass $maxcpushares } result ] } {
	ddl print error "pool cannot be created"
	return 0
    }

    #
    # Now associate the processor set to pool
    #
    if { [ catch { associateResourceToPool $poolname "pset" $poolname } \
			result ] } {
	ddl print error "could not associate processor set to pool"
	return 0
    }

    #
    # Inform the PsrSetTable to refresh the table, 
    # so that the currently created processor set shows up in that table.
    #
    toe_send [ locate ProcessorSets.PsrSetTable ] refreshValue 
    
    return 1
}


##########################################################################
# Procedure: deleteSystemPool
# Description: Deletes a system pool. 
# Input : index - row index of the row to be deleted
# Output: success/failure
##########################################################################

proc deleteSystemPool { index } {
    #
    # Delete the processor set from the system
    #
    set  pntoe [ locate PoolEntry.poolName ]

    if { ! [ catch { toe_send $pntoe getValue $index } name ] } {
        deletePool $name 
        deletePset  $name
    }

    set tblObj [ locate PoolTable.PoolEntry ]

    if { $tblObj != "" } {
	toe_send $tblObj cleanupRow $index CLEAR_PARMS
    }

    toe_send [ locate ProcessorSets.PsrSetTable ] refreshValue 
    return 1
}

#######################################################################
#
# General Purpose Procedures
#
#######################################################################



##########################################################################
# Procedure: defineSliceForActiveRow
# Description: This procedure defines a slice for use by the 
#    corresponding property rule. This rule is defined only when the 
#    already existing property instance is altered. This is
#    done by checking the rowstatus instance node. If it is 1(active), 
#    then row is already present.
# Input: slice - slicename
#        index - key for slicename
#        value - value for the slicename key
# Output: slice defined { key value } pair
##########################################################################

proc defineSliceForActiveRow { slice index value } {

    if { ! [ catch { getRowValue rowstatus $index } rowstatus ] } {
	if { $rowstatus == 1 } {
	    #
	    # The row already exist and this is an individual set.
	    # Hence, define a slice  
	    #
	    define $slice $index $value
	}
    }
}

#########################################################################
# Procedure: defineSliceForNewHostResRow
# Description: Defines a slice for new pool
#
# Input: slice - slicename
#        index - key
#        value - value
# Output: slice defined with { key value } pair
#########################################################################
proc defineSliceForNewHostResRow { slice index value } {
    set rstoe [ locate PoolEntry.rowstatus ]

    if { [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
	define $slice $index $value
    } 
}

########################################################################
# Procedure: checkResult
# Description: This is a callback procedure for any shell scripts 
#    executed in a procedure call. The result would be of the 
#    following format
#     0 - operation successful 
#     1 - operation failed
#     2 - syntax error or other failures
# Input: result - result of the script execution
# Output: None - It sets the globalscriptresult
########################################################################
proc checkResult { result } {
    lextract $result 2 status
    global globalscriptresult

    set globalscriptresult [ string match "*status 0" $status ]
}

##########################################################################
# Procedure: executeScript
# Description: Common procedure to execute the given script name with
# arguments. 
# Input: Script name - name of the script
#        option - options for scripts
#        projname - project name for project creation/deletion/updation
#        projid - project id for project creation/deletion/updation
#        comment - comment for project creation
#        users - users association to project
#        groups - groups association to project
#        resctls - resource controls for project
#        tasklist - for writing exended accounting
#        filename - for writing acctadm output and also to change exacct file
#
# Output: 1 on success, and 0 on failure
##########################################################################
proc executeScript { scriptname option zonename projname projid comment users \
			groups pool resctls tasklist filename matchexpr } {

    global env

    if { [ sysinfo osversion ] == "5.10" } {
        set shellCommand "srmShellAsync"
    } else {
        set shellCommand "srmShell"
    }

    switch -exact -- $scriptname {
	prj_add.sh {
               set result [ $shellCommand $env(ESROOT)/modules/sbin/prj_add.sh \
		-z $zonename -n $projname -p $projid -c $comment -u $users \
		-g $groups -r $resctls ]
	}

	prj_mod.sh {
	    set arg ""
	    switch -exact -- $option {
	        -c { set arg $comment }
		-u { set arg $users }
		-g { set arg $groups }
	    }

              set result [ $shellCommand $env(ESROOT)/modules/sbin/prj_mod.sh \
	  	 $zonename $projname $option $arg ]
	}

	prj_del.sh {
              set result [ $shellCommand $env(ESROOT)/modules/sbin/prj_del.sh $zonename $projname $projid ]
	}

	prj_rctl_mod.sh {
	    if { $pool != "" } {
                set result [ $shellCommand $env(ESROOT)/modules/sbin/prj_rctl_mod.sh \
		     -z $zonename -n $projname -r $pool $option $resctls ]
	    } else {
                set result [ $shellCommand $env(ESROOT)/modules/sbin/prj_rctl_mod.sh \
		     -z $zonename -n $projname $option $resctls ]
	    }
	}

	prj_wracct.sh {
            set result [ $shellCommand $env(ESROOT)/modules/sbin/prj_wracct.sh $zonename "$tasklist" ]
	}

	prj_acctadm.sh {
            set result [ $shellCommand $env(ESROOT)/modules/sbin/prj_acctadm.sh $zonename $option \
                 "$filename" ]
	}

	prj_userattr.sh {
	    set userlist [ split $users , ]
            set result [ $shellCommand $env(ESROOT)/modules/sbin/prj_userattr.sh $zonename $projname $option \
		 $userlist ]
	}

	prj_matchexpr.sh {
            set result [ $shellCommand $env(ESROOT)/modules/sbin/prj_matchexpr.sh \
		 "$matchexpr" ]
	}

        lnode_functions.sh {
            if { $option == "-c" } {
                set result [ srmShell $env(ESROOT)/modules/sbin/lnode_functions.sh $option \
                    $projname -C $resctls ]
            } else {
                set result [ srmShell $env(ESROOT)/modules/sbin/lnode_functions.sh $option \
                    $projname ]
            }

	    #
	    # For lnode_functions.sh case, we return the result
	    # as it is for further analysis at the place where
	    # this script is called. This is to handle some situations
	    # where lnode does not exists and project entry exists,
	    # and also, user trying to delete an lnode, where there
	    # are some processes running in it.
	    #
	    return $result
        }
    }

    if { $result == 0 }  {
	return 1
    } else  {
	return 0
    }
}

##########################################################################
# Procedure: binarysearch
# Description: Does a binary search of the list of lists of the form
# 	{ a b c } { d e f } { g h i }
#       The list must be sorted by the first number in every list
# Input: list - List of lists of the form { { } { } { } }
#        item - item to search the list
# Output: position in the list if match found, otherwise -1
##########################################################################

proc binarysearch { list item } {
    set low 0
    set high [ expr [ llength $list ] - 1 ]

    while { $low <= $high } {
        set mid [ expr ( $high + $low ) / 2 ]

        lextract $list $mid entries
        lextract $entries 0 first      

        if { $first == $item } {
            return $mid
        }

        if { $first > $item } {
            set high [ expr $mid - 1 ]
        } else {
            set low [ expr $mid + 1 ]
        }
    }

    return -1
}

##########################################################################
#
# Procedure: setUserOrGroupDefaultProject
# Description: During container creation, the user specifies a list of
# users, primary usrs, for whom the project name should set as the default
# project in the /etc/user_attr file. This procedures, validates each user
# and if this is a individual set on the primary user node, then updates
# the /etc/user_attr database, otherwise, the createContainer procedure
# would take care of updating the /etc/user_attr file.
#
##########################################################################
proc setUserOrGroupDefaultProject { index value } {
    #
    # Split the comma separated user/group list.
    # The variable userlist is used to represent both
    # users and groups here.
    #
    # 
    # 
    # During container creation, when a primary user is specified,
    # we check whether the user exists, and if not we will not create
    # a container. Checking this condition is a feature, but, this may
    # causes some concerns due to insufficient error information on the
    # seahaven console. For the time being, this part of the code is
    # commented, and we will wait till the RR. If we find that this is
    # needed, we will totally delete the following lines, but, for the
    # time being, just commenting the code.
    #
    #set userlist [ split $value , ]
    #
    #
    #foreach user $userlist {
    #
    #	if { ! [ checkUserOrGroupExists $user ] } {
    #
    # The user is not a valid user or a group
    # So, we will just return failure here
    #
    #	    return 0
    #	}
    #}

    #
    # Now set the user's default project
    #
    set defaultCont [ checkForDefaultContainer $index ]

    if { $defaultCont == 1 } {
	ddl print warning "Editing on default container is not allowed"
	ddl print info "setUserOrGroupDefaultProject $index $value"
	return 0
    }

    setUserDefaultProject $index $value 0
    return $value 
}

##########################################################################
#
# Procedure: setUserDefaultProject
# Description: Sets the users default project in /etc/user_attr file
# This function is called on 2 occasions.
# 1. During container creation
# 2. During editing of the primaryUses node for that row.
#
# The contCreation parameter indicates whether it is a container creation
# or an individual set on the primaryUsers column
#
##########################################################################
proc setUserDefaultProject { index users contCreation } {

    if { ! [ catch { toe_send [ locate iso*ContEntry.rowstatus ] \
		getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {
	    set projname [ toe_send [ locate iso*ContEntry.projectName ] \
		getValue $index ]

	    set zonename [ toe_send [ locate iso*ContEntry.zoneName ] \
		getValue $index ]
	    #
	    # First remove the old users, if there are any
	    #
	    if { $contCreation == 0 } {
	        set olduserlist [ toe_send [ locate iso*ContEntry.primaryUsers ] \
		    getValue $index ]

	        if { $olduserlist != "" } {
		    ddl print info "setUserDefaultProject: old users: $olduserlist"
	            set result [ executeScript "prj_userattr.sh" -d $zonename $projname \
	    		     "" "" $olduserlist "" "" "" "" "" "" ]
	        }
	    }

	    if { $users != "" } {
		ddl print info "setUserDefaultProject: new users = $users"
	        set result [ executeScript "prj_userattr.sh" -a $zonename $projname \
			 "" "" $users "" "" "" "" "" "" ]
	        if { $result == 0 && $contCreation == 0 } {
		    ddl print error "could not make $projname as default project"
		    return 0
	        }
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console. If it is a container creation operation, this will
	    # be done in the createContainer procedure
	    #
	    if { $contCreation == 0 } {
	        define refreshtrap $index 1
	        saveContainerData
	    }
	}
    }

    return 1
}

##########################################################################
#
# Procedure: sendRefreshValueTrap
# Description: This is a general procedure to check if the set happened
# 	       on for a given node only, then we send the refresh trap
#	       to the console. If the set happens as part of the row-creation
#	       rowstatus node will take care of sending refresh trap.
#		
#	       Whenever there is a set on the individual node, the individual 
#	       node's validateCommand checks if rowstatus value is 1, then
#	       defines a slice called, refreshtrap, in the table context.
#	       The sendRefreshValueTrap should be called in that table context.
#	       It checks if the refreshtrap slice is defined. If so, it
#	       undefines the slice and sends out a refresh trap to the console.
#
##########################################################################
proc sendRefreshValueTrap { index } {
    if { ! [ catch { ilookup refreshtrap $index } rtrap ] } {
	undefine refreshtrap $index
	refreshValueAndTrap
    }
}

##########################################################################
#
# Procedure: schedContTimers
# Description: Called when the container table is activated. 
#  	       Registers 2 timers
#		a. /var/adm/exacct directory cleanup timer, and
#			Every 24 hours, it deletes files in /var/adm/exacct
#			directory which are more than 24 hours older. 
#		b. Move processes timer
#			Every 30 minutes, checks for processes, matching
#			the matchexpr criteria and moves them under the 
#			specified projectname.
#
##########################################################################

proc schedContTimers { } {
    #
    # We need the following timers, when the module is loaded, and 
    # the container table is activated.
    #
    # a. A timer to periodically check for files under /var/adm/exacct
    #    directory greater than 24 hours from the current time and delete
    #    them
    # b. A timer to periodically check for processes matching the matchexpr
    #    registered, and move them under the corresponding container.
    #    In this case, whenever a container is created, and if the matchexpr
    #    is specified, we add that projectname and the matchexpr to a slice
    #    called projmatchexpr. The timer looks for the key,value (projectname,
    #    matchexpr) pairs and calls prj_matchexpr.sh script to move processes.
    #

    #
    # When the container table is activated, we first cleanup the 
    # /var/adm/exacct directory, and set the timer for every 24 hours
    #
    exacctDirCleanup

    schedExacctDirCleanup

    schedMoveProcesses
}

##########################################################################
#
# Procedure: schedExacctDirCleanup
# Description: schedules timer to cleanup /var/adm/exacct directory 
# every 24 hours
#
##########################################################################
proc schedExacctDirCleanup { } {
    define value exacctfiletimer [ registerOneShot 86400 \
		exacctDirCleanup ]
}

##########################################################################
#
# Procedure: exacctDirCleanup
# Description: Cleans /var/adm/exacct directory every 24 hours for files
#		older than 24 hours. This procedure has a catch block.
#		It is because, this function is called in schedContTimers
#		before scheduling it, and hence, without catch block, it
#		generate error when undefining the timer.
#
##########################################################################
proc exacctDirCleanup { } {
    if { ! [ catch { lookup value exacctfiletimer } ] } {
        undefine value exacctfiletimer
        schedExacctDirCleanup
    }

    #
    # Cleanup the directory here. 
    #
    cleanupExacctDir
}

##########################################################################
#
# Procedure: schedMoveProcesses
# Description: Schedules a timer to move processes into a container for
#		every 30 minutes.
#
##########################################################################
proc schedMoveProcesses { } {

    # processes cannot be moved from one container to other in Solaris 8
    if { [ sysinfo osversion ] != "5.8" } {
        define value moveproctimer [ registerOneShot 300 \
		moveProcesses ]
    }
}

##########################################################################
#
# Procedure: moveProcesses
# Description: Moves processes into a container
#		The procedure is, it collects all the <projectname> 
#		<matchexpr> pairs and sends them to prj_matchexpr.sh script.
#
##########################################################################
proc moveProcesses { } {


    # processes cannot be moved from one container to other in Solaris 8
    if { [ sysinfo osversion ] == "5.8" } {
        return 0
    }
    undefine value moveproctimer
    schedMoveProcesses
    #
    # Move the processes here.
    #

    set contentrytoe [ locate ContEntry ]
    set projnametoe [ locate ContEntry.projectName ]
    set zonenametoe [ locate ContEntry.zoneName ]
    set mexprtoe [ locate ContEntry.matchexpr ]
    set projExprs ""

    toe_send $mexprtoe sliceforeach key matchexpr data {
	if { $matchexpr != "" } {
	    set projname [ toe_send $projnametoe getValue $key ]
	    set zonename [ toe_send $zonenametoe getValue $key ]
	    append projExprs "$zonename $projname $matchexpr "
	}
    }

    # 
    # The result container <zonename> <projname> <matchexpr> pairs
    # Pass this result to prj_matchexpr.sh script and it takes care
    # of moving the processes.
    #
    if { $projExprs != "" } {
        set result [ executeScript "prj_matchexpr.sh" "" "" "" \
		 "" "" "" "" "" "" "" "" "$projExprs" ]
    }
}

##########################################################################
#
# Procedure: moveProcsToContainer 
# Description: If the set is an individual set on the matchexpr node,
# and if the matchexpr is specified, move the processes matching the
# new expr. Also, replace the old expr in the slice with the new one.
# For new row, just add a slice and the createContainer procedure would
# take care of moving the processes after the container creation is successful.
#
##########################################################################
proc moveProcsToContainer { index rowname newmatchexpr contCreation username } {

    set curruser [ checkProjAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "moveProcsToContainer: Invalid access. Permission denied for user to complete this operation"
        return 0
    }


    # processes cannot be moved from one container to other in Solaris 8
    if { [ sysinfo osversion ] == "5.8" } {
        return $newmatchexpr
    }

    if { ! [ catch { toe_send [ locate ContEntry.rowstatus ] \
			getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {

	    if { $newmatchexpr != "" } {

                set projname [ toe_send [ locate ContEntry ] \
				getRowValue projectName $rowname ]
                set zonename [ toe_send [ locate ContEntry ] \
				getRowValue zoneName $rowname ]
		set projexprpair ""
		append projexprpair "$zonename $projname $newmatchexpr"
                set result [ executeScript "prj_matchexpr.sh" "" "" "" \
		             "" "" "" "" "" "" "" "" "$projexprpair" ]
	    }

	    if { $contCreation == 0 } {
	        saveContainerData
	    }
	}
    } else {
	#
	# This is a new row
	# Store the newmatchexpr for use by the createContainer.
	#
	if { $newmatchexpr != "" } {
	    define projmatchexpr $index $newmatchexpr
	}
    }

    return $newmatchexpr
}

##########################################################################
# 
# Procedure: updateProjectFileTimestamp
# Description: When we create/delete/modify a container row in the container
# table, it affects the timestamp of the project database on the system.
# Hence, we need to sync the timestamp we maintain in the projectTimestamp
# node in the Admin branch to the new timestamp to avoid unnecessary
# refresh operation from the service layer.
# 
##########################################################################
proc updateProjectFileTimestamp { } {
    set projtoe [ locate iso*Admin.projectTimestamp ]
    file stat "/etc/project" timestamp
    set curtimestamp $timestamp(ctime)
    toe_send $projtoe setValue 0 $curtimestamp
}

##########################################################################
#
# Procedure: updatePoolconfTimestamp
# Description: When we create/delete/update the resource pool definition
# it affects the pooladm configuration file's timestamp. To sync this time
# stamp with the one we maintain in the poolconfTimestamp of the Admin
# branch, we call this procedure.
#
##########################################################################
proc updatePoolconfTimestamp { } {
    set pooltoe [ locate iso*Admin.poolconfTimestamp ]

    if { [ catch { file stat "/etc/pooladm.conf" timestamp } ] }  {
        set curtimestamp 0
        set curaccesstime 0
    } else {
        set curtimestamp $timestamp(ctime)
        set curaccesstime $timestamp(atime)
    }

    toe_begin $pooltoe
    setValue 0 $curtimestamp

    if { ! [ catch { lookup internal poolconfaccesstimestamp } at ] } {
        undefine internal poolconfaccesstimestamp
    }

    define internal poolconfaccesstimestamp $curaccesstime

    #
    # Take care of the dynamic configuration time stamp
    #
    set poolDynLoc [ getPoolDynamicLocation ]
    if { [ catch { file stat $poolDynLoc timestamp } ] }  {
	set poolDynCurTime 0
    } else {
	set poolDynCurTime $timestamp(ctime)
    }

    if { ! [ catch { lookup internal poolDynTimestamp } t ] } {
	undefine internal poolDynTimestamp
    }

    define internal poolDynTimestamp $poolDynCurTime

    toe_end
}

########################################################################
#
# Procedure: refreshNupdateContainer
# Description: When the container is modified, this proc is 
# called to update the project file timestamp and also to save
# the container data. It initializes the refresh trap for the
# console
#
########################################################################

proc refreshNupdateContainer { index } {
    define refreshtrap $index 1
    updateProjectFileTimestamp 
    saveContainerData
}

########################################################################
#
# Procedure: refreshProjectFileTimestamp
# Description: Compare the previously value with the current value.
# If the /etc/project is modified between 2 refresh intervals, then
# send a refresh trap to the console for refreshing.
#
########################################################################

proc refreshProjectFileTimestamp { } {
    file stat "/etc/project" timestamp
    set curtimestamp $timestamp(ctime)

    if { ! [ catch { getValue 0 } oldtimestamp ] } {

	if { $oldtimestamp != $curtimestamp } {
	    #
	    # The project file has changed, hence, issue
	    # a refreshTrap. And store the current timestamp
	    #
            setTrapInfo refreshOID
	}
    }

    return $curtimestamp
}


########################################################################
#
# Procedure: verifyPoolconfFileTimestamp
# Description: There 2 timestamps that need to be checked. The change time
# and the access time. If the file has changed, then send a refresh trap.
# Otherwise, check whether the access time of the pooladmconf has changed.
# The scenario here is to catch the dynamic reconfiguration of the system
# through CLI and update the service layer database. The user can change
# the pooladm conf file and commit it to the system. We need to catch this
# and let the service layer know of the changes through refresh trap.
# The reason behind the access time is, the user can change the static
# configuration and later commits it. The committing requires the pooladm
# command to access this file, and we catch it using the access time of this
# file.
#
########################################################################

proc refreshPoolconfFileTimestamp { } {
    #
    # As pools does not exist on solaris 8, returning with a constant
    #
    if { [ sysinfo osversion ] == "5.8" } {
        return 1063879010
    }

    set trapTriggered 0

    #
    # See whether the active pool has changed by
    # comparing the file change time. The user does pooladm -c
    # or pooladm -x command to commit/delete a configuration, 
    # and we have to catch this.
    #

    set poolDynLoc [ getPoolDynamicLocation ]
    if { [ catch { file stat $poolDynLoc timestamp } ] } {
	set poolDynCurTime 0
    } else {
	set poolDynCurTime  $timestamp(ctime)
    }

    if { [ catch { lookup internal poolDynTimestamp } poolDynOldTime ] } {
	#
	# This can happen for the first time, hence,
	# define a slice and store the current value
	#
	define internal poolDynTimestamp $poolDynCurTime
    } else {
	if { $poolDynOldTime != $poolDynCurTime } {
	    #
	    # Update the time slice with the current time
	    #
	    undefine internal poolDynTimestamp
	    define internal poolDynTimestamp $poolDynCurTime

	    #
	    # Refresh the pool table and the processor set table
	    #
    	    toe_send [ locate iso*Pools.PoolTable ] refreshValue 
    	    toe_send [ locate iso*ProcessorSets.PsrSetTable ] refreshValue 

	    #
	    # Send a refresh trap to the service to sync the data
	    #
            setTrapInfo refreshOID
	    set trapTriggered 1
	}
    }

    #
    # Now, check for the static file configuration
    #
    if { [ catch { file stat "/etc/pooladm.conf" timestamp } ] } {
	set curtimestamp 0
	set curaccesstime 0
    } else {
        set curtimestamp $timestamp(ctime)
        set curaccesstime $timestamp(atime)
    }

    if { [ catch { getValue 0 } oldtimestamp ] } {
	#
	# This occurs for the first time
	#
	define internal poolconfaccesstimestamp $curaccesstime
    } else {
	set oldaccesstime [ lookup internal poolconfaccesstimestamp ]

	if { $oldtimestamp != $curtimestamp || $oldaccesstime != $curaccesstime } {
	    undefine internal poolconfaccesstimestamp
	    define internal poolconfaccesstimestamp $curaccesstime

	    #
	    # See if we have already triggered a refresh trap, if not 
	    # trigger it. 
	    #
	    if { $trapTriggered == 0 } {
	        #
	        # First refresh the pool table and then send the trap
	        #
    	        toe_send [ locate iso*Pools.PoolTable ] refreshValue 
    	        toe_send [ locate iso*ProcessorSets.PsrSetTable ] refreshValue 

                setTrapInfo refreshOID
	    }
	} 
    }
    
    return $curtimestamp
}


########################################################################
#
# Procedure: checkForDefaultContainer 
# Description: Checks wether the given index is a default
#	       container
# Input: index
# Output: 1 - if default container
#	  0 - otherwise
#
########################################################################
proc checkForDefaultContainer { index } {
    #
    # For the default containers, we will not let the user
    # edit it.
    #
    set defCont [ locate ContEntry.defaultContainer ]

    if { ! [ catch { toe_send $defCont getValue $index } default ] } {
	#
	# It is either 0 or 1
	#
	return $default
    }
    return 0
}

########################################################################
#
# Procedure: nodeSetValue
# Description: Returns the value set on the node
# Input: index, value
# Output: value
########################################################################
proc nodeSetValue { index value } {
    return $value
}

#######################################################################
# Procedure: checkForGlobalZone
# Description: Checks whether the given index is of the global zone
#
# Input: index - index of the zone table row
# output: 1-yes, 0-no
#
#######################################################################
proc checkForGlobalZone { index } {
    if { ! [ catch { toe_send [ locate ZoneEntry.zoneName ] getValue $index } zonename ] } {
	if { "$zonename" == "global" } {
	    return 1
	}
    }

    return 0
}


#######################################################################
#
# Procedure: createOrDeleteZone
# Description: As part of the pre-validate-set operation on the rowstatus
#   node, we either create or delete the zone based on the rowstatus
#   value.
#
# Input: index - Zone row index
#        value - Rowstatus value to confirm zone creation/deletion
# output: Success(1)/Failure(0)
#
#######################################################################
proc createOrDeleteZone { index value rowname username } {

    set result [ checkResourceAvailForZone $index $value ] 

    if { $result == 0 } {
	return 0
    }

    if { $value == 4 || $value == 5 } {
	#
	# Create zone
	#
	set result [ createZone $index $rowname  $username ]
    } else {
	if { $value == 6 } {
	    #
	    # Delete container
	    #
	    set result [ deleteZone $index $username ]
	} else {
	    ddl print warning "not a valid operation on this row"
	    return 0
	}
    }

    #
    # Can the zone config file /etc/zones/<zonename> file be edited
    # by the user outside SCM1.1 ? If so, we need to keep the 
    # timestamp of this file and update that timestamp here for this
    # operation. The admin node should have one more node to keep the
    # timestamp of /etc/zones/<zonename>.xml file.
    #

    if { $result != 0 } {
        return $value
    } else {
	return 0
    }
}

##########################################################################
# Procedure checkZoneAuth
# TO check user autherization for zone admin commands
# 
##########################################################################
proc checkZoneAuth { username } { 

    ddl print debug "checkZoneAuth"
  
    set auth "rejected"
    set curruser $username
    set ulistfd [ open "/var/opt/SUNWsymon/cfg/scm-zoneadm-users.cfg" ]
    set ulist [ gets $ulistfd ]
    set users [ split $ulist "," ]
    set iszadmin "0"
    foreach u $users {
       if { $curruser == $u } {
           if { $u == "root" } {
               set iszadmin "1"
           } else {
               set iszadmin [ isZoneManager $curruser ]
           }
           if { $iszadmin == 1 } {
               set auth "approved"
           }
           break
       }
    }
    return $auth
}

##########################################################################
# Procedure checkProjAuth
# TO check user autherization for project admin commands
#
##########################################################################
proc checkProjAuth { username } {

    ddl print debug "checkProjAuth"
 
    set auth "rejected"
    set curruser $username
    set ulistfd [ open "/var/opt/SUNWsymon/cfg/scm-zoneadm-users.cfg" ]
    set ulist [ gets $ulistfd ]
    set users [ split $ulist "," ]
    set ispadmin "0"
    foreach u $users {
       if { $curruser == $u } {
           if { $u == "root" } { 
               set ispadmin "1"
           } else {
               set ispadmin [ isProjManager $curruser ]
           }
           if { $ispadmin == 1 } {
               set auth "approved"
           }
           break
       }
    }
    return $auth
}

##########################################################################
# Procedure checkPoolAuth
# TO check user autherization for project admin commands
#
##########################################################################
proc checkPoolAuth { username } {

    ddl print debug "checkPoolAuth"

    set auth "rejected"
    set curruser $username
    set ulistfd [ open "/var/opt/SUNWsymon/cfg/scm-zoneadm-users.cfg" ]
    set ulist [ gets $ulistfd ]
    set users [ split $ulist "," ]
    set ispadmin "0"
    foreach u $users {
       if { $curruser == $u } {
           if { $u == "root" } { 
               set ispadmin "1"
           } else {
               set ispadmin [ isPoolManager $curruser ]
           }
           if { $ispadmin == 1 } {
               set auth "approved"
           }
           break
       }
    }
    return $auth
}


########################################################################## 
# Procedure: createZone 
# Description: Create a zone on the system with given resource
#   controls.
# 
# Input: index - zone row index
#        rowname - Zone Name
# Output: Success(1) - Zone creation succeeded
#         Failure(0) - to fail the row creation
########################################################################## 
proc createZone { index rowname username } {

    ddl print info "Entered createZone: $index $rowname"
    set zonename $rowname

    set auth [ checkZoneAuth $username ] 
    if { $auth == "rejected" } {
        ddl print warning "createZone: The user $username does not have zone administration pivileges"
        return 0
    }

    if { [ catch { ilookup zoneid $index } zoneid ] } {
        set zoneid "\"\""
    }

    if { [ catch { ilookup zonepath $index } zonepath ] } {
        set zonepath "\"\""
    }

    if { [ catch { ilookup zonepoolname $index } zonepool ] } {
	ddl print error "Poolname for the zone $zonename missing"
	return 0
    }

    if { [ catch { ilookup zonecpureservation $index } mincpu ] } {
	#
	# Assume 1 share if not specified
	#
        set mincpu "1"
    }

    if { [ catch { ilookup zoneipaddr $index } zoneip ] } {
	set zoneip "\"\""
    }

    if { [ catch { ilookup zonephysaddr $index } zonephys ] } {
	set zonephys "\"\""
    } 

    if { [ catch { ilookup zonehostname $index } zonehost ] } {
	set zonehost "\"\""
    } 

    if { [ catch { ilookup zonestate $index } zonestate ] } {
	set zonestate "ready"
    } 

    if { [ catch { ilookup ipbandwidth $index } ipbandwidth ] } {
	set ipbandwidth "0"     
    } 

    if { [ catch { ilookup opbandwidth $index } opbandwidth ] } {
	set opbandwidth "0"  
    } 

    if { [ catch { ilookup maxzonecpushares $index } maxzoneshares ] } {
	set maxzoneshares "100"       # 100 default
    }

    if { [ catch { ilookup zoneautoboot $index } zoneautoboot ] } {
        set zoneautoboot "false"
    }

    if { [ catch { ilookup extrazoneattr $index } extrazoneattr ] } {
        set extrazoneattr ""
    }

    set hostresentrytoe [ locate iso*PoolEntry ]
    set psrsetentrytoe [ locate iso*PsrSetEntry ]

    #
    # Get the Max CPU shares for this pool 
    #
    set maxShares [ getCpuSharesForPool $zonepool ]

    #
    # Get the processor set name associated with this pool
    #
    if { [ catch { toe_send $hostresentrytoe \
		getRowValue psetName "$zonepool" } psetname ] } {
	ddl print error "processor set not found for pool"
        return 0
    }

    #
    # Get the cpusize of the processor set associated with the given pool
    #
    if { [ catch { toe_send $psrsetentrytoe \
		getRowValue cpuSize "$psetname" } cpusize ] } {
	ddl print error "processor set has no cpus"
	return 0
    }

    #
    # Check, whether the cpusize of the processor set is zero.
    #

    if { $cpusize <= 0 } {
	ddl print warning "Resource Set has zero cpus - Can't Create Zone"
	return 0
    }

    #
    # Now, check, if the given minimum value is greater than the 
    # maxShares of the pool
    #
    if { $mincpu > $maxShares } {
	ddl print error "Given cpu value exceeds the pool max shares"
	return 0
    }

    # Now check if the given value fits in the maxShares with other 
    # zone reservations as well.
    set totalShares $mincpu
    set pooltoe [ locate iso*ZoneEntry.poolName ]
    set cputoe [ locate iso*ZoneEntry.cpuRes ]

    toe_send $pooltoe sliceforeach key tmpPool data {
                if { $key == $index } {
                   continue;
                }
                if { $tmpPool == $zonepool } {
                    set tmpcpuval [ toe_send $cputoe getValue $key ]
                    set totalShares [ expr $totalShares + $tmpcpuval ]
                }
    }

    if { $totalShares > $maxShares } {
	ddl print error "Given cpu value exceeds the pool max shares"
	return 0
    }

    set cpushares [ expr int ($mincpu) ]
    set ipbandwidth [ expr int ($ipbandwidth) ]
    set opbandwidth [ expr int ($opbandwidth) ]
#    ddl print info "Final Values: zonename: $zonename, zoneid:$zoneid
    set scriptname "zone_create.sh"
    set option "\"\""

    set result [ executeZoneScript $scriptname $option $zonename $zoneid $zonepath \
			 $zonehost $zonephys $zoneip $zonepool $cpushares $zonestate $ipbandwidth $opbandwidth $maxzoneshares $zoneautoboot $extrazoneattr ]

    #
    # Try to update the sysid information
    #
    registerOneShot 300 updateZoneSysidInfo

    #
    # wait for a sec to first configure the zone to apear
    # in the zone table.
    #
    after 1000

    return 1
}

########################################################################## 
# Procedure: deleteZone 
# Description: Delete a given zone
# 
# Input: index - zone row index
# Output: Success(1) - Zone deletion succeeded
#         Failure(0) - to fail the row deletion
########################################################################## 
proc deleteZone { index username } {

    set auth [ checkZoneAuth $username ] 
    if { $auth == "rejected" } {
        ddl print warning "deleteZone: The user $username does not have zone administration pivileges"
        return 0
    }
    set zonename [ toe_send [ locate iso*ZoneEntry.zoneName ] getValue $index ]

    set result [ executeZoneScript "zone_del.sh" "" $zonename "" "" "" "" "" "" "" "" "" "" "" "" "" ]

    set tblObj [ locate ZoneEntry ]

    if { $tblObj != "" } {
        toe_send $tblObj cleanupRow $index CLEAR_PARMS
    }

    return 1
}

########################################################################## 
# Procedure: loadZonesInfo 
# Description: Loads all the zones on the system
# Input: None
# Output: List of all the zone and their information
########################################################################## 
proc loadZonesInfo { } {
    
    set zonelist [ loadZones ]
    set result ""
    set numZones [ llength $zonelist ]

    for { set i 0 } { $i < $numZones } { incr i } {
	lextract $zonelist $i zoneinfo
	lextract $zoneinfo 0 zonename 1 zoneid 2 zonepath 3 pool 4 shares 5 ipaddr 6 phys 7 hostname 8 state 9 ipqosenabled 10 ipbw 11 opbw 12 maxshares 13 autoboot
	#
	# Convert the cpu shares to actual cpus
	#
	set cpures [ expr int ($shares) ]

        set zoneEntryToe [ locate iso*ZoneEntry ]

        if { [ catch { toe_send $zoneEntryToe \
		getRowValue ipbwUsed $zonename } oldipbwUsed ] } {
                set oldipbwUsed 0
        }

        if { [ catch { toe_send $zoneEntryToe \
		getRowValue opbwUsed $zonename } oldopbwUsed ] } {
                set oldopbwUsed 0
        }

        set kstatphys [ string trim "$phys" ]
        if { "$kstatphys" == "" } {
            set ipbwUsed $oldipbwUsed
            set ipbwUsage 0
        } elseif { [ catch { zoneBandWidthUsage $zonename $phys input $oldipbwUsed } bwData ] } {
            set ipbwUsed $oldipbwUsed
            set ipbwUsage 0
        } else {
            lextract $bwData 0 newipbwUsed 1 ipbwDelta
            if { $newipbwUsed == -1 } {
                set ipbwUsed $oldipbwUsed
            } else {
                set ipbwUsed $newipbwUsed
            }
            set ipbwUsage $ipbwDelta
        }
        
        if { "$kstatphys" == "" } {
            set opbwUsed $oldopbwUsed
            set opbwUsage 0
        } elseif { [ catch { zoneBandWidthUsage $zonename $phys output $oldopbwUsed } bwData ] } {
            set opbwUsed $oldopbwUsed
            set opbwUsage 0
        } else {
            lextract $bwData 0 newopbwUsed 1 opbwDelta
            if { $newopbwUsed == -1 } {
                set opbwUsed $oldopbwUsed
            } else {
                set opbwUsed $newopbwUsed
            }
            set opbwUsage $opbwDelta
        }

	lappend result $zonename $zoneid $zonepath "$pool" "$cpures" "$ipaddr" "$phys" "$hostname" "$state" "$ipqosenabled" "$ipbw" "$opbw" "$ipbwUsed" "$ipbwUsage" "$opbwUsed" "$opbwUsage" "$maxshares" "$autoboot" ""
        # The last empty string is for extra zone commands.
    }

    if { [ sysinfo osversion ] == "5.10" } {
        # find out the timestamp of the /etc/zones dir and /etc/zones/index file
        # if it's later than the /var/opt/SUNWsymon/scm-ipqos.cfg, apply ipqos conf
        # run ipqos config. don't worry about the status

        file stat "/etc/zones" timestamp
        set zdstamp $timestamp(ctime)
        file stat "/etc/zones/index" timestamp
        set zistamp $timestamp(ctime)
        if { [ catch { file stat "/var/opt/SUNWsymon/scm/scm-ipqos.cfg" timestamp } ] } {
            set ipqosstamp 0
        } else {
            set ipqosstamp $timestamp(ctime)
        }

        if { $ipqosstamp < $zdstamp } {
            ddl print debug "applying ipqos configuration"
            srmShellAsync ipqos-adm.sh -a
        } elseif { $ipqosstamp < $zistamp } {
            ddl print debug "applying ipqos configuration"
            srmShellAsync ipqos-adm.sh -a
        } else {
            ddl print debug "ignoring ipqos configuration"
        }

    }

    return $result
}

########################################################################## 
# Procedure: updateZonePoolName 
# Description: Updates the zone's poolname
# 
# Input: index - zone row index
#	 value - New pool name
# Output: Success(1) - Zone's pool name is updated
#         Failure(0) - failed to update the zones' poolname
########################################################################## 
proc updateZonePoolName { index value username } {
    ddl print info "updateZonePoolName: $index $value"

    set auth [ checkZoneAuth $username ] 
    if { $auth == "rejected" } {
        ddl print warning "updateZonePoolName: The user $username does not have zone administration pivileges"
        return 0
    }

    set globalZone [ checkForGlobalZone $index ]

    if { $globalZone == 1 } {
	ddl print warning "Editing on global Zone is not allowed"
	ddl print info "updateZonePoolName $index $value"
	return 0
    }

    #
    # Check whether the given pool exists either for
    # zone creation or for zone poolname modification
    #
    set result [ checkPoolExists $value ]

    if { $result == 0 } {
	ddl print warning "Poolname $value does not exists"
	return 0
    }

    set rstoe [ locate iso*ZoneEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {

	    #
	    # This is a preset operation, and the getRowValue 
	    # should give you old value. The new value is in the 
	    # value argument.
	    #
	    set oldpool [ toe_send [ locate iso*ZoneEntry.poolName ] \
				getValue $index ]

	    if { $oldpool == $value } {
		return $value
	    }

	    #
	    # Before we update the pool, make sure that, 
            # we can accomodate the current zone's cpu reservation 
	    # in the newly specified pool.
	    #

	    set pooltoe [ locate iso*ZoneEntry.poolName ]
	    set cputoe [ locate iso*ZoneEntry.cpuRes ]

	    #
	    # Get the current zone's cpu reservation
	    #
	    set mincpu [ toe_send $cputoe getValue $index ]

            set totalCpus $mincpu
	    set cpusize [ getCpusizeForPool $value ]
            set maxShares [ getCpuSharesForPool $value ]

	    toe_send $pooltoe sliceforeach key tmpPool data {
	        if { $tmpPool == $value } {
		    set tmpcpuval [ toe_send $cputoe getValue $key ]
	            set totalCpus [ expr $totalCpus + $tmpcpuval ]
	        }
            }

            if { $totalCpus > $maxShares } {
	        ddl print error "cannot accomodate new cpu reservation"
		return 0
            }

            #
            # Now, we have found that, the cpu reservation can be 
	    # given to this container Calculate the cpu-shares for 
	    # this container
            #

	    #
	    # Calculate the shares
	    #
            set cpushares [ expr int ($mincpu) ]
	    set zonename [ toe_send [ \
		  locate iso*ZoneEntry.zoneName ] getValue $index ]

	    ddl print info "Executing the zone_mod.sh for $zonename and $value"
    	    set result [ executeZoneScript "zone_mod.sh" -p $zonename "" ""\
			 "" "" "" $value "" "" "" "" "" "" "" ]
	    if { $result == 0 } {
	        ddl print error "updateZonePoolName: Could not update the pool"
		return 0
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
    	    define refreshtrap $index 1
        }
    }
    ddl print info "Returning from updateZonePoolName"
    return $value
}


########################################################################## 
# Procedure: updateZoneCpuReservation 
# Description: Update the zone's CPU reservation
# 
# Input: index - zone row index
#	 value - New CPU Reservation for the zone
# Output: Success(1) - Successfully updated the zone's CPU reservation
#         Failure(0) - Failed to update the zone's CPU reservation
########################################################################## 
proc updateZoneCpuReservation { index value username } {

    set auth [ checkZoneAuth $username ] 
    if { $auth == "rejected" } {
        ddl print warning "updateZoneCpuReservation: The user $username does not have zone administration pivileges"
        return 0
    }

    set globalZone [ checkForGlobalZone $index ]

    if { $globalZone == 1 } {
	ddl print warning "Editing on default Zone is not allowed"
	ddl print info "updateZoneCpuReservation $index $value"
	return 0
    }

    set rstoe [ locate ZoneEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {

	    set pool [ toe_send [ locate iso*ZoneEntry.poolName ] \
			getValue $index ]
	    set cpusize [ getCpusizeForPool $pool ]
            set maxShares [ getCpuSharesForPool $pool ]

	    if { $value > $maxShares } {
	        ddl print error "The value is greater than pool max shares"
		return 0
	    }

	    #
	    # Now, visit all the zones having this pool name
	    # and find out the target reservation alotted to them. If the
	    # current value does not fit into the size of the pool name,
	    # then return an error
	    #

	    set totalCpus $value
	    set pooltoe [ locate iso*ZoneEntry.poolName ]
	    set cpurestoe [ locate iso*ZoneEntry.cpuRes ]

	    toe_send $pooltoe sliceforeach key tmppool data {
		if { $key == $index } {
		    continue
		}

		if { $tmppool == $pool } {
		    set cpures [ toe_send $cpurestoe getValue $key ]
	            set totalCpus [ expr $totalCpus + $cpures ]
		}
	    }

	    if { $totalCpus > $maxShares } {
		#
		# The minimum value exceeds the maxshares of the
		# pool. Hence, we cannot set it.
		#
		ddl print error "udpateZoneCpuReservation: cannot reserve the cpu"
		return 0
	    }

	    #
	    # Now, the specified minimum value can be allocated.
	    # Try allocating it.
	    #

	    #
	    # Calculate the shares for the specified minimum value
	    #
            set shares [ expr int ($value) ]
	    set zonename [ toe_send [ locate iso*ZoneEntry.zoneName \
				] getValue $index ]

            #
            # Now call the script 
            #
	    set oldshares [ toe_send [ locate ZoneEntry.cpuRes ] getValue $index ]
            set oldshares [ expr int ($oldshares) ]
	    set sharevalues "$shares $oldshares"
		    set result [ executeZoneScript "zone_mod.sh" -c $zonename "" ""\
				 "" "" "" "" $sharevalues "" "" "" "" "" "" ]
	    if { $result == 0 } {
		ddl print error "updateZoneCpuReservation: update CPU error"
		return 0
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
    	    define refreshtrap $index 1
        }
    }

    return 1
}

########################################################################## 
# Procedure: updateMaxZoneCPUShares
# Description: Update the zone's MAX CPU Shares Configuration
# 
# Input: index - zone row index
#	 value - New Max Shares value for the zone
# Output: Success(1) - Successfully updated the zone configuration
#         Failure(0) - Failed to update the zone configuration
########################################################################## 
proc updateMaxZoneCPUShares { index value username } {

    set auth [ checkZoneAuth $username ] 
    if { $auth == "rejected" } {
        ddl print warning "updateMaxZoneCPUShares: The user $username does not have zone administration pivileges"
        return 0
    }


    set rstoe [ locate ZoneEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {

            set zone [ toe_send [ locate iso*ZoneEntry.zoneName ] \
                         getValue $index ]

	    set zonetoe [ locate iso*ContEntry.zoneName ]
	    set cputoe [ locate iso*ContEntry.cpuRes ]

	    #
	    # Now, visit all the container having this zone name
	    # and find out the target reservation alotted to them. 
	    #

            set cpuRes 0
            set totalCpus 0

	    toe_send $zonetoe sliceforeach key tmpzone data {
		if { $tmpzone == $zone } {
		    set cpures [ toe_send $cputoe getValue $key ]
	            set totalCpus [ expr $totalCpus + $cpures ]
		}
	    }

	    if { $totalCpus > $value } {
		#
		# The reservation value exceeds the new maxshares of the
		# zone. Hence, we cannot set it.
		#
		ddl print error "udpateMaxZoneCPUShares: cannot set the value $value"
		return 0
	    }

	    #
	    # Now, the specified value can be set.
	    # Try setting it.
            # call the script 
            #
	    set maxsharevalue [ expr int ($value) ]
    	    set result [ executeZoneScript "zone_mod.sh" -m $zone "" ""\
			 "" "" "" "" "" "" "" "" $maxsharevalue "" "" ]
	    if { $result == 0 } {
		ddl print error "updateMaxZoneCPUShares: update failed"
		return 0
	    }

	    #
	    # Define the slice refreshtrap. The sendRefreshValueTrap 
	    # procedure needs this slice to send the refresh trap to the 
	    # console.
	    #
    	    define refreshtrap $index 1
        }
    }

    return 1
}

##########################################################################
# Procedure: executeZoneScript
# Description: To execute script related to zones
# Input: Script name - name of the script
#        option - options for scripts
#        zonename - Zone Name
#        zoneid   - Zone ID
#        zonepath - Zone Path where zone exists
#        zonehost - Zone's hostname
#        zonephys - Zone's physical interface
#        zonepool - Zone's poolname
#        zonecpu  - Zone's CPU reservation in terms of shares
#        zonestate- State of the zone (ready, running, halt, boot )
#
# Output: 1 on success, and 0 on failure
##########################################################################
proc executeZoneScript { scriptname option zonename zoneid zonepath \
			 zonehost zonephys zoneip zonepool zonecpu zonestate \
                         ipbandwidth opbandwidth maxzoneshares zoneautoboot \
                         extrazoneattr } {

    ddl print info "executeZoneScript: $zonename $zoneid $zonepath $zonehost $zonephys $zoneip $zonepool $zonecpu $zonestate $ipbandwidth $opbandwidth $maxzoneshares $zoneautoboot $extrazoneattr"
    global env

    switch -exact -- $scriptname {
	zone_create.sh {
	    ddl print info "executing the zone_create.sh from $env(ESROOT)"
    	    set result [ srmShellAsync $env(ESROOT)/modules/sbin/zone_create.sh \
		-n "$zonename" -r "$zonepath" -i "$zoneip" -h "$zonehost" \
		-e "$zonephys" -p "$zonepool" -c "$zonecpu" -o "$zonestate" \
                -I "$ipbandwidth" -O "$opbandwidth" -m "$maxzoneshares" \
                -a "$zoneautoboot" -E "$extrazoneattr" ]
	}

	zone_del.sh {
    	    set result [ srmShellAsync  $env(ESROOT)/modules/sbin/zone_del.sh $zonename ] 
	}

	zone_mod.sh {
	    switch -exact -- $option {
		-p {
    	            set result [ srmShellAsync $env(ESROOT)/modules/sbin/zone_mod.sh -n $zonename -p $zonepool ] 
		}

		-c {
		    lextract $zonecpu 0 newshares 1 oldshares
    	            set result [ srmShellAsync $env(ESROOT)/modules/sbin/zone_mod.sh -n $zonename -c $newshares -o $oldshares ] 
		}
		-m {
    	            set result [ srmShellAsync $env(ESROOT)/modules/sbin/zone_mod.sh -n $zonename -m $maxzoneshares ] 
		}
	    }
	}

	zone_halt.sh {
    	    set result [ srmShellAsync $env(ESROOT)/modules/sbin/zone_halt.sh $zonename ] 
	}

	zone_boot.sh {
    	    set result [ srmShellAsync $env(ESROOT)/modules/sbin/zone_boot.sh $zonename ] 
	}
    }

    if { $result == 0 }  {
	#
	# Success
	#
	return 1
    } else  {
	#
	# Failed
	#
	return 0
    }
}

########################################################################## 
# Procedure: changeZoneState 
# Description: Change the zone state accordingly
# 
# Input: index - zone row index
#	 value - New state
# Output: Success(1) - Successfully changed the state
#         Failure(0) - Failed to change the state
########################################################################## 
proc changeZoneState { index value username } {

    ddl print info "changeZoneState: $index $value"

    set auth [ checkZoneAuth $username] 
    if { $auth == "rejected" } {
        ddl print warning "changeZoneState: The user $username does not have zone administration pivileges"
        return 0
    }


    set globalZone [ checkForGlobalZone $index ]

    if { $globalZone == 1 } {
	ddl print warning "Global Zone state cannot be changed"
	return 0
    }

    set rstoe [ locate ZoneEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {
	   set zonename [ toe_send [ locate ZoneEntry.zoneName ] getValue $index ]
	   ddl print info "Executing switch statement"

	   switch -exact -- $value {
	       halt {
		   set scriptname "zone_halt.sh"
	       }

	       boot {
		   set scriptname "zone_boot.sh"
	       }
	   }

	   ddl print info "changeZoneState: executing $scriptname"
    	   set result [ executeZoneScript $scriptname "" $zonename "" "" \
			 "" "" "" "" "" "" "" "" "" "" "" ]
	   if { $result == 0 } {
	        ddl print error "changeZoneState: Could not change the zone state"
		return 0
	   }
	}
    }

    ddl print info "Exiting changeZoneState"

    return $value
}


proc checkRowStatus { tableEntry } {
#    ddl print info "checkRowStatus $tableEntry"
    set tDepth [ toe_send [ locate $tableEntry ] getTableDepth ]
    set rstoe [ locate $tableEntry.rowstatus ]

    for { set index 1 } { $index <= $tDepth } { incr index } {

        if { [ toe_csend val $rstoe getValue $index ] } {
	    #
	    # The rowstatus is not set for this row
	    # Set it to active
	    #
	    toe_send $rstoe setValue $index 1
        }
    }

    #
    # as a result of changing the system configuration, like,
    # deleting a pool, the row in this table may not appear, but, the rowstatus
    # would still appear as 1, so, clear them too
    #
    while { 1 } {
	if { [ catch { toe_send $rstoe getValue $index } val ] } {
	    break
	} else {
	    #
	    # clear the rowstatus node
	    #
	    toe_send $rstoe setValue $index 6

	    incr index
	}
    }
#    ddl print info "Exiting checkRowStatus"
}

#######################################################################
#
# Procedure: checkResourceAvailForZone
# Description: 
#
# Input: index - Zone row index
#        Value - rowstatus to confirm row creation operation
# Output: Success(1) - If specified pool accomdates given 
#                      resources
#         Failure (0) - If there are no enough resources
#######################################################################

proc checkResourceAvailForZone { index value } {
    ddl print info "Entered checkResourceAvailForZone: $index $value"

    if { $value == 6 } {
	#
	# When row is deleted, we need not perform this check
	#
	return 1
    }

    #
    # Get the pool name for this zone. 
    #
    if { [ catch { ilookup zonepoolname $index } pool ] } {
	ddl print error "poolname is not defined for index $index"
	return 0
    }

    #
    # Get the minimum cpu value for this Zone
    #
    if { [ catch { ilookup zonecpureservation $index } mincpu ] } {
	ddl print error "CPU reservation is not defined for index $index"
	return 0
    }

    #
    # Get the cpusize of the processor set associated with this pool 
    #
    set cpusize [ getCpusizeForPool $pool ]

    #
    # Get the Max CPU shares for this pool 
    #
    set maxShares [ getCpuSharesForPool $pool ]

    if { $cpusize == 0 } {
	ddl print warning "CPU Size is zero for pool : $pool"
	undefine zonecpureservation $index
	undefine zonepoolname $index
	return 0

    } else {
        set totalCpus $mincpu
	set pooltoe [ locate ZoneEntry.poolName ]
	set cpurestoe [ locate ZoneEntry.cpuRes ]

	toe_send $pooltoe sliceforeach key tmpPool data {
	    if { $key == $index } {
	        continue
	    }

	    if { $tmpPool == $pool } {
	        set mincpuval [ toe_send $cpurestoe getValue $key ] 
	        set totalCpus [ expr $totalCpus + $mincpuval ]
	    }
        }

        if { $totalCpus > $maxShares } {
	    ddl print error "checkResourceAvailForZone: cannot reserve cpu"
	    undefine zonecpureservation $index
	    undefine zonepoolname $index
	    return 0
        }
    }

    return 1
}

#######################################################################
#
# Procedure: 
# Description: 
#
# Input: 
# Output: 
#######################################################################
#proc runScript { scriptname, option, zonename zoneid, zonepath,\
#		 zonehost, zonephys, zonepool, shares, zonestate }
#{
#   set result  [ executeZoneScript $scriptname $option $zonename $zoneid $zonepath \
#			 $zonehost $zonephys $zonepool $shares $zonestate ]
#   return
#}


proc loadZoneContainers { } {
    set result ""
    set contentrytoe [ locate ContEntry ]
    set depth [ toe_send $contentrytoe getTableDepth ]
    set rowIdx 1

    #
    # Get the containers in all the zones
    # The returned valued is in the below form:
    # { zoneid { { name id pool memcap users groups comment procsRunning } {...} ... } }
    # { zoneid { { name id pool memcap users groups comment procsRunning } {...} ... } }
    #

    set zones [ getAllZonesProjects ]

    #
    # Now get the usage of all the projects in all the zones
    # The usage format is as below:
    # { zoneid { { proj cpuusage memusage } { proj cpuusage memusage } ... } }
    # { zoneid { { proj cpuusage memusage } { proj cpuusage memusage } ... } }
    #
    set numzones [ toe_send [ locate iso*ZoneEntry ] getTableDepth ]
    set zonesUsages [ getAllZonesUsage $numzones ]
    set length [ llength $zones]

    for { set idx 0 } { $idx < $length } { incr idx } {
#    foreach zone $zones {
	lextract $zones $idx zone
	lextract $zone 0 zoneid 1 projects

	if { [ catch { getZoneName $zoneid } zoneName ] } {
	    set zoneName ""
	}

	set pos [ binarysearch $zonesUsages $zoneid ]

	if { $pos != -1 } {
	    lextract $zonesUsages $pos zoneUsage
	    lextract $zoneUsage 0 dummyZoneId 1 projectsUsages
	} else {
	    set projectsUsages {}
	}


	#
	# Here we should also get the rcapstat info
	# for each zone, by entering the zone and getting
	# the details. For the prototype, this part is
	# skipped due to time constraint. Currently, we
	# use only the resource usage of all the projects by
	# scanning the /proc directory.
	# ????? IMPLEMENT THE RCAPSTAT FOR EACH ZONE ??????
	#

	#
	# Find the default pool of this zone
	#
	set zonePool [ getZonePoolname $zoneid ]

	foreach project $projects {
	    lextract $project 0 projname 1 projid 2 pool 3 shares 4 memcap \
			      5 users 6 groups 7 comment 8 procsRunning 9 maxshm

            set pool [ string trim $pool ]
	    if { $pool == "" } {
		set pool $zonePool
	    } else {
		#
		# check for the pool existence
		#
		set poolexists [ checkPoolExists $pool ]
		if { $poolexists == 0 } {
		    set pool $zonePool
		}
	    }

	    if { $pool == "" } {
	        set poolsize 0
	        set maxShares 0
	    } else {
	        set poolsize [ getCpusizeForPool $pool ]
	        set maxShares [ getCpuSharesForPool $pool ]
	    }

	    if { $shares == -1 } {
	        set shares 1
	    }


	    set cpures [ expr int ($shares) ]

	    #
	    # It's time to find the values for container name, 
	    # container id, etc. as described in the description above.
	    #

	    set rowname [ toe_send $contentrytoe getRowName $rowIdx ]

	    if { $depth >= 1 && $rowIdx <= $depth } {

	        #
	        # Get the existing values 
	        #

	        set contName [ toe_send $contentrytoe getRowValue contName \
	    			    $rowname ]
	        set contID [ toe_send $contentrytoe getRowValue contID $rowname ]
	        set matchExpr [ toe_send $contentrytoe getRowValue matchexpr \
				$rowname ]
	        set primaryUsers [ toe_send $contentrytoe getRowValue primaryUsers \
				$rowname ]
	    } else {
	        #
	        # If this condition is true, then, either we have no 
	        # pre-existing containers or some projects are added on the 
	        # command line. In such cases, make the container name, 
	        # container id, and other values, as described above
	        #


	        set contID ""
	        set matchExpr ""
	        set primaryUsers ""
		set contName ""
	    }

	    #
	    # The default projects, system, default, user.root,
	    # noproject, and group.staff have been given default
	    # container names by the service layer. The following
	    # logic assigns the default container name for the 
	    # default projects. If the project name is none of the
	    # default projects, then the container name would be
	    # project name.
	    #

	    if { $projname == "system" } {
	        set defaultContainer 1
	        set contName "System Processes"
	    } elseif { $projname == "default" } {
	        set defaultContainer 1
	        set contName "Default"
	    } elseif { $projname == "user.root" } {
	       set defaultContainer 1
	       set contName "Root User"
	    } elseif { $projname == "group.staff" } {
	        set defaultContainer 1
	        set contName "Users with Group Staff"
	    } elseif { $projname == "noproject" } {
		set defaultContainer 1
		set contName "Processes with No Project"
	    } else {
		set defaultContainer 0
		if { $contName != "" } {
		    set contName $projname
		}
	    }
	    # 
	    # Get the system memory and convert it to MB
	    #
	    set sysMemory [ expr [ sysinfo hwmemory ] / 1024 ]

	    #
	    # Get the rcapstat information from the list for this project id
	    #

	    #
	    # Virtual memory is set to 0, since, we are not
	    # collecting the rcapstat info from each of the zones
	    # once, we implement that feature, we have to fill this
	    # value ???????????? TO BE IMPLEMENTED ?????????????
	    #
	    set virtualmem "0"
	    set pos [ binarysearch $projectsUsages $projid ]
	    if { $pos == -1 } {
	        set cpuused 0
	        set cpuPctUsed 0.0
	        set cpuROI 0.0
	        set rss "0"
		set memROI 0.0
		set memPctUsed 0.0
	    } else {
	        lextract $projectsUsages $pos cpumemusage
	        lextract $cpumemusage 1 cpuPctUsed 2 rss
	        set onlinecpucount [ getOnlineCpuCount ]
	        set cpuused [ expr ( $cpuPctUsed * $onlinecpucount ) / 100 ]

                # We need to calculate return on investment
                # i.e. cpu used against cpu reserved.
                # The shares for absolute 1 CPU is maxShares / poolsize
	        if { $poolsize > 0 } {
                    set onecpu [ expr $maxShares / $poolsize ]
                } else {
                    set onecpu 0
                }

                # cpu shares used is
                set cpusharesused [ expr $cpuused * $onecpu ]

	        #
	        # It's possible that, the user has made the shares to 0 by
	        # editing the /etc/project entry, and that would cause 
	        # divide by zero error for the below operation.
	        #
	        if { $cpures > 0.0 } {
	            set cpuROI [ expr ( $cpusharesused * 100.0 ) / $cpures ]
	        } else {
		    set cpuROI 0.0
	        }

	        #
	        # Now calculate the percentage value. The rss value
	        # is made float to capture less than 1% of the memory. 
	        #
	        set memPctUsed [ expr ( $rss * 100.0 ) / $sysMemory ]
                if { [ isZero64 $memcap ] > 0 } {
	            set memROI [ expr ( $rss * 100.0 ) / $memcap ]
	        } else {
	            set memROI 0.0
	        }
	    }

	    incr rowIdx

	    lappend result $zoneName $contID "$contName" $projid $projname "$primaryUsers" \
		$pool $poolsize $cpures $cpuused $cpuPctUsed $cpuROI $memcap $virtualmem $rss \
		$memPctUsed $memROI "$users" "$groups" "$comment" "$matchExpr" $procsRunning $defaultContainer $maxshm
	}
    }

    return $result
}


proc getZonePoolname { zoneid } {
    set zoneIDToe [ locate iso*ZoneEntry.zoneID ]
    set poolNameToe [ locate iso*ZoneEntry.poolName ]

    set poolToReturn ""

    toe_send $zoneIDToe sliceforeach key idval data {
        if { $zoneid == $idval } {
            set poolToReturn [ toe_send $poolNameToe getValue $key ]
        }
    }
    if { $poolToReturn == "" } {
        set poolToReturn "pool_default"
    }
    return $poolToReturn
}

proc refreshSysidTable { } {

    set result ""
    set depth [ toe_send [ locate SysidEntry ] getTableDepth ]

    for { set idx 1 } { $idx <= $depth } { incr idx } {
	set zoneName [ toe_send [ locate SysidEntry.zoneName ] getValue $idx ]
	set hostName [ toe_send [ locate SysidEntry.hostName ] getValue $idx ]
	set locale [ toe_send [ locate SysidEntry.locale ] getValue $idx ]
	set rootPasswd [ toe_send [ locate SysidEntry.rootPasswd ] getValue $idx ]
	set termType [ toe_send [ locate SysidEntry.termType ] getValue $idx ]
	set nameSvc [ toe_send [ locate SysidEntry.nameService ] getValue $idx ]
	set tz [ toe_send [ locate SysidEntry.timeZone ] getValue $idx ]

	lappend result $zoneName $hostName $locale $rootPasswd $termType $nameSvc $tz
    }

    return $result
}

proc updateZoneSysidInfo { } {
    ddl print info "updateZoneSysidInfo entered"
    set timer [ ilookup -d "" value sysidtimer ]

    if { $timer != "" } {
       undefine value sysidtimer
    }

    #
    # See for entries in the sysid table. If found
    # configure that zone with the sysid info
    #
    set depth [ toe_send [ locate iso*SysidTable.SysidEntry ] getTableDepth ]
    set deleteList ""

    for { set idx 1 } { $idx <= $depth } { incr idx } {
	set zoneName [ toe_send [ locate iso*SysidTable.SysidEntry.zoneName ] getValue $idx ]
	set hostName [ toe_send [ locate iso*SysidTable.SysidEntry.hostName ] getValue $idx ]
	set locale [ toe_send [ locate iso*SysidTable.SysidEntry.locale ] getValue $idx ]
	set passwd [ toe_send [ locate iso*SysidTable.SysidEntry.rootPasswd ] getValue $idx ]
	set term [ toe_send [ locate iso*SysidTable.SysidEntry.termType ] getValue $idx ]
	set ns [ toe_send [ locate iso*SysidTable.SysidEntry.nameService ] getValue $idx ]
	set tz [ toe_send [ locate iso*SysidTable.SysidEntry.timeZone ] getValue $idx ]

	#
	# Set the sysid info only if the zone directory is available
	#
        set zonebase [ toe_send [ locate iso*ZoneEntry ] getRowValue zonePath $zoneName ]
	set zonediravl [ file exists $zonebase/root/etc ]

        
	if { $zonediravl == 1 } {
	    ddl print info "Setting the sysid info for zone $zoneName"
            global env
    	    set result [ srmShellAsync $env(ESROOT)/modules/sbin/zone_sysidcfg.sh \
		-n "$zoneName" -h "$hostName" -l "$locale" -p "$passwd" \
		-t "$term" -s "$ns" -z "$tz" ]

	    if { $result != 0 } {
	        ddl print error "Error in setting the sysid info for zone $zoneName"
	    } else {
	        #
	        # Mark the row for deletion
                ddl print info "updateZoneSysidInfo: cleaning up zone $zoneName sysidcfg"
	        #
	        lappend deleteList $idx
	    }
        }
    }

    #
    # Once we are done with updating the sysid info for each
    # zone, we can delete the corresponding rows in sysid table
    #
    foreach row $deleteList {
	set rowstatustoe [ locate iso*SysidEntry.rowstatus ]
	toe_send $rowstatustoe setValue 1 6
    }

    #
    # Now see if there are some rows in this table. If not,
    # timer is not needed
    #
    if { [ toe_send [ locate iso*SysidTable.SysidEntry ] getTableDepth ] > 0 } {
        define value sysidtimer [ registerOneShot 300 \
		updateZoneSysidInfo ]
    }
}

proc createSysidRow { index value rowname } {
    #ddl print info "Entered createSysidRow idx = $index, val = $value, rowname = $rowname"
    refreshValueAndTrap
    return $value
}

#######################################################################
#
# Procedure: moveProcess
# Description: 
#
# Input:  value - the process id of the process to be moved.
# Output: Success(1) - If the process is moved successfully
#         Failure (0) - If the process move fails.
#######################################################################

proc moveProcess { value username } {
    #ddl print info "Entered moveProcess: $value"
    if { [ sysinfo osversion ] == "5.8" } {
        ddl print warning "moveProcess: Operation not supported on Solaris 8"
        return 0
    }

    set curruser [ checkProjAuth $username ]
    if { $curruser == "rejected" } {
        ddl print info "moveProcess: Invalid access. Permission denied for user to complete this operation"
        return 0
    }

    if { [ catch { set pid [ lindex [ split $value : ] 0 ] } spliterr ] } {
        ddl print warning "moveProcess: Invalid parameters"
        return 0
    }

    if { [ catch { set projname [ lindex [ split $value : ] 1 ] } spliterr ] } {
        ddl print warning "moveProcess: Invalid parameters"
        return 0
    }

    if { [ catch { set zonename [ lindex [ split $value : ] 2 ] } spliterr ] } {
        ddl print warning "moveProcess: Invalid parameters"
        return 0
    }

    if { $pid == "" } {
        ddl print warning "moveProcess: Invalid parameters"
	return 0
    }

    if { $projname == "" } {
        ddl print warning "moveProcess: Invalid parameters"
	return 0
    }

    if { $zonename == "" } {
        ddl print warning "moveProcess: Invalid parameters"
        return 0
    }

    if { [ sysinfo osversion ] == "5.9" } {
        set result [ srmShell newtask -c $pid -p $projname ]
    } elseif { [ sysinfo osversion ] == "5.10" } {
        if { $zonename == "global" } {
            set result [ srmShell newtask -c $pid -p $projname ]
        } else {
            set result [ srmShell zlogin $zonename newtask -c $pid -p $projname ]
        }
    }

    unset pid projname zonename

    return $result
}

#######################################################################
# Procedure: updateipBandWidth
# Description: Update the inbound committed_rate and committed_burst IPQoS 
#              parameters in the configuration file
# Input : The bandwidth threashold value
# Output: The bandwidth value if set, Default (100 mbps) if not set.
#
#######################################################################

proc updateipBandWidth { index value username } {

    ddl print info "updateipBandWidth: $index $value"

    set auth [ checkZoneAuth $username ] 
    if { $auth == "rejected" } {
        ddl print warning "updateipBandWidth: The user $username does not have zone administration pivileges"
        return 0
    }


    set value [ expr int ($value) ]
    set rstoe [ locate iso*ZoneEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {

            # Get the zonename for which ipqos is being configured.
    
            set zonename [ toe_send [ \
        	  locate iso*ZoneEntry.zoneName ] getValue $index ]

            set result [ srmShellAsync "ipqos-adm.sh" -u $zonename "scm.zone.inputbw" $value ] 
            if { $result != 0 } {
                ddl print error "updateipBandWidth: Could not update the input committed_rate"
        	return 0
            }

            #
            # Define the slice refreshtrap. The sendRefreshValueTrap 
            # procedure needs this slice to send the refresh trap to the 
            # console.
            #
            refreshNupdateContainer $index
        }
    }

    if { $value == 0 } {
        return " "
    }
    return $value
}

#######################################################################
# Procedure: updateopBandWidth
# Description: Update the inbound committed_rate and committed_burst IPQoS 
#              parameters in the configuration file
# Input : The bandwidth threashold value
# Output: The bandwidth value if set, Default (100 mbps) if not set.
#
#######################################################################

proc updateopBandWidth { index value username } {

    ddl print info "updateopBandWidth: $index $value"

    set auth [ checkZoneAuth $username ] 
    if { $auth == "rejected" } {
        ddl print warning "updateopBandWidth: The user $username does not have zone administration pivileges"
        return 0
    }

    set value [ expr int ($value) ]
    set rstoe [ locate iso*ZoneEntry.rowstatus ]

    if { ! [ catch { toe_send $rstoe getValue $index } rowstatus ] } {
        if { $rowstatus == 1 } {
            # Get the zonename for which ipqos is being configured.
    
            set zonename [ toe_send [ \
	          locate iso*ZoneEntry.zoneName ] getValue $index ]

            set result [ srmShellAsync "ipqos-adm.sh" -u $zonename "scm.zone.outputbw" $value ] 
            if { $result != 0 } {
                ddl print error "updateopBandWidth: Could not update the input committed_rate"
        	return 0
            }

            #
            # Define the slice refreshtrap. The sendRefreshValueTrap 
            # procedure needs this slice to send the refresh trap to the 
            # console.
            #
            refreshNupdateContainer $index
        }
    }
    if { $value == 0 } {
        return " "
    }
    return $value
}

################################################################################
# Procedure: getInterfaceList
#            The procedure reads the MIB2Instrumentation module data
#            and returns the comma separated list the table values.
#
# Output   : list of comma separated network interfaces 
#
################################################################################
proc getInterfaceList {} {
        set ifList ""
        toe_send [locate iso*ifDescr ] sliceforeach key val data {
                if { ! [ string match $val "lo0" ] } {
                        if { [ string length $ifList ] != 0 } {
                                append ifList ","
                        }
                        append ifList $val
                }
        }
        return "$ifList"
}

proc getLocaleList {} {
    set lcList ""
    return "C,zh,zh_TW,ko,fr"
    
}

proc getTermTypeList {} {
    set ttList ""
    return "dtterm,xterms,vt100,ansi,heath,zenith,pc,sun-cmd,sun,tvi910,tvi925,wy50"

}

proc getTimezoneList {} {
    set tzList ""
    return "US/Eastern,US/Pacific,US/Central,Asia/Singapore,Asia/Calcutta"
}

