/*
 * Copyright 2004-2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */
package samples.sfsbfailover.client;

import java.io.*;
import java.util.*;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.rmi.PortableRemoteObject;

import samples.sfsbfailover.tools.*;
import samples.sfsbfailover.ejb.*;

/**
 * A simple java client. This demonstrates the use of stateful session bean. In this process
 * the client first creates a cart object, adds few books to that cart, lists the content of
 * the cart, removes few books from the cart. Overall, it demonstrates how the session bean
 * retains state information across method calls and within the same session.
 * <p>Client does the following, in order
 * <ul>
 * <li>Locates the home interface of the enterprise bean
 * <li>Gets a reference to the remote interface
 * <li>Invokes business methods
 * </ul>
 * <br>
 * <b>Locating the home interface:</b>
 * <blockquote><pre>
 *	Context initial = new InitialContext();
 *	Object objref = initial.lookup("java:comp/env/ejb/Cart");
 *	CartHome home = (CartHome)PortableRemoteObject.narrow(objref, CartHome.class);
 * </pre></blockquote>
 * <br>
 * <b>Creating the remote interface:</b>
 * <blockquote><pre>
 *	Cart shoppingCart = home.create();
 * </pre></blockquote>
 * <br>
 * <b>Invoking business methods:</b>
 * <blockquote><pre>
 *	...
 *	shoppingCart.addBook("The Martian Chronicles");
 *	...
 *	shoppingCart.removeBook("Alice in Wonderland");
 *	...
 * </pre></blockquote>
 * <br>
 *
 * <b>Output:</b>
 * <pre>
 *  Cart Contents before failover.....
 *	The Martian Chronicles
 *	2001 A Space Odyssey
 *	The Left Hand of Darkness
 *
 * 
 *	Alice in Wonderland not in cart.
 * </pre>
 *
 *
 */
public class CartClient {
    
    /**
     * The main method of the client. This invokes the <code>CartBean</code> to use
     * its services. It then asks the stateful bean to add few books, list the contents
     * of the cart and remove few books from the cart.
     * See <code>appclient</code> documentation in SunONE app server to run the clinet.
     *
     */
    public static void main(String[] args) {
        try {
            Properties properties = new Properties();
            boolean acc = false;
            Context ic = null;
            Object objref = null;
            String title = null;
            
            if (args.length == 0) {
                acc = true;
            }
            
            if (!acc) {
                String key;
                String value;
                int index;
                
                for(int i=0; i < args.length; i++) {
                    key = null;
                    value = null;
                    index = args[i].indexOf("=");
                    
                    if(index > -1) {
                        key = args[i].substring(0,index);
                        value = args[i].substring(index+1);
                        
                        if(key != null && value != null)
                            properties.put(key, value);
                    }
                }
                
                ic = new InitialContext(properties);
                
            } else {
                Context ic1 = new InitialContext();
                ic = (Context)ic1.lookup("java:comp/env");
            }
            
            System.out.println("\nCreated Initial Context");
            waitForUserInput("Press <Enter> to continue....");
            
            //do a lookup on the ejb
            objref = ic.lookup("ejb/Cart");
            CartHome home = (CartHome)PortableRemoteObject.narrow(objref, CartHome.class);
            System.out.println("\nObtained EJB home reference object.");
            waitForUserInput("Press <Enter> to continue....");
            
            Cart shoppingCart = home.create();
            System.out.println("\nCreated EJB remote reference object.");
            waitForUserInput("Press <Enter> to continue....");
            
            invokeBusinessMethods(shoppingCart);
            
        } catch (Exception ex) {
            System.err.println("Caught an unexpected exception!");
            ex.printStackTrace();
            System.exit(1);
        }
        
        System.exit(0);
    }
    
    private static void invokeBusinessMethods(Cart shoppingCart) {
        String option = null;
        String bookName = null;
        Vector bookList = null;
        Enumeration enumer = null;
        
        do {
            System.out.println("\nDo you want to - ");
            System.out.println("1. Add a book to the cart ");
            System.out.println("2. Remove a book from the cart ");
            System.out.println("3. List cart contents ");
            System.out.println("4. Exit ");
            option = waitForUserInput("Please specify the option....");
            
            try {
                if(option != null) {
                    if(option.equals("1")) {
                        bookName = waitForUserInput("\nEnter the name of the book to be added...");
                        if((bookName != null) && (bookName.trim().length() > 0)) {
                            shoppingCart.addBook(bookName);
                            System.out.println("Added the book to the cart.");
                            
                        } else
                            System.out.println("\nInvalid book name");
                        
                    } else if(option.equals("2")) {
                        bookName = waitForUserInput("\nEnter the name of the book to be removed...");
                        if((bookName != null) && (bookName.trim().length() > 0)) {
                            shoppingCart.removeBook(bookName);
                            System.out.println("Removed the book from the cart.");
                        } else
                            System.out.println("\nInvalid book name");
                        
                    } else if(option.equals("3")) {
                        bookList = shoppingCart.getContents();
                        System.out.println("\nCart contents....");
                        
                        enumer = bookList.elements();
                        
                        while (enumer.hasMoreElements()) {
                            bookName = (String) enumer.nextElement();
                            System.out.println(bookName);
                        }
                    }
                }
                
            } catch (BookException bookEx) {
                System.err.println(bookEx.getMessage());
                
            } catch (Exception ex) {
                System.err.println("Caught an unexpected exception!");
                ex.printStackTrace();
                System.exit(1);
            }
            
        } while((option == null) || (! option.equals("4")));
    }
    
    private static String waitForUserInput(String msg) {
        String line = "";
        System.out.println(msg);
        //		System.out.println("Press <Enter> to continue.....");
        
        try {
            BufferedReader reader = new BufferedReader(
            new InputStreamReader(System.in));
            line=reader.readLine();
            
        } catch(Exception e) {
            System.out.println("Exception occured while waiting for <Enter>: " +
            e.toString());
        }
        
        return line;
    }
}
