/*
 * Copyright 2004-2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */
package samples.mqfailover.client;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.Properties;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.jms.Topic;
import javax.jms.Queue;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicPublisher;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;

import javax.naming.Context;
import javax.naming.InitialContext;

import javax.swing.JOptionPane;

/**
 * A simple java client to demonstrate Message failover in SJSAS 8.1 SE/EE. 
 * In this sample, the client first looks up a Topic Connection Factory, and a 
 * Topic called PublishTopic and publishes 2 text messages to the publishtopic, whose
 * responses need to be sent back to a ReplyQueue.
 *
 * The client then pops a dialog that prompts the user to kill one or more broker instance in
 * the broker cluster
 *
 * An CMT MDB that listens to the publishTopic, on receipt of a message from the appclient,
 * sleeps for 30 seconds so that MQ brokers can be killed to simulate broker failure/shutdown. 
 * The MDB then sends back a reply to the ReplyQueue.
 * 
 * Another CMT MDB that listens to the ReplyQueue shows a dialog when a message is received. 
 * It can be onserved that JMS messages addressed to a Topic, are delivered in a random
 * load-balanced manner to MDBs in the different application server instances in the 
 * application server cluster. JMS Messages delivery are failed over to alternate brokers 
 * even when brokers are killed/restared during delivery to CMT MDBs.
 *
 * @author Sivakumar Thyagarajan
 */
public class MQFailoverTestClient {
    
    private static Context ic = null;
    private static int NUMBER_OF_MESSAGES = 2;
    private static String messageText = " MQ Failover Test Message ";

    public static void main(String[] args) {
        try {
            Properties properties = new Properties();
            boolean acc = false;
            Object objref = null;
            String title = null;
            
            if (args.length == 0) {
                acc = true;
            }
            
            if (!acc) {
                String key;
                String value;
                int index;
                
                for(int i=0; i < args.length; i++) {
                    key = null;
                    value = null;
                    index = args[i].indexOf("=");
                    
                    if(index > -1) {
                        key = args[i].substring(0,index);
                        value = args[i].substring(index+1);
                        
                        if(key != null && value != null)
                            properties.put(key, value);
                    }
                }
                
                ic = new InitialContext(properties);
                
            } else {
		System.out.println("No properties. Not Initialized to comp/env");
                ic = new InitialContext();
                //Context ic1 = new InitialContext();
                //ic = (Context)ic1.lookup("java:comp/env");
            }
            
            System.out.println("\nCreated Initial Context");
            waitForUserInput("Press <Enter> to continue....");
            
            //sendTestMessages
            sendTestMessages();
            
        } catch (Exception ex) {
            System.err.println("Caught an unexpected exception!");
            ex.printStackTrace();
            System.exit(1);
        }
        
        System.exit(0);
    }


    /**
     * Send test messages to the publish Topic
     */
    private static void sendTestMessages() {
        Topic publishTopic = null; 
        Queue receiveQueue = null;
        TopicConnectionFactory factory = null;
        TopicSession topicSession = null;
        TopicConnection topicConnection = null;
        TopicPublisher publisher = null;
 
        

            System.out.println("\n Publishing messages");
        
            for(int i = 0; i < NUMBER_OF_MESSAGES; i++) {
            try {
            //Lookup TCF and Publish Topic, Receive Queue
            factory = (TopicConnectionFactory) ic.lookup("java:comp/env/jms/PublishClientTCF");
            System.out.println("*** After getting factory");
            topicConnection = factory.createTopicConnection();
            System.out.println("*** After getting connection");
            
            
            publishTopic = (Topic) ic.lookup("java:comp/env/jms/publishTopic");
            receiveQueue =(Queue) ic.lookup("java:comp/env/jms/receiveQueue");
            topicSession = topicConnection.createTopicSession(false, Session.AUTO_ACKNOWLEDGE);
            publisher = topicSession.createPublisher(publishTopic);
            topicConnection.start();

            System.out.println("Sending message #: " + i);
            TextMessage sendMessage = topicSession.createTextMessage( messageText + i);
            sendMessage.setText("MQ Failover Demo Test Message # " + i);
            sendMessage.setJMSReplyTo(receiveQueue);
            publisher.publish(sendMessage);
 
            if (i % 2 != 0) { 
                      JOptionPane.showConfirmDialog(null, "Perform one of the following to demonstrate JMS Failover." +
                                "\n Kill a JMS broker and restart the broker instance. \n OR \n " +
                                "Stop an application server instance and restart it",
                                "JMS Failover sample",
                                JOptionPane.DEFAULT_OPTION,
                                JOptionPane.INFORMATION_MESSAGE);
                      try {
                          System.out.println("Sleeping for 30 seconds");
                          Thread.sleep(30 * 1000);
                      } catch (InterruptedException ex) {
                          //ignore
                      }

            }
            
        } catch(Exception e) {
            System.err.println("Error while publishing messages :: " 
                            + e.getMessage());
            e.printStackTrace();
        } finally {
            try {
                if (topicSession != null ){
                    topicSession.close();
                }
                if (topicConnection != null) {
                    topicConnection.close();
                }
            } catch (JMSException jmsex) {
                System.err.println("Error while closing connection/session :: " 
                                + jmsex.getMessage());
                jmsex.printStackTrace();
            }
        }
        }
    }
    
    private static String waitForUserInput(String msg) {
        String line = "";
        System.out.println(msg);
        try {
            BufferedReader reader = new BufferedReader(
            new InputStreamReader(System.in));
            line=reader.readLine();
            
        } catch(Exception e) {
            System.out.println("Exception occured while waiting for <Enter>: " +
            e.toString());
        }
        
        return line;
    }
}
