/*
 * Copyright 2004-2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

package com.sun.enterprise.management.agent.ws;

import java.util.Set;

import javax.management.AttributeList;
import javax.management.Attribute;
import javax.management.MBeanInfo;
import javax.management.QueryExp;

import java.rmi.RemoteException;

 /**
 * @author Hans Hrasna
 */
public interface ManagementWSIF extends java.rmi.Remote {

    /**
     * Gets the names of managed objects controlled by the MEJB. This method
     * enables any of the following to be obtained: The names of all managed objects,
     * the names of a set of managed objects specified by pattern matching on the
     * <CODE>ObjectName</CODE>, a specific managed object name (equivalent to
     * testing whether a managed object is registered). When the object name is
     * null or no domain and key properties are specified, all objects are selected.
     * It returns the set of J2EEObjectNames for the managed objects selected.
     *
     * @param objectName The object name pattern identifying the managed objects to be retrieved. If
     * null or no domain and key properties are specified, all the managed objects registered will be retrieved.
     *
     * @return  A set containing the ObjectNames for the managed objects selected.
     * If no managed object satisfies the query, an empty set is returned.
     *
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     */
   
    Set queryNames(String objectName,  String query) throws RemoteException;

    /**
     * Checks whether a managed object, identified by its object name, is already registered
     * with the MEJB.
     *
     * @param name The object name of the managed object to be checked.
     *
     * @return  True if the managed object is already registered in the MEJB, false otherwise.
     *
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     */
    boolean isRegistered(String name) throws RemoteException;

    /**
     * Returns the number of managed objects registered in the MEJB.
     *
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     */
    Integer getMBeanCount() throws RemoteException;

    /**
     * This method discovers the attributes and operations that a managed object exposes
     * for management.
     *
     * @param name The name of the managed object to analyze
     *
     * @return  An instance of <CODE>MBeanInfo</CODE> allowing the retrieval of all attributes and operations
     * of this managed object.
     *
     * @exception IntrospectionException An exception occurs during introspection.
     * @exception InstanceNotFoundException The managed object specified is not found.
     * @exception ReflectionException An exception occurred when trying to perform reflection on a managed object
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     */
    //MBeanInfo getMBeanInfo(String name) throws IntrospectionException, InstanceNotFoundException, ReflectionException, RemoteException;
      String getMBeanInfo(String name) throws RemoteException;

    /**
     * Gets the value of a specific attribute of a named managed object. The managed object
     * is identified by its object name.
     *
     * @param name The object name of the managed object from which the attribute is to be retrieved.
     * @param attribute A String specifying the name of the attribute to be
     * retrieved.
     *
     * @return  The value of the retrieved attribute.
     *
     * @exception AttributeNotFoundException The attribute specified is not accessible in the managed object.
     * @exception MBeanException  Wraps an exception thrown by the managed object's getter.
     * @exception InstanceNotFoundException The managed object specified is not registered in the MEJB.
     * @exception ReflectionException An exception occurred when trying to invoke the getAttribute method of a Dynamic MBean
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     */
    Object getAttribute(String name, String attribute) throws RemoteException;

    /**
     * Enables the values of several attributes of a named managed object. The managed object
     * is identified by its object name.
     *
     * @param name The object name of the managed object from which the attributes are
     * retrieved.
     * @param attributes A list of the attributes to be retrieved.
     *
     * @return The list of the retrieved attributes.
     *
     * @exception InstanceNotFoundException The managed object specified is not registered in the MEJB.
     * @exception ReflectionException An exception occurred when trying to invoke the getAttributes method of a Dynamic MBean.
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     */
    String getAttributes(String name, Set attributes) throws RemoteException;

    /**
     * Sets the value of a specific attribute of a named managed object. The managed object
     * is identified by its object name.
     *
     * @param name The name of the managed object within which the attribute is to be set.
     * @param attributeName The identification of the attribute to be set and the value it is to be set to.
     *
     * @return  The value of the attribute that has been set.
     *
     * @exception InstanceNotFoundException The managed object specified is not registered in the MEJB.
     * @exception AttributeNotFoundException The attribute specified is not accessible in the managed object.
     * @exception InvalidAttributeValueException The value specified for the attribute is not valid.
     * @exception MBeanException Wraps an exception thrown by the managed object's setter.
     * @exception ReflectionException An exception occurred when trying to invoke the setAttribute method of a Dynamic MBean.
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     */
    void setAttribute(String name, String attributeName, String value) throws RemoteException;
    
    /**
     * Sets the values of several attributes of a named managed object. The managed object is
     * identified by its object name.
     *
     * @param name The object name of the managed object within which the attributes are to
     * be set.
     * @param serializedAttributeList A list of attributes: The identification of the
     * attributes to be set and  the values they are to be set to.
     *
     * @return  The list of attributes that were set, with their new values.
     *
     * @exception InstanceNotFoundException The managed object specified is not registered in the MEJB.
     * @exception ReflectionException An exception occurred when trying to invoke the setAttributes method of a Dynamic MBean.
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     *
     */    
    String setAttributes(String name, String serializedAttributeList) throws RemoteException;

    /**
     * Invokes an operation on a managed object.
     *
     * @param name The object name of the managed object on which the method is to be invoked.
     * @param operationName The name of the operation to be invoked.
     * @param params An array containing the parameters to be set when the operation is
     * invoked
     * @param signature An array containing the signature of the operation. The class objects will
     * be loaded using the same class loader as the one used for loading the managed object on which the operation was invoked.
     *
     * @return  The object returned by the operation, which represents the result of invoking the operation on the
     * managed object specified.
     *
     * @exception InstanceNotFoundException The managed object specified is not registered in the MEJB.
     * @exception MBeanException  Wraps an exception thrown by the managed object's invoked method.
     * @exception ReflectionException  Wraps a <CODE>java.lang.Exception</CODE> thrown while trying to invoke the method.
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     */
    Object invoke(String name, String operationName, Set params, Set signature) throws RemoteException;

    /**
     * Returns the default domain name of this MEJB.
     * @exception RemoteException A communication exception occurred during the execution of a remote method call
     */
    String getDefaultDomain() throws RemoteException;

   /*
    * Returns the listener registry implementation for this MEJB. The listener registry implements the methods
    * that enable clints to add and remove event notification listeners managed objects
    * @return An implementation of <CODE>javax.management.j2ee.ListenerRegistration</CODE>
    *
    * @exception RemoteException A communication exception occurred during the execution of a remote method call
    */
    
    public void addNotificationListener( String objectName, String callbackUri,
        String filter, String handbackObject ) throws RemoteException;

    public void removeNotificationListener(String objectName, String callbackURI) throws RemoteException;
    
}
