/*
 * Copyright 2001 Sun Microsystems, Inc.  All rights reserved.
 * PROPRIETARY/CONFIDENTIAL.  Use of this product is subject to license terms.
 */

package com.sun.portal.search.demo;

import com.sun.portal.search.soif.*;

import java.applet.Applet;
import java.awt.Button;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.Panel;
import java.awt.TextField;
import java.awt.Event;
import java.awt.AWTEvent;

import java.io.*;


/**
 * Performs a simple search and displays its results.
 */
class SimpleSearch {
    
    String RDMServer;
    String dbname;
    String SOIFOutputFile;
    
    /**
     * SimpleSearch constructor
     * @param rdm - the rdm search server, eg, http://portal.siroe.com:2222/ps/search
     */
    public SimpleSearch(String rdm, String db) {
	System.out.println("Sun ONE Search Java Demo");
	RDMServer = rdm;
	dbname = db;
    }
    
    public void doSearch(String scope) throws IOException {
	
	/* The Search class encapsulates the search.
	 ** It's parameters are:
	 **  1) the search string
	 **  2) the attributes you want returned, comma delimited
	 **  3) sort order, comma delimited, - descending, + ascending
	 **  4) first hit
	 **  5) number of hits
	 **  6) query language, eg search, taxonomy-basic, schema-basic, etc
	 **  7) database to search
	 **  8) The RDM server URL, eg, http://portal.siroe.com:2222/ps/search
	 **  9) Access token (null for anonymous access, or valid
	 **     Sun ONE Identity Server session id)
	 */
	Search  search = new Search(
	scope,
	"score,url,title,description",
	"-score",
	1,
	20,
	"search",
	dbname,
	RDMServer,
	null
	);
	
	/* Execute the query. */
	
	System.out.println("\nSearch results for \"" + scope + "\"");
	
	DataOutputStream sos = null;
	if (SOIFOutputFile != null) {
	    try {
		sos = new DataOutputStream(new FileOutputStream(SOIFOutputFile));
	    }
	    catch (Exception e1) {
		System.out.println("Error: failed to create output file: " + e1);
	    }
	}
	
	int  pagenum = 1;
	int  pagesize = 10;
	SOIFBuffer firstPageSOIF = new SOIFBuffer();
	
	for (; pagenum <= 5; pagenum++)	{
	    
	    int firstHit = (pagenum-1)*pagesize+1;
	    
	    try {
		search.doQuery(firstHit, pagesize);
	    }
	    catch (Exception ex) {
		ex.printStackTrace();
		break;
	    }
	    
	    // Check the result count. -1 indicates an error.
	    if (search.getResultCount() <= 0)
		break;
	    
	    System.out.println("==========================================");
	    System.out.println("page " + pagenum
	    + ": hits " + search.getFirstHit()
	    + " to " + (search.getFirstHit() + search.getResultCount() - 1)
	    + " out of " + search.getHitCount()
	    + " across " + search.getDocumentCount() + " documents");
	    System.out.println("==========================================");
	    System.out.println();
	    
	    SOIFInputStream resultStream = search.getResultStream();
	    SOIF soif;
	    
	    /* Examine the results of the search.  The
	     ** results can also be returned as a string
	     ** using getResult(), but the SOIF version
	     ** parses the string for you.  The following
	     ** block loops through a list of SOIF instances.
	     */
	    for (soif = resultStream.readSOIF(); soif != null; soif = resultStream.readSOIF()) {
		
		if (pagenum == 1)
		    firstPageSOIF.write(soif.toByteArray());
		
		/* use the getValue() method to get
		 ** a value associated w/ the specified
		 ** attribute.
		 */
		String  u   = soif.getURL();
		String  t   = soif.getValue("title");
		String  d   = soif.getValue("description");
		String  sc  = soif.getValue("score");
		
		/* do something with the results */
		System.out.println(
		"TITLE:       " + t  + "\n" +
		"URL:         " + u  + "\n" +
		"SCORE:       " + sc + "\n" +
		"DESCRIPTION: " + d  + "\n" +
		"--------------------------------------------\n"
		);
		if (sos != null) {
		    try {
			sos.writeBytes(soif.toString());
		    }
		    catch (Exception e1) {
			System.out.println("Error: failed to write to SOIF output file: " + e1);
		    }
		}
		
	    }
	    
	    /* Break if largest hit already displayed */
	    if (search.getHitCount() <= (firstHit + pagesize - 1))
		break;
	    
	}
	
	if (firstPageSOIF == null)
	    System.out.println("No matching documents found.");
	
    }
    
    /**
     * @param filename - a file to dump raw SOIF results into - only
     * use if running from the comand line or an applet with file
     * system access
     */
    public void setSOIFfile(String filename) {
	SOIFOutputFile = filename;
    }
    
}


/**
 * A panel with a search box that calls the simple search class.
 */
class SearchPanel extends Panel implements java.awt.event.ActionListener {
    
    Button	searchBtn;
    TextField	searchTF;
    SimpleSearch ss;
    
    /** C'tor */
    public SearchPanel(SimpleSearch ss) {
	this.ss = ss;
	searchBtn = new Button("Search");
	searchTF = new TextField("java", 40);
	setLayout(new FlowLayout(FlowLayout.CENTER));
	add(searchTF);
	add(searchBtn);
	searchTF.addActionListener(this);
	searchBtn.addActionListener(this);
    }
    
    public void actionPerformed(java.awt.event.ActionEvent e) {
	if (e.getID() == Event.ACTION_EVENT) {
	    searchTF.setEditable(false);
	    searchBtn.setEnabled(false);
	    try {
		ss.doSearch(searchTF.getText());
	    }
	    catch (Exception ex) {
		System.out.println("Exception during search: " + ex);
	    }
	    searchTF.setEditable(true);
	    searchBtn.setEnabled(true);
	}
	else
	    super.processEvent(e);
    }
    
}


/**
 * Applet/application for simple query interface.  Can be used as an
 * example for those who want to create their own java interface.
 * This example demonstrates search only.  Browse, determining
 * the schema of the search server and obtaining the taxonomy
 * of the search server will be demonstrated in other examples.
 */
public class SearchDemo extends Applet {
    
    /** Run as an applet. */
    public void init() {
	String rdm = getParameter("RDMServer");
	String db = getParameter("Database");
	SimpleSearch ss = new SimpleSearch(rdm, db);
	SearchPanel sp = new SearchPanel(ss);
	setLayout(new FlowLayout(FlowLayout.CENTER));
	add(sp);
    }
    
    /** Run as an application. */
    public static void main(String argv[]) throws Exception {
	
	int args = argv.length;
	String SOIFOutputFile = null;
	
	if (args != 1 && args != 2 && args != 3) {
	    System.out.println("args: RDMServer [query] [soif_output_file_name]");
	    return;
	}
	
	String rdm = argv[0];	// rdm search server, eg, http://portal.siroe.com:2222/ps/search
	String db = null;	// server will use default db
	
	SimpleSearch ss = new SimpleSearch(rdm, db);
	
	if (args == 3) {
	    --args;
	    ss.setSOIFfile(argv[2]); // dump raw soif results to this file
	}
	
	if (args == 1) {
	    // run from a search box
	    Frame f = new Frame();
	    SearchPanel sp = new SearchPanel(ss);
	    f.add(sp);
	    f.pack();
	    f.show();
	}
	else {
	    // run from command line
	    String query = argv[1];
	    ss.doSearch(query);
	}
    }
    
}

