/*
 *  Copyright (c) 2000 Sun Microsystems.  All rights reserved.
 *  All rights reserved.
 * 
 *  Use of this software is governed by the terms of the license agreement 
 *  for the Sun ONE Compass Server between the parties.
 */


/* ------------------------------------------------------------------------ */


/*
 * Copyright 2003 Sun Microsystems, Inc. All
 * rights reserved. Use of this product is subject
 * to license terms. Federal Acquisitions:
 * Commercial Software -- Government Users
 * Subject to Standard License Terms and
 * Conditions.
 *
 * Sun, Sun Microsystems, the Sun logo, and Sun ONE
 * are trademarks or registered trademarks of Sun Microsystems,
 * Inc. in the United States and other countries.
 */


/* ------------------------------------------------------------------------ */


#ifndef PUBLIC_LIBCS_PBLOCK_H
#define PUBLIC_LIBCS_PBLOCK_H

/*
 * pblock.h: Header for Parameter Block handling functions
 * 
 *
 * A parameter block is a set of name=value pairs which are generally used 
 * as parameters, but can be anything. They are kept in a hash table for 
 * reasonable speed, but if you are doing any intensive modification or
 * access of them you should probably make a local copy of each parameter
 * while working.
 *
 * When creating a pblock, you specify the hash table size for that pblock.
 * You should set this size larger if you know that many items will be in 
 * that pblock, and smaller if only a few will be used or if speed is not
 * a concern.
 *
 * The hashing function is very simple right now, and only looks at the 
 * first character of name.
 *
 * Rob McCool
 */

/* ------------------------------ Structures ------------------------------ */


typedef struct {
    char *name,*value;
} libcs_pb_param;

struct libcs_pb_entry {
    libcs_pb_param *param;
    struct libcs_pb_entry *next;
};

typedef struct {
    int hsize;
    struct libcs_pb_entry **ht;
} libcs_pblock;


#ifdef __cplusplus
extern "C" {
#endif

/* ------------------------------ Prototypes ------------------------------ */


/*
 * SPparam_create creates a parameter with the given name and value. If name
 * and value are non-NULL, they are copied and placed into the new pb_param
 * struct.
 */

NSAPI_PUBLIC libcs_pb_param *libcs_param_create(char *name, char *value);

/*
 * SPparam_free frees a given parameter if it's non-NULL, and returns 1 if
 * p was non-NULL, and 0 if p was NULL.
 * 
 * Useful for error checking SPpblock_remove.
 */

NSAPI_PUBLIC int libcs_param_free(libcs_pb_param *pp);

/* 
 * SPpblock_create creates a new pblock with hash table size n.
 * 
 * It returns the newly allocated pblock.
 */

NSAPI_PUBLIC libcs_pblock *libcs_pblock_create(int n);

/*
 * SPpblock_free frees the given pblock and any entries inside it.
 * 
 * If you want to save anything in a pblock, remove its entities with 
 * SPpblock_remove first and save the pointers you get.
 */

NSAPI_PUBLIC void libcs_pblock_free(libcs_pblock *pb);

/* 
 * SPpblock_find finds the entry with the given name in pblock pb.
 *
 * If it is successful, it returns the param block. If not, it returns NULL.
 */

#define libcs_pblock_find(name, pb) (_libcs_pblock_fr(name,pb,0))

/*
 * SPpblock_findval finds the entry with the given name in pblock pb, and
 * returns its value, otherwise returns NULL.
 */

NSAPI_PUBLIC char *libcs_pblock_findval(const char *name, libcs_pblock *pb);

/*
 * SPpblock_remove behaves exactly like SPpblock_find, but removes the given
 * entry from pb.
 */

#define libcs_pblock_remove(name, pb) (_libcs_pblock_fr(name,pb,1))

/*
 * SPpblock_nvinsert creates a new parameter with the given name and value
 * and inserts it into pblock pb. The name and value in the parameter are
 * also newly allocated. Returns the pb_param it allocated (in case you 
 * need it).
 *
 * SPpblock_nninsert inserts a numerical value.
 */

NSAPI_PUBLIC libcs_pb_param *libcs_pblock_nvinsert(const char *name, const char *value, libcs_pblock *pb);
NSAPI_PUBLIC libcs_pb_param *libcs_pblock_nninsert(const char *name, int value, libcs_pblock *pb);

/*
 * SPpblock_pinsert inserts a pb_param into a pblock.
 */

NSAPI_PUBLIC void libcs_pblock_pinsert(libcs_pb_param *pp, libcs_pblock *pb);

/*
 * SPpblock_str2pblock scans the given string str for parameter pairs
 * name=value, or name="value". Any \ must be followed by a literal 
 * character. If a string value is found, with no unescaped = signs, it
 * will be added with the name 1, 2, 3, etc. depending on whether it was
 * first, second, third, etc. in the stream (zero doesn't count).
 * 
 * Returns the number of parameters added to the table, or -1 upon error.
 */

NSAPI_PUBLIC int libcs_pblock_str2pblock(const char *str, libcs_pblock *pb);
#define libcs_str2pblock libcs_pblock_str2pblock

/*
 * SPpblock_pblock2str places all of the parameters in the given pblock 
 * into the given string (NULL if it needs creation). It will re-allocate
 * more space for the string. Each parameter is separated by a space and of
 * the form name="value"
 */

NSAPI_PUBLIC char *libcs_pblock_pblock2str(libcs_pblock *pb, char *str);
#define libcs_pblock2str libcs_pblock_pblock2str

/*
 * SPpblock_copy copies the entries in the given source pblock to the 
 * destination one. The entries are newly allocated so that the original
 * pblock may be freed or the new one changed without affecting the other.
 */

NSAPI_PUBLIC void libcs_pblock_copy(libcs_pblock *src, libcs_pblock *dst);

/*
 * SPpblock_dup creates a new pblock and copies the given source pblock
 * into it.  The entries are newly allocated so that the original pblock
 * may be freed or the new one changed without affecting the other.
 */

NSAPI_PUBLIC libcs_pblock *libcs_pblock_dup(libcs_pblock *src);

/* Compass-specific -- apply */
NSAPI_PUBLIC void libcs_pblock_apply(libcs_pblock *pb, void (*fn)(libcs_pb_param
 *, void *), void *);

/* --------------------------- Internal things ---------------------------- */


NSAPI_PUBLIC libcs_pb_param *_libcs_pblock_fr(const char *name, libcs_pblock *pb, int remove);
NSAPI_PUBLIC char * libcs_pblock_replace(const char *name,char * new_value,libcs_pblock *pb);

/* extra libcs stuff */
NSAPI_PUBLIC void libcs_pblock_remove_n_free(char *n, libcs_pblock *pb);
NSAPI_PUBLIC void libcs_pblock_pblock2str_add_nl(char *s);
#define libcs_pblock2str_add_nl libcs_pblock_pblock2str_add_nl
NSAPI_PUBLIC void libcs_pblock_form2pblock(char *buf, int sz, libcs_pblock *pb);
#define libcs_form2pblock libcs_pblock_form2pblock

#ifdef __cplusplus
};
#endif

#endif /* !PUBLIC_BASE_PBLOCK_H */
