/*
 *  rdm.h - Resource Description Messaging (RDM)
 *
 *  For further details: http://www.w3.org/pub/WWW/TR/NOTE-rdm.html
 *
 * Copyright 2003 Sun Microsystems, Inc. All
 * rights reserved. Use of this product is subject
 * to license terms. Federal Acquisitions:
 * Commercial Software -- Government Users
 * Subject to Standard License Terms and
 * Conditions.
 *
 * Sun, Sun Microsystems, the Sun logo, and Sun ONE
 * are trademarks or registered trademarks of Sun Microsystems,
 * Inc. in the United States and other countries.
 */
#ifndef _RDM_H_
#define _RDM_H_

#include "soif.h"	/* Requires SOIF API */

#ifdef __cplusplus
extern "C" {
#endif

/*  ---------------------------------------------------------------------- */

/** To find out the version of the RDM library, use RDM_Version() */
NSAPI_PUBLIC const char *RDM_Version(void);

/*  ---------------------------------------------------------------------- */

/** RDM API objects */
typedef struct rdm_csid_s 	CSID;
typedef struct rdm_header_s 	RDMHeader;
typedef struct rdm_query_s 	RDMQuery;
typedef struct rdm_request_s 	RDMRequest;
typedef struct rdm_response_s 	RDMResponse;
typedef struct rdm_schema_s 	RDMSchema;
typedef struct rdm_server_s 	RDMServer;
typedef struct rdm_taxonomy_s 	RDMTaxonomy;
typedef struct rdm_class_s 	RDMClassification;
typedef struct rdm_view_s 	RDMView;
typedef struct rdm_viewattr_s 	RDMViewAttr;
typedef struct rdm_viewhits_s 	RDMViewHits;
typedef struct rdm_vieworder_s 	RDMViewOrder;
typedef RDMResponse *(*RDMServiceFn)(RDMRequest *);

/** RDM Enumerated types */
typedef enum rdm_csid_types_e {		/* Used for CSID types */
	CSID_UNDEF,			/* unknown */
	CSID_NORMAL,			/* x-catalog */
	CSID_SSL			/* x-catalogs */
} CSIDType;

typedef enum rdm_tax_order_e {		/* Used for Taxonomy traversals */
	RDM_TAX_INORDER,
	RDM_TAX_PREORDER,
	RDM_TAX_POSTORDER
} RDMTaxonomyOrder;

typedef enum rdm_sort_order_e {		/* Used for RDMViewOrder */
	RDM_ASCENDING,
	RDM_DESCENDING
} RDMSortOrder;

typedef enum rdm_type_e {		/* Used to identify RDM message type */
	RDM_MSGTYPE_UNKNOWN = 0,	/* undefined or unknown */
	RDM_RD_UPD_REQ,			/* request a RD stream for update */
	RDM_RD_DEL_REQ,			/* request a RD stream for delete */
	RDM_RD_UPD_RES,			/* a RD stream for update */
	RDM_RD_DEL_RES,			/* a RD stream for delete */
	RDM_SD_REQ,			/* request a server description */
	RDM_SD_RES,			/* a server description */
	RDM_SCH_REQ,			/* request a schema description */
	RDM_SCH_RES,			/* a schema description */
	RDM_TAX_REQ,			/* request a taxonomy description */
	RDM_TAX_RES,			/* a taxonomy description */
	RDM_STAT_REQ,			/* a status request (existence) */
	RDM_STAT_RES			/* a status message */
} RDMType;
#define RDM_RD_REQ	RDM_RD_UPD_REQ	/* UPDATE stream is default */
#define RDM_RD_RES	RDM_RD_UPD_RES	/* UPDATE stream is default */

/** RDM MIME type for use with HTTP/1.0 */
#define RDM_MIME_TYPE		"application/x-rdm"		/* CS1.0 */

/** RDM Header - @RDMHEADER SOIF Attribute Names */
#define A_RDM_VERSION		"rdm-version"			/* CS1.0 */
#define A_RDM_TYPE		"rdm-type"			/* CS1.0 */
#define A_RDM_QL		"rdm-query-language"		/* CS1.0 */
#define A_RDM_RESP		"rdm-response-interpret"	/* CS1.0 */
#define A_RDM_EMSG		"rdm-error-message"		/* CS1.0 */
#define A_RDM_ERR		"rdm-error-number"		/* CS1.0 */
#define A_RDM_CSID		"catalog-service-id"		/* CS1.0 */
#define A_RDM_USER		"rdm-user"			/* CSHana */
#define A_RDM_PASSWORD		"rdm-password"			/* CSHana */
#define A_RDM_ACCESS_TOKEN	"rdm-access-token"		/* CSHana */

/** RDM Query - @RDMQUERY SOIF Attribute Names */
#define A_RDM_SCOPE		"scope"				/* CS1.0 */
#define A_RDM_MODE		"mode"				/* CS3.01C */
#define A_RDM_VIEWATT		"view-attributes"		/* CS1.0 */
#define A_RDM_VIEWHIT		"view-hits"			/* CS1.0 */
#define A_RDM_DATABASE		"database"			/* 6.0 */
#define A_RDM_VIEWORD		"view-order"			/* CS1.0 */
#define A_RDM_VIEWTMPL		"view-template"			/* CS2.0 */

/** RDM Classification/Taxonomy - SOIF Attribute Names */
#define A_RDM_ID		"id"				/* CS1.0 */
#define A_RDM_PARENT		"parent-id"			/* CS1.0 */
#define A_RDM_TAX		"taxonomy-id"			/* CS1.0 */
#define A_RDM_DESC		"description"			/* CS1.0 */
#define A_RDM_XREF		"cross-reference-to"		/* CS2.0 */
#define A_RDM_LMT		"last-modified"			/* CS1.0 */
#define A_RDM_MAINT		"maintainer"			/* CS1.0 */
#define A_RDM_MATCHRULE		"matching-rule"			/* CS2.0 */
#define A_RDM_NSUBCAT		"subclassification-count"	/* CS2.0 */
#define A_RDM_NCATDOC		"categorized-count"		/* CS2.0 */
#define A_RDM_NSUBDOC		"subcategorized-count"		/* CS3.01C */

/** RDM Schema - SOIF Attribute Names */
#define A_RDM_SDL_VERSION \
        "schema-definition-language-version"			/* CS1.0 */
#define A_RDM_SCHEMA_NAME	"schema-name"			/* CS1.0 */
#define A_RDM_NUM_ENTRIES	"number-of-entries"		/* CS1.0 */
#define A_RDM_SOIF_ATTR		"soif-attribute"		/* CS1.0 */
#define A_RDM_DATA_TYPE		"data-type"			/* CS1.0 */
#define A_RDM_CONTENT_TYPE	"content-type"			/* CS1.0 */
#define A_RDM_ENFORCE_UNIQ	"enforce-uniqueness"		/* CS1.0 */
#define A_RDM_INDEX_ATTR	"index-attribute"		/* CS1.0 */
#define A_RDM_DB_INDEX_ATTR	"db-index-attribute"		/* CS2.0 */
#define A_RDM_EDIT_ATTR		"edit-attribute"		/* CS2.0 */
#define A_RDM_INTERNAL		"is-internal"			/* CS1.0 */
#define A_RDM_VIEWORD_DEFAULT	"default-view-order"		/* CS1.0 */
#define A_RDM_VIEWATT_DEFAULT	"default-view-attribute"	/* CS1.0 */
#define A_RDM_TBLNAME		"table-name"			/* CS1.0 */
#define A_RDM_COLNAME		"column-name"			/* CS1.0 */
#define A_RDM_SYSTBLNAME	"system-table-name"		/* CS1.0 */
#define A_RDM_SYSCOLNAME	"system-column-name"		/* CS1.0 */
#define A_RDM_FK_SYSTBLNAME	"foreign-key-system-table-name"	/* CS1.0 */
#define A_RDM_FK_SYSCOLNAME	"foreign-key-system-column-name"/* CS1.0 */
#define A_RDM_INROOT		"in-root-table"			/* CS1.0 */

/** RDM Server - SOIF Attribute Names */
#define A_RDM_SUPTYPE		"supported-rdm-type"		/* CS1.0 */
#define A_RDM_SUPQL		"supported-rdm-query-language"	/* CS1.0 */
#define A_RDM_SUPCSID 		"supported-catalog-service-id"	/* CS1.0 */
#define A_RDM_SD_LMT		"sd-last-modified"		/* CS1.0 */
#define A_RDM_SD_EXPIRE		"sd-expires"			/* CS1.0 */
#define A_RDM_SAMPLE_RD		"sample-rd"			/* CS1.0 */
#define A_RDM_COLL		"collection"			/* Ent3.0 */

/** SOIF Schema Names for all RDM objects */ 
#define A_SN_RDM_CLASS		"CLASSIFICATION"		/* CS1.0 */
#define A_SN_RDM_TAX		"TAXONOMY"			/* CS1.0 */
#define A_SN_RDM_SCHEMA		"SCHEMA"			/* CS1.0 */
#define A_SN_RDM_HDR		"RDMHEADER"			/* CS1.0 */
#define A_SN_RDM_QRY		"RDMQUERY"			/* CS1.0 */
#define A_SN_RDM_SERVER		"RDMSERVER"			/* CS1.0 */

/*  ---------------------------------------------------------------------- */

/** RDM Request Message
 *
 *  The message contains an RDMHeader with an appropriate request type:
 *	RDM_RD_UPD_REQ
 *	RDM_RD_DEL_REQ
 *	RDM_SD_REQ
 *	RDM_SCH_REQ
 *	RDM_TAX_REQ
 *	RDM_STAT_REQ
 *
 *  RDMRequest_Parse() will create a new RDMRequest object by
 *  parsing the RDMHeader and RDMQuery objects from the given SOIF Stream.
 */
NSAPI_PUBLIC RDMRequest *RDMRequest_Parse(SOIFStream *ss);
NSAPI_PUBLIC RDMRequest *RDMRequest_Create(RDMHeader *hdr, RDMQuery *qry);
NSAPI_PUBLIC int RDMRequest_Free(RDMRequest *msg);

struct rdm_request_s {
	RDMHeader *header;	/* RDM Header object */
	RDMQuery *query;	/* RDM Query object */

	/** To service the given request, supply this service() function.
	 *  The service function uses the given RDMRequest * object, 
	 *  resolves the request, then returns a RDMResponse * object.
	 *
	 *  By default, this service function is not defined (e.g., == NULL).
	 *  The caller must supply a service() function.
	 */
	RDMServiceFn service;

	void *caller_data;	/* caller can set this to whatever */
	void *internal;		/* DO NOT USE */
};

/*  ---------------------------------------------------------------------- */

/** RDM Response Message
 *
 *  The message contains an RDMHeader with an appropriate response type
 *	RDM_RD_UPD_RES
 *	RDM_RD_DEL_RES
 *	RDM_SD_RES
 *	RDM_SCH_RES
 *	RDM_TAX_RES
 *	RDM_STAT_RES;
 *  and a SOIF Stream containing the result set of the objects that
 *  matched the request.
 *
 *  RDMResponse_Parse() will create a new RDMResponse object, and
 *  parse the RDMHeader from the given SOIF Stream, and cue the
 *  SOIF Stream to be ready for parsing the result set.
 */
NSAPI_PUBLIC RDMResponse *RDMResponse_Parse(SOIFStream *ss);
NSAPI_PUBLIC RDMResponse *RDMResponse_Create(RDMHeader *hdr, SOIFStream *ss);
NSAPI_PUBLIC int RDMResponse_Free(RDMResponse *msg);

struct rdm_response_s {
	RDMHeader *header;	/* RDM Header object */
	SOIFStream *results;	/* Result set as stream of SOIF objects */

	void *caller_data;	/* caller can set this to whatever */
	void *internal;		/* DO NOT USE */
};

/*  ---------------------------------------------------------------------- */

/** RDM Query
 *
 *  Valid SOIF Attribute names for @RDMQUERY objects:
 *	Scope					REQUIRED
 *	Mode					OPTIONAL
 *	View-Attributes				OPTIONAL
 *	View-Hits				OPTIONAL
 *	View-Order				OPTIONAL
 *	View-Template				OPTIONAL
 */
NSAPI_PUBLIC RDMQuery *RDMQuery_Parse(SOIFStream *ss);
NSAPI_PUBLIC RDMQuery *RDMQuery_Create(const char *scope);
NSAPI_PUBLIC int RDMQuery_Free(RDMQuery *);

struct rdm_query_s {
	SOIF *soif;			/* Direct access to SOIF object */
	void *internal;			/* DO NOT USE */
};

/** To merge data from a SOIF object into the RDMQuery object */
NSAPI_PUBLIC int RDMQuery_Merge(RDMQuery *, SOIF *);

/** Macros for accessing @RDMQUERY values 
 *  Use prototype: char *RDMQuery_GetXXX(RDMQuery *q).
 */
#define RDMQuery_GetScope(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_SCOPE))
#define RDMQuery_GetMode(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_MODE))
#define RDMQuery_GetViewAttr(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_VIEWATT))
#define RDMQuery_GetViewHits(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_VIEWHIT))
#define RDMQuery_GetDatabaseName(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_DATABASE))
#define RDMQuery_GetViewOrder(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_VIEWORD))
#define RDMQuery_GetViewTemplate(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_VIEWTMPL))

/** Macros for defining @RDMQUERY values 
 *  Use prototype: int RDMQuery_SetXXX(RDMQuery *q, char *newvalue).
 */
#define RDMQuery_SetScope(c,s) \
	SOIF_Replace((c)->soif, A_RDM_SCOPE, (s), strlen(s))
#define RDMQuery_SetMode(c,s) \
	SOIF_Replace((c)->soif, A_RDM_MODE, (s), strlen(s))
#define RDMQuery_SetViewAttr(c,s) \
	SOIF_Replace((c)->soif, A_RDM_VIEWATT, (s), strlen(s))
#define RDMQuery_SetViewHits(c,s) \
	SOIF_Replace((c)->soif, A_RDM_VIEWHIT, (s), strlen(s))
#define RDMQuery_SetDatabaseName(c,s) \
	SOIF_Replace((c)->soif, A_RDM_DATABASE, (s), strlen(s))
#define RDMQuery_SetViewOrder(c,s) \
	SOIF_Replace((c)->soif, A_RDM_VIEWORD, (s), strlen(s))
#define RDMQuery_SetViewTemplate(c,s) \
	SOIF_Replace((c)->soif, A_RDM_VIEWTMPL, (s), strlen(s))

/** RDM View
 *
 *  Holds RDMViewAttr, RDMViewHits, and RDMViewOrder objects.
 */
NSAPI_PUBLIC RDMView *RDMView_Create(RDMViewAttr *, 
				     RDMViewHits *, 
				     RDMViewOrder *);
NSAPI_PUBLIC int RDMView_Free(RDMView *);

struct rdm_view_s {			/* RDM View object */
	RDMViewAttr *attr;
	RDMViewHits *hits;
	RDMViewOrder *order;
	void *internal;			/* DO NOT USE */
};

/** RDMView_Parse() loads a view from an RDMQuery.
 *  Use prototype: RDMView *RDMView_Parse(RDMQuery *);
 */
#define RDMView_Parse(req) \
	RDMView_Create( \
		RDMViewAttr_Parse(RDMQuery_GetViewAttr((req)->query)), \
		RDMViewHits_Parse(RDMQuery_GetViewHits((req)->query)), \
		RDMViewOrder_Parse(RDMQuery_GetViewOrder((req)->query)))

/** RDM View Attributes
 *
 *  The View Attributes is a simple list of SOIF Attributes (all lowercase)
 *  separated by commas.  Use _Parse() to load a new object.
 *
 *  Example:
 *	RDMViewAttr *vap = RDMViewAttr_Parse("scope,title,description,url");
 */
NSAPI_PUBLIC RDMViewAttr *RDMViewAttr_Parse(const char *str);
NSAPI_PUBLIC RDMViewAttr *RDMViewAttr_Create(void);
NSAPI_PUBLIC int RDMViewAttr_Free(RDMViewAttr *);
struct rdm_viewattr_s {			/* RDM View Attributes */
	char **attrs;			/* Each SOIF Attribute */
	int nattrs;			/* Number of Valid SOIF Attributes */

	/** To test if the given attribute is part of the view,
	 *  use contains().
	 */
	boolean_t (*contains)(RDMViewAttr *, char *attribute);

	/** To convert the object into its RDM string representation,
	 *  use asString().
	 */
	const char *(*asString)(RDMViewAttr *);

	/** To modify the array of attributes, use add() and delete(). */
	int (*add)(RDMViewAttr *, char *attribute);
	int (*remove)(RDMViewAttr *, char *attribute);

	void *internal;			/* DO NOT USE */
};

/** RDM View Hits
 *
 *  The View Hits object specifies a range for result sets.  
 *  Use _Parse() to load a new object from an RDM string.
 *
 *  Example:
 *	RDMViewHits *vhp1 = RDMViewHits_Parse("100")	for 1..100
 *	RDMViewHits *vhp2 = RDMViewHits_Parse("1..100")	for 1..100
 */
#define RDMVIEWHITS_UNBOUNDED	(-1)
#define RDMVIEWHITS_MAX		1000000	/* million */

NSAPI_PUBLIC RDMViewHits *RDMViewHits_Parse(const char *str);
NSAPI_PUBLIC RDMViewHits *RDMViewHits_Create(int min, int max);
NSAPI_PUBLIC int RDMViewHits_Free(RDMViewHits *);
struct rdm_viewhits_s {			/* RDMViewHits object */
	int min;			/* minimum hit number */
	int max;			/* maximum hit number */
	
	/** To generate a string representation of the View-Hits, 
	 *  use asString().  It'll generate the following strings:
	 *
	 *	min	max	string
	 *	1	10	"1..10"
	 *	UNBOUND	100	"1..100"
	 *	UNBOUND	UNBOUND	"1..1000000"
	 *	10	UNBOUND	"10..1000000"
	 *	100	UNBOUND	"100..1000000"
	 */
	const char *(*asString)(RDMViewHits *);

	void *internal;			/* DO NOT USE */
};

/** RDM View Order
 *
 *  The View Order specifies the sort order for the result set.
 *  It's in the same format as RDM View Attributes, except that
 *  each SOIF Attribute is preceeded by a '+' to indicate ascending
 *  order, or a '-' to indicate descending order.
 *
 *  Use _Parse() to load the object from a string:
 *
 *	RDMViewOrder *vp = RDMViewOrder_Parse("-score,+title");
 */
NSAPI_PUBLIC RDMViewOrder *RDMViewOrder_Parse(const char *str);
NSAPI_PUBLIC RDMViewOrder *RDMViewOrder_Create(void);
NSAPI_PUBLIC int RDMViewOrder_Free(RDMViewOrder *);

struct rdm_vieworder_s {		/* RDMViewOrder object */
	RDMSortOrder *sorts;
	char **attrs;
	int nattrs;

	/** To generate a string representation of the View-Order, 
	 *  use asString().  It'll generate the following strings:
	 *
	 *	SortOrder	Attr	String
	 *	RDM_ASCENDING	Score	"+score"
	 *
	 *	RDM_ASCENDING	Score	
	 *	RDM_ASCENDING	Title	"+score,+title"
	 *
	 *	RDM_DESCENDING	Score	
	 *	RDM_ASCENDING	Title	"-score,+title"
	 */
	const char *(*asString)(RDMViewOrder *);

	/** To test if the given attribute is part of the view,
	 *  use contains().
	 */
	boolean_t (*contains)(RDMViewOrder *, char *attribute);


	/** To add or delete items from the order, use add() or delete() */
	int (*add)(RDMViewOrder *, char *attribute, RDMSortOrder);
	int (*remove)(RDMViewOrder *, char *attribute);

	void *internal;			/* DO NOT USE */
};

/*  ---------------------------------------------------------------------- */

/** RDM Header Object 
 *
 *  Valid SOIF Attribute names for @RDMHEADER objects:
 *	RDM-Version					REQUIRED
 *	RDM-Type					REQUIRED
 *	Catalog-Service-ID				RECOMMENDED
 *	RDM-Query-Language				REQUIRED if a request
 *	RDM-Response-Interpret				OPTIONAL if a response
 *	RDM-Error-Message				OPTIONAL if a response
 *	RDM-Error-Number				OPTIONAL if a response
 */
NSAPI_PUBLIC RDMHeader *RDMHeader_Parse(SOIFStream *ss);
NSAPI_PUBLIC RDMHeader *RDMHeader_Create(RDMType t);
NSAPI_PUBLIC RDMHeader *RDMHeader_CreateRequest(RDMType, char *ql, CSID *);
NSAPI_PUBLIC RDMHeader *RDMHeader_CreateResponse(RDMType, char *ri, CSID *);
NSAPI_PUBLIC int RDMHeader_Free(RDMHeader *r);

struct rdm_header_s {			/* RDM Header object */
	SOIF *soif;			/* Direct access to SOIF object */
	void *internal;			/* DO NOT USE */
};

/** Macros for accessing @RDMHEADER values 
 *  Use prototype: char *RDMHeader_GetXXX(RDMHeader *h).
 */
#define RDMHeader_GetType(c) /* NOTE: returns RDMType */ \
	RDMType_Parse((*(c)->soif->findval)((c)->soif, A_RDM_TYPE))
#define RDMHeader_GetVersion(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_VERSION))
#define RDMHeader_GetQueryLanguage(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_QL))
#define RDMHeader_GetCSID(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_CSID))
#define RDMHeader_GetResponseInterpret(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_RESP))
#define RDMHeader_GetErrorMessage(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_EMSG))
#define RDMHeader_GetErrorNumber(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_ERR))
#define RDMHeader_GetAccessToken(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_ACCESS_TOKEN)) 
#define RDMHeader_GetUser(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_USER)) 
#define RDMHeader_GetPassword(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_PASSWORD))

/** Macros for defining @RDMHEADER values 
 *  Use prototype: int RDMHeader_SetXXX(RDMHeader *h, char *newvalue).
 */
#define RDMHeader_SetType(c,s) 	/* NOTE: Use (RDMHeader *, RDMType new) */ \
	SOIF_ReplaceStr((c)->soif, A_RDM_TYPE, (char *)RDMType_asString(s))
#define RDMHeader_SetVersion(c,s) \
	SOIF_Replace((c)->soif, A_RDM_VERSION, (s), strlen(s))
#define RDMHeader_SetQueryLanguage(c,s) \
	SOIF_Replace((c)->soif, A_RDM_QL, (s), strlen(s))
#define RDMHeader_SetCSID(c,s) \
	SOIF_Replace((c)->soif, A_RDM_CSID, (s), strlen(s))
#define RDMHeader_SetResponseInterpret(c,s) \
	SOIF_Replace((c)->soif, A_RDM_RESP, (s), strlen(s))
#define RDMHeader_SetErrorMessage(c,s) \
	SOIF_Replace((c)->soif, A_RDM_EMSG, (s), strlen(s))
#define RDMHeader_SetErrorNumber(c,s) \
	SOIF_Replace((c)->soif, A_RDM_ERR, (s), strlen(s))
#define RDMHeader_SetAccessToken(c,s) \
	SOIF_Replace((c)->soif, A_RDM_ACCESS_TOKEN, (s), strlen(s))
#define RDMHeader_SetUser(c,s) \
	SOIF_Replace((c)->soif, A_RDM_USER, (s), strlen(s))
#define RDMHeader_SetPassword(c,s) \
	SOIF_Replace((c)->soif, A_RDM_PASSWORD, (s), strlen(s))

/** To merge data from a SOIF object into the RDMHeader object */
NSAPI_PUBLIC int RDMHeader_Merge(RDMHeader *, SOIF *);

/*  ---------------------------------------------------------------------- */

/** RDM Message type conversions 
 *
 *  Example:
 *	RDMType t1 = RDMType_Parse("rd-request");
 *	RDMType t2 = RDM_SD_REQ;
 *	RDMType_asString(t2) == "schema-description-request";
 */
NSAPI_PUBLIC RDMType RDMType_Parse(const char *s);
NSAPI_PUBLIC const char *RDMType_asString(RDMType r);

/*  ---------------------------------------------------------------------- */

/** Catalog Service ID 
 *
 *  CSIDs are used to identify servers which contain catalog information.
 *  The CSIDs are written much like URLs using the following syntax:
 *	x-catalog://host:port/id
 *
 *  CSIDs support servers which host many catalogs, by further identifying
 *  the catalog with the given 'id'.
 *
 *  Example:
 *	x-catalog://www.iplanet.com:1234/ProductCatalog
 */
NSAPI_PUBLIC CSID *CSID_Parse(const char *str);
NSAPI_PUBLIC int CSID_Free(CSID *c);

struct rdm_csid_s {
	CSIDType type;		/* The access type (e.g., "x-catalog") */
	char *host;		/* The hostname */
	int port;		/* The port */
	char *id;		/* The id or name at that server */

	/** To view the CSID in its URL syntax, then
	 *  use asURL().
	 *
	 *  Example:
	 *	CSID *csid = CSID_Parse("x-catalog://foobar:80/stuff");
	 *	printf("CSID = %s\n", (*csid->asURL)(csid));
	 */
	const char *(*asURL)(CSID *c);

	void *internal;		/* DO NOT USE */
};

/*  ---------------------------------------------------------------------- */

/*
 *  RDM Taxonomy/Classification Support
 *
 *  RDM supports a hierarchical taxonomy which has cross-references.
 *  Very much analogous to a Unix directory structure with symbolic
 *  links between directories.  A taxonomy definition is written in
 *  SOIF containing:
 *	- a single @TAXONOMY object 
 *  	- many @CLASSIFICATION objects organized in a hierarchy
 *
 *  Example:
 * 
 *      @TAXONOMY { -					SchName REQ / URL Opt
 *	Id{x}:	Excite					REQUIRED
 *	Last-Modified{x}: Sun, 06 Nov 1994 08:49:37 GMT OPTIONAL
 *	Maintainer{x}: joeblow@iplanet.com		OPTIONAL
 *	}
 *
 *      @CLASSIFICATION { -				SchName REQ / URL Opt
 *	Id{x}:	Arts					REQUIRED
 *	Parent-Id{x}:	ROOT				REQUIRED
 *	Taxonomy-Id{x}:	Excite				REQUIRED
 *	Description{x}:	Stuff about Art			OPTIONAL
 *	}
 *
 *      @CLASSIFICATION { -				SchName REQ / URL Opt
 *	Id{x}:	Arts:Books				REQUIRED
 *	Parent-Id{x}:	Arts				REQUIRED
 *	Taxonomy-Id{x}:	Excite				REQUIRED
 *	Description{x}:	Stuff about Books		OPTIONAL
 *	}
 *
 *      @CLASSIFICATION { -				SchName REQ / URL Opt
 *	Id{x}:	Arts:Music				REQUIRED
 *	Parent-Id{x}:	Arts				REQUIRED
 *	Taxonomy-Id{x}:	Excite				REQUIRED
 *	Description{x}:	Stuff about Music		OPTIONAL
 *	}
 *
 *      @CLASSIFICATION { -				SchName REQ / URL Opt
 *	Id{x}:	Sports					REQUIRED
 *	Parent-Id{x}:	ROOT				REQUIRED
 *	Taxonomy-Id{x}:	Excite				REQUIRED
 *	Description{x}:	Stuff about Sports		OPTIONAL
 *	}
 *
 *	Describes a taxonomy that looks like:
 *		- Arts
 *			- Books
 *			- Music
 *		- Sports
 */

/** RDM Taxonomy
 *
 *  Represents a hierarchy of Classifications (or Categories).
 *  The Taxonomy supports basic lookup and enumeration primitives
 *  for understanding the organization and content of the Classifications.
 *
 *  To load a Taxonomy definition encoded in SOIF from a file or memory, 
 *  use RDMTaxonomy_Parse().
 *
 *  Example:
 *	SOIFStream *ss = SOIF_ParseInitFile(stdin);
 *	t = RDMTaxonomy_Parse(ss);
 *
 *  To create a Taxonomy structure with no Classifications, 
 *  use RDMTaxonomy_Create().
 *
 *  To free an entire Taxonomy and all of its Classifications, 
 *  use RDMTaxonomy_Free().
 */
NSAPI_PUBLIC RDMTaxonomy *RDMTaxonomy_Parse(SOIFStream *ss);
NSAPI_PUBLIC RDMTaxonomy *RDMTaxonomy_Create(const char *id);
NSAPI_PUBLIC int RDMTaxonomy_Free(RDMTaxonomy *t);

struct rdm_taxonomy_s {			/* RDMTaxonomy object */
	RDMClassification *root;	/* Root of the Taxonomy Hierarchy */
	SOIF *soif;			/* Direct access to SOIF object */

	/** To lookup a Classification Id in the taxonomy, use find().
	 *
	 *  Example:
	 *	RDMClassification *musicp = (*t->find)(t, "Arts:Music");
	 */
	RDMClassification *(*find)(RDMTaxonomy *t, const char *classification_id);

	/** To insert a new classification, use insert().  
	 *  The given Classification object is inserted into the Taxonomy
	 *  structure.  Use delete() to release the classification object
	 *  from the Taxonomy. 
	 */
	int (*insert)(RDMTaxonomy *t, RDMClassification *newc);
	int (*remove)(RDMTaxonomy *t, RDMClassification *oldc);

	/** To traverse the Taxonomy in the given order, use apply() 
	 *  and give an explicit traversal policy:
	 *	RDM_TAX_INORDER		- node, then children (recommended)
	 *	RDM_TAX_PREORDER	- same as INORDER
	 *	RDM_TAX_POSTORDER	- children, then node
	 *
	 *  Example:
	 *	(*t->apply)(t, RDM_TAX_INORDER, MyFn, NULL);
	 *	...
	 *	void MyFn(RDMClassification *c, void *unused)
	 *	{
	 *		printf("Classification = %s\n", 
	 *			RDMClassification_GetId(c));
	 *	}
	 */
	void (*apply)(RDMTaxonomy *t,
		      RDMTaxonomyOrder order,
		      void (*apply_fn)(RDMClassification *c, void *data),
		      void *data);

	void *internal;			/* DO NOT USE */
};

/** To merge data from a SOIF object into the Taxonomy object */
NSAPI_PUBLIC int RDMTaxonomy_Merge(RDMTaxonomy *, SOIF *);

/** To find the depth of this Classification from the Taxonomy root */
NSAPI_PUBLIC int RDMTaxonomy_Depth(RDMTaxonomy *, RDMClassification *c);

/** Macros for accessing @TAXONOMY values 
 *  Use prototype: char *RDMTaxonomy_GetXXX(RDMTaxonomy *t).
 */
#define RDMTaxonomy_GetId(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_ID))
#define RDMTaxonomy_GetDescription(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_DESC))
#define RDMTaxonomy_GetLMT(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_LMT))
#define RDMTaxonomy_GetMaintainer(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_MAINT))

/** Macros for defining @TAXONOMY values 
 *  Use prototype: int RDMTaxonomy_SetXXX(RDMTaxonomy *t, char *newvalue).
 */
#define RDMTaxonomy_SetId(c, s) \
	SOIF_Replace((c)->soif, A_RDM_ID, (s), strlen(s))
#define RDMTaxonomy_SetDescription(c, s) \
	SOIF_Replace((c)->soif, A_RDM_DESC, (s), strlen(s))
#define RDMTaxonomy_SetLMT(c, s) \
	SOIF_Replace((c)->soif, A_RDM_LMT, (s), strlen(s))
#define RDMTaxonomy_SetMaintainer(c, s) \
	SOIF_Replace((c)->soif, A_RDM_MAINT, (s), strlen(s))

/** Taxonomy's Root identifier */
#define RDM_TAXONOMY_ROOT	"ROOT"

/*  ---------------------------------------------------------------------- */

/** RDM Classification
 *
 *  A Classification (or "Category") is how data is organized into
 *  a taxonomy.  A Classification is analogous to a directory or folder 
 *  in a file system.
 *
 *  To load a Classification definition encoded in SOIF from a file or memory, 
 *  use RDMTaxonomy_Parse().
 *
 *  Example:
 *	SOIFStream *ss = SOIF_ParseInitFile(stdin);
 *	c = RDMClassification_Parse(ss);
 *
 *  To create a basic Classification structure, use RDMClassification_Parse().
 *  To free a Classification (only node, no children or xrefs),
 *  use RDMClassification_Free().
 */
NSAPI_PUBLIC RDMClassification *RDMClassification_Parse(SOIFStream *ss);
NSAPI_PUBLIC RDMClassification *RDMClassification_Create(const char *id);
NSAPI_PUBLIC int RDMClassification_Free(RDMClassification *c);

struct rdm_class_s {			/* RDMClassification object */
	RDMClassification *parent;	/* Pointer up the Taxonomy tree */
       	RDMClassification **children;	/* its children sorted by Id */
       	RDMClassification **xrefs;	/* its cross-references sorted by Id */
	char *subcategory;		/* subcategory -- id's last component */
	int depth;			/* number of hops from ROOT */
	int ndescendant;		/* number of descendants */
	int ndocs;			/* docs in this class */
	int ndescdocs;			/* docs in this plus all lower classes */
       	SOIF *soif;			/* Direct access to SOIF object */

	/** To lookup by classification id, use find(). 
	 *
	 *  Example:
	 *	RDMClassification *musicp = (*c->find)(c, "Arts:Music");
	 */
	RDMClassification *(*find)(RDMClassification *c, const char *class_id);

	/** To insert a new classification that is the child of the current 
	 */
	int (*insert_child)(RDMClassification *c, RDMClassification *newchild);
	int (*delete_child)(RDMClassification *c, RDMClassification *oldchild);

	/** To insert a cross-reference to another classification 
	 */
	int (*insert_xref)(RDMClassification *c, const char *xref);
	int (*delete_xref)(RDMClassification *c, const char *xref);

	/** To traverse the Taxonomy in the given order starting at this 
	 *  Classification, use apply().  Similar to RDMTaxonomy::apply().
  	 */
	void (*apply)(RDMClassification *c,
		      RDMTaxonomyOrder order,
		      void (*apply_fn)(RDMClassification *c, void *data),
		      void *data);

	void *internal;			/* DO NOT USE */
};

/** To merge data from a SOIF object into the Classification object */
NSAPI_PUBLIC int RDMClassification_Merge(RDMClassification *, SOIF *);

/** Macros for accessing children and cross references */
#define RDMClassification_NthChild(c,n)	((c)->children[n])
#define RDMClassification_NthXref(c,n)	((c)->xrefs[n])
NSAPI_PUBLIC int RDMClassification_NChildren(RDMClassification *);
NSAPI_PUBLIC int RDMClassification_NXrefs(RDMClassification *);

/** Returns the portion of the id that is 'depth' levels deep.
 *  For example, 
 *    ("a", 1) --> "a"  
 *    ("a", 2) --> NULL
 *    ("a:b", 2) --> "b"
 *    ("a:b:c:d", 1) --> "a"
 *    ("a:b:c:d", 2) --> "b"
 *    ("a:b:c:d", 3) --> "c"
 *    ("a:b:c:d", 4) --> "d"
 *    ("a:b:c:d", 5) --> NULL
 *    ("a:b:c:d", 0) --> NULL
 *  
 *  Return malloc'ed value if successful; otherwise returns NULL;
 */     
NSAPI_PUBLIC char *RDMClassification_Subcategory(const char *id, int depth);

/** Macros for accessing @CLASSIFICATION values 
 *  Use prototype: char *RDMClassification_GetXXX(RDMClassification *c).
 */
#define RDMClassification_GetId(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_ID))
#define RDMClassification_GetParentId(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_PARENT))
#define RDMClassification_GetTaxonomyId(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_TAX))
#define RDMClassification_GetDescription(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_DESC))
#define RDMClassification_GetMatchingRule(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_MATCHRULE))

/** Macros for defining @CLASSIFICATION values 
 *  Use prototype: int RDMClassification_SetXXX(RDMClassification *c, 
 *                                              char *newval).
 */
#define RDMClassification_SetId(c,s) \
	SOIF_Replace((c)->soif, A_RDM_ID, (s), strlen(s))
#define RDMClassification_SetParentId(c,s) \
	SOIF_Replace((c)->soif, A_RDM_PARENT, (s), strlen(s))
#define RDMClassification_SetTaxonomyId(c,s) \
	SOIF_Replace((c)->soif, A_RDM_TAX, (s), strlen(s))
#define RDMClassification_SetDescription(c,s) \
	SOIF_Replace((c)->soif, A_RDM_DESC, (s), strlen(s))
#define RDMClassification_SetMatchingRule(c,s) \
	SOIF_Replace((c)->soif, A_RDM_MATCHRULE, (s), strlen(s))

/*  ---------------------------------------------------------------------- */

/** RDM Schema
 *
 *  A Schema specifies how to interpret the SOIF objects for a given
 *  Schema-Name.  For example, a schema may define that the SOIF object
 *  called @DOCUMENT may contain the following Attributes:
 *	Author, Last-Modified, Description, Content-Length, etc.
 *  Furthermore, define how to interpret the individual values for 
 *  those Attributes.  For example, the Author attribute's value may be
 *  defined as a string.
 *
 *  The RDM @SCHEMA definition contains the following attributes:
 *	column-name					multivalued
 *	content-type					multivalued
 *	data-type					multivalued
 *	default-view-attribute				multivalued
 *	default-view-order				multivalued
 *	enforce-uniqueness				multivalued
 *	foreign-key-system-column-name			multivalued
 *	foreign-key-system-table-name			multivalued
 *	in-root-table					multivalued
 *	index-attribute					multivalued
 *	is-internal					multivalued
 *	number-of-entries				single-value
 *	schema-definition-language-version		single-value
 *	schema-name					single-value
 *	soif-attribute					multivalued
 *	system-column-name				multivalued
 *	system-table-name				multivalued
 *	table-name					multivalued
 */
NSAPI_PUBLIC RDMSchema *RDMSchema_Parse(SOIFStream *ss);
NSAPI_PUBLIC RDMSchema *RDMSchema_Create(const char *schema_name);
NSAPI_PUBLIC int RDMSchema_Free(RDMSchema *);
struct rdm_schema_s {			/* RDM Schema object */
	SOIF *soif;			/* Direct access to SOIF object */
	void *internal;			/* DO NOT USE */
};

/** To merge data from a SOIF object into the RDMSchema object */
NSAPI_PUBLIC int RDMSchema_Merge(RDMSchema *, SOIF *);

/** To remove a "slot" from the RDMSchema object */
NSAPI_PUBLIC int RDMSchema_DeleteSlot(RDMSchema *, int);

/** Macros for accessing single-value @SCHEMA values 
 *  Use prototype: char *RDMSchema_GetXXX(RDMSchema *s)
 */
#define RDMSchema_GetVersion(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_SDL_VERSION))
#define RDMSchema_GetName(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_SCHEMA_NAME))
#define RDMSchema_GetNumEntries(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_NUM_ENTRIES))

/** Macros for defining single-value @SCHEMA values 
 *  Use prototype: int RDMSchema_SetXXX(RDMSchema *schema, char *newval)
 */
#define RDMSchema_SetVersion(c,s) \
	SOIF_Replace((c)->soif, A_RDM_SDL_VERSION, (s), strlen(s))
#define RDMSchema_SetName(c,s) \
	SOIF_Replace((c)->soif, A_RDM_SCHEMA_NAME, (s), strlen(s))
#define RDMSchema_SetNumEntries(c,s) \
	SOIF_Replace((c)->soif, A_RDM_NUM_ENTRIES, (s), strlen(s))

/** Macros for accessing common multi-valued @SCHEMA values 
 *  Use prototype: char *RDMSchema_GetXXX(RDMSchema *schema, 
 *                                        int MultivalSlotNum).
 */
#define RDMSchema_GetSOIFAttribute(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_SOIF_ATTR, (slot))
#define RDMSchema_GetDataType(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_DATA_TYPE, (slot))
#define RDMSchema_GetContentType(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_CONTENT_TYPE, (slot))
#define RDMSchema_GetEnforceUnique(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_ENFORCE_UNIQ, (slot))
#define RDMSchema_GetIndexAttribute(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_INDEX_ATTR, (slot))
#define RDMSchema_GetDBIndexAttribute(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_DB_INDEX_ATTR, (slot))
#define RDMSchema_GetEditAttribute(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_EDIT_ATTR, (slot))
#define RDMSchema_GetIsInternal(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_INTERNAL, (slot))
#define RDMSchema_GetViewOrderDefault(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_VIEWORD_DEFAULT, (slot))
#define RDMSchema_GetViewAttributeDefault(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_VIEWATT_DEFAULT, (slot))
#define RDMSchema_GetTableName(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_TBLNAME, (slot))
#define RDMSchema_GetColumnName(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_COLNAME, (slot))
#define RDMSchema_GetSysTableName(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_SYSTBLNAME, (slot))
#define RDMSchema_GetSysColumnName(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_SYSCOLNAME, (slot))
#define RDMSchema_GetFKSysTableName(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_FK_SYSTBLNAME, (slot))
#define RDMSchema_GetFKSysColumnName(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_FK_SYSCOLNAME, (slot))
#define RDMSchema_GetInRoot(c,slot) \
	SOIF_FindvalMV((c)->soif, A_RDM_INROOT, (slot))

/** Macros for defining common multi-valued @SCHEMA values 
 *  Use prototype: int RDMSchema_SetXXX(RDMSchema *schema, 
 *                                      int MultivalSlotNum, char *newval).
 */
#define RDMSchema_SetSOIFAttribute(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_SOIF_ATTR, (slot), (s), strlen(s))
#define RDMSchema_SetDataType(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_DATA_TYPE, (slot), (s), strlen(s))
#define RDMSchema_SetContentType(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_CONTENT_TYPE, (slot), (s), strlen(s))
#define RDMSchema_SetEnforceUnique(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_ENFORCE_UNIQ, (slot), (s), strlen(s))
#define RDMSchema_SetIndexAttribute(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_INDEX_ATTR, (slot), (s), strlen(s))
#define RDMSchema_SetDBIndexAttribute(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_DB_INDEX_ATTR, (slot), (s), strlen(s))
#define RDMSchema_SetEditAttribute(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_EDIT_ATTR, (slot), (s), strlen(s))
#define RDMSchema_SetIsInternal(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_INTERNAL, (slot), (s), strlen(s))
#define RDMSchema_SetViewOrderDefault(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_VIEWORD_DEFAULT, (slot),(s),strlen(s))
#define RDMSchema_SetViewAttributeDefault(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_VIEWATT_DEFAULT, (slot),(s),strlen(s))
#define RDMSchema_SetTableName(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_TBLNAME, (slot), (s), strlen(s))
#define RDMSchema_SetColumnName(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_COLNAME, (slot), (s), strlen(s))
#define RDMSchema_SetSysTableName(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_SYSTBLNAME, (slot), (s), strlen(s))
#define RDMSchema_SetSysColumnName(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_SYSCOLNAME, (slot), (s), strlen(s))
#define RDMSchema_SetFKSysTableName(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_FK_SYSTBLNAME, (slot), (s), strlen(s))
#define RDMSchema_SetFKSysColumnName(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_FK_SYSCOLNAME, (slot), (s), strlen(s))
#define RDMSchema_SetInRoot(c,slot,s) \
	SOIF_ReplaceMV((c)->soif, A_RDM_INROOT, (slot), (s), strlen(s))

/*  ---------------------------------------------------------------------- */

/** RDM Server Description
 *
 *  Server Descriptions provide system-level access information to the
 *  client (such as supported RDM query languages), help distributed
 *  query routing clients better select promising catalog services to
 *  search, and provide human-readable descriptions of a catalog service.
 *  A server description is written as a single SOIF object with the
 *  object type of RDMSERVER. 
 *
 *  The @RDMSERVER definition contains the following attributes:
 *	Supported-RDM-Type 				single-value,comma-sep
 *	Supported-RDM-Query-Language 			single-value,comma-sep
 *	SD-Last-Modified 				single-value
 *	SD-Expires 					single-value
 *	Description 					single-value
 *	Supported-Catalog-Service-ID 			single-value,comma-sep
 *	Maintainer 					single-value
 *	Sample-RD					multivalued
 */
NSAPI_PUBLIC RDMServer *RDMServer_Parse(SOIFStream *ss, const char *url);
NSAPI_PUBLIC RDMServer *RDMServer_Create(const char *url);
NSAPI_PUBLIC int RDMServer_Free(RDMServer *);
struct rdm_server_s {			/* RDMServer object */
	char *url;			/* URL to the Server */
	SOIF *soif;			/* Direct access to SOIF object */
	void *internal;			/* DO NOT USE */
};

/** To merge data from a SOIF object into the RDMServer object */
NSAPI_PUBLIC int RDMServer_Merge(RDMServer *, SOIF *);

/** Macros for accessing single-value @RDMSERVER values 
 */
#define RDMServer_GetSupportedRDMType(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_SUPTYPE))
#define RDMServer_GetSupportedRDMQL(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_SUPQL))
#define RDMServer_GetSupportedCSID(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_SUPCSID))
#define RDMServer_GetLastModified(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_SD_LMT))
#define RDMServer_GetExpires(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_SD_EXPIRE))
#define RDMServer_GetDescription(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_DESC))
#define RDMServer_GetMaintainer(c) \
	((*(c)->soif->findval)((c)->soif, A_RDM_MAINT))

/** Macros for defining single-value @RDMSERVER values 
 */
#define RDMServer_SetSupportedRDMType(c) \
	SOIF_Replace((c)->soif, A_RDM_SUPTYPE, (s), strlen(s))
#define RDMServer_SetSupportedRDMQL(c) \
	SOIF_Replace((c)->soif, A_RDM_SUPQL, (s), strlen(s))
#define RDMServer_SetSupportedCSID(c) \
	SOIF_Replace((c)->soif, A_RDM_SUPCSID, (s), strlen(s))
#define RDMServer_SetLastModified(c) \
	SOIF_Replace((c)->soif, A_RDM_SD_LMT, (s), strlen(s))
#define RDMServer_SetExpires(c) \
	SOIF_Replace((c)->soif, A_RDM_SD_EXPIRE, (s), strlen(s))
#define RDMServer_SetDescription(c) \
	SOIF_Replace((c)->soif, A_RDM_DESC, (s), strlen(s))
#define RDMServer_SetMaintainer(c) \
	SOIF_Replace((c)->soif, A_RDM_MAINT, (s), strlen(s))

/*  ---------------------------------------------------------------------- */

#ifdef __cplusplus
};
#endif

#endif	/* _RDM_H_ */
