#!/bin/sh
#
# Copyright (c) 2004, Sun Microsystems, Inc. 
# 
# RESTRICTED RIGHTS: Use, duplication or disclosure by the government is 
# subject to the restrictions as set forth in 
# subparagraph (c) (1) (ii) of the Rights in Technical Data and Computer
# Software Clause as DFARS 252.227-7013 and FAR 52.227-19. 
# 
# SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
# THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
# PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
# SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS
# SOFTWARE OR ITS DERIVATIVES. 
# 
# By using or copying this Software, Licensee agrees to abide by the 
# intellectual property laws, and all other applicable laws of the U.S., 
# and the terms of this license. 
# 

# ----------------------------------------------------------------------
#
#  crashdump-report.sh -- Script to print out System Crash Dump Analysis
#
#
#  Files this script creates:
#
#    /tmp/crashdump-report.{pid}
#
#  Notes:
#
#    This script must be run on the same kernel architecture and
#    under the same OS as the system crash dump we are analyzing.
#
# ----------------------------------------------------------------------

#
# Set cpu, thread, proc related parameters for i386
#
set_cpuparam_i386 ()
{
    case "$rel" in
    5.9)
        # x86 SunOS 5.9 supports only 32 bit.
        Cpu2Thread=c;  CpuIdle=10;  CpuNext=2c; Thread2Proc=b8; Proc2Cmd=2f4
        ;;
    5.10)	
        if [ `/usr/bin/isainfo -k` = "amd64" ]
        then
            # x86 SunOS 5.10 64 bit amd64 support
            Cpu2Thread=18;  CpuIdle=20;  CpuNext=60; Thread2Proc=168; Proc2Cmd=5b0
        else
            # x86 SunOS 5.10 32 bit.
            Cpu2Thread=10;  CpuIdle=14;  CpuNext=38; Thread2Proc=d4; Proc2Cmd=3dc
        fi
        ;;
    *)
        echo "Operating system release $rel is not supported by this script."
        exit
    esac
}


#
# Set cpu, thread, proc related parameters for sparc
#
set_cpuparam_sparc ()
{
    case "$rel" in
    5.6) # sparc solaris 5.6 only supports 32 bit.
         #
        Cpu2Thread=24; CpuIdle=10; CpuNext=2c; Thread2Proc=a8; Proc2Cmd=2b0
        ;;

    5.7) # sparc solaris 7 supports both 32 and 64 bit.
         # But the "file" command doesn't give the bit information.
         # So we get it from "isainfo" command.
         #
        osbit=`isainfo -b`
        if [ "$osbit" = "32" ]; then
            Cpu2Thread=c;  CpuIdle=10;  CpuNext=28; Thread2Proc=a8; Proc2Cmd=2f8
        elif [ "$osbit" = "64" ]; then
            Cpu2Thread=10;  CpuIdle=18;  CpuNext=48; Thread2Proc=130; Proc2Cmd=508
        else
            echo "Unknown Operating System architecture $osbit bit"
            exit
        fi
        ;;

    5.8) # sparc solaris 8 supports both 32 and 64 bit.
         #
        filebit=`file $corename | awk '{ print $5 }' | cut -f1 -d '-'`
        if [ "$filebit" = "32" ]; then
            Cpu2Thread=c;  CpuIdle=10;  CpuNext=30; Thread2Proc=ac; Proc2Cmd=300
        elif [ "$filebit" = "64" ]; then
            Cpu2Thread=10;  CpuIdle=18;  CpuNext=50; Thread2Proc=130; Proc2Cmd=510
        else
            echo "Unknown file architecture $osbit bit"
            exit
        fi
        ;;

    5.9) # sparc solaris 9 supports both 32 and 64 bit.
         #
        filebit=`file $corename | awk '{ print $5 }' | cut -f1 -d '-'`
        if [ "$filebit" = "32" ]; then
            Cpu2Thread=c;  CpuIdle=10;  CpuNext=3c; Thread2Proc=a8; Proc2Cmd=2f0
        elif [ "$filebit" = "64" ]; then
            Cpu2Thread=10;  CpuIdle=18;  CpuNext=50; Thread2Proc=118; Proc2Cmd=480
        else
            echo "Unknown file architecture $osbit bit"
            exit
        fi
        ;;
    5.10) # sparc solaris 10 supports both 32 and 64 bit.
        #
        filebit=`file $corename | awk '{ print $5 }' | cut -f1 -d '-'`
        if [ "$filebit" = "32" ]; then
            Cpu2Thread=c;  CpuIdle=10;  CpuNext=3c; Thread2Proc=a8; Proc2Cmd=2f0
        elif [ "$filebit" = "64" ]; then
            Cpu2Thread=18;  CpuIdle=28;  CpuNext=70; Thread2Proc=138; Proc2Cmd=590
        else
            echo "Unknown file architecture $osbit bit"
            exit
        fi
        ;;

    *)
        echo "Operating system release $rel is not supported by this script."
        exit
    esac
}


#
# Main entry point of the script
#
if [ ! $# = 2 ]
then
  echo
  echo "Usage:     crashdump-report.sh unixfile  corefile  > your-output-file"
  echo
  echo "Examples:  crashdump-report.sh  unix.3  vmcore.3"
  echo "           crashdump-report.sh  /dev/ksyms  /dev/mem"
  echo
  exit
fi

#
#  Make sure we can see the files.
#
if [ ! -f $1 ] && [ ! $1 = "/dev/ksyms" ]
then
  echo; echo "Unable to locate $1.  Exiting."
  echo; exit
fi
if [ ! -f $2 ] && [ ! $2 = "/dev/mem" ]
then
  echo; echo "Unable to locate $2.  Exiting."
  echo; exit
fi

corename=$2

PATH=${PATH}:/etc
export PATH

# Find the platform and release we're dealing with.
plat=`uname -p`
rel=`uname -r`
mach=`uname -m`

#
#  The cpu, thread, and u-area structures differ from platform to platform
#  and release to release.  Set the following variables so that we can get 
#  the name of the running processes later on: 
#      Cpu2Thread, CpuIdle, CpuNext, Thread2Proc, Proc2Cmd
#

if [ "$plat" = "i386" ]; then
    set_cpuparam_i386
elif [ "$plat" = "sparc" ]; then
    set_cpuparam_sparc
else
    echo "Operating system platform $plat is not supported by this script."
    exit
fi


#
#  Okay, we are ready!  Create and put some stuff into output file.
#
cat <<EOC
******************************************************************************
System Crash Dump Analysis Output                          crashdump-report.sh 
`date`
******************************************************************************

EOC

#
#  We will be creating our own macros for use in adb sessions.
#  Odds are we will create a unique filename if we use the PID.
#  If it is not unique, let the user do something about it.
#
WHERE=/tmp/crashdump-report.$$
if [ -f $WHERE ]
then
  echo; echo "File $WHERE already exists.  Exiting."
  echo; exit
fi

#
#  The following will get the name of the processes running on each
#  CPU.  The structures from release to release differ, so we use
#  variables throughout.  The variables were computed earlier, above.
#  (Rev 1.5 edit)
#
 
cat > $WHERE <<EOC
*(<c+${CpuNext})>n
<c+${Cpu2Thread}/X"Thread address"
*(<c+${Cpu2Thread})>p
<p+${Thread2Proc}/X"Proc address"
*(<p+${Thread2Proc})>j
<j+${Proc2Cmd}/s
.,#((*(<c+${Cpu2Thread}))-(*(<c+${CpuIdle})))="This CPU was idle"
0,#(#(<n))&#(#(<n-<e))=n"Next CPU..."n
<n>c
<n,#(#(<n))&#(#(<n-<e))\$<$WHERE
EOC


if [ "$rel" = "5.6" -o "$rel" = "5.7" -o "$rel" = "5.8" -o "$rel" = "5.9" ]
then

#
#  Get initial information from adb
#
cat <<EOC
************************************
**  Initial information from adb  **
************************************

EOC

adb -k $1 $2 <<EOA
\$<utsname
srpc_domain/s16t"Domain name"
lbolt>a
*time-(*<a%0t100)=Y16t"Time of boot"
time/y16t"Time of crash"
,#(*audit_active)=n"Auditing is not enabled"
,##(*audit_active)=n"Auditing is enabled"
,#(*quotas_initialized)=n"Quotas are not enabled"
,##(*quotas_initialized)=n"Quotas are enabled"
=nn"**  Panic String  **"
="--------------------"
*panicstr/s
=nn"**  Stack Backtrace  **"
="-----------------------"
\$c
=nn"**  Per CPU information  **"
="---------------------------"
ncpus/X"# of CPUs present"
ncpus_online/X"# of CPUs online"
=nn
*cpu_list>c
<c>e
<c,#(<c)="The cpu_list pointer is NULL.  Not a good sign."n
<c,#(#(<c))\$<$WHERE
=nn
.,##(*(utsname+408)-63)="**  Stacktrace  **"
.,##(*(utsname+408)-63)="-----------------"
<sp,##(*(utsname+408)-63) $<stacktrace
=nn
="**  CPU structures  **"
="--------------------"
\$<cpus
=nn
="**  Msgbuf  **"
="------------"
\$<msgbuf
EOA

else

cat <<EOC
************************************
**  Initial information from mdb  **
************************************

EOC
mdb -k $1 $2 <<EOA
="** Status for core file **"
::status
=n"** System information **"
utsname::print struct utsname
srpc_domain/s16t"Domain name"
lbolt>a
*time-(*<a%0t100)=Y16t"Time of boot"
time/y16t"Time of crash"
,#(*audit_active)=n"Auditing is not enabled"
,##(*audit_active)=n"Auditing is enabled"
,#(*quotas_initialized)=n"Quotas are not enabled"
,##(*quotas_initialized)=n"Quotas are enabled"
=nn"**  Panic String  **"
="--------------------"
*panicstr/s
=nn"**  Stack Backtrace  **"
="-----------------------"
\$c
=nn"**  Per CPU information  **"
="---------------------------"
ncpus/X"# of CPUs present"
ncpus_online/X"# of CPUs online"
=nn
*cpu_list>c
<c>e
<c,#(<c)="The cpu_list pointer is NULL.  Not a good sign."n
<c,#(#(<c))\$<$WHERE
=nn"CPU/Thread"
::cpuinfo
=nn"** Stacktrace for the kernel threads that each cpu was running. **"
::walk cpu| ::print cpu_t cpu_thread | ::findstack -v
=nn
="**  CPU structures  **"
="--------------------"
::walk cpu|::print struct cpu
=nn
="**  Msgbuf  **"
="------------"
::msgbuf
EOA

fi


if [ "$mach" = "sun4d" -o "$mach" = "sun4m" ]
then
cat > $WHERE <<EOC
=nn"**  System is sun4m architecture - Checking enable_sm_wa  **"
="------------------------------------------------------------"
cpunodes/s
enable_sm_wa/D
,##(*enable_sm_wa)="Fix to bug 1146924 is installed"
,#(*enable_sm_wa)="Fix to bug 1146924 is not installed"
EOC

adb -k $1 $2 <<EOA
0,#(*(utsname+408)-6d000000)\$<$WHERE
EOA

fi


#
#  Get process information by using crash/mdb.
#
case "$rel" in
    5.4|5.5*|5.6|5.7|5.8)
cat <<EOC


**************************************
**  Process information from crash  **
**************************************

EOC
crash -d $2 -n $1 <<EOC
p -e
EOC
    ;;
5.9|5.10)
cat <<EOC


**************************************
**  Process information from mdb **
**************************************

EOC
/usr/bin/mdb $1 $2  <<EOC
::ps
$q
EOC
    ;;
esac


#
#	Get more of the message buffer, if possible, because the
#	msgbuf macro has a tendency to truncate things.
#
case "$rel" in
5.4|5.5*|5.6)
#
#
#  Get strings output to capture message buffer.
#
cat <<EOC



******************************************************
**  Strings output of complete message ring buffer  **
******************************************************

EOC
strings $2 | head -100

	;;

#
# For 5.7/5.8, the message buffer is not contained in the first
# part of memory, so strings won't work.  It's now in a streams
# buffer.
#
5.7|5.8)
	;;

*)
	;;
esac


#
#  Additional adb stuff
#
cat <<EOC

***********************
**  Some Statistics  **
***********************
EOC

#
#  Get DNLC statistics.  Divide all stats by 1000 for the larger
#  servers which tend to get odd hit rate percentages out of vmstat
#  due to using integer arithmetic.
#
# This works up through 2.6, but Solaris 7 and 8 have changed
# things.  The long_enter and long_look entries are deprecated in
# 5.7, and the whole accounting method changes in 5.8 (although the
# old structure is maintained for backward compatibility).
#
case "$rel" in
5.7|5.8)
adb -k $1 $2 <<EOA
=nn"**  Directory Name Lookup Cache Statistics  **"
="----------------------------------------------"
ncsize/D"Directory name cache size"
ncstats/D"# of cache hits that we used"
+/D"# of misses"
+/D"# of enters done"
+/D"# of enters tried when already cached"
ncstats+18/D"# of times entry moved to front"
+/D"# of purges of cache"
="(See /usr/include/sys/dnlc.h for more information)"
EOA
	;;
5.*)
adb -k $1 $2 <<EOA
=nn"**  Directory Name Lookup Cache Statistics  **"
="----------------------------------------------"
ncsize/D"Directory name cache size"
ncstats/D"# of cache hits that we used"
+/D"# of misses"
+/D"# of enters done"
+/D"# of enters tried when already cached"
+/D"# of long names tried to enter"
+/D"# of long name tried to look up"
+/D"# of times LRU list was empty"
+/D"# of purges of cache"
*ncstats%1000>a
*(ncstats+4)%1000>b
*(ncstats+14)%1000>c
<a+<b+<c>n
<a*0t100%<n=D"Hit rate percentage"
="(See /usr/include/sys/dnlc.h for more information)"
EOA
	;;
esac


#
#  Not all stats are available on all releases of the OS, so
#  back to checking the release before we go on...
#
case "$rel" in
5.4|5.5*|5.6|5.7|5.8)
cat <<EOC

**  Kernel Memory Request Statistics from crash **
    -------------------------------------------
EOC
crash -d $2 -n $1 <<EOC
kmastat
EOC
    ;;

5.9|5.10)
cat <<EOC

**  Kernel Memory Request Statistics from mdb **
    -----------------------------------------
EOC
/usr/bin/mdb $1 $2 << EOC
::kmastat
$q
EOC
     ;;

5.*)
adb -k $1 $2 <<EOA
=nn"**  Kernel Memory Request Statistics  **"
="----------------------------------------"
="Small"16t"Large"16t"Outsized"
kmeminfo/3X"Owned by kmem"
+/3X"Mem allocated"
+/3X"# of failures"n
pagesize/D"Memory page size"
="(See /usr/include/sys/sysinfo.h for more information)"
=nn"**  Streams Statistics  **"
="--------------------------"
="In use"16t"Total"16t"Maximum"16t"Failures"
strst/4X"Streams"
+/4X"Queues"
+/4X"MsgBlks"
+/4X"LinkBlks"
="(See /usr/include/sys/strstat.h for more information)"
EOA
	;;
*)
	;;
esac
#
#  end of case
#


if [ "$rel" = "5.6" -o "$rel" = "5.7" -o "$rel" = "5.8" ]
then

#
#  Print out some of the tunable variables.  Do this via macros
#  so that we don't get alot of symbol not found messages if
#  the driver was not modloaded for use.
#
cat > $WHERE <<EOC
=nn"**  Shared Memory Tuning Variables (if in use)  **"
="--------------------------------------------------"
shminfo_shmmax/D"Max segment size"
+/D"Min segment size"
+/D"Max identifiers"
+/D"Max attached shm segs per proc"
EOC

adb -k $1 $2 <<EOA
\$<$WHERE
EOA

cat > $WHERE <<EOC
=nn"**  Semaphore Tuning Variables (if in use)  **"
="----------------------------------------------"
seminfo_semmap/D"Entries per map"
+/D"Max identifiers"
+/D"Max in system"
+/D"Max undos"
+/D"Max sems per id"
+/D"Max ops per semop"
+/D"Max undos per proc"
+/D"Max bytes in undos"
+/D"Max sem value"
+/D"Max adjust on exit"
EOC

adb -k $1 $2 <<EOA
\$<$WHERE
EOA


cat > $WHERE <<EOC
=nn"**  Message Queue Tuning Variables (if in use)  **"
="--------------------------------------------------"
msginfo_msgmap/D"Max entries in map"
+/D"Max message size"
+/D"Max bytes on queue"
+/D"Max msg queue ids"
+/D"Max segment size (word size multiple)"
+/D"Max system message headers"
+/d16t"Max msg segments (must be < 32768)"
EOC

adb -k $1 $2 <<EOA
\$<$WHERE
EOA

else
# Solaris 9 and Solaris 10
mdb -k $1 $2 <<EOA
=nn"** System V IPC information **"
::ipcs
=n"** Threads blocking on Conditional Variables or Semaphores **"
::wchaninfo -v
$q
EOA

fi


##
## optional - eats up lots of space
##
#
#############################################
#panicstring=`echo "*panicstr/s" | adb -k $1 $2 | sed -e '1d' -e 's/^.*	//' | head -1`
#if [ "$panicstring" = "zero" ] ; then
#adb -k $1 $2 << EOA
#\$<threadlist
#EOA
#fi
#

rm $WHERE

#
##	Get some basic configuration information which is always
##	useful.
#
cat << EOC

************************************
**  Current patch revision status **
************************************
EOC
showrev -p

cat << EOC

****************************************
** Hardware Configuration Information **
****************************************
EOC
prtconf -vp

echo "*****"
echo "Done!"

############################################
#
#  +---------------------------------------------------------------------+
#  |  For more information about system crash dump analysis, refer to    |
#  |  the SunSoft Press book, "Panic! UNIX System Crash Dump Analysis",  |
#  |  ISBN 0-13-149386-8, published by Prentice Hall.                    |
#  +---------------------------------------------------------------------+
#
############################################
#
#  end of crashdump-report.sh
#
