// "$Id: ThreadsWindow.java,v 1.59 2002/07/11 00:44:11 gah Exp $ Sun Microsystems"
/*
 * {START_JAVA_COPYRIGHT_NOTICE
 * Copyright 2002 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 * END_COPYRIGHT_NOTICE}
 */


package com.sun.tools.dbxgui.debugger;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.io.ObjectStreamException;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

import org.openide.*;
import org.openide.windows.*;
import org.openide.util.actions.SystemAction;
import org.openide.actions.PropertiesAction;
import org.openide.util.HelpCtx;

import org.openide.nodes.Node;
import org.openide.nodes.Children;
import org.openide.nodes.AbstractNode;
import org.openide.nodes.Sheet;
import org.openide.nodes.PropertySupport;
import org.netbeans.modules.debugger.support.View2;
import org.netbeans.modules.debugger.support.DelegatingView2;
import java.lang.reflect.InvocationTargetException;
import java.io.IOException;

import com.sun.forte.st.glue.dbx.*;

import com.sun.tools.dbxgui.debugger.options.DebuggingOption;
import com.sun.tools.dbxgui.debugger.actions.*;
import com.sun.tools.dbxgui.utils.*;
import com.sun.tools.dbxgui.DbxGuiModule;


/**
 * Window where you can view the list of threads. A context menu
   allows you to select which of the columns should be visible.
   Selection sets the current thread context.
   <p>
   Todo:
   <ul>
    <li> Right clicking on a header doesn't work
    <li> Popup menu should work everywhere
    <li> How do I get proportional column display? If I turn off auto
         resizing, the columns don't fill up the width. Ah, perhaps this
	 is where I can query the size of the viewport and apply my
	 own heuristics?? But once I resize I need to be told so that
	 I can recompute. Then of course there's the issue of preserving
	 the user's manual column size adjustments...
    <li> Use only system actions, not JMenuItems directly
    <li> Prevent unselection of the current thread?
    <li> Perhaps show a thread count somewhere as is done in WorkShop?
    <li> Hyperlinks to editor for Start Function and Current Function
    <li> Display all fields correctly:
    <li> TreeTable implementation for thread groups
    <li> Should have some kind of thread-related breakpoints setting
         from the threads component
    <li> When multiple rows are selected, desensitize thread specific
         operations (and stack etc. should go blank)
    <li> Bug: if you click on the currently selected thread, it currently
         doesn't reactivate the link! This is a limitation of the JTable
	 MouseInputListener (mousePressed) in plaf/basic/BasicListUI.java .
    <li> It looks like I'll have to write my own MouseInputListener to get
         past these limitations (no unselection, no reselection).
    <li> The stacksize field is just set to -1. We need DbxFrame to
         include this info.
   </ul>
 */
public final class ThreadsWindow extends DebuggerTableWindow {

    /** generated Serialized Version UID */
    static final long serialVersionUID = -4952954247683058294L;

    /** Currently selected thread */
    private IpeThread selectedThread = null;
    
    private static ThreadsWindow singleton = null;
    public static ThreadsWindow getSingleton() {
	return singleton;
    }
    
    /** Create a new ThreadsWindow tied to the given debugger. */
    public ThreadsWindow(DbxDebugger debugger, SessionDataSource source) {
	super(debugger, source);
	if (singleton == null) {
	    singleton = this;
	}
	createComponents();
	initializeA11y();
	setInView(true);
	// setNodes();
	addPopupListener(this);
    }

    public ThreadsWindow() {
	this(null, null);
    }

    /** Get access to the data source for this component */
    public void setSession(DbxDebugSession oldSession,
			   DbxDebugSession newSession) {
	/*System.out.println("ThreadsWindow.setSession " + newSession + " old:" + oldSession);

	if (oldSession != null) {
	    System.out.println("  old: " + oldSession.getShortName());
	}
	if (newSession != null) {
	    System.out.println("  new: " + newSession.getShortName());
	}
	*/
	
	if (oldSession != null) {
	    DbxDebuggerEngine engine = oldSession.getEngine();
	    if (engine != null) {
		Dbx dbx = (Dbx)engine;
		dbx.removeThreadsListener(this);
	    }
	}
	if (newSession != null) {
	    DbxDebuggerEngine engine = newSession.getEngine();
	    if (engine != null) {
		Dbx dbx = (Dbx)engine;
		if (dbx.hasThreadsListeners()) {
		    // We already have stack listeners over there,
		    // so we're not going to get a new update. Since
		    // we've cleared out the old contents (in componentHidden),
		    // fetch the values that the other listener has
		    // most recently received
		    show(dbx.getTotalThreads(),
			 dbx.getShownThreads(),
			 dbx.getThreads(),
			 dbx.getThreadFlags());
		}
		dbx.addThreadsListener(this);
	    }
	    setMultiThreaded(newSession.isMultiThreaded());
	} else {
	    // Switched away from any sessions (no selection);
	    // clear out the window
	    //setMultiThreaded(false);
	    show(0, 0, null, 0);
	}
	
    }
    
    private void createComponents() {
	// open(); // XXX should it be here instead of below????
	setLayout (new BorderLayout ());

	notThreadedLabel = new JLabel(
		      DbxDebugger.getText("MSG_NotMultiThreaded"), // NOI18N
		      SwingConstants.LEFT);
	notThreadedLabel.setBorder(BorderFactory.createEtchedBorder());
	add(notThreadedLabel, java.awt.BorderLayout.CENTER);
	
	addPopupListener(notThreadedLabel);
	
	//createTable();
	//isThreaded = true;
	//add(tableScrollPane, java.awt.BorderLayout.CENTER);
	
	setName(DbxDebugger.getText("TITLE_ThreadsWindow")); //NOI18N
	setIcon(org.openide.util.Utilities.loadImage("org/netbeans/core/resources/threads.gif")); // NOI18N
    }

    /** Create a table-specific model */
    protected DebuggerTableModel createModel() {
	return new ThreadsTableModel();
    }
    
    /** The given item has been selected (if row == -1, nothing is selected) */
    protected void selectedItem(int row, Hyperlink l) {
	if (row != -1) {
	    // SelectedRow is selected
	    selectedThread = (IpeThread)model.getObjectAt(row);

	    // onSetCurrent();
	    
	    if (l != null) {
		l.activate(this);
	    }
	} else {
	    selectedThread = null;
	}
    }

    /** Enable items that depend on selection */
    protected void enableSelectionItems() {
	DbxDebugSession session = ((SessionDataSource)source).getSession();
	if (session == null) {
	    return;
	}
	DbxDebuggerEngine engine = session.getEngine();
	if (engine == null) {
	    return;
	}
	Dbx dbx = (Dbx)engine;	
	SystemAction.get(SuspendAction.class).setEnabled(dbx.isSuspendSupported());
	SystemAction.get(ResumeAction.class).setEnabled(dbx.isResumeSupported());
    }

    /** Disable items that depend on selection */
    protected void disableSelectionItems() {
	SystemAction.get(SuspendAction.class).setEnabled(false);
	SystemAction.get(ResumeAction.class).setEnabled(false);
    }
    

    
    // I probably need more about the event here in order to
    // determine which thread it was applied to, right?
    public void onSuspend() {
	// Suspend is invoked
	if (IpeUtils.trace) {
	    System.out.println("ThreadsWindow.onSuspend()"); // NOI18N
	}
	if (selectedThread == null)
	    return;
	DbxDebugSession session = ((SessionDataSource)source).getSession();
	if (session == null)
	    return;
	DbxDebuggerEngine engine = session.getEngine();
	if (engine == null)
	    return;
	engine.suspendThread(selectedThread.getId());
    }

    // I probably need more about the event here in order to
    // determine which thread it was applied to, right?
    public void onResume() {
	// Resume is invoked
	if (IpeUtils.trace) {
	    System.out.println("ThreadsWindow.onResume()"); // NOI18N
	}

	if (selectedThread == null)
	    return;
	DbxDebugSession session = ((SessionDataSource)source).getSession();
	if (session == null)
	    return;
	DbxDebuggerEngine engine = session.getEngine();
	if (engine == null)
	    return;
	engine.resumeThread(selectedThread.getId());
    }

    public void onSetCurrent() {
	if (IpeUtils.trace) {
	    System.out.println("ThreadsWindow.onSetCurrent()"); // NOI18N
	}

	if (selectedThread == null) {
	    Toolkit.getDefaultToolkit().beep();
	    return;
	}
	DbxDebugSession session = ((SessionDataSource)source).getSession();
	if (session != null) {
	    DbxDebuggerEngine engine = session.getEngine();
	    if (engine != null) {
		if (selectedThread == null) {
		    Toolkit.getDefaultToolkit().beep();
		} else {
		    engine.selectThread(selectedThread.getId());
		}
	    }
	}
    }
    
    /**
     * Show the list of threads
     * @param threads List of threads to show, of null if none
     */
    public void show(int tot, int shown,
		     IpeThread threads[], int flags) {

	if (IpeUtils.asserts) {
	    IpeUtils.ensure(SwingUtilities.isEventDispatchThread());
	}
	
	if (!isThreaded) {
	    if (tot > 0) {
		setMultiThreaded(true);
	    } else {
		return;
	    }
	}

	selectedRow = -1;
	sensitivity();

	((ThreadsTableModel)model).setThreads(shown, threads);
    }

    /**
     * Select the given thread
     * @param tid Thread Id
     */
    public void select(int tid) {
	((ThreadsTableModel)model).setSelectedThread(tid);
    }
    
    private boolean isThreaded = false;
    private JLabel notThreadedLabel = null;
    
    /** Indicate whether or not the program is multithreaded. If
	it is not, provide a useful message to the user. */
    public void setMultiThreaded(boolean threaded) {
	//System.out.println("In ThreadsWindow.setMultiThreaded(" + threaded + ")  ....isThreaded is " + isThreaded);

	if (IpeUtils.asserts) {
	    IpeUtils.ensure(SwingUtilities.isEventDispatchThread());
	}

	if (threaded == isThreaded) {
	    return;
	}
	isThreaded = threaded;
	if (threaded) {
	    // Hide the message about the program not being multithreaded
	    remove(notThreadedLabel);
	    if (tableScrollPane == null) {
		createTable();
	    }
	    add(tableScrollPane, java.awt.BorderLayout.CENTER);

	    // I would think validate would be enough to ensure that
	    // new geometry is calculated, but I had to call repaint() as
	    // well since I found that the validate() call wouldn't cause
	    // any changes until I resized or exposed the component
	    validate();
	    repaint();
	    
	    /* No point doing this here -- this window ain't listening
	       when it's not visible....
	    if (isThreaded && DebuggingOption.OPEN_THREADS.isEnabled()) {
		debugger.makeWindowVisible(this);
	    }
	    */
	} else {
	    // Add a message about the program being multithreaded
	    remove(tableScrollPane);
	    add(notThreadedLabel, java.awt.BorderLayout.CENTER);

	    validate();
	    repaint();
	    return;
	}
    }

    
    // Implements ComponentListener:

    /** The component is now showing: tell debugger to send us updates! */
    public void componentShown(ComponentEvent e) {
	if (IpeUtils.trace)
	    System.out.println("ThreadsWindow.componentShown!"); // NOI18N

	if (viewShowing) {
	    if (IpeUtils.asserts) {
		IpeUtils.ensure(false);
	    }
	    return;
	}
	
	// Start listening to the current stack
	if (source != null) {
	    DbxDebugSession session = ((SessionDataSource)source).getSession();
	    setSession(null, session);
	}

	super.componentShown(e);
    }
    
    /** The component is no longer showing: tell debugger to stop computing
	updates on our behalf! */
    public void componentHidden(ComponentEvent e) {
	if (IpeUtils.trace)
	    System.out.println("ThreadsWindow.componentHidden!"); // NOI18N

	if (!viewShowing) {
	    // For some reason, inside NetBeans top managers we often seem
	    // to get componentHidden BEFORE anything else!
	    // if (IpeUtils.asserts) {
	    //	  IpeUtils.ensure(false);
	    // }
	    return;
	}

	// Unregister ourselves from the old session if any
	if (source != null) {
	    DbxDebugSession session = ((SessionDataSource)source).getSession();
	    if (session != null) {
		setSession(session, null);
	    }
	}

	super.componentHidden(e);
    }

    /** Don't care - but must implement full ComponentListener interface */
    public void componentResized(ComponentEvent e) {
	// Don't care
    }
    
    /** Don't care - but must implement full ComponentListener interface */
    public void componentMoved(ComponentEvent e) {
	// Don't care
    }

    /** Table Model optimized for showing threads.
	It knows about a list of columns from which a subset of
	columns can be shown in the table. I'll refer to the
	complete set of columns as the physical columns, and the
	ones shown in the table as the virtual columns.
     */
    class ThreadsTableModel extends DebuggerTableWindow.DebuggerTableModel
	implements Comparator {

	public ThreadsTableModel() {
	    super(
	      12, // Total number of columns
	      4,  // Default number of visible columns (specified below)
	      -1, // Initial sorting column (-1 == don't sort)
	      
	      // Column headers
	      new String[] {
		  DbxDebugger.getText("Threads_Table_Name"), // NOI18N
		  DbxDebugger.getText("Threads_Table_ID"), // NOI18N
		  DbxDebugger.getText("Threads_Table_State"), // NOI18N
		  DbxDebugger.getText("Threads_Table_Priority"), // NOI18N
		  DbxDebugger.getText("Threads_Table_Flags"), // NOI18N
		  DbxDebugger.getText("Threads_Table_LWP"), // NOI18N
		  DbxDebugger.getText("Threads_Table_LWPAssoc"), // NOI18N
		  DbxDebugger.getText("Threads_Table_Signals"), // NOI18N
		  DbxDebugger.getText("Threads_Table_CurrentFunc"), // NOI18N
		  DbxDebugger.getText("Threads_Table_StartFunc"), // NOI18N
		  DbxDebugger.getText("Threads_Table_Address"), // NOI18N
		  DbxDebugger.getText("Threads_Table_StackSize") // NOI18N
	      },
	      // Tooltips for the columns
	      new String[] {
		  DbxDebugger.getText("Threads_Table_NameTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_IDTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_StateTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_PriorityTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_FlagsTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_LWPTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_LWPAssocTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_SignalsTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_CurrentFuncTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_StartFuncTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_AddressTip"), // NOI18N
		  DbxDebugger.getText("Threads_Table_StackSizeTip") // NOI18N
	      },
	      // Which columns are visible? (user selectable)
	      new boolean[] {
		  false, true, true,
		  false, false, false, false, false, true,
		  true, false, false
	      },
	      // Visible column mapping. Only defined up to index
	      // numVisible-1.
	      new int[] {
		  1, 2, 8, 9, -1, -1, -1, -1, -1, -1, -1, -1
	      },
	      // Preferred width
	      new int[] {
		  75, 50, 50, 50, 50, 50, 50, 75, 150, 100, 75,
		  75
	      },
              new char[] {
		  DbxDebugger.getText("Threads_Table_NameMnemonic").charAt(0),        //NOI18N
		  DbxDebugger.getText("Threads_Table_IDMnemonic").charAt(0),          //NOI18N
		  DbxDebugger.getText("Threads_Table_StateMnemonic").charAt(0),       //NOI18N
		  DbxDebugger.getText("Threads_Table_PriorityMnemonic").charAt(0),    //NOI18N
		  DbxDebugger.getText("Threads_Table_FlagsMnemonic").charAt(0),       //NOI18N
		  DbxDebugger.getText("Threads_Table_LWPMnemonic").charAt(0),         //NOI18N
		  DbxDebugger.getText("Threads_Table_LWPAssocMnemonic").charAt(0),    //NOI18N
		  DbxDebugger.getText("Threads_Table_SignalsMnemonic").charAt(0),     //NOI18N
		  DbxDebugger.getText("Threads_Table_CurrentFuncMnemonic").charAt(0), //NOI18N
		  DbxDebugger.getText("Threads_Table_StartFuncMnemonic").charAt(0),   //NOI18N
		  DbxDebugger.getText("Threads_Table_AddressMnemonic").charAt(0),     //NOI18N
		  DbxDebugger.getText("Threads_Table_StackSizeMnemonic").charAt(0)    //NOI18N
              });
	}

	/** Threads data */
	private IpeThread[] threads = null;

	/** Return the value of a particular cell in the table */
        public Object getRealValueAt(int row, int realColumn) {
	    if ((threads == null) || (numRows == 0)) {
		return "";
	    }
	    IpeThread t = threads[row];
	    /*
	    if (t == null) {
		return "";
	    }
	    */
	    switch (realColumn) {
		// Name -- JAVA only. Same as Root function!
	    case 0: return t.getRootFunction();
		// ID
	    case 1: return t.getTID();
	    	// State
	    case 2: return t.getState();
	        // Priority
	    case 3: return "?";
		// Flags
	    case 4: return "?";
		// LWP
	    case 5: return t.getLID();
		// LWP Association
	    case 6: return t.getLwpAssociation();
		// Signals
	    case 7: return "?";
		// Current Function
	    case 8: return t.getCurrentFunction();
		// Start Function
	    case 9: return t.getRootFunction();
		// Address
	    case 10: return t.getAddress();
		// Stack Size
	    case 11: return "?";
		// Unused: "event", "current", "lrelation", "db_suspended",
		// "stop_reason"
	    default:
		// Error
		if (IpeUtils.asserts)
		    IpeUtils.ensure(false);
		return "";
	    }
        }

	// Extra methods (these are additional ThreadsModel methods,
	// not AbstractTableModel methods)

	/** Return the thread on a particular row */
	public Object getObjectAt(int row) {
	    return threads[row];
	}

	/** Return a hyperlink for the given row, column */
	public Hyperlink getRealLink(int row, int realColumn) {
	    IpeThread t = threads[row];
	    switch (realColumn) {

	    // Current Function
	    case 8: {
		if (t.getCurrentFunction() instanceof Hyperlink) {
		    return (Hyperlink)t.getCurrentFunction();
		} else {
		    return null;
		}
	    }

	    // Start Function
	    case 9: {
		if (t.getRootFunction() instanceof Hyperlink) {
		    return (Hyperlink)t.getRootFunction();
		} else {
		    return null;
		}
	    }

	    // Not a hyperlink column
	    default: return null;
	    }
	}
	
	/** Return true iff this column can contain a hyperlink. */
	public boolean isRealColumnHyperLinked(int rcol) {
	    return ((rcol == 8) || (rcol == 9));
	}

	/** Return a cell renderer to be used for this column. */
	// XXX make "real" one of these as well
	protected DefaultTableCellRenderer getRealCellRenderer(int rCol) {
	    if (rCol == 1) {
		// Function: want my own cell renderer capable of
		// showing both images and text
		return new ImageLabelCellRenderer();
	    } else {
		// Use default - e.g. DefaultTableCellRenderer
		return null;
	    }
	}
	
	/** Compare function used to sort the threads */
	public int compare(Object o1, Object o2) {
	    IpeThread t1 = (IpeThread)o1;
	    IpeThread t2 = (IpeThread)o2;

	    if (t1 == null) {
		//System.out.println("ThreadsWindow.compare: t1 is null!");
		return -1;
	    }
	    if (t2 == null) {
		//System.out.println("ThreadsWindow.compare: t2 is null!");
		return 1;
	    }

	    int ret = 0;
	    switch(sortColumn) {
		// Name
	    case 0: {
		ret = 0;
		break; // Not valid: don't sort
	    }

	    // ID
	    case 1: {
		// XXX if we have names for Java threads, use those
		// here (and sort alphabetically)
		ret = t1.getId() - t2.getId(); break;
	    }

	    // State
	    case 2: {
		// Unfortunately, we have to create the state object
		// here, no easy way to do alphabetic comparison of
		// the glue dbx flags
		ret = t1.getState().compareTo(t2.getState()); break;
	    }

	    // Priority
	    case 3: {
		ret = 0;
		// XXX ret = t1.getPriority() - t2.getPriority();
		break;
	    }

	    // Flags
	    // XXX
	    case 4: {
		ret = 0;
		break; // Not valid: don't sort
	    }

	    // LWP
	    case 5: {
		ret = t1.getLWP() - t2.getLWP();
		break;
	    }

	    // LWP Association
	    // XXX
	    case 6: {
		ret = 0;
		break; // Not valid: don't sort
	    }

	    // Signals
	    case 7: {
		ret = 0;
		break; // Not valid: don't sort
	    }

	    // Current function
	    case 8: {
		String l = t1.getThread().current_function;
		String r = t2.getThread().current_function;
		if (l == null) {
		    if (r == null) {
			ret = 0;
		    } else {
			ret = -1;
		    }
		} else if (r == null) {
		    // we know l != null
		    ret = 1;
		} else {
		    ret = l.compareTo(r);
		}
		break;
	    }

	    // Start Function
	    case 9: {
		String l = t1.getThread().root_function;
		String r = t2.getThread().root_function;
		if (l == null) {
		    if (r == null) {
			ret = 0;
		    } else {
			ret = -1;
		    }
		} else if (r == null) {
		    // we know l != null
		    ret = 1;
		} else {
		    ret = l.compareTo(r);
		}
		break;
	    }

	    // Address
	    case 10: {
		ret = (int)(t2.getThread().address-t1.getThread().address);
		break; // only care about sign
	    }

	    // Stack Size
	    case 11: {
		ret = 0;
		break; // Not valid: don't sort
	    }

	    // Error
	    default:
		//		if (IpeUtils.asserts)
		//		    IpeUtils.ensure(false);
		ret = 0; break;
	    }

	    // Sort descending rather than ascending order?
	    if (sortDescending) {
		ret = -ret;
	    }
	    return ret;
	}

	/*
	private void printThreads() {
	    for (int i = 0; i < numRows; i++) {
		IpeThread t = threads[i];
		System.out.print(i + ") ");
		if (t == null) {
		    System.out.println("<null>");
		}
		System.out.println(t.getTID() + " " + t.getCurrentFunction());
	    }
	}
	*/
	
	/** Resort the table and update */
	public void sort() {
	    if ((numRows > 1) && (sortColumn != -1)) {
		sort(threads);
	    }

	    // XXX Later we might be able to do a bit better job of
	    // determining how much changed, and perhaps indicate
	    // that say only a single row changed (that might improve
	    // redraw performance and minimize flicker)
	    // Update GUI
	    fireTableDataChanged();

	    selectCurrentThread();
	}

	private void selectCurrentThread() {
	    if (IpeUtils.asserts) {
		IpeUtils.ensure(SwingUtilities.isEventDispatchThread());
	    }

	    // Select the current thread
	    if (threads != null) {
		for (int i = 0; i < numRows; i++) {
		    if (threads[i] != null && (threads[i].isCurrent())) {
			// Found the current thread: mark it as selected
			ignoreSelection = true;
			table.changeSelection(i, 0, false, false);
			selectedThread = threads[i];
			ignoreSelection = false;
			break;
		    }
		}
	    }
	}

	/** Select a particular thread */
	public void setSelectedThread(int tid) {
	    if (IpeUtils.asserts) {
		IpeUtils.ensure(SwingUtilities.isEventDispatchThread());
	    }

	    for (int i = 0; i < numRows; i++) {
		if (threads[i].getId() == tid) {
		    // Found the current thread: mark it as selected
		    threads[i].getThread().current = true;
		    threads[i].refresh();
		    ignoreSelection = true;
		    table.changeSelection(i, 0, false, false);
		    ignoreSelection = false;
		    fireTableRowsUpdated(i, i);
		} else if (threads[i].getThread().current) {
		    threads[i].getThread().current = false;
		    threads[i].refresh();
		    fireTableRowsUpdated(i, i);
		}
	    }
	}
	
	/** Set the model data (the threads data structure) */
	public void setThreads(int newNum, IpeThread[] newThreads) {

	    // Sort threads according to sort criterion
	    // sort the newThreads array since we want to do a diff
	    // against the threads array afterwards

	    // XXX find out if dbx (or the OS) already sorts the threads
	    // by thread id, and if so, optimize out sorting by column 1
	    if ((newNum > 1) && (sortColumn != -1)) {
		sort(newThreads);
	    }

	    // XXX Later we might be able to do a bit better job of
	    // determining how much changed, and perhaps indicate
	    // that say only a single row changed (that might improve
	    // redraw performance and minimize flicker)

	    // Well, let's take a first cut at it here...
	    // For each new thread, see if it was there in the old list...
	    // This is an O(n*n) algorithm, so don't do it for huge thread
	    // lists (for now, say n=100)
	    // For a huge list I can do it faster: copy the entire
	    // list into a LinkedList; then after each match remove
	    // the matched thread. This is still O(n*n) but faster for
	    // larger lists.
	    /*	    
	    if (threadIsNew == null) {
		if (newNum < 20) {
		    threadIsNew = new boolean[20];
		} else {
		    threadIsNew = new boolean[newNum];
		}
	    }
	    if (newNum > threadIsNew.length) {
		// Duplication of the Vector/ArrayList functionality
		// but (1) allow booleans instead of something derived
		// from Object to be inserted, and (2) allow direct
		// array manipulation
		boolean[] newArray = new boolean[threadIsNew.length*2];
		for (int i = 0; i < threadIsNew.length; i++) {
		    newArray[i] = threadIsNew[i];
		}
		threadIsNew = newArray;
	    }
	    if (newNum < 100) {
		for (int n = 0; n < newNum; n++) {
		    boolean isNew = true;
		    int tid = newThreads[n].tid;
		    for (int o = 0; o < numRows; o++) {
			if (threads[o].tid == tid) {
			    isNew = false;
			    break;
			}
		    }
		    threadIsNew[n] = isNew;
		}
	    } // else: disabled - don't bother showing diffs for large
	    // thread lists (or is this where users need it???)
	    */
	    
	    threads = newThreads;
	    numRows = newNum;
    
	    
	    // Update GUI
	    fireTableDataChanged();

	    // Select the current thread
	    selectCurrentThread();
	}
    }


    public void setNodes() {    
	if (isThreaded) {
	    super.setNodes();
	} else {
	    setActivatedNodes(null);
	}
    }
    
    /** Overrides superclass method. Gets actions for this top component. */
    public SystemAction[] getSystemActions() {
	//SystemAction[] sa = super.getSystemActions ();

	DebuggerTableColumnAction dtca = (DebuggerTableColumnAction)
	    SystemAction.get(DebuggerTableColumnAction.class);
        dtca.setTable(this);

	//SessionDataSourceAction sdsa = (SessionDataSourceAction)
	//    SystemAction.get(SessionDataSourceAction.class);
        //sdsa.setDebuggerWindow(this);

	DebugWinDockAction dock = (DebugWinDockAction)
	    SystemAction.get(DebugWinDockAction.class);
        dock.setDebuggerWindow(this);
	// XXX alternatively I could set it to DockIntoAction
	// if top component is in view!
	
	SystemAction[] threadActions = new SystemAction[] {
	    SystemAction.get(SetCurrentAction.class),
	    null,
	    SystemAction.get(SuspendAction.class),
	    SystemAction.get(ResumeAction.class),
	    null,
	    dtca,
	    null,
	    //sdsa,
	    //null,
	    dock,
	    /*
	    null,
	    SystemAction.get(PropertiesAction.class)
	    */
 	};
	//return SystemAction.linkActions (sa, threadActions);
	return threadActions;
    }

    // Implements PropertyChangeListener
    public void propertyChange(PropertyChangeEvent evt) {
	// Note - these strings have all been interned (they are all literals
	// so no need for full .equals check)
	if (SessionDataSource.PROP_SESSION_CHANGED == evt.getPropertyName()) {
	    // Handle session switching.
	    if (viewShowing) {
		DbxDebugSession oldSession = (DbxDebugSession)evt.getOldValue();
		DbxDebugSession session = (DbxDebugSession)evt.getNewValue();
		setSession(oldSession, session);
	    }

	    super.propertyChange(evt);

	    // XXX Later I need to translate this into a threads object
	    // change which in turn will lead to a property change
	    // broadcast to anyone listening to my selection.
	} else if (DbxDebugSession.PROP_THREADS_CHANGED == evt.getPropertyName()) {
	    // This is not ideal. Eventually (hopefully in time for
	    // Krakatoa) we will have a separate Threads object, which
	    // the Threads viewer is attached to. For now we're just
	    // fetching threads info out of the debug session engine.
	    //DbxDebugSession session = (DbxDebugSession)evt.getNewValue();
	    //if (session != null) {
	    //Dbx dbx = (Dbx)session.getEngine();
	    Dbx dbx = (Dbx)evt.getNewValue();
	    if (dbx != null) {
		show(dbx.getTotalThreads(),
		     dbx.getShownThreads(),
		     dbx.getThreads(),
		     dbx.getThreadFlags());
	    }
	} else if (DbxDebugSession.PROP_THREADSELECTION_CHANGED == evt.getPropertyName()) {
	    //DbxDebugSession session = (DbxDebugSession)evt.getNewValue();
	    //if (session != null) {
	    //Dbx dbx = (Dbx)session.getEngine();
	    Dbx dbx = (Dbx)evt.getNewValue();
	    if (dbx != null) {
		select(dbx.getSelectedThreadId());
	    }
	} else if (DbxDebugSession.PROP_ISTHREADED_CHANGED == evt.getPropertyName()) {
	    Dbx dbx = (Dbx)evt.getNewValue();
	    if (dbx != null) {
		DbxDebugSession session = dbx.getSession();
		if (session != null) {
		    setMultiThreaded(session.isMultiThreaded());
		}
	    }
	} else {
	    super.propertyChange(evt);
	}
    }

    protected void populateMenu(Object source, int xpos, int ypos,
				JPopupMenu popup) {
	if (IpeUtils.trace) {
	    System.out.println("In StackWindow.PopupAdapter.createPopup"); // NOI18N
	}

	// DebuggerTableWindow will use the position to select
	// the current column etc.
	super.populateMenu(source, xpos, ypos, popup);
	
	// Remainder of menu: use system actions
	addSystemActionsToMenu(popup);
    }
    
    public HelpCtx getHelpCtx() {
        return new HelpCtx("Debugging_threads"); // NOI18N
    }

    Object readResolve() throws ObjectStreamException {
	ThreadsWindow w = getSingleton();
	resolvedTo(w);
	return w;
    }
    
    /** Get the debugger view associated with this window */
    public View2 getView() {
	return DbxGuiModule.threadsView;
    }    

    /** Get the delegating debugger view associated with this window */
    public DelegatingView2 getDelegatingView() {
	return DbxGuiModule.threadsDView;
    }    
}
