// "$Id: BreakpointsWindow.java,v 1.46 2002/07/11 00:44:10 gah Exp $ Sun Microsystems"
/*
 * {START_JAVA_COPYRIGHT_NOTICE
 * Copyright 2002 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 * END_COPYRIGHT_NOTICE}
 */


package com.sun.tools.dbxgui.debugger;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.border.*;
import java.awt.*;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.io.ObjectStreamException;

import org.openide.*;
import org.openide.windows.*;
import org.openide.TopManager;
import org.openide.NotifyDescriptor;

import com.sun.forte.st.glue.dbx.*;

import org.openide.util.actions.*;
import org.openide.actions.CutAction;
import org.openide.actions.CopyAction;
import org.openide.actions.PasteAction;
import org.openide.actions.DeleteAction;
import org.openide.util.actions.SystemAction;
import org.openide.util.actions.CallableSystemAction;
import org.openide.util.HelpCtx;
import org.netbeans.modules.debugger.support.View2;
import org.netbeans.modules.debugger.support.DelegatingView2;


import com.sun.tools.dbxgui.debugger.actions.*;
import com.sun.tools.dbxgui.debugger.breakpoints.*;
import com.sun.tools.dbxgui.utils.*;
import com.sun.tools.dbxgui.DbxGuiModule;

/**
 * Window where you can view, and edit, the breakpoints
   <p>
   Todo:
   <ul>
    <li> Handler_batch_begin and _end: use it to reduce the number
         of redraws.
    <li> What is the "valid" attribute? Is it important?
    <li> I may remove the selection if I update the properties of
         an item which is selected
    <li> Implement the actions
    <li> Make sure that on session switch we replace the contents!
         Add as session listener, not just as breakpt listener...
    <li> Make it smart such that if you are not showing the window
         it only records the data (like the DbxEventRecord array), and
	 you only process this (to update columns etc. with the right
	 icon) when the window is made visible.
    <li> Handle caching the latest fired-handlers struct such that
         I can retrieve it!
    <li> Intern strings!	 
    <li> custom1, 2, 3
    <li> Change defaults
    <li> Complete the sorting function, the summary function, and the
         editing function!
    <li> Editing of rows
    <li> sorting
    <li> Reorganize the columns: get rid of column 0, function column;
         move countLimit up to count; get rid of unused ones,
	 improve the defaults: summary, enabled, count, countlimit.
	 Custom1? Defaults should probably depend on eventual width of the
	 window as well!
    <li> Scale the widths of the default column by the screen resolution?
         Or perhaps the font size? (Would also have to be locale sensitive...)
   </ul>
 */

public final class BreakpointsWindow extends DebuggerTableWindow
    implements ActionListener {

    /** generated Serialized Version UID */
    static final long serialVersionUID = -260872708500732373L;

    /** Currently selected handler */
    private IpeHandler selectedHandler = null;
    
    
    
    private static BreakpointsWindow singleton = null;
    public static BreakpointsWindow getSingleton() {
	return singleton;
    }
    
    /** Create a new BreakpointsWindow tied to the given debugger. */
    public BreakpointsWindow(DbxDebugger debugger, DataSource source) {
	super(debugger, source);
	if (singleton == null) {
	    singleton = this;
	}
	createComponents();
	setInView(true);
    }

    public BreakpointsWindow() {
	this(null, null);
    }
    
    /** Get access to the data source for this component */
    public void setSession(DbxDebugSession oldSession,
			   DbxDebugSession newSession) {
	/*
	System.out.println("StackWindow.setSession " + newSession + " old:" + oldSession);
	if (oldSession != null) {
	    System.out.println("  old: " + oldSession.getShortName());
	}
	if (newSession != null) {
	    System.out.println("  new: " + newSession.getShortName());
	}
	*/

	if (oldSession != null) {
	    DbxDebuggerEngine engine = oldSession.getEngine();
	    if (engine != null) {
		Dbx dbx = (Dbx)engine;
		dbx.removeBreakptListener(this);
	    }
	}
	if (newSession != null) {
	    DbxDebuggerEngine engine = newSession.getEngine();
	    if (engine != null) {
		Dbx dbx = (Dbx)engine;
		((BreakptTableModel)model).setHandlers(
			dbx.getNumHandlers(), dbx.getHandlers(),
			dbx.getEvents());
		dbx.addBreakptListener(this);
	    }
	} else {
	    // Switched away from any sessions (no selection);
	    // clear out the window
	    ((BreakptTableModel)model).setHandlers(0, null, null);
	}
    }
    
    /** Helper-function for startDebugger: creates GUI components
     *  such as the Dbx Commands Window
     */
    private void createComponents() {
	setLayout (new BorderLayout ());

	add(createTable(), java.awt.BorderLayout.CENTER);

	/* This didn't look good - see bug
	   4658487: breakpoints view shows vertical lines
	   so we no longer show vertical lines.
	// Parent will set showHorizontal lines -- we want to show vertical
	// lines as well since it's an editable table.
	table.setShowVerticalLines(true);
	*/
	
	setName(DbxDebugger.getText("TITLE_BreakpointsWindow")); //NOI18N
	setIcon(org.openide.util.Utilities.loadImage("org/netbeans/core/resources/breakpoints.gif")); // NOI18N
    }

    /** Create a table-specific model */
    protected DebuggerTableModel createModel() {
	return new BreakptTableModel();
    }
    
    /** The given item has been selected (if row == -1, nothing is selected) */
    protected void selectedItem(int row, Hyperlink l) {
	// SelectedRow is selected
	if (row != -1) {
	    selectedHandler= (IpeHandler)model.getObjectAt(row);
	    
	    if (l != null) {
		// XXX I don't have to (unless handlerno == visited???)
		// because switching the thread context automatically
		// triggers an editor request from dbx!
		// XXX Tor, you have to activate the link because
		// the link could be an address url!!!
		l.activate(this);
	    }
	} else {
	    selectedHandler = null;
	}
    }
    
    /** Enable items that depend on selection */
    protected void enableSelectionItems() {
	    SystemAction.get(SuspendAction.class).setEnabled(true);
    }

    /** Disable items that depend on selection */
    protected void disableSelectionItems() {
	    SystemAction.get(SuspendAction.class).setEnabled(false);
    }
    
    /** The component is now showing: tell debugger to send us updates! */
    public void componentShown(ComponentEvent e) {
	if (IpeUtils.trace)
	    System.out.println("BreakpointWindow.componentShown!"); // NOI18N

	if (viewShowing) {
	    if (IpeUtils.asserts) {
		IpeUtils.ensure(false);
	    }
	    return;
	}
	
	if (source != null) {
	    DbxDebugSession session = ((SessionDataSource)source).getSession();
	    setSession(null, session);
	}
	
	super.componentShown(e);
    }

    /** The component is no longer showing: tell debugger to stop computing
	updates on our behalf! */
    public void componentHidden(ComponentEvent e) {
	if (IpeUtils.trace)
	    System.out.println("BreakpointWindow.componentHidden!"); // NOI18N

	if (!viewShowing) {
	    // For some reason, inside NetBeans top managers we often seem
	    // to get componentHidden BEFORE anything else!
	    // if (IpeUtils.asserts) {
	    //	  IpeUtils.ensure(false);
	    // }
	    return;
	}
	
	selectedHandler = null;

	// Unregister ourselves from the old session if any
	if (source != null) {
	    DbxDebugSession session = ((SessionDataSource)source).getSession();
	    if (session != null) {
		setSession(session, null);
	    }
	}
	
	super.componentHidden(e);
    }

    /** Don't care - but must implement full ComponentListener interface */
    public void componentResized(ComponentEvent e) {
	// Don't care
    }
    
    /** Don't care - but must implement full ComponentListener interface */
    public void componentMoved(ComponentEvent e) {
	// Don't care
    }

    /** Create a combo box model suitable for editing breakpoint
	count limits */
    public static DefaultComboBoxModel createCountLimitModel() {
	DefaultComboBoxModel model =
	    new DefaultComboBoxModel(new String[] {
		"1", // NOI18N
		DbxDebugger.getText("CurrCount"), // NOI18N
		DbxDebugger.getText("Infinity"), // NOI18N
	    });
	return model;
    }

    
    /** Table Model optimized for showing breakpt.
	It knows about a list of columns from which a subset of
	columns can be shown in the table. I'll refer to the
	complete set of columns as the physical columns, and the
	ones shown in the table as the virtual columns.
     */
    class BreakptTableModel extends DebuggerTableWindow.DebuggerTableModel
	implements Comparator {

	public BreakptTableModel() {
	    super(
		15, // Total number of columns
		4, // Default number of visible columns (specified below)
		-1, // Initial sorting column (-1 == don't sort)
		// Column headers
		new String[] {
		  DbxDebugger.getText("Breakpt_Table_Summary"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_HID"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Custom1"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Custom2"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Custom3"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Enabled"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Valid"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Count"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Limit"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Temp"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Persistence"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Condition"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Thread"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_LWP"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Script") //NOI18N
		},
		// Column headers
		new String[] {
		  DbxDebugger.getText("Breakpt_Table_SummaryTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_HIDTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Custom1Tip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Custom2Tip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Custom3Tip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_EnabledTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_ValidTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_CountTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_LimitTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_TempTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_PersistenceTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_ConditionTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_ThreadTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_LWPTip"), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_ScriptTip") //NOI18N
		},
		/** Which columns are visible? (user selectable) */
		new boolean[] {
		  true,  false, false, false, false,
		  true, false, true, true, false,
		  false, false, false, false, false,
		  false, false
		},
		// Visible column mapping. Only defined up to index
		// numVisible-1.
		new int[] {
		  0, 5, 7, 8, -1,
		  -1, -1, -1, -1, -1,
		  -1, -1, -1, -1, -1,
		  -1, -1
		},
		// Preferred width
		new int[] {
		  200, 25,  100,  50, 50,
		  25, 25, 50, 50, 100,
		  100, 100, 100, 100, 100,
		  100
		},
                new char[] {
		  DbxDebugger.getText("Breakpt_Table_SummaryMnemonic").charAt(0),     //NOI18N
		  DbxDebugger.getText("Breakpt_Table_HIDMnemonic").charAt(0),         //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Custom1Mnemonic").charAt(0),     //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Custom2Mnemonic").charAt(0),     //NOI18N
		  DbxDebugger.getText("Breakpt_Table_Custom3Mnemonic").charAt(0),     //NOI18N
		  DbxDebugger.getText("Breakpt_Table_EnabledMnemonic").charAt(0),     //NOI18N
		  DbxDebugger.getText("Breakpt_Table_ValidMnemonic").charAt(0),       //NOI18N
		  DbxDebugger.getText("Breakpt_Table_CountMnemonic").charAt(0),       //NOI18N
		  DbxDebugger.getText("Breakpt_Table_LimitMnemonic").charAt(0),       //NOI18N
		  DbxDebugger.getText("Breakpt_Table_TempMnemonic").charAt(0),        //NOI18N
		  DbxDebugger.getText("Breakpt_Table_PersistenceMnemonic").charAt(0), //NOI18N
		  DbxDebugger.getText("Breakpt_Table_ConditionMnemonic").charAt(0),   //NOI18N
		  DbxDebugger.getText("Breakpt_Table_ThreadMnemonic").charAt(0),      //NOI18N
		  DbxDebugger.getText("Breakpt_Table_LWPMnemonic").charAt(0),         //NOI18N
		  DbxDebugger.getText("Breakpt_Table_ScriptMnemonic").charAt(0)       //NOI18N
                });
	}

	/** Breakpt data */
	//private ArrayList handlers = new ArrayList(20);
	private IpeHandler[] handlers = null;

	
	/** Return the value of a particular cell in the table */
        public Object getRealValueAt(int row, int realColumn) {
	    if ((handlers == null) || (numRows == 0)) {
		return "";
	    }
	    IpeHandler h = handlers[row];
	    /*
	    if (h == null) {
		// XXX why is this necessary?
		return null;
	    }
	    */
	    
	    switch (realColumn) {
		// Summary
	    case 0: return h.getSummary();

	    	// Handler ID
	    case 1: return h.getId();

		// Custom 1
	    case 2: return h.getCustom1();

		// Custom 2
	    case 3: return h.getCustom2();

		// Custom 3
	    case 4: return h.getCustom3();

		// Enabled
	    case 5: return h.getEnabled();

		// Valid
	    case 6: return h.getValid();

		// Count
	    case 7: return h.getCount();

		// Count Limit
	    case 8: return h.getCountLimit();

		// Temp
	    case 9: return h.getTemp();

		// Permanent
	    case 10: return h.getPermanent();

		// Condition
	    case 11: return h.getCondition();

		// Thread
	    case 12: return h.getThread();

		// LWP
	    case 13: return h.getLwp();

		// Script
	    case 14: return h.getScript();
		
	    default:
		// Error
		if (IpeUtils.asserts)
		    IpeUtils.ensure(false);
		return "";
	    }
        }
	
	public void setRealValueAt(Object value, int row, int realColumn) {
	    if (handlers == null) {
		return;
	    }

	    DbxDebugSession session =
		((SessionDataSource)source).getSession();
	    if (session == null) {
		// XXX or can I update rows in just the GUI
		// when there is no running debugger??
		return;
	    }
	    DbxDebuggerEngine engine = session.getEngine();
	    if (engine == null) {
		return;
	    }
	    
	    IpeHandler h = handlers[row];
	    
	    switch (realColumn) {

		// Custom 1
	    case 2: {
		if (h.getEvent() == null) {
		    // XXX internal error
		    if (IpeUtils.asserts) {
			IpeUtils.ensure(false);
		    }
		    return;
		}
		if (h.getEvent() instanceof LineBreakpoint) {
		    h.updateEvent(" at " + value.toString()); // NOI18N
		}
		// XXX Must implement MethodBreakpoint etc. here!
		engine.updateHandler(h, true);
		break;
	    }
		
		// Enabled
	    case 5: {
		int hid = h.getHid();

		if (value instanceof Boolean) {
		    engine.setHandlerEnabled(hid,
					     ((Boolean)value).booleanValue());
		}
		break;
	    }

	    case 7: {
		int hid = h.getHid();
		int count = h.getHandler().count;
		if (value instanceof Integer) {
		    count = ((Integer)value).intValue();
		} else if (value instanceof String) {
		    String s = (String)value;
		    if (s.equals(DbxDebugger.getText("Reset"))) { //NOI18N
			count = 0;
		    } else {
			try {
			    count = Integer.parseInt(s);
			} catch (java.lang.NumberFormatException e) {
			    // Restore field to show previous value?
			    // No, give the user a chance to edit it
			    // easily
			    return;
			}
		    }
		    if (!(count == h.getHandler().count)) {
			if (count == 0) {
			    engine.resetHandlerCount(hid);
			} else {
			    if (IpeUtils.asserts)
				IpeUtils.ensure(false);
			}
		    }
		//} else {
		//    System.out.println("BreakpointsWindow.setRealValueAt(" + row + "," + realColumn + ") is " + value);
		}
		break;
	    }

		// Count Limit
	    case 8: {
		int hid = h.getHid();

		if (value instanceof Integer) {
		    engine.setHandlerCountLimit(hid,
						((Integer)value).intValue());
		} else if (value instanceof String) {
		    String s = (String)value;
		    int count;
		    if (s.equals(DbxDebugger.getText("Infinity"))) { //NOI18N
			count = -1;
		    } else if (s.equals(DbxDebugger.getText("CurrCount"))) { //NOI18N
			int currCount = h.getHandler().count;
			if (currCount > 0) {
			    count = currCount;
			} else {
			    count = 0;
			}
		    } else {
			try {
			    count = Integer.parseInt(s);
			} catch (java.lang.NumberFormatException e) {
			    // Restore field to show previous value?
			    // No, give the user a chance to edit it
			    // easily
			    return;
			}
		    }
		    engine.setHandlerCountLimit(hid, count);
		//} else {
		//    System.out.println("BreakpointsWindow.setRealValueAt(" + row + "," + realColumn + ") is " + value);
		}
		break;
	    }

		// Temp
	    case 9: {
		if (value instanceof Boolean) {
		    boolean temp = ((Boolean)value).booleanValue();
		    h.updateModifier(temp, "-temp"); // NOI18N
		    engine.updateHandler(h, true);
		}
		break;
	    }

		// Permanent
	    case 10: {
		if (value instanceof Boolean) {
		    boolean temp = ((Boolean)value).booleanValue();
		    h.updateModifier(temp, "-perm"); // NOI18N
		    engine.updateHandler(h, true);
		}
		break;
	    }

		// Condition
	    case 11: {
		h.updateModifier("-if", value.toString()); // NOI18N
		engine.updateHandler(h, true);
		break;
	    }

		// Thread
	    case 12: {
		String thread = value.toString();
		// See if the user just entered an integer instead of the
		// proper form "t@<integer>".
		try {
		    int tid = Integer.parseInt(thread);
		    // Yes, the user did. Fix it.
		    thread = "t@" + Integer.toString(tid); // NOI18N
		} catch (java.lang.NumberFormatException e) {
		    // Nope, so let's just hope that value is of the proper
		    // form...
		}
		h.updateModifier("-thread", thread); // NOI18N
		engine.updateHandler(h, true);
		break;
	    }

		// LWP
	    case 13: {
		String lwp = value.toString();
		// See if the user just entered an integer instead of the
		// proper form "t@<integer>".
		try {
		    int tid = Integer.parseInt(lwp);
		    // Yes, the user did. Fix it.
		    lwp = "l@" + Integer.toString(tid); // NOI18N
		} catch (java.lang.NumberFormatException e) {
		    // Nope, so let's just hope that value is of the proper
		    // form...
		}
		h.updateModifier("-lwp", lwp); // NOI18N
		engine.updateHandler(h, true);
		break;
	    }

		/*
		
		// Summary
	    case 0: 

	    	// Handler ID
	    case 1: 

		// Custom 1
	    case 2: 

		// Custom 2
	    case 3: 

		// Custom 3
	    case 4: 

		// Valid
	    case 6: 


		// Script
	    case 14:
		*/		
	    default:
		if (IpeUtils.asserts) {
		    // Error: assignment not allowed here!
		    TopManager.getDefault().getErrorManager().log("BreakpointsWindow.setRealValueAt(" + row + "," + realColumn + ") is " + value + " : UNKNOWN COLUMN!"); // NOI18N
		}
	    }	    

	    // Redraw
	    updatedRealCell(row, realColumn);
	}

	
	// Extra methods (these are additional BreakptModel methods,
	// not AbstractTableModel methods)

	/*
	 * Return a hyperlink for the given row, column.
	 */
	public Hyperlink getRealLink(int row, int realColumn) {
	    // There are no hyperlinks in this table... except possibly
	    // for the custom fields...
	    switch (realColumn) {
	    case 2: {
		IpeHandler h = handlers[row];
		Object o = h.getCustom1();
		if (o instanceof Hyperlink) {
		    return (Hyperlink)o;
		} else {
		    return null;
		}
	    }
	    case 3: {
		IpeHandler h = handlers[row];
		Object o = h.getCustom2();
		if (o instanceof Hyperlink) {
		    return (Hyperlink)o;
		} else {
		    return null;
		}
	    }
	    case 4: {
		IpeHandler h = handlers[row];
		Object o = h.getCustom3();
		if (o instanceof Hyperlink) {
		    return (Hyperlink)o;
		} else {
		    return null;
		}
	    }
	    // Not a hyperlink column
	    default: return null;
	    }
	}
	
	
	/** Return true iff this column can contain a hyperlink. */
	public boolean isRealColumnHyperLinked(int rcol) {
	    return ((rcol == 2) || (rcol == 3) || (rcol == 4));
	}

	/** Return a cell renderer to be used for this column. */
	protected DefaultTableCellRenderer getRealCellRenderer(int rCol) {
	    if (rCol == 0) {
		// Summary: want my own cell renderer capable of
		// showing both images and text
		return new ImageLabelCellRenderer();
	    } else {
		// Use default - e.g. DefaultTableCellRenderer
		return null;
	    }
	}

	/** Return the cell editor to be used for this column.
	    If null, no editing.
	*/
	protected TableCellEditor getRealCellEditor(int rCol) {
	    switch(rCol) {
	    case 7: {
		// Count
		JComboBox comboBox = new JComboBox();
		comboBox.addItem("0"); // NOI18N
		comboBox.addItem(DbxDebugger.getText("Reset")); // NOI18N
		comboBox.setEditable(false);
		// Try to make the text right justified as well
		Component c = comboBox.getEditor().getEditorComponent();
		if (c instanceof JTextField) {
		    ((JTextField)c).setHorizontalAlignment(JTextField.RIGHT);
		}
		return new DefaultCellEditor(comboBox);
	    }
		// Count Limit
	    case 8: {
		DefaultComboBoxModel model = createCountLimitModel();
		JComboBox comboBox = new JComboBox(model);
		//comboBox.setModel(model);
		comboBox.setEditable(true);

		// Try to make the text right justified as well
		Component c = comboBox.getEditor().getEditorComponent();
		if (c instanceof JTextField) {
		    ((JTextField)c).setHorizontalAlignment(JTextField.RIGHT);
		}
		return new DefaultCellEditor(comboBox);
	    }
	    default: return null;
	    }
	}

        public Class getRealColumnClass(int rCol) {
	    // We want to make sure we set the type here, since in some
	    // columns (such as the count limit) the field can occasionally
	    // be a String ("infinity"), and we still want number-rendering
	    // for most of the column (it's okay if "infinity" is right
	    // justified)
	    switch (rCol) {
	    case 5:
	    case 6:
	    case 9:
	    case 10:
		return Boolean.class;
	    case 1:
	    case 7:
	    case 8:
	    case 12: // this is a string (thread: t@1) but behaves like int
		//XXX case 13: // this is a string (lwp: l@1) but behaves like int
		return Integer.class;
	    default:
		return getRealValueAt(0, rCol).getClass();
	    }
        }
	
	/*
	 * Don't need to implement this method unless your table's
	 * editable.
	 */
	public boolean isRealCellEditable(int row, int rcol) {
	    return ((rcol >= 2) && (rcol <= 13));
	}
	
	/** Compare function used to sort the breakpt handlers */
	public int compare(Object o1, Object o2) {
	    IpeHandler f1 = (IpeHandler)o1;
	    IpeHandler f2 = (IpeHandler)o2;

	    if (f1 == null) {
		//System.out.println("BreakpointWindow.compare: f1 is null!");
		//if (f2 != null) {
		//  System.out.println("  f2 was " + f2);
		//}
		return -1;
	    }
	    if (f2 == null) {
		//System.out.println("BreakpointWindow.compare: f2 is null!");
		//if (f1 != null) {
		//  System.out.println("  f1 was " + f1);
		//}
		return -1;
	    }

	    int ret = 0;

	    switch(sortColumn) {
	    case 0: {
		// ret = f1.eventspec.compareTo(f2.eventspec);
		// Actually, I think David John and I agreed that sorting
		// by this column should be done by dbx id! (creation order)
		// This is because the id column is typically not visible and
		// this is the "natural" ordering.
		if ((f1.getHandler() != null)  && f2.getHandler() != null) {
		    ret = f1.getHandler().id - f2.getHandler().id;
		} else if (f1.getHandler() == null) {
		    ret = -1;
		} else {
		    ret = 1;
		}
		break;
	    }

		// handler id
	    case 1: {
		if ((f1.getHandler() != null)  && f2.getHandler() != null) {
		    ret = f1.getHandler().id - f2.getHandler().id;
		} else if (f1.getHandler() == null) {
		    ret = -1;
		} else {
		    ret = 1;
		}
		break;
	    }

		// Custom 1
	    case 2: {
		ret = f1.getCustom1().compareTo(f2.getCustom1());
		break;
	    }
		
		// Custom 2
	    case 3: {
		ret = f1.getCustom2().compareTo(f2.getCustom2());
		break;
	    }
		
		// Custom 3
	    case 4: {
		ret = f1.getCustom3().compareTo(f2.getCustom3());
		break;
	    }
		
		// enabled
	    case 5: {
		ret = IpeUtils.boolCompare(f1.getEnabled().booleanValue(),
					   f2.getEnabled().booleanValue());
		break;
	    }

		// count
	    case 7: {
		int co1 = 0;
		int co2 = 0;
		if (f1.getHandler() != null) {
		    co1 = f1.getHandler().count;
		}
		if (f2.getHandler() != null) {
		    co2 = f2.getHandler().count;
		}
		ret = co1 - co2;
		break;
	    }
		
	    // Count Limit
	    case 8: {
		int co1 = 0;
		int co2 = 0;
		if (f1.getHandler() != null) {
		    co1 = f1.getHandler().count_limit;
		}
		if (f2.getHandler() != null) {
		    co2 = f2.getHandler().count_limit;
		}
		ret = co1 - co2;
		break;
	    }

		// temp
	    case 9: {
		ret = IpeUtils.boolCompare(f1.getTemp().booleanValue(),
					   f2.getTemp().booleanValue());
		break;
	    }
		
		// Permanent
	    case 10: {
		ret = IpeUtils.boolCompare(f1.getPermanent().booleanValue(),
					   f2.getPermanent().booleanValue());
		break;
	    }

		// Condition
	    case 11: {
		ret = f1.getCondition().compareTo(f2.getCondition());
		break;
	    }

		// Thread
	    case 12: {
		ret = f1.getThread().compareTo(f2.getThread());
		break;
	    }

		// LWP
	    case 13: {
		ret = f1.getLwp().compareTo(f2.getLwp());
		break;
	    }

		// Script
	    case 14: {
		ret = f1.getScript().compareTo(f2.getScript());
		break;
	    }
		
	    default: {
		ret = 0;
		if (IpeUtils.asserts) {
		    IpeUtils.ensure(false);
		}
		break; // Not valid key: don't sort
	    }
	    }
	    
	    // Sort descending rather than ascending order?
	    if (sortDescending) {
		ret = -ret;
	    }
	    return ret;
	}

	/** Resort the table and update */
	public void sort() {
	    if ((numRows > 1) && (sortColumn != -1)) {
		sort(handlers);

		// XXX Later we might be able to do a bit better job of
		// determining how much changed, and perhaps indicate
		// that say only a single row changed (that might improve
		// redraw performance and minimize flicker)
		// Update GUI
		fireTableDataChanged();
	    }
	}

	public void addHandler(IpeHandler h, int num,
			       IpeHandler[] newHandlers) {
	    if ((numRows > 1) && (sortColumn != -1)) {
		sort(newHandlers);
	    }
	    handlers = newHandlers;
	    numRows = num;

	    // Handlers are typically added to the end so look there
	    // first
	    if (handlers[numRows-1] == h) {
		// Yup! Appended!
		fireTableRowsInserted(numRows-1, numRows-1);
	    } else {
		int i = getHandlerIndex(h.getHid());
		if (i == -1) {
		    // Error: we didn't find the handler... refresh full
		    // table just in case
		    fireTableDataChanged();
		} else {
		    fireTableRowsInserted(i, i);
		}
	    }
	}

	/**
	 * Replace an existing handler
	 * @param handler The handler to replace handler.id with
	 */
	public final void replaceHandler(IpeHandler handler) {
	    // When handlers change the may affect the sorting...
	    if ((numRows > 1) && (sortColumn != -1)) {
		sort(handlers);
	    }

	    // Redraw the relevant row
	    // Look for the new handler to see which line it ended
	    // up on
	    int i = getHandlerIndex(handler.getHid());
	    if (i == -1) {
		// Error: we didn't find the handler... refresh full table just
		// in case
		fireTableDataChanged();
	    } else {
		fireTableRowsInserted(i, i);
	    }
	}

	/** Return the index of the particular handler */
 	private int getHandlerIndex(int id) {
	    for(int i= 0; i < numRows; i++) {
		// will (handlers[i] == h) work? Slightly faster, but not much
		if (handlers[i].getHid() == id) {
		    return i;
		}
	    }
	    return -1;
	}
	
	/**
	 * Delete a handler
	 * @param handler The handler to be deleted
	 */
	public void deleteHandler(IpeHandler h,
				  int num, IpeHandler[] newHandlers) {
	    // Remove the given row

	    // No need to sort - deletion doesn't affect order

	    // Find index before deletion
	    int index = getHandlerIndex(h.getHid());
	    if (index == -1) {
		// Error - just update the whole table
		fireTableDataChanged();
	    } else {
		fireTableRowsDeleted(index, index);
	    }

	    handlers = newHandlers;
	    numRows = num;
	    selectedRow = -1;
	}
	
    
	/**
	 * Make a handler defunct
	 * @param handler The handler to be made defunct
	 */
	public final void handlerDefunct(IpeHandler handler) {
	    int i = getHandlerIndex(handler.getHid());
	    if (i == -1) {
		// Error: we didn't find the handler... refresh full table just
		// in case
		fireTableDataChanged();
	    } else {
		fireTableRowsUpdated(i, i);
	    }
	}

    
	/**
	 * Make a handler enabled or disabled (according to
	 * handler.enabled)
	 * @param handler The handler to be enabled/disabled
	 */
	public final void enableHandler(IpeHandler handler) {
	    int i = getHandlerIndex(handler.getHid());
	    if (i == -1) {
		// Error: we didn't find the handler... refresh full table just
		// in case
		fireTableDataChanged();
	    } else {
		fireTableRowsUpdated(i, i);
	    }
	}
	
	/**
	 * The handler count (and or handler count limit) has been
	 * updated.
	 * @param handler The handler whose count/count limit is updated
	 */
	public final void handlerCountUpdated(IpeHandler handler) {
	    int i = getHandlerIndex(handler.getHid());
	    if (i == -1) {
		// Error: we didn't find the handler... refresh full table just
		// in case
		fireTableDataChanged();
	    } else {
		fireTableRowsUpdated(i, i);
	    }
	}

	/**
	 * The list of fired handlers has changed
	 */
	public final void firedHandlersUpdated(DbxEventRecord[] events,
					       boolean refresh) {

	    // XXX
	    /*
	      I probably should be doing the event<=>breakpoint correlation
	      in dbx, such that when we get multiple breakpoints windows for
	      the same session, this is computed only once. But for now
	      I want to prevent this work from being done when the breakpoints
	      window is not showing.

	      XXX IDEA: add a "refresh" attribute to each IpeHandler object
	      such that dbx can mark handlers in need of a refresh! That
	      way I can just iterate over the array!
	     */
	    
	    // Clear out previously fired handlers
	    for (int row = 0; row < numRows; row++) {
		IpeHandler h = handlers[row];
		if (h.isFired()) {
		    h.setFired(false);
		    if (refresh) {
			h.doRefresh = true;
		    }
		}
	    }

	    if (events != null) {
		for (int i = 0; i < events.length && events[i] != null; i++) {
		    //System.out.println("event[" + i + "]: hid " + events[i].hid + " origin " + events[i].origin + " descr " + events[i].description);
		    if (events[i].hid <= 0) { // XXX is that the right criterion?
			continue;
		    }

		    int index = getHandlerIndex(events[i].hid);
		    if (index != -1) {
			IpeHandler h = handlers[index];
			h.setFired(true);
			if (refresh) {
			    h.doRefresh = true;
			}
		    }
		}
	    }

	    if (refresh) {
		// Update the table
		for (int row = 0; row < numRows; row++) {
		    IpeHandler h = handlers[row];
		    if (h.doRefresh) {
			fireTableRowsUpdated(row, row);
			h.doRefresh = false;
		    }
		}
	    }
	}
	
	/**
	 * Set the list of breakpoints. This replaces the entire set
	 * of handlers, as opposed to the add/delete methods above.
	 * @param handlers A hashmap which lists all new breakpoints to
	 *  be inserted, or null, to delete all.
	 */
	public final void setHandlers(int num, IpeHandler[] newHandlers,
				      DbxEventRecord[] events) {
	    // XXX Optimize sorting by handler number - dbx already sends us
	    // this in order
	    if ((num > 1) && (sortColumn != -1)) {
		sort(newHandlers);
	    }

	    handlers = newHandlers;
	    numRows = num;
	    selectedRow = -1;

	    // Since I will redraw the whole table shortly I can
	    // avoid a redraw here
	    firedHandlersUpdated(events, false);
	    
	    fireTableDataChanged();
	}

	/** Cut & Delete support: delete the actual breakpoint */
	public void delete(int[] selectedRows) {
	    if (handlers == null) {
		return;
	    }
	    DbxDebugSession session = ((SessionDataSource)source).getSession();
	    if (session == null) {
		return;
	    }
	    DbxDebuggerEngine engine = session.getEngine();
	    if (engine == null) {
		return;
	    }

	    int n = selectedRows.length;
	    int[] hids = new int[n];
	    for (int i = 0; i < n; i++) {
		hids[i] = handlers[selectedRows[i]].getHid();
	    }
	    engine.deleteHandlers(hids);
	}


	/** On paste: Create the actual object */
	public void paste(Object[] objects) {
	    if (handlers == null) {
		return;
	    }
	    DbxDebugSession session = ((SessionDataSource)source).getSession();
	    if (session == null) {
		return;
	    }
	    DbxDebuggerEngine engine = session.getEngine();
	    if (engine == null) {
		// XXX why can't I paste into a dead session? Shouldn't these
		// be the same as breakpoints created in editor before
		// debugger start???
		return;
	    }
	    
	    int n = objects.length;
	    for (int i = 0; i < n; i++) {
		engine.updateHandler((IpeHandler)objects[i], false);
	    }
	}

	/** Return the handler on a particular row */
        public IpeHandler getHandler(int row) {
	    if (handlers == null) {
		return null;
	    }
	    if ((row >= 0) && (row < numRows)) {
		return handlers[row];
	    }
	    return null;
	}
	
	/** Return the breakpt handler on a particular row */
	public Object getObjectAt(int row) {
	    return handlers[row];
	}
    }

    /** Overrides superclass method. Gets actions for this top component. */
    public SystemAction[] getSystemActions() {
	//SystemAction[] sa = super.getSystemActions ();

	DebuggerTableColumnAction dtca = (DebuggerTableColumnAction)
	    SystemAction.get(DebuggerTableColumnAction.class);
        dtca.setTable(this);

	//SessionDataSourceAction sdsa = (SessionDataSourceAction)
	//  SystemAction.get(SessionDataSourceAction.class);
        //sdsa.setDebuggerWindow(this);

	DebugWinDockAction dock = (DebugWinDockAction)
	SystemAction.get(DebugWinDockAction.class);
        dock.setDebuggerWindow(this);
	// XXX alternatively I could set it to DockIntoAction
	// if top component is in view!
	
	SystemAction[] bptActions = new SystemAction[] {
            null,
	    dtca,
	    null,
	    //sdsa,
	    //null,
	    dock
	};
	//return SystemAction.linkActions (sa, bptActions);
	return bptActions;
    }

    // Implements PropertyChangeListener
    public void propertyChange(PropertyChangeEvent evt) {
	// Note - these strings have all been interned (they are all literals
	// so no need for full .equals check)
	if (SessionDataSource.PROP_SESSION_CHANGED == evt.getPropertyName()) {
	    // Handle session switching.
	    if (viewShowing) {
		DbxDebugSession oldSession = (DbxDebugSession)evt.getOldValue();
		DbxDebugSession session = (DbxDebugSession)evt.getNewValue();
		setSession(oldSession, session);
	    }

	    super.propertyChange(evt);

	    // XXX Later I need to translate this into a stackframe object
	    // change which in turn will lead to a property change
	    // broadcast to anyone listening to my selection.
	    
	} else if (Dbx.PROP_NEW_HANDLER.equals(evt.getPropertyName())) {
	    Dbx dbx = (Dbx)evt.getOldValue();
	    IpeHandler h = (IpeHandler)evt.getNewValue();
	    ((BreakptTableModel)model).addHandler(h,
			dbx.getNumHandlers(), dbx.getHandlers());
	} else if (Dbx.PROP_HANDLER_REPLACE.equals(evt.getPropertyName())) {
	    IpeHandler h = (IpeHandler)evt.getNewValue();
	    ((BreakptTableModel)model).replaceHandler(h);
	} else if (Dbx.PROP_HANDLER_DELETE.equals(evt.getPropertyName())) {
	    Dbx dbx = (Dbx)evt.getOldValue();
	    IpeHandler h = (IpeHandler)evt.getNewValue();
	    ((BreakptTableModel)model).deleteHandler(h,
			     dbx.getNumHandlers(), dbx.getHandlers());
	} else if (Dbx.PROP_HANDLER_DEFUNCT.equals(evt.getPropertyName())) {
	    IpeHandler h = (IpeHandler)evt.getNewValue();
	    ((BreakptTableModel)model).handlerDefunct(h);
	} else if (Dbx.PROP_HANDLER_ENABLE.equals(evt.getPropertyName())) {
	    IpeHandler h = (IpeHandler)evt.getNewValue();
	    ((BreakptTableModel)model).enableHandler(h);
	} else if (Dbx.PROP_HANDLER_COUNT.equals(evt.getPropertyName())) {
	    IpeHandler h = (IpeHandler)evt.getNewValue();
	    ((BreakptTableModel)model).handlerCountUpdated(h);
	} else if (Dbx.PROP_HANDLERS_FIRED.equals(evt.getPropertyName())) {
	    Dbx dbx = (Dbx)evt.getNewValue();
	    ((BreakptTableModel)model).firedHandlersUpdated(dbx.getEvents(), true);
	} else {
	    super.propertyChange(evt);
	}
    }

    public void onSetCurrent() {
	if (IpeUtils.trace) {
	    System.out.println("BreakpointsWindow.onSetCurrent()"); // NOI18N
	}
	
	if (selectedHandler == null) {
	    Toolkit.getDefaultToolkit().beep();
	    return;
	}
	if (UsageTracking.enabled) {
	    UsageTracking.sendAction("Show Breakpoint Source", null); // NOI18N
	}

	// Pick the next annotation
	BreakpointAnnotation annotation = selectedHandler.getNextAnnotation();
	if (annotation != null) {
	    annotation.show();
	}
    }

    public HelpCtx getHelpCtx() {
        return new HelpCtx("Debugging_breakpoints"); // NOI18N
    }

    /* Not yet fully implemented
    private JMenuItem editItem = null;
    */
    private JMenuItem gotoSourceItem = null;
    private JMenuItem enableItem = null;
    private JMenuItem disableItem = null;
    private JMenuItem deleteAllItem = null;
    public void actionPerformed(java.awt.event.ActionEvent actionEvent) {
	Dbx dbx = getDbx();
	if (dbx == null) {
	    return;
	}
	
	if (actionEvent.getSource() == gotoSourceItem) {
	    onSetCurrent();
	} else if (actionEvent.getSource() == enableItem) {
	    if (UsageTracking.enabled) {
		UsageTracking.sendAction("Enable Handler", null); // NOI18N
	    }

	    int[] selectedRows = table.getSelectedRows();
	    int n = selectedRows.length;
	    for (int i = 0; i < n; i++) {
		IpeHandler h =
		    ((BreakptTableModel)model).getHandler(selectedRows[i]);
		if (IpeUtils.asserts) {
		    IpeUtils.ensure(h != null);
		}
		int hid = h.getHid();
		dbx.setHandlerEnabled(hid, true);
	    }
	} else if (actionEvent.getSource() == disableItem) {
	    if (UsageTracking.enabled) {
		UsageTracking.sendAction("Disable Handler", null); // NOI18N
	    }

	    int[] selectedRows = table.getSelectedRows();
	    int n = selectedRows.length;
	    for (int i = 0; i < n; i++) {
		IpeHandler h =
		    ((BreakptTableModel)model).getHandler(selectedRows[i]);
		if (IpeUtils.asserts) {
		    IpeUtils.ensure(h != null);
		}
		int hid = h.getHid();
		dbx.setHandlerEnabled(hid, false);
	    }
	} else if (actionEvent.getSource() == deleteAllItem) {
	    if (UsageTracking.enabled) {
		UsageTracking.sendAction("Delete All Handlers", null); // NOI18N
	    }
	    dbx.deleteHandlers(new int[] { 0 });
	    /* Not yet fully implemented!
	} else if (actionEvent.getSource() == editItem) {
	    if (UsageTracking.enabled) {
		UsageTracking.sendAction("Edit Handler", null); // NOI18N
	    }
	    // It's never enabled so this error won't occur in the product -
	    // hence not a user string.
	    System.out.println("Not yet implemented!!!"); // NOI18N
	    */
	}
    }
    
    protected void populateMenu(Object source, int xpos, int ypos,
				JPopupMenu popup) {
	if (IpeUtils.trace) {
	    System.out.println("In SessionsWindow.PopupAdapter.createPopup"); // NOI18N
	}
	
	// DebuggerTableWindow will use the position to select
	// the current column etc.
	super.populateMenu(source, xpos, ypos, popup);

	gotoSourceItem =
	    new JMenuItem(DbxDebugger.getText("Breakpoints_gotoSource")); // NOI18N
	popup.add(gotoSourceItem);
	gotoSourceItem.addActionListener(this);
	
	popup.addSeparator();
	
	CallableSystemAction tmca =
	    (CallableSystemAction)SystemAction.get(org.netbeans.modules.debugger.support.actions.AddBreakpointAction.class);
	JMenuItem addWatchItem = tmca.getPopupPresenter();
	popup.add(addWatchItem);

	    /* Not yet fully implemented!
	editItem =
	    new JMenuItem(DbxDebugger.getText("Breakpoints_edit")); // NOI18N
	popup.add(editItem);
	editItem.addActionListener(this);
	
	popup.addSeparator();
	    */
	
	popup.addSeparator();
	
	enableItem =
	    new JMenuItem(DbxDebugger.getText("Breakpoints_enable")); // NOI18N
	popup.add(enableItem);
	enableItem.addActionListener(this);
	
	disableItem =
	    new JMenuItem(DbxDebugger.getText("Breakpoints_disable")); // NOI18N
	popup.add(disableItem);
	disableItem.addActionListener(this);
	
	popup.addSeparator();

	CallableSystemAction sa =
	    (CallableSystemAction)SystemAction.get(CutAction.class);
	popup.add(sa.getPopupPresenter());

	sa = (CallableSystemAction)SystemAction.get(CopyAction.class);
	popup.add(sa.getPopupPresenter());

	sa = (CallableSystemAction)SystemAction.get(PasteAction.class);
	popup.add(sa.getPopupPresenter());

	popup.addSeparator();	

	sa = (CallableSystemAction)SystemAction.get(DeleteAction.class);
	popup.add(sa.getPopupPresenter());

	
	deleteAllItem =
	    new JMenuItem(DbxDebugger.getText("Breakpoints_deleteAll")); // NOI18N
	popup.add(deleteAllItem);
	deleteAllItem.addActionListener(this);
	
	DbxDebuggerState state = null;
	Dbx dbx = getDbx();
	if (dbx != null) {
	    state = dbx.getState();
	} else {
	    state = new DbxDebuggerState();
	}

	boolean hasSource = false;
	IpeHandler h = null;
	if (selectedRow != -1) {
	    h = ((BreakptTableModel)model).getHandler(selectedRow);
	    if (IpeUtils.asserts) {
		IpeUtils.ensure(h != null);
	    }
	    ArrayList annotations = h.getAnnotations();
	    if ((annotations != null) && (annotations.size() > 0)) {
		hasSource = true;
	    }
	}

	boolean hasEnabled = false;
	boolean hasDisabled = false;
	int[] selectedRows = table.getSelectedRows();
	int n = selectedRows.length;
	for (int i = 0; i < n; i++) {
	    h = ((BreakptTableModel)model).getHandler(selectedRows[i]);
	    if ((h != null) && (h.getHandler() != null)) {
		if (h.getHandler().enabled) {
		    hasEnabled = true;
		} else {
		    hasDisabled = true;
		}
	    }
	}
	
	if (IpeUtils.asserts) {
	    // Why did this fire?
	    IpeUtils.ensure((h == null) || (selectedHandler == h));
	}
	
	// Always true: addWatchItem.setEnabled(true)

	/* Not yet fully implemented!
	//editItem.setEnabled((selectedRow != -1) && !state.isRunning());
	editItem.setEnabled(false); // not yet implemented
	*/
	
	gotoSourceItem.setEnabled(hasSource);
	enableItem.setEnabled(!state.isRunning() && hasDisabled);
	disableItem.setEnabled(!state.isRunning() && hasEnabled);
	deleteAllItem.setEnabled(model.getRowCount() > 0);

	// Remainder of menu: use system actions
	addSystemActionsToMenu(popup);
    }

    Object readResolve() throws ObjectStreamException {
	BreakpointsWindow w = getSingleton();
	resolvedTo(w);
	return w;
    }
    
    /** Get the debugger view associated with this window */
    public View2 getView() {
	return DbxGuiModule.breakpointsView;
    }

    /** Get the delegating debugger view associated with this window */
    public DelegatingView2 getDelegatingView() {
	return DbxGuiModule.breakpointsDView;
    }
}
