// "$Id: SessionsWindow.java,v 1.64 2002/07/11 00:44:11 gah Exp $ Sun Microsystems"
/*
 * {START_JAVA_COPYRIGHT_NOTICE
 * Copyright 2002 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 * END_COPYRIGHT_NOTICE}
 */


package com.sun.tools.dbxgui.debugger;

import java.awt.*;
import java.awt.event.*;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.PropertyChangeEvent;
import java.io.ObjectStreamException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

import org.openide.*;
import org.openide.actions.*;
import org.openide.awt.*;
import org.openide.util.actions.*;
import org.openide.util.HelpCtx;
import org.openide.util.Utilities;
import org.openide.windows.*;
import org.netbeans.modules.debugger.multisession.*;
import org.netbeans.modules.debugger.support.View2;
import org.netbeans.modules.debugger.support.DelegatingView2;
import org.netbeans.modules.debugger.support.actions.*;

import com.sun.forte.st.glue.dbx.*;

import com.sun.tools.dbxgui.DbxGuiModule;
import com.sun.tools.dbxgui.debugger.actions.*;
import com.sun.tools.dbxgui.debugger.options.DebuggingOption;
import com.sun.tools.dbxgui.utils.*;

/**
 * Window where you can view the list of processes. A context menu
   allows you to select which of the columns should be visible.
   Selection sets the current debug session/
   <p>
   Todo:
   <ul>
    <li> there seems to be a problem with the sorting when I don't
         have a well-defined order (sometimes I say two null objects
	 are equal, but this shouldn't cause loss should it? Why does it?)
	 To try this, set sortingColumn to one where I don't have
	 one defined (so ret always returns 0) and look at what I end up
	 with.
    <li> Proper column handling
    <li> Popup menu should work everywhere
    <li> Use only system actions, not JMenuItems directly
    <li> Select the session, and prevent its unselection
    <li> Display all fields correctly:
    <li> When multiple rows are selected, desensitize session specific
         operations (and threads etc. should go blank)
   </ul>
 */
public final class SessionsWindow extends DebuggerTableWindow
    implements ComponentListener, SessionDataSource, ActionListener {

    static final long serialVersionUID = 8247929904368592896L;

    /** Currently selected thread */
    private DbxDebugSession selectedSession = null;
    
    /** Create a new SessionsWindow tied to the given debugger. */
    public SessionsWindow(DbxDebugger debugger, DataSource source,
			   DbxDebugSession current) {
	super(debugger, source);
	if (singleton == null) {
	    singleton = this;
	}
	createComponents();
	initializeA11y();
	currentSession = current;
	setInView(true);
	addPopupListener(this);
    }

    public SessionsWindow() {
	this(null, null, null);
    }
    

    private void createComponents() {
	// open(); // XXX should it be here instead of below????
	setLayout (new BorderLayout ());

	add(createTable(), java.awt.BorderLayout.CENTER);
	
	setName(DbxDebugger.getText("TITLE_SessionsWindow")); //NOI18N
	setIcon(Utilities.loadImage("org/netbeans/modules/debugger/multisession/resources/sessionView/Sessions.gif")); // NOI18N
    }

    /** Create a table-specific model */
    protected DebuggerTableModel createModel() {
	return new SessionsTableModel();
    }
    
    /** The given item has been selected (if row == -1, nothing is selected) */
    protected void selectedItem(int row, Hyperlink l) {
	if (row != -1) {
	    // SelectedRow is selected
	    selectedSession = (DbxDebugSession)model.getObjectAt(row);

	    // onSetCurrent()
	    
	    if (l != null) {
		System.out.println("Activating link " + l.getUrl());
		l.activate(this);
	    }
	} else {
	    selectedSession = null;
	}
    }

   public void onSetCurrent() {
	if (IpeUtils.trace) {
	    System.out.println("SessionsWindow.onSetCurrent()"); // NOI18N
	}

	if (selectedSession == null) {
	    Toolkit.getDefaultToolkit().beep();
	    return;
	}

	if (selectedSession.isForeign()) {
	    // Foreign session - switch debuggers
	    //System.out.println("Foreign session - switch debuggers");
	    try {
		((EnterpriseDebugger)TopManager.getDefault ().getDebugger ()).
		    setCurrentSession(selectedSession.getCoreSession());
	    } catch (org.openide.debugger.DebuggerNotFoundException e) {
	    }
	} else {
	    debugger.switchTo(selectedSession);
	}
    }
    
    
    /** Enable items that depend on selection */
    protected void enableSelectionItems() {
	/*	
	    SystemAction.get(SuspendAction.class).setEnabled(true);
	    popItem.setEnabled(true);
	    popToCurItem.setEnabled(true);
	*/
    }

    /** Disable items that depend on selection */
    protected void disableSelectionItems() {
	/*
	    SystemAction.get(SuspendAction.class).setEnabled(false);
	    popItem.setEnabled(false);
	    popToCurItem.setEnabled(false);
	*/
    }
    
    /**
     * Show the sessions
     * @param current Which session is currently "active" in this window
     * @param numSession Total number of sessions
     * @param sessions Array of session objects
     */
    public final void show(DbxDebugSession current, int numSessions,
			   DbxDebugSession[] sessions) {

	if (IpeUtils.trace) {
	    System.out.println("SessionsWindow.show[" + //NOI18N
		    Thread.currentThread().getName() + "]: " +  numSessions + //NOI18N
		   ", " + current + ", " + sessions.length + ", " + sessions); // NOI18N
	}
	    
	selectedRow = -1;
	sensitivity();
	DbxDebugSession old = currentSession;
	currentSession = current;
	// Don't notify others of foreign sessions
	if ((old != currentSession) && ((currentSession == null) ||
					!currentSession.isForeign())) {
	    pcs.firePropertyChange(SessionDataSource.PROP_SESSION_CHANGED,
				   old, currentSession); // NOI18N
	}
	((SessionsTableModel)model).setSessions(current, numSessions, sessions);
    }

    /** The component is now showing: tell debugger to send us updates! */
    public void componentShown(ComponentEvent e) {
	if (IpeUtils.trace)
	    System.out.println("SessionsWindow.componentShown!");
	// Nothing to do
	// XXX consider optimizing here such that if the window is not shown,
	// we don't do any work, and when switching to it, we populate the
	// full list. (We should also clear out the window in componentHidden,
	// such that when returning to the tab we don't briefly flash the
	// old contents before the new is generated.

	super.componentShown(e);

    }
    
    /** The component is no longer showing: tell debugger to stop computing
	updates on our behalf! */
    public void componentHidden(ComponentEvent e) {
	if (IpeUtils.trace)
	    System.out.println("SessionsWindow.componentHidden!");
	// Nothing to do

	super.componentHidden(e);
    }

    /** Don't care - but must implement full ComponentListener interface */
    public void componentResized(ComponentEvent e) {
	// Don't care
    }
    
    /** Don't care - but must implement full ComponentListener interface */
    public void componentMoved(ComponentEvent e) {
	// Don't care
    }
    
    /** Table Model optimized for showing stack.
	It knows about a list of columns from which a subset of
	columns can be shown in the table. I'll refer to the
	complete set of columns as the physical columns, and the
	ones shown in the table as the virtual columns.
     */
    class SessionsTableModel extends DebuggerTableWindow.DebuggerTableModel
	implements Comparator {

	/** Row which is current (session) */
	private int currentRow = -1;
	
	/** Sessions data */
	private DbxDebugSession[] sessions = null;

	public SessionsTableModel() {
	    super(10, 4, -1,
		// Column headers
		new String[] {
		  DbxDebugger.getText("Sessions_Table_Name"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_PID"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_Mode"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_State"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_Args"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_Location"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_CoreLoc"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_Memory"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_Access"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_Profiling") // NOI18N
		},
		// Column headers
		new String[] {
		  DbxDebugger.getText("Sessions_Table_NameTip"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_PIDTip"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_ModeTip"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_StateTip"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_ArgsTip"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_LocationTip"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_CoreLocTip"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_MemoryTip"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_AccessTip"), // NOI18N
		  DbxDebugger.getText("Sessions_Table_ProfilingTip") // NOI18N
		},
		// Which columns are visible? (user selectable)
		new boolean[] {
		    true, true, false, true, true, false, false, false,
		    false, false
		},
		// Visible column mapping. Only defined up to index
		// numVisible-1.
		new int[] {
		    0, 4, 1, 3, -1, -1, -1, -1, -1, -1, -1
		},
		// Preferred Width
		new int[] {
		    100, 50, 75, 75, 150, 75, 75, 75, 75, 75
		},
                new char[] {
		  DbxDebugger.getText("Sessions_Table_NameMnemonic").charAt(0),     // NOI18N
		  DbxDebugger.getText("Sessions_Table_PIDMnemonic").charAt(0),      // NOI18N
		  DbxDebugger.getText("Sessions_Table_ModeMnemonic").charAt(0),     // NOI18N
		  DbxDebugger.getText("Sessions_Table_StateMnemonic").charAt(0),    // NOI18N
		  DbxDebugger.getText("Sessions_Table_ArgsMnemonic").charAt(0),     // NOI18N
		  DbxDebugger.getText("Sessions_Table_LocationMnemonic").charAt(0), // NOI18N
		  DbxDebugger.getText("Sessions_Table_CoreLocMnemonic").charAt(0),  // NOI18N
		  DbxDebugger.getText("Sessions_Table_MemoryMnemonic").charAt(0),   // NOI18N
		  DbxDebugger.getText("Sessions_Table_AccessMnemonic").charAt(0),   // NOI18N
		  DbxDebugger.getText("Sessions_Table_ProfilingMnemonic").charAt(0) // NOI18N
                });
	}
	
	/** Return the value of a particular cell in the table */
        public Object getRealValueAt(int row, int realColumn) {
	    if ((sessions == null) || (numRows == 0)) {
		return "";
	    }
	    DbxDebugSession s = sessions[row];
	    if (s == null) {
		return "";
	    }
	    RunConfig c = s.getConfig();
	    switch (realColumn) {
		// Name
	    case 0: {
		return s.getName();
		/*
		if ((c.getProgram() != null)
		    && (c.getProgram().getExecutableBaseName()
			!= null)) {
		    return c.getProgram().getExecutableBaseName();
		} else {
		    return "";
		}
		*/
	    }
	    // PID
	    case 1: {
		if (s.getPidObject().longValue() == -1) {
		    return ""; // XXX this is created only ONCE right???
		} else {
		    return s.getPidObject(); // XXX convert to space if necessary
		}
	    }
	    // Mode
	    case 2: {
		if (c == null) {
		    return "";
		}
		if (s.getJavaMode() == DbxJN_mode.DBX_JAVA) {
		    return DbxDebugger.getText("Sessions_Table_ModeJava"); // NOI18N
		} else {
		    // Plain, or native mode java
		    return DbxDebugger.getText("Sessions_Table_ModeNative"); // NOI18N
		}
	    }
	    // State
	    case 3: {
		if (s.getEngine() != null) {
		    return s.getEngine().getStateDesc();
		} else {
		    return new ImageLabel(DbxDebugger.getText(
			      "Sessions_Table_StateNone"), null); // NOI18N
		}
	    }
	    // Arguments
	    case 4: {
		if ((c == null) || (c.getArgsFlattened() == null)) {
		    return "";
		} else {
		    return c.getArgsFlattened();
		}
	    }
	    
	    // Location
	    case 5: {
		if (s.isForeign()) {
		    return s.getCoreSession().getLocationName();
		} else if ((c.getProgram() != null) &&
			   (c.getProgram().getExecutableName() != null)) {
		    return c.getProgram().getExecutableName();
		} else {
		    return "";
		}
	    }
	    // Core Location
	    case 6: {
		if (s.getCorefile() != null) {
		    return s.getCorefile();
		} else {
		    return "";
		}
	    }
	    // Memory Use
	    case 7: {
		if (s.isMemuseMode()) {
		    return DbxDebugger.getText("On"); // NOI18N
		} else {
		    return DbxDebugger.getText("Off"); // NOI18N
		}
	    }
	    // Memory Access
	    case 8: {
		if (s.isAccessMode()) {
		    return DbxDebugger.getText("On"); // NOI18N
		} else {
		    return DbxDebugger.getText("Off"); // NOI18N
		}
	    }
	    // Profiling
	    case 9: {
		if (s.isProfilingMode()) {
		    return DbxDebugger.getText("On"); // NOI18N
		} else {
		    return DbxDebugger.getText("Off"); // NOI18N
		}
	    }
	    default:
		// Error
		if (IpeUtils.asserts)
		    IpeUtils.ensure(false);
		return "";
	    }
        }

	/** Return the value of a particular cell in the table */
	public void setRealValueAt(Object value, int row, int realColumn) {
	    if ((sessions == null) || (numRows == 0)) {
		return;
	    }
	    DbxDebugSession s = sessions[row];
	    if (s == null) {
		return;
	    }
	    RunConfig c = s.getConfig();
	    switch (realColumn) {
		/*
		// XXX Name - perhaps you should get to change this...
	    case 0: {
		break s.getName();
	    }
		*/
	    // Mode
	    case 2: {
		if (value.toString().equals(DbxDebugger.getText(
					"Sessions_Table_ModeJava"))) { // NOI18N
		    // Switch to java mode
		    if (s.getEngine() != null) {
			Dbx dbx = (Dbx)(s.getEngine());
			dbx.setJavaMode(DbxJN_mode.DBX_JAVA);
		    }
		} else {
		    if (IpeUtils.asserts) {
			IpeUtils.ensure(value.toString().equals(
				DbxDebugger.getText(
				    "Sessions_Table_ModeNative"))); // NOI18N
		    }
		    if (s.getEngine() != null) {
			Dbx dbx = (Dbx)(s.getEngine());
			dbx.setJavaMode(DbxJN_mode.DBX_JAVANATIVE);
		    }
		}
		break;
	    }
		// yes, I should let you change it!
	    // Arguments
	    case 4: {
		if (c != null) {
		    c.setUnparsedArgs(value.toString());

		    // See if we can't get this stuff updated in the engine
		    // as well right away...
		    if ((s != null) && (s.getEngine() != null)) {
			Dbx dbx = (Dbx)s.getEngine();
			dbx.applyEnvironment();
		    }
		}
		break;
	    }

	    // Memory Use
	    case 7: {
		if (s.getEngine() != null) {
		    Dbx dbx = (Dbx)(s.getEngine());
		    if (value.toString().equals(
				DbxDebugger.getText("On"))) { // NOI18N
			// Enable RTC
			if (!dbx.isMemuseEnabled()) {
			    dbx.setMemuseChecking(true);
			}
		    } else {
			// Disable RTC
			if (dbx.isMemuseEnabled()) {
			    dbx.setMemuseChecking(false);
			}
		    }
		}
		break;
	    }
	    // Memory Access
	    case 8: {
		if (s.getEngine() != null) {
		    Dbx dbx = (Dbx)(s.getEngine());
		    if (value.toString().equals(
				DbxDebugger.getText("On"))) { // NOI18N
			// Enable RTC
			if (!dbx.isAccessCheckingEnabled()) {
			    dbx.setAccessChecking(true);
			}
		    } else {
			// Disable RTC
			if (dbx.isAccessCheckingEnabled()) {
			    dbx.setAccessChecking(false);
			}
		    }
		}
		break;
	    }
		/*
	    // Profiling
	    case 9: {
		if (s.isProfilingMode()) {
		    break DbxDebugger.getText("On"); // NOI18N
		} else {
		    break DbxDebugger.getText("Off"); // NOI18N
		}
	    }
		*/
	    default:
		if (IpeUtils.asserts) {
		    // Error: assignment not allowed here!
		    TopManager.getDefault().getErrorManager().log("SessionsWindow.setRealValueAt(" + row + "," + realColumn + ") is " + value + " : UNKNOWN COLUMN!"); // NOI18N
		}
		break;
	    }

	    // Redraw
	    updatedRealCell(row, realColumn);
        }
	

	/*
	 * Don't need to implement this method unless your table's
	 * editable.
	 */
	public boolean isRealCellEditable(int row, int rcol) {
	    // XXX I can use the row number here to prevent you from
	    // trying to edit a session which is purely native
	    return ((rcol == 2) || (rcol == 4) || (rcol == 7) || (rcol == 8));
	}
	
	// Extra methods (these are additional SessionsModel methods,
	// not AbstractTableModel methods)

	/** Return a cell renderer to be used for this column. */
	protected DefaultTableCellRenderer getRealCellRenderer(int rCol) {
	    if ((rCol == 0) || (rCol == 3)) {
		// Name: want my own cell renderer capable of
		// showing both images and text
		return new ImageLabelCellRenderer();
	    } else {
		// Use default - e.g. DefaultTableCellRenderer
		return null;
	    }
	}

	/** Return the cell editor to be used for this column.
	    If null, no editing.
	*/
	protected TableCellEditor getRealCellEditor(int rCol) {
	    switch(rCol) {
	    case 2: {
		JComboBox comboBox = new JComboBox();
		comboBox.addItem(DbxDebugger.getText(
				     "Sessions_Table_ModeJava")); // NOI18N
		comboBox.addItem(DbxDebugger.getText(
				     "Sessions_Table_ModeNative")); // NOI18N
		comboBox.setEditable(false);
		return new DefaultCellEditor(comboBox);
	    }

	    case 7:
	    case 8: {
		JComboBox comboBox = new JComboBox();
		comboBox.addItem(DbxDebugger.getText("On")); // NOI18N
		comboBox.addItem(DbxDebugger.getText("Off")); // NOI18N
		comboBox.setEditable(false);
		return new DefaultCellEditor(comboBox);
	    }
	    default: return null;
	    }
	}
	
	/** Compare function used to sort the stack sessions */
	public int compare(Object o1, Object o2) {
	    DbxDebugSession s1 = (DbxDebugSession)o1;
	    DbxDebugSession s2 = (DbxDebugSession)o2;

	    if (s1 == null) {
		System.out.println("s1 is null!");
		return -1;
	    }
	    if (s2 == null) {
		System.out.println("s2 is null!");
		return -1;
	    }

	    RunConfig c1 = s1.getConfig();
	    RunConfig c2 = s2.getConfig();
	    
	    int ret = 0;
	    switch(sortColumn) {
		// XXX need to provide frameno sorting
	    case 0: {
		String n1 = null;
		String n2 = null;
		if (s1.isForeign()) {
		    n1 = s1.getCoreSession().getSessionName();
		} else if (c1.getProgram() != null &&
		    c1.getProgram().getExecutableBaseName() != null) {
		    n1 = c1.getProgram().getExecutableBaseName();
		}
		if (s2.isForeign()) {
		    n2 = s2.getCoreSession().getSessionName();
		} else if (c2.getProgram() != null &&
		    c2.getProgram().getExecutableBaseName() != null) {
		    n2 = c2.getProgram().getExecutableBaseName();
		}
		if ((s1 != null) && (s2 != null)) {
		    ret = n1.compareTo(n2);
		} else if ((s1 == null) && (s2 == null)) {
		    // XXX does this cause data loss??
		    ret = 0;
		} else if (s1 != null) {
		    ret = 1;
		} else {
		    ret = -1;
		}
		break;
	    }

		// pid
	    case 1: {
		ret = (int)(s1.getPid()-s2.getPid());
		break; // only care about sign
	    }

	    // Mode
	    case 2: {
		ret = s1.getJavaMode() - s2.getJavaMode();
		break;
	    }

		// State
	    case 3: {
		String st1;
		String st2;
		if (s1.getEngine() != null) {
		    st1 = s1.getEngine().getStateDesc().getLabel();
		} else {
		    st1 = DbxDebugger.getText("Sessions_Table_StateNone"); // NOI18N
		}
		if (s2.getEngine() != null) {
		    st2 = s2.getEngine().getStateDesc().getLabel();
		} else {
		    st2 = DbxDebugger.getText("Sessions_Table_StateNone"); // NOI18N
		}
		ret = st1.compareTo(st2);
		break;
	    }

		// Arguments
	    case 4: {
		String a1;
		String a2;
		
		if ((c1 == null) || (c1.getArgsFlattened() == null)) {
		    a1 = "";
		} else {
		    a1 = c1.getArgsFlattened();
		}
		if ((c2 == null) || (c2.getArgsFlattened() == null)) {
		    a2 = "";
		} else {
		    a2 = c2.getArgsFlattened();
		}
		ret = a1.compareTo(a2);
		break;
	    }

	    // Location
	    case 5: {
		String l1;
		String l2;
		if (s1.isForeign()) {
		    l1 = s1.getCoreSession().getLocationName();
		} else if ((c1.getProgram() != null) &&
			   (c1.getProgram().getExecutableName() != null)) {
		    l1 = c1.getProgram().getExecutableName();
		} else {
		    l1 = "";
		}
		if (s2.isForeign()) {
		    l2 = s2.getCoreSession().getLocationName();
		} else if ((c2.getProgram() != null) &&
			   (c2.getProgram().getExecutableName() != null)) {
		    l2 = c2.getProgram().getExecutableName();
		} else {
		    l2 = "";
		}
		ret = l1.compareTo(l2);
		break;
	    }
		
	    // Core Location
	    case 6: {
		String co1;
		String co2;
		
		if (s1.getCorefile() != null) {
		    co1 = s1.getCorefile();
		} else {
		    co1 = "";
		}
		if (s2.getCorefile() != null) {
		    co2 = s2.getCorefile();
		} else {
		    co2 = "";
		}
		ret = co1.compareTo(co2);
		break;
	    }
		
	    // Memory Use
	    case 7: {
		ret = IpeUtils.boolCompare(s1.isMemuseMode(),
					   s2.isMemuseMode());
		break;
	    }
	    // Memory Access
	    case 8: {
		ret = IpeUtils.boolCompare(s1.isAccessMode(),
					   s2.isAccessMode());
		break;
	    }
		
	    // Profiling
	    case 9: {
		ret = IpeUtils.boolCompare(s1.isProfilingMode(),
					   s2.isProfilingMode());
		break;
	    }
		
	    default:
		ret = 0;
		if (IpeUtils.asserts) {
		    IpeUtils.ensure(false);
		}
		break; // Not valid: don't sort
	    }

	    // Sort descending rather than ascending order?
	    if (sortDescending) {
		ret = -ret;
	    }
	    return ret;
	}

	/*
	private void printSessions() {
	    for (int i = 0; i < numRows; i++) {
		DbxDebugSession s = sessions[i];
		System.out.print(i + ") ");
		if (s == null) {
		    System.out.println("<null>");
		}
		System.out.println(s.getConfig().getProgram().getExecutableBaseName());
	    }
	}
	*/
	
	/** Resort the table and update */
	public void sort() {
	    if ((numRows > 1) && (sortColumn != -1)) {
		sort(sessions);
	    }

	    // XXX Later we might be able to do a bit better job of
	    // determining how much changed, and perhaps indicate
	    // that say only a single row changed (that might improve
	    // redraw performance and minimize flicker)
	    // Update GUI
	    fireTableDataChanged();

	    selectCurrentSession();
	}

	public DbxDebugSession getSession(int index) {
	    if (sessions == null) {
		return null;
	    }
	    return sessions[index];
	}
	
	public void setSessions(DbxDebugSession selSession,
				int newNumSessions,
				DbxDebugSession[] newSessions) {

	    if ((numRows == 1) && (newNumSessions > 1)) {
		// On transition from single-session to multisession
		// debugging, expose the sessions window
		if (DebuggingOption.OPEN_SESSIONS.isEnabled()) {
		    debugger.makeWindowVisible(SessionsWindow.this);
		}
	    }
		
	    // Sort threads according to sort criterion
	    // sort the newThreads array since we want to do a diff
	    // against the threads array afterwards

	    // XXX find out if dbx (or the OS) already sorts the threads
	    // by thread id, and if so, optimize out sorting by column 1
	    if ((newNumSessions > 1) && (sortColumn != -1)) {
		sort(newSessions);
	    }

	    sessions = newSessions;
	    currentSession = selSession;
	    numRows = newNumSessions;

	    // XXX Later we might be able to do a bit better job of
	    // determining how much changed, and perhaps indicate
	    // that say only a single row changed (that might improve
	    // redraw performance and minimize flicker)
	    // Update GUI
	    fireTableDataChanged();

	    selectCurrentSession();
	}

	private void selectCurrentSession() {
	    // Select the current session
	    if (sessions != null) {
		for (int i = 0; i < numRows; i++) {
		    if (sessions[i] == currentSession) {
			currentRow = i;
			// Found the current session: mark it as selected
			ignoreSelection = true;
			table.changeSelection(i, 0, false, false);
			selectedSession = sessions[i];
			ignoreSelection = false;
			fireTableRowsUpdated(i, i);
			break;
		    }
		}
	    }
	}

	/** Show the given session as the currently selected one */
	public void selectSession(DbxDebugSession s) {
	    currentSession = s;
	    selectCurrentSession();
	}

	/** The given session has been updated: reflect this in the GUI */
	public void updateSession(DbxDebugSession s) {
	    // Select the current session
	    if (sessions != null) {
		for (int i = 0; i < numRows; i++) {
		    if (sessions[i] == s) {
			fireTableRowsUpdated(i, i);
			// if we messed with the current item, selection
			// will disappear
			if (s == currentSession) {
			    selectCurrentSession();
			}
			return;
		    }
		}
		/* XXX
		if (IpeUtils.asserts) {  // make sure we found the session
		    IpeUtils.ensure(false);
		}
		*/
	    }
	}

	/** Return the stack frame on a particular row */
	public Object getObjectAt(int row) {
	    return sessions[row];
	}
    }

    /** Overrides superclass method. Gets actions for this top component. */
    public SystemAction[] getSystemActions() {
	//SystemAction[] sa = super.getSystemActions ();

	DebuggerTableColumnAction dtca = (DebuggerTableColumnAction)
	    SystemAction.get(DebuggerTableColumnAction.class);
        dtca.setTable(this);

	DebugWinDockAction dock = (DebugWinDockAction)
	SystemAction.get(DebugWinDockAction.class);
        dock.setDebuggerWindow(this);
	// XXX alternatively I could set it to DockIntoAction
	// if top component is in view!
		
	SystemAction[] sessActions = new SystemAction[] {
	    null,
	    dtca,
	    null,
	    dock
	};
	//return SystemAction.linkActions (sa, sessActions);
	return sessActions;
    }

    /** Currently selected session in this window */
    private DbxDebugSession currentSession = null;


    // Implement SessionDataSource interface:

    public DbxDebugSession getSession() {
	return currentSession;
    }

    private transient PropertyChangeSupport pcs = new PropertyChangeSupport(this);
    
    /**
    * Adds property change listener.
    *
    * @param l new listener.
    */
    public void addPropertyChangeListener (PropertyChangeListener l) {
	//System.out.println("SessionsWindow.addPropertyChangeListener: " + l.getClass().getName());
        pcs.addPropertyChangeListener (l);
	// I should also add the listener to listen on the thread itself
	// such that the thread can notify it of changes in data without
	// our involvement
	if (currentSession != null) {
	    currentSession.addPropertyChangeListener(l);
	}
    }

    /**
    * Removes property change listener.
    *
    * @param l removed listener.
    */
    public void removePropertyChangeListener (PropertyChangeListener l) {
	//System.out.println("SessionsWindow.removePropertyChangeListener: " + l.getClass().getName());
        pcs.removePropertyChangeListener (l);
	// Remove this window from listening on the thread directly
	if (currentSession != null) {
	    currentSession.removePropertyChangeListener(l);
	}
    }

    /** Return a user-readible short name of the session.
	XXX Probably (eventually) user modifiable.
	Used to list session in popup menus etc. so it needs
	to be short but precise and distinguishable from other
	similar sessions. */
    public String getShortName() {
	return MessageFormat.format(DbxDebugger.getText(
		     "CurrSelWin"), // NOI18N
		 new String[] { getWindowSetTag() });
    }

    /** Show the given session as the currently selected one */
    public void selectSession(DbxDebugSession s) {
	((SessionsTableModel)model).selectSession(s);
    }
    
    // Implements PropertyChangeListener
    public void propertyChange(PropertyChangeEvent evt) {
	//System.out.println("SessionsWindow.PropertyChange: " + evt.getPropertyName());
	
	// Note - these strings have all been interned (they are all literals
	// so no need for full .equals check)
	if (SessionListDataSource.PROP_SESSIONS_CHANGED == evt.getPropertyName()) {
	    DbxDebugSession newCurrSession =
		(DbxDebugSession)evt.getOldValue();
	    DbxDebugSession [] sessArray = (DbxDebugSession [])evt.getNewValue();
	    show(newCurrSession,
		 (sessArray != null) ? sessArray.length : 0,
		 sessArray);

	    // Notify listeners (e.g. windows wired to this one)
	    // that session changed such that they can refetch contents
	    if (currentSession != newCurrSession) {
		pcs.firePropertyChange(SessionDataSource.PROP_SESSION_CHANGED,
				       currentSession,
				       newCurrSession);
	    }
	} else if (SessionListDataSource.PROP_CURRSESSION_CHANGED == evt.getPropertyName()) {
	    DbxDebugSession oldSession = (DbxDebugSession)evt.getOldValue();
	    DbxDebugSession newSession = (DbxDebugSession)evt.getNewValue();
	    if (IpeUtils.asserts) {
		IpeUtils.ensure(oldSession == currentSession);
	    }

	    if (currentSession != newSession) {
		// XXX For now, the editor is tied to this session selection
		// as well
		if ((newSession != null) && (newSession.getEngine() != null)) {
		    Dbx dbx = (Dbx)newSession.getEngine();
		    DbxLocation loc = dbx.getCurrentLocation();
		    DbxLocation vloc = dbx.getVisitedLocation();
		    DbxLocation sloc = dbx.getStartLocation();
		    if ((loc != null) && (loc.src != null)) {
			debugger.makeCurrent(dbx, loc.src, loc.line,
				     loc.func,
				     ((loc.flags & DbxLocation.NO_SRC) == 0) &&
				     (loc.src != null),
				     true, false, false);
		    } else if ((vloc != null) && (vloc.src != null)) {
			debugger.makeCurrent(dbx, vloc.src, vloc.line,
				     vloc.func,
				     ((vloc.flags & DbxLocation.NO_SRC) == 0) &&
				     (vloc.src != null),
				     true, false, true);
		    } else if ((sloc != null) && (sloc.src != null)) {
			debugger.makeCurrent(dbx, sloc.src, sloc.line,
				     sloc.func,
				     ((sloc.flags & DbxLocation.NO_SRC) == 0) &&
				     (sloc.src != null),
				     false, false, false);
		    } else {
			debugger.setCurrentLine(null);
		    }

		    pcs.firePropertyChange(
				   SessionDataSource.PROP_SESSION_CHANGED,
				   currentSession, newSession); // NOI18N
		}

		currentSession = newSession;
	    }
	    ((SessionsTableModel)model).selectSession(currentSession);

	} else if (SessionListDataSource.PROP_SESSIONSTATE_CHANGED == evt.getPropertyName()) {

	    DbxDebugSession session = (DbxDebugSession)evt.getNewValue();
	    if (IpeUtils.asserts) {
		IpeUtils.ensure(session != null);
	    }

	    ((SessionsTableModel)model).updateSession(session);
	    /*
        // EVENT FROM ENTERPRISE DEBUGGER:
	} else if (EnterpriseDebugger.PROP_CURRENT_SESSION.equals(
				       evt.getPropertyName())) { // NOI18N
 	    // Current session changed...
	    Session session = (Session)evt.getNewValue();
	    if (session != null) {
		System.out.println("Switched to session  " +
				   session.getSessionName());
	    } else {
		System.out.println("Switched session: current is now null.");
	    }
	    // Compare to info objects? Or perhaps just lobby for access to
	    // the debuggerinfo object? Or is there some way at session
	    // creation to set up an association
	    */
	} else if (SessionListDataSource.PROP_SESSION_QUIT == evt.getPropertyName()) {
	    pcs.firePropertyChange(SessionDataSource.PROP_SESSION_QUIT,
				   evt.getOldValue(), evt.getNewValue());
	} else {
	    super.propertyChange(evt);
	}
    }

    private JMenuItem contItem = null;
    private JMenuItem terminateItem = null;
    private JMenuItem finishItem = null;
    private JMenuItem restartItem = null;
    private JMenuItem pauseItem = null;
    private JMenuItem detachItem = null;
    private JMenuItem configItem = null;

   public void actionPerformed(java.awt.event.ActionEvent actionEvent) {
       if (selectedRow == -1) {
	   return;
       }
      
       int[] selectedRows = table.getSelectedRows();
       int n = selectedRows.length;
       for (int i = 0; i < n; i++) {
	   DbxDebugSession s = ((SessionsTableModel)model).getSession(
								      selectedRows[i]);
	   if (IpeUtils.asserts) {
	       IpeUtils.ensure(s != null);
	   }
	   if (s.getEngine() == null) {
	       continue;
	   }
	   Dbx dbx = (Dbx)s.getEngine();
	   if (actionEvent.getSource() == contItem) {
	       dbx.go();
	   } else if (actionEvent.getSource() == terminateItem) {
	       dbx.terminate();
	   } else if (actionEvent.getSource() == finishItem) {
	       dbx.quit();
	   } else if (actionEvent.getSource() == restartItem) {
	       dbx.restart();
	   } else if (actionEvent.getSource() == pauseItem) {
	       dbx.interrupt();
	   } else if (actionEvent.getSource() == detachItem) {
	       dbx.detach();
	   } else if (actionEvent.getSource() == configItem) {
	       RunConfigDialog.showWindow(s.getConfig());
	   }
       }
    }

    protected void populateMenu(Object source, int xpos, int ypos,
				JPopupMenu popup) {
	if (IpeUtils.trace) {
	    System.out.println("In SessionsWindow.PopupAdapter.createPopup"); // NOI18N
	}

	// DebuggerTableWindow will use the position to select
	// the current column etc.
	super.populateMenu(source, xpos, ypos, popup);

	CallableSystemAction tmca;
	tmca = (CallableSystemAction)SystemAction.get(SetCurrentAction.class);
	JMenuItem setCurrentItem = tmca.getPopupPresenter();
	popup.add(setCurrentItem);

	popup.addSeparator();
	
	pauseItem =
	    new JMenuItem(DbxDebugger.getText("Session_pause")); // NOI18N
	popup.add(pauseItem);
	pauseItem.addActionListener(this);

	contItem =
	    new JMenuItem(DbxDebugger.getText("Session_cont")); // NOI18N
	popup.add(contItem);
	contItem.addActionListener(this);

	popup.addSeparator();
	
	tmca = (CallableSystemAction)SystemAction.get(EditRunConfigAction.class);
	/* Don't do this; we need to override the action's selection
	   of which configuration to use (it defaults to the activated
	   node or current session, but you might have right-clicked on
	   the non-current-session
	JMenuItem jmi = tmca.getPopupPresenter();
	//jmi.setIcon(tmca.getIcon());
	popup.add(jmi);
	*/
	// XXX should we be using Actions.cutAmpersand() instead???
	configItem = new JMenuItem();
	//curMenuItem.setIcon(tmca.getIcon());
	String label = tmca.getName();
	Actions.setMenuText(configItem, label, true);
	configItem.addActionListener(this);
	popup.add(configItem);

	popup.addSeparator();
	
	restartItem =
	    new JMenuItem(DbxDebugger.getText("Session_restart")); // NOI18N
	popup.add(restartItem);
	restartItem.addActionListener(this);

	detachItem =
	    new JMenuItem(DbxDebugger.getText("Session_detach")); // NOI18N
	popup.add(detachItem);
	detachItem.addActionListener(this);

	terminateItem =
	    new JMenuItem(DbxDebugger.getText("Session_terminate")); // NOI18N
	popup.add(terminateItem);
	terminateItem.addActionListener(this);

	finishItem =
	    new JMenuItem(DbxDebugger.getText("Session_finish")); // NOI18N
	popup.add(finishItem);
	finishItem.addActionListener(this);

	int [] selectedRows = table.getSelectedRows();
	int n = selectedRows.length;
	boolean def = false;
	if (n > 0) {
		def = true;
	}
	boolean canContinue = def;
	boolean canTerminate = def;
	boolean canDetach = def;
	boolean canRestart = def;
	boolean canPause = def;
	boolean canSetCurrent = (n == 1);
	boolean canFinish = (n == 1);
	boolean canConfigure = (n == 1);
	
	for (int i = 0; i < n; i++) {
	    DbxDebugSession s =
		((SessionsTableModel)model).getSession(selectedRows[i]);

	    // Foreign objects can't do most of these, all we can do is switch
	    // to them (actually, I could probably implement some of these)
	    if (s.isForeign()) {
		canContinue = false;
		canTerminate = false;
		canDetach = false;
		canPause = false;
		canFinish = false;
		canConfigure = false;
		canRestart = false;
	    }
	    
	    if ((s != null) && (s.getEngine() != null)) {
		DbxDebuggerState st = s.getEngine().getState();
		if (!(!st.isRunning() && !st.isCore() && st.isLoaded())) {
			canContinue = false;
		}
	        if (!st.isProcess()) {
			canTerminate = false;
			canDetach = false;
		}
		if (!st.isLoaded()) {
			canRestart = false;
		}
		if (!st.isRunning()) {
			canPause = false;
		}
	    }
	}

	contItem.setEnabled(canContinue);
	terminateItem.setEnabled(canTerminate);
	detachItem.setEnabled(canDetach);
	finishItem.setEnabled(canFinish);
	restartItem.setEnabled(canRestart);
	pauseItem.setEnabled(canPause);
	setCurrentItem.setEnabled(canSetCurrent);
	configItem.setEnabled(canConfigure);
	
	// Remainder of menu: use system actions
	addSystemActionsToMenu(popup);
    }
    
    public HelpCtx getHelpCtx() {
        return new HelpCtx("Debugging_sessions"); // NOI18N
    }

    private static SessionsWindow singleton = null;
    public static SessionsWindow getSingleton() {
	return singleton;
    }
    
    Object readResolve() throws ObjectStreamException {
	//System.out.println("SessionsWindow: Resolving " + getName());
	SessionsWindow w = getSingleton();
	resolvedTo(w);
	return w;
    }
    
    /** Get the debugger view associated with this window */
    public View2 getView() {
	return DbxGuiModule.sessionsView;
    }

    /** Get the delegating debugger view associated with this window */
    public DelegatingView2 getDelegatingView() {
	return DbxGuiModule.sessionsDView;
    }
}
