// "@(#)DataBrowserWindow.java	1.28 01/11/02 Sun Microsystems"
/*
 * {START_JAVA_COPYRIGHT_NOTICE
 * Copyright 2002 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 * END_COPYRIGHT_NOTICE}
 */


package com.sun.tools.dbxgui.debugger;

import com.sun.forte.st.glue.dbx.DbxDisplayItem;
import com.sun.forte.st.glue.dbx.DbxHEvalResult;
import com.sun.tools.dbxgui.debugger.actions.*;
import com.sun.tools.dbxgui.debugger.expression.*;
import com.sun.tools.dbxgui.DbxGuiModule;
import com.sun.tools.dbxgui.utils.*;
import java.awt.*;
import java.awt.event.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.*;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.JTableHeader;
import javax.swing.tree.DefaultMutableTreeNode;
import org.openide.*;
import org.openide.actions.CopyAction;
import org.openide.actions.CutAction;
import org.openide.actions.DeleteAction;
import org.openide.actions.PasteAction;
import org.openide.actions.FileSystemAction;
import org.openide.awt.JPopupMenuPlus;
import org.openide.awt.MouseUtils;
import org.openide.util.actions.SystemAction;
import org.openide.util.actions.CallableSystemAction;
import org.openide.util.actions.SystemAction;
import org.openide.util.HelpCtx;
import org.netbeans.modules.debugger.support.View2;
import org.netbeans.modules.debugger.support.DelegatingView2;


/**
 * Panel which shows the contents of the evaluate window
 * (Created and maintained with the NetBeans form editor)
 */
public class DataBrowserWindow
    extends DebuggerWindow
    implements ActionListener,
               ComponentListener,
               RoutingPropertyChangeListener,
               TreeSelectionListener {
    
    private static final boolean trace = System.getProperty("trace.DataBrowser") != null;//NOI18N
    
    static final long serialVersionUID = 6804460262021184535L;
    
    private static javax.swing.JButton colsButton = null;
            
    private static CopyAction copy = null;
    private static CutAction cut = null;
    private static DeleteAction delete = null;
    private static PasteAction paste = null;
    
    /** The current serialization version if data read/written externally.*/
    private static final int SERIALIZATION_VERSION = 1;

    /** Label for the expression combo box. */
    JLabel expressionLabel = null;

    /** Combo box for stack of expressions.*/
    private transient javax.swing.JComboBox expressionStackCombo = null;
    
    /** Expression combo box model */
    private transient DbxComboBoxModel expressionStackModel;
    
    private static ImageIcon ic = null;
    
    /** List of all open expression items.*/
    private final transient ArrayList items = new ArrayList(20);
    
    private transient ExpressionTreeTableModel model = null;
    
    /** A panel that provides input of new expressions.*/
    private transient JPanel newExprPanel = null;
    
    /** Cached model for the null session.*/
    private transient ExpressionTreeTableModel nullSessionModel = null;
    
    /** Number of expressions in the scroll pane.*/
    private transient int numberOfExpressions = 0;
    
    /** Context menu item to remove an item.*/
    private JMenuItem removeMenuItem = null; // NOI18N
    
    /** Context menu item to remove all items.*/
    private JMenuItem removeAllMenuItem = null; // NOI18N
    
    /** Unique routine token for this model.*/
    private final RoutingToken routingToken = RoutingToken.TREETABLE.getUniqueRoutingToken();
    
    /** Item which the user right-clicked on.*/
    private transient ListItem selectedItem = null;
    
    /** Current session. */
    private transient DbxDebugSession session = null;
    
    /** Map of session/model tuples. */
    private transient Hashtable sessionMap = new Hashtable();
    
    /** Number of times we've called componentShown without componentHidden */
    private transient int shown = 0;

    private static DataBrowserWindow singleton = null;
    
    /** The TreeTable widget. */
    private transient TreeTable treeTable = null;
    
    private final static String updateNowMenuItemString = DbxDebugger.getText("LBL_UpdateNow"); //NOI18N
    
    private JCheckBoxMenuItem updateAtStopCommand = null; //NOI18N
    
    private JMenuItem updateNowCommand = null; //NOI18N
    
    
    /** Evaluate window */
    public DataBrowserWindow(DbxDebugger debugger, DataSource source) {
        super(debugger, source);
        if (singleton == null) {
            singleton = this;
            ic = new ImageIcon(DbxStackFrame.class.getResource(
                "/com/sun/tools/dbxgui/icons/cols.gif"));//NOI18N
            colsButton = new javax.swing.JButton(ic);
            colsButton.addActionListener(this);
        }
        
        initComponents();
        // Fill in the label/text field to accept new expressions.
        expressionStackModel = new DbxComboBoxModel();

        expressionStackCombo = new javax.swing.JComboBox();
        expressionStackCombo.setModel(expressionStackModel);
        expressionStackCombo.setEditable(true);
        expressionStackCombo.setActionCommand(""); //NOI18N
        expressionStackCombo.addActionListener(this);        
        expressionStackCombo.requestDefaultFocus();

        expressionLabel = new JLabel(DbxDebugger.getText("LBL_Expression")); //NOI18N

        // Create a panel to hold the label/expression value for new expressions.
        newExprPanel = new JPanel();
        newExprPanel.setLayout(new BorderLayout());
        newExprPanel.setOpaque(true);
        newExprPanel.add(expressionLabel, BorderLayout.WEST);
        newExprPanel.add(expressionStackCombo, BorderLayout.CENTER);
        
        copy = (CopyAction) CopyAction.findObject(CopyAction.class, true);
        cut = (CutAction) CutAction.findObject(CutAction.class, true);
        delete = (DeleteAction) DeleteAction.findObject(DeleteAction.class, true);
        paste = (PasteAction) PasteAction.findObject(PasteAction.class, true);
        
        additionalInitComponents(null);
        initializeA11y();
        setInView(true);
    }
    
    
    /** Null constructor, used when creating window from configuration
     * files etc. */
    public DataBrowserWindow() {
        this(null, null);
    }
    
    
    // For RoutingPropertyChangeListener interface.
    public boolean acceptsExternalWatches() {
        return false;
    }
    
    
    public void actionPerformed(java.awt.event.ActionEvent actionEvent) {
        if (trace) {
            System.err.println("@@@ DataBrowser.actionPerformed source: "+actionEvent.getSource());//NOI18N
        }
        if (actionEvent.getSource() == updateAtStopCommand) {
            if (UsageTracking.enabled)
                UsageTracking.sendAction("DataBrowser: udpateAtStop", null); // NOI18N
            
            // Note!!! For some reason, this even apparently can get fired twice
            // on one mouse click, with the same sate for the button.
            if (!updateAtStopCommand.getState()) {
                if (selectedItem.updateOnStop) {
                    // Was true, make false.
                    selectedItem.updateOnStop = false;
                    updateOnStop(false);
                }
            } else {
                if (!selectedItem.updateOnStop) {
                    // Was false, make true.
                    selectedItem.updateOnStop = true;
                    updateOnStop(true);
                }
            }
        } else if (actionEvent.getSource() == updateNowCommand) { // NOI18N
            if (UsageTracking.enabled)
                UsageTracking.sendAction("DataBrowser: updateNow", null); // NOI18N
            
            DbxDebugSession session = ((SessionDataSource) source).getSession();
            if (session != null) {
                DbxDebuggerEngine engine = session.getEngine();
                if ((engine != null) && (selectedItem != null)) {
                    String expression = selectedItem.expression;
                    DisplayItemManager.manager.evaluate(model.getPropertyChangeListener(),
                    session,
                    selectedItem.node,
                    expression);
                } else {
                    TopManager.getDefault().getErrorManager().log("DataBrowserWindow: INTERNAL ERROR engine or selection null, selection: "+selectedItem);//NOI18N
                    if (IpeUtils.asserts) {
                        IpeUtils.ensure(false);
                    }
                }
            }
        } else if (actionEvent.getSource() == removeMenuItem) {
            if (trace) {
                System.err.println("@@@ DataBrowserWindow.actionPerformed remove selected items");//NOI18N
            }
            if (selectedItem != null) {
                items.remove(selectedItem);
                remove();
            }
        } else if (actionEvent.getSource() == removeAllMenuItem) {
            if (trace) {
                System.err.println("@@@ DataBrowserWindow.actionPerformed remove all " + items.size() + " items");//NOI18N
            }
            Iterator iter = items.iterator();
            while (iter.hasNext()) {
                selectedItem = (ListItem) iter.next();
                if (trace) {
                    System.err.println("... ask to remove item: "+selectedItem.node.toString());//NOI18N
                }
                if (selectedItem != null) {
                    remove();
                    iter.remove();
                }
            }
        } else if (actionEvent.getSource() == colsButton) {
            treeTable.selectVisibleColumns();
        } else if (actionEvent.getSource() == expressionStackCombo) {
            if (source == null) {
                return;
            }
            
            if (!actionEvent.getActionCommand().equals("comboBoxEdited")) {//NOI18N
                // Hack. actionPerformed gets called twice if the user enters a
                // new string and hits ENTER. The first call has an empty string
                // (as does the call you get when the user selects an entry from
                // the JComboBox pop-up), and the second call (only when you type
                // a new entry and hit ENTER) is the string "comboBoxEdited".
                if (trace) {
                    System.err.println("...return because getActionCommand is not comboBoxEdited,  getActionCommand is: \'"+actionEvent.getActionCommand()+"\'");//NOI18N
                }
                return;
            }
            
            // Evaluate the given expression
            String expression = expressionStackCombo.getSelectedItem().toString();
            if (!expression.trim().equals("")) { //NOI18N
                newExpression(expression);
            }
        }
    }
    
    
    /** Extra Swing setup stuff not done by the gui builder code (initComponents)
     */
    private void additionalInitComponents(ExpressionTreeTableModel newModel) {
        model = newModel;
        setName(DbxDebugger.getText("TITLE_DataBrowserWindow")); //NOI18N
        setIcon(org.openide.util.Utilities.loadImage("com/sun/tools/dbxgui/icons/databrowser.gif")); // NOI18N
        
        if (model == null) {
            model = new ExpressionTreeTableModel(session, DbxDebugger.getText("TITLE_DataBrowserWindow"), false, source, true);//NOI18N
            if (session != null) {
                sessionMap.put(session, model);
            } else {
                nullSessionModel = model;
            }
        }
        
        if (scrollPane != null) {
            remove(scrollPane);
        }
        
        if (model != null) {
            treeTable = new TreeTable(this, model);
            scrollPane = new JScrollPane(treeTable);
            scrollPane.getVerticalScrollBar().setUnitIncrement(20);//!!!
            scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
            scrollPane.setCorner(JScrollPane.UPPER_RIGHT_CORNER, colsButton);
            add(scrollPane, java.awt.BorderLayout.CENTER);
            scrollPane.getViewport().setBackground(treeTable.getBackground());
            
            treeTable.getTree().addTreeSelectionListener(this);
            
            addPopupListener(scrollPane);
            addPopupListener(treeTable);
            addPopupListener(treeTable.getTableHeader());
            
            expressionPanel.setBackground((Color) UIManager.get("Table.background"));//NOI18N
            
            // Add the new panel as the last entry in the window.
            add(newExprPanel, java.awt.BorderLayout.SOUTH);
        }
        
        validate();
        repaint();
    }
    
    
    protected void closeNotify() {
        if (trace) {
            System.err.println("@@@ DataBrowserWindow.closeNotify()");//NOI18N
        }
        
        super.closeNotify();
    }
    
    
    protected void componentActivated() {
        disableCutCopy();
        super.componentActivated();
    }
    
    
    protected void componentDeactivated() {
        disableCutCopy();
        super.componentDeactivated();
    }
    
    
    /** The component is no longer showing: tell debugger to stop computing
     * updates on our behalf! */
    public void componentHidden(ComponentEvent event) {
        if (IpeUtils.trace)
            System.out.println("DataBrowserWindow.componentHidden!"); //NOI18N
        
        if (trace) {
            System.err.println("@@@ DataBrowserWindow.componentHidden");//NOI18N
        }
        
        if (!viewShowing) {
            // For some reason, inside NetBeans top managers we often seem
            // to get componentHidden BEFORE anything else!
            // if (IpeUtils.asserts) {
            //	  IpeUtils.ensure(false);
            // }
            return;
        }
        
        if (shown > 0) {
            // It's silly to decrement below zero.
            shown--;
            if (source != null) {
                DbxDebugSession oldSession = ((SessionDataSource) source).getSession();
                setSession(oldSession, null);
            }
        }
        
        super.componentHidden(event);
    }
    
    
    /** Don't care - but must implement full ComponentListener interface */
    public void componentMoved(ComponentEvent event) {
        // Do nothing.
    }
    
    
    /** Don't care - but must implement full ComponentListener interface */
    public void componentResized(ComponentEvent event) {
        // Do nothing.
    }
    
    
    /** The component is now showing: tell debugger to send us updates! */
    public void componentShown(ComponentEvent event) {
        if (IpeUtils.trace)
            System.out.println("DataBrowserWindow.componentShown!"); //NOI18N
        
        if (trace) {
            System.err.println("@@@ DataBrowserWindow.componentShown");//NOI18N
        }
        
        if (viewShowing) {
            if (IpeUtils.asserts) {
                IpeUtils.ensure(false);
            }
            return;
        }
        
        shown++;
        if (source != null) {
            DbxDebugSession newSession = ((SessionDataSource) source).getSession();
            setSession(null, newSession);
        }
        
        super.componentShown(event);
    }
    
    
    /** Disable copy, cut. */
    private void disableCutCopy() {
        copy.setActionPerformer(null);
        cut.setActionPerformer(null);
    }
    
    
    /** Get the delegating debugger view associated with this window */
    public DelegatingView2 getDelegatingView() {
        return DbxGuiModule.browserDView;
    }
    
    
    public HelpCtx getHelpCtx() {
        return new HelpCtx("Debugging_databrowser"); // NOI18N
    }
    
    
    // For RoutingPropertyChangeListener interface.
    public RoutingToken getRoutingToken() {
        return routingToken;
    }
    
    
    public static DataBrowserWindow getSingleton() {
        return singleton;
    }
    
    
    /** Overrides superclass method. Gets actions for this top component. */
    public SystemAction[] getSystemActions() {
        TreeTableAction treeTableAction = (TreeTableAction) SystemAction.get(TreeTableAction.class);
        treeTableAction.setWindow(this);
        treeTableAction.setTable(treeTable);
        treeTableAction.setEnabled(selectedItem != null);
        
        DebugWinDockAction dock = (DebugWinDockAction)
        SystemAction.get(DebugWinDockAction.class);
        dock.setDebuggerWindow(this);
        
        SystemAction[] actions = new SystemAction[] {
            treeTableAction,
            null,
            dock
        };
        
        return actions;
    }
    
    
    /** Get the debugger view associated with this window */
    public View2 getView() {
        return DbxGuiModule.browserView;
    }
    
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the FormEditor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents
        scrollPane = new javax.swing.JScrollPane();
        expressionPanel = new javax.swing.JPanel();

        setLayout(new java.awt.BorderLayout());

        setBackground(java.awt.Color.white);
        expressionPanel.setLayout(new java.awt.GridBagLayout());

        scrollPane.setViewportView(expressionPanel);

        add(scrollPane, java.awt.BorderLayout.CENTER);

    }//GEN-END:initComponents
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JScrollPane scrollPane;
    private javax.swing.JPanel expressionPanel;
    // End of variables declaration//GEN-END:variables
    
    
    /** Set accessible names and descriptions as necessary */
    void initializeA11y() {
        super.initializeA11y();
        expressionStackCombo.getAccessibleContext().setAccessibleDescription(
        DbxDebugger.getText("ACSD_LBL_Expression")); // NOI18N
    }
    
    
    public void newExpression(String expression) {
        if (trace) {
            System.err.println("@@@ DataBrowserWindow.newExpression expression: "+expression);//NOI18N
        }
        DbxDebugSession session = ((SessionDataSource) source).getSession();
        if (session != null) {
            DbxDebuggerEngine engine = session.getEngine(); {
                if (engine != null) {
                    DisplayItemManager.manager.evaluate(this, session, null, expression);
                }
            }
        } else {
            TopManager.getDefault().getErrorManager().log("!!! INTERNAL ERROR - DataBrowserWindow.actionPerformed, NULL session");//NOI18N
            if (IpeUtils.asserts) {
                IpeUtils.ensure(false);
            }
        }
        
        expressionStackModel.add(expression);
        
        // Select the text such that it's easy to either edit the
        // current text or completely replace it ("pending delete")
        Component c = expressionStackCombo.getEditor().getEditorComponent();
        if (c instanceof JTextField) {
            final JTextField jtf = (JTextField) c;
            // We have to defer this on the event queue because
            // something related to this actionPerformed (but later)
            // is causing the selection to be undone
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    jtf.selectAll();
                }
            });
        }
    }
    
    
    protected void populateMenu(Object source, int xpos, int ypos, JPopupMenu popup) {
        if (IpeUtils.trace) {
            System.out.println("@@@ DataBrowserWindow.populateMenu"); // NOI18N
        }
        
        Point p = new Point(xpos, ypos);
        selectItemAtPoint(p);
        treeTable.singleClicked(p, source);
        
        updateAtStopCommand = new JCheckBoxMenuItem(DbxDebugger.getText("LBL_UpdateAtStop"));
        updateAtStopCommand.setState((selectedItem == null) ? false : selectedItem.updateOnStop);
        updateAtStopCommand.setEnabled(selectedItem != null);
        popup.add(updateAtStopCommand);
        updateAtStopCommand.addActionListener(this);
        
        updateNowCommand =
            new JMenuItem(updateNowMenuItemString.replace('&', ' '), //NOI18N
                  updateNowMenuItemString.charAt(updateNowMenuItemString.indexOf('&') + 1));
        updateNowCommand.setEnabled(selectedItem != null);
        popup.add(updateNowCommand);
        updateNowCommand.addActionListener(this);
        
        popup.addSeparator();
        
        removeMenuItem = new JMenuItem(DbxDebugger.getText("LBL_Remove"));
        popup.add(removeMenuItem);
        removeMenuItem.addActionListener(this);
        
        removeAllMenuItem = new JMenuItem(DbxDebugger.getText("Breakpoints_deleteAll"));
        popup.add(removeAllMenuItem);
        removeAllMenuItem.addActionListener(this);
        
        popup.addSeparator();
        
        // Remainder of menu: use system actions
        addSystemActionsToMenu(popup);
        
        removeMenuItem.setEnabled((items.size() > 0) && (selectedItem != null));
        removeAllMenuItem.setEnabled(items.size() > 0);
    }
    
    
    // Implements PropertyChangeListener
    public void propertyChange(PropertyChangeEvent event) {
        if (trace) {
            System.err.println("===== DataBrowserWindow.propertyChange: event: "+event.getPropertyName()+", newValue: "+event.getNewValue());//NOI18N
        }
        if (DbxDebugSession.PROP_EVALUATE_RESULT.equals(event.getPropertyName())) {
            Object[] tuple = (Object[]) event.getNewValue();
            DbxDebugSession session = (DbxDebugSession) tuple[1];
            int routingToken = session.getRoutingToken();
            if (session != null) {
                if (session.getEvaluationResult() != null) {
                    if (routingToken == routingToken) {
                        // This was the result of creating a new expression.
                        if (trace) {
                            System.err.println("...routingToken is broadcast token: "+session.getRoutingToken());//NOI18N
                        }
                        result(session, session.getEvaluationResult());
                    } else {
                        // This was the result of expanding a node.
                        // The result will be handled in the appropriate model.
                        if (trace) {
                            System.err.println("...routingToken is NOT broadcast token: "+session.getRoutingToken());//NOI18N
                        }
                    }
                } else {
                    TopManager.getDefault().getErrorManager().log("!!! INTERNAL ERROR - DataBrowserWindow propertyChange, NULL getEvaluationResult");//NOI18N
                    if (IpeUtils.asserts) {
                        IpeUtils.ensure(false);
                    }
                }
            } else {
                TopManager.getDefault().getErrorManager().log("!!! INTERNAL ERROR - DataBrowserWindow propertyChange, NULL getEvaluationResult");//NOI18N
                if (IpeUtils.asserts) {
                    IpeUtils.ensure(false);
                }
            }
        } else if (DbxDebugSession.PROP_SESSION_CHANGED.equals(event.getPropertyName())) {
            // Handle session switching.
            DbxDebugSession newSession = (DbxDebugSession) event.getNewValue();
            DbxDebugSession oldSession = (DbxDebugSession) event.getOldValue();
            setSession(oldSession, newSession);
            super.propertyChange(event);
        } else if (DbxDebugSession.PROP_SESSION_QUIT.equals(event.getPropertyName())) {
            DbxDebugSession deadSession = (DbxDebugSession) event.getNewValue();
            ExpressionTreeTableModel oldModel = (ExpressionTreeTableModel) sessionMap.get(deadSession);
            sessionMap.remove(deadSession);
            if (deadSession == session) {
                setSession(session, null);
            }
            
            if (trace) {
                System.out.println("...session quit: " + deadSession.getShortName());//NOI18N
            }
        } else {
            super.propertyChange(event);
        }
    }
    
    
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
        if (IpeUtils.trace) {
            System.out.println("@@@ DataBrowserWindow.readExternal (createid is " + myid + ")"); // NOI18N
        }
        
        // DO NOT CHANGE SERIALIZATION!!!
        super.readExternal(in);
        
        // Get the serialization version.
        // For now, only one serialization version so no special handling here.
        int version = in.read();
        
        if (version != SERIALIZATION_VERSION) {
            if (IpeUtils.asserts) {
                IpeUtils.ensure(version == SERIALIZATION_VERSION);
            }
        }
        
        // Get persistent fields.
        // -- none yet --
    }
    
    
    Object readResolve() throws ObjectStreamException {
        DataBrowserWindow w = getSingleton();
        resolvedTo(w);
        return w;
    }
    
    
    private void remove() {
        if (trace) {
            System.err.println("@@@ DataBrowserWindow.remove selectedItem.node: "+(((selectedItem!=null)&&(selectedItem.node!=null))?selectedItem.node:null));//NOI18N
        }
        if (selectedItem != null) {
            // Do _NOT_ do "items.remove(selectedItem)" here.
            // Instead, do this in the caller. This allows the
            // caller to remove the item within an iterator.
            // Otherwise, you will get a concurrent access error
            // if the caller is using an iterator on the list and
            // you try to remove the item from the list here.
            treeTable.clearSelection();
            model.remove(selectedItem.node);
            selectedItem = null;
        }
    }
    
    
    /** Called when readResolve has decided to use "win" instead of this
     * window. We need to copy any state deserialized into the new object.
     */
    protected void resolvedTo(DebuggerWindow win) {
        super.resolvedTo(win);
    }
    
    
    /** Update the view to show the given set of evaluations.
     * @param evaluate A vector containing strings with evaluate dump info.
     */
    public void result(DbxDebugSession session, DbxHEvalResult result) {
        if (trace) {
            System.err.println("@@@ DataBrowserWindow.result lhs: "+result.plain_lhs);//NOI18N
        }
    /* FYI
    ! public class DbxHEvalResult {
    !     public int flags;
    !     public int is_a_pointer;
    !     public String plain_lhs;
    !     public String qualified_lhs;
    !     public String reevaluable_lhs;
    !     public String static_type;
    !     public String rhs;
    !     public String rhs_vdl;
    !     public String addr;
    !     public DbxHEvalResult() { }
    !     public DbxHEvalResult(MsgRcv _rcv) {
    !       decode(_rcv);
    !     }*/
        final ListItem listItem = new ListItem();
        Value value = null;
        
        // Parse VDL Lisp.
        try {
            VDLLispParser vp = new VDLLispParser(result.rhs_vdl);
            value = vp.getValue();
        } catch (Exception e) {
            TopManager.getDefault().getErrorManager().log("!!! UNEXPECTED EXCEPTION: DataBrowserWindow.result: VDLParser construction exception: " + e);//NOI18N
            TopManager.getDefault().getErrorManager().notify(ErrorManager.INFORMATIONAL, e);
            if (IpeUtils.asserts) {
                IpeUtils.ensure(false);
            }
            value = null;
        }
        
        listItem.expression = result.plain_lhs;
        items.add(listItem);
        ExpressionNode node = model.add(session,
        -1,
        result.plain_lhs,
        value,
        result.reevaluable_lhs,
        result.is_a_pointer != 0);
        listItem.node = node;
        if (trace) {
            System.err.println("...new node: "+node.describe());//NOI18N
        }
    }
    
    
    /** Change the session shown in this component.
     *  A non tested assertion is the oldSession != newSession.
     */
    public void setSession(DbxDebugSession oldSession,
    DbxDebugSession newSession) {
        if (trace) {
            System.err.println("@@@ DataBrowserWindow.setSession old: " + oldSession+", new: " + newSession);//NOI18N
        }
        if (oldSession != null) {
            if (trace) {
                System.err.println("... hiding old session");//NOI18N
            }
            if (model != null) {
                model.hide();
            }
        }
        
        session = newSession;
        if (session == null) {
            model = nullSessionModel;
        } else {
            model = (ExpressionTreeTableModel) sessionMap.get(session);
        }
        
        additionalInitComponents(model);
        if (model != null) {
            model.show();
        }
    }
    
    
    private void selectItemAtPoint(Point p) {
        int cap = treeTable.columnAtPoint(p);
        int rap = treeTable.rowAtPoint(p);
        int realCol = treeTable.convertColumnIndexToModel(cap);
        javax.swing.tree.TreePath path = treeTable.getTree().getPathForRow(rap);
        if (path != null) {
            ExpressionNode node = (ExpressionNode) path.getPathComponent(1);
            Iterator iter = items.iterator();
            while (iter.hasNext()) {
                ListItem item = (ListItem) iter.next();
                if (node == item.node) {
                    selectedItem = item;
                    break;
                }
            }
        }
    }
    
    
    protected void singleClicked(Point p, Object source) {
        super.singleClicked(p, source);
        selectItemAtPoint(p);
        if (trace) {
            System.err.println("@@@ DataBrowserWindow.singleClicked selectedItem: "+selectedItem);//NOI18N
        }
        treeTable.singleClicked(p, source);
    }
    
    
    // For RoutingPropertyChangeListener interface.
    public String toString() {
        return "DataBrowserWindow[hash:" + this.hashCode() + //NOI18N
               ", routingToken:" + routingToken +//NOI18N
               ", acceptsExternalWatches:" + acceptsExternalWatches() +//NOI18N
               "]";//NOI18N
    }
    
    
    public void updateOnStop(boolean doUpdate) {
        if (selectedItem != null) {
            if (model != null) {
                model.updateOnStop(doUpdate, selectedItem.node);
            } else {
                TopManager.getDefault().getErrorManager().log("!!! INTERNAL ERROR: DataBrowserWindow.updateOnStop null selectedItem.model");//NOI18N
                if (IpeUtils.asserts) {
                    IpeUtils.ensure(false);
                }
            }
        } else {
            TopManager.getDefault().getErrorManager().log("!!! INTERNAL ERROR: DataBrowserWindow.updateOnStop null selectedItem");//NOI18N
            if (IpeUtils.asserts) {
                IpeUtils.ensure(false);
            }
        }
    }
    
    
    public void valueChanged(TreeSelectionEvent event) {
        ExpressionNode node = (ExpressionNode) event.getPath().getPathComponent(1);
        Iterator iter = items.iterator();
        while (iter.hasNext()) {
            ListItem item = (ListItem) iter.next();
            if (node == item.node) {
                selectedItem = item;
                break;
            }
        }
    }
    
    
    public void writeExternal(java.io.ObjectOutput oo) throws java.io.IOException {
        if (IpeUtils.trace) {
            System.out.println("@@@ DataBrowserWindow.writeExternal (createid is " + myid + ")"); // NOI18N
        }
        
        // DO NOT CHANGE SERIALIZATION!!!
        super.writeExternal(oo);
        
        // Write out serialization version to handle upgrades in the future
        oo.write(SERIALIZATION_VERSION); // SERIAL VERSION
        
        // Write persistant fields.
        // -- none yet --
    }
    
    
    /** Class holding expression item info such that we can look
     * up lhs etc.
     */
    public class ListItem {
        public String expression = null; // left hand size of expression
        public ExpressionNode node = null; // The associated expression node.
        public boolean updateOnStop = false; // Current update model.
        
        public void clear() {
            expression = null;
            node = null;
            updateOnStop = false;
        }
    }
    
    
}
