#
# perl module - utility routines
# - MS specific
#   - get_hosted_domains
#   - check_configaccess
#   - check_servers
#   - stop_servers
#   - isHAconfigured
#   - perform_checks
# - logging
#   - getLogfile
#   - log_init
#   - log_msg
#   - log_only
#   - log_cmd
#   - logonly_cmd
#   - debug_log
#   - abort
#   - warning
#   - ContinueYesNo
#   - ReadAnswer
#

use Cwd;                        # for cwd

########################################
# to get list of hosted domains
# - does not include default domain
# - uses global variable $basedir
# returns @domains
########################################
sub get_hosted_domains {
  my (@domains,$defdomain,$key,$value,$found,$schematype,$currDir);
  my ($ughost,$ugport,$binddn,$passwd,$dctree,$dctree,$tmpfile,$Cmd,$output);
  my (@dc_rdns,@rdns,$num,$domain);

  $defdomain = `$basedir/sbin/configutil -o local.defdomain`;
  chomp $defdomain;
  #print "defdomain $defdomain\n";
  @domains = ();

  # this only works if MTA is configured
#   open(DOM, "echo enumerate|$basedir/sbin/imsimta test -domain_map |") or die "Failed to execute imsimta test -domain_map -";
#   while (<DOM>) {
#     chomp;
#     # trim leading and trailing whitespace
#     s/^\s+//;
#     s/\s+$//;
#     #print "QQQ hosted domain $_\n";
#     if ($_ ne $defdomain) {
#       push(@domains,$_);
#     }
#   }
#   close(DOM);

  #
  # do  a search for all DNs that have objectclass=maildomain
  #   for schema 1, the mail domain will be the dn,
  #      e.g. dn: dc=wst,dc=sun,dc=com,$dctree
  #   for schema 2, look at the sunPreferredDomain

  # get schema type
  # one way to do it
  # $schematype = `grep ^LDAP-SCHEMALEVEL config/option.dat | awk -F= '{print $2}'
  # $schematype is also the value of msg.schematype in Devsetup.properties

  $found = 0;
  open(OPTION, "$basedir/config/option.dat") or die "Failed to open $basedir/config/option.dat -";
  while (<OPTION>) {
    chomp;
    ($key,$value) = split(/=/,$_,2);
    # remove leading and trailing whitespace
    $value =~ s/^\s+//;
    $value =~ s/\s+$//;
    $key =~ s/^\s+//;
    $key =~ s/\s+$//;
    if ( $key eq "LDAP_SCHEMALEVEL" ) {
      $found = 1;
      $schematype = $value;
    }
  }
  if ( $found == 0 ) {
    die "did not find LDAP_SCHEMALEVEL in $basedir/config/option.dat -";
  }

  # do ldapsearch to find domains
  $ughost = `$basedir/sbin/configutil -o local.ugldaphost`;
  chomp $ughost;
  $ugport = `$basedir/sbin/configutil -o local.ugldapport`;
  chomp $ugport;
  $binddn = `$basedir/sbin/configutil -o local.ugldapbinddn`;
  chomp $binddn;
  $passwd = `$basedir/sbin/configutil -o local.ugldapbindcred`;
  chomp $passwd;
  $dctree = `$basedir/sbin/configutil -o service.dcroot`;
  chomp $dctree;
  $tmpfile = "/tmp/.patch-config.$$";
  open(TMPFILE, ">$tmpfile") or die "Failed to open temporary file $tmpfile -";
  print TMPFILE "$passwd\n";
  close(TMPFILE);
  $currDir = cwd();
  $Cmd = "$basedir/lib/ldapsearch -h $ughost -p $ugport -D \"$binddn\" -j $tmpfile -b $dctree objectclass=maildomain";
  if ($schematype == 2 || $schematype == 1.5) {
    chdir "$basedir/lib";
    foreach $_ (`$Cmd sunPreferredDomain`) {
      if ($_ =~ /^sunPreferredDomain/i) {
        chomp;
        ($key,$value) = split(/: /, $_, 2);
        if ($value ne $defdomain) {
          push(@domains,$value);
        }
      } elsif ($_ =~ /^ldap_simple_bind/i) {
        if ($_ =~ /Can\'t connect/i) {
          log_msg("Directory server appears not running or not listening to port $ugport.\n");
        } elsif ($_ =~ /invalid credentials/i) {
          log_msg("The Directory Server credentials are incorrect.\n");
        } else {
          $_ =~ s/^ldap_simple_bind: //i;
          log_msg("Error: $_.\n");
        }
        last;
      }
    }
    chdir $currDir;
    unlink($tmpfile);
  } elsif ($schematype == 1) {
    @dc_rdns = split(/,/, $dctree);
    chdir "$basedir/lib";
    foreach $_ (`$Cmd dn`) {
      if ($_ =~ /^dn/i) {
        chomp;
        ($key,$value) = split(/: /, $_, 2);
        # convert dn to a domain, sample:
        #  dn: dc=budha2,dc=west,dc=sun,dc=com,o=internet
        #  becomes budah2.west.sun.com

        # break out into RDNs
        @rdns = split(/,/, $value);
        $num = @rdns - @dc_rdns;
        ($key,$domain) = split(/=/,$rdns[0],2);
        $domain =~ s/^\s+//;
        $domain =~ s/\s+$//;
        for ($i = 1; $i < $num; $i++) {
          ($key,$value) = split(/=/,$rdns[$i],2);
          $value =~ s/^\s+//;
          $value =~ s/\s+$//;
          $domain = $domain . "." . "$value";
        }

        if ($domain ne $defdomain) {
          push(@domains,$domain);
        }
      } elsif ($_ =~ /^ldap_simple_bind/i) {
        if ($_ =~ /Can\'t connect/i) {
          log_msg("Directory server appears not running or not listening to port $ugport.\n");
        } elsif ($_ =~ /invalid credentials/i) {
          log_msg("The Directory Server credentials are incorrect.\n");
        } else {
          $_ =~ s/^ldap_simple_bind: //i;
          log_msg("Error: $_.\n");
        }
        last;
      }
    }
    chdir $currDir;
    unlink($tmpfile);
  } else {
    die "Unrecognized LDAP_SCHEMALEVEL $schematype found in $basedir/config/option.dat -";
  }
  log_only("get_hosted_domains found @domains\n");
  return(@domains);
}

########################################
# check to make sure config is accessible
#  use global variable: $basedir
########################################
sub check_configaccess
{
  if ( ! -w "$basedir/config/imta_tailor" ) {
    log_msg("-- Could not access the config area $basedir/config\n");
    log_msg("-- This utility requires access to the config area\n");
    die "  ";
  }
}

########################################
# check to make sure servers are stopped
#  use global variable: $basedir
########################################
sub check_servers
{
  my($stat);
  log_only("-- check_servers\n");
  if ( -x "$basedir/lib/msstart" ) {
    $Cmd = "$basedir/lib/msstart -l >> $LOGFILE 2>&1";
    $stat = logonly_cmd($Cmd);
    log_only("-- msstart -l returned $stat\n");
    if ( $stat == 0 ) {
      log_only("-- No servers running\n");
    } else {
      $Cmd = "$basedir/lib/msstart -l";
      log_msg("--\n");
      log_msg("-- Please stop the services before applying the patch\n");
      log_msg("--\n");
      die "  ";
    }
  }
}

########################################
# stop servers
#  use global variable: $basedir
########################################
sub stop_servers
{
  log_only("-- stop_servers\n");
  if ( -x "$basedir/sbin/stop-msg" ) {
    log_msg("-- Stopping services: $basedir/sbin/stop-msg\n");
    $Cmd = "$basedir/sbin/stop-msg";
    log_cmd($Cmd);
  }
}

########################################
# determine if configured for HA
# - set the variable isHAconfigured to 0 (false) or 1 (true)
########################################
sub isHAconfigured
{
  $VCSCMD = "/opt/VRTSvcs/bin/hares";
  $SC3CMD = "/usr/cluster/lib/rgm/rtreg/SUNW.ims";
  $IP_INTERFACE = `$basedir/lib/configutil -o service.listenaddr 2> $LOGFILE`;
  chomp $IP_INTERFACE;
  if ( "$IP_INTERFACE" ne "INADDR_ANY" && ( -f "$VCSCMD" || -f "$SC3CMD" ) ) {
    $isHAconfigured = 1;
  } else {
    $isHAconfigured = 0;
  }
  log_only("-- isHAconfigured = $isHAconfigured\n");
}


########################################
# perform_checks
########################################
sub perform_checks
{
  my($Cmd,$imtaEnable);
  log_only("-- perform_checks being run\n");
  $imtaEnable = `$basedir/sbin/configutil -o local.imta.enable`;
  chomp $imtaEnable;

  if ($imtaEnable == 0) {
    return;
  }

  log_msg("-- Running clbuild, chbuild, cnbuild, test rewrite...\n");

  $Cmd = "$basedir/sbin/imsimta clbuild -image_file=IMTA_COMMAND_DATA IMTA_BIN:pmdf.cld";
  log_cmd($Cmd);

  $Cmd = "$basedir/sbin/imsimta chbuild";
  log_cmd($Cmd);

  $Cmd = "$basedir/sbin/imsimta cnbuild";
  log_cmd($Cmd);

  $Cmd = "$basedir/sbin/imsimta version";
  log_cmd($Cmd);

  $Cmd = "$basedir/sbin/imsimta test -rewrite -debug postmaster";
  logonly_cmd($Cmd);
}

########################################
# generate a a timestamp YYYYMMDDHHMMSS
# args:
# return: string - the timestamp
########################################
sub genTimestamp {
  my ($t,$now,$datestr);
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst);
  $t = time;
  $now = localtime($t);
  ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($t);
  $year += 1900;
  $mon = $mon + 1;
  $datestr = sprintf ("%d%02d%02d%02d%02d%02d", $year, $mon, $mday, $hour, $min, $sec);
  return($datestr);
}

########################################
# generate a logfile name. The format is
#   <directory><delim><prefix>_<timestamp>.log
# e.g.
#   for
#     <directory> = /tmp
#     <delim> = /
#     <prefix> = patch-config
#   gives
#     /tmp/patch-config_20050915110030.log
# args:
#   dir - directory where the LOGFILE will live
#   delim - pathname delimiter
#   prefix - the prefix for the logfile
# return: string - the logfile name
# side effect: set $LOGFILE global variable
########################################
sub getLogfile {
  my($dir,$delim,$prefix) = @_;
  my ($t,$now,$datestr);
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst);
  $t = time;
  $now = localtime($t);
  ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($t);
  $year += 1900;
  $mon = $mon + 1;
  $datestr = sprintf ("%d%02d%02d%02d%02d%02d", $year, $mon, $mday, $hour, $min, $sec);
  $LOGFILE = "$dir" . "$delim" . "$prefix" . "_$datestr.log";
  return($LOGFILE);
}

########################################
# setup logging, output is LOG file descriptor - as a global variable
# TODO: log routines need to be in a utility module
########################################
sub log_init
{
  my($LOGFILE) = @_;
  my($stat);
  $stat = open(LOG, ">>$LOGFILE") or die("ERROR: Cannot open $LOGFILE -");
  $date = `date`;
  chomp $date;
  print LOG "============ LOGFILE started $date ==========\n";
  print LOG "LOGFILE = $LOGFILE\n";
  print LOG "Running $^X $0 @ARGV\n";
#  if ($^O eq "MSWin32") {
#    print LOG "  Windows Version was determined to be $windowsversion\n";
#  }
}

########################################
# log a message to the screen and the logfile
########################################
sub log_msg
{
  print "$_[0]";
  print LOG "$_[0]";
}

########################################
# log a message to the logfile only
########################################
sub log_only
{
  print LOG "$_[0]";
}

########################################
# run a command and log it
#  return the status
########################################
sub log_cmd
{
  my ($Cmd) = @_;
  my ($output, $stat);
  log_msg("$Cmd\n");
  $output = `$Cmd`;
  $stat = $? / 256;
  foreach $_ ($output) {
    log_msg("$_");
  }
  log_msg("stat = $stat\n");
  return($stat);
}

########################################
# run a command and output to the log only
########################################
sub logonly_cmd
{
  my ($Cmd) = @_;
  my ($output, $stat);
  log_only("$Cmd\n");
  $output = `$Cmd`;
  $stat = $? / 256;
  foreach $_ ($output) {
    log_only("$_");
  }
  log_only("stat = $stat\n");
  return($stat);
}

########################################
# print debugging message, if $DEBUG != 0
#  uses Global variable: $DEBUG
########################################
sub debug_log
{
  if ( $DEBUG != 0 ) {
    log_msg($_[0]);
  }
}

########################################
# print message and abort
########################################
sub abort
{
  log_msg("\n$_[0]\n\n");
  exit 1;
}

########################################
# print warning
########################################
sub warning
{
  log_msg("\n WARNING: $_[0]\n\n");
}

########################################
# print question and ask whether to continue or not
########################################
sub ContinueYesNo
{
  my ($ans);
  $ans = YesOrNo("Do you want to continue ?", "yes");
  if ($ans eq "y") {

  } elsif ( $ans eq "n") {
    log_msg ( "Exiting....");
    exit (0);
  }
  return;
}

########################################
# print question and read the answer
########################################
sub ReadAnswer
{
  if ($CmdLineFlag ) {
    return;
  }

  ($question, $default) = @_;

  log_msg("\n$question");

  $answer = <STDIN>;
  chomp $answer;

  # Do not put password in logfile.
  if ($question ne "Password: ") {
    log_only($answer);
  }
  if ( $answer ) {
    return ( $answer );
  } else {
    return ( $default );
  }
}

########################################
# YesOrNo
# args
# - $question - the question
# - $default - the default answer (must be one of y/n/yes/no)
# returns
# - y or n
########################################
sub YesOrNo ()
{
  my($question,$default) = @_;
  my($ans);
  while (1) {
    log_msg("$question [$default] : ");
    $ans = <STDIN>;
    log_only($ans);
    chomp $ans;
    if ( !$ans ) {
      $ans = $default;
    }
    if ( $ans =~ /^y$|^yes$/i ) {
      $ans = "y";
    } elsif ( $ans =~ /^n$|^no$/i ) {
      $ans = "n";
    } else {
      log_msg("Invalid answer ($ans), please answer y or n\n");
      next;
    }
    last;
  }
  return($ans);
}

# required by require
return 1;
