/*
 * Copyright 2002 Sun Microsystems, Inc. All Rights Reserved 
 * Use of this product is subject to license terms.
 */
/************************************************************

 testpwdstore.c

 This source file provides an example of a password storage scheme
 plug-in with trivial functions for encoding and comparing password
 attribute values.

 ACTIVATING THIS PLUG-IN
 -----------------------
 1. Build the library containing the plug-in.
 
 2. Configure the server to load the plug-in.
 
    Create a file containing the following entry, named
    testpwdstore.ldif, replacing the value of nsslapd-pluginPath with
    the path appropriate for your installation:

dn: cn=XOR,cn=Password Storage Schemes,cn=plugins,cn=config
objectclass: top
objectclass: nsSlapdPlugin
cn: XOR
nsslapd-pluginPath: <ServerRoot>/plugins/slapd/slapi/examples/<LibName>
nsslapd-pluginInitFunc: xor_init
nsslapd-pluginType: pwdstoragescheme
nsslapd-pluginEnabled: on
nsslapd-pluginId: xor-password-storage-scheme
nsslapd-pluginVersion: 5.2
nsslapd-pluginVendor: Sun Microsystems, Inc.
nsslapd-pluginDescription: Exclusive-or example (XOR)

    Add the configuration entry to the directory. For example:

    $ ldapmodify -a -p <port> -D "cn=directory manager" -w <password> -f testpwdstore.ldif
   
 3. Restart the server.

************************************************************/
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include "slapi-plugin.h"

static char * name           ="XOR";   /* Storage scheme name */
static Slapi_PluginDesc desc = {
    "xor-password-storage-scheme",     /* Plug-in identifier  */
    "Sun Microsystems, Inc.",          /* Vendor name         */
    "5.2",                             /* Revision number     */
    "Exclusive-or example (XOR)"       /* Plug-in description */
};

#define PREFIX_START '{'
#define PREFIX_END   '}'

/* Encode a password.
   Server frees memory in which encoded password is returned. */
static char *
xorenc(char * pwd)
{
    char * tmp    = NULL;              /* Used for encoding   */
    char * head   = NULL;              /* Encoded password    */
    char * cipher = NULL;              /* Prefix, then pwd    */
    int i, len;
  
    /* Allocate space to build the encoded password           */
    len = strlen(pwd);
    tmp = slapi_ch_malloc(len + 1);
    if (tmp == NULL) return NULL;

    memset(tmp, '\0', len + 1);
    head = tmp;

    /* Encode. This example is not secure by any means.       */
    for (i = 0; i < len; i++, pwd++, tmp++) *tmp = *pwd ^ 42;
  
    /* Add the prefix to the cipher                           */
    if (tmp != NULL) {
        cipher = slapi_ch_malloc(3 + strlen(name) + strlen(head));
        if (cipher != NULL) {
            sprintf(cipher,"%c%s%c%s",PREFIX_START,name,PREFIX_END,head);
        }
    }
    slapi_ch_free((void **) &head);
  
    return (cipher);                   /* Server frees cipher */
}

/* Compare password provided with stored, encoded password.   */
static int 
xorcmp(char * userpwd, char * dbpwd)
{
    /* Check the correspondence of the two char by char       */       
    int i, len = strlen(userpwd);
    for (i = 0; i < len; i++) {
        if ((userpwd[i] ^ 42) != dbpwd[i])
            return 1;                  /* Different passwords */
    } 
    return 0;                          /* Identical passwords */
}

/* Register the plug-in with the server                       */
#ifdef _WIN32
__declspec(dllexport)
#endif
int
xor_init(Slapi_PBlock * pb)
{
    int rc = 0;                        /* 0 means success     */
    rc |= slapi_pblock_set(            /* Plug-in API version */
        pb,
        SLAPI_PLUGIN_VERSION,
        (void *) SLAPI_PLUGIN_CURRENT_VERSION
    );
    rc |= slapi_pblock_set(            /* Plug-in description */
        pb,
        SLAPI_PLUGIN_DESCRIPTION,
        (void *) &desc
    );
    rc |= slapi_pblock_set(            /* Storage scheme name */
        pb,
        SLAPI_PLUGIN_PWD_STORAGE_SCHEME_NAME,
        (void *) name
    );
    rc |= slapi_pblock_set(            /* Encode password     */
        pb,
        SLAPI_PLUGIN_PWD_STORAGE_SCHEME_ENC_FN,
        (void *) xorenc
    );
    rc |= slapi_pblock_set(            /* Compare password    */
        pb,
        SLAPI_PLUGIN_PWD_STORAGE_SCHEME_CMP_FN,
        (void *) xorcmp
    );
    rc |= slapi_pblock_set(            /* Never decode pwd    */
        pb,
        SLAPI_PLUGIN_PWD_STORAGE_SCHEME_DEC_FN,
        NULL
    );
    return rc;
}
