/* 
 * Portions Copyright 2003 Sun Microsystems, Inc. All Rights Reserved 
 * Use is subject to license terms. 
 * Some preexisting portions Copyright 1997-2001 Netscape Communications Corp.
 * All rights reserved.
 */
/************************************************************

 testentry.c

 This source file provides examples of entry store and
 entry fetch plug-in functions.  These plug-in functions are
 called by the server before writing an entry to disk and
 after reading an entry from disk.

 Entry store and entry fetch plug-in functions are passed 
 the string representation (in LDIF -- LDAP Data Interchange 
 Format) of the entry.

 In this example, the entry store function performs a bitwise
 exclusive-or operation on each character in the entry 
 against the value 0xaa (10101010).  The entry fetch 
 function performs this again to revert each character 
 back to its initial value.

 NOTE: The Directory Server caches recently added and retrieved
 entries in memory.  The entry fetch plug-in function is called 
 only when reading the entry from the disk, *not* when reading
 the entry from the cache.  

 For example, if you add an entry and search for it, you will 
 not see a message in the server error log indicating that 
 the entry fetch plug-in function was called. In the process 
 of adding the entry to the directory, the server also added 
 the entry to the cache; the server then reads the entry from 
 the cache instead of from the disk and does not need to call 
 the entry fetch plug-in function.

 You can flush the cache by shutting down the server. 

 ACTIVATING THIS PLUG-IN
 -----------------------
 1. Build the library containing the plug-in.
         
 2. Configure the server to log messages and load the plug-in.
           
    Set the nsslapd-infolog-area attribute on the configuration entry,
    dn: cn=config, to turn on logging of information messages from
    plug-ins. The value is that of SLAPI_LOG_INFO_AREA_PLUGIN in
    slapi-plugin.h. One way of doing this:

    $ ldapmodify -p <port> -D "cn=directory manager" -w <password>
    dn: cn=config
    changetype: modify
    replace: nsslapd-infolog-area
    nsslapd-infolog-area: 65536

    Create a file containing the following entry, named testentry.ldif,
    replacing the value of nsslapd-pluginPath with the path appropriate
    for your installation:

dn: cn=Test entry,cn=plugins,cn=config
objectClass: top
objectClass: nsSlapdPlugin
objectClass: extensibleObject
cn: Test entry
nsslapd-pluginPath: <ServerRoot>/plugins/slapd/slapi/examples/<LibName>
nsslapd-pluginInitfunc: testentry_init
nsslapd-pluginType: ldbmentryfetchstore
nsslapd-pluginEnabled: on
nsslapd-plugin-depends-on-type: database
nsslapd-pluginId: test-entry
nsslapd-pluginVersion: 5.2
nsslapd-pluginVendor: Sun Microsystems, Inc.
nsslapd-pluginDescription: Sample entry store/fetch plug-in

    Add the configuration entry to the directory. For example:

    $ ldapmodify -a -p <port> -D "cn=directory manager" -w <password> -f testentry.ldif

 3. Restart the server.
         

*************************************************************/
#include <stdio.h>
#include <string.h>
#include "slapi-plugin.h"
#ifdef _WIN32
typedef unsigned int uint;
#endif

Slapi_PluginDesc entrypdesc = {
    "test-entry",                      /* plug-in identifier      */
    "Sun Microsystems, Inc.",          /* vendor name             */
    "5.2",                             /* plug-in revision number */
    "Sample entry store/fetch plug-in" /* plug-in description     */
};

/* Scramble the entry to store and note the work in the log.      */
#ifdef _WIN32
__declspec(dllexport)
#endif
int
testentry_scramble(char ** entry, uint * len)
{
    uint i;

    /* Scramble using bitwise exclusive-or on each character.     */
    for (i = 0; i < *len - 1; i++) { (*entry)[i] ^= 0xaa; }
    
    slapi_log_info_ex(
        SLAPI_LOG_INFO_AREA_PLUGIN,
        SLAPI_LOG_INFO_LEVEL_DEFAULT,
        SLAPI_LOG_NO_MSGID,
        SLAPI_LOG_NO_CONNID,
        SLAPI_LOG_NO_OPID,
        "testentry_scramble in test-entry plug-in",
        "Entry data scrambled.\n"
    );

    return 0;
}

/* Unscramble scrambled entries and note the work in the log.     */
#ifdef _WIN32
__declspec(dllexport)
#endif
int
testentry_unscramble(char ** entry, uint * len)
{
    uint i;

    /* Return now if the entry is not scrambled.                  */
    if (!strncmp(*entry, "dn:", 3)) { return 0; }
    
    /* Unscramble using bitwise exclusive-or on each character.   */
    for (i = 0; i < *len - 1; i++) { (*entry)[i] ^= 0xaa; }
    
    slapi_log_info_ex(
        SLAPI_LOG_INFO_AREA_PLUGIN,
        SLAPI_LOG_INFO_LEVEL_DEFAULT,
        SLAPI_LOG_NO_MSGID,
        SLAPI_LOG_NO_CONNID,
        SLAPI_LOG_NO_OPID,
        "testentry_unscramble in test-entry plug-in",
        "Entry data unscrambled.\n"
    );

    return 0;
}

/* Register the plug-in with the server.                          */
int 
testentry_init(Slapi_PBlock *pb)
{
    int rc = 0;                        /* 0 means success         */
    rc |= slapi_pblock_set(            /* Plug-in API version     */
        pb,
        SLAPI_PLUGIN_VERSION,
        SLAPI_PLUGIN_CURRENT_VERSION
    );
    rc |= slapi_pblock_set(            /* Plug-in description     */
        pb,
        SLAPI_PLUGIN_DESCRIPTION,
        (void *) &entrypdesc
    );
    rc |= slapi_pblock_set(            /* Entry store function    */
        pb,
        SLAPI_PLUGIN_ENTRY_STORE_FUNC,  
        (void *) testentry_scramble
    );
    rc |= slapi_pblock_set(            /* Entry fetch function    */
        pb,
        SLAPI_PLUGIN_ENTRY_FETCH_FUNC,  
        (void *) testentry_unscramble
    );
    return rc;
}
