/*
 *  Copyright (c) 2000 Sun Microsystems.  All rights reserved.
 *  All rights reserved.
 * 
 *  Use of this software is governed by the terms of the license agreement 
 *  for the Sun ONE Compass Server between the parties.
 */


/* ------------------------------------------------------------------------ */


/*
 * adt.h - ADT's for the Compass Server (CS): list, hash table, queues, etc.
 * Copyright 2003 Sun Microsystems, Inc. All
 * rights reserved. Use of this product is subject
 * to license terms. Federal Acquisitions:
 * Commercial Software -- Government Users
 * Subject to Standard License Terms and
 * Conditions.
 *
 * Sun, Sun Microsystems, the Sun logo, and Sun ONE
 * are trademarks or registered trademarks of Sun Microsystems,
 * Inc. in the United States and other countries.
 */

#ifndef _LIBCS_ADT_H_
#define _LIBCS_ADT_H_

#include <stdio.h>		/* FILE */
#include <sys/types.h>		/* size_t */

#ifdef __cplusplus
extern "C" {
#endif

typedef unsigned int (*CSHashFn)(void *);
typedef int (*CSCompareFn)(void *, void *);
typedef void *(*CSCopyFn)(void *);
typedef void (*CSFreeFn)(void *);
typedef struct cs_list_node CSListNode;
typedef struct cs_list CSList;
typedef struct cs_list CSStack;
typedef struct cs_list CSQueue;
typedef struct cs_list CSPriQ;
typedef struct cs_hashtbl CSHashTable;
typedef struct cs_hashtblstats CSHashTableStats;
typedef struct _ncs_io_access_s CSIO;

#define CSIO_MEMORY		0x01
#define CSIO_FILE		0x10
#define CSIO_BOTH		((CSIO_MEMORY)|(CSIO_FILE))
#define CSIO_IS_IN_FILE(p) 	((p)->location & CSIO_FILE)
#define CSIO_IS_IN_MEMORY(p) 	((p)->location & CSIO_MEMORY)
#define CSIO_BUFSZ		4096	/* 4k is good for disk */

struct _ncs_io_access_s {
	unsigned location;		/* Data is in memory, file, or both */

	char *data;			/* Memory buffer */
	size_t datalen;			/* len of valid data in data buffer */
	size_t current;			/* current position in buffer */
	size_t allocated;		/* #bytes allocated in mem buffer */
	size_t max_alloc;		/* max memory buffer to allocate */
	size_t chunksize;		/* size of the chunks to allocate */

	char *filename;			/* File which contains data */
	int   fd;			/* file pointer, for current */
	int   is_temporary;		/* Is this a temporary file? */
	char *fdbuf;			/* to speed up small appecs */
	size_t fdbufsz;			/* amount of data in fdbuf */

	int finished_writing;		/* done writing or not? */
};

struct cs_list_node {
	void *data;
	struct cs_list_node *next;
};

struct cs_list {
	CSListNode *head, *tail;
	unsigned int nnodes;
	CSFreeFn fr;
	CSCompareFn cmp;
	CSCopyFn cp;
};

struct cs_hashtbl {
	CSList **buckets;
	u_int hashsize;
	u_int nnodes;
	CSHashFn hf;
	CSCopyFn cp;
	CSCompareFn cmp;
	CSFreeFn fr;
};

struct cs_hashtblstats {
	int nobjs;
	int nbuckets;
	u_int longest_chain;
	int total_chain;
	double mean;
	double stddev;
};

/* CSList -- Singularly linked list */
NSAPI_PUBLIC int cslist_create(CSList **);
NSAPI_PUBLIC int cslist_set_compare(CSList *, CSCompareFn);
NSAPI_PUBLIC int cslist_set_free(CSList *, CSFreeFn);
NSAPI_PUBLIC int cslist_set_copy(CSList *, CSCopyFn);
NSAPI_PUBLIC int cslist_append(CSList *, void *);
NSAPI_PUBLIC int cslist_prepend(CSList *, void *);
NSAPI_PUBLIC int cslist_delete(CSList *, void *);
NSAPI_PUBLIC void *cslist_find(CSList *, void *);
NSAPI_PUBLIC void *cslist_behead(CSList *);
NSAPI_PUBLIC int cslist_do_free(CSList *, int);
NSAPI_PUBLIC int cslist_do_reset(CSList *, int);
NSAPI_PUBLIC int cslist_count(CSList *);
NSAPI_PUBLIC void cslist_apply(CSList *l, void (*fn)(void *));
NSAPI_PUBLIC void *cslist_copy_nop(void *);
NSAPI_PUBLIC void cslist_free_nop(void *);
NSAPI_PUBLIC int cslist_cmp_nop(void *, void *);
NSAPI_PUBLIC int cslist_create_with_nop(CSList **);
#define cslist_insert(l, d) 		cslist_append((l), (d))
#define cslist_free(l)			cslist_do_free((l), 1)
#define cslist_reset(l)			cslist_do_reset((l), 1)
#define cslist_free_readonly(l)	cslist_do_free((l), 0)
/* the (l) might be null, etsal */
#define cslist_head(l)			((l)? (l)->head : NULL)

/* CSHashTable -- static hashing with singularly linked-list chaining */
NSAPI_PUBLIC int cshashtable_create(int, CSHashTable **);
NSAPI_PUBLIC int cshashtable_set_hash(CSHashTable *, CSHashFn);
NSAPI_PUBLIC int cshashtable_set_copy(CSHashTable *, CSCopyFn);
NSAPI_PUBLIC int cshashtable_set_compare(CSHashTable *, CSCompareFn);
NSAPI_PUBLIC int cshashtable_set_free(CSHashTable *, CSFreeFn);
NSAPI_PUBLIC int cshashtable_insert(CSHashTable *, void *, void *);
NSAPI_PUBLIC int cshashtable_insert_unique(CSHashTable *, void *, void *);
NSAPI_PUBLIC int cshashtable_delete(CSHashTable *, void *);
NSAPI_PUBLIC void *cshashtable_find(CSHashTable *, void *);
NSAPI_PUBLIC void *cshashtable_findhv(CSHashTable *, void *, unsigned int);
NSAPI_PUBLIC int cshashtable_free(CSHashTable *);
NSAPI_PUBLIC void *cshashtable_iterate(CSHashTable *, void *, void *);
NSAPI_PUBLIC void cshashtable_stats(CSHashTable *, CSHashTableStats *);
NSAPI_PUBLIC unsigned int cshashtable_hf_default(void *);
#define cshashtable_hashsize(ht)	(ht)->hashsize
#define cshashtable_bucket(ht,i)	(ht)->buckets[i]

/* CSStack -- dynamic push/pop stack; read-only values */
NSAPI_PUBLIC int csstack_create(CSStack **);
NSAPI_PUBLIC int csstack_push(CSStack *, void *);
NSAPI_PUBLIC void *csstack_pop(CSStack *);
NSAPI_PUBLIC int csstack_free(CSStack *);

/* CSQueue -- dynamic FIFO queue; read-only values */
NSAPI_PUBLIC int csqueue_create(CSQueue **);
NSAPI_PUBLIC int csqueue_put(CSQueue *, void *);
NSAPI_PUBLIC int csqueue_delete(CSQueue *, void *);
NSAPI_PUBLIC void *csqueue_get(CSQueue *);
NSAPI_PUBLIC void *csqueue_peek(CSQueue *);
NSAPI_PUBLIC int csqueue_free(CSQueue *);

/* CSPriQ -- dynamic FIFO + priority level queue */
NSAPI_PUBLIC int cspriq_create(CSPriQ **);
NSAPI_PUBLIC int cspriq_put(CSPriQ *, void *, int);
NSAPI_PUBLIC void *cspriq_get(CSPriQ *);
NSAPI_PUBLIC void *cspriq_getpri(CSPriQ *, int);
NSAPI_PUBLIC int cspriq_free(CSPriQ *);

/* CSIO -- routines for I/O buffers/accumulators: handles both file/memory */
NSAPI_PUBLIC void csio_create(CSIO **);
NSAPI_PUBLIC void csio_increase(CSIO *, size_t);
NSAPI_PUBLIC void csio_destroy(CSIO *);
NSAPI_PUBLIC void csio_reset(CSIO *);
NSAPI_PUBLIC int csio_copy(CSIO *, char *, size_t);
NSAPI_PUBLIC void csio_convert2file(CSIO *);
NSAPI_PUBLIC void csio_dup2file(CSIO *);
NSAPI_PUBLIC void csio_rewind(CSIO *);
NSAPI_PUBLIC int csio_append(CSIO *, char *, size_t);
NSAPI_PUBLIC int csio_releasefd(CSIO *);
NSAPI_PUBLIC int csio_finish(CSIO *);
#define csio_appendstr(x,y) csio_append((x), (y), strlen(y))
#define csio_flush(x) csio_append((x), (char *)0, (int)0)

#ifdef __cplusplus
};
#endif

#endif
