package PDM::ConfigFile;
#<copyright>
# ----------------------------------------------------------
# Sun Proprietary/Confidential Code
# Copyright 2001, Sun Microsystems, Inc. All rights reserved.
# ----------------------------------------------------------
#</copyright>

#  $Name:  $ 
#  $Id: ConfigFile.pm,v 1.26 2003/12/18 07:12:18 ccadieux Exp $

use strict;
use System;
use Debug;
sub revision {'$Revision: 1.26 $'}
use vars qw($CONFIG $ERR);

#  INTERFACE:

#  ($renv, $hosts, $devices, $notifs, $Config) = PDM::ConfigFile->read();
#  $c->getSlaveConfig : used by http to return a custom config to a slave.
#  $c->read           : used by http and rasagent to get their configfile.
#  $c->write          : used by http to update it's configFile.
#

# get master config and change local slave config.
# return undef on failure

sub configFromMaster {
   my($class) = @_;
   my $F = System->get_home() . "/DATA/MASTER.time";
   my($master) = Util->findMaster();
   $ERR = undef;
   return 1 if (!$master);
   require Util::Http;

   #my $current = (stat(System->get_home() . "/DATA/rasagent.conf"))[9];

   my ($err, $age) =  Util::Http->fileAge($master, "rasagent.conf");
   if ($err) {
      Debug->errNoRepeat(MASTER_CONFIG => $master, 12);
      $ERR = "Cannot read Master config $master";
      return undef;
   }
   my $last_age;
   if (open(O, $F)) {
      $last_age = <O>; close(O);
   }
   if ($age > $last_age) {  # update slave
      my ($err, $out) = Util::Http->readFile($master, "rasagent.conf");
      if ($err || length($out) < 50) {
        $ERR = "Error reading rasagent.conf: $err";
        return undef;
      }
      my $local_config = $class->getSlaveConfig(System->hostname(), 0, $out);         
      if ($local_config && open(W, ">" . System->get_home() . "/DATA/rasagent.conf")) {
        print W $local_config;
        close(W);
        open(O, ">$F"); print O $age; close(O);
      } else {
        $ERR = $local_config ? "Error:Cannot open local rasagent.conf" : "Error:Slave not found in master config";
        return undef;
      }
   }
   return 1;
}



#
# returns a slave's config-file : STRING
# used also by master http
# force_local is for testing where a SLAVE and a MASTER on the same machine.
#
sub getSlaveConfig {
  my($this, $hostid, $force_local, $master_string) = @_;
  my($config_renv, $config_hosts, $config, $renv, $devices, $hosts, $notifs);
  my($master, $local_ip);

  if ($master_string) {  # running on a slave
    ($renv, $devices, $hosts,$notifs) = $this->readFromString($master_string);
    $master   = $renv->{hostname};
    $local_ip = Util->findMaster();

  } else {               # running on the master
    ($renv, $devices, $hosts,$notifs) = $this->read();
    $master   = $renv->{hostname};
    $local_ip = Util->name2ip($master);
  }

  my ($name, $local_host, $hcnt, $ipflag, $roleflag);
  $hcnt = "host0";

  foreach my $h (@$hosts) {
    if ($h->{hostname} eq $hostid || ($hostid eq $h->{hostid}) ) {
      if (!exists $h->{solution}) {
        $h->{solution} = "N";
        $h->{solution_model} = "";
      }
      $h->{role}     = "M" if (!exists $h->{role});
      $name = $h->{hostname};
      foreach my $k (sort keys %$h) {
         next if (substr($k,0,5) eq '_name');
         if ($k eq "ipno") {
            $local_host .= "ipno=$local_ip\n";
            $ipflag = 1;
         } else {
            $local_host .= "$k=$renv->{$k}\n" if (exists $renv->{$k}) ; # exchange 
         }
         $renv->{$k} = $h->{$k};
      }
    } else {
      $config_hosts .= "\n[$h->{_name}]\n";
      $hcnt = $h->{_name};
      foreach my $k (sort keys %$h) {
         next if ($k eq "_name");
         $config_hosts .= "$k=$h->{$k}\n";
      }
    }
  }
  $hcnt++;
  $local_host   .= "ipno=$local_ip\n" if (!$ipflag);
  $config_hosts .= "\n[$hcnt]\n$local_host\n";

  return undef if (!$name);
  foreach my $x (sort keys %$renv) {
     $config_renv .= "$x=$renv->{$x}\n";
  }

  foreach my $d (@$devices) {
    $d->{datahost} = $d->{host} if (!$d->{datahost});

#   if ($d->{host} eq $name || ($d->{datahost} eq $name) ) {
    $d->{host} = $master if (!$d->{host});

    $config .= "\n[$d->{_name}]\n";
    foreach my $x (sort keys %$d) {
      next if ($x eq "_name");
      $config .= "$x=$d->{$x}\n";
    }
#   }
  }

  foreach my $not (@$notifs) {
    $config .= "\n[$not->{_name}]\n";
    foreach my $x (sort keys %$not) {
      next if ($x eq "_name");
      $config .= "$x=$not->{$x}\n";
    }
  }

  return $config_renv . $config_hosts . $config;
}

  

sub readFromString {
  my($cfg, $string) = @_;

  my @F = split(/\n/, $string);
  return &_parse_config(\@F, wantarray);
}
 
#
#  returns the  proper configfile
#  used by master-http and all rasagents
#  PDM::ConfigFile->read("bardog.conf");
#
sub read0 {
  my($cfg, $file) = @_;
  $CONFIG = undef;
  return $cfg->read($file);
}
 
sub read {
  my($cfg, $file) = @_;
  my($renv, $devices, @F, $hosts) ;
  $ERR = "";
  if ($CONFIG && !wantarray) {
     return $CONFIG;
  }
  my($err, $config) = $cfg->_request_config($file);
  if (!$err) {
    @F = split(/\n/, $config);
    return &_parse_config(\@F, wantarray);
  }
  $ERR = $err;
  return ();
}

#
#   GET CONFIG FILE
#     If you are a slave , read the pushed configfile or get a new one from the master
#     else, get the complete local one
#
sub _request_config {
  my($class, $file) = @_;
  my($use_config, $local, $localC);
  my $renv = System->get_renv();

  $file = "rasagent.conf" if (!$file);

  $use_config = 1 if ($file eq "rasagent.conf");

  if ($renv->{portal} && $Http::HTTPQ->{COOKIE_HID}) {
    $use_config = 1;
    $local  = System->get_home() . "/NSCC.DATA/S" . $Http::HTTPQ->{COOKIE_HID} . "/rasagent.conf";
  } else {
    $local  = System->get_home() . "/DATA/$file";
  }
  $localC = System->get_home() . "/System/config";
  my($push) =  $local;
  my($config, $l);

  my($slave) = Util->findMaster();

  if ($use_config && open(O, $localC)) { # read system config
     while ($l = <O>) {
        $config .= "STATIC.$l";
     }
     close(O);
  }
  if (open(O, $local)) {  # read current config
     while ($l = <O>) {
        $config .= $l;
     }
     close(O);
     return ("", $config);
  } else {
     my($err) = $!;
     Debug->errNoRepeat(CONFIG_READ => undef, 8,"rasagent.conf: $err");
     return ("Cannot read $file: $err", undef);
  }
}


sub _parse_config {
  my($F, $want) = @_;
  my($x, $section, $k);
  my(%renv, @devices, %last, @hosts, $host);
  my(@notifs, $notif, %HIP);
  $section = -1;
  $host = -1;
  $notif = -1;
  my $in = "";
  for ($x=0; $x <= $#$F; $x++) {
      my $l = $F->[$x];
      next if (substr($l,0,1) eq "#" || $l =~ /^ *$/);
      if ($l =~ /\[(.*)\]/) {
        my $n = $1;
        if (substr($n,0,1) eq "h") {
           $host++;
           $hosts[$host]{_name} = $n;
           $in = "h";
        } elsif (substr($n,0,1) eq "n") {
           $notif++;
           $notifs[$notif]{_name} = $n;
           $in = "n";
        } else {
           $section++;
           $devices[$section]{_name} = $n;
           $in = "s";
        }
      } elsif ($l =~ /^([\w\.]+)= *(.*)$/) {
        if ($in eq "") {
          $renv{$1} = $2;

        } elsif ($in eq "n") {
          $notifs[$notif]{$1} = $2;

        } elsif ($in eq "h") {
          my $k = $1; my $val = $2;
          $hosts[$host]{$k} = $val;
          if ($k eq "ipno") {
             $HIP{$hosts[$host]{hostname}} = $val;
          }
        } else {
          my $k = $1; my $val = $2;
          if ($devices[$section]{$k}) {
             $last{"$section $k"}++;
             $k = $1 . $last{"$section $k"};
             $devices[$section]{$k} = $val;
          } else {
             $devices[$section]{$k} = $val;
             $last{"$section $k"} = 1;
          }
        }
      }
  }
  foreach $k (keys %renv) {
     if ($k =~ /STATIC\.(.*)/) {
        $renv{$1} = $renv{$k};
        delete $renv{$k};
     }
  } 
  for ($x=0;$x <= $#devices; $x++) {
     if ($devices[$x]{host}) {
        $devices[$x]{hostIpno} = $HIP{$devices[$x]{host}};
     }
     bless($devices[$x], 'ConfigDev');
  }
  # solution = N,se,se2
  my $se_slave = 0;
  for ($x=0;$x <= $#hosts ; $x++) {
     $hosts[$x]{ip} =  $hosts[$x]{hostname} if (!$hosts[$x]{ip});
     if (!exists($hosts[$x]{solution})) {
       $hosts[$x]{solution} = "N" ;
     } elsif ($hosts[$x]{solution} eq "Y") {
       $hosts[$x]{solution} = "se" ;
     }
     $se_slave = 1 if ($hosts[$x]{solution} eq "se");
     bless($hosts[$x], 'ConfigHost');
  }
  $renv{se_slave} = $se_slave;

  if (!exists($renv{solution})) {
    $renv{solution} = "N";
  } elsif ($renv{solution} eq "Y") {
    $renv{solution} = "se";
  }

  my $C = { renv => \%renv, devices => \@devices, hosts => \@hosts, notifs => \@notifs };

  bless($C, 'PDM::ConfigFile');
  $CONFIG = $C;
  if ($want) {
     return (\%renv, \@devices, \@hosts,  \@notifs, $C);
  } else {
     return $C;
  }
}

# add a device to a configfile, use hostname if present.

sub addDevice0 {
  my($Config, $d, $hostname) = @_;
  if (!$Config->deviceByKey($d->{key})) {
     my($devs) = $Config->devices();
     my $x = $#$devs+1;
     foreach my $v (keys %$d) {
       $devs->[$x]{$v} = $d->{$v} if ($v ne "_name");
     }
     $devs->[$x]{host} = $hostname if ($hostname);
     return 1;
  }
  return undef;
  
}


sub updateDevice {
  my($class, $device) = @_;
  my $lease;
  require Lease;
  if (!($lease = Lease->get("rasagent_conf", 5, 6)) ) {
    Debug->print2("cannot lease rasagent_conf");
    return undef;
  }   
  my $Conf = PDM::ConfigFile->read0();
  my $dev  = $Conf->deviceByKey($device->{key});
  if ($dev) {
    foreach my $e (keys %$device) {
       next if (substr($e,0,1) eq "_");
       $dev->{$e} = $device->{$e};
    }
    $Conf->save();
  }
  $lease->release();
}
  
sub addDevice {
  my($cfg, $devs, $new) = @_;

  my $cnt = $#$devs+1;
  $new->{_name}      = "device$cnt";
  $new->{time_added} = Util->get_today();
  push(@$devs, $new);
  return $devs->[$#$devs];
}



# returns a list of categories with at least one device

sub findActiveCategories {
  my($C, $host) = @_;
  my (%CAT);
  my $devs = $C->devices();
  my $renv = $C->renv();
  $host = $renv->{hostname} if (!$host);

  foreach my $d (@$devs) {
      my $h = $d->{host} || $renv->{hostname};
      next if ($h ne $host);
      if ( $d->{userLabel} eq "Switch1") {  # for 1G FCSW-2 switch only
         $CAT{switch}=1;
      } else {
         $CAT{$d->{type}} = 1;
      }
  }
  my $list;
  foreach my $el (keys %CAT) {
     $list .= "$el|";
  }
  return $list;
}

sub myDevices {
  my($c) = @_;
  my(@L);
  my($devs) = $c->devices();
  foreach my $d (@$devs) {
     if (Util->isMineToMonitor($d)) {
        push(@L, $d);
     }
  }
  return \@L;
}

sub adminGroups {
  my($c) = @_;
  my %H;
  my $hosts = $c->{hosts};
  my $renv  = $c->{renv};
  my $devs  = $c->devices();
  my $type  = $renv->{solution} eq "N" ? "host" : "sp";

  foreach my $h (@$hosts) {
    $H{"$type:$h->{hostname}"} = $h->{adm_group} if ($h->{adm_group});
  }
  $H{"$type:$renv->{hostname}"} = $renv->{adm_group} if ($renv->{adm_group});

  foreach my $d (@$devs) {
     $H{$d->{type} . ":" . $d->{key}} = $d->{adm_group} if ($d->{adm_group});
  }
  return \%H;
}


sub notifs {
  my($c) = @_;
  return $c->{notifs};
}

sub hostsList {
  my($c, $arg) = @_;
  my @L;
  my $include_ip = $arg->{include_ip};
  my $renv = $c->{renv};
  my $hosts = $c->{hosts};
  if ($include_ip) {
    push(@L, [$renv->{hostname}, $renv->{ipno} || $renv->{ip}] );
  } else {
    push(@L, $renv->{hostname});
  }
  foreach my $h (@$hosts) {
    if ($include_ip) {
      push(@L, [$h->{hostname}, $h->{ipno} || $h->{ip}] );
    } else {
      push(@L, $h->{hostname});
    }
  }
  return \@L;
}

# generate sorted <option>
#
sub slaveSelect {
  my($Config, $current) = @_;
  my $hosts = $Config->{hosts};
  my ($options, @H);
  foreach my $h (@$hosts) {
     next if($h->{dh} eq "Y");
     push(@H, $h->{hostname});
  }
  foreach my $h (sort @H) {
    my $ck = ($h eq $current) ? "selected" : "";
    $options .= "<option $ck>$h</option>";
  }
  return $options;
}
  


sub hosts {
  my($c) = @_;
  return $c->{hosts};
}

sub devices {
  my($c) = @_;
  return $c->{devices};
}
 
sub renv {
  my($c, $val) = @_;

  if ($val) {
    return $c->{renv}{$val};
  } else {
    return $c->{renv};
  }
}

# return ipno of a hostname, in hosts, or ipName of a device
sub findIpNo {
  my($Config, $host) = @_;
  
  my $devs  = $Config->{devices};
  my $hosts = $Config->{hosts};
  foreach my $h (@$hosts) {
      return $h->{ipno} if ($h->{hostname} eq $host);
  }
  foreach my $d (@$devs) {
      return $d->{ipno} if ($d->{ipName} eq $host);
  }
  return undef;
}
  

  

sub deviceByWWN {
  my($c, $wwn, $last, $use_wwns) = @_;

  my $devs = $c->{devices};
  foreach my $d (@$devs) {
     if ($last) {
       if (substr($d->{wwn}, -$last)  eq substr($wwn, -$last)) {
          return wantarray? ($d, 0) : $d;
       } elsif ($d->{wwn2} && 
          substr($d->{wwn2}, -$last) eq substr($wwn, -$last)) {
          return wantarray? ($d, 1) : $d;
       } 
     } else {
       if ($use_wwns && $d->{wwns} && index($d->{wwns}, $wwn) >= 0) {
          my(@L) = split(/[\|,]/, $d->{wwns});
          my($x);
          for ($x=0; $x <= $#L; $x++) {
             if ($L[$x] eq $wwn) {
                return wantarray? ($d, $x) : $d;
             }
          }
          return wantarray? () : undef;

       } elsif ($d->{wwn} eq $wwn) {
          return wantarray? ($d, 0) : $d;
       } elsif ($d->{wwn2} && $d->{wwn2} eq $wwn) {
          return wantarray? ($d, 1) : $d;
       }
     }
  }
  return undef;
}

sub hostByIpNo {
  my($c, $key) = @_;

  if (substr($key,0,5) eq "host:") {
    $key = substr($key,5);
  } elsif (substr($key,0,3) eq "sp:") {
    $key = substr($key,3);
  }
  my $renv = $c->{renv};
  my $hosts = $c->{hosts};
  if (Util->name2ip()  eq $key) {
    return { hostname => $renv->{hostname},
             ip       => $renv->{ip},
             ipno     => $renv->{ipno},
             solution => $renv->{solution},
             solution_model => $renv->{solution_model},
             role     => $renv->{role}
           };
  }
  foreach my $h (@$hosts) {
     if ($h->{ipno} eq $key) {
         return $h;
     }
  }
  return undef;
}
sub hostByKey {
  my($c, $key) = @_;

  if (substr($key,0,5) eq "host:") {
    $key = substr($key,5);
  } elsif (substr($key,0,3) eq "sp:") {
    $key = substr($key,3);
  }
  my $renv = $c->{renv};
  my $hosts = $c->{hosts};
  if (lc($renv->{hostname}) eq lc($key)) {
    return { hostname => $renv->{hostname},
             ip       => $renv->{ip},  
             ipno     => $renv->{ipno},
             solution => $renv->{solution},
             solution_model => $renv->{solution_model},
             role     => $renv->{role}
           };
  }
  foreach my $h (@$hosts) {
     if (lc($h->{hostname}) eq lc($key)) {
         return $h;
     } elsif ($h->{ipno} && ($h->{ipno} eq $key)) {
         return $h;
     }
  }
  return undef;
}


sub deviceByType {
  my($c, $type) = @_;
  my $devs = $c->{devices};
  foreach my $d (@$devs) {
     return $d if ($d->{type} eq $type);
  }
  return undef;
}


sub deviceByKey {
  my($c, $key) = @_;

  my $ix = index($key, ":");
  if ($ix > 0) {
     $key = substr($key, $ix+1);
  }
  my $devs = $c->{devices};
  foreach my $d (@$devs) {
     my $k = $d->{key} || $d->{wwn};
     if ($k eq $key) {
        return $d;
     }
  }
  return undef;
}

# return all device types present
sub typeList {
  my($c) = @_;
  my (%T, $list);
  my $devs = $c->{devices};
  foreach my $d (@$devs) {
     $T{$d->{type}} = 1;
  }
  foreach my $d (sort keys %T) {
     $list .= "$d=" . Util->abb("$d.medium") . "|";
  }
  return $list;
}

sub deviceByName {
  my($c, $name) = @_;

  my $devs = $c->{devices};
  foreach my $d (@$devs) {
     if ($d->{name} eq $name) {
        return $d;
     }
  }
  return undef;
}

sub deviceByIP {
  my($c, $name, $key) = @_;

  $key = "ipno" if (!$key);

  my $devs = $c->{devices};
  foreach my $d (@$devs) {
     if ($d->{$key} eq $name) {
        return $d;
     }
  }
  return undef;
}


# hosts and devices

sub deviceHash {
  my($c) = @_;
  my (%H);

  my $renv = $c->{renv};
  my $hosts = $c->{hosts};

  $H{$renv->{hostname}} = $c->hostAsDevice($renv);

  foreach my $h (@$hosts) {
    $H{$h->{hostname}} =  $c->hostAsDevice($h);
  }
  my $devs  = $c->{devices};
  foreach my $d (@$devs) {
     $H{$d->{key}} = $d;
  }
  return \%H;
}

sub hostAsDevice {
  my($c, $h) = @_;
  my $renv = System->get_renv();
  my $type = $renv->{solution} eq "N" ? "host" : "sp";

  return { 
     type => $type,
     name => $h->{hostname}, 
      key => $h->{hostname},
     ipno => $h->{ipno}, 
   active => "Y",
     ip   => $h->{ip},    
     dh   => $h->{dh},
        };
}


sub hostByName {
  my($c, $name) = @_;

  my $hosts = $c->{hosts};
  foreach my $h (@$hosts) {
     if ($h->{hostname} eq $name) {
        return $h;
     }
  }
  return undef;
}

# return all devices monitored by a host
# blank host mean local devices

sub devicesByHost {
  my($c, $host) = @_;
  my(@L);
  my $local = $c->{renv}{hostname};
  $host = $local if (!$host);

  my $devs = $c->{devices};
  foreach my $d (@$devs) {
      my $dh = $d->{host} || $local;
      if ($dh eq $host) {
         push(@L, $d);
      }
  }
  return \@L;
}

sub save {
  my ($config, $no) = @_;

  return undef if ($no);
  $config->write($config->renv(), $config->devices(), $config->hosts(), 
                 $config->notifs());
}


sub write {
 my($class, $renv, $devices, $hosts, $notifs) = @_;
 my($file, $cnt, $x, $y);
 $file = System->get_home() . "/DATA/rasagent.conf";
 $CONFIG = undef;
 if (!open(O,">$file")) {
   return undef;
 } else {
   System->set_renv($renv);
   foreach $x (sort keys %$renv) {
     print O "$x=$renv->{$x}\n";
   }
   $cnt = 1;
   foreach $x (@$hosts) {
     next if ($x->{_name} eq "");
     print O "\n[host$cnt]\n";
     $cnt++;
     foreach my $y (sort keys %$x) {
        next if ($y eq "_name");
        print O "$y=$x->{$y}\n";
     }
   }
   $cnt = 1;
   foreach $x (@$devices) {
     next if ($x->{_name} eq "");
     print O "\n[device$cnt]\n";
     $cnt++;
     foreach $y (sort keys %$x) {
        next if ($y eq "_name");
        print O "$y=$x->{$y}\n";
     }
   }

   $cnt = 1;
   foreach $x (@$notifs) {
     next if ($x->{_name} eq "");
     print O "\n[notification$cnt]\n";
     $cnt++;
     foreach $y (sort keys %$x) {
        next if ($y eq "_name");
        print O "$y=$x->{$y}\n";
     }
   }
   close(O);
   return 1;
 }
}

package ConfigDev;

sub new {
  my($class, $arg) = @_;
  my $obj = {};
  foreach my $el (keys %$arg) {
     $obj->{$el} = $arg->{$el};
  }
  bless($obj, "ConfigDev");
  return $obj;
}

sub key {
  my($dev) = @_;

  return $dev->{key} || $dev->{wwn};
}

sub topoKey {
  my($dev) = @_;

  my $k = $dev->{key} || $dev->{wwn};
  return $dev->{type} . ":$k";

}
sub hostIpno {
  my($dev) = @_;
  return $dev->{hostIpno};
}

sub isInSolution {
  my($dev) = @_;

  my($renv)    = System->get_renv();
  my $IP = $renv->{solution_prefix} || "XX";

  return (substr($dev->{ipno},0,length($IP) ) eq $IP);
}

#
# this device is the responsibility of the current host
#
sub isMineToMonitor {
  my($dev) = @_;
  my($renv)    = System->get_renv();

  my $hostname = $dev->{host} || $renv->{hostname};

  return ($hostname eq $renv->{hostname});
}



package ConfigHost;




1;


#
# last lines of the module...
#
# 1;
#


__END__

=head1 NAME

PDM:ConfigFile - ConfigFile acess class


=head2 SYNOPSIS

 use PDM::ConfigFile

 
  $c->getSlaveConfig : used by http to return a custom config to a slave.
  my($renv, $devices, $hosts, $notifs) = $c->read();
  $c->write(($renv, $devices, $hosts, $notifs);



=head2 DESCRIPTION

This module read and write a configfile into a set of hashes and arrays.


B<hostid> - HostId information

=back 4

=head2 METHODS

=over 4

=item read()

Read a configFile


=item write($renv, $devices, $hosts, $notifs)

Write back a config File

=back 4

=head2 AUTHOR

 Christian Cadieux



=head2 COPYRIGHT

Copyright SUN Microsystems


=cut
