package PDM;
#<copyright>
# ----------------------------------------------------------
# Sun Proprietary/Confidential Code
# Copyright 2001, Sun Microsystems, Inc. All rights reserved.
# ----------------------------------------------------------
#</copyright>

#  $Name:  $ 
#  $Id: PDM.pm,v 1.40 2004/01/12 22:27:35 ccadieux Exp $

use strict;
use MaxEvents;
use Thresholds;
use Carp;
use Report;
use Debug;
use Util::Http;
use PDM::ConfigFile;
use System;
use Map;
use State;
use RasDB;
use LockManager;

#
#  Interface:
#   new
#   serialize : called by the displatcher.
#   saveReport:
#   getDocSequence:
#   getLastReport:
#   saveReport:
#   saveMessage:
#   getCacheHandle;
#   inCache :  boolean
#      
use vars qw(@EVENT_DOCS $Error %V @Reports_CB $CACHE %GLOBAL $TOT_DISCO $ED $ED_SEQ $EVENTS);

$Error      = 0;
%V          = ();
@Reports_CB = ();

sub error {
  return $Error;
}


#  my($pdm) = PDM->new({  dir => "$HOME/DATA",
#                      renv => $renv ,
#                   devices => $devices,
#                     hosts => $hosts,
#                    notifs => $notifs, http => 1});

sub new {
  my($class, $arg) = @_;
  my( $N, %T);
  my($pdm) = $arg;
  my $renv = System->get_renv();
  require NWS::Schema;

  my $D = System->get_home() . "/DATA";

  $pdm->{dir} = System->get_home() . "/DATA" if (!$pdm->{dir});

  if ($arg->{http}) {
    $class->readAllCache();
 
  } else {
#    $N = "$D/OLD_REPORTS";
#    $OLD_REPORTS = &_read($N, ":");

    $class->readAllCache();
    State->init(); 
    NWS::Schema->init();
  }

  my $master = Util->findMaster();  # erase master catalog if config was pushed recently.
  if ($master) {
    my $config_data_age  = (stat(System->get_home() . "/DATA/rasagent.conf"))[9];

    TO->resetIncomplete();
  }

  my($devs) = System->get_configDevices();
  foreach my $d (@$devs) {
     next if (!Util->isMineToMonitor($d));
     $T{lc($d->{type})}{$d->{key}} = 1;
  }
  if (!$renv->{keep_old_reports}) {
    $D = System->get_home() . "/DATA/OLD_REPORTS";
    #
    # REMOVE OLD_REPORTS
    #
    opendir(O1, $D) ;
    my(@a) = readdir(O1); closedir(O1);
    my($VAR1);
    foreach my $f (@a) {
       next if ($f !~ /.+:.+/);
       open(O1, "$D/$f");
       my(@b) = <O1>; close(O1);
       eval "@b";
       my($name) = $VAR1->{_id}{deviceName};
       my($type) = lc($VAR1->{_id}{category});
       my $type0 = $type; $type0 =~ s/message$//;
       my $rack = ",se,se2" if ($renv->{solution} ne "N");
       my $rack = $renv->{solution} eq "N" ? ",host," : ",se,se2,sp,";

       if ( (index("$rack,message,san,topo,datahost,master,backup,", ",$type,") < 0) &&
             $name && $type && !$T{$type}{$name} && !$T{$type0}{$name}) {
         unlink("$D/$f");
         State->clearComponent($f);
         Debug->print2("Report $f not in config file, removed.");
       }
    }
    my $hosts = System->get_configHosts();
    #
    # REMOVE OLD REMOTE_LISTS
    #
    $D = System->get_home() . "/DATA/REMOTE_LISTS";
    opendir(O1, $D) ;
    my @REM = readdir(O1); closedir(O1);
    foreach my $f (@REM) {
       next if (substr($f,0,1) eq ".");
       my $found;
       foreach my $h (@$hosts) {
           if ($h->{hostname} eq $f) {
              $found = 1; last;
           }
       }
       if (!$found && ($f ne $renv->{hostname}) ) {
         unlink "$D/$f";
         Debug->print2("$f removed from REMOTE_LISTS");
       }
    }
  }
  bless($pdm, "PDM");
  return $pdm;
}


sub serialize {
  my($pdm) = @_;

# Each reports has it's own file in the OLD_REPORTS dir.

  $DB::single = 1;
  $pdm->_write($CACHE, "CACHE");
  State->cleanState();
  State->write(1);
}

sub _write {
  my($pdm, $x, $dir) = @_;
  require Data::Dumper;
  my($k);
  my($f) = System->get_home() . "/DATA/$dir";
  mkdir $f, 0777 if (!-d $f);
  $Data::Dumper::Indent = 1;
  foreach $k (keys %$x) {
     open(O, ">$f/$k");
     print O Data::Dumper::Dumper($x->{$k});
     close(O);
  }
}


sub _read {
  my($dir, $pat) = @_;
  my(%O) = ();
  my($file, $v, $l1);
  my($f) = $dir;
  opendir(O, $f);
  my(@f) = readdir(O); closedir(O);

  foreach $file (@f) {
    next if (substr($file, 0,1) eq ".");
    next if ($pat && $file !~ /$pat/);
    open(O, "$f/$file");
    my($VAR1) = undef;
    my($l) = "";
    while ($l1 = <O>) { $l .= $l1}
    close(O);
    eval "$l";
    $O{$file} = $VAR1;
  }
  return \%O; 
    
}

sub writeAllCache {
  my($class) = @_;
  $class->_write($CACHE, "CACHE");
}

sub readAllCache {
  my($class) = @_;
  $CACHE = &_read( System->get_home() . "/DATA/CACHE");
}

sub readCache {
  my($class, $name) = @_;
  my($VAR1);
  my($file) = System->get_home() . "/DATA/CACHE/$name";
  if (open(O, $file)) {
    my(@L) = <O>; close(O);
    eval "@L";
    return $VAR1;
  } else {
    return undef;
  }
}

sub writeCache {
  my($class, $cache, $file) = @_;
  require Data::Dumper;
  open(O, ">" . System->get_home() . "/DATA/CACHE/$file");
  print O Data::Dumper::Dumper($cache);
  close(O);
}

sub writeCacheObject {
  my($class, $file) = @_;
  require Data::Dumper;
  open(O, ">" . System->get_home() . "/DATA/CACHE/$file");
  print O Data::Dumper::Dumper($CACHE->{$file});
  close(O);
}

use vars qw(%Stale);

sub setStaleCatalog {
  my($class ,$host) = @_;

  $Stale{$host} = 1;
}

sub getStaleCatalog {
  my($class) = @_;
  my(@a);
  foreach my $k (keys %Stale) {
    push(@a, $k);
  }
  return @a;
}

# change the key of a  component in the configfile

sub fixConfig {
  my($class, $type, $old, $new, $extra) = @_;
  my $rc;
  my $master  = Util->findMaster();
  if (!$master) {
     $rc = &get_fixConfig({ type => $type, old => $old, new => $new, extra => $extra});
  } else {
     Util::Http->runit($master, "/sbin/change_wwn -o $old -n $new", 20);
  }
  return $rc;
}

sub get_fixConfig {
  my($q) = @_;
  my $warning;
  my ($renv, $devs, $hosts, $notifs) = PDM::ConfigFile->read();
  foreach my $dev (@$devs) {
     if ($dev->{key} eq $q->{old}) {
       $dev->{wwn} = $dev->{key} = $q->{new};
       $warning .= "wwn of device $dev->{name}/$dev->{ipno} changed from $q->{old} to $q->{new}\n";
     }
  }
  if ($warning) {
    open(O, ">>" . System->get_home() . "/DATA/master_warning");
    print O $warning;
    close(O);
    PDM::ConfigFile->write($renv, $devs, $hosts, $notifs);
  }
  my $rc = $warning ? "OK" : "ERR No device found";
  if ($q->{HTTP}) {
     print "$rc\n";
  } else {
     return $rc;
  }
}
  


###########################################################
#  DEVICE STATE
###########################################################

# static method

sub getDeviceStateMap {
  my($pdm, $section) = @_;
  return Map->getDeviceStateMap($section);
}


###########################################################
#  REPORTS STUFF
###########################################################


sub getOldReport {
  my($pdm, $key) = @_;
  return Report->readNew($key);
}

#  name, category, ip, key_type/key_value, report_type, report_value => {};
# Save a new report.
# $arg = {skip_health => 1}

sub saveReport {
  my($pdm, $report) = @_;

# Only update reports if you got a good one.

  return if ($report->status eq Report->STATUS_DONT_SAVE);

  if ($report->status eq Report->STATUS_OK) {
      $report->saveNew();

  } else {  # update report
      $report->updateStatus(); # change the status of the last good report.
  }
  return;
}

sub saveReportNU {
  my($pdm,  $report, $arg) = @_;
  my($i, $hm, $filter, $cb, $rcb);
  $DB::single = 1;

  if ($arg->{skip_health}) {
    $pdm->_saveReport($report);
    return;
  }

# read all callback, check the filter, call if good.
  my %DONE;
  for ($i=0; $i <= $#Reports_CB; $i++) {  
      $rcb    = $Reports_CB[$i];
      $hm     = $rcb->[0];
      my($hm0, $rest) = split(/\=/, $hm);
      $cb     = $rcb->[1];
      $filter = $rcb->[2];
      next if ($DONE{"$hm0.$cb"});
      next if ($filter->{INITIAL} || $filter->{FINAL});
      next if ($filter->{category} && $filter->{category} ne $report->category);
      next if ($filter->{report_type} && $filter->{report_type} ne $report->get_report_type);
      $hm->$cb($report);  # call callback with this report.
      $DONE{"$hm0.$cb"} = 1;
  }
  $pdm->_saveReport($report);

}



###########################################################
#  EVENT STUFF
###########################################################
# 
# Executed on the slave, appends to the SLAVE_DATA file all new events found.
#
sub push {
  my($class, $master) = @_;

  my $renv = System->get_renv();
  return if (!$master);

  my $S = $class->getMessages({severityCount => 1});
  if ($S->[1] + $S->[2] + $S->[3] > 0) {
    my($err, $ans)  = Util::Http->saveFile($master, "Events/$renv->{hostname}.push",
                          "ALERTS FOUND", 20);
  }
}

# save locally in case the master is down, 
# send everything when the master is back
#
sub sendMessages {
  my($pdm, $MASTER) = @_;
  my($x, $ed);
  my $renv = System->get_renv();

  my $SD = System->get_home() . "/DATA/SLAVEDATA/$renv->{hostname}";

  if (!open(O3,  ">>$SD")) {
    Debug->err(CANNOT_READ => "SLAVE_DATA $!");
    return;
  }

  $ED = RasDB->new("EDOCS");
  my $hash = $ED->hash();
  $ED->ReadLock();
  foreach my $el (sort keys %$hash) {
     $ed = $hash->{$el};
     print O3 $ed->toC;
  }
  $ED->UnLock();
  close(O3);
       
  my $avail =  Util::Http->getCommand($MASTER, "INFO1", 10);

  if ($avail =~ /version/) {
     my($l, $out);
     if (open(O3, $SD)) {
        while ($l = <O3>) {
          $out .= $l;
        }
        if ($out) {
           Util::Http->appendFile($MASTER, "SLAVEDATA/$renv->{hostname}",
                        $out, 30);
        }
        close(O3);
        unlink $SD;
     }
  }
}

sub storeMessages {
  my($pdm, $skip_mgmt) = @_;
  my($x, $ed);
  my $renv = System->get_renv();

  my $F =  System->get_home() . "/DATA/SLAVE_DATA";
  if (open(O, ">>$F")) {
    $ED = RasDB->new("EDOCS");
    my $hash = $ED->hash();
    $ED->ReadLock();
    foreach my $el (sort keys %$hash) {
       $ed = $hash->{$el};
       my $ev = $ed->instances(0);
       if ($renv->{solution} ne "N") { # racks
          my $agg_list  = $ed->id("agg_list");
          my $agg       = $ev->value("Aggregate");
          my $desc      = $ev->value("Description");
          if ($agg > 0) {
             require EventDB;
             my($involved, $sub) = EventDB->sub_events($agg_list, 0);
             $ev->setValue("Data", $sub);
             $ev->setValue("Description", "$desc:") if (substr($desc,-1) ne ":");
          }
       }  
       next if ($skip_mgmt && $ev->value("MgmtLevel") eq "DS");
       print O $ed->toC;
    }
    $ED->UnLock();
    close(O);
    System->rotate($F, 10, 1);
  } else {
    Debug->err(CANNOT_READ => "SLAVE_DATA $!");
  }
}

#
# store toString as html file
#
sub docMessage {
  my($class, $edoc) = @_;
  my($etype, $desc);
  my($ev)   = $edoc->instances(0);
  my($cat)  = $edoc->id('category');
  my($renv) = System->get_renv;

  return if (Debug->level() <= 2 && $renv->{hostname} !~ /sun.com/);

  if ($edoc->type() eq 'E') {
    $etype = $ev->propertyByName('EventType')->value;
    $desc = $ev->propertyByName('Caption')->value;
    $desc = substr($desc,0,8) if (substr($etype,0,5) eq "host." && index($desc,".lun.") >= 0);
    $etype .= ".$desc" if ($desc);
    my $D;
    if (-d "/net/code/usr6/Events") {
       $D = "/net/code/usr6/Events/toc";
       open(O2, ">$D/$etype.toc");
       print O2 $edoc->toC();
       close(O2);
    }
    $D = System->get_home() . "/DATA/tos";
    if (-d $D) {
       open(O2, ">$D/$etype.toc");
       print O2 $edoc->toString();
       close(O2);
    }

  } else {
    $etype = $ev->className();
    $desc = $ev->propertyByName('AlertType')->value;
    $etype .= ".$desc" if ($desc);
  }
}

sub getDiscoveryCount {
   return $TOT_DISCO;
}

sub saveMessage {
  my($class, $edoc, $fromSlave, $ignoreMax) = @_;
  my($xx);
  if (!$edoc->instances(0)->type()) {
     Debug->err(CIM_ERR => "Event has no type");
     return;
  }
  $fromSlave = "local" if (!$fromSlave);
  croak("Invalid type for event: " . ref($edoc)) if (ref($edoc) ne "Message");
  $class->docMessage($edoc) ;
  $edoc->id(fromSlave => $fromSlave);

  my($ev) = $edoc->instances(0);
  my($time) = $ev->value("EventTime");
  my($desc) = $ev->value("Description");
  my $target= $ev->value('Target');
  my $sev   = int($ev->value("Severity")+0.5);
  my $caption = $ev->value('Caption');
  my $deviceName =  $edoc->id('deviceName');
  my $et    = $ev->value("EventType") ;
  $TOT_DISCO++ if ($et =~ /DiscoveryEvent/ || $et =~ /AuditEvent/ );
  my $code    = $ev->value("GridCode");
  my $grid_no = $ev->value("GridNo");
  my $action = $ev->value("Actionable");

  if ($action eq "TRUE") { # check if diags were running in last hour
     my $diags = PDM->getCacheHandle("last_diag_run");

     if (exists($diags->{$target}) && (time - $diags->{$target}) < 60 * 60) {
       $ev->setValue("Actionable", "FALSE");
       my $col = substr($desc, -1);
       $desc .= " (This event is now non-actionable because a diagnostic was running in the last hour)";
       $desc .= $col if ($col eq ":");
       $ev->setValue("Description", $desc);
     }
     my $lock = LockManager->new();
     my $locki = $lock->exists("system");
     if ($locki->{info}) {
       my($app, $name, $desc, $date, $expires, $ip) = split(/\|/, $locki->{info});
       my $col = substr($desc, -1);
       $desc .= " ($app action in progress [user:$name, start:$date, desc=$desc])";
       $desc .= $col if ($col eq ":");
       $ev->setValue("Description", $desc);
     }
  }
  if ($fromSlave eq "local") {
       my ($last, $last_email) = MaxEvents->maxEvents("events", $ev, "", 6, 8) if (!$ignoreMax);
       my $col = substr($desc,-1);
       $col = "" if ($col ne ":");
       if ($last_email) {
          $ev->setValue("Description", "$desc (last event for next 8 hours)$col");
        
       } elsif ($last) {
          Debug->print2("  ==>No event generated: Exceeded 6 events in 8 hours for $target/$caption");
          return;
       }
  }

  my $ev_name = $ev->value("EventType");
  my $ev_desc = $ev->value("Description");
  my $ev_sev  = $ev->value("Severity");
  my $ev_act  = $ev->value("Actionable");
  my $ev_cap  = $ev->value("Caption");
  Debug->print1("==> New Event: $ev_name: $ev_desc (sev=$ev_sev, act=$ev_act, cap=$ev_cap, etype=$et, no=$grid_no)");

  my $renv = System->get_renv();
  if ($renv->{event_trace_dir} && -d $renv->{event_trace_dir}) {
        my $et = $ev->value("EventType");
        my $out = $edoc->toC();
        my $ts = Util->get_today();
        $ts =~ s/[ \:]//g;
        if (open(OO, ">$renv->{event_trace_dir}/$ts.$et")) {
           print OO $out;
           close(OO);
        }
  }
  push(@EVENT_DOCS, { Severity => $sev,
                          Caption => $caption, 
                         GridCode => $code,
                        EventType => $et,
                           Target => $target,
                              Key => $deviceName,
                        Component => $ev->value("Component"),
                       });

  $ED = RasDB->new("EDOCS");
  my $hash = $ED->hash();
  my $seq = time . ".$$." . ++$ED_SEQ;
  $edoc->id("edoc_id", $seq);
  $hash->{$seq} = $edoc;
  return $seq;
}

#
# CALL saveState and log events
#
sub saveStateLog {
  my($pdm, $arg) = @_;
  require EventDB;

  $ED = RasDB->new("EDOCS");
  my $renv = System->get_renv();

  my $hash = $ED->hash();
  $ED->ReadLock();
  my (@LIST, @SEV);

  foreach my $el (sort keys %$hash) {
     my $ed        = $hash->{$el};
     my $edoc_id   = $ed->id("edoc_id");
     my $state     = $ed->state();
     my $ev        = $ed->instances(0);
     my $time      = $ev->value("EventTime");
     my $fromSlave = $ed->id('fromSlave');
     my $desc      = $ev->value("Description");
     my $target    = $ev->value('Target');
     my $sev       = int($ev->value("Severity")+0.5);
     my $caption   = $ev->value('Caption');
     my $deviceName= $ed->id('deviceName');
     my $et        = $ev->value("EventType") ;
     my @et0       = split(/\./, $et);
     my $code      = $ev->value("GridCode");
     my $action    = $ev->value("Actionable");
     my $agg_list  = $ed->id("agg_list");
     my $agg       = $ed->id("aggregate");
     my $solid     = $ev->value("SolutionId");
     my $parent    = $ev->value("Parent");
     my $mgmtLevel = $ev->value("MgmtLevel") ;
     next if ($mgmtLevel eq "DS");

     $agg          = -1 if ($parent);

     my $grid      = Grid->getInfo(undef, undef, {code => $code});

     # LOG EVENT
     $desc =~ s/\n/ /g;
     my $data = "$time\t$et\t$caption\t$desc" .  "\tSev:$sev" .
                "\tAction:$action" .
                "\tEnc:$deviceName"  .  
                "\tmgmtLevel:"  . $ev->value('MgmtLevel') .
                "\tAgentH:"  . $fromSlave  .  
                "\tAgg:"  . $agg .
                "\tECode:"   . $code ;

     if (substr($desc,-1) eq ":") {  # get more
       my $v1 = substr($ev->value("Data"), 0, 1000);
       my @d1 = split(/\n/, $v1);
       my ($xx);
       if ($#d1 >= 0) {
         $data .= "\n";
         for ($xx = 0; $xx <= $#d1 && $xx < 30; $xx++) {
           last if (index($d1[$xx] , "#INSTRUMENTATION_REPORT") >= 0);
           $data .= "L$d1[$xx]\n";
         }
       }
     }
     if ($solid && $renv->{solution} eq "N") {
       $pdm->writeTruncate(System->get_home() . "/DATA/EventsSys.log", $data);
     } else {
       $pdm->writeTruncate(System->get_home() . "/DATA/Events.log", $data);
     }

     # CALL State->saveState()
     my($key, $keyval);
     if ($state->[2]) {
       ($key, $keyval) = split(/\:/, $state->[2]);
     } else {
       ($key, $keyval) = split(/\:/, $target);
     }
     next if (index(",AuditEvent,DiscoveryEvent,Statistics,", ",$et0[1],") >= 0);
     next if ($et0[0] eq "agent" && $sev == 0);  # ship agent events

     State->saveState($key, $keyval, $state->[0], $deviceName, 
                      $ed->severity2(), 
                      $desc, $state->[1], $ev , 
		      {   absolute => $state->[3], 
		         aggregate => $agg,
	 		   service => $grid->{service} ? 1:0,
   	                  agg_list => $agg_list,
	              } );
     EventDB->write($ed);
  }

  $ED->UnLock();
}



#######################
#  EVENT_DOCS functions: these are not CIM events anymore
#######################

sub get_event {
  my($class, $no) = @_;
  return $EVENT_DOCS[$no];
}

sub get_events {
  my($class, $no, $no2) = @_;
  $no2 = $#EVENT_DOCS if (!defined($no2));

  my ($x,@list);

  for ($x=$no+1; $x <= $no2; $x++) {
     push(@list, $EVENT_DOCS[$x]);
  }
  return \@list;
}

sub get_last_event {
  my($class) = @_;
  return $#EVENT_DOCS;
}
###############################




sub setMessages {
  my($pdm, $MESSAGES) = @_;
  return;
  foreach my $m (@$MESSAGES) {
     #push(@EVENT_DOCS, $m);
  }
}


sub writeTruncate {
  my($pdm, $file, $data) = @_;
  if ($data) {
     if (open(O,">>$file")) {
       print O "$data\n";
       close(O);
     } else {
       Debug->err(CANNOT_WRITE => $file);
     }
  }
}


#  Request event-docs from the PDM, executed by providers.
#  $edocs = $pdm->getMessages({ ClassName => "className" });
#  $edocs = $pdm->getMessages({ ClassName => "className" });
#  $edocs = $pdm->getMessages({ ClassName => [<set of classes>] });
#  foreach $ed (@$edocs) {
#    ...
#
sub getMessages {
  my($pdm, $arg) = @_;

  $ED = RasDB->new("EDOCS");
  my $sevCnt = $arg->{severityCount};

  my $hash = $ED->hash();
  $ED->ReadLock();
  my (@LIST, @SEV);

  foreach my $el (sort keys %$hash) {
     my $ed = $hash->{$el};
     if ($sevCnt) {
       my $ev = $ed->instances(0);
       my $sev = $ev->value("Severity");
       $SEV[$sev]++;
     } else {
       push(@LIST, $ed);
     }
  }
  $ED->UnLock();
  if ($sevCnt) {
    return \@SEV;
  } else {
    return \@LIST;
  }
} 

sub getMessagesFromList {
  my($pdm, $idlist) = @_;

  $ED = RasDB->new("EDOCS");
  my $hash = $ED->hash();

  my (@LIST, @SEV);
  $ED->ReadLock();

  foreach my $el (@$idlist) {
     my $ed = $hash->{$el};
     push(@LIST, $ed);
  }
  $ED->UnLock();

  return \@LIST;
} 


sub deleteMessages {
  my($pdm) = @_;
  $ED = RasDB->new("EDOCS");
  my $hash = $ED->hash();

  $ED->Lock();
  foreach my $el (keys %$hash) {
    delete $hash->{$el};
  }
  $ED->UnLock();
}

sub getEventIdList {
  my($pdm) = @_;
  $ED = RasDB->new("EDOCS");
  my $hash = $ED->hash();
  $ED->ReadLock();
  my @L = (sort keys %$hash);
  $ED->UnLock();
  return \@L;
}

sub getEventCount {
  my($pdm) = @_;
  $ED = RasDB->new("EDOCS");
  my $hash = $ED->hash();
  $ED->ReadLock();
  my @L = (keys %$hash);
  $ED->UnLock();
  return $#L+1;
}

#
# Reports_CB = [health_monitor, $function, $filter]
# filter can be on category, report_type or 'INITIAL'
# INITIAL : will execute automatically at the request of the dispatcher.
#

sub reportRequest {
  my($pdm, $hm, $callback, $filter) = @_;

  push(@Reports_CB, [$hm, $callback, $filter]);

}



%GLOBAL= ();

sub getGlobal {
  my($pdm, $key) = @_;
  my($obj) = {};

  if (!defined($GLOBAL{$key})) {
    $GLOBAL{$key} = $obj;
    return $obj;
  } else {
    return $GLOBAL{$key};
  }
}

  
sub dumpCache {
  my($pdm) = @_;
  $CACHE = undef;
}

# always returns a handle to an object or a new empty object.
#
sub getCacheHandle {
  my($pdm, $key) = @_;
  my($obj) = {};

  if (!$CACHE) {
    my($D) = System->get_home() . "/DATA/CACHE";
    $CACHE = &_read($D);
  }

  if (!defined($CACHE->{$key})) {
    $CACHE->{$key} = $obj;
    return $obj;
  } else {
    return $CACHE->{$key};
  }
}

sub inCache {
  my($pdm, $key) = @_;
  if ($CACHE->{$key}) {
    return 1;
  } else {
    return 0;
  }
}

###########################################################
# MISC.
###########################################################
use vars qw($SEQ);


sub getEventSequence {
  my($pdm) = @_;
  my($s);
 
  $SEQ++;
  return time . "." . System->hostid() .  ".$$.$SEQ";
  
  my($file) = "EventSequence";
  if (open(O, System->get_home() . "/System/$file") ) {
    $s = <O>;
    $s++;
    close(O);
  } else {
    $s = 1;
  }
  open(O , ">" . System->get_home() . "/System/$file");
  print O $s;
  close(O);
  return System->hostid . "." . $s;
}



1;

__END__

=head1 NAME

PDM.pm - Persistent Data Manager


=head3 SYNOPSIS

 use PDM;

 $rep = PDM->new( {...} );


=head3 DESCRIPTION

This module include persistence functions.


=head3 CONSTRUCTOR

=over 1

=item new()


 $pdm => PDM->new( {dir => "cache dir", renv => , $devices => , hosts => , notifs => });

=back 1

=head3 METHODS

=over 4

=item serialize();

Serialize the PDM data stores, called by the dispatcher.

  $pdm->serialize();


=item saveReport();

Saves a report, called by instrumentation

=item getLastReport();

returns an old report.

=item initialHealth();

Execute initial health modules, created with the INITIAL flag.

=item finalHealth();

Execute final health modules, created with the FINAL flag.



=item saveMessage();

Store one Message in memory.

=item storeMessages();

Serialize to the disk all eventDocs, called by slaves dispatchers.

=item getMessages();

returns a list of eventDocs available.

=item reportRequest();

called at the creation of a health monitor to register a callback.

=item getCacheHandle();

Returns a handle to an object given a cache key. If the object does not
exist, it will create an empty one and return a handle so that the caller
can still use the handle to store stuff. This will be serialized by the
dispatcher at the end of the rasagent run.

  Example:
    $h1 = PDM->getCacheHandle('savethis');
    $h1->{'here'} = "some stuff";
    $h1->{'there'} = "more stuff";
    $h1->{'count'}++;

    # This information will be available on the next run of rasagent.>

=item inCache();

returns TRUE if the key has an object in the cache.

=item getTimer($timer);

Will initialize a timer or return the number of minutes since the last reset
of the timer. By default, the timer is 'heartbeat' unless 'timer' is defined.

=item resetTimer($timer);

Will reset a timer to zero. Called manually and by getTimer when no timer is
found.

=item getEventSequence();

Get a unique event sequence.

=item getDocSequence();

Get a unique Document sequence for a set of events.

=back 4

=head3 SEE ALSO




=head3 AUTHOR

 Christian Cadieux (ccadieux@central.sun)



=head3 COPYRIGHT

Copyright (c) 2000 Sun Microsystems

=cut
