package FSA;
use System;
use PDM;
use FSA::Indy;
use strict;
use Events;
use TO;
use Message;
use Grid;

use vars qw(%IN_AGGREGATE);

#      OLD          NEW        Agg.
#  ------------+-------------+------+
#  |    [C]    |    [C]      | [A]  |
#  +-----------+-------------+------+
#  will run all FSA rules
#  make old + [C] + [A] the current events
#  move new (non-C) event in old.
#
sub test {
   my (@LIST);

   push(@LIST, [ [ "E5" ], ["sw:c:0", "sw:x:2"]   ]);
   push(@LIST, [ [ "E1" ], ["sw:a:0", "sw:x:0"]   ]);
   push(@LIST, [ [ "E4" ], ["sw:b:1", "sw:y:1"]   ]);
   push(@LIST, [ [ "E2" ], ["sw:a:1", "sw:y:0"]   ]);
   push(@LIST, [ [ "E3" ], ["sw:b:0", "sw:x:1"]   ]);
   push(@LIST, [ [ "E6" ], ["sw:x"]            ]);
   push(@LIST, [ [ "E7" ], ["sw:y"]            ]);

   &merge(\@LIST);
   use Data::Dumper;
   print Dumper(\@LIST);
}

sub clean {
  my($new) = @_;
  my (@clean);
  foreach my $ed (@$new) {
      next if ($ed->type() eq Message::TYPE_ALERT);
      my $ev = $ed->instances(0);
      my $sev = $ev->value("Severity");
      next if (!$sev);  # warn and err only
      push(@clean, $ed);
  }
  return \@clean;
}
      

sub RUN {
  my($class, $TEST) = @_;
  my $renv = System->get_renv();
  if (!$renv->{fsa}) {
     Debug->print2("FSA is off.");
     return 1;
  }
  $DB::single = 1;

  my $topo =  TO->readExistingTopo("MERGE-MASTER") ||  TO->readExistingTopo();
  if (!$topo) {
     Debug->print2("FSA: Cannot find topo");
     return 1;
  }
  my($renv, $devs, $hosts, $notifs, $Config) = PDM::ConfigFile->read();

  Debug->print2("Running FSA...");
  my $OLDF = "previous_events";
  my $new;
  my $old = [];

  $new = PDM->getMessages();
  my $EX = ",AuditEvent,DiscoveryEvent,Statistics,backup,topoBackup,PatchInfo,TopologyEvent,LocationChange,";

  my @LIST;   # [ [event_list] , [ candidate_node_list] ]
  foreach my $ed (@$new) {
      next if ($ed->type() eq Message::TYPE_ALERT);
      my $ev  = $ed->instances(0);
      my $sev = $ev->value("Severity");
      my $e0  = $ev->value('EventType');
      my($cat, $eventType, $rest) = split(/\./, $e0);
      next if ($cat eq "agent");
      next if (!$sev);  # warn and err only
      #next if (index($EX, $eventType) >= 0);

      my $target0   =  $ev->value('Target');
      my($type, $target1, $port) = split(/\:/, $target0);
      my $mgmtLevel = $ev->value("MgmtLevel");
      my $solid = $ev->value("SolutionId");
      next if ($solid && $renv->{solution} eq "N"); # already processed.
      next if ($mgmtLevel eq "DS");                 # skip rack-level events for now.
      my $component =  $ev->value('Component');
      my $caption   =  $ev->value('Caption');
      my $target = "$type:$target1";
      my $ix = index($target0, ":");
      my $ix2 = rindex($target0, ":");

      my @indy = FSA::Indy->extra_keywords($cat, $eventType, $target, $ev);
      my @keywords = (@indy);

      my $sid = $ev->value("SolutionId");
      if ($eventType eq "CommunicationLostEvent") {
         if ($caption eq "ib") {
           push(@LIST, [ [ $ed ], [@keywords , $target] ]);
         } else {
           push(@LIST, [ [ $ed ], [@keywords, $target, "ETHERNET"] ]);
         }
      } elsif ($eventType =~ /^LinkEvent/) {
         push(@LIST, [ [ $ed ], [@keywords, "$target:$port", &nodeFrom($topo, "$target:$port")] ]);

      # logEvent that was identified as FC_PATH related
      } elsif ($eventType =~ /LogEvent/ && 
               ($caption eq "PATH" || substr($caption,0,7) eq "driver.") ) {
         push(@LIST, [ [ $ed ], [@keywords, $target, &nodesFrom($topo, $target)] ]);

      } elsif ($component =~ /^port.(\d+)$/ || $component =~ /^fcPort.(\d+)$/) {  # 
         my $port = $1;
         if ($type eq "host") {
            push(@LIST, [ [ $ed ], [@keywords, "$target:$port", &nodesFrom($topo, "$target:$port")] ]);
         } else {
            push(@LIST, [ [ $ed ], [@keywords, "$target:$port", &nodeFrom($topo, "$target:$port")] ]);
         }
      } else {
         if ($type eq "host") {
            push(@LIST, [ [ $ed ], [@keywords, $target, &nodesFrom($topo, $target)] ]);
         } else {
           push(@LIST, [ [ $ed ], [@keywords, $target] ] );
         }
      }
  }
  if ($TEST) {
    showList(\@LIST, "before");
    &merge(\@LIST);
    showList(\@LIST, "after");
  } else {
    &merge(\@LIST);
  }

  foreach my $l (@LIST) {
     my $elist = $l->[0];
     my $nodes = $l->[1];
     if ($#$elist > 0) { # at least 2 events were aggregated.
        $class->findSignature($topo, $nodes, $elist, $Config, $TEST);
     }
  }

  &update_parent();
  return 1;
}



sub update_parent {

  my $ED = RasDB->new("EDOCS");
  my $hash = $ED->hash();
  $ED->Lock();
  foreach my $el (sort keys %$hash) {
     my $ed     = $hash->{$el};
     my $ev     = $ed->instances(0);
     my $edoc_id   = $ed->id("edoc_id");
     my $parent;
     if ($parent = $IN_AGGREGATE{$edoc_id}) {
         $ev->setValue("Parent", $parent);
         $hash->{$el} = $ed;
     }
  }
  $ED->UnLock();
}

  


sub eventGroups {
  my($class, $elist) = @_;
  my %G;
  foreach my $e (@$elist) {
     my $ev = $e->instances(0);
     my $type = $ev->value("EventType");
     my ($cat, $et) = split(/\./, $type, 2);
     $G{EventType}{$et}{count}++;
     push(@{$G{EventType}{$et}{events}}, $e);

     $G{Category}{$cat}{count}++;
     push(@{$G{Category}{$cat}{events}}, $e);

     $G{EventType2}{$type}{count}++;
     push(@{$G{EventType2}{$type}{events}}, $e);
  }
  return \%G;
}
     

sub showList {
  my ($LIST, $label) = @_;
  my ($y,$x);
  for ($x=0; $x <= $#$LIST; $x++) {
     my $l = $LIST->[$x];
     my $elist = $l->[0];
     my $nodes = $l->[1];
     next if (!$nodes);
     print "$label-LIST $x\n";
     for ($y=0; $y <= $#$elist; $y++) {
       my $e = $elist->[$y];
       my $ev = $e->instances(0);
       print "      $y:" . $e->description() . ", et=" . $ev->value("EventType") . ", cap=" . 
                    $ev->value("Caption") . ", comp=" . $ev->value("Component") .
                    ", sev=" . $ev->value("Severity") . "\n";
     }
     print   "   Nodes:". join(", ", @$nodes) . "\n" if ($nodes);
     print "--------------------------\n";
  }
}

# all nodes visible from this target (port is optional)
#
sub nodesFrom {
  my($topo, $target) = @_;
  my @L;
  my $list = $topo->visit($target);
  foreach my $el (@$list) {
     push(@L, $el->name());
  }
  return @L;
}

# first nodeport visible from this nodeport
#
sub nodeFrom {
  my($topo, $nodeport) = @_;
  my $target_nodeport = $topo->nodeportTarget($nodeport);
  return ($target_nodeport);
}

sub merge {
  my($LIST) = @_;

  my ($x, $y);
  my $again = 1;
  my $loop = 0;
  while ($again && $loop++ < 100) {
     for ($x=0; $x <= $#$LIST; $x++) {
        my $E1 = $LIST->[$x];
        next if (!$E1);
        for ($y = $x+1; $y <= $#$LIST; $y++) {
           my $E2 = $LIST->[$y];
           next if (!$E2);
           my $int;
           if (($int = &intersect($E1->[1], $E2->[1])) ) {
              my $E1_elist = $E1->[0];
              my $E2_elist = $E2->[0];
              push(@$E1_elist, @$E2_elist);
              if ($E1->[2]) {
                #&uniq($E1->[1], $int);
                $E1->[1] = $int;
              } else {
                $E1->[1] = $int;
                $E1->[2] = 1;
              }
              $LIST->[$y] = undef;
              $again = 1;
           }
        }
     }
  }
}

sub uniq {
  my($l1, $l2) = @_;

  foreach my $el (@$l2) {
      my $found = 0;
      foreach my $el2 (@$l1) {
         if ($el2 eq $el) {
            $found = 1; last;
         }
      }
      if (!$found) {
         push(@$l1, $el);
      }
  }
}
#
# X:1 means enc X, port 1
# X   means enc X, all ports
#
sub intersect {
  my($E1, $E2) = @_;
  my ($x, $y);
  my @INT;
  foreach my $node1 (@$E1) {
     foreach my $node2 (@$E2) {
        my $l1 = length($node1);
        my $l2 = length($node2);
        my ($t1, $k1, $p1) = split(/\:/, $node1);  
        my ($t2, $k2, $p2) = split(/\:/, $node2);  
        if ($node1 eq $node2) {
           push(@INT, $node1);
        } elsif ($l1 < $l2 && substr("$node1:",0,$l1+1) eq substr($node2,0,$l1+1) ) {
           push(@INT, "$t1:$k1");

        } elsif ($l2 < $l1 && substr("$node2:",0,$l2+1) eq substr($node1,0,$l2+1) ) {
           push(@INT, "$t2:$k2");
        }
     }
  }
  return undef if ($#INT < 0);
  return \@INT;
}

sub swap {
  my($class, $elist, $x1, $x2) = @_;
  my $save = $elist->[$x1];
  $elist->[$x1] = $elist->[$x2];
  $elist->[$x2] = $save;
}

# RUN all rules from FSA/, stop if a new event is generated or
# a new event is identifned as the new leader.
#
sub rule_run {
  my($class, $Config, $topo, $targets, $elist) = @_;
  my($r);
  foreach my $type ('RuleSet1') {
    my $pk = "FSA::$type";
    require "FSA/$type.pm";
    for ($r=1; $r <= 100; $r++) {  # rules must start at 1, sequential
       my $r1 = "RULE$r";
       if ($pk->can($r1)) {
          my($ev, $pertains, $done) = $pk->$r1($Config, $topo, $targets, $elist);
          if ($ev || $done) {
           Debug->print2("FSA: Rule $pk->$r1 fired!");
           return ($ev, $pertains, "$pk->$r1");
          }
       } else {
         last;
       }
    }
  }
  return ();
}

sub newEvent {
  my($class, $elist, $targets, $high_event, $desc, $arg) = @_;

  my($cat, $key1, $comp1) = split(/\:/, $targets->[0]);
  Grid->setCode($arg->{GridCode} || $high_event->value("GridCode"));
  my $id = PDM->getEventSequence(); 
  if (!$key1) {
    ($cat, $key1) = split(/\:/, $high_event->value("Target"));
  }
  if (!$key1) {
     Debug->print2("FSA::newEvent: no key for this event, gridcode=" . 
                     $high_event->value("GridCode"));
     return ();
  }
        
  my $sev = exists $arg->{Severity}  ? $arg->{Severity}  : $high_event->value("Severity");
  my $act = exists $arg->{Actionable}? $arg->{Actionable}: $high_event->value("Actionable");

  my $ev = CIM::Instance->new('NWS_Event', [
               [ EventType   => $arg->{EventType}   || $high_event->value("EventType") ],
               [ EventId     => $id       ],
               [ Target      => $arg->{Target}      || $high_event->value("Target")    ],
               [ TargetName  => $arg->{TargetName}  || $high_event->value("TargetName")],
               [ MgmtLevel   => $high_event->value("MgmtLevel") ],
               [ Component   => $arg->{Component}   || $high_event->value("Component") ],
               [ PriorValue  => $arg->{PriorValue}  || $high_event->value("PriorValue") ],
               [ CurrentValue=> $arg->{CurrentValue}|| $high_event->value("CurrentValue") ],
               [ Caption     => $arg->{Caption}     || $high_event->value("Caption")   ],
               [ SourceIP    => $arg->{SourceIP}    || $high_event->value("SourceIP")  ],
               [ Actionable  => $act          ],
               [ Severity    => $sev          ],
               [ Fault       => $arg->{Fault} ],
               [ Aggregate   => ($#$elist+1)  ], 
               [ Description => $desc || $high_event->value("Description")],
               [ Data        => $arg->{Data}  ],
                            ]);
   my $key = CIM::Key->new( ['NWS_System', Name  => $key1, 
                                    CreationClassName => 'NWS_System'] );
     
   my $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                     [ Event       => $ev  ],
                     [ Element     => $key ],
                     ]);
   return ($ev, $pertains);
}


################################################
#      CREATE AGGREGATED EVENTS

sub findSignature {
  my($class, $topo, $targets, $elist, $Config, $TEST) = @_;
  my ($w);

  CIM->version("1.1");
  my $Config = PDM::ConfigFile->read();

  my($targetName, $target_list, $source_ip) = $class->target_list($targets, $Config);

  $class->orderEvents($elist);

  # this may return $ev and $pertains || nothing
  # may swap some events in elist
  my($ev, $pertains, $rule_no) = $class->rule_run($Config, $topo, $targets, $elist);
  
  my($cnt, $max_sev, $agg_list, $high_event) = $class->event_stats($elist);

  if (!$ev) {  # create generic event with first event as the aggregated event.
     #if ($#$targets == 0) {
        my $hd = $high_event->value("Description");
        ($ev, $pertains) = FSA->newEvent($elist, $targets, $high_event);
     #}
  } else {
     $max_sev = $ev->value("Severity");
  }

  if ($ev) { # send event
    my $sd    = Events->sourceDetector({ event => $ev });
    my($category, $rest) = split(/\./, $ev->value("EventType"));

    my $ev_id = { aggregate  => ($#$elist+1), 
                  deviceName => $ev->value("Target"), 
		  agg_list   => $agg_list,
                  category   => $category };

    my $ed    = Message->new({    id => $ev_id, 
                            severity => $max_sev,
                               state => [$ev->value("Component"), 0 ],
                           instances => [$ev, @$sd, $pertains] 
                              });

    my $edoc_id = PDM->saveMessage($ed, undef, 1);

    foreach my $ed (@$elist) {
       $IN_AGGREGATE{$ed->id("edoc_id")} = $edoc_id;
    }
  }
}

sub inAggregate {
  my($class, $id) = @_;
   
  return $IN_AGGREGATE{$id};
}


##################################################################
# FORM A LIST OF TARGETS and TARGET_NAMES
#
sub target_list {
  my($class, $targets, $Config) = @_;
  my ($cat, $key, $targetName, $target_list, $dev ,$source_ip);

  foreach my $t (@$targets) {
      $target_list .= "$t|";
      if (($dev = $Config->deviceByKey($t)) ) {
        $targetName .= "$dev->{type}:$dev->{name}, ";
        $source_ip = $dev->{ipno} if (!$source_ip);
      } else {
        $targetName .= "$t, ";
      }
  }
  if ($targetName) {
    chop($targetName); chop($targetName) ;
  }
  chop($target_list) if ($target_list);

  return ($targetName, $target_list, $source_ip);
}



#  SUMMARY SECTION OF THE AGG. EVENT
#
sub orderEvents {
  my($class, $elist) = @_;
  my ($actionable);
  my $cnt = 0;
  my $agg_list; # ordered by severity
  my (%DATA,  $max_sev, $x, $high_event, @elist2);

  for ($x=0; $x <= $#$elist; $x++) {
     my $ed      = $elist->[$x];
     my $ev      = $ed->instances(0);
     my $date    = $ev->value("EventTime");
     my $sev2    = $ed->severity2();
     my $act     = $ev->value("Actionable");
     $date       = substr($date,4,10);
     $cnt++;
     my $order = $sev2 * 10;
     $order += 5 if ($act eq "TRUE");
     $DATA{sprintf("%2.2d-%s-%d", 99 - $order, $date, $x)} = 1;
     $elist2[$x] = $elist->[$x];
  }
  my $y=0;
  foreach my $x (sort keys %DATA) {
     my @s = split(/\-/, $x);
     $elist->[$y] = $elist2[$s[2]];
     $y++;
  }
}

sub event_stats {
  my($class, $elist) = @_;
  my($x, $max_sev, $actionable, $agg_list, $cnt, $high);

  for ($x=0; $x <= $#$elist; $x++) {
     my $ed      = $elist->[$x];
     my $ev      = $ed->instances(0);
     $high       = $ev if (!$high);
     my $act     = $ev->value("Actionable");
     $cnt++;
     $actionable = 1 if ($act eq "TRUE");
     my $sev2    = $ed->severity2();
     if ($sev2 > $max_sev) {
        $max_sev = $sev2; 
     }
     $agg_list .= $ev->value("EventId") . ",";
  }
  return ($cnt, $max_sev, $agg_list, $high);
}
  


sub wrap {
  my($desc, $space) = @_;
  my ($cnt, $out, $len);
  my @words = split(/\s/, $desc);

  foreach my $w (@words) {
     if ($len+length($w) > 60) {
       $out .= "\n$space"; $len = 0;
     }
     $out .= "$w "; $len += length($w);
  }
  return $out;
}

sub RUN2 {
  my $renv = System->get_renv();
  my $OLDF = "previous_events";
  my $new = PDM->getMessages();
  my $old = Util->deserialize($OLDF);

  if (!$old) {
    Util->serialize($new, $OLDF);
    return;
  }
  my $rules = Modules->load("FSA");
  foreach my $rule0 (@$rules) {
     my $rule = "FSA::$rule0";
     if ($rule->can("RUN")) {
        $rule->RUN( $old, $new);
     }
  }
  my $new = PDM->getMessages();
  my (@CURRENT, @OLD) ;
  foreach my $e (@$old) {
      push(@CURRENT, $e);
  }
  foreach my $e (@$new) {
      if ($e->id("inAggregate") || $e->id("aggregate") ) {
         push(@CURRENT, $e);
      } else {
         push(@OLD, $e);
      }
  }
  Util->serialize(\@OLD, $OLDF);
  PDF->setMessages(\@CURRENT);

}

1;

