#
# Copyright 2003 Sun Microsystems, Inc., All Rights Reserved.
#
package Client::Device;
use Debug;
use Client;
use strict;
use Util;
use Matrix;
use Revision;
use Process;

#
sub help {
  print Client->http_OK();
  print "Help: In progress...\n";
}


# Adds a device to be monitored.
# Arguments ip - Ip address of device.
#           type - hint at type of device.
#
# Returns the discovered properties for the device that have been 
#  written to the rasagent.conf file.
sub Add {
  require Logic::Subnet;

  my($q) = @_;
  my $in_ip = $q->{ip};
  my $intype = $q->{type};
  my $timeout = $q->{timeout} || "10";
  my $format = $q->{format} || "tab";

  Debug->logLine("Client::Device::Add $in_ip $intype");

  my $stuff =
  Logic::Subnet->snmpget1($in_ip, "public 1.1.0", $timeout);
  my ($line) = @$stuff;
  my($class, $type, $ip, $wwn, $key, $wwn2, $ctrl, $devname, $label) = 
    split(/\|/, $line) ;
  my($ip0, $port) = split(/\:/, $ip);

  if ($class eq "ERR") {
    print Client->error($format, 101, "$type");
    return;
  }

  if (!$key){
    print Client->error($format, 103, "Device not found for $in_ip");
    return;
  }

  # Add device to Config file
  my($renv, $devs, $hosts,$notifs) = PDM::ConfigFile->read();
  my $new_dev = $#$devs ;
  my $cnt=1;
  my %F;
  foreach my $d (@$devs) {
    $F{$d->{key}} = $cnt;
    $cnt++;
  }
#  if ($F{$key}) {
#    print Client->error($format, 102, "Device exists ($type $ip $key)");
#    return;
#  }

  my $timeAdded = Util->get_today();

  $new_dev++;
  $devs->[$new_dev]{_name}      = "device" . $new_dev;
  $devs->[$new_dev]{name}       = $devname;
  $devs->[$new_dev]{type}       = $type;
  $devs->[$new_dev]{class}      = $class;
  $devs->[$new_dev]{userLabel}  = $label if ($label);
  $devs->[$new_dev]{ip}         = $ip;
  $devs->[$new_dev]{ipno}       = $ip;
  $devs->[$new_dev]{wwn}        = $wwn;
  $devs->[$new_dev]{wwn2}       = $wwn2 if ($wwn2);
  $devs->[$new_dev]{key}        = $key;
  $devs->[$new_dev]{host}       = ""; #$q->{host};
  $devs->[$new_dev]{active}     = "Y";
  $devs->[$new_dev]{mgmtLevel}  = "D";
  $devs->[$new_dev]{time_added} = $timeAdded;

  # Save conf if device is new.
  if (!$F{$key}) {
    PDM::ConfigFile->write( $renv, $devs, $hosts,$notifs);
  }

  print Client->http_OK();
  if ( $q->{format} eq "xml"){
    print "<?xml version=\"1.0\"?>";
    print "<CLIENT_ADD>";
    print "<VALUE ID='class'>$class</VALUE>\n";
    print "<VALUE ID='type'>$type</VALUE>\n";
    print "<VALUE ID='ip'>$ip</VALUE>\n";
    print "<VALUE ID='wwn'>$wwn</VALUE>\n";
    print "<VALUE ID='key'>$key</VALUE>\n";
    print "<VALUE ID='wwn2'>$wwn2</VALUE>\n" if ($wwn2);
    print "<VALUE ID='name'>$devname</VALUE>\n";
    print "<VALUE ID='userLabel'>$label</VALUE>\n" if ($label);
    print "<VALUE ID='time_added'>$timeAdded</VALUE>\n";
    print "<VALUE ID='mgmtLevel'>D</VALUE>\n";
    print "<VALUE ID='active'>Y</VALUE>\n";
    print "</CLIENT_ADD>";
  }
  else {
    print "#<pre>\n";
    print "class\t$class\n";
    print "type\t$type\n";
    print "ip\t$ip\n";
    print "wwn\t$wwn\n";
    print "key\t$key\n";
    print "wwn2\t$wwn2\n" if ($wwn2);
    print "name\t$devname\n";
    print "userLabel\t$label\n" if ($label);
    print "time_added\t$timeAdded\n";
    print "mgmtLevel\tD\n";
    print "active\tY\n";
    print "#DONE";
  }
}


#
# Find the properties for a given device
#
sub Properties {
  my($q) = @_;
  my $format = $q->{format} || "tab";
  my($type, $key) = split(/\:/, $q->{key}) ;

  Debug->logLine("Client::Device::Remove $type $key");

  my($renv, $devs, $hosts,$notifs) = PDM::ConfigFile->read();
  my $numDevs = $#$devs ;
  my $index=0;
  my %F;
  foreach my $d (@$devs) {
    if ($type eq $d->{type}) {
      if ($key eq $d->{key}){

	print Client->http_OK();
	if ( $q->{format} eq "xml"){
	  print "<?xml version=\"1.0\"?>\n";
	  print "<DEVICE_PROPERTIES>\n";
	  for my $xprop ( keys %{$d} ) {
	    print "<VALUE ID='$xprop'>$d->{$xprop}</VALUE>\n";
	  }
	  print "</DEVICE_PROPERTIES>\n";
	}
	else {
	  print "#<pre>\n";
	  for my $prop ( keys %{$d} ) {
	    print "$prop\t$d->{$prop}\n";
	  }
	  print "#DONE";
	}
	return;
      }
    }
    $index++;
  }
  print Client->error($format, 101, "No device exists for $type $key");
}


#
# Remove a device from the monitor list.
#
sub Remove {
  my($q) = @_;
  my $format = $q->{format} || "tab";
  my($type, $key) = split(/\:/, $q->{key}) ;

  Debug->logLine("Client::Device::Remove $type $key");

  my($renv, $devs, $hosts,$notifs) = PDM::ConfigFile->read();
  my $numDevs = $#$devs ;
  my $index=0;
  my %F;
  foreach my $d (@$devs) {
    if ($type eq $d->{type}) {
      if ($key eq $d->{key}){
	my @$matchDevs = splice(@$devs, $index, 1);
	PDM::ConfigFile->write( $renv, $devs,
				$hosts,$notifs);
	 print Client->http_OK();
	 print "OK";
	 return;
      }
    }
    $index++;
  }
  print Client->error($format, 101, "No device exists for $type $key");
}


#
# Update all devices or if a key defined a single device.
#
sub Update {
  my($q) = @_;
  my $format = $q->{format};
  my($type, $key) = split(/\:/, $q->{dev}) ;

  my @l = split(/\|/, $q->{devlist});
  foreach my $k (@l) {
    my($ktype, $kvalue) = split(/\:/, $k);
    if ($key){
      $key .= ":".$kvalue;
    }
    else {
      $key = $kvalue;
    }
  }

  my $ras_cmd = System->get_home() . "/bin/rasagent";
  my $ras_args;

  if ($key){
    $ras_args = "-l $key";
  }

  my $command =  "$ras_cmd $ras_args >> /dev/null 2>&1 &";
  system($command);

  Debug->logLine("Client::Device::Update $command");

  print Client->http_OK();
  print "OK $command";
}

#
# Run ras_revcheck on a single device
#
sub Revision {

  my($q) = @_;

  my $key     = $q->{key};
  my $format  = $q->{format} || "xml";
  my $MODULES = $q->{module} || "ALL";
  my @REPORTS;
  my $TT = "XREV";

  Debug->logLine("Client::Device::Revision $MODULES $key");

  if ($key){
    print Client->error($format, 99,
			"Revision Check for single device not implemented.");
    return;
  }

  my $pid = fork();

  if ($pid){ #parent
    return;
  }

  if (defined $pid) {  # child
    my $include;
    my ($env)      = Revision->hbas_present();
    my $MATRIX     = Matrix->defaultFile();
    my $matrix     = Matrix->read($MATRIX, $env);
    my $installedp = Revision->readInstalledPatches();
    my $mods       = Modules->load("Revision");
    
    foreach my $x (@$mods) {
      if ($MODULES eq "ALL" || index(lc("|$MODULES|"), lc("|$x|")) >= 0) {
	$include .= " $x|";
	my $mod = "Revision::$x";
	my $db;
	eval {
	  $db = $mod->RUN($matrix, $installedp);
	};
	if (!$db) {
	  #$db = Revision->warning2($x);
	  #push(@REPORTS, @$db);
	} elsif ($@) {
	  #$errors .= "ERROR in $mod: $@ \n";
	} else {
	  push(@REPORTS, @$db);
	}
	#open(O, ">>$LOG"); print O "module $x done\n"; close(O);
	chop($include);
      }
    }
    
    my($renv, $devs) = PDM::ConfigFile->read();
    my %KEY;
    foreach my $d (@$devs) {
      $KEY{$d->{name}} = $d->{key};
      $KEY{$d->{ipno}} = $d->{key};
    }
    
    print Client->http_OK();
    print "<REVISION_REPORT>\n";
    foreach my $ele (@REPORTS){
      my $devKey = $ele->[8];
      if (!$devKey){
	$devKey = $KEY{$ele->[1]};
      }
      if (!$devKey){
	my @N = split(/\./, $ele->[1]);
	$devKey = $KEY{$N[0]};
      }
      my $devType;
      foreach my $d (@$devs) {
	if ($d->{key} eq $devKey) {
	  $devType = $d->{type};
	}
      }
      my $fruKey = $ele->[9];
      
      print "<ITEM\n";
      print "Type='$ele->[0]'\n";
      print "DevType='$devType'\n" if $devType;
      print "DevKey='$devKey'\n" if $devKey;
      print "Fru='$fruKey'\n" if $fruKey;
      print "Name='$ele->[1]'\n" if $ele->[1];
      print "Status='$ele->[2]'\n" if $ele->[2];
      print "CurrVers='$ele->[3]'\n" if $ele->[3];
      print "ExptVers='$ele->[4]'\n" if $ele->[4];
      print "CurrPatch='$ele->[5]'\n" if $ele->[5];
      print "ExptPatch='$ele->[6]'\n" if $ele->[6];
      print "Message='$ele->[10]'\n" if $ele->[10];
      print ">\n";
      print "$ele->[7]\n" if $ele->[7];
      print "</ITEM>\n"
    }
    print "</REVISION_REPORT>\n";
  }
  else {
    print Client->error($format, 98,
			"Unable to run command.");
  }
}


#
# Lease a device
#
sub LeaseRequest {
  my($q) = @_;

  my $format   = $q->{format};

  Debug->logLine("Client::Device::LeaseRequest $format");

  print Client->error($format, 99,
		      "Lease Request not implemented.");
}

#
# Clear a device from a lease
#
sub LeaseRelease {
  my($q) = @_;

  my $format   = $q->{format};

  Debug->logLine("Client::Device::LeaseRelease $format");

  print Client->error($format, 99,
		      "Lease Release not implemented.");
}


1;
