package Agent::3310;
#<copyright>
# ----------------------------------------------------------
# Sun Proprietary/Confidential Code
# Copyright 2001, Sun Microsystems, Inc. All rights reserved.
# ----------------------------------------------------------
#</copyright>

#  $Name:  $ 
#  $Id: 
use strict;
use Agent;
use base 'Agent';

sub isSelectable {0}
sub revision {'$Revision: 1.78 $'}

sub category {Report::CAT_3310}


sub new {
  my($self) = Agent->new();

  bless ($self, 'Agent::3310');
  return $self;
}

# translate port to the right  CIM key 

sub getPortKey {
  my($class, $wwn, $port) = @_;
  return "$wwn.$port";
}

sub RUN {
  my($agent, $static_list, $pass) = @_;
  my($HBA, $logfile, $r, @results);
  my($xml, @lux_dif, $dif, $ev, $found);
  my($log_err, $lines, $timelapse);
  my(%DONE, $report, $device, $key, $id, $x, $l, $connect_errs);
  $DB::single=1;
  
  my $renv = System->get_renv();
  my $type = $agent->category();
  my $ping_to  = $renv->{'timeout.ping'} || 10;

  my $today = Util->today("YMDH");
  my $processed_list = [];
  my($dc) = 0;

  foreach $device ( $agent->deviceList($static_list)) {
    next if ($device->{active} eq "N");
    $dc++;
    Debug->print1("-> " . uc($device->{type}) . ": Reading device $device->{name}/$device->{wwn}/$device->{path}");
    my $problems = PDM->get_last_event();

    if (($device->{path} =~ /dev/) ||
        ($device->{ipno} && Util->ping2($device->{ipno}, $ping_to))
       ) {
      my($err, $lines) = $agent->readLog($device, $report);
      my($warn_lines)  = $agent->find_alert($lines, $device);
      if ($warn_lines) {
        $id = {
              deviceName  =>  $device->{key},
              active      => "Y",
              display     => $device->{name},
              name        => $device->{name},
              class       => "storage.$type",  # Report::CAT_3310MESSAGE
              category    => $type . "message",
              ip          => "",
           };
        $report->{"id.name"} = $device->{name};
        
        my $new_rep = Report->new($id, $report , $warn_lines, Report::STATUS_OK);
        require Health::Message;
        Health::Message->all_logic($new_rep);
        PDM->saveReport( $new_rep );
      }
    }

    ($key, $connect_errs, $report) = $agent->INSTRUMENTATION($device);
    if ($connect_errs eq "BUSY") {
       Debug->print2("Device $device->{name} is busy, skipping!");
       next;
    }

    $key= lc($key);

    $id = {  
              deviceName  => $device->{key},
              active      => $device->{active},
              logFile     => "",
              display     => "$device->{name} (wwn=$device->{key})",
              name        => $device->{name},
              class       => $device->{class},
              category    => $type,
              ip          => "",
           };
     $report = {} if (!$report);
     $agent->copyDev($device, $report);
     my $wwns = Agent::3310->get_wwns($report);
     if (length($wwns) > 10 && $wwns ne $device->{wwns}) {
        Debug->print2("The port-wwns of $device->{name} changed.");
        $device->{wwns} = $wwns;
        PDM::ConfigFile->updateDevice($device);
     }
     $HBA = $device->{hba} || System->get_hba;
     $report->{"id.name"} = $device->{name};
     $report->{"id.ipno"} = System->get_ipno();
     $report->{"id.wwn"}  = $device->{key};
     my $new_rep;
     if ($connect_errs || !$key) {
        $report->{'id.connect_errs'} = $connect_errs;
        $new_rep = Report->new($id, $report , undef, Report::STATUS_CANNOT_CONNECT);
     } else {
        $new_rep = Report->new($id, $report , undef);
     }
     require Health::3310;
     Health::3310->all_logic($new_rep); # create event
     PDM->saveReport($new_rep);

     my ($broke, $abort) = $agent->new_events($problems, $device, 1);
     push(@$processed_list, $device);
     last if ($broke && $pass == 1 || $abort);

  } 
  return $processed_list;
}


############################################################################

use vars qw ($ERR);


#
#  Fast: ONLY CALLS sccli show ip-address and show port-wwn
#
sub getFastWWN {
  my($agent, $name, $host) = @_;
  my($wwn);
  $ERR = undef;
  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;


  if (!$host) {
     return &get_WWNF({name => $name});
  } else {
     my $rc = Util::Http->getCommand($host, "Agent::3310::WWNF&name=$name&HTTP=1" , $TO);
     if ($rc !~ /OK /) {
        $ERR = "3310: getWWN: cannot identify $name on $host";
        return undef;
     } else {
        my $VAR1;
        my $ix = index($rc, "OK ");
        eval substr($rc,$ix+3);
        return $VAR1;
     }
  }
}


sub getWWN {
  my($agent, $name, $host) = @_;
  my($wwn);
  $ERR = undef;
  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;


  if (!$host) {
     return &get_WWN({name => $name});
  } else {
     my $rc = Util::Http->getCommand($host, "Agent::3310::WWN&name=$name&HTTP=1" , $TO);
     if ($rc !~ /OK /) {
        $ERR = "3310: getWWN: cannot identify $name on $host";
        return undef;
     } else {
        my $VAR1;
        my $ix = index($rc, "OK ");
        eval substr($rc,$ix+3);
        return $VAR1;
     }
  }
}

sub FCfromDevice {
  my($class, $device, $report) = @_;
  my(%SW, $x , %X, $err);

    my $R = {};
    $err = $class->getDiskFC($device, $report);
    return {} if ($err);
    $err = $class->getFC($device, $R);
    return {} if ($err);

    foreach my $el (keys %$R) {
        $report->{$el} = $R->{$el};
    }
    for ($x=0; $x < 8; $x++) {
        my $key = "|$device->{key}|port.$x|$device->{type}";
        $X{$key} = $R->{"channel.$x.error.link"}   . "\t" . 
                      $R->{"channel.$x.error.signal"} . "\t" .
                      $R->{"channel.$x.error.seq"}    . "\t" .
                      $R->{"channel.$x.error.crc"}    . "\t" .
                      $R->{"channel.$x.error.sync"}   . "\t" .
                      $R->{"channel.$x.error.itw"};
    }
    return { data => \%X,  hba => {} };
}

sub getDiskFC {
  my($class, $device, $report) = @_;
  my($err, $com, $chan, $disk);
  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;


  my $sccli = &sccli_path();
  my $w = $device->{path} || $device->{ipno};

  for ($disk=0; $disk <= $report->{"info.top_disk"}; $disk++) {
    next if (!exists $report->{"components.disk.$disk.model"});

    # my $ch = $report->{"components.disk.$disk.ch"};
    # need all drive channel list 

    my $ch;
    my $drv_chan_list = $class->get_drv_chan($w); 
    foreach my $ch ( @$drv_chan_list) { 
       ($err,$com) = Util->run_command("$sccli $w diag error -chan $ch -targ $disk", "chan.txt" , $TO);
       $class->parseFC($com, $report, "components.disk.$disk.error.chan_$ch");
    }
  }
}

sub get_drv_chan {
   my ($class, $device) = @_;
   my($err, $com);
   my @list;
 
   my($renv) = System->get_renv();
   my($TO) = $renv->{'timeout.sccli'} || 400;
   my $sccli = &sccli_path();

   ($err,$com) = Util->run_command("$sccli $device show chan", "drv_chan.txt" , $TO);
   foreach my $el (@$com) {
        next if ($el!~ /Drive/);
        my $l = Util->trim($el);
        my ($ch_id, $tem) = split(/\s+/,$l);
        push(@list,$ch_id);
   } 
   return \@list;
}

sub getFC {
  my($class, $device, $rep) = @_;
  
  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;


  my $sccli = &sccli_path();
  my $w = $device->{path} || $device->{ipno};
  my($err, $com, $chan, $chan0);
  for ($chan0=0; $chan0 < 6; $chan0++) {
    ($err,$com) = Util->run_command("$sccli $w diag error -chan $chan0", "chan.txt" , $TO);
    return $err if ($err);
    my $chan = $chan0 + 1;
    $class->parseFC($com, $rep, "channel.$chan.error");
  }
  return undef;
}

sub parseFC {
  my($class, $com, $rep, $init) = @_;

  foreach my $el (@$com) {
     last if ($el =~ /Times of/);
      my($el, $v) = split(/\: /, $el, 2);
      if ($el =~ /LIP/){
         $rep->{"$init.lip"} = $v;

      } elsif ($el =~ /Link/) {
         $rep->{"$init.link"} = $v;

      } elsif ($el =~ /Sync/) {
         $rep->{"$init.sync"} = $v;

      } elsif ($el =~ /Signal/) {
         $rep->{"$init.signal"} = $v;

      } elsif ($el =~ /Primitive/) {
         $rep->{"$init.seq"} = $v;

      } elsif ($el =~ /Transmission/) {
         $rep->{"$init.itw"} = $v;

      } elsif ($el =~ /CRC/) {
         $rep->{"$init.crc"} = $v;
      }
  }
}

sub sccli_path {
  return System->get_home() . "/bin/sccli";
}


sub get_WWNF {  # fast
  my($q) = @_;
  my $report = {};
  my $rc = {};
  if ( ($q->{name} =~ /dev/) || Util->ping($q->{name}, 10) ){
    my $sccli = System->get_home() . "/bin/sccli";
    
    my($err,$com) = Util->run_command("$sccli $q->{name} show ip-address 2>&1", "luxadm.txt" , 30);

    if($err){
       $rc->{error} = "Error on sccli, $err";
    } else {  
       if ("@$com" =~ /\[SUN StorEdge (\w+) SN#(\w+)\]/) {
          my $type = $1;
          my $sn   = lc($2);
          $rc->{key}      = "206000c0ff$sn";
          $rc->{userLabel} = $type;
          $type           = "3310" if ($type =~ /N/);
          $rc->{type}     = $type || "3510";
          $rc->{error}    = $err;
          $rc->{units}    = 1;
          $rc->{ip}       = $report->{'info.ip'};
          $rc->{ipno}     = $com->[1];
          if ($type =~ /3310/){
            # Currently there is a bug in 3310 which causes IO to go to 5% when we 
	    # probe, removing support for now.
            $rc->{error} = "$q->{name} is a 3310. 3310 is not a supported device.";
	    $rc->{key} = undef;
          } else {
            Agent::3310->getPortWWNS({ipno => $q->{name}} , $report);
            $rc->{wwn2} = Agent::3310->get_wwns($report, $rc);
            $rc->{wwn}  = $rc->{key} if (!$rc->{wwn});
            $rc->{report}   = $report;
          }
       }  
    }
  } else {
    $rc->{error} = "Cannot ping $q->{name}";
  }

  if ($q->{HTTP}) {
    $Data::Dumper::Indent = 0;
    print "\nOK " . Data::Dumper::Dumper($rc) . "\n";
  } else {
    return $rc;
  }

}


sub get_WWN {
  my($q) = @_;
  my $report = {};
  my $rc = {};
  if ( ($q->{name} =~ /dev/) || Util->ping($q->{name}, 10) ){
    my $err = Agent::3310->getConfig({ipno => $q->{name} }, $report);
    if($err){
       $rc->{error} = "Error on getConfig, $err";
    }else{  
       my $sn = sprintf("%6.6x", hex($report->{'info.unique_id'}));
       $rc->{key}      = "206000c0ff$sn" if ($report->{'info.unique_id'});
  
       my $type        = $report->{'info.model'};
       $type           =~ s/StorEdge\s+//;
       $rc->{userLabel} = $type;
       $type           = "3310" if ($type =~ /N/);
       $rc->{type}     = $type || "3510";
       $rc->{error}    = $err;
       
       $rc->{model}    = $report->{'info.model'};
       $rc->{disk_map} = $report->{"info.disk_map"};
       $rc->{top_disk} = $report->{"info.top_disk"};
       $rc->{units}    = 1;
       $rc->{wwns}     = "";
       $rc->{revision} = $report->{'info.firmware_version'};
       $rc->{serial}   = $report->{'info.primary_sn'};
       $rc->{ip}       = $report->{'info.ip'};
       $rc->{ipno}     = Util->name2ip($rc->{ip});
       if($type =~ /3310/){
          # Currently there is a bug in 3310 which causes IO to go to 5% when we 
	  # probe, removing support for now.
          $rc->{error} = "$q->{name} is a 3310. 3310 is not a supported device.";
	  $rc->{key} = undef;
       }else{
          Agent::3310->getPortWWNS({ipno => $q->{name}} , $report);
       
          $rc->{wwn2} = Agent::3310->get_wwns($report, $rc);
          $rc->{wwn}  = $rc->{key} if (!$rc->{wwn});
          $rc->{report}   = $report;
       }
     }  
  } else {
    $rc->{error} = "Cannot ping $q->{name}";
  }

  if ($q->{HTTP}) {
    $Data::Dumper::Indent = 0;
    print "\nOK " . Data::Dumper::Dumper($rc) . "\n";
  } else {
    return $rc;
  }

}

sub get_wwns {
  my($class, $report, $rc) = @_;

  my($x, $wwns);

  for ($x=0; $x <= 10; $x++) {
     my $ww = $report->{"port.$x.wwn"};
     if (exists( $report->{"port.$x.wwn"}) && index($wwns, $ww) < 0 ) {
        $rc->{wwn} = $ww if ($rc && !$rc->{wwn});
        $wwns  .= "$ww,";
     }
  }
  return $wwns;
}

sub get_WWN0 {
  my($q) = @_;
  my $name = $q->{name};
  my $sccli = System->get_home() . "/bin/sccli";
  my($err,$com) = Util->run_command("$sccli $name inquiry", 
                          "luxadm.txt" , 30);
  if ($err) {
    $q->{HTTP}? print "\nERR $err" : return $err;
  }
  my $rc = {};
  foreach my $l (@$com) {
     if ($l =~ /Product: StorEdge (.+)/) {
        $rc->{model} = $1;
     } elsif ($l =~ /Revision: (.+)/) {
        $rc->{revision} = $1;
     } elsif ($l =~ /Serial Number: (.+)/) {
        $rc->{serial} = $1;
     } elsif ($l =~ /IP Address: (.+)/) {
        $rc->{ip} = $1;
     }
  }
  $rc->{key}   = "204000c0ff0" . $rc->{serial};
  $rc->{wwn}   = $rc->{key};
  $rc->{type}  = "3310";
  $rc->{units} = 1;
  $rc->{wwns}  = "";

  if ($q->{HTTP}) {
    $Data::Dumper::Indent = 0;
    print "\nOK " . Data::Dumper::Dumper($rc) . "\n";
  } else {
    return $rc;
  }
}



sub INSTRUMENTATION {
  my($agent, $device) = @_;
  my(@s, %dev, $in, $key, $state, $num);
  my $report = {};

  if (System->get_testMode()) {
     my $testrep = Report->readTest($device);
     return ($device->{key}, "", $testrep);
  }

  my $renv = System->get_renv();
  my $ping_to  = $renv->{'timeout.ping'} || 10;

  $report->{'info.ethernet_status'} = "ok";

  if (!$device->{path} && $device->{ipno} && !Util->ping($device->{ipno},$ping_to)) {
     $report->{'info.ethernet_errs'} = "Ping failed to $device->{ipno} (timeout=$ping_to)";
     $report->{'info.ethernet_status'} = "ping_failed";
     return ($device->{key}, "Ping failed", $report);
  }
  Timelapse->startDev( $device); 
  my $err1 = $agent->getInquiry($device, $report);
  return ($device->{key}, $err1, $report) if ($err1);

  my $err2 = $agent->getConfig($device, $report);
  return ($device->{key}, $err2, $report) if ($err2);

  $err2    = $agent->getPortWWNS($device, $report);
  return ($device->{key}, $err2, $report) if ($err2);

  $err2    = $agent->getLunMaps($device, $report);
  return ($device->{key}, $err2, $report) if ($err2);

  if (index("|$renv->{categories}|", "|san|") >= 0) {
     $report->{"FC_COUNTERS"} = $agent->FCfromDevice($device, $report);
  }


  Timelapse->stopDev( $device); 
  return ($device->{key}, "", $report);
}

use Policies;
sub find_alert {
  my($class, $lines, $device) = @_;
  my @err;
  my $policies = Policies->new("3310_policies");
  my($x, %DONE);
  for ($x = 0; $x <= $#$lines; $x++ ) {
     my $line = $lines->[$x];
     my $comp1 = "other";
     if ($line =~ / Controller /) {
         $comp1  = "controller";
     } elsif ($line =~ / CPU /) {
         $comp1  = "cpu";
     } elsif ($line =~ / PS \d/) {
         $comp1  = "power";
     } elsif ($line =~ / Fan \d/) {
         $comp1  = "fan";
     }
     if ($line =~ /\[([0-9A-F][0-9A-F][0-9A-F][0-9A-F])\]/) {
        my $error_code = $1;
        next if ($error_code eq "01A5" && $DONE{'0125'});
        $DONE{$error_code} = 1;
     }
     my $key0 = $device->{type} . ":" . $device->{key};
     my $key  = "$key0:$comp1";
     #                            ix  disk    t3b0             key   
     $policies->run(\@err, $lines,\$x,$comp1, $device->{name}, $key, $device->{ipno}, $key0, 1);
  }
  return ( \@err);

}
use vars qw($DB);

# reset the cache pointed when discovering a device.

sub resetLogCache {
  my($class, $device) = @_;

  my $w = $device->{path} || $device->{ipno};

  $DB = RasDB->new("CACHE");
  my $cache = $DB->hash();
  delete $cache->{"$w.start"};
}


sub readLog {
  my($class, $device, $report) = @_;
  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;
  my @lines;

  if (System->get_testMode()) {
     my $l;
     my $F = System->get_snapshot() ? System->get_snapshot() . "/messages.3310" :
              "/var/adm/messages.3310";
     if (open(O, $F)) {
       while ($l= <O>) {
            push(@lines, $l);
       }
       close(O);
     }
     return (undef, \@lines);
         
  }

  my $sccli = &sccli_path();
  my $renv = System->get_renv();
  my $w = $device->{path} || $device->{ipno};
  $DB = RasDB->new("CACHE");
  my $cache = $DB->hash();

  my $last_date = $cache->{"$w.start"};
  # TEST PURPOSE: uncomment the following line to get the agent to read the same
  #               events from the command.
  # $last_date = "";

  my($err,$com) = Util->run_command("$sccli $w show events last 500", "luxadm.txt" , $TO);
  return ($err, $com) if ($err);
  my ($max, $date, $x);
  open(O, ">>/var/adm/messages.3310");

  for ($x=0; $x <= $#$com; $x++) {
     my $l = $com->[$x];
     if ($l =~ /(\w+) +(\w+) +(\d+) (\d\d\:\d\d\:\d\d) (\d+)/) {
        my $mth = $Util::MTH{$2};
        my $time = substr($4,0,8);
        $date = sprintf("%2.2d-%2.2d-%2.2d %s", $5, $mth, $3, $time);
        $max = $date if ($date gt $max);
        $x++ if ($last_date && $date le $last_date);

     } elsif ($l =~ /\[(\w+)\] \#\d+\: .*SN\#(\w+)/ ) {
         my $ecode = $1;
         my $sn    = $2;
         my $sev   = 0;
         $date = Util->get_today() if (!$date);
         push(@lines, "$date $l");
         print O "$date $w $l\n";    
     }
  }
  $cache->{"$w.start"} = &round($max);

  Debug->print2("Found " . ($#lines+1) . " entries in 3310/3510 log file.");
  close(O);

  if ($renv->{delete_3300_log}) {
    my($err,$com) = Util->run_command("$sccli $w clear events", "luxadm.txt" , $TO);
  }
  
  return (undef, \@lines);
}

sub round {
  my($d) = @_;

  my $secs = substr($d,11,2) * 3600 + substr($d,14,2) * 60 + substr($d,17);
  $secs += 2;

  my $h = int($secs / 3600);
  $secs -= $h * 3600;
  my $m = int($secs / 60);
  $secs -= $m * 60;

  my $d2 = substr($d, 0, 11) . sprintf("%2.2d:%2.2d:%2.2d\n", $h, $m, $secs);
  return $d2;
}



# sccli diag229 show port-wwns 
# sccli: selected se3000://172.20.67.229:58632 [SUN StorEdge 3510 SN#000003]
#Ch  Id   WWPN
#-------------------------
# 0  112  216000C0FF000003
# 1  113  226000C0FF100003
# 4  114  256000C0FF200003
# 5  115  266000C0FF300003


sub getPortWWNS {
  my($class, $device, $report) = @_;

  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;

  my $sccli = &sccli_path();
  my $w = $device->{path} || $device->{ipno};

  my($err,$com) = Util->run_command("$sccli $w show port-wwns", "luxadm.txt" , $TO);
  return $err if ($err);
  foreach my $l (@$com) {
     $l = Util->ltrim($l);
     if ($l =~ /(\d+)\s+(\d+)\s+(\w+)/) {
         my $port = $1;
         my $id   = $2;
         my $wwn  = $3;
         $report->{"port.$port.id"} = $id;
         $report->{"port.$port.wwn"} = lc($wwn);
     }
  }
  return undef;
}

#sccli diag229 show lun-maps 
#sccli: selected se3000://172.20.67.229:58632 [SUN StorEdge 3510 SN#000003]
#Ch Tgt LUN   ld/lv  ID-Partition  Assigned
#-------------------------------------------
# 0 112   0   ld0    7DE450A8-00   Primary  
# 0 112   3   ld0    7DE450A8-00   Primary  
# 1 113   0   ld1    29947214-00   Secondary
# 4 114   0   ld0    7DE450A8-00   Primary  
# 4 114   1   ld0    7DE450A8-00   Primary  


sub getLunMaps {
  my($class, $device, $report) = @_;

  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;


  my $sccli = &sccli_path();
  my $w = $device->{path} || $device->{ipno};

  my($err,$com) = Util->run_command("$sccli $w show lun-maps", "luxadm.txt" , $TO);
  return $err if ($err);
  my $lun = 0;
  foreach my $l (@$com) {
     $l = Util->ltrim($l);
     if ($l =~ /(\d+)\s+(\d+)\s+(\d+)\s+(\w+)\s+([^\s]+)\s+(\w+)/) {
         my $ch     = $1;
         my $target = $2;
         my $lun    = $3;
         my $ld     = $4;
         my $part   = $5;
         my $ass    = $6;
         $report->{"lun.$lun.channel"} = $ch;
         $report->{"lun.$lun.target"}  = $target;
         $report->{"lun.$lun.lun"}     = $lun;
         $report->{"lun.$lun.ld"}      = $ld;
         $report->{"lun.$lun.partition"}  = $part;
         $report->{"lun.$lun.assigned"}   = $ass;
         $lun++;
     }
  }
  return undef;
}
  


sub getConfig {
  my($class, $device, $report) = @_;

  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;

  my $sccli = &sccli_path();
  my $w = $device->{path} || $device->{ipno};

  my($err,$com, $std) = Util->run_command("$sccli $w show configuration -x", "luxadm.txt" , $TO);
  return $err if ($err);

  my $w2 = $w;
  $w2 =~ s/\W/_/g;
  open(OO, ">" . System->get_home() . "/DATA/tmp/$w2.xml");
  print OO join("\n", @$com);
  close(OO);

  if ($err || $#$com < 0) {
    return "$err $std";
  }
  while ($com->[$#$com] !~ /raidbaseview/ && $#$com > 10) {
       $#$com--;
  }
  my $in;
  require XML::LibXML;
  my $parser = XML::LibXML->new();
  my $dom;
  eval {
    $dom  = $parser->parse_string("@$com");
  };
  if ($@) {
     Debug->print2("Error parsing XML in 'show configuration': " . substr("@$com", 0, 100));
     return "Error parsing XML";
  }
  my $elem = $dom->getDocumentElement();
  my @raid = $elem->findnodes("raidsystem");
  my $raid1 = $raid[0];
  my %REP;
  &xml2report( \%REP, $raid1, "", 0);
  #
  # FIX THE SLOT INFO
  #
  my (%LD, %DISK_SLOT, $x, $top_disk, $disk_cnt);

  for ($x=0; $x <= 50; $x++) {
     if (exists $REP{"logical_drive.$x.idx"}) {
       $LD{$x} = $REP{"logical_drive.$x.idx"};
     }
  }

  for ($x=0; $x <= 100; $x++) {
     if (exists $REP{"components.disk.$x.target"}) {
       $DISK_SLOT{$x} = $REP{"components.disk.$x.target"};
       $top_disk = $DISK_SLOT{$x} if ($DISK_SLOT{$x} > $top_disk);
       $disk_cnt++;
     }
  }
  $report->{"info.top_disk"} = $top_disk;
  my @IN;
  foreach my $el (keys %REP) {
     if ($el =~ /^logical_drive\.(\d+)\.(.*)/) {
        my $d = $1;
        my $e = $2;
        $report->{"logical_drive.$LD{$d}.$e"} = $REP{$el};

     } elsif ($el =~ /^components\.disk\.(\d+)\.(.*)/) {
        my $d = $1;
        my $e = $2;
        $report->{"components.disk." . $DISK_SLOT{$d} . ".$e"} = $REP{$el};
        $IN[$DISK_SLOT{$d}+0] = 1;

     } else {
        $report->{$el} = $REP{$el};
     }
  }
  my ($in);
  for ($x=0; $x <= $top_disk; $x++) {
     $in .= $IN[$x] ? "1," : ",";
  }
  $report->{"info.disk_map"} = $in;
  return undef;   
}



# translate xml to snmp structure.

sub xml2report {
  my( $report, $node, $prefix, $level)  = @_;
  my(%MAP);
  my $prefix0 = "$prefix." if ($prefix);
  require XML::LibXML;

  foreach my $el ($node->getChildNodes()) {
    next if ($el->getType() != &XML::LibXML::ELEMENT_NODE() );
    my $name = $el->getName();
    my $pr;
    my @ch1 = $el->getChildNodes();

    if (index(",fru,channel,logical_volume,logical_drive,disk,partition,", ",$name,") >= 0) {
      $MAP{$name}++;
      $pr = "$prefix0$name." . $MAP{$name};
    } elsif ($level == 0 && $#ch1 == 0) {
      $pr = "info.$name";
    } else {
      $name = "components" if ($name eq "config_components");
      $pr = "$prefix0$name";
    }
    if ($#ch1 == 0) {
       $report->{$pr} = $el->textContent();
    } else {
       &xml2report($report, $el, $pr, $level+1);
    }
  }
}

# returns array of hashes

sub discover {
  my($class, $host, $args) = @_;
  my($rc);
  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;


  my $VAR1 = [];
  if (!$host) {
     return &get_discover($args);
  } else {
     my $rc = Util::Http->getCommand($host, "Agent::3310:discover&HTTP=1" , 
                                       $TO + 10);
     if (substr($rc,0,3) eq "ERR") {
        $ERR = $rc;
        return [];
     } else {
        eval substr($rc,3);
        return $VAR1;
     }
  }
}

sub get_discover {
  my($q) = @_;
 
  my @disco;
  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;


  my $sccli = &sccli_path();
  my $master = Util->findMaster();
  my $renv   = System->get_renv();
  my $hname = $master ? $renv->{hostname} : "";

  if (-x $sccli) {
     my ($err, $L) = Util->run_command("$sccli --list", "sccli", $TO);
     foreach my $l (@$L) {
        if ($l =~ /\/dev\/([^ ]+) /) {
           my $path = "/dev/$1";
           my $rc = &get_WWN({name => $path});
           $rc->{path} = $path;
           push(@disco, $rc);
           if ($q->{callback}) {
              my $f = $q->{callback};
              my $type = $rc->{type};
              my $name = "$type-" . substr($rc->{wwn},-4);
              my $line = "storage.$type|$type|$rc->{ip}|$rc->{wwn}|$rc->{key}||$path|$name|$rc->{userLabel}||$hname";
	      $line = "ERR|No Key found for $path $rc->{error}" if (!$rc->{key});
              &$f($line);
           }
        }
     }
  }
  if ($q->{HTTP}) {
    require Data::Dumper;
    print "OK " . Data::Dumper::Dumper(\@disco);
  } else {
    return \@disco;
  }
}

  

sub getInquiry {
  my($class, $device, $rep) = @_;

  my($renv) = System->get_renv();
  my($TO) = $renv->{'timeout.sccli'} || 400;

  my $sccli = &sccli_path();

  my $w = $device->{path} || $device->{ipno};

  my($err,$com) = Util->run_command("$sccli $w inquiry", "luxadm.txt" , $TO);
  return $err if ($err);
  return "ERR: $device->{name}/$w not responding to sccli inquiry" if ("@$com" =~ /device not found/ || $#$com < 0) ;

  foreach my $l (@$com) {
    if ($l =~ /([^\:]+): (.*)/) {
       my $name = Util->trim($1);
       my $value = Util->trim($2);
       $name =~ s/Page .. //;
       $name =~ s/ /_/g;
       $rep->{"info.$name"}  = $value;
    }
  }
  return undef;
}



sub FRUS {
  my($class, $r, $name) = @_;
  my($v) = $r->{_value};
   my @FRUS;
  my $devtype = $v->{"id.device_type"} || "3310";

  foreach my $el (sort keys %$v) {
     if ($el =~ /components.disk.(\d+).model/) {
        my $ix = $1;
        my $d = "components.disk.$ix";
        my $model  = $v->{"$d.model"};
        my $vendor = $v->{"$d.manufacturer"};
        my $serial = $v->{"$d.serial_number"};
        my $rev    = $v->{"$d.product_revision"};
        my $status = $v->{"$d.status"};

        push(@FRUS, [ $name ,$devtype, "disk", "disk.$ix",
             $vendor || " N/A", $model||" N/A", $serial||" N/A", $rev, $status]);
     }
  }
  return \@FRUS;
}

sub REPORT {
  my($class, $host, $r, $arg) = @_;

  my($v) = $r->{_value};
  my($out);
  my($tableCnt) = $arg->{tableCnt};
  my($host0) = $host || System->hostname();
  my $comm = ($r->{_status} eq "CC") ? "<font color=red>Communication Lost: Data may be old!</font>" : "";
 

  $out .= $class->reportHead($v->{'info.product'}, $r);
  my $cnt;
  $out .= "<tr> <td colspan=4><center><b>$v->{'info.Product'} $v->{'info.unique_id'}</td>";
  $out .= "<tr> <td colspan=4><b><center>$comm</td>";
  foreach my $el (sort keys %$v) {
     if ($el =~ /^info\.(.+)/)  {
       my $n = $1;
       next if (index(",Serial_Number,manufacturer,model,Vendor,name,sccli,unique_id,", ",$n,") >= 0);
       next if (index(",Product,disk_map,top_disk,Device_Type,", ",$n,") >= 0);
       $n =~ s/id_of_//;
       $n =~ s/^total_//;
       $n =~ s/_/ /g;
       $out .= "<tr>" if ($cnt++ % 2 == 0);
       my $v1 = $v->{$el};
       my $le = length($v1);
       if ($le > 20) {
         my $half = int($le/2);
         $v1 = "<small>" . substr($v1,0,$half) . " " . substr($v1,$half);
       }
       $out .= "<td bgcolor=$Style::LIGHT align=right>$n:</td>
                   <td>&nbsp;$v1</td>";
     }
  }
  $out .= "</table>";

# CHANNELS

  $out .= "<table border=1 cellspacing=0 width=95% bgcolor=white>
    <tr bgcolor=$Style::DARK><td colspan=7><font color=white><b>Channels</td>
    <tr bgcolor=$Style::LIGHT>
        <th>#
        <th>Port
        <th>Type
        <th>Media
        <th>Speed
        <th>Width
        <th>PID/SID
   ";
  my($x);
  my $port= 0;
  for ($x=1; $x <= 100; $x++) {
     last if (!$v->{"channel.$x.type"});
     my $mode = $v->{"channel.$x.mode"};
     my $p;
     if ($mode eq "Host") {
        $port++;
        $p = $port;
     }
     $out .=<<EOF;
        <tr><td>$v->{"channel.$x.idx"}</td>
         <td>&nbsp;$p</td>
         <td>$mode</td>
         <td><center>$v->{"channel.$x.type"}</td>
         <td><center>$v->{"channel.$x.curclk"} 
         <td><center>$v->{"channel.$x.defwid"}
         <td><center>$v->{"channel.$x.pid"}/$v->{"channel.$x.sid"}&nbsp;
EOF
  }
  $out .= "</table>";

######################
# DISKS

  my %DD;
  $out .= "<table border=1 cellspacing=0 width=95% bgcolor=white>
    <tr bgcolor=$Style::DARK><td colspan=6><font color=white><b>Disks</td>
    <tr bgcolor=$Style::LIGHT>
        <th>Ch-Id
        <th>Size
        <th>Status
        <th>Model / Serial#
        <th>Rev
   ";
  my $x;
  my $top_disk = $v->{"info.top_disk"};
  for ($x=0; $x <= $top_disk; $x++) {
     next if (!$v->{"components.disk.$x.model"});
     my $s = $v->{"components.disk.$x.capacity"};
     my $ch = $v->{"components.disk.$x.ch"};
     my $tg = $v->{"components.disk.$x.target"};
     $DD{"$ch.$tg"} = $v->{"components.disk.$x.model"} . " " . 
                      $v->{"components.disk.$x.serial_number"} . " $s";
     my $st = $v->{"components.disk.$x.status"};
     my $col = "bgcolor=#FFC0C0" if ($st =~ /Bad/);
     $s="N/A" if ($st =~ /Bad/);
                  
     $out .=<<EOF;
        <tr><td><center>$ch-$tg</td>
         <td align=right>$s&nbsp;</td>
         <td $col><center>$st</td>
         <td><small>$v->{"components.disk.$x.model"} / 
                    $v->{"components.disk.$x.serial_number"}&nbsp;
         <td>$v->{"components.disk.$x.product_revision"} 
EOF
  }
  $out .= "</table>";


# FRUS

  $out .= "<table border=1 cellspacing=0 width=95% bgcolor=white>
    <tr bgcolor=$Style::DARK><td colspan=6><font color=white><b>Frus</td>
    <tr bgcolor=$Style::LIGHT>
        <th>#
        <th>Name
        <th>Vendor/Model/Serial
        <th>Rev.
   ";
  my $x;
  for ($x=1; $x <= 100; $x++) {
     last if (!$v->{"fru.$x.model"});
     my $item = $v->{"fru.$x.description"};
     $out .=<<EOF;
        <tr>
              <td>$v->{"fru.$x.idx"}</td>
              <td><small>$item</td>
              <td><small>$v->{"fru.$x.vendor_jedec_id"}/ $v->{"fru.$x.model"}/ $v->{"fru.$x.serial_number"}</td>
              <td><center>$v->{"fru.$x.revision"}</td>
EOF
  }
  $out .= "</table>";

######################
# LOGICAL DRIVE

  $out .= "<table border=1 cellspacing=0 width=95% bgcolor=white>
    <tr bgcolor=$Style::DARK><td colspan=6><font color=white><b>Logical Drives</td>
    <tr bgcolor=$Style::LIGHT>
        <th>#
        <th>Role
        <th>Drives
        <th>Status
        <th>Size
        <th>Raid
   ";
  my ($x, $y);
  for ($x=0; $x <= 50; $x++) {
     next if (!$v->{"logical_drive.$x.status"});
     my $s = "&nbsp;"; # sprintf("%.3f GB", $v->{"logical_drive.$x.size"}/1024);

     $out .=<<EOF;
        <tr><td><b><center>ld$v->{"logical_drive.$x.idx"}
            <td><center>$v->{"logical_drive.$x.assignment"}
            <td><center>$v->{"logical_drive.$x.number_of_drives"}
            <td><center>$v->{"logical_drive.$x.status"}
            <td align=right>$s
            <td><center>$v->{"logical_drive.$x.raid_level"}
EOF
     my $details = &logical_drives($v, "logical_drive.$x", \%DD);
     my $parts   = &logical_partitions($v, "logical_drive.$x");

     $out .= "<tr><td>&nbsp;</td><td colspan=5>$details $parts</td>";
  }
  $out .= "</table>";

######################
# LOGICAL VOLUME

  $out .= "<table border=1 cellspacing=0 width=95% bgcolor=white>
    <tr bgcolor=$Style::DARK><td colspan=6><font color=white><b>Logical Volumes</td>
    <tr bgcolor=$Style::LIGHT>
        <th>#
        <th>Assignment
        <th>Status
        <th>Size
   ";
  my ($x, $y, $p, $d);
  for ($x=1; $x <= 2000; $x++) {
     last if (!exists $v->{"logical_volume.$x.idx"});
     my $idx = $v->{"logical_volume.$x.idx"};
     my $ass = $v->{"logical_volume.$x.assignment"};
     my $size= &size($v->{"logical_volume.$x.size"});
     my $status= $v->{"logical_volume.$x.status"};
     my $pre = "logical_volume.$x";
     $out .= "
        <tr><td><b><center><b>lv$idx</td>
            <td><center>$ass</td>
            <td>$status</td>
            <td align=right>$size</td>";

     my $parts   = &logical_partitions($v, "logical_volume.$x");
     my ($details, $dd);
     $details = "<table border=1 cellspacing=0 width=100%><tr bgcolor=$Style::LIGHT>
        <th>Drive#</th>
        <th>Role</th>
        <th>Drives</th>
        <th>Status</th>
        <th>Size</th>
        <th>Raid</th>";

     for ($dd=1; $dd <= 1000; $dd++) {
        last if (!exists $v->{"$pre.logical_drive.$dd.idx"});
        my $idx = $v->{"$pre.logical_drive.$dd.idx"};
        my $ass = $v->{"$pre.logical_drive.$dd.assignment"};
        my $cnt = $v->{"$pre.logical_drive.$dd.number_of_drives"};
        my $level = $v->{"$pre.logical_drive.$dd.raid_level"};
        my $status= $v->{"$pre.logical_drive.$dd.status"};
        my $size  = &size($v->{"$pre.logical_drive.$dd.size"});
        $details .= "<tr>
                   <td><b>ld$idx</td>
                   <td>$ass</td>
                   <td>$cnt</td>
                   <td>$status</td>
                   <td>$size</td>
                   <td>$level</td>";

        $details .= "<tr><td>&nbsp;</td><td colspan=5> " . 
                     &logical_drives($v, "$pre.logical_drive.$dd", \%DD) . "</td>";
     }
     $details .= "</table>";
     $out .= "<tr><td colspan=1>&nbsp;</td><td colspan=3>$parts $details</td>";
  }
  $out .= "</table>&nbsp;<p>&nbsp;<p>";

  return $out;

}



sub logical_drives {
  my($v, $pre, $DD) = @_;
  my $details;
  my @dd = split(/\s+/, $v->{"$pre.physical_drive"});

  foreach my $d (@dd) {
       my $d0 = $d; $d0 =~ s/\./-/;
       my $dinfo = 
       $details .= "&nbsp;[Disk $d0] <small>$DD->{$d}</small> <br>";
  }
  return $details;
}

sub logical_partitions {
  my($v, $pre) = @_;
  my ($details);
  my $part_cnt = $v->{"$pre.total_partitions"};
  my($y);
  for ($y=1; $y <= $part_cnt; $y++) {
       my $eff0 = $v->{"$pre.partition.$y.effective_size"};
       my $map  = $v->{"$pre.partition.$y.mapping"};
       my $eff  = sprintf("%.3f GB", $eff0 / 1024);
       my $idx  = $v->{"$pre.partition.$y.idx"};
       $details .= "&nbsp;Partition-$idx: [$eff] Map: $map<br>";
  }
  return $details;
}



sub size {
  my($v) = @_;
  return sprintf("%.3f GB", $v/1024);
}


1;

