#!/bin/ksh
#**********************************************************************#
#*                                                                    *#
#* Copyright (c) 2002 by Sun Microsystems, Inc.                       *#
#* All rights reserved.                                               *#
#*                                                                    *#
#**********************************************************************#

# Sun Cluster Data Services Builder template version 1.0
#
# Validate method for scmbm.
# ex: When the resource is being created command args will be
#   
# scmbm_validate -c -R <..> -G <...> -T <..> -r <sysdef-prop=value>... 
#       -x <extension-prop=value>.... -g <resourcegroup-prop=value>....
#
# when the resource property is being updated
#
# scmbm_validate -u -R <..> -G <...> -T <..> -r <sys-prop_being_updated=value>
#   OR
# scmbm_validate -u -R <..> -G <...> -T <..> -x <extn-prop_being_updated=value>
#

###############################################################################
# Parse program arguments.
#
function parse_args # [args ...]
{
	typeset opt

	while getopts 'cur:x:g:R:T:G:' opt
	do
                if [ "$EMP_DEBUG" = "ON" ]
                then
		  logger -p ${SYSLOG_FACILITY}.notice \
		  -t [$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME] \
		  "Validate method received option -$opt argument $OPTARG"
                fi

		case "$opt" in

		R)
		# Name of the scmbm resource.
		RESOURCE_NAME=$OPTARG
		;;

		G)
		# Name of the resource group in which the resource is
		# configured.
		RESOURCEGROUP_NAME=$OPTARG
		;;

		T)
		# Name of the resource type.
		RESOURCETYPE_NAME=$OPTARG
		;;

		r)      
		# We are not accessing any system defined property.
		# So, this is a no-op
		;;

		g)
		# This is a no-op as we are not bothered about 
		# Resource group properties
		;;

		c)
		# This is a no-op as this just a flag which indicates
		# that the validate method is being called while
		# creating the resource.
		;;

		u)
		# This is a flag to indicate the updating of property 
		# of an existing resource.
		UPDATE_PROPERTY=1
		;;

		x)
                if [ "$EMP_DEBUG" = "ON" ]
                then
		  logger -p ${SYSLOG_FACILITY}.notice \
		  -t [$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME] \
		  "Validate method received extended option $OPTARG"
                fi
                KEY=`echo $OPTARG | awk -F= '{print $1}'`
                VALUE=`echo $OPTARG | awk -F= '{print $2}'`
                case "$KEY" in
                  "Debug"|"DEBUG")
                  EMP_DEBUG=$VALUE
                  ;;
                  "EMP_BASEDIR")
                  EMP_BASEDIR=$VALUE
                  ;;
                  "EMP_ADMIN")
                  EMP_ADMIN=$VALUE
                  ;;
                esac
		;;

		*)
		logger -p ${SYSLOG_FACILITY}.err \
		-t [$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME] \
		"ERROR: Option $OPTARG unknown"
		exit 1
		;;

		esac
	done
}

###############################################################################
# MAIN
##############################################################################

DEBUG_PARMS="$*"

export PATH=/bin:/usr/bin:/usr/cluster/bin:/usr/sbin:/usr/proc/bin:$PATH

# Obtain the syslog facility to use. This will be used to log the messages.
SYSLOG_FACILITY=`scha_cluster_get -O SYSLOG_FACILITY`

UPDATE_PROPERTY=0

# Parse the arguments that have been passed to this method.
parse_args "$@"

SYSLOG_TAG=$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME

if [ "$EMP_DEBUG" = "ON" ]
then
	logger -p ${SYSLOG_FACILITY}.notice \
		-t [$SYSLOG_TAG] \
		"${ARGV0} Validate method for scmbm has been started with parameters $DEBUG_PARMS"
fi

if [[ -f /opt/SUNWscmbm/bin/hasp_check && \
	-x /opt/SUNWscmbm/bin/hasp_check ]]; then
	/opt/SUNWscmbm/bin/hasp_check "$@" >/dev/null 2>&1
	hasp_status=$?
        if [ "$EMP_DEBUG" = "ON" ]
        then
	        logger -p ${SYSLOG_FACILITY}.notice \
		        -t [$SYSLOG_TAG] \
		        "${ARGV0} Validate: hasp_check returns $hasp_status"
        fi
else
	# the binary doesn't exist so we cannot call it
	hasp_status=2
        if [ "$EMP_DEBUG" = "ON" ]
        then
	        logger -p ${SYSLOG_FACILITY}.notice \
		        -t [$SYSLOG_TAG] \
		        "${ARGV0} Validate: hasp_check not found status set to $hasp_status"
        fi
fi

case "$hasp_status" in

	1) 
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: Internal Error. Failed to check status of SUNW.HAStoragePlus resource."
	exit 1
	;;

	2)
        if [ "$EMP_DEBUG" = "ON" ]
        then
	  logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: This resource doesn't depend on any SUNW.HAStoragePlus resources. Proceeding with the normal checks."
        fi

	;;

	3)
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: One or more of the SUNW.HAStoragePlus resources that this resource depends on is in a different RG. Failing validate method configuration checks."
	exit 1
	;;

	4)
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: One or more of the SUNW.HAStoragePlus resources that this resource depends on is not online anywhere. Failing validate method."
	exit 1
	;;

	5)
        if [ "$EMP_DEBUG" = "ON" ]
        then
	  logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: All the SUNW.HAStoragePlus resources that this resource depends on are not online on the local node. Skipping the checks for the existence and permissions of the start/stop/probe commands."
        fi
	exit 0
	;;

	6)
        if [ "$EMP_DEBUG" = "ON" ]
        then
	  logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: All the SUNW.HAStoragePlus resources that this resource depends on are online on the local node. Proceeding with the checks for the existence and permissions of the start/stop/probe commands."
        fi
	;;

	*)
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: Unknown status code $hasp_status"
	;;

esac
if [ ! -z "$EMP_DEBUG" ]
then
	case $EMP_DEBUG in
	     ON|OFF)
	     ;;
	     *)
	        logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		      "${ARGV0} Validate: Debug option $EMP_DEBUG is invalid. Valid options are ON or OFF"
	        exit 1
	     ;;
	esac
fi

if [ ! -z "$EMP_BASEDIR" ]
then
  if [ ! -d "$EMP_BASEDIR" ]
  then
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: Directory EMP_BASEDIR=$EMP_BASEDIR is invalid"
	exit 1
  else
        if [ "$EMP_DEBUG" = "ON" ]
        then
	  logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: Directory EMP_BASEDIR=$EMP_BASEDIR is valid"
        fi
  fi

  start_cmd_prog="$EMP_BASEDIR/pack/scbin/mbm_start"
  if [[ ! -f $start_cmd_prog || ! -x $start_cmd_prog ]]; then
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: File $start_cmd_prog is missing or not executable"
	exit 1
  else
        if [ "$EMP_DEBUG" = "ON" ]
        then
	  logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: File $start_cmd_prog is OK"
        fi
  fi

  stop_cmd_prog="$EMP_BASEDIR/pack/scbin/mbm_stop"
  if [[ ! -z $stop_cmd_prog ]]; then
	if [[ ! -f $stop_cmd_prog || ! -x $stop_cmd_prog ]]; then
		logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
			"${ARGV0} Validate: File $stop_cmd_prog is missing or not executable"
		exit 1
        else
                if [ "$EMP_DEBUG" = "ON" ]
                then
	          logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		        "${ARGV0} Validate: File $stop_cmd_prog is OK"
                fi
	fi
  fi

  probe_cmd_prog="$EMP_BASEDIR/pack/scbin/mbm_probe"
  if [[ ! -z $probe_cmd_prog ]]; then
	if [[ ! -f $probe_cmd_prog || ! -x $probe_cmd_prog ]]; then
		logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
			"${ARGV0} Validate: File $probe_cmd_prog is missing or not executable"
		exit 1
        else
                if [ "$EMP_DEBUG" = "ON" ]
                then
	          logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		        "${ARGV0} Validate: File $probe_cmd_prog is OK"
                fi
	fi
  fi

  batchenv_path="$EMP_BASEDIR/batchenv"
  if [[ ! -z $batchenv_path ]]; then
	if [[ ! -f $batchenv_path ]]; then
		logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
			"${ARGV0} Validate: File $batchenv_path is missing"
		exit 1
        else
                if [ "$EMP_DEBUG" = "ON" ]
                then
	          logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		        "${ARGV0} Validate: File $batchenv_path is OK"
                fi
	fi
  fi

else
  if [ "$EMP_DEBUG" = "ON" ]
  then
    logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
	"${ARGV0} Validate: Directory EMP_BASEDIR=$EMP_BASEDIR has not been updated"
  fi
fi

if [ ! -z "$EMP_ADMIN" ]
then
  su "$EMP_ADMIN" -c echo "Hello" >/dev/null 2>&1
  if [ $? -ne 0 ]
  then
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: user EMP_ADMIN=$EMP_ADMIN is invalid"
	exit 1
  else
        if [ "$EMP_DEBUG" = "ON" ]
        then
	  logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		"${ARGV0} Validate: user EMP_ADMIN=$EMP_ADMIN is OK"
        fi
  fi
else
  if [ "$EMP_DEBUG" = "ON" ]
  then
    logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
	"${ARGV0} Validate: user EMP_ADMIN=$EMP_ADMIN has not been updated"
  fi
fi

# User added code -- BEGIN vvvvvvvvvvvvvvv
# User added code -- END   ^^^^^^^^^^^^^^^

# Log a message indicating that validate method was successful.
if [ "$EMP_DEBUG" = "ON" ]
then
  logger -p ${SYSLOG_FACILITY}.notice \
	-t [$SYSLOG_TAG] \
	"${ARGV0} Validate method for resource $RESOURCE_NAME completed successfully"
fi

exit 0
