#!/bin/ksh
#**********************************************************************#
#*                                                                    *#
#* Copyright (c) 2002 by Sun Microsystems, Inc.                       *#
#* All rights reserved.                                               *#
#*                                                                    *#
#**********************************************************************#

# Sun Cluster Data Services Builder template version 1.0
#
# Stop method for scmbm
#

###############################################################################
# Parse program arguments.
#
function parse_args # [args ...]
{
	typeset opt

	while getopts 'R:G:T:' opt
	do
		case "$opt" in
		R)
		# Name of the scmbm resource.
		RESOURCE_NAME=$OPTARG
		;;

		G)
		# Name of the resource group in which the resource is
		# configured.
		RESOURCEGROUP_NAME=$OPTARG
		;;

		T)
		# Name of the resource type.
		RESOURCETYPE_NAME=$OPTARG
		;;

		*)
		logger -p ${SYSLOG_FACILITY}.err \
		-t [$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME] \
		"ERROR: Option $OPTARG unknown"
		exit 1
		;;

		esac
	done

}

##############################################################################
# MAIN
##############################################################################
DEBUG_PARMS="$*"

export PATH=/bin:/usr/bin:/usr/cluster/bin:/usr/sbin:/usr/proc/bin:$PATH

# Obtain the syslog facility to use. This will be used to log the messages.
SYSLOG_FACILITY=`scha_cluster_get -O SYSLOG_FACILITY`

# Parse the arguments that have been passed to this method.
parse_args "$@"

# get the Timeout value allowed for stop method from the RTR file
STOP_TIMEOUT=`scha_resource_get -O STOP_TIMEOUT -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME`

# We will try to wait wait for 65% of the stop_method_timeout value when we
# send a SIGTERM through PMF to the Data service. This is to make sure that
# the application stops in a decent manner. If the application does not
# respond favourably to this then we use SIGKILL to stop the data service
# and this will be done for a 15% of the Stop_method_timeout value. However,
# if the data service has not stopped by now, we conclude that there was a
# Failure in the stop method and exit non-zero. The remaining 5% of the 
# stop method timeout is for other needs. 
((SMOOTH_TIMEOUT=$STOP_TIMEOUT * 65/100))
((BAM_STOP_TIMEOUT=$STOP_TIMEOUT * 50/100))
((TERM_SMOOTH_TIMEOUT=$STOP_TIMEOUT * 15/100))
((BAM_CLEAN_TIMEOUT=$STOP_TIMEOUT * 15/100))

((HARD_TIMEOUT=$STOP_TIMEOUT * 15/100))

PMF_TAG=$RESOURCEGROUP_NAME,$RESOURCE_NAME,0.svc
SYSLOG_TAG=$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME

emp_debug_info="`scha_resource_get -O Extension -R $RESOURCE_NAME \
        -G $RESOURCEGROUP_NAME Debug`"
EMP_DEBUG=`echo $emp_debug_info | awk '{print $2}'`
export EMP_DEBUG

# We need to know the full path for the gethostnames utility which resides
# in the directory <RT_BASEDIR>. Get this from the RT_BASEDIR property of the
# resource type.
RT_BASEDIR=`scha_resource_get -O RT_BASEDIR -R $RESOURCE_NAME \
        -G $RESOURCEGROUP_NAME`

# We need to know the full path for the mbm utilities which reside in the
# directory <EMP_BASEDIR>. Get this from the EMP_BASEDIR property of the
# resource type.
emp_basedir_info="`scha_resource_get -O Extension -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME EMP_BASEDIR`"
EMP_BASEDIR=`echo $emp_basedir_info | awk '{print $2}'`

EMP_SETUP=$EMP_BASEDIR/batchenv

# We need to know the mbm administrator name to start the mbm utilities
# with his/her user id. Get this from the EMP_ADMIN property of the
# resource type.
emp_admin_info="`scha_resource_get -O Extension -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME EMP_ADMIN`"
EMP_ADMIN=`echo $emp_admin_info | awk '{print $2}'`


hostnames=`$RT_BASEDIR/gethostnames -R $RESOURCE_NAME -G $RESOURCEGROUP_NAME \
        -T $RESOURCETYPE_NAME`

EMP_RESOURCE_ID="type $RESOURCETYPE_NAME, group $RESOURCEGROUP_NAME, name $RESOURCE_NAME, host $hostnames"; export EMP_RESOURCE_ID

stop_cmd_args="$EMP_BASEDIR/pack/scbin/mbm_stop -S $EMP_SETUP -D $EMP_DEBUG -I \"$EMP_RESOURCE_ID\" -L \"$SYSLOG_TAG\" -F \"$SYSLOG_FACILITY\" >/dev/null 2>&1"

stop_cmd_prog="$EMP_BASEDIR/pack/scbin/mbm_stop"

typeset -i SEND_KILL=0
typeset -i DO_CLEANUP=0

if [ "$EMP_DEBUG" = "ON" ]
then
	logger -p ${SYSLOG_FACILITY}.notice \
		-t [$SYSLOG_TAG] \
		"${ARGV0} Starting stop method for scmbm with parameters $DEBUG_PARMS"
fi
# User added code -- BEGIN vvvvvvvvvvvvvvv
# User added code -- END   ^^^^^^^^^^^^^^^

pmfadm -q $PMF_TAG >/dev/null 2>&1
if [[ $? == 0 ]]; then
	if [[ -f $stop_cmd_prog && -x $stop_cmd_prog ]]; then
                if [ "$EMP_DEBUG" = "ON" ]
                then
	               logger -p ${SYSLOG_FACILITY}.notice \
		               -t [$SYSLOG_TAG] \
		               "${ARGV0} Stop method scmbm still registered"
                fi
		# User added code -- BEGIN vvvvvvvvvvvvvvv
		# User added code -- END   ^^^^^^^^^^^^^^^

		pmfadm -s $PMF_TAG >/dev/null 2>&1
		if [[ $? != 0  ]]; then
			logger -p ${SYSLOG_FACILITY}.warn \
				-t [$SYSLOG_TAG] \
				"${ARGV0} Failed to take scmbm out of PMF control; trying to stop using custom stop command"
			# execute the user specified stop_cmd using hatimerun
			if [ "$EMP_DEBUG" = "ON" ]
			then
			        logger -p ${SYSLOG_FACILITY}.notice \
				        -t [$SYSLOG_TAG] \
					"${ARGV0} Stop method executes su $EMP_ADMIN -c hatimerun -k KILL -t $BAM_STOP_TIMEOUT $stop_cmd_args"
			fi
			su $EMP_ADMIN -c "hatimerun -k KILL -t $BAM_STOP_TIMEOUT $stop_cmd_args" >/dev/null 2>&1
			if [[ $? != 0 ]]; then
			        logger -p ${SYSLOG_FACILITY}.warn \
			                -t [$SYSLOG_TAG] \
				        "${ARGV0} Failed to stop scmbm using the custom stop command; trying SIGKILL now."
			fi
			SEND_KILL=1
		else
			# User added code -- BEGIN vvvvvvvvvvvvvvv
			# User added code -- END   ^^^^^^^^^^^^^^^

			# execute the user specified stop_cmd using hatimerun
                        if [ "$EMP_DEBUG" = "ON" ]
                        then
	                        logger -p ${SYSLOG_FACILITY}.notice \
		                        -t [$SYSLOG_TAG] \
		                        "${ARGV0} Stop method executes su $EMP_ADMIN -c hatimerun -k KILL -t $BAM_STOP_TIMEOUT $stop_cmd_args"
                        fi
			su $EMP_ADMIN -c "hatimerun -k KILL -t $BAM_STOP_TIMEOUT $stop_cmd_args" >/dev/null 2>&1
			if [[ $? != 0 ]]; then
				logger -p ${SYSLOG_FACILITY}.warn \
					-t [$SYSLOG_TAG] \
					"${ARGV0} Failed to stop scmbm using the custom stop command; trying SIGTERM/SIGKILL now."
                                DO_CLEANUP=1
			fi

			# Regardless of whether the command succeeded or not we
			# send TERM signal to the pmf tag. This will ensure
			# that the process tree goes away if it still exists.
			# If it doesn't exist by then, we return NOERR.

		        # User added code -- BEGIN vvvvvvvvvvvvvvv
		        # User added code -- END   ^^^^^^^^^^^^^^^

		        # Send a SIGTERM signal to the Data service and wait for
		        # 15% of the total timeout value.
                        if [ "$EMP_DEBUG" = "ON" ]
                        then
	                        logger -p ${SYSLOG_FACILITY}.notice \
		                        -t [$SYSLOG_TAG] \
		                        "${ARGV0} Stop method executes pmfadm -s $PMF_TAG -w $TERM_SMOOTH_TIMEOUT TERM"
                        fi
		        pmfadm -s $PMF_TAG -w $TERM_SMOOTH_TIMEOUT TERM >/dev/null 2>&1
		        # We compare the exit status of pmfadm to be greater than 2
		        # because "exit status = 1" means nametag doesn't exist, which
		        # is a OK, for the stop method has to be idempotent.
		        if [[ $? -ge 2 ]]; then 
			        logger -p ${SYSLOG_FACILITY}.warn \
				        -t [$SYSLOG_TAG] \
				        "${ARGV0} Failed to stop scmbm with SIGTERM; retry with SIGKILL"
			        SEND_KILL=1;
		        fi

			# Regardless of whether the command succeeded or not we
			# send KILL signal to the pmf tag. This will ensure
			# that the process tree goes away if it still exists.
			# If it doesn't exist by then, we return NOERR.
			SEND_KILL=1
		fi
	else
		# User added code -- BEGIN vvvvvvvvvvvvvvv
		# User added code -- END   ^^^^^^^^^^^^^^^

		# Send a SIGTERM signal to the Data service and wait for
		# 65% of the total timeout value.
                if [ "$EMP_DEBUG" = "ON" ]
                then
	                logger -p ${SYSLOG_FACILITY}.notice \
		                -t [$SYSLOG_TAG] \
		                "${ARGV0} Stop method executes pmfadm -s $PMF_TAG -w $SMOOTH_TIMEOUT TERM"
                fi
		pmfadm -s $PMF_TAG -w $SMOOTH_TIMEOUT TERM >/dev/null 2>&1
		# We compare the exit status of pmfadm to be greater than 2
		# because "exit status = 1" means nametag doesn't exist, which
		# is a OK, for the stop method has to be idempotent.
		if [[ $? -ge 2 ]]; then 
			logger -p ${SYSLOG_FACILITY}.warn \
				-t [$SYSLOG_TAG] \
				"${ARGV0} Failed to stop scmbm with SIGTERM; retry with SIGKILL"
			SEND_KILL=1;
		fi
	fi

	if [[ $SEND_KILL == 1 ]]; then 
		# User added code -- BEGIN vvvvvvvvvvvvvvv
		# User added code -- END   ^^^^^^^^^^^^^^^

		# Since the Data service did not stop with a SIGTERM we will
		# use a SIGKILL now and wait for another 15% of total timeout.
                if [ "$EMP_DEBUG" = "ON" ]
                then
	                logger -p ${SYSLOG_FACILITY}.notice \
		                -t [$SYSLOG_TAG] \
		                "${ARGV0} Stop method executes pmfadm -s $PMF_TAG -w $HARD_TIMEOUT KILL"
                fi
		pmfadm -s $PMF_TAG -w $HARD_TIMEOUT KILL >/dev/null 2>&1
		# We compare the exit status of pmfadm to be greater than 2
		# because "exit status = 1" means nametag doesn't exist, which
		# is a OK, for the stop method has to be idempotent.
		if [[ $? -ge 2 ]]; then
			logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
			"${ARGV0} Failed to stop scmbm; exiting UNSUCCESFUL"
			exit 1
		fi

		if [[ $DO_CLEANUP == 1 ]]; then 
			# execute the user specified stop_cmd using hatimerun 
			# for clean-up purposes.
			if [ "$EMP_DEBUG" = "ON" ]
			then
				logger -p ${SYSLOG_FACILITY}.notice \
					-t [$SYSLOG_TAG] \
					"${ARGV0} Stop method executes su $EMP_ADMIN -c hatimerun -k KILL -t $BAM_CLEAN_TIMEOUT $stop_cmd_args again"
			fi
			su $EMP_ADMIN -c "hatimerun -k KILL -t $BAM_CLEAN_TIMEOUT $stop_cmd_args" >/dev/null 2>&1
			if [[ $? != 0 ]]; then
				logger -p ${SYSLOG_FACILITY}.warn \
					-t [$SYSLOG_TAG] \
					"${ARGV0} Failed to clean up scmbm using the custom stop command; trying SIGTERM/SIGKILL now."
			fi
		fi
	fi
else 
	# The Data service is not running as of now. Log a message and 
	# exit suceess.
        if [ "$EMP_DEBUG" = "ON" ]
        then
          logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] \
		"scmbm found not running"
        fi

	# Even if the scmbm is not running, we exit success, for idempotency of
	# the STOP method.
	exit 0
fi

# User added code -- BEGIN vvvvvvvvvvvvvvv
# User added code -- END   ^^^^^^^^^^^^^^^

# could successfully stop scmbm. Log a message and exit success.
if [ "$EMP_DEBUG" = "ON" ]
then
  logger -p ${SYSLOG_FACILITY}.notice -t [$SYSLOG_TAG] "scmbm successfully stopped"
fi
exit 0
