#!/bin/ksh
#**********************************************************************#
#*                                                                    *#
#* Copyright (c) 2002 by Sun Microsystems, Inc.                       *#
#* All rights reserved.                                               *#
#*                                                                    *#
#**********************************************************************#

# Sun Cluster Data Services Builder template version 1.0
#
# Start Method for scmbm.
#
# The data service is started under the control of PMF. Prior to starting the 
# process(es) for scmbm some sanity checks are done. The PMF tries to start the
# service specified number of times and if the number of attempts exceeds this 
# value within a specified interval of time the PMF reports a failure to start
# the service. The number of times to retry and the interval in which it is to
# be tried are both properties of the resource set in RTR file. 

###############################################################################
# Parse program arguments.
#
function parse_args # [args ...]
{
	typeset opt

	while getopts 'R:G:T:' opt
	do
		case "$opt" in

		R)
		# Name of the scmbm resource.
		RESOURCE_NAME=$OPTARG
		;;

		G)
		# Name of the resource group in which the resource is
		# configured.
		RESOURCEGROUP_NAME=$OPTARG
		;;

		T)
		# Name of the resource type.
		RESOURCETYPE_NAME=$OPTARG
		;;

		*)
		logger -p ${SYSLOG_FACILITY}.err -t \
		[$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME] \
		"ERROR: Option $OPTARG unknown"
		exit 1
		;;

		esac
	done

}

###############################################################################
# MAIN
##############################################################################
DEBUG_PARMS="$*"

export PATH=/bin:/usr/bin:/usr/cluster/bin:/usr/sbin:/usr/proc/bin:$PATH

# Obtain the syslog facility to use. This will be used to log the messages.
SYSLOG_FACILITY=`scha_cluster_get -O SYSLOG_FACILITY`

# Parse the arguments that have been passed to this method
parse_args "$@"

PMF_TAG=$RESOURCEGROUP_NAME,$RESOURCE_NAME,0.svc
SYSLOG_TAG=$RESOURCETYPE_NAME,$RESOURCEGROUP_NAME,$RESOURCE_NAME

emp_debug_info="`scha_resource_get -O Extension -R $RESOURCE_NAME \
        -G $RESOURCEGROUP_NAME Debug`"
EMP_DEBUG=`echo $emp_debug_info | awk '{print $2}'`

# We need to know the full path for the gethostnames utility which resides
# in the directory <RT_BASEDIR>. Get this from the RT_BASEDIR property of the
# resource type.
RT_BASEDIR=`scha_resource_get -O RT_BASEDIR -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME`

# We need to know the full path for the mbm utilities which reside in the
# directory <EMP_BASEDIR>. Get this from the EMP_BASEDIR property of the
# resource type.
emp_basedir_info="`scha_resource_get -O Extension -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME EMP_BASEDIR`"
EMP_BASEDIR=`echo $emp_basedir_info | awk '{print $2}'`

EMP_SETUP=$EMP_BASEDIR/batchenv

# We need to know the mbm administrator name to start the mbm utilities
# with his/her user id. Get this from the EMP_ADMIN property of the
# resource type.
emp_admin_info="`scha_resource_get -O Extension -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME EMP_ADMIN`"
EMP_ADMIN=`echo $emp_admin_info | awk '{print $2}'`

emp_child_info="`scha_resource_get -O Extension -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME Child_mon_level`"
EMP_CHILD_MON_LEVEL=`echo $emp_child_info | awk '{print $2}'`

if [ $EMP_CHILD_MON_LEVEL -eq -1 ]
then
  CHILD_LEVEL=" "
else
  CHILD_LEVEL="-C $EMP_CHILD_MON_LEVEL"
fi

# get the Timeout value allowed for start method from the RTR file
START_TIMEOUT=`scha_resource_get -O START_TIMEOUT -R $RESOURCE_NAME \
            -G $RESOURCEGROUP_NAME`
((WAIT_START_TIMEOUT=$START_TIMEOUT * 10/100))

hostnames=`$RT_BASEDIR/gethostnames -R $RESOURCE_NAME -G $RESOURCEGROUP_NAME \
	-T $RESOURCETYPE_NAME`

EMP_RESOURCE_ID="type $RESOURCETYPE_NAME, group $RESOURCEGROUP_NAME, name $RESOURCE_NAME, host $hostnames"; export EMP_RESOURCE_ID

start_cmd_args="$EMP_BASEDIR/pack/scbin/mbm_start -S $EMP_SETUP -D $EMP_DEBUG -I \"$EMP_RESOURCE_ID\" -L \"$SYSLOG_TAG\" -F \"$SYSLOG_FACILITY\" >/dev/null 2>&1"

start_cmd_prog=`echo $start_cmd_args | nawk '{print $1}'`

if [[ ! -f $start_cmd_prog || ! -x $start_cmd_prog ]]; then
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} File $start_cmd_prog is missing or not executable"
	exit 1
fi

probe_cmd_args="$EMP_BASEDIR/pack/scbin/mbm_probe -S $EMP_SETUP -D $EMP_DEBUG -I \"$PMF_TAG\" -L \"$SYSLOG_TAG\" -F \"$SYSLOG_FACILITY\" >/dev/null 2>&1"

probe_cmd_prog="$EMP_BASEDIR/pack/scbin/mbm_probe"

if [[ ! -f $probe_cmd_prog || ! -x $probe_cmd_prog ]]; then
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} File $probe_cmd_prog is missing or not executable"
	exit 1
fi

# Obtain the time-out value allowed for the probe. For nslookup the default 
# time-out is 1.5 minutes. This value is set in the extension property 
# PROBE_TIMEOUT of the data service.
probe_timeout_info=`scha_resource_get -O Extension -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME Probe_timeout`
PROBE_TIMEOUT=`echo $probe_timeout_info | awk '{print $2}'`

# Get the value for pmf specific retry count from the RTR file.
emp_retry_cnt=`scha_resource_get -O Extension -R $RESOURCE_NAME \
	-G $RESOURCEGROUP_NAME PMF_Retry_Count`
RETRY_CNT=`echo $emp_retry_cnt | awk '{print $2}'`

# Get the value for retry interval from the RTR file. The value for the 
# RETRY_INTERVAL in the RTR file will be in seconds. Convert this value from 
# seconds to minutes for passing on to pmfadm. Note that this is necessarily
# a conversion with round-up, eg. 59 seconds --> 1 minute.
((RETRY_INTRVAL = (`scha_resource_get -O Retry_Interval -R $RESOURCE_NAME -G \
	$RESOURCEGROUP_NAME` + 59) / 60 ))

if [ "$EMP_DEBUG" = "ON" ]
then
	logger -p ${SYSLOG_FACILITY}.notice \
		-t [$SYSLOG_TAG] \
		"${ARGV0} Starting start method for scmbm resource with parameters $DEBUG_PARMS"
fi
# User added code -- BEGIN vvvvvvvvvvvvvvv
# User added code -- END   ^^^^^^^^^^^^^^^

# start the daemon under the control of Sun Cluster Process Monitor
# Facility. Let it crash and restart up to $RETRY_COUNT times in a period of 
# $RETRY_INTERVAL; if it crashes more often than that, the process monitor
# facility will cease trying to restart it
if [ "$EMP_DEBUG" = "ON" ]
then
	logger -p ${SYSLOG_FACILITY}.notice \
		-t [$SYSLOG_TAG] \
		"${ARGV0} Start method executes pmfadm -c $PMF_TAG -n $RETRY_CNT -t $RETRY_INTRVAL $CHILD_LEVEL su $EMP_ADMIN -c \"$start_cmd_args\""
fi
pmfadm -c $PMF_TAG -n $RETRY_CNT -t $RETRY_INTRVAL $CHILD_LEVEL su $EMP_ADMIN -c "$start_cmd_args" >/dev/null 2>&1

if [ "$EMP_DEBUG" = "ON" ]
then
        logger -p ${SYSLOG_FACILITY}.notice \
                -t [$SYSLOG_TAG] \
                "${ARGV0} Start method waiting $WAIT_START_TIMEOUT seconds"
fi
sleep $WAIT_START_TIMEOUT

pmfadm -q $PMF_TAG >/dev/null 2>&1
pmfadm_status=$?
if [ "$EMP_DEBUG" = "ON" ]
then
	logger -p ${SYSLOG_FACILITY}.notice \
		-t [$SYSLOG_TAG] \
		"${ARGV0} Start method \"pmfadm -q $PMF_TAG\" returns $pmfadm_status"
fi

typeset -i probe_status=1
while [ $probe_status -ne 0 ]
do
    sleep 5
    if [ "$EMP_DEBUG" = "ON" ]
    then
	    logger -p ${SYSLOG_FACILITY}.notice \
		    -t [$SYSLOG_TAG] \
		    "${ARGV0} Start method executes su $EMP_ADMIN -c hatimerun -t $PROBE_TIMEOUT $probe_cmd_args"
    fi
    su $EMP_ADMIN -c "hatimerun -t $PROBE_TIMEOUT $probe_cmd_args" >/dev/null 2>&1
    probe_status=$?
    if [ "$EMP_DEBUG" = "ON" ]
    then
	    logger -p ${SYSLOG_FACILITY}.notice \
		    -t [$SYSLOG_TAG] \
		    "${ARGV0} Probe $probe_cmd_args returns $probe_status"
    fi
    pmfadm -q $PMF_TAG >/dev/null 2>&1
    pmfadm_status=$?
    if [ $pmfadm_status -ne 0 ]
    then
	if [ "$EMP_DEBUG" = "ON" ]
	then
		logger -p ${SYSLOG_FACILITY}.notice \
			-t [$SYSLOG_TAG] \
			"${ARGV0} Start method \"pmfadm -q $PMF_TAG\" returns $pmfadm_status"
	fi
	logger -p ${SYSLOG_FACILITY}.err -t [$SYSLOG_TAG] \
		"${ARGV0} Failed to stay up. PMF status $pmfadm_status"
	exit $pmfadm_status
    fi
done

# User added code -- BEGIN vvvvvvvvvvvvvvv
# User added code -- END   ^^^^^^^^^^^^^^^

if [ "$EMP_DEBUG" = "ON" ]
then
	logger -p ${SYSLOG_FACILITY}.notice \
		-t [$SYSLOG_TAG] \
		"${ARGV0} Start method exit with status $probe_status"
fi
exit $probe_status
