#ifndef csICalendarDatabase_h___
#define csICalendarDatabase_h___

/*
 * This file presents the calendar database interface a CSAPI database plugin would need to
 * implement to provide a different database implementation.
 *
 * The included file, caldb.h, represents the structures and typedefs that these implementations
 * need to care and know about.  Please see that file for more pertinent information.
 */

#include "nsISupports.h"
#include "caldb.h"
#include "caldbapi.h"

/**
 * This is the abstract interface to the calendar database. It has been designed 
 * as a CSAPI interface so that it can be implemented using multiple database 
 * technologies. Every effort has been made to hide database-technology-specific
 * capabitilites. The interface is focused on maintenance of persistent calendar
 * data.
 */
class csICalendarDatabase : public nsISupports 
{

public:
    /**
     * Always gets caled after the object is created
     *
     * @param aServer
     * @return NS_OK if successful
     */

    NS_IMETHOD Init(
      nsISupports * aServer) = 0;

    /**
     * Close down the database subsystem.
     * @param pDBH  handle to the database
     * @return 0    if successful
     */

    NS_IMETHOD_(int) CloseDB(CALDB_HANDLE pDBH) = 0;

    /**
     * CreateCalProps
     * 
     * Create the CalProps entry for the specified relativeCalID.
     * @param pDBH    handle to the calendar
     * @param psRCID  relative calendar identifier
     * @param pp      newly created calprops struct
     * @return 0 if successful
     */

    NS_IMETHOD_(int) CreateCalProps(
      CALDB_HANDLE pDBH,
      char* psRCID, 
      CALDB_CALPROPS** pp) = 0;

    /**
     * Delete the CalProps entry for the specified relativeCalID.
     *
     * @param pDBH    handle to the calendar db
     * @param psRelativeCalID - relative calendar identifier
     * @return 0 if successful
     */

    NS_IMETHOD_(int) DeleteCalProps(
      CALDB_HANDLE pDBH,
      char* psRelativeCalID) = 0;

    /**
     * DeleteDB
     * 
     * @description   Delete the entire database. This means all log files and 
     *                any files associated with the database.
     * @param psURL   pointer to the root location of this db. NULL means use
     *                the default location.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) DeleteDB(char* psURL) = 0;

    /**
     * Delete a specific event identified by the supplied UID and recurrence id.
     *
     * @param pDBH    handle to the calendar db
     * @param pCalProps calendar properties
     * @param psUID     calendar uid
     * @param rid       calendar recurrence id
     * @param flags     If CALDB_FLAG_TOMBSTONE is present the event
     *                  is marked as a tombstone.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) DeleteEvent(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      char* psUID, 
      time_t rid, 
      PRUint32 flags, CALDB_COMPONENT* pComp) = 0;

    /**
     * Delete a specific component identified by the supplied UID and recurrence id.
     *
     * @param pDBH    handle to the calendar db
     * @param pCalProps calendar properties
     * @param iType     CALDB_COMP_EVENT or CALDB_COMP_TODO
     * @param psUID     calendar uid
     * @param rid       calendar recurrence id
     * @param flags     If CALDB_FLAG_TOMBSTONE is present the event
     *                  is marked as a tombstone.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) DeleteComponent(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      PRInt32 iType, 
      char* psUID, 
      time_t rid, 
      PRUint32 flags, CALDB_COMPONENT* pComp) = 0;

    /**
     * Read the specific event identified by the supplied uid and recurrence id.
     * 
     * @param pDBH       handle to the calendar db
     * @param pCalProps  calendar properties
     * @param psUID      calendar uid
     * @param rid        calendar recurrence id
     * @param flags      if CALDB_FLAG_TOMBSTONE is present, events marked as 
     *                   tombstones are read included in the returned events. 
     *                   Otherwise, tombstone events are ignored.
     * @param ppEvent    the found event. This event structure is allocated in ReadEvent.
     *                   Call FreeEvent to free the memory associated with this event.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) ReadEvent(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      char* psUID, 
      time_t rid, 
      PRUint32 flags, 
      CALDB_VEVENT** ppEvent) = 0;

    /**
     * Generalized routine to read or delete events and todos
     * 
     * @param pDBH       handle to the calendar db
     * @param pCalProps  calendar properties
     * @param iQuery     which query
     * @param pData      the data describing the conditions
     * @param ppCompList pointer to a list of component pointers. May be
     *                   passed in as a null when a delete query is specified.
     *                   De-allocated by a call to an be FreeComponentList
     * @return 0 if successful
     */

    NS_IMETHOD QueryComponents(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      PRInt32 iQuery, 
      CALDB_COMPQUERYDATA* pData, 
      array_t** ppCompList) = 0;

    /**
     * ReadCalProps
     * 
     * @description - Fetch the calprops entry for the supplied relative calid
     * 
     * @param pDBH       handle to the calendar db
     * @param psRelativeCalID - calendar relative calid
     * @param ppCalProps      - calendar properties
     * @return 0 if successful
     */

    NS_IMETHOD_(int) ReadCalProps(
      CALDB_HANDLE pDBH,
      char* psRelativeCalID, 
      CALDB_CALPROPS** ppCalProps) = 0;
    

    /**
     * Place a link in one calendar to an event in another calendar.
     * 
     * @param pDBH       handle to the calendar db
     * @param pSource        source calendar
     * @param pUID           unique id of the event to link
     * @param dtRecurrencID  recurrence id of the event to link
     * @param pDestCal       calendar in wich to insert event
     * @return 0 if successful
     */

    NS_IMETHOD_(int) LinkEvent(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pSource, 
      char* pUID, 
      time_t dtRecurrenceID, 
      CALDB_CALPROPS* pDestCal,
      PRUint32 flags) = 0;
    
    /**
     * Generic call to open the database subsystem.
     *
     * @param flags          flags to control the db subsystem. Currently
     *                       there are flags to control whether or not the
     *                       deadlock detection thread and the checkpoint
     *                       thread are started.
     * @param DBOpenFlags    generic database flags (database technology specific).
     * @param pURL           A url for the db subsystem
     * @param ppDBH          on a successful open, the returned handle to 
     *                       the calendar db
     * @return 0 if successful
     */

    NS_IMETHOD_(int) OpenDB(
      PRUint32 flags, 
      PRUint32 DBOpenFlags, 
      char* pURL,
      CALDB_HANDLE* ppDBH
      ) = 0;
    
    /**
     * Generic call determine the database version.
     * 
     * @param pDBH       handle to the calendar db
     * @param psBuf     user string buffer
     * @param iBufSize  size of the buffer
     * @return 0 if successful
     */

    NS_IMETHOD_(int) GetDBVersion(
      CALDB_HANDLE pDBH,
      char* psBuf, 
      PRUint32 iBufSize) = 0;
    
    /**
     * Generic call determine the database subsystem (ie, the code) version.
     * 
     * @param pDBH       handle to the calendar db
     * @param psBuf     user string buffer
     * @param iBufSize  size of the buffer
     * @return 0 if successful
     */

    NS_IMETHOD_(int) GetDBPluginVersion(
      CALDB_HANDLE pDBH,
      char* psBuf, 
      PRUint32 iBufSize) = 0;
    
    /**
     * Reload all preferences that can be changed while the
     * db subsystem is running.
     * 
     * @param pDBH       handle to the calendar db
     * @return 0 if successful
     */

    NS_IMETHOD_(int) ReloadPrefs(
      CALDB_HANDLE pDBH
      ) = 0;
    
    /**
     * Initialize all database statistic counts to 0
     * 
     * @param pDBH       handle to the calendar db
     * @return 0 if successful
     */

    NS_IMETHOD_(int) InitStatCounts(
      CALDB_HANDLE pDBH
      ) = 0;
    
    /**
     *  Fill a structure with requested information about the database.
     *  Calculating some of the values may take time, so a flags variable
     *  is provided to allow the caller to choose which of the values it
     *  wants filled in.  All values can be selected by setting it to 
     *  DBSTATS_ALL
     *
     *  @param pDBH       handle to the calendar db
     *  @param pStats  pointer to a DB_STATISTICS structure to fill
     *  @param flags   indicates which statistics to compute. The value
     *                 is a logical or of DBSTATS_SIZE, DBSTATS_CALS or
     *                 can be all values by setting it to DBSTATS_ALL
     */

    NS_IMETHOD_(int) GetStats(
      CALDB_HANDLE pDBH,
      DB_STATISTICS* pStats, 
      PRUint32 flags) = 0;
    
    /**
     * Write a CALDB_CALPROPS structure to the database. If the
     * entry exists in the database it will be completely overwritten.
     * No flags are defined at this time and will be used in a future
     * version.
     * 
     * @param pDBH       handle to the calendar db
     * @param pCalProps      the structure to write
     * @param flags          should be set to 0, no flags at this time
     * @return 0 if successful
     */

    NS_IMETHOD_(int) WriteCalProps(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      PRUint32 flags) = 0;
    
    /**
     * Write a CALDB_VEVENT structure to the database. If the 
     * entry exists in the database it will be completely overwritten.
     *
     * If the CALDB_FLAG_NOALARMCHANGE flag is set, no updates will be
     * made to the alarm database or the active alarm queue. This is
     * an optimization that requires the server to do significantly
     * less work on write updates. But it is the responsibility of the
     * caller to do the right thing. If anything about the alarms have
     * changed, its trigger time, its notify list, or any other 
     * attribute, the caller should not set this flag. Misuse of this
     * command could cause alarms to be removed or sent at the wrong
     * time.
     * 
     * @param pDBH       handle to the calendar db
     * @param pCalProps      the calendar in which to write the event
     * @param pEvent         pointer to the event structure to write
     * @param flags          bitwise OR of flag bits. Only 
     *                       CALDB_FLAG_NOALARMCHANGE is supported at
     *                       this time.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) WriteEvent(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      CALDB_VEVENT* pEvent, 
      PRUint32 flags) = 0;
    
    /**
     * Used to write a CALDB_VEVENT or CALDB_VTODO structure to the 
     * database.
     * 
     * @param pDBH           database handle
     * @param pCalProps      the calendar in which to write the event
     * @param pComp          pointer to the event or todo structure to 
     *                       write
     * @param flags          bitwise OR of flag bits. Only 
     *                       CALDB_FLAG_NOALARMCHANGE is supported at
     *                       this time.
     * @return 0 if successful
     */
    NS_IMETHOD_(int) WriteComponent(
      CALDB_HANDLE pDBH, 
      CALDB_CALPROPS* pCalProps, 
      CALDB_COMPONENT* pComp, 
      PRUint32 flags
      ) = 0;

    /**
     * Used to read a CALDB_VEVENT or CALDB_VTODO structure to the 
     * database.
     * 
     * @param pDBH           database handle
     * @param pCalProps      the calendar in which to write the event
     * @param iType          Component Type
     * @param psUID          UID of the component
     * @param rid            recurrence id
     * @param flags          special behavior
     * @aparam ppComponent   returned allocated copy of the component read.
     * @return 0 if successful
     */
    NS_IMETHOD_(int) ReadComponent(
      CALDB_HANDLE pDBH, 
      CALDB_CALPROPS* pCalProps, 
      PRInt32 iType,
      char* psUID,
      time_t rid,
      PRUint32 flags,
      CALDB_COMPONENT** ppComponent
      ) = 0;

    /**
     * Write a CALDB_GSE_ENTRY structure to the database.
     * 
     * @param pDBH     database handle
     * @param pInfo    pointer to info describing options for the write
     * @param pEntry   the entry to write. 
     * @return 0 if successful
     */

    NS_IMETHOD_(int) WriteGseEntry(
      CALDB_HANDLE pDBH, 
      CALDB_GSE_WRITEINFO* pInfo, 
      CALDB_GSE_ENTRY* pEntry
      ) = 0;
    

    /**
     * Read a CALDB_GSE_ENTRY structure from the database.
     * 
     * @param pDBH     database handle
     * @param pInfo    pointer to info describing options for the write
     * @param ppEntry  pointer the the entry pointer returned. Must be
     *                 freed by the caller. use caldb_FreeGseEntry. 
     * @return 0 if successful
     */

    NS_IMETHOD_(int) ReadGseEntry(
      CALDB_HANDLE pDBH, 
      CALDB_GSE_QUERYINFO* pInfo, 
      CALDB_GSE_ENTRY** ppEntry
      ) = 0;
    

    /**
     * Read a CALDB_GSE_ENTRY structure from the database.
     * 
     * @param pDBH     database handle
     * @param pInfo    pointer to info describing options for the write
     * @param ppEntry  pointer the the entry pointer returned. Must be
     *                 freed by the caller. use caldb_FreeGseEntry. 
     * @return 0 if successful
     */

    NS_IMETHOD_(int) QueryGseEntries(
      CALDB_HANDLE pDBH, 
      CALDB_GSE_QUERYINFO* pInfo, 
      array_t** ppList
      ) = 0;
    

    /**
     * Delete a CALDB_GSE_ENTRY structure from the database.
     * 
     * @param pDBH     database handle
     * @param pInfo    pointer to info describing options for the write
     * @return 0 if successful
     */

    NS_IMETHOD_(int) DeleteGseEntry(
      CALDB_HANDLE pDBH, 
      CALDB_GSE_QUERYINFO* pInfo
      ) = 0;
    
    /**
     * Examine every CALDB_CALPROPS entry in the database by calling the 
     * supplied compare procedure. The compare procedure examines the
     * CALDB_CALPROPS entry and returns 1 the entry matches the criteria
     * and 0 if it does not. A list of all matching entries is returned.
     * It is the caller's responsibility to free the list and associated
     * data when finished. See FreeCalPropsList.
     * 
     * @param pDBH       handle to the calendar db
     * @param ppPropList     pointer to the pointer for the list of matching
     *                       CALDB_CALPROPS structures.
     * @param pData          a data value that is passed into the compare
     *                       procedure as a void*. This is managed entirely 
     *                       by the caller.
     * @param pCompareProc   Pointer to a procedure whose arguments are a
     *                       pointer to a CALDB_CALPROPS structure and a
     *                       void*. This procedure examines the supplied
     *                       structure and returns 1 if it matches and 0
     *                       if it does not.
     * @return 0 if successful
     */

    NS_IMETHOD  ScanCalProps(
      CALDB_HANDLE pDBH,
      array_t** ppPropList, 
      CALDB_CALPROPS_QUERYDATA* pData, 
      int(*pCompareProc)(const CALDB_CALPROPS*, CALDB_CALPROPS_QUERYDATA*) ) = 0;

    
    /**
     * Query the calendars (calendar properties database) for all the 
     * calendars that match the supplied input filtering data.
     * It is the caller's responsibility to free the list and associated
     * data when finished. See FreeCalPropsList.
     * 
     * @param pDBH       handle to the calendar db
     * @param iQuery         The desired query or delete
     * @param pData          a data value that is passed into the compare
     *                       procedure as a void*. This is managed entirely 
     *                       by the caller.
     * @param ppPropList     Pointer a returned list of CALDB_CALPROPS 
     *                       structures.
     * @return 0 if successful
     */

    NS_IMETHOD QueryCalendars(
      CALDB_HANDLE pDBH,
      PRInt32 iQuery, 
      CALDB_CALPROPS_QUERYDATA* pData, 
      array_t** ppPropList) = 0;

    /**
     *  Search the CalProps table for entries that match the supplied string.
     *  The string will be searched for in fields defined by flags. The flags
     *  can be a logical OR of the following defines in any combination:
     * 
     *     CALDB_CALPROPS_RELATIVECALID
     *     CALDB_CALPROPS_NAME
     *     CALDB_CALPROPS_OWNERS
     * 
     *  If the string appears in _any_ of these fields, the CalProps structure
     *  is returned. There are several options for how the string is matched:
     *
     *     CALDB_SEARCH_CONTAINS
     *     CALDB_SEARCH_BEGINSWITH
     *     CALDB_SEARCH_ENDSWITH
     *     CALDB_SEARCH_EXACT
     *
     *  One of these options is specified in searchOpts.  A maximum of 
     *  iMaxResults are returned.
     *
     *  @param pDBH       handle to the calendar db
     *  @param pString    the search string
     *  @param flags
     *  @param searchOpts
     *  @param iMaxResults
     *  @param ppPropList returned table of props
     *  @return 0 on success
     */
    NS_IMETHOD SearchCalProps(
      CALDB_HANDLE pDBH,
      const char* pString, 
      PRUint32 flags, 
      PRUint32 searchOpts, 
      PRUint32 iMaxResults, 
      array_t** ppPropList) = 0;

    /**
     * Delete a specific to-do identified by the supplied UID and recurrence id.
     *
     * @param pDBH       handle to the calendar db
     * @param - CALDB_CALPROPS* pCalProps - calendar properties
     * @param - char* psUID               - calendar uid
     * @param - time_t rid                - calendar recurrence id
     * @param - PRUint32 flags            - If CALDB_FLAG_TOMBSTONE is present the event
     *                                      is marked as a tombstone.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) DeleteToDo(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps,
      char* psUID, 
      time_t rid, 
      u_int32_t flags, CALDB_COMPONENT* pComp ) = 0;

    /**
     * Place a link in one calendar to an to-do in another calendar.
     * 
     * @param pDBH       handle to the calendar db
     * @param pSource        source calendar
     * @param pUID           unique id of the to-do to link
     * @param dtRecurrencID  recurrence id of the to-do to link
     * @param pDestCal       calendar in wich to insert to-do
     * @return 0 if successful
     */

    NS_IMETHOD_(int)  LinkToDo(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pSource, 
      char* pUID, 
      time_t dtRecurrenceID, 
      CALDB_CALPROPS* pDestCal,
      PRUint32 flags) = 0;

    /**
     * Write a CALDB_VTODO structure to the database. If the 
     * entry exists in the database it will be completely overwritten.
     *
     * If the CALDB_FLAG_NOALARMCHANGE flag is set, no updates will be
     * made to the alarm database or the active alarm queue. This is
     * an optimization that requires the server to do significantly
     * less work on write updates. But it is the responsibility of the
     * caller to do the right thing. If anything about the alarms have
     * changed, its trigger time, its notify list, or any other 
     * attribute, the caller should not set this flag. Misuse of this
     * command could cause alarms to be removed or sent at the wrong
     * time.
     * 
     * @param pDBH       handle to the calendar db
     * @param pCalProps      the calendar in which to write the to-do
     * @param pToDo          pointer to the to-do structure to write
     * @param flags          bitwise OR of flag bits. Only 
     *                       CALDB_FLAG_NOALARMCHANGE is supported at
     *                       this time.
     * @return 0 if successful
     */

    NS_IMETHOD_(int)  WriteToDo(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      CALDB_VTODO* pToDo, 
      u_int32_t flags) = 0;


    /**
     * Read the specific to-do identified by the supplied uid and recurrence id.
     * 
     * @param pDBH       handle to the calendar db
     * @param pCalProps  - calendar properties
     * @param psUID      - calendar uid
     * @param rid        - calendar recurrence id
     * @param flags      - if CALDB_FLAG_TOMBSTONE is present the to-do is marked as a tombstone.. 
     * @param ppToDo     - the found to-do. This to-do structure is allocated in ReadToDo.
     *                     Call FreeToDo to free the memory associated with this to-do.
     * @return 0 if successful
     */

    NS_IMETHOD_(int)  ReadToDo(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps,
      char* psUID,
      time_t rid, 
      u_int32_t flags, 
      CALDB_VTODO** ppToDo) = 0;

    NS_IMETHOD ReadDeleteLogEntry(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      CALDB_DELETELOG_QUERYINFO* pData, 
      array_t** ppList) = 0;

    NS_IMETHOD WriteDeleteLogEntry(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      CALDB_COMPONENT* pComp) = 0;

    NS_IMETHOD DeleteDeleteLogEntry(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      CALDB_DELETELOG_ENTRY* pInfo) = 0;

    NS_IMETHOD QueryDeleteLogEntries(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps, 
      CALDB_DELETELOG_QUERYINFO* pData, 
      array_t** ppList) = 0;

    NS_IMETHOD GetServerTime(
      CALDB_HANDLE pDBH,
      CALDB_CALPROPS* pCalProps,
      CALTIME* servertime) = 0;


    /**
     * Free a single CALDB_CALPROPS structure
     * 
     * @param ppCalProps pointer to the structure pointer. Set to 0
     *                   on successful return.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) FreeCalProps(
      CALDB_CALPROPS** ppCalProps) = 0;
    
    /**
     * Free a list of CALDB_CALPROPS structures
     * 
     * @param ppPropList pointer to the list of calprops. Set to 0
     *                   on successful return.
     * @return 0 if successful
     */

    NS_IMETHOD FreeCalPropsList(
      array_t** ppPropList)=0;
    
    /**
     * Free a single CALDB_VEVENT structure
     * 
     * @param ppEvent pointer to the structure pointer. Set to 0
     *                on successful return.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) FreeEvent(
      CALDB_VEVENT** ppEvent) = 0;
    
    /**
     * Free a list of CALDB_VEVENT structures
     * 
     * @param ppEventList - pointer to the list of CALDB_VEVENTs. Set to 0
     *                      on successful return.
     * @return 0 if successful
     */

    NS_IMETHOD FreeEventList(
      array_t** ppEventList) = 0;
    
    /**
     * Free a list of CALDB_COMPONENT structures
     * 
     * @param ppComponentList - pointer to the list of CALDB_COMPONENTs. Set to 0
     *                      on successful return.
     * @return 0 if successful
     */

    NS_IMETHOD FreeComponentList(
      array_t** ppComponentList) = 0;
    
    /**
     * Free a single CALDB_VTODO structure
     * 
     * @param ppToDo - pointer to the structure pointer. Set to 0
     *                  on successful return.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) FreeToDo(
      CALDB_VTODO** ppToDo) = 0;
    
    /**
     * Free a list of todo structures
     * 
     * @param ppToDoList - pointer to the list of CALDB_VTODOs. Set to 0
     *                     on successful return.
     * @return 0 if successful
     */

    NS_IMETHOD FreeToDoList(
      array_t** ppToDoList) = 0;
    
    /**
     * Free a single Attendee structure
     * 
     * @param ppAttendee - pointer to the structure pointer. Set to 0
     *                  on successful return.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) FreeAttendee(
      CALDB_ATTENDEE** ppAttendee) = 0;
    
    /**
     * Free a single organizer structure
     * 
     * @param ppOrganizer - pointer to the structure pointer. Set to 0
     *                  on successful return.
     * @return 0 if successful
     */

    NS_IMETHOD_(int) FreeOrganizer(
      CALDB_ORGANIZER** ppOrganizer) = 0;
    
    /**
     * Generate a globally unique id.
     * 
     * @param pG             pointer to the CALDB_GUID generated
     * @return 0 if successful
     */

    NS_IMETHOD_(PRInt32) GenerateGUID(
      CALDB_GUID* pG) = 0;
    
    /**
     *  Allocate and initialize a new event structure to its default
     *  state.
     *
     *  @return a pointer to the new event.
     */
    NS_IMETHOD_(int) NewEvent(CALDB_VEVENT*) = 0;

    /**
     *  Allocate and initialize a new todo structure to its default
     *  state.
     *
     *  @return a pointer to the new todo.
     */
    NS_IMETHOD_(int) NewToDo(CALDB_VTODO*) = 0;

    /**
     *  Allocate and initialize a new alarm structure to its default
     *  state.
     *
     *  @return a pointer to the new alarm.
     */
    NS_IMETHOD_(int) NewAlarm(CALDB_VALARM*) = 0;

    /**
     *  Allocate and initialize a new attendee structure to its default
     *  state.
     *
     *  @return a pointer to the new attendee.
     */
    NS_IMETHOD_(int) NewAttendee(CALDB_ATTENDEE*) = 0;
    
    /**
     *  Allocate and initialize a new organizer structure to its default
     *  state.
     *
     *  @return a pointer to the new organizer.
     */
    NS_IMETHOD_(int) NewOrganizer(CALDB_ORGANIZER*) = 0;

/********************************************************************************/
/********************************************************************************/
/********************************************************************************/

/*
 * THE FOLLOWING CALLS SHOULD BE REMOVED FROM CALDBAPI.  They are internal calls.
 */

    NS_IMETHOD_(int) StartServerAlarmDispatch(CALDB_HANDLE pDBH,const char* psHost, PRUint32 iPort, int (*pFunc)(CALDB_EXTERNAL_INFO_QUERY*)) = 0;
    NS_IMETHOD_(int) StartServerPurgeDeleteLogDispatch(CALDB_HANDLE pDBH) = 0;
    
    NS_IMETHOD_(int) StopServerAlarmDispatch(CALDB_HANDLE pDBH) = 0;
    NS_IMETHOD_(int) StopServerPurgeDeleteLogDispatch(CALDB_HANDLE pDBH) = 0;
    
    NS_IMETHOD_(int)  WriteAlarmQueue(
      CALDB_HANDLE pDBH,
      array_t* pAlarmList, 
      time_t rid, 
      PRBool bTombstone, 
      CALDB_COMP_HASH* pHash, PRUint32 flags) = 0;
    
    /* called by ReadEvent, Alarm Queue, and others. */
    NS_IMETHOD_(int)  ReadEventCore(
      CALDB_HANDLE pDBH,
      DBT* pKey, 
      DBT* pData, 
      PRUint32 flags, 
      CALDB_VEVENT** ppEvent) = 0;

    NS_IMETHOD_(int)  ReadToDoCore(
      CALDB_HANDLE pDBH,
      DBT* pKey, 
      DBT* pData, 
      PRUint32 flags, 
      CALDB_VTODO** pp) = 0;

    /**
     * Create a hash number based on the calendar and the event. The
     * result is stored in a CALDB_COMP_HASH structure.
     * 
     * @param pCalProps      pointer to the calendar structure
     * @param pEvent         pointer to the event structure
     * @param pHash          pointer to the structure in which the 
     *                       hash results are stored.
     * @return 0 if successful
     */

    NS_IMETHOD_(void) EventHash(
      CALDB_CALPROPS* pCalProps, 
			CALDB_VEVENT* pEvent, 
      CALDB_COMP_HASH* pHash) = 0;
    
    /**
     * GetCalDB
     *
     * Get a Calendar DB Object
     *
     * @param pDBSS
     * @return !NULL if successful
     */

    NS_IMETHOD_(void*) GetCalDB() = 0;

    /**
     * Get Serialized Data in specified MIME type
     *
     * @param - char * mimetype     - mimetype
     * @param - CALDB_SDOP * sdop   - serialized data operand
     * @param - void ** buf         - output buffer
     * @param - PRUint32 * len      - length of returned data
     *                                 
     * @return 0 if successful
     */

    NS_IMETHOD_(int) GetSerializedData(char * mimetype, 
                                       CALDB_SDOP * sdop, 
                                       void ** buf,       
                                       PRUint32 * len) = 0; 


    NS_IMETHOD_(int) RecoverScanComponents(CALDB_HANDLE pDBH,CALDB_HANDLE pToDBH,PRInt32 iType, PRBool bCalpropsOnly) = 0;
    NS_IMETHOD_(int) RecoverScanCalProps(CALDB_HANDLE pDBH,CALDB_HANDLE pToDBH,PRInt32 iType, PRBool bDontScanComps) = 0;
    NS_IMETHOD_(int) RecoverAlarms(CALDB_HANDLE pDBH,CALDB_HANDLE pToDBH) = 0;
   NS_IMETHOD_(int) VerifyDB(CALDB_HANDLE pDBH,char *pSource, PRInt32 verifytype) = 0;
    NS_IMETHOD_(int) DeserializeComponent(CALDB_CALPROPS* pContainer, CALDB_COMPONENT* pComp, char* pBuf, size_t iBufLen, char** ppNext) = 0;

/* begin LDAP CLD migration */
    NS_IMETHOD_(int) CldvdMigrate(CALDB_HANDLE pDBH,CALDB_HANDLE pToDBH, hashtable_t *ht, PRFileDesc *fd) = 0;
/* end LDAP CLD migration */
    NS_IMETHOD_(int) Rename(CALDB_HANDLE pDBH,CALDB_HANDLE pToDBH, hashtable_t *ht) = 0;

    /*
     * Delete one or more events identified by the supplied UID, recurrence id,
     * and modifier flags. If the recurrence id is 0, and the UID points to a 
     * recurring sequence, the entire recurring series is deleted. Note the 
     * flags that provide operations on subsets.
     * 
     * @param pCalProps - calendar properties
     * @param psUID     - calendar uid
     * @param rid       - calendar recurrence id
     * @param flags     - If CALDB_FLAG_TOMBSTONE is present the event
     *                    is marked as a tombstone. 
     *                  - If CALDB_FLAG_THISANDPRIOR is present all
     *                    events prior to and including the one specified 
     *                    by the recurrence id are deleted. 
     *                  - If CALDB_FLAG_THISANDFUTURE is present all
     *                    events after and including the one specified
     *                    by the recurrence id are deleted.
     * @param piEventCount - number of events deleted (returned)
     * @return 0 if successful
     */
/*
    NS_IMETHOD_(int) DeleteEventsByID(
      CALDB_CALPROPS* pCalProps, 
      char* psUID, 
      time_t rid, 
      PRUint32 flags, 
      PRUint32* piEventCount) = 0;
*/

    /*
     * Delete the events that overlap range of time. The CALDB_FLAG_TOMBSTONE
     * flag can be used to mark the events as deleted but leave them in
     * the database. Normal read operations ignore events that are tombstones.
     * However, tombstone events can be read with an appropriate flag and
     * the tombstone attribute can be removed from an event.
     * 
     * @param - pCalProps    - calendar properties
     * @param - dtStart      - calendar start time
     * @param - dtEnd        - calendar end time
     * @param - flags        - if CALDB_FLAG_TOMBSTONE is present the event
     *                         is marked as a tombstone. 
     * @param - piEventCount - number of events
     * @return 0 if successful
     */

/*
    NS_IMETHOD_(int) DeleteEventsByRange(
      CALDB_CALPROPS* pCalProps, 
      time_t dtStart, 
      time_t dtEnd, 
      PRUint32 flags, 
      PRUint32* piEventCount) = 0;
*/

    /*
     * Read one or more events identified by the supplied uid and 
     * recurrence id.
     * 
     * @param pCalProps    - calendar properties
     * @param psUID        - calendar uid
     * @param rid          - calendar recurrence id
     * @param flags        - if CALDB_FLAG_TOMBSTONE is present, events marked as 
     *                       tombstones are read included in the returned events. 
     *                       Otherwise, tombstone events are ignored.
     * @param ppEventList  - pointer to a list pointer of events. Allocated 
     *                       here. De-allocate by calling FreeEventList 
     * @return 0 if successful
     */

/*
    NS_IMETHOD ReadEventsByID(
      CALDB_CALPROPS* pCalProps, 
      char* psUID, 
      time_t rid, 
      PRUint32 flags, 
      array_t** ppEventList) = 0;
*/    

    /*
     * Read events that overlap the supplied start and end times.
     * 
     * @param pCalProps    - calendar properties
     * @param dtStart      - calendar start time
     * @param dtEnd        - calendar end time
     * @param flags        - if CALDB_FLAG_TOMBSTONE is present, events marked as 
     *                       tombstones are read included in the returned events. 
     *                       Otherwise, tombstone events are ignored.
     * @param ppEventList  - pointer to a list pointer of events. Allocated here.
     *                       De-allocated by a call to an be FreeEventList
     * @return 0 if successful
     */

/*    NS_IMETHOD ReadEventListByRange(CALDB_CALPROPS* pCalProps, 
      time_t dtStart, 
      time_t dtEnd, 
      PRUint32 flags, 
      array_t** ppEventList) = 0;
*/
    /**
     * Read events where the last modified time has a relationship
     * to the the supplied start and end times.
     *
     * If both dtStart and dtEnd are 0, all events are matched.
     *
     * If dtStart has a non-zero value and dtEnd has a 0 value it will match
     * all entries where the last modified value is >= dtStart.
     *
     * If dtStart is 0 and dtEnd is non-zero, all events with a last modified
     * value less than dtEnd are matched.
     *
     * If both dtStart and dtEnd are non-zero, events with last modified >=
     * dtStart and < dtEnd are matched.
     * 
     * @param pCalProps    - calendar properties
     * @param dtStart      - calendar start time
     * @param dtEnd        - calendar end time
     * @param flags        - if CALDB_FLAG_TOMBSTONE is present the event is 
     *                       marked as a tombstone.
     * @param ppEventList  - pointer to a list pointer of events. Allocated here.
     *                       De-allocated by a call to an be FreeEventList
     * @return 0 if successful
     */

/*
    NS_IMETHOD ReadEventListByLastModified(CALDB_CALPROPS* pCalProps, 
      time_t dtStart, 
      time_t dtEnd, 
      PRUint32 flags, 
      array_t** ppEventList) = 0;
*/

    /*
     * Read events with alarms that have trigger times in the supplied time range
     * 
     * @param pCalProps    - calendar properties
     * @param dtStart      - calendar start time
     * @param dtEnd        - calendar end time
     * @param flags        - if CALDB_FLAG_TOMBSTONE is present the event is marked 
     *                       as a tombstone. 
     * @param ppEventList  - the list of found events
     * @return 0 if successful
     */
/*
    NS_IMETHOD ReadEventListByAlarmRange(CALDB_CALPROPS* pCalProps, 
			time_t dtStart, 
      time_t dtEnd, 
      PRUint32 flags, 
      array_t** ppEventList) = 0;
  */
    
    /*
     * ReadCalPropsListByOwner
     * 
     * Read the calprop entries where the supplied name is an Owner.
     * 
     * @param pOwner     - calendar relative calid
     * @param ppPropList - list of matching calendar properties
     * @return 0 if successful
     */
#if 0
    NS_IMETHOD ReadCalPropsListByOwner(
      const char* pOwner, 
      array_t** ppPropList) = 0; 
#endif

    /*
     * Find the list of calprops where at least one category matches
     * the supplied category
     * 
     * @param pCategory  - calendar relative calid
     * @param ppPropList - list of matching calendar properties
     * @return 0 if successful
     */
/*
    NS_IMETHOD ReadCalPropsListByCategory(
      const char* pCategory, 
      array_t** ppPropList) = 0; 
*/
    
    /*
     * Delete one or more to-dos identified by the supplied UID, recurrence id,
     * and modifier flags. If the recurrence id is 0, and the UID points to a 
     * recurring sequence, the entire recurring series is deleted. Note the 
     * flags that provide operations on subsets.
     * 
     * @param - CALDB_CALPROPS* pCalProps - calendar properties
     * @param - char* psUID               - calendar uid
     * @param - time_t rid                - calendar recurrence id
     * @param - PRUint32 flags            - If CALDB_FLAG_TOMBSTONE is present the to-do
     *                                      is marked as a tombstone. 
     *                                    - If CALDB_FLAG_THISANDPRIOR is present all
     *                                      to-dos prior to and including the one specified 
     *                                      by the recurrence id are deleted. 
     *                                    - If CALDB_FLAG_THISANDFUTURE is present all
     *                                      to-dos after and including the one specified
     *                                      by the recurrence id are deleted.
     * @param - PRUint32* piCount         - number of to-dos deleted (returned)
     * @return 0 if successful
     */

/*    NS_IMETHOD_(int)  DeleteToDosByID(
      CALDB_CALPROPS* pCalProps, 
      char* psUID, 
      time_t rid, 
      u_int32_t flags, 
      u_int32_t* piCount) = 0;
*/

    /**
     * Delete the to-dos that have been marked as completed in the specified
     * range of time. The CALDB_FLAG_TOMBSTONE flag can be used to mark the
     * to-dos as deleted but leave them in the database. Normal read operations
     * ignore to-dos that are tombstones. However, tombstone to-dos can be read
     * with an appropriate flag and the tombstone attribute can be removed from 
     * a to-do.
     * 
     * @param - CALDB_CALPROPS* pCalProps - calendar properties
     * @param - time_t dtStart            - calendar start time
     * @param - time_t dtEnd              - calendar end time
     * @param - PRUint32 flags            - if CALDB_FLAG_TOMBSTONE is present the to-do
     *                                      is marked as a tombstone. 
     * @param - PRUint32* piToDoCount    - number of to-dos
     * @return 0 if successful
     */

/*    NS_IMETHOD_(int)  DeleteToDosCompletedByRange(
      CALDB_CALPROPS* pCalProps, 
      time_t dtStart, 
      time_t dtEnd, 
      PRUint32 flags, 
      PRUint32* piToDoCount) = 0;
*/

    /*
     * Read one or more to-dos identified by the supplied uid and 
     * recurrence id.
     * 
     * @param pCalProps    - calendar properties
     * @param psUID        - calendar uid
     * @param rid          - calendar recurrence id
     * @param flags        - if CALDB_FLAG_TOMBSTONE is present, to-dos marked as 
     *                       tombstones are read included in the returned to-dos. 
     *                       Otherwise, tombstone to-dos are ignored.
     * @param ppToDoList   - pointer to a list pointer of to-dos. Allocated 
     *                       here. De-allocate by calling FreeToDoList 
     * @return 0 if successful
     */

/*    NS_IMETHOD  ReadToDoListByID(
      CALDB_CALPROPS* pCalProps, 
      char* psUID, 
      time_t rid, 
      u_int32_t flags, 
      array_t** ppToDoList) = 0;
*/

    /*
     * Read to-dos that are either not marked as completed or that have been
     * marked as completed at some point between the supplied start and end times.
     * 
     * @param pCalProps    - calendar properties
     * @param dtStart      - calendar start time
     * @param dtEnd        - calendar end time
     * @param flags        - if CALDB_FLAG_TOMBSTONE is present, to-dos marked as 
     *                       tombstones are read included in the returned to-dos. 
     *                       Otherwise, tombstone to-dos are ignored.
     * @param ppToDoList  - pointer to a list pointer of to-dos. Allocated here.
     *                       De-allocated by a call to an be FreeToDoList
     * @return 0 if successful
     */

/*
    NS_IMETHOD  ReadToDoListByRange(
      CALDB_CALPROPS* pCalProps, 
      time_t dtStart, 
      time_t dtEnd, 
      u_int32_t flags, 
      array_t** ppToDoList) = 0;
*/

    /*
     * Read to-dos with alarms that have trigger times in the supplied time range
     * 
     * @param pCalProps    - calendar properties
     * @param dtStart      - calendar start time
     * @param dtEnd        - calendar end time
     * @param flags        - if CALDB_FLAG_TOMBSTONE is present the to-do is marked
     *                       as a tombstone. 
     * @param ppToDoList   - the list of found to-dos
     * @return 0 if successful
     */
/*
    NS_IMETHOD  ReadToDoListByAlarmRange(
      CALDB_CALPROPS* pCalProps, 
      time_t dtStart, 
      time_t dtEnd, 
      PRUint32 flags, 
      array_t** ppToDoList) = 0;
*/
    /**
     * Read to-dos where the last modified time has a relationship
     * to the the supplied start and end times.
     *
     * If both dtStart and dtEnd are 0, all to-dos are matched.
     *
     * If dtStart has a non-zero value and dtEnd has a 0 value it will match
     * all entries where the last modified value is >= dtStart.
     *
     * If dtStart is 0 and dtEnd is non-zero, all to-dos with a last modified
     * value less than dtEnd are matched.
     *
     * If both dtStart and dtEnd are non-zero, to-dos with last modified >=
     * dtStart and < dtEnd are matched.
     * 
     * @param pCalProps    - calendar properties
     * @param dtStart      - calendar start time
     * @param dtEnd        - calendar end time
     * @param flags        - if CALDB_FLAG_TOMBSTONE is present the to-do is 
     *                       marked as a tombstone.
     * @param ppToDoList   - pointer to a list pointer of to-dos. Allocated here.
     *                       De-allocated by a call to an be FreeToDoList
     * @return 0 if successful
     */
#if 0
    NS_IMETHOD  ReadToDoListByLastModified(
      CALDB_CALPROPS* pCalProps, 
      time_t dtStart, 
      time_t dtEnd, 
      PRUint32 flags, 
      array_t** ppToDoList) = 0;
#endif


};

#define CS_ICALENDARDATABASE_IID                     \
{ /* f7bed500-d65f-11d2-9e72-00105ae811b7 */         \
    0xf7bed500,                                      \
    0xd65f,                                          \
    0x11d2,                                          \
    {0x9e, 0x72, 0x00, 0x10, 0x5a, 0xe8, 0x11, 0xb7} \
}


#endif /* csICalendarDatabase_h___ */




