/**********************************************************************/
/*                                                                    */
/* Copyright (c) 2003 by Sun Microsystems, Inc.                       */
/* All rights reserved.                                               */
/*                                                                    */
/**********************************************************************/

//
// File: Sample3.java
//
// IMPORTANT NOTE: To execute this program successfully, you must have 
// available an MTP Telnet3270 end-system on which the ACCT transaction
// is available.
//
// This program searches for ACCT account numbers for the given Surname on the
// given host. It demonstrates navigation through a series of Terminal screens.
//
// This program takes the parameters:
//   The host name	 e.g. myhost
//   The port number	 e.g. 9993
//   The search string	 e.g. Greenfield 
//     NOTE: The search string has a maximum length of 12 characters.
//
// e.g. java Sample3 myhost 9993 Greenfield
//
// This program also demonstrates two techniques for interrogating the 
// Terminal screen:
//
// Technique 1 - Requesting a field.
// This technique asks the Terminal to return the field at the specified
// point on the Terminal. This is a TerminalField which can be interrogated with
// a variety of methods, explained further in the JavaDoc API of TerminalField.
//
// Technique 2 - Requesting the display buffer.
// Requesting the display buffer from the Terminal results in a character
// array of the screen. This can be manipulated in the usual manner.
//

package samples.bean;

import com.sun.emp.pathway.bean.Terminal;
import com.sun.emp.pathway.bean.TerminalField;

import java.util.Vector;

public class Sample3 {
    public static void main(String[] args) {
	//
	// If the number of supplied parameters is not 3 or the 3rd parameter
	// is greater that 12 characters in length then display the usage
	// message.
	//
	if ((args.length != 3) || (args[2].length() > 12)) {
	    System.out.println("Usage: java Sample3 <host> <port> <searchstring>");
	    System.out.println("  Where the search string has a max length of 12 characters");
	    System.exit(0);
	}

	System.out.println("------------------------------------------------");

	//
	// Run the method to obtain the account numbers and store the results
	// in a vector.
	//
	Vector v = getAccountNumbers(args);

	//
	// Did we get any results (is the vector empty) ?
	//
	if (v.isEmpty()) {
	    System.out.println("Sorry, no matches for " + args[2]);
	} else {
	    //
	    // Display the results on the screen.
	    //
	    for ( int i=0;i<v.size();i++ ) {
		System.out.println(" Account number: "+v.elementAt(i));
	    }
	}

	System.out.println("------------------------------------------------");
	System.exit(0);
    }

    //
    // This method returns the account numbers (as a vector) for the
    // specified searchName.
    //
    // It achieves this by logging on to the specified host,
    // then navigates a series of screens, enters the given searchName
    // into the Terminal and returns the account numbers (if any) for that
    // searchName.
    //
    // This method assumes that:
    //   args[0] is the host
    //   args[1] is the port (NOTE: This should be numeric only)
    //   args[2] is the searchName
    //
    private static Vector getAccountNumbers(String[] args) {
	//
	// Setup the Terminal and returnVector.
	//
	Terminal terminal = new Terminal();
	Vector returnVector = new Vector();

	//
	// Set the Terminal host and port settings.
	//
	terminal.setTN3270Host(args[0]);
	terminal.setTN3270Port(Integer.parseInt(args[1]));
	String searchName = args[2];

	//
	// Connect the Terminal
	//
	terminal.connect();
	terminal.waitUntilConnected();
	terminal.waitUntilKeyboardUnlocked();

	//
	// The Terminal should now be at the "Welcome screen".
	//
	terminal.pressClear();
	terminal.waitUntilKeyboardUnlocked();

	//
	// The Terminal should now be on a blank screen.
	//
	terminal.typeString("ACCT");
	terminal.pressEnter();
	terminal.waitUntilKeyboardUnlocked();

	//
	// The Terminal should now be on the ACCT screen.
	//
	terminal.typeString(searchName);
	terminal.pressEnter();
	terminal.waitUntilKeyboardUnlocked();

	//
	// Get the account numbers.
	//
	while(true) {
	    for (int i=0; i<6; i++) {
		//
		// Get the appropriate field from the Terminal.
		//
		TerminalField tf = terminal.findField(terminal.offsetFromRowColumn(17+i,2));

		//
		// Try to get a substring of the first 5 characters of the
		// field. A failure indicates an empty field.
		//
		try {
		    String accountNumber  = tf.getText().substring(0,5);
		    //
		    // Store the resultant String in the vector.
		    //
		    returnVector.addElement(accountNumber);
		} catch(IndexOutOfBoundsException e) {
		    //
		    // An empty field, so exit the loop.
		    //
		    break;
		}
	    }

	    //
	    // Check to see if we have reached the end of the list.
	    // This technique inspects a substring of the display buffer.
	    //
	    String s = new String(terminal.getDisplayBuffer(),
				    terminal.offsetFromRowColumn(24,39), 3);

	    //
	    // If PA2 is on the screen we need to go round the loop again.
	    // If it isn't then we have reached the end of the list.
	    // 
	    if ( !(s.equals("PA2")) ) {
	    	break;
	    }

	    //
	    // There are more results, so go round the loop again. PA2 moves
	    // to the next set of results.
	    //
	    terminal.pressPA(2);
	    terminal.waitUntilKeyboardUnlocked();
	}

	//
	// Test to see if we obtained any results. If not, we need to perform
	// the clear operation an extra time.
	//
	if (returnVector.isEmpty()) {
	    terminal.pressClear();
	    terminal.waitUntilKeyboardUnlocked();
	}

	//
	// Now perform the logging-off procedure
	//
	terminal.pressClear();
	terminal.waitUntilKeyboardUnlocked();

	//
	// The Terminal should have returned to the blank screen
	//
	terminal.typeString("CSSF LOGOFF");
	terminal.pressEnter();
	terminal.waitUntilDisconnected();

	//
	// Return the results
	//
	return returnVector;
    }
}
