package Health;
#<copyright>
# ----------------------------------------------------------
# Sun Proprietary/Confidential Code
# Copyright 2001, Sun Microsystems, Inc. All rights reserved.
# ----------------------------------------------------------
#</copyright>

#  $Name:  $ 
#  $Id: Health.pm,v 1.110 2003/06/04 23:13:43 ccadieux Exp $

use Report;
use Carp;
use Transition;
use Thresholds;
use CIM::Instance;
use Grid;
use strict;
use Message;
use State;

sub new {
  my($class) = @_;

  my($x) = {};

  bless($x, "Health");
  return $x;
}

  


sub saveAlert {
  my($hm, $ev, $level, $id, $report, $cause, $desc) = @_;
  return;
  my($pdm) = $hm->{pdm};

  my($type) = $ev->propertyByName("EventType")->value;
  my($cap) = $ev->propertyByName("Caption")->value;
  $type .= ".$cap" if ($cap);

  my($alert) = CIM::Instance->new('NWS_AlertIndication', [
                  [ AlertId      => $id  ],
                  [ Description  => $desc],
                  [ AlertType    => $type],
                  [ Severity     => $level],
                 [ ProbableCause => $cause],
                  ]);
  my($pertains) =  CIM::Instance->new('NWS_AlertPertainsToEvent', [
                  [ Event        => $ev ],
                  [ Alert        => $alert ],
                  ]);

  my($ed) = Message->new({ id     => $report->id,
                            instances => [$alert, $pertains],
                            type      => Message::TYPE_ALERT ,
                            });
  $pdm->saveMessage($ed);
}


sub locationChangeEvent {
  my($hm, $report, $rep, $orep, $keyval) = @_;

  my($ev, $key, $sd, $pertains, $ed);
  my($pdm) = $hm->{pdm};
  my $renv = System->get_renv();
  return if ($renv->{solution} ne "N");
  my($cat) = $report->category();
  my($name) = $report->id('display');
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';

  my($old) = "Contract:"   . $orep->get("location.contractNo")  . ", " . 
             "HostId:"     . $orep->get('location.hostId')      . ", " .
             "CustomerNo:" . $orep->get("location.customerNo")  . ", " .
             "Site:" .       $orep->get("location.siteName")    . ", " . 
                             $orep->get("location.siteAddress") . ", " .
                             $orep->get("location.siteCity")    . ", " .
                             $orep->get("location.siteState")   . ", " .
                             $orep->get("location.siteZip")     . ", " .
                             $orep->get("location.siteContact") . "," .
                             $orep->get("location.siteEmail") ;

  my($new) = "Contract:"   . $rep->get("location.contractNo")   . ", " . 
             "HostId:"     . $rep->get('location.hostId')       . ", " .
             "CustomerNo:" . $rep->get("location.customerNo")   . ", " . 
             "Site:" .       $rep->get("location.siteName")    . ", " . 
                             $rep->get("location.siteAddress") . ", " .
                             $rep->get("location.siteCity")    . ", " .
                             $rep->get("location.siteState")   . ", " .
                             $rep->get("location.siteZip")     . ", " .
                             $rep->get("location.siteContact") . "," .
                             $rep->get("location.siteEmail") ;

  # MAKE SURE THE SITE INFO WAS ENTERED IN THE FIRST PLACE.
  my $site  = $rep->get("location.siteName");
  my $osite = $orep->get("location.siteName");
  return if (!$site || !$osite);

  if ($old ne $new) {
    #my $desc = Events->info('LocationChange', $cat, $name)->serialize();
    my $desc = "Location of $cat $name was changed:";
    Grid->setCode("$cat.LocationChangeEvent");
    $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => lc($cat) . ".LocationChange"  ],
                  [ EventId     => $pdm->getEventSequence  ],
                  [ MgmtLevel   => $mgmtLevel ],
                  [ Description => $desc ], 
                  [ Component   => "location"],
                  [ Data        => "Location Changed from:\n$old\nto:\n$new" ],
                         ]);
     my($key) = CIM::Key->new( ['NWS_System',
                   'Name'       => $keyval,
                    CreationClassName => 'NWS_System']);

     $sd = Events->sourceDetector({ event => $ev });

     $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

     my($loc) = NWS->location($rep);

     my($locp) = CIM::Instance->new('NWS_AgentLocation', [
                  [ Agent       => $sd->[0]  ],
                  [ Element     => $loc ],
                    ]);


     $ed = Message->new( { id        => $report->id,
                            instances => [$ev, @$sd, $pertains, $loc, $locp ],
                          });
     $pdm->saveMessage($ed);
  }
     
}

# $hm->alarm($report, $orep, "BROCADE", "sensor.temperature.1", "status",
#                             $id, $id, $wwn, "sensor.temperature.status");
#      sensor.temperature.1        : used as a key in the state DB
#      sensor.temperature.1.status : used to read the agent-report
#      sensor.temperature.status   : used as a key in the Avail. Map file.

#  id = display id, use by saveState only
#  wwn= cim key
#  shortid = name
#  map_id = used to assign on topology
#  arg->{info} : extra info
#  $arg->{use_map_id}   : use the map_id for the events.
#  $arg->{severity}
#  $arg->{actionable} = Y/N
#  $arg->{state}  : use $arg->{state} for State

sub alarm {
  my($class, $report, $orep, $cat, $comp, $el, $shortid,
     $display_id, $cimkey, $map_id, $arg) = @_;

  my $rep          = $report->content();
  my $s2           = $orep->{"$comp.$el"} || "[Undefined]";
  my $s1           = $rep->{"$comp.$el"}  || "[Undefined]";
  my $extra        = "($arg->{info})" if ($arg->{info});
  my $desc_display = "$comp.$el";
  my $state_comp;
  if ($arg->{state}) {
    $state_comp = $arg->{state};
  } else {
    $state_comp   = ($arg->{use_map_id})? $map_id : $comp;
  }

  my($map) = PDM->getDeviceStateMap(lc($cat) . ".availability");

  $map_id = $el if (!$map_id);
  my($ostatus, $status, $map_sev, $map_action) = $map->transition("$map_id.$s2", "$map_id.$s1", $orep, $rep);

  if (System->get_bad_health()) {
     $s1 = "BAD"; $status = 0;
     delete $orep->{"$comp.${el}_time"};
  }
  my ($severity, $sign);

  if ($status eq "X") {
    $sign = "X";
    $severity = 2;
  } elsif ($status == 0) {
    $sign = "M";
    $severity = 2;
  } else {
    $sign = "P";
    $severity = 0;
  }
  $severity = $map_sev if ($map_sev >= 0);

  my $al = 0;

  # NEW TEST
  if ($s1 ne $s2 || ($rep == $orep && $status != 1)) {
     my $desc = $s1 ne $s2 ? 
           "The state of '$desc_display' on $shortid changed from '$s2' to '$s1' $extra":
           "The state of '$desc_display' on $shortid is '$s1' $extra";
        
     Grid->setSign($sign);
     my $ev = $class->alarmEvent($comp, "", $report, $cimkey, $desc, $severity, 
                  "$sign.$comp.$el" , {noState => 1, actionable => $map_action, nomap => 1 });

     State->saveState($cat, $cimkey, $state_comp, $display_id, 
             $severity, $desc, $status, $ev);
  }
  return;
  # /NEW TEST

  if ($s1 ne $s2 && ($s2 ne "[Undefined]") ) {
     my $desc = "The state of '$desc_display' on $shortid changed from '$s2' to '$s1' $extra";
     Grid->setSign($sign);
     my $ev = $class->alarmEvent($comp, "", $report, $cimkey, $desc, $severity, "$sign.$comp.$el" , {noState => 1, actionable => $map_action, nomap => 1 });
     $al =1;
     if ($status == 1) {
       State->saveState($cat, $cimkey, $state_comp, $display_id, 
             $severity, $desc, $status, $ev);
     }
  }
  my $time_id = "$comp.${el}_time";
  $rep->{$time_id} = $orep->{$time_id} if (exists($orep->{$time_id}));

  if ($status != 1) {
    if ($s1 ne "[Undefined]" ) { # in comment to fix host lun monitoring problem.
       my $desc = "The state of '$desc_display' on $shortid is '$s1' $extra";
       my $time = $rep->{$time_id};
       if (!$time) {
         Grid->setSign($sign);
         my $ev = $class->alarmEvent($comp, "", $report, $cimkey, $desc, 
              $severity, "$sign.$comp.$el", {noState => 1, actionable => $map_action, nomap=> 1 }) if (!$al);
         $rep->{$time_id} = time;
         # could be error or warning if status == 0 or 'X');
         # warning the first time around or if last_value was 0/X

         State->saveState($cat, $cimkey, $state_comp, $display_id, 
                           $severity, $desc, $status, $ev);
       }
    }
  } else {
     delete $rep->{$time_id};
  }
}

# parent FILTER, can be implemented on each device Class

sub FILTER {
  my($hm, $rep, $ev) = @_;
  return if (!$ev);
  if ($rep->{'info.noActionable'}) {
    $ev->setValue("Actionable", "FALSE");
    my $desc = $ev->value("Description");
    my $col = substr($desc, -1);
    $desc .= " (" . $rep->{'info.noActionable'} . ")";
    $desc .= $col if ($col eq ":");
    $ev->setValue("Description", $desc);
  }
}

sub diagEvent {
  my($hm, $report, $sev, $caption, $command, $target, $start, $end, $desc) = @_;

  my $rep = $report->content();
  my $cat = $report->category();
  my $code = "$cat.DiagnosticTest";
  $code .= "-" if ($sev);
  Grid->setCode("$code.$caption");

  my $action = $sev > 1 ? 1 : 0;

  my $ev = CIM::Instance->new('NWS_DiagnosticTestComplete', [
                  [ EventType   => "$cat.DiagnosticTest"  ],
                  [ EventId     => PDM->getEventSequence  ],
                  [ Severity    => $sev                   ],
                  [ Caption     => $caption               ],
                  [ FailureCode => 2                      ],
                  [ Target      => "$cat:$target"         ],
                  [ Actionable  => $action                ],
                  [ Severity    => $sev                   ],
                  [ StartDate   => $start                 ],
                  [ EndDate     => $end                   ],
                  [ Description => $desc                  ],
                   ]);

  $hm->FILTER($rep, $ev);

  my $sd = Events->sourceDetector({ event => $ev , host => 1, rep => $rep});

  my $pertains = CIM::Instance->new('NWS_EventPertainsToSoftwareElement', [
                  [ Event         => $ev      ],
                  [ Element       => $sd->[0] ],
                  [ DiscoveryType => "D"      ],
                    ]);
  my $ed = Message->new( { id        => {deviceName => $target },
                            instances => [$ev, @$sd, $pertains ],
                            severity  => $sev });
  PDM->saveMessage($ed);

  State->saveState($cat,  $target, "alarm", "Alarm", $sev, $desc, 1, $ev);
  return $ev;
}


sub identificationEvent {
  my($class, $report, $old, $new, $wwn) = @_;

  return undef if (!$report);
  my $old = lc($old);
  my $new = lc($new);
  if ($old && (substr($old,-4) ne substr($new,-4)) ) {
    my $cat = $report->category();
    Grid->setCode("$cat.AlarmEvent.identification");
    $class->alarmEvent("system","", $report, $wwn, 
      "The identification of this device changed from '$old' to '$new'",
      1, "identification");
  }
}



# $arg->{actionable} = Y/N
# by default, warnings(1) are not actionable, errors(2) are.

sub alarmEvent {
  my($hm, $comp, $data, $report, $keyval, $desc, $sev, $caption, $arg) = @_;

  my($ev, $key, $sd, $pertains, $ed);
  my($pdm) = $hm->{pdm};
  my($cat) = $report->category();
  my $captionIsState = $arg->{captionIsState};
  my($id)  = $pdm->getEventSequence ;
  my($rep) = $report->content();
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';
  $caption = "AlarmEvent" if (!$caption);

  my $av = ['X', -1,undef];
  if (!$arg->{nomap}) {
    my $map = PDM->getDeviceStateMap(lc($cat) . ".availability");
    $av  = $map->get2("alarmEvent.$caption");
  }
  if ($av->[1] >= 0) {
     $sev = $av->[1];
  } else {
     $sev = 1 if (!defined($sev)); # warning by default
  }

  my $actionable = 0;
  if ($av->[2]) {
     $actionable = $av->[2] eq "Y" ? 1:0;
  } elsif ($arg->{actionable}) {
     $actionable = $arg->{actionable} eq "Y" ? 1:0;
  } elsif ($sev >= 2 ) {
     $actionable = 1;
  }
  my $target = lc($cat) . ":" . $keyval;

  $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => lc($cat) . ".AlarmEvent"  ],
                  [ EventId     => $id          ],
                  [ Data        => $data        ],
                  [ Caption     => $caption     ],
                  [ Target      => $target      ],
                  [ TargetName  => $report->id('display') ],
                  [ Actionable  => $actionable  ],
                  [ MgmtLevel   => $mgmtLevel   ],
                  [ SourceIP    => $rep->{'id.ipno'}],
                  [ Severity    => $sev         ],
                  [ Component   => $comp        ],
                  [ Description => $desc        ],
                         ]);
   $hm->FILTER($rep, $ev);

   $key = CIM::Key->new( ['NWS_System',
                   'Name'       => $keyval,
                    CreationClassName => 'NWS_System']);
   $sd = Events->sourceDetector({ event => $ev });

   $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

   $ed = Message->new( { id        => $report->id,
                          instances => [$ev, @$sd, $pertains ],
                        });
   $pdm->saveMessage($ed);

   $hm->saveAlert($ev,  $sev, $id, $report, 
       $report->id("logFile") . "\n $data" , $desc);

   if (!$arg->{noState}) {
     my $c = $captionIsState ? $caption : "alarm.$caption";
     State->saveState($cat,  $keyval, $c, "Alarm", $sev, $desc, 1, $ev);
   }
   return $ev;
}

# against any switch

sub portStatEvents {
  my($hm , $report, $PORTS, $rep, $orep, $prefix) = @_;
  my $wwn = $report->id('deviceName');
  my $id  = $report->id('display');
  my $cat = $report->category();
  $prefix = "port" if (!$prefix);

  my($x, $port, @P);
  for ($x=0; $x <= $PORTS; $x++) {
     $port = "$prefix.$x";
     my $state = $rep->get("$port.state");
     if(!$state){
       $state = $rep->get("$port.mode");
     }
     next if(!$state);
     $state = lc($state);
     next if ($state =~ /offline/ || $state =~ /unknown/  || $state =~ /not-logged-in/);
     my $rep1 = $rep->subset("$port.error");
     foreach my $val (keys %$rep1) {
         next if ($val eq "C3Discards");
         my($old) = $orep->get("$port.error.$val");
         my($new) = $rep->get("$port.error.$val");
         if ($new > $old) {
           my($level, $cnt, $desc, $mins) = Thresholds->test($cat, $val, $x, $new-$old);
           if ($level eq "W" || $level eq "E") {
             $P[$x] .= "Received $cnt '$desc' in $mins mins (value=$new)\n";
           }
         }
     }
  }
  for ($x=0; $x <= $PORTS; $x++) {
    if ($P[$x]) {
      my $data = $P[$x];
      chop($data);
      if ($data =~ /\n/) {
         $hm->alarmEvent("$prefix.$x", $data, $report,  $wwn, 
            "Change in statistics on $cat $id, $prefix-$x:", 
            Message::SEVERITY_WARNING, "$prefix.$x.statistics", {captionIsState => 1, nomap => 1});
      } else { 
         $hm->alarmEvent("$prefix.$x", "", $report,  $wwn, 
            "$cat $id, $prefix-$x: $data",
            Message::SEVERITY_WARNING, "$prefix.$x.statistics", {captionIsState => 1, nomap => 1});
      }
    }
  }

}

# {actionable => 1}

sub logEvent {
  my($hm, $log, $report, $keyval, $desc, $sev, $caption, $arg) = @_;

  my($ev, $key, $sd, $pertains, $ed);
  my($pdm) = $hm->{pdm};
  my($cat) = $report->category();
  my($id)  =  $pdm->getEventSequence ;
  my($id2) =  $report->id('display');
  my $comp = $arg->{component};
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';

  my($rep) = $report->content();
  my($name)= $rep->get('id.name') if ($rep);
  $caption = "MessageLog" if (!$caption);

  $desc =  "$cat.MessageLog on $id2:" if (!$desc);
  my $action= 0;
  $action = 1 if (($sev == Message::SEVERITY_ERROR) || $arg->{actionable});


  $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => lc($cat) . ".LogEvent"  ],
                  [ EventId     => $id ],
                  [ Data        => $log ],
                  [ Caption     => $caption ],
                  [ Target      => lc($cat) . ":" . $keyval],
                  [ TargetName  => $report->id('display')],
                  [ Actionable  => $action],
                  [ MgmtLevel   => $mgmtLevel ],
                  [ SourceIP    => $rep->{'id.ipno'}],
                  [ Component   => $comp ],
                  [ Severity    => $sev ],
                  [ Description => $desc ] 
                         ]);
   $hm->FILTER($rep, $ev);

   $key = CIM::Key->new( ['NWS_System',
                   'Name'       => $keyval,
                    CreationClassName => 'NWS_System']);
   $sd = Events->sourceDetector({ event => $ev });

   $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

   $sev -= 0.2 if ($comp && $sev); # logEvent < OtherEvents < StateEvent

   my $rid = $report->id();
   $ed = Message->new( { id         => {%$rid}, 
                          severity  => $sev, 
                          instances => [$ev, @$sd, $pertains ],
                        });
   $pdm->saveMessage($ed);

   $hm->saveAlert($ev, $sev, $id, $report, $report->id("logFile") . "\n $log", $desc );

   State->saveState($cat,  $keyval, $comp || "alarm.${cat}Log", "Alarm", $sev, $desc, 1, $ev);

   return $ev;
}


sub fcEvent {
  my($hm, $FC) = @_;

  my($id)    =  PDM->getEventSequence ;
  my $renv = System->get_renv();
  my $tt1 = "san.fcEvent";
  require Data::Dumper;
  $Data::Dumper::Indent = 0;
  my $out = Data::Dumper::Dumper($FC);

  my $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => $tt1 ],
                  [ EventId     => $id  ],
                  [ Actionable  => 1],
                  [ Data        => $out ],
                         ]);

   my $ed = Message->new({ id  => {}, instances => [ $ev ],
                      });

   PDM->saveMessage($ed);
}
  

# pc = probable cause
# ra = recomm actions

sub linkEvent {
  my($hm, $report, $type1, $key1, $type2, $key2, $desc, $reads, $writes, $hba,
      $fc_type) = @_;

  my($ev, $key, $sd, $pertains, $ed, $ehba);
  my($id)    =  PDM->getEventSequence ;
  my $renv = System->get_renv();
  my $mgmtLevel = $renv->{solution} =~ /^se/? "C" : "D"; 

  my ($t1, $k1, $p1) = split(/\:/, $key1);
  my $keyval = $k1;

  my($e_hba, $k1, $host_hba, $f1, $f2, $f3, $hostid, $link1, $link2);
  my(@hba_list, $ix, $node1, $loc);

  my $tt1 = "san.LinkEvent_" . $fc_type;
  my $grid = Grid->getInfo($tt1, "$type1|$type2");
  Grid->setCode("$tt1.$grid->{comp}");

  $ev = CIM::Instance->new('NWS_Event', [
                  [ EventType   => $tt1 ],
                  [ EventId     => $id  ],
                  [ Target      => $key1], 
                  [ TargetName  => $key1],
                  [ Caption     => "${type1}|$type2" ],
                  [ Actionable  => 1],
                  [ SourceIP    => System->get_ipno() ],
                  [ MgmtLevel   => $mgmtLevel ],
                  [ Description => $desc ],
                  [ Data        => "target2=$key2" ],
                  [ Component   => "port.$p1" ],
                  [ Severity    => Message::SEVERITY_WARNING ],
                         ]);

   $sd = Events->sourceDetector({ event => $ev });

   my $system = CIM::Key->new( ['NWS_System',
                   'Name'             => $keyval,
                    CreationClassName => 'NWS_System']);

   $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $system ],
                    ]);

   $ed = Message->new({ id        => $report->id,
                        instances =>
                        [$ev, @$sd, @hba_list, $pertains ],
                      });

   PDM->saveMessage($ed);

   State->saveLinkState($type1, $key1, $type2, $key2, 
         Message::SEVERITY_WARNING, $desc, $reads, $writes, $ev);

   $hm->saveAlert($ev,  Message::SEVERITY_WARNING, $id, $report, "", $desc );

}
  
sub tempEvent {
  my($class, $report, $orep, $cat, $comp, $el, $shortid,
     $display_id, $cimkey, $high, $low) = @_;

  my($rep) = $report->content();
  my $temp = $rep->get("$comp.$el");
  return if ($temp !~ /\d+/ || $temp > 500);

  my($tran, $reps) = Transition->getTransition2({      
                             key => $cimkey,
                            code => "temp_$comp",
                           in_value => ($temp > $high),
                           out_value => ($temp < $low),
                                 });
  if ($tran =~ /IN/) { #lost  or lost-repeat
     $class->alarmEvent($comp, "", $report, $cimkey, 
            "$comp.$el in $cat $display_id: Temperature($temp) over $high",
             2, "$comp.Temp", {captionIsState => 1, nomap => 1});
 
  }
  if ($tran eq "OUT") { # can connect
     $class->alarmEvent($comp, "", $report, $cimkey, 
            "$comp.$el in $cat $display_id: Temperature($temp) back under $low",
             0, "$comp.Temp", {captionIsState => 1, nomap => 1} );
  }
}
  
# 2=outoband, 1=inband 3=both

sub connectionEvent {
  my($hm, $keyval, $report, $args) = @_;
  my($key, $ev, $sd, $pertains, $ed, $id, $text, $rc);
  my($cat) = $report->category();
  my($rep) = $report->content();
  my $name = $rep->get('id.name') if ($rep);

  if ($args->{method} == 3) {  # BOTH
    $text = "OutOfBand";

    my($tran, $reps) = Transition->getTransition({      key => $report->deviceName(),
                                         code => 'status',
                                        value => $report->status(),
                             transition_value => Report::STATUS_CANNOT_CONNECT,
                                       repeat => '12h',
                                 });
    $rc = $hm->connection("e", $keyval, $tran, $report, $text, $reps, $args);

    if (defined($rep->{'info.IB_status'})) {  # T3 only so far.
       my $host = $rep->{'info.IB_host'}; 

       $text = "InBand(" . Util->shortHostname($host) . ")";

       my($tran, $reps) = Transition->getTransition( {  key => $report->deviceName(),
                                         code => 'statusIB',
                                        value => $rep->{'info.IB_status'},
                             transition_value => Report::STATUS_CANNOT_CONNECT,
                                       repeat => '12h',
                                       });
       $hm->connection("ib", $keyval, $tran, $report, $text, $reps, $args);
    }
    return $rc;

  } else {
    my $ho = System->hostname();
    my $ei;
    if ($args->{method} == 1) {
       $text = "InBand($ho)";
       $ei = "ib";
    } else {
       $text = "OutOfBand($ho)";
       $ei = "e";
    }

    my($tran, $reps) = Transition->getTransition({      key => $report->deviceName(),
                                         code => 'status',
                                        value => $report->status(),
                             transition_value => Report::STATUS_CANNOT_CONNECT,
                                       repeat => '12h',
                                 });
    return $hm->connection($ei, $keyval, $tran, $report, $text, $reps, $args);
  }

}

sub connection {
  my($hm, $comp, $keyval, $tran, $report, $text, $reps, $args) = @_;
  my($pdm) = $hm->{pdm};
  my($key, $id, $desc);
  my($ev, $sd, $pertains, $ed);
  my($cat) = $report->category();
  my($rep) = $report->content();
  my($name)= $rep->get('id.name') if ($rep);
  my($connect_errs) = $args->{connect_errs} || $rep->get('id.connect_errs');

  my $mgmtLevel  = $report->value('id.mgmtLevel') || 'D';
  my $egrid_info = $comp eq "ib" ? "ib": "oob";

  my($id2)  = $report->id('display');

  $key = CIM::Key->new( ['NWS_System',
                   'Name'             => $keyval,
                    CreationClassName => 'NWS_System']);

  return 1 if ($tran eq "LOST");
  my $comp1 = $comp;
  $comp1 = "$comp." . $rep->{'id.caption'} if ($rep && $rep->{'id.caption'});

  if ($tran =~ /IN/) { #lost  or lost-repeat
     return 1 if (!$keyval);  # never had good connection
     return 1 if ($reps > 6);
     $id =  $pdm->getEventSequence;
     my $repeat = "(lost for " . ($reps*8) . " hours)" if ($reps > 0);
     $desc = "Lost communication ($text) with $cat $id2 $repeat: " .
             $connect_errs;
     Grid->setCode(lc($cat) . ".CommunicationLostEvent.$egrid_info");

     $ev = CIM::Instance->new('NWS_CommunicationLostEvent', [
                  [ EventType   => lc($cat) . ".CommunicationLostEvent"  ],
                  [ EventId     => $id  ],
                  [ Caption     => $comp1 ],
                  [ Description => $desc  ],
                  [ Target      => lc($cat) . ":" . $keyval],
                  [ TargetName  => $report->id('display')],
                  [ Method      => $args->{method} ],
                  [ MgmtLevel   => $mgmtLevel ],
                  [ Actionable  => 1    ],
                  [ SourceIP    => $rep->{'id.ipno'}],
                  [ Severity    => Message::SEVERITY_DOWN],
                  [ Timeout     => 20],
                         ]);
     $hm->FILTER($rep, $ev);
     $sd = Events->sourceDetector({ event => $ev });

     $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

     State->saveState($cat,  $keyval, $comp, $name, 
         Message::SEVERITY_DOWN, "Lost Communication ($text) with $id2", 0, $ev);

     $ed = Message->new( { id        => $report->id,
                            instances => [$ev, @$sd, $pertains ],
                          });
     $pdm->saveMessage($ed);

     $hm->saveAlert($ev,  Message::SEVERITY_DOWN, $id, $report, "", $desc);
     return 1;
  }
  if ($tran eq "OUT") { # can connect
     $desc = "Regained communication($text) with $cat $id2";
     $id =  $pdm->getEventSequence;
     Grid->setCode(lc($cat) . ".CommunicationEstablishedEvent.$egrid_info");

     $ev = CIM::Instance->new('NWS_CommunicationEstablishedEvent', [
                  [ EventType   => lc($cat) . ".CommunicationEstablishedEvent"  ],
                  [ EventId     => $id ],
                  [ Description => $desc ],
                  [ Target      => lc($cat) . ":" . $keyval],
                  [ TargetName  => $report->id('display')],
                  [ Caption     => $comp1 ],
                  [ SourceIP     => $rep->{'id.ipno'}],
                  [ Severity    => Message::SEVERITY_GREEN],
                  [ Method      => 1],
                         ]);
     $sd = Events->sourceDetector({ event => $ev });


     $pertains = CIM::Instance->new('NWS_EventPertainsToSystem', [
                  [ Event       => $ev  ],
                  [ Element     => $key ],
                    ]);

     State->saveState($cat,  $keyval, $comp, $name, 
                  Message::SEVERITY_GREEN, "Regained Communication ($text)", 1, $ev);

     $ed = Message->new( { id        => $report->id,
                            instances => [$ev, @$sd, $pertains ],
                            });

     $pdm->saveMessage($ed);

     $hm->saveAlert($ev,  Message::SEVERITY_WARNING, $id, $report,"",$desc);
  }
  return 0;  # keep going
}

#
# returns availability-state of the device (old,new, description) :
#    returns (1/0, 1/0, "Old=..., New=...");

sub status {
   croak("Health::status is abstract, must implement in device Health");
}

#
# arg = {logical => 1, descIsValue=> 1, info => "extra info"}
#
sub stateEvent {
  my($hm, $component, $report,  $rep, $orep, $cim1, $nserial, $keyval, $arg) = @_;
  my($key, $ev, $sd, $pertains, $ed , $id);
  my($ovalue, $nvalue, $old, $new, $map_sev, $map_act) = $hm->status($rep, $orep, $component);
  if (System->get_bad_health()) {
      $nvalue = "BAD"; $new = 0;
  }
  my $name   = $rep->get('id.name');
  my $ix     = index($component, ".");
  my $extra  = "($arg->{info})" if ($arg->{info});
  my $id2    = $report->id('display');
  my $error  = 0;
  my $cat    = $report->category();
  my $catcap = uc($cat);
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';

  my($comp0) = $component;
   if ($ovalue ne $nvalue || ($rep == $orep && !$new) ) { 
         my($newval); 
         if ($new eq "X") {
            $newval = "Unknown";
         } else {
            $newval = ($new)?"Available":"Not-Available";
         }
         if ($rep == $orep) {
            $old = 1;
            $ovalue = "unknown";
         }
         $id =  PDM->getEventSequence;
         my $xx = (index($component, $nserial) < 0) ? "($nserial) ":" ";
         my($desc);
         if ($arg->{descIsValue}) {
            $desc = "'$component'${xx}in $catcap $id2 is now $newval: $nvalue$extra";
         } else {
            $desc = "'$component'${xx}in $catcap $id2 is now $newval (state changed from '$ovalue' to '$nvalue')$extra";
         }

         if (!$nserial) {
             $arg->{noState} = 1;
             my $sev0 = $map_sev >= 0 ? $map_sev : 2;
             $arg->{nomap} = 1;
             my @clist = split(/\./, Grid->getCode(), 3);
             Grid->setCode("$clist[0].AlarmEvent.$clist[2]");
             Grid->setSign($new != 1 ? "M": "P");
             my $ev0 = $hm->alarmEvent($component, "", $report, $keyval, $desc, $sev0,"$cat.$component", $arg);
             if ($new != 1) { #  bad comp
                State->saveState($cat,  $keyval, $component, $name, $sev0, $desc, $new, $ev0);
                 $error++;
             } else {
                State->saveState($cat,  $keyval, $component, $name, Message::SEVERITY_GREEN, $desc, $new, $ev0);
             }
             return $error;
         }
         $old = 0 if ($old eq "X");
         my $yellow;
         my $sign = "X";
         if ($new eq "X") {
            $new = $old;
            $yellow =1;
         }
         my ($severity, $action);
         if (!$new) { #  bad comp
           $severity = Message::SEVERITY_ERROR;
           $action = 1;
           $sign = "M";
           $error++;
         } elsif ($yellow) {
           $severity = Message::SEVERITY_ERROR;
           $action = 1;
           $error++;
         } else {
           $severity = Message::SEVERITY_GREEN;
           $action = 0;
           $sign = "P";
         }
         $severity = $map_sev if ($map_sev >= 0);
         if ($map_act) {
           $action   = $map_act eq "Y"? 1:0;
         }
         Grid->setSign($sign);


         $ev = CIM::Instance->new('NWS_StateChangeEvent', [
             [ EventType    => lc($cat) . ".StateChangeEvent"   ],
             [ EventId      => $id ],
             [ PriorState   => $old],
             [ CurrentState => $new],
             [ MgmtLevel    => $mgmtLevel ],
             [ PriorValue   => $ovalue],
             [ CurrentValue => $nvalue],
             [ Actionable   => $action  ],
             [ Caption      => "$sign.$comp0"],
             [ Severity    => $severity ],
             [ Description  => $desc ],
             [ Target       => lc($cat) . ":" . $keyval],
             [ TargetName  => $report->id('display')],
             [ Component    => $component],
             [ SourceIP     => $rep->{'id.ipno'}],
                    ]);
         $hm->FILTER($rep, $ev);

         $sd = Events->sourceDetector({ event => $ev });

         $key = CIM::Key->new( [$cim1,
                  Tag               => $nserial,
                  CreationClassName => $cim1]);

         $pertains = CIM::Instance->new(
                $arg->{logical} ? 'NWS_EventPertainsToLogicalDevice': 
                                  'NWS_EventPertainsToPhysicalElement', [
             [ Event       => $ev  ],
             [ Element     => $key ],
               ]);
         my $sev2 = $severity;
         $sev2 += 0.1 if ($sev2); # logEvent < OtherEvents < StateEvent
         $ed = Message->new( { id        => $report->id,
                               severity  => $sev2, 
                               instances => [$ev, @$sd, $pertains ],
                       });

         PDM->saveMessage($ed);
         State->saveState($cat,  $keyval, $component, $name, $sev2, $desc, $new, $ev);
         $hm->saveAlert($ev,  $severity, $id, $report, "", $desc);

   }
   my $time_id = "$component.statusTime";
   $rep->{$time_id} = $orep->{$time_id};

   return $error;

}

# old/new are severities, ovalue and nvalue are descriptions
# new=1 (warning) new=2 (error) etc..

sub sevStateEvent {
  my($hm, $component, $report,  $rep, $orep, $cim1, $nserial, $keyval, $arg) = @_;
  my($key, $ev, $sd, $pertains, $ed , $id);

  my($ovalue, $nvalue, $old, $new, $oavail, $navail) = $hm->status($rep, $orep, $component);

  my $name   = $rep->get('id.name');
  my $data   = "";
  my $ix     = index($component, ".");
  my $id2    = $report->id('display');
  my $error  = 0;
  my $cat    = $report->category();
  my $catcap = uc($cat);
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';
  my $sign   = $navail eq "X" ? "X" : ($navail == 1 ? "P": "M");
  my($comp0) = $component;

  if ($ovalue ne $nvalue || ($rep == $orep && $new) ) { # availability change or new report  and unavailable
         $id =  PDM->getEventSequence;
         my $nvalue2 = $nvalue;
         $nvalue2 = substr($nvalue2,19) if ($nvalue2 =~ /^\d+\-\d\d/);
         my $desc = "$arg->{label} $catcap-$id2: $nvalue2";
         my $org  = State->eventHash($rep->{"$component.status-details"});

         my($g_info, $g_cause, $g_action, $g_code) = 
                      Grid->getInfoString($org->{EventType}, $org->{Caption});

         $ev = CIM::Instance->new('NWS_StateChangeEvent', [
             [ EventType    => lc($cat) . ".StateChangeEvent"   ],
             [ EventId      => $id ],
             [ PriorState   => "$oavail"],
             [ CurrentState => "$navail"],
             [ MgmtLevel    => $mgmtLevel ],
             [ PriorValue   => $ovalue],
             [ CurrentValue => $nvalue],
             [ Actionable   => ($new > 1)   ],
             [ Caption      => "$sign.$comp0"],
             [ Severity     => $new],
             [ Description  => $desc ],
             [ Target       => lc($cat) . ":" . $keyval],
             [ TargetName  => $report->id('display')],
             [ Data         => $data],
             [ Component    => $component],
             [ SourceIP     => $rep->{'id.ipno'}],
             [ OriginalEvent => $rep->{"$component.status-details"}. "|grid_code=$g_code" ],
                    ]);
         $hm->FILTER($rep, $ev);

         $sd = Events->sourceDetector({ event => $ev });

         if ($arg->{logical}) {              # logical (like NWS_Slot,
           $key = CIM::Key->new( [$cim1,    
                  CreationClassName => $cim1, 
                  SystemName        => $rep->{'id.wwn'} . "." . $component,
                  DeviceID          => $nserial 
                  ]);
         } else {
           $key = CIM::Key->new( [$cim1,    # physical (like NWS_PhysicalPackage)
                  Tag               => $nserial,
                  CreationClassName => $cim1
                  ]);
         }

         $pertains = CIM::Instance->new(
                $arg->{logical} ? 'NWS_EventPertainsToLogicalDevice': 
                                  'NWS_EventPertainsToPhysicalElement', [
             [ Event       => $ev  ],
             [ Element     => $key ],
               ]);

         $ed = Message->new( { id => $report->id,
                        instances => [$ev, @$sd, $pertains ], });

         PDM->saveMessage($ed);
         State->saveState($cat,  $keyval, $component, $name, $new, $desc, $navail, $ev,
                           { absolute => 1} );
         $hm->saveAlert($ev,  $new, $id, $report, "", $desc);

   }
   my $time_id = "$component.statusTime";
   $rep->{$time_id} = $orep->{$time_id};

   return $error;

}

sub removeCompEvent {
  my($hm, $component, $pf, $report,  $cim1, $oserial, $keyval, $arg) = @_;
  
  my($key, $ev, $sd, $pertains, $ed, $contain);
  my($pdm) = $hm->{pdm};
  my($cat) = $report->category();
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';
  my $extra_desc = $arg->{desc};
  my $logical = $arg->{logicalDevice};

  my($ix) = index($component, ".");
  my($comp0) = $component;

  my($id)  =  $pdm->getEventSequence;
  my($rep) = $report->content();
  my($name)= $rep->get('id.name') if ($rep);
  my($comp1) = $component;
  $comp1  = "" if (index($oserial, $component) >= 0);
  my($id2) = $report->id('display');
  my($desc) = "$comp1($oserial) was removed from $id2 $extra_desc";


  $ev = CIM::Instance->new('NWS_Event', [
             [ EventType    => lc($cat) . ".ComponentRemoveEvent"  ],
             [ Caption      => "remove.$comp0"],
             [ Description  => $desc ],
             [ MgmtLevel    => $mgmtLevel ],
             [ Actionable   => 1    ],
             [ Severity     => Message::SEVERITY_ERROR ],
             [ SourceIP     => $rep->{'id.ipno'}],
             [ Target       => lc($cat) . ":" . $keyval],
             [ Component       => $component],
             [ TargetName  => $report->id('display')],
             [ EventId     => $id ]]);
   $hm->FILTER($rep, $ev);

   $sd = Events->sourceDetector({ event => $ev });

   my @pert;
   if ($logical) {
      $key = CIM::Key->new( [$cim1,   # key of the disk
                  SystemName     => $oserial,
                  DeviceID       => $oserial]);

      $pertains = CIM::Instance->new('NWS_EventPertainsToLogicalDevice', [
             [ Event       => $ev  ],
             [ Element     => $key ],
               ]);
     push(@pert, $pertains);

   } else {
     $key = CIM::Key->new( [$cim1,   # key of the disk
                  Tag               => $oserial,
                  CreationClassName => $cim1]);

     $contain = CIM::Instance->new( 'CIM_Container', [
                  [GroupComponent => $pf],            # frame is a package
                  [PartComponent  => $key],           # package is an element
                  [LocationWithinContainer => $component],
                   ]);

     $pertains = CIM::Instance->new('NWS_EventPertainsToContainer', [
                  [Event          => $ev   ],
                  [Association    => $contain ],
                  [DiscoveryType => "D"   ],   # delete
               ]);
     push(@pert, $contain, $pertains);
   }

   State->saveState($cat,  $keyval, $component, $name, Message::SEVERITY_ERROR, $desc, 0, $ev);

   $ed = Message->new( { id        => $report->id,
                       instances => [$ev, @$sd, @pert ],
                       severity  => Message::SEVERITY_ERROR});

   $pdm->saveMessage($ed);
   $hm->saveAlert($ev,  Message::SEVERITY_ERROR, $id, $report, "",$desc);
}

sub insertCompEvent {
  my($hm, $component, $pf, $report, $part, $nserial, $keyval, $arg) = @_;
  my($key, $ev, $sd, $pertains, $ed, $contain, $pertains2);

  my($pdm) = $hm->{pdm};
  my($cat) = $report->category();
  my $extra_desc = $arg->{desc};
  my $logical = $arg->{logicalDevice};

  my($ix) = index($component, ".");
  my($comp0) = $component;
  my $mgmtLevel = $report->value('id.mgmtLevel') || 'D';

  my($rep) = $report->content();
  my($name)= $rep->get('id.name') if ($rep);

  my($id) =  $pdm->getEventSequence;
  my($id2) = $report->id('display');

  my($comp1) = $component;
  $comp1  = "" if (index($nserial, $component) >= 0);

  my($desc) = "$comp1($nserial) was added to $id2 $extra_desc";

  $ev = CIM::Instance->new('NWS_Event', [  
             [ EventType    => lc($cat) . ".ComponentInsertEvent"  ],
             [ Caption      => "add.$comp0"],
             [ Description  => $desc ],
             [ MgmtLevel    => $mgmtLevel ],
             [ Target       => lc($cat) . ":" . $keyval],
             [ TargetName   => $report->id('display')],
             [ Severity     => Message::SEVERITY_NORMAL ],
             [ Component    => $component],
             [ SourceIP     => $rep->{'id.ipno'}],
             [ EventId      => $id ]]);

  $sd = Events->sourceDetector({ event => $ev });
  my @pert;
  if ($logical) {
     $pertains = CIM::Instance->new('NWS_EventPertainsToLogicalDevice', [
             [ Event       => $ev  ],
             [ Element     => $part->[0]],
               ]);
     push(@pert, $pertains);
  } else {
    $pertains = CIM::Instance->new('NWS_EventPertainsToPhysicalElement', [
             [ Event       => $ev  ],
             [ Element     => $part->[1]],
               ]);

    $contain = CIM::Instance->new('CIM_Container', [
              [GroupComponent           => $pf],         #PhysicalFrame
              [PartComponent            => $part->[1]],
              [LocationWithinContainer  => $component],
               ]);

    $pertains2 = CIM::Instance->new('NWS_EventPertainsToContainer', [
              [Event                    => $ev],
              [Association             => $contain  ],
              [DiscoveryType            => "A"       ],
               ]);
     push(@pert, $pertains, $contain, $pertains2);
  }

  State->saveState($cat,  $keyval, $component, $name, Message::SEVERITY_NORMAL, $desc, 1, $ev);

  $ed = Message->new( { id        => $report->id,
                       instances => [$ev, @$sd, @$part, @pert],
                       severity  => Message::SEVERITY_ERROR});

  $pdm->saveMessage($ed);
  $hm->saveAlert($ev,  Message::SEVERITY_WARNING, $id, $report, "", $desc);
}

1;
