/*
 * Copyright 1999-2002 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

/*
 * Modified by Fujitsu Limited to provide support
 * for Solaris on Fujitsu's hardware platforms
 */

#ifndef	_SYS_CPC_EVENT_H
#define	_SYS_CPC_EVENT_H

#pragma ident	"@(#)cpc_event.h	1.4	02/10/28 SMI"

/*
 * CPU Performance Counters measure 'events', as captured
 * by the processor dependent data structures shown below.
 * The meaning of the events, and the pattern of bits placed
 * in the control register is managed by the routines in libcpc.
 */

#include <sys/inttypes.h>
#include <sys/time.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(__sparc)

#ifdef	FJCPU

/*
 * Fujitsu SPARC64-III processor:
 *
 * There are 40 performance counters on this processor.  Each counter is
 * 32-bit. Performance events can be captured simultaneously on all of
 * these counters.
 */

#define	SPARC64_NPMC		39
#define	SPARC64_PMC_VISIBLE	6
#define	SPARC64_PMC_NREGS	2

/*
 * Unlike UltraSPARC, we do not have the %pcr register to control
 * performance monitoring in FJCPU.  In stead, performance monitoring
 * control bits are spread in two registers:
 *
 * 	The two performance counter views (pm0_view and pm1_view)
 * 	are selected by writing an 8 bit value in %as30.
 *
 * 	On the other hand, user access and mode selection bits are located
 * 	in %scr (bits [13:15]).
 * 
 * We still retain the ce_pcr field as a "software register" which holds
 * all the control bits in one place and of course, having the same 
 * field name makes porting much easier.  The composite ce_pcr has the
 * follwoing format:
 *
 * +---------------------+--------+-----+-------+-----------+-----------+
 * |    UNUSED           |PMEN_SEL|PM_US|Unused | pm1_view# | pm0_view# |
 * +---------------------+--------+-----+-------+-----------+-----------+
 * 63                    15     14    13 12      7           3          0
 */

typedef struct _cpc_event {
	int ce_cpuver;
	hrtime_t ce_hrt;	/* gethrtime() */
	uint64_t ce_tick;	/* virtualized %tick */
	uint64_t ce_pcr;	/* PM related bits in the %scr and %asr30 */
	uint64_t ce_pic[SPARC64_PMC_VISIBLE];	/* virtualized perf. counters */
} cpc_event_t;

#else	/* FJCPU */

/*
 * UltraSPARC I, II and III processors
 *
 * The performance counters on these processors allow up to two 32-bit
 * performance events to be captured simultaneously from a selection
 * of metrics.   The metrics are selected by writing to the performance
 * control register, and subsequent values collected by reading from the
 * performance instrumentation counter registers.  Both registers are
 * priviliged by default, and implemented as ASRs.
 */

typedef struct _cpc_event {
	int ce_cpuver;
	hrtime_t ce_hrt;	/* gethrtime() */
	uint64_t ce_tick;	/* virtualized %tick */
#if defined(FJPS1)
	uint64_t ce_pic[16];	/* virtualized %pic */
#else /* FJPS1 */
	uint64_t ce_pic[2];	/* virtualized %pic */
#endif /* FJPS1 */
	uint64_t ce_pcr;	/* %pcr */
#if defined(FJPS1)
	uint_t ce_selpic[16];	/* virtualized %pic selection */
#endif /* FJPS1 */
} cpc_event_t;

#endif	/* FJCPU */

#define	CPC_TICKREG_NAME	"%tick"
#define	CPC_TICKREG(ev)		((ev)->ce_tick)

/*
 * "Well known" bitfields in the UltraSPARC %pcr register
 * The interfaces in libcpc should make these #defines uninteresting.
 */
#define	CPC_ULTRA_PCR_USR		2
#define	CPC_ULTRA_PCR_SYS		1
#define	CPC_ULTRA_PCR_PRIVPIC		0

#define	CPC_ULTRA_PCR_PIC0_SHIFT	4
#define	CPC_ULTRA2_PCR_PIC0_MASK	UINT64_C(0xf)
#define	CPC_ULTRA3_PCR_PIC0_MASK	UINT64_C(0x3f)
#define	CPC_ULTRA_PCR_PIC1_SHIFT	11
#define	CPC_ULTRA2_PCR_PIC1_MASK	UINT64_C(0xf)
#define	CPC_ULTRA3_PCR_PIC1_MASK	UINT64_C(0x3f)

#if defined(FJPS1)

#define	CPC_SPARC64V_NPIC		8

/* FJPS1 PCR extension */

#define	CPC_SPARC64V_PCR_ULRO		3
#define	CPC_SPARC64V_PCR_SC_SHIFT	18
#define	CPC_SPARC64V_PCR_SC_MASK	UINT64_C(0x7)
#define	CPC_SPARC64V_PCR_NC_SHIFT	22
#define	CPC_SPARC64V_PCR_NC_MASK	UINT64_C(0x7)
#define	CPC_SPARC64V_PCR_OVRO		26
#define	CPC_SPARC64V_PCR_OVF_SHIFT	32
#define	CPC_SPARC64V_PCR_OVF_MASK	UINT64_C(0xffff)

#endif /* FJPS1 */
#ifdef	FJCPU

/* Performance Monitoring bits in %scr ([13:15]) */

#define	CPC_SPARC64_PM_U_ACCESS		0x2000
#define	CPC_SPARC64_PM_U_ONLY		0x4000
#define	CPC_SPARC64_PM_S_ONLY		0x8000
#define	CPC_SPARC64_PM_MASK		0xC000

/* Performance Monitoring bits in %asr30 */

#define	CPC_SPARC64_PMV0_SHIFT		0
#define	CPC_SPARC64_PMV0_MASK	UINT64_C(0xf)

#define	CPC_SPARC64_PMV1_SHIFT		4
#define	CPC_SPARC64_PMV1_MASK	UINT64_C(0xf)

#define	CPC_SPARC64_PMV_MASK					\
	((CPC_SPARC64_PMV1_MASK << CPC_SPARC64_PMV1_SHIFT) |	\
	 					CPC_SPARC64_PMV0_MASK)

#define	CPC_SPARC64_V0_MAX	8	/* Legal values for pm0_view: 0 - 8 */
#define	CPC_SPARC64_V1_MAX	6	/* Legal values for pm1_view: 0 - 6 */

#define	CPC_SPARC64_PMV_MAX	\
	((CPC_SPARC64_V1_MAX << CPC_SPARC64_PMV1_SHIFT) | CPC_SPARC64_V0_MAX)

#endif	/* FJCPU */


#elif defined(__i386)

/*
 * Pentium I, II and III processors
 *
 * These CPUs allow pairs of events to captured.
 * The hardware counters count up to 40-bits of significance, but
 * only allow 32 (signed) bits to be programmed into them.
 * Pentium I and Pentium II processors are programmed differently, but
 * the resulting counters and timestamps can be handled portably.
 */

typedef struct _cpc_event {
	int ce_cpuver;
	hrtime_t ce_hrt;	/* gethrtime() */
	uint64_t ce_tsc;	/* virtualized rdtsc value */
	uint64_t ce_pic[2];	/* virtualized PerfCtr[01] */
	uint32_t ce_pes[2];	/* Pentium II */
#define	ce_cesr	ce_pes[0]	/* Pentium I */
} cpc_event_t;

#define	CPC_TICKREG_NAME	"tsc"
#define	CPC_TICKREG(ev)		((ev)->ce_tsc)

/*
 * "Well known" bit fields in the Pentium CES register
 * The interfaces in libcpc should make these #defines uninteresting.
 */
#define	CPC_P5_CESR_ES0_SHIFT	0
#define	CPC_P5_CESR_ES0_MASK	0x3f
#define	CPC_P5_CESR_ES1_SHIFT	16
#define	CPC_P5_CESR_ES1_MASK	0x3f

#define	CPC_P5_CESR_OS0		6
#define	CPC_P5_CESR_USR0	7
#define	CPC_P5_CESR_CLK0	8
#define	CPC_P5_CESR_PC0		9
#define	CPC_P5_CESR_OS1		(CPC_P5_CESR_OS0 + 16)
#define	CPC_P5_CESR_USR1	(CPC_P5_CESR_USR0 + 16)
#define	CPC_P5_CESR_CLK1	(CPC_P5_CESR_CLK0 + 16)
#define	CPC_P5_CESR_PC1		(CPC_P5_CESR_PC0 + 16)

/*
 * "Well known" bit fields in the Pentium Pro PerfEvtSel registers
 * The interfaces in libcpc should make these #defines uninteresting.
 */
#define	CPC_P6_PES_INV		23
#define	CPC_P6_PES_EN		22
#define	CPC_P6_PES_INT		20
#define	CPC_P6_PES_PC		19
#define	CPC_P6_PES_E		18
#define	CPC_P6_PES_OS		17
#define	CPC_P6_PES_USR		16

#define	CPC_P6_PES_UMASK_SHIFT	8
#define	CPC_P6_PES_UMASK_MASK	(0xffu)

#define	CPC_P6_PES_CMASK_SHIFT	24
#define	CPC_P6_PES_CMASK_MASK	(0xffu)

#define	CPC_P6_PES_PIC0_MASK	(0xffu)
#define	CPC_P6_PES_PIC1_MASK	(0xffu)

#else
#error	"performance counters not available on this architecture"
#endif

/*
 * Flag arguments to cpc_bind_event and cpc_ctx_bind_event
 */
#define	CPC_BIND_LWP_INHERIT	(0x1)
#define	CPC_BIND_EMT_OVF	(0x2)

/*
 * ce_cpuver values
 */

#define	CPC_ULTRA1		1000
#define	CPC_ULTRA2		1001	/* same as ultra1 for these purposes */
#define	CPC_ULTRA3		1002
#define	CPC_ULTRA3_PLUS		1003
#define	CPC_ULTRA3_I		1004

#define	CPC_PENTIUM		2000
#define	CPC_PENTIUM_MMX		2001
#define	CPC_PENTIUM_PRO		2002
#define	CPC_PENTIUM_PRO_MMX	2003

#define	CPC_SPARC64_III		3000
#define	CPC_SPARC64_V		3002

#ifdef __cplusplus
}
#endif

#endif	/* _SYS_CPC_EVENT_H */
