#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_shelf.pm 1.8     02/10/31 SMI"
# ------------------------------------------------------------------------------

BEGIN { 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_switch.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_drawer.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_utils.pl"; 
}

# ------------------------------------------------------------------------------

package sl_shelf;

# ID fields separator
my $ID_SEP = '@';

# ID min and max values
$ID_MIN = 0;
$ID_MAX = 255;

#  node object attributes
my @SHELF_ATTR = qw( Id
                     Name
		     SwitchLst
		     DrawerLst );

%SHELF_DEF_COMP_MASK = ( 'Id' => 0,
                	 'Name' => 1,
			 'SwitchLst' => 0,
			 'DrawerLst' => 1 );

# ------------------------------------------------------------------------------
#
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} ) &&
	( defined $attr{'Name'} ) &&
	( defined $attr{'SwitchLst'} ) &&
	( defined $attr{'DrawerLst'} ) ) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      $self->{'SwitchLst'} = $attr{'SwitchLst'};
      $self->{'DrawerLst'} = $attr{'DrawerLst'};
   }
   elsif ( defined $attr{'ShelfObject'} ) {
      $self->_construct( $attr{'ShelfObject'} );
   }
   else {
      &sl_traces::error( "Unable to create %s object. ".
                         "Invalid creation method", 
			 "- sl_shelf::new - ".__LINE__,
			 ( "sl_shelf" ) );
   }
   &sl_traces::trace( 2, "- sl_shelf::new - Creating shelf object ".
                "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $shelfObject ) = @_;
   &sl_traces::trace( 3, "- sl_shelf::_construct - Building shelf object" );
   my $shelf = sl_iterator->new( 'Object' => $shelfObject );
   $self->{'Id'} = $shelf->extract;
   $self->{'Name'} = $shelf->extract;
   @{$self->{'SwitchLst'}} = 
      $shelf->extractObjectArray( "sl_switch", "SwitchObject" );
   @{$self->{'DrawerLst'}} = 
      $shelf->extractObjectArray( "sl_drawer", "DrawerObject" );
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_shelf::serialize - Serializing shelf object" );
   my $shelfObject = sl_iterator->new( 'Object' => "" );
   foreach $attr ( @SHELF_ATTR ) {
      if ( $attr eq 'DrawerLst' ) {
         $shelfObject->serializeObjectArray( @{$self->{'DrawerLst'}} );
      }
      elsif ( $attr eq 'SwitchLst' ) {
         $shelfObject->serializeObjectArray( @{$self->{'SwitchLst'}} );
      }
      else {
         $shelfObject->serialize( $self->{$attr} );
      }
   }
   return $shelfObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_shelf::display - Displaying shelf" );
   print STDOUT "\nShelf\n";
   print STDOUT "-----\n";
   foreach ( @SHELF_ATTR ) {
      if ( $_ eq 'DrawerLst' ) {
         print STDOUT "$_ = \n";
         foreach $drawer (@{$self->{'DrawerLst'}}) {
	    $drawer->display( $brief );
	 }
      }
      elsif ( $_ eq 'SwitchLst' ) {
         print STDOUT "$_ = \n";
         foreach $switch (@{$self->{'SwitchLst'}}) {
	    $switch->display( $brief );
	 }
      }
      else {
         print STDOUT "$_ = ";
         print STDOUT "$self->{$_}" if ( defined $self->{$_} );
         print STDOUT "\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub getDrawerLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_shelf::getDrawerLst - Getting < $self->{'Name'} > ".
                "drawer list" );
   if ( defined $self->{'DrawerLst'} ) {
      return @{$self->{'DrawerLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getSwitchLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_shelf::getSwitchLst - Getting ".
                     "< $self->{'Name'} > switch list" );
   if ( defined $self->{'SwitchLst'} ) {
      return @{$self->{'SwitchLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getBoardLst {
   my $self = shift;
   my ( $drawerOid ) = @_;
   &sl_traces::trace( 2, "- sl_shelf::getBoardLst - Getting shelf ".
                "< $self->{'Name'} > board list" );
   my @drawerLst = $self->getDrawerLst;
   my @boardLst = ();
   foreach $drawer ( @drawerLst ) {
      push( @boardLst, $drawer->getBoardLst );
   }
   return @boardLst;
}

# ------------------------------------------------------------------------------
#
sub getDiskLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_shelf::getDiskLst - Getting shelf ".
                      "< $self->{'Name'} > disk list" );
   my @drawerLst = $self->getDrawerLst;
   my @diskLst = ();
   foreach $drawer ( @drawerLst ) {
      push( @diskLst, $drawer->getDiskLst );
   }
   return @diskLst;
   
}

# ------------------------------------------------------------------------------
#
sub getDrawer {
   my $self = shift;
   my ( $drawerOid ) = @_;
   &sl_traces::trace( 2, "- sl_shelf::getDrawer - Getting drawer ".
                     "< ${drawerOid} >" );
   foreach $drawer ( @{$self->{'DrawerLst'}} ) {
      return $drawer if ( $drawer->iD eq $drawerOid ); 
   }
   return undef;
}

# ------------------------------------------------------------------------------
#
sub getBoard {
   my $self = shift;
   my ( $boardOid ) = @_;
   &sl_traces::trace( 2, "- sl_shelf::getBoard - Getting board ".
                     "< ${boardOid} >" );
   my @boardLst = $self->getBoardLst;
   foreach $board ( @boardLst ) {
      return $board if ( $board->getName eq $boardOid ); 
   }
   return undef;
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_shelf::getName - Getting < $self->{'Name'} > ".
                      "Name attribute" );
   return $self->{'Name'}; 
}

# ------------------------------------------------------------------------------
#
sub setDrawerLst {
   my $self = shift;
   my ( @drawerLst ) = @_;
   &sl_traces::trace( 2, "- sl_shelf::setDrawerLst - Setting ".
                      "< $self->{'Name'} > Drawer list" );
   if ( defined $drawerLst[0] ) {
      @{$self->{'DrawerLst'}} = @drawerLst;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_shelf::setDrawerLst - ".__LINE__,
			 ( "Drawer list" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub setSwitchLst {
   my $self = shift;
   my ( @switchLst ) = @_;
   &sl_traces::trace( 2, "- sl_shelf::setSwitchLst - Setting ".
                      "< $self->{'Name'} > Switch list" );
   if ( defined $switchLst[0] ) {
      @{$self->{'SwitchLst'}} = @switchLst;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_shelf::setSwitchLst - ".__LINE__,
			 ( "Switch list" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub compare {
   my $class = shift;
   my ( $shelf1, $shelf2 ) = @_;
   my $shelf1Name = ( defined $shelf1 ) ? $shelf1->getName:"<undefined>";
   my $shelf2Name = ( defined $shelf2 ) ? $shelf2->getName:"<undefined>";
   &sl_traces::trace( 2, "- sl_shelf::compare - Comparing shelves ".
                     "< ${shelf1Name} > and < ${shelf2Name} >" );
   # String holding differences if any
   my $diffStr = undef;
   # Extract the differences if any
   foreach $attr ( keys %SHELF_DEF_COMP_MASK ) {
      $diffStr1 = "Comparing shelf : ${shelf1Name}.";
      $diffStr2 = "       to shelf : ${shelf2Name}.";
      if ( $SHELF_DEF_COMP_MASK{$attr} ) {
         if ( &sl_utils::testAttribute( $shelf1->{$attr} ) and 
	      &sl_utils::testAttribute( $shelf2->{$attr} ) ) {
            if ( $attr eq 'DrawerLst' ) {
	       # Sort the shelves lists for comparison
	       $shelf1->setDrawerLst( 
	          &sl_utils::sortListByName( $shelf1->getDrawerLst ) );
	       $shelf2->setDrawerLst( 
	          &sl_utils::sortListByName( $shelf2->getDrawerLst ) );
               # Look for different drawers
	       @shelf1DrawerLst = $shelf1->getDrawerLst;
	       @shelf2DrawerLst = $shelf2->getDrawerLst;
	       for ($i=0; $i < scalar @shelf1DrawerLst; $i++) {
		  $diff = sl_drawer->compare( 
		           $shelf1DrawerLst[$i], $shelf2DrawerLst[$i] );
		  $diffStr .= $diffStr1."DrawerLst[$i]\n".
		              $diffStr2."DrawerLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
            elsif ( $attr eq 'SwitchLst' ) {
	       # Sort the shelves lists for comparison
	       $shelf1->setSwitchLst( 
	          &sl_utils::sortListByName( $shelf1->getSwitchLst ) );
	       $shelf2->setSwitchLst( 
	          &sl_utils::sortListByName( $shelf2->getSwitchLst ) );
               # Look for different switches
	       @shelf1SwitchLst = $shelf1->getSwitchLst;
	       @shelf2SwitchLst = $shelf2->getSwitchLst;
	       for ($i=0; $i < scalar @shelf1SwitchLst; $i++) {
		  $diff = sl_switch->compare( 
		           $shelf1SwitchLst[$i], $shelf2SwitchLst[$i] );
		  $diffStr .= $diffStr1."SwitchLst[$i]\n".
		              $diffStr2."SwitchLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
	    else {
	       if ( $shelf1->{$attr} ne $shelf2->{$attr} ) {
        	  $diffStr1 .= "${attr}=".$shelf1->{$attr}."\n";
		  $diffStr2 .= "${attr}=".$shelf2->{$attr}."\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }
            }
	 }
         elsif ( ! &sl_utils::testAttribute( $shelf1->{$attr} ) and
	           &sl_utils::testAttribute( $shelf2->{$attr} ) )  {
            unless ( ( $attr eq 'SwitchLst' ) or
                     ( $attr eq 'DrawerLst' ) ) {
	       $diffStr1 .= "${attr}= <undefined>\n";
	       $diffStr2 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
	    }
	    else {
	       if ( scalar @{$shelf2->{$attr}} > 0 ) {
		  $diffStr1 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
		}       
	    }
         }
         elsif ( ! &sl_utils::testAttribute( $shelf2->{$attr} ) and
	           &sl_utils::testAttribute( $shelf1->{$attr} ) ) {
            unless ( ( $attr eq 'SwitchLst' ) or
                     ( $attr eq 'DrawerLst' ) ) {
               $diffStr2 .= "${attr}= <undefined>\n";
	       $diffStr1 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
 	    }
	    else {
	       if ( scalar @{$shelf1->{$attr}} > 0 ) {
		  $diffStr2 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }       
	    }
	 }
      }
   }
   return $diffStr;
}

# ------------------------------------------------------------------------------
#
sub deploy {
   my $self = shift;
   ( $destDir ) = @_;
   &sl_traces::trace( 2, "- sl_shelf::deploy - Configuring shelf ".
                     "< $self->{'Name'} >" );
   # Configure switches
   my @switchLst = $self->getSwitchLst;
   foreach $switch ( @switchLst ) {
      $switch->deploy( $destDir );
   }
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   $id = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_shelf::iD - Shelf iD is < ${id} >" );
   return $id;
}

# ------------------------------------------------------------------------------

return 1;
