#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_node.pm 1.18     02/11/13 SMI"
# ------------------------------------------------------------------------------


BEGIN { 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_package.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_rnfs.pl"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_cmm.pl"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_board.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_group.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_utils.pl"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_registry.pl"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
}

# ------------------------------------------------------------------------------

package sl_node;

# Node Type constants
%NODE_TYPE = ( 'MASTER_ELIGIBLE' => "MASTER_ELIGIBLE", 
               'DISKLESS' => "DISKLESS",
	       'DATALESS' => "DATALESS" );
	       
# Node Type constants
%NODE_SERVICE_TYPE = ( 'MASTER_ELIGIBLE' => "MASTER_ELIGIBLE", 
                       'DISKLESS' => "DISKLESS",
	               'DATALESS' => "DATALESS" );

# Node Os constants
%NODE_OS = ( 'SOLARIS' => "SOLARIS",
             'LINUX' => "LINUX"  );

# Node hidden enum values
%NODE_HIDDEN_VALUES = (
	     'MASTER_ELIGIBLE' => 0, 
	     'DISKLESS' => 0,
	     'DATALESS' => 0,
	     'SOLARIS' => 0,
	     'LINUX' => 1 );

# Node ID range 
$NODE_ID_MIN = 1;
$NODE_ID_MAX = ( 16**2 ) * 255 + ( 16**1 ) * 255 + ( 16**0 ) + 255;

# Mapping OS / directory
%OS2DIR = ( "${NODE_OS{'SOLARIS'}}" => "solaris" );

# ID fields separator
my $ID_SEP = '@';

#  node object attributes
my @NODE_ATTR = qw( Id
                    Name
		    Type
		    Os
		    Arch
		    Drawer
		    Board
		    GroupName
		    DiskLst );

%NODE_DEF_COMP_MASK = ( 'Id' => 1,
                        'Name' => 1,
		        'Type' => 1,
			'Os' => 1,
			'Arch' => 1,
			'Drawer' => 0,
			'Board' => 1,
			'GroupName' => 1,
			'DiskLst' => 1 );

# Node specific config. install. scripts
$NODE_PRE_INSTALL_SCRIPT = "confinstall.pre.sh";
$NODE_POST_INSTALL_SCRIPT = "confinstall.post.sh";
$NODE_DATA_INSTALL_SCRIPT = "confinstall.sh";

# ------------------------------------------------------------------------------
# 
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} ) && 
        ( defined $attr{'Name'} ) ) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      # Set optional attributes
      $self->{'Arch'} = $attr{'Arch'} 
         if ( defined $attr{'Arch'} );
      $self->{'Drawer'} = $attr{'Drawer'} 
         if ( defined $attr{'Drawer'} );
      $self->{'DiskLst'} = $attr{'DiskLst'} 
         if ( defined $attr{'DiskLst'} );
      $self->{'Type'} = $attr{'Type'} 
         if ( defined $attr{'Type'} );
      $self->{'Os'} = $attr{'Os'} 
         if ( defined $attr{'Os'} );
      $self->{'Board'} = $attr{'Board'} 
         if ( defined $attr{'Board'} );
      $self->{'GroupName'} = $attr{'GroupName'} 
         if ( defined $attr{'GroupName'} );
   }
   elsif ( defined $attr{'NodeObject'} ) {
      $self->_construct( $attr{'NodeObject'} );
   }
   else {
      &sl_traces::error( "Unable to create node object. ".
                	 "Invalid creation method", 
			 "- sl_node::new - ".__LINE__,
			 ( "sl_node" ) );
   }
   &sl_traces::trace( 2, "- sl_node::new - Creating node object ".
                "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $nodeObject ) = @_;
   &sl_traces::trace( 3, "- sl_node::_construct - Building node object" );
   my $node = sl_iterator->new( 'Object' => $nodeObject );
   $self->{'Id'} = $node->extract;
   $self->{'Name'} = $node->extract;
   $self->{'Type'} = $node->extract;
   $self->{'Os'} = $node->extract;
   $self->{'Arch'} = $node->extract;
   $self->{'Drawer'} = $node->extract;
   $self->{'Board'} = $node->extract;
   $self->{'GroupName'} = $node->extract;
   @{$self->{'DiskLst'}} = $node->extractScalarArray;
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::serialize - Serializing node object" );
   my $nodeObject = sl_iterator->new( 'Object' =>  "" );
   foreach $attr ( @NODE_ATTR ) {
      if ( $attr eq 'DiskLst' ) {
	 $nodeObject->serializeScalarArray( @{$self->{'DiskLst'}} );
      }
      else {
	 $nodeObject->serialize( $self->{$attr} );
      }
   }
   return $nodeObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_node::display - Displaying node" );
   print STDOUT "\nNode\n";
   print STDOUT "----\n";
   foreach ( @NODE_ATTR ) {
      if ( defined $self->{$_} ) {
         if ( $_ eq 'DiskLst' ) {
            print STDOUT "$_ = ";
            foreach $diskId ( @{$self->{'DiskLst'}} ) {
	       print STDOUT $diskId." ";
	    }
	    print STDOUT "\n";
	 }
	 else {
	    print STDOUT "$_ = ";
	    print STDOUT "$self->{$_}";
	    print STDOUT "\n";
	 }
      }
      else {
	 print STDOUT "$_ =\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub compare {
   my $class = shift;
   my ( $node1, $node2 ) = @_;
   my $node1Name = ( defined $node1 ) ? $node1->getName:"<undefined>";
   my $node2Name = ( defined $node2 ) ? $node2->getName:"<undefined>";
   &sl_traces::trace( 2, "- sl_node::compare - Comparing node ".
                     "< ${node1Name} > and < ${node2Name} >" );
   # String holding differences if any
   my $diffStr = undef;
   my $item1 = undef;
   my $item2 = undef;
   # Extract the differences if any
   foreach $attr ( keys %NODE_DEF_COMP_MASK ) {
      $diffStr1 = "Comparing node : ${node1Name}.";
      $diffStr2 = "       to node : ${node2Name}.";
      if ( $NODE_DEF_COMP_MASK{$attr} ) {
         if ( &sl_utils::testAttribute( $node1->{$attr} ) and 
	      &sl_utils::testAttribute( $node2->{$attr} ) ) {
	    if ( $attr eq 'DiskLst' ) {
               # Look for differences
	       @node1DiskLst = sort $node1->getDiskOidLst;
	       @node2DiskLst = sort $node2->getDiskOidLst;
	       for ($i=0; $i < scalar @node1DiskLst; $i++) {
		  if ( $node1DiskLst[$i] ne $node2DiskLst[$i] ) {
		     $diffStr .= 
		        $diffStr1.$attr."[$i]=".$node1DiskLst[$i]."\n".
		        $diffStr2.$attr."[$i]=".$node2DiskLst[$i]."\n";
	          }
	       }
	    }
	    else {
	       if ( $node1->{$attr} ne $node2->{$attr} ) {
        	  $diffStr1 .= "${attr}=".$node1->{$attr}."\n";
		  $diffStr2 .= "${attr}=".$node2->{$attr}."\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }
	    }
	 }
         elsif ( ! &sl_utils::testAttribute( $group1->{$attr} ) and
	           &sl_utils::testAttribute( $group2->{$attr} ) )  {
            unless ( $attr eq 'DiskLst' ) {
	       $diffStr1 .= "${attr}= <undefined>\n";
	       $diffStr2 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
	    }
	    else {
	       if ( scalar @{$node2->{$attr}} > 0 ) {
		  $diffStr1 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
		}       
	    }
         }
         elsif ( ! &sl_utils::testAttribute( $group2->{$attr} ) and
	           &sl_utils::testAttribute( $group1->{$attr} ) ) {
            unless ( $attr eq 'DiskLst' ) {
               $diffStr2 .= "${attr}= <undefined>\n";
	       $diffStr1 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
 	    }
	    else {
	       if ( scalar @{$node1->{$attr}} > 0 ) {
		  $diffStr2 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }       
	    }
	 }
      }
   }
   return $diffStr;
}

# ------------------------------------------------------------------------------
#
sub setBoardObject {
   my $self = shift;
   my ( $board ) = @_;
   &sl_traces::trace( 2, "- sl_node::setBoardObject - Setting board object".
                      "to the node < $self->{'Name'} >" );
   $self->{'BoardObject'} = $board;
}

# ------------------------------------------------------------------------------
#
sub getBoardObject {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getBoardObject - Getting board object ".
                      "of the node < $self->{'Name'} >" );
   return $self->{'BoardObject'};
}

# ------------------------------------------------------------------------------
#
sub setDrawer {
   my $self = shift;
   my ( $drawerOid ) = @_;
   &sl_traces::trace( 2, "- sl_node::setDrawer - Setting drawer reference ".
                      "< ${drawerOid} > to the node < $self->{'Name'} >" );
   $self->{'Drawer'} = $drawerOid;
}

# ------------------------------------------------------------------------------
#
sub getDrawer {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getDrawer - Getting drawer attribute ".
                "of node < $self->{'Name'} >" );
   return $self->{'Drawer'};
}

# ------------------------------------------------------------------------------
#
sub isMasterEligible {
   my $self = shift;
   &sl_traces::trace( 3, "- sl_node::isMasterEligible - Determining node Type for ".
                "< $self->{'Name'} >" );
   if ( $self->{'Type'} eq $NODE_TYPE{'MASTER_ELIGIBLE'} ) {
      return 1;
   }
   else {
      return 0;
   }
}

# ------------------------------------------------------------------------------
#
sub isDataless {
   my $self = shift;
   &sl_traces::trace( 3, "- sl_node::isDataless - Determining group Type for ".
                "< $self->{'Name'} >" );
   if ( $self->{'Type'} eq $NODE_TYPE{'DATALESS'} ) {
      return 1;
   }
   else {
      return 0;
   }
}

# ------------------------------------------------------------------------------
#
sub isDiskFull {
   my $self = shift;
   &sl_traces::trace( 3, "- sl_node::isDiskFull - Determining if node ".
                "< $self->{'Name'} > has disk" );
   if ( defined  $self->{'DiskLst'} ) {
      return 1;
   }
   else {
      return 0;
   }
}

# ------------------------------------------------------------------------------
#
sub getGroupName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getGroupName - Getting node < $self->{'Id'} > ".
                "GroupName" );
   return $self->{'GroupName'};
}

# ------------------------------------------------------------------------------
#
sub getType {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getType - Getting node < $self->{'Id'} > ".
                "type" );
   return $self->{'Type'};
}

# ------------------------------------------------------------------------------
#
sub getOs {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getOs - Getting node < $self->{'Id'} > ".
                "os" );
   return $self->{'Os'};
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getName - Getting node ".
                     "< $self->{'Name'} name" );
   return $self->{'Name'};
}

# ------------------------------------------------------------------------------
#
sub getArch {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getArch - Getting node < $self->{'Id'} > ".
                "arch" );
   return $self->{'Arch'};
}

# ------------------------------------------------------------------------------
#
sub getId {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getId - Getting node < $self->{'Id'} > ".
                "Id" );
   return $self->{'Id'};
}

# ------------------------------------------------------------------------------
#
sub getBoard {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getBoard - Getting node ".
                     "< $self->{'Name'} board" );
   return $self->{'Board'};
}

# ------------------------------------------------------------------------------
#
sub getDiskOidLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_node::getDiskOidLst - Getting node ".
                      "< $self->{'Name'} > DiskLst" );
   if ( defined @{$self->{'DiskLst'}} ) {
      return @{$self->{'DiskLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getDiskLst {
   my $self = shift;
   my ( @shelfLst ) = @_;
   &sl_traces::trace( 2, "- sl_node::getDiskLst - Getting node ".
                      "< $self->{'Name'} > Disk object list" );
   my @nodeDiskOidLst = $self->getDiskOidLst;
   my @nodeDiskLst = ();
   my @diskLst = ();
   # Get the dsk list from the shelves
   foreach $shelf ( @shelfLst ) {
      push( @diskLst, $shelf->getDiskLst );
   }
   # Build the disk list for the current node
   foreach $disk ( @diskLst ) {
	 foreach $diskOid ( @nodeDiskOidLst ) {
            if ( $diskOid eq $disk->getName ) {
	       push( @nodeDiskLst, $disk );
            }
	 } 
      }
   return sort @nodeDiskLst;
}

# ------------------------------------------------------------------------------
#
sub _getMntPoint {
   my $self = shift;
   my ( $role, $drawer ) = @_; 
   my $mntPoint = undef;
   &sl_traces::trace( 3, "- sl_node::_getMntPoint - Getting mount point for ".
                     "file system < ${role} > DiskLst" );
   my @diskOidLst = $self->getDiskOidLst;
   DISK: foreach $diskOid ( @diskOidLst ) {
      $disk = $drawer->getDisk( $diskOid );
      foreach $slice ( $disk->getSliceLst ) {
	 $fileSys = $slice->getFileSys;
	 if ( defined $fileSys ) {
	    $fileSysRole = $fileSys->getRole;
	    if ( $role eq $fileSysRole ) {
	       $mntPoint = $fileSys->getMntPoint;
	       last DISK;
	    }
	 }
      }
   }
   return $mntPoint;
}

# ------------------------------------------------------------------------------
#
sub deploy {
   my $class = shift;
   my ( $clusterDir, $swlVersion, $clusterName, @groupLst ) = @_;
   &sl_traces::trace( 2, "- sl_node::deploy - Finishing nodes ".
                      "configuration" );
   my $command = "";
   my $vendorType = "NLS_VENDOR=";
   my %vendorTab = ();
   my @menLst = ();
   my $menGroupDir = "";
   my @menNodeDirLst = ();
   foreach $group ( @groupLst ) {
      my @nodeLst = $group->getNodeLst;
      my $groupName = $group->getName;
      my $groupArch = $group->getArch;
      my $groupOs = $group->getOs;
      my $groupDir = "${clusterDir}/${groupName}";
      my $arch = $sl_board::ARCH2DIR{$groupArch};
      my $os = $sl_node::OS2DIR{$groupOs};
      if ( ( ! $group->isMasterEligible ) and ( ! $group->isDataless ) ) {
	 my $exportRoot = &sl_registry::getValue( $clusterName, "EXPORT_ROOT" );
	 my $groupPrefix = 
         "${exportRoot}/${swlVersion}/${arch}/${os}/${groupName}";
         # Terminate crfs configuration: add shared usr for diskless nodes group
         my $sharedDir = "share -F nfs -o rw,anon=0 ${groupPrefix}/usr";
         ${sl_rnfs::RNFS_SHARED_DIR_TAB}{$sharedDir} = $sharedDir;
      }
      elsif ( $group->isMasterEligible ) {
         $menGroupDir = $groupDir;
      }
      foreach $node ( @nodeLst ) {
         # Determine etc directory
	 my $nodeId = $node->getId;
         my $nodeDir = "${groupDir}/${nodeId}";
         my $etcDir = "${nodeDir}/etc";
         my $hostsFile = "${etcDir}/hosts";
	 # Add $nodeDir to the nodeDir list used further in this method
	 push( @menNodeDirLst, $nodeDir )
	    if ( $node->isMasterEligible );
         # Append common hosts file body to the node specific hosts file
         $command = "$ENV{'SMCT_CAT'} $ENV{'SMCT_TMP_DIR'}/hosts >> ".
	            "${hostsFile};";	
         &sl_utils::execCmd( $command );
         # Copy netmasks under each /etc node directory
	 $command = "$ENV{'SMCT_CP'} $ENV{'SMCT_TMP_DIR'}/netmasks ".
	            "${nodeDir}/etc";
         &sl_utils::execCmd( $command );
         # Copy defaultrouter under each /etc node directory
	 $command = "$ENV{'SMCT_CP'} $ENV{'SMCT_TMP_DIR'}/defaultrouter ".
	            "${nodeDir}/etc";
         &sl_utils::execCmd( $command );
	 # Copy cluster_nodes_table under each local node directory
         my $tableDir = "${nodeDir}/etc/opt/SUNWcgha";
         ( -d $tableDir ) || &sl_utils::createDir( $tableDir );
	 $command = "$ENV{'SMCT_CP'} ".
	    "$ENV{'SMCT_TMP_DIR'}/${sl_cmm::CMM_CLUSTER_NODES_TABLE} ".
	    "${tableDir};";
         &sl_utils::execCmd( $command );
         if ( $node->isMasterEligible ) {
	    push( @menLst, $node );
	    my $sharedRoot = 
	       &sl_registry::getValue( $clusterName, "SHARED_ROOT" );
	    my $dataRoot = 
	       &sl_registry::getValue( $clusterName, "DATA_ROOT" );
	    
	    # Move NLT to the appropriate directory
	    my $tmpNltConf = "$ENV{'SMCT_TMP_DIR'}/${sl_nls::NLT_CONF}";
	    if ( -f $tmpNltConf ) {
	       my $sharedRoot = 
                  &sl_registry::getValue( $clusterName, "SHARED_ROOT" );
	       my $dataRoot = 
	          &sl_registry::getValue( $clusterName, "DATA_ROOT" );
	       my $nltConf = "${nodeDir}${sharedRoot}/${dataRoot}".
	                     "${sl_nls::NLT_DIR}/${sl_nls::NLT_CONF}";
	       $command = "$ENV{'SMCT_CP'} ${tmpNltConf} ${nltConf};";
               &sl_utils::execCmd( $command );
	    }
	 }
         elsif ( !$node->isDataless ) {
	    # get vendor type information for NLS config. 
	    #my $tmpNltConf = "$ENV{'SMCT_TMP_DIR'}/${sl_nls::NLT_CONF}";
	    #if (  -f $tmpNltConf ) {
	       my $boardOid = $node->getBoard;
	       my $board = $node->getBoardObject;
	       my @vendorLst = $board->getVendorType;
	       foreach $vendor ( @vendorLst ) {
	          $vendorTab{$vendor} = $vendor;
	       }
	    #}
	 }
      }
   }
   # Append tmp crfs.share to MEN crfs.share
   $command = "";
   foreach $nodeDir ( @menNodeDirLst ) {
      my $crfsDir = "${nodeDir}${sl_fs::FS_CONF_DIR}";
      my $crfsShare = "${crfsDir}/${sl_fs::FS_CONF}";
      my $sharedNb = 0;
      foreach $sharedDir ( sort keys %{sl_rnfs::RNFS_SHARED_DIR_TAB} ) {
	 $sharedNb++;
	 $command .= "echo \"$sl_fs::FS_RNFS_PROPERTIES{'SHARE'}.".
	 "${sharedNb}=${sharedDir}\"".
	 ">> ${crfsShare};";
      }
   }
   &sl_utils::execCmd( $command );
   # Terminate the NLS configuration
   my $tmpNltConf = "$ENV{'SMCT_TMP_DIR'}/${sl_nls::NLT_CONF}";
   my $tmpNlsConf = "$ENV{'SMCT_TMP_DIR'}/${sl_nls::NLS_CONF}";
   if ( -f $tmpNltConf ) {
      foreach $vendor ( keys %vendorTab ) {
         $vendorType .= "${vendor} ";
      }
      # Suppress trailing spaces from vendorType string
      $vendorType =~ s/(\s)+$//;
      foreach $node ( @menLst ) {
         my $nodeId = $node->getId;
	 my $groupName = $node->getGroupName;
         my $groupDir = "${clusterDir}/${groupName}";
         my $nodeDir = "${groupDir}/${nodeId}";
	 my $nlsConfDir = "${nodeDir}${sl_nls::NLS_DIR}";
	 my $nlsConfFile = "${nlsConfDir}/${sl_nls::NLS_CONF}";      
	 $command = "echo \"${vendorType}\" >> ${nlsConfFile};";
         if ( -f $tmpNlsConf ) {
            # Move tmp nls config file to MEN config files
	    $command .= "$ENV{'SMCT_CAT'} ${tmpNlsConf} >> ${nlsConfFile};";
         }
	 &sl_utils::execCmd( $command );
      }
   }
   # Temporary -> waiting for C implementation of NLS
   &sl_nls::finalize( $menGroupDir, %vendorTab );
}

# ------------------------------------------------------------------------------
#
sub install {
   my $self = shift;
   my ( $groupDir ) = @_;
   my $nodeId = $self->getId;
   &sl_traces::info( 2, "Installing node < $self->{'Name'} >" );
   my $nodeDir = "${groupDir}/${nodeId}";
   &sl_utils::createDir( $nodeDir );
}

# ------------------------------------------------------------------------------
#
sub checkInstall {
   my $self = shift;
   my ( $hasSharedSoftware, $hasEmbeddedSoftware, @shelfLst ) = @_;
   my $nodeName = $self->getName;
   &sl_traces::trace( 2, "- sl_node::checkInstall - Checking < ${nodeName} >" );
   # Check node disk existence and set MEN and DLN mandatory files systems
   my %mandatoryFilesys = ();
   my %mandatoryFileSysLst = ();
   my @diskOidLst = $self->getDiskOidLst;
   if ( $self->isMasterEligible ) {
      &sl_traces::error( "The %s < %s > has no disk defined", 
			 "- sl_node::checkInstall - ".__LINE__,
			 ( "MEN", $nodeName ) )
         unless ( scalar  @diskOidLst > 0 );
      $mandatoryFilesys{$sl_filesys::FILESYS_ROLE{'ROOT'}} = 0;
      $mandatoryFilesys{$sl_filesys::FILESYS_ROLE{'SWAP'}} = 0;
      $mandatoryFilesys{$sl_filesys::FILESYS_ROLE{'SHARED'}} = 0;
      $mandatoryFilesys{$sl_filesys::FILESYS_ROLE{'DATA'}} = 0;
      $mandatoryFilesys{$sl_filesys::FILESYS_ROLE{'SERVICES'}} = 0
         if ( $hasSharedSoftware );
      $mandatoryFilesys{$sl_filesys::FILESYS_ROLE{'EXPORT'}} = 0
         if ( $hasEmbeddedSoftware );
   }
   elsif ( $self->isDataless ) {
      &sl_traces::error( "The %s < %s > has no disk defined", 
			 "- sl_node::checkInstall - ".__LINE__,
			 ( "DLN", $nodeName )  )
         unless ( scalar @diskOidLst  > 0 );
      $mandatoryFilesys{$sl_filesys::FILESYS_ROLE{'ROOT'}} = 0;
      $mandatoryFilesys{$sl_filesys::FILESYS_ROLE{'SWAP'}} = 0;      
   }
   else {
      # Check that diskless nodes have no disk
      &sl_traces::error( "The diskless node ".
                         "< %s > cannot have disk(s) defined",
			 "- sl_node::checkInstall - ".__LINE__,
			 ( $nodeName ) )
         if ( scalar @diskOidLst  > 0 );
   }
   # Look for the mandatory files systems within the node disks list
   my %fileSysMntPtLst = ();
   my @diskLst = $self->getDiskLst( @shelfLst );
   my $sliceSizeTot = 0;
   my %metaPhysSliceLst = ();
   my %softPhysSliceLst = ();
   my %sliceSizeLst = ();
   foreach $disk ( @diskLst ) {
      my $sliceSizeTot = 0;
      my $diskName = $disk->getName;
      my $diskSize = 1000 * $disk->getSize;
      my @sliceLst = $disk->getSliceLst;
      # Extract slice parameters to check
      my @repSliceLst = ();
      my @bitmapSliceLst = ();
      my @fileSysLst = ();
      my %fileSysSize = ();
      my %doubleFileSys = ();
      foreach $slice ( @sliceLst ) {
	 my $fileSys = $slice->getFileSys;
         my $sliceName = $slice->getName;
	 my $sliceSize = $slice->getSize;
	 # Capture Slice size for future checks
	 $sliceSizeLst{$sliceName} = $sliceSize;
	 if ( $slice->isMeta ) {
	    my $physSliceName = $slice->getPhysSlice;
	    $metaPhysSliceLst{$sliceName} = $physSliceName;
	 }
	 elsif ( $slice->isSoft ) {
	    my $physSliceName = $slice->getPhysSlice;
	    $softPhysSliceLst{$sliceName} = $physSliceName;
	 }
	 else {
	    $sliceSizeTot += $sliceSize;
	 }
	 # Check that metaDB slice has no fils susyem defined
	 if ( $slice->isMetaDB ) {
	    &sl_traces::error( "%s slice ".
	       "< %s > cannot have a filesystem defined".
	       "\nFile system = %s",
	       "- sl_node::checkInstall - ".__LINE__,
	       ( "MetaDB", $sliceName, $fileSys->getName ) )	 	 
               if ( defined $fileSys );
	 }
         # Process bitmap slices
	 if ( $slice->isBitmap ) {
	    &sl_traces::error( "%s slice ".
	       "< %s > cannot have a filesystem defined".
	       "\nFile system = %s",
	       "- sl_node::checkInstall - ".__LINE__,
	       ( "Bitmap", $sliceName, $fileSys->getName ) )	 
	       if ( defined $fileSys );
	    $managedSlice = $slice->getRepSlice;
	    push( @bitmapSliceLst, $managedSlice  );
	    # Check minimum size
	    &sl_traces::error( "Insufficient size for the bitmap slice ".
	       "< %s > of the disk < %s >.".
	       "\nCurrent size = %s".
	       "\nMinimum size = %s",
	        "- sl_node::checkInstall - ".__LINE__,
		( $sliceName, $diskName, 
		  $sliceSize, $sl_slice::SLICE_SIZE{'bitmap'} ) )
		unless ( $sliceSize >= $sl_slice::SLICE_SIZE{'bitmap'} );
	 }
	 elsif ( $slice->isReplicated ) {
	    push( @repSliceLst, $slice->getName );
	 }
	 # Process non-bitmap slices   
	 if ( ( defined $fileSys ) and
	      ( ( $slice->isPhys ) or
		( $slice->isMeta ) or
		( $slice->isSoft ) ) ) {
	    my $fileSysSize = $fileSys->getSize;
	    my $fileSysRole = $fileSys->getRole;
            my $fileSysName = $fileSys->getName;
	    my $fileSysMntPt = $fileSys->getMntPoint;
	    # Check that the filesys size is <= to the slice size
	    &sl_traces::error( "The size for the slice ".
	       "< %s > of the disk < %s >".
	       "\nis less than the file system < %s > size.".
	       "\nSlice size = %s".
	       "\nFile system size = %s",
	        "- sl_node::checkInstall - ".__LINE__,
		( $sliceName, $diskName, $fileSysName, 
		  $sliceSize, $fileSysSize ) )
	       unless ( $sliceSize >= $fileSysSize );
	    # Check that a mnt point is not used more than one time
            if ( $fileSysMntPt ne "-" ) {
	       if ( defined $fileSysMntPtLst{$fileSysMntPt} ) {
                  &sl_traces::error( "The mount point < %s > is ".
		     "already used by one file system of the node ".
		     "< %s >".
		     "\nPrevious file system: %s".
		     "\nCurrent file system : %s",
		     "- sl_node::checkInstall - ".__LINE__,
		     ( $fileSysMntPt, $nodeName, 
		       $fileSysMntPtLst{$fileSysMntPt}, $fileSysName ) );
	       }
	       else {
	          $fileSysMntPtLst{$fileSysMntPt} = $fileSysName;
	       }
	    }
	    push( @fileSysLst, $fileSys );
	    # Control that a file system is not referenced by more than one 
            # partition of the disk
            if ( defined $doubleFileSys{$fileSysName} ) {
	       &sl_traces::error( "The file system < %s > is ".
	          "referenced by more than 1 slice of the disk < %s >".
		  "\nSlice < %s >".
		  "\nSlice < %s >",
		  "- sl_node::checkInstall - ".__LINE__,
		  ( $fileSysName, $diskName, 
		    $sliceName, $doubleFileSys{$fileSysName} ) )
 	    }
	    else {
	       $doubleFileSys{$fileSysName} = $sliceName;
	       $fileSysSize{$fileSysRole} = $sliceSize;
	    }
	    # Take also the exported files systems
	    my @expFileSysLst = $fileSys->getExportedFileSysLst;
	    if ( scalar @expFileSysLst > 0 ) {
               # Check that NFS file systems are only defined for MEN
	       &sl_traces::error( "Only MEN can ".
	                          "have NFS file system definitions. ".
				  "Current node is < %s >",
			          "- sl_node::checkInstall - ".__LINE__,
				  ( $nodeName ) )
                   unless ( $self->isMasterEligible );
	       foreach $expFileSys ( @expFileSysLst ) {
	          # Check that the type of the exported file system is NFS
                  $expFileSystemName = $expFileSys->getName;
		  &sl_traces::error( "The exported ".
	             "file system < %s > of the node ".
		     "< %s > must have a type equal to NFS", 
		     "- sl_node::checkInstall - ".__LINE__,
		     ( $expFileSystemName, $nodeName ) )
		     unless ( $expFileSys->isNFS );
	          push( @fileSysLst, $expFileSys );
	       }
	    }
	 }
      }
      # Check that the sum of the slice size do not exceed the disk size
      &sl_traces::error( "Insufficient disk size".
         "\nDisk : %s".
	 "\nCurrent size = %s Mbyte".
	 "\nTotal slices size = %s Mbyte",
	  "- sl_node::checkInstall - ".__LINE__,
	  ( $diskName, $diskSize, $sliceSizeTot  ) )
	  unless ( $diskSize >= $sliceSizeTot );       
      # Check file system size (but bitmap slice size checked above)
      foreach $key ( keys %fileSysSize ) {
         if ( defined $sl_slice::SLICE_SIZE{$key} ) {
      	 &sl_traces::error( "Insufficient size for the slice associated ".
	    "to the < %s > file system of the disk < %s >.".
	    "\nCurrent size = %s".
	    "\nMinimum size = %s",
	     "- sl_node::checkInstall - ".__LINE__,
	     ( $key, $diskName, $fileSysSize{$key}, 
	       $sl_slice::SLICE_SIZE{$key} ) )
	     unless ( $fileSysSize{$key} >= $sl_slice::SLICE_SIZE{$key} );
	 }
      }
      # Control disk replicated and bitmap slices
      &sl_traces::error( "Replicated and bitmap slices ".
			 "of the node < %s > are not consistent:".
			 "\nReplicated slices: %s".
			 "\nMirrored slices  : %s",
			 "- sl_node::checkInstall - ".__LINE__,
			 ( $nodeName, join( " ", @repSliceLst ),
			   join( " ", @bitmapSliceLst ) ) )
         unless ( &sl_utils::compare( ( @repSliceLst, " ", @bitmapSliceLst ) ) );
      # Control existence and unicity of the mandatory files systems
      foreach $fileSys ( @fileSysLst ) {
         $role = $fileSys->getRole;
	 $fileSysName = $fileSys->getName;
         foreach $key ( keys %mandatoryFilesys ) {
	    if ( $key eq $role ) {
               $mandatoryFilesys{$key}++;
	       &sl_traces::error( "More than 1 file ".
	                	  "system has been defined with the role ".
				  "< %s > for the node < %s >".
				  "\nCurrent file system is < %s >",
				  "- sl_node::checkInstall - ".__LINE__,
				  ( $key, $nodeName, $fileSysName ) )
		  if ( $mandatoryFilesys{$key} > 1 );
               $mandatoryFileSysLst{$key} = $fileSys;
            }
	 }
      }
      foreach $key ( keys %mandatoryFilesys ) {
	 &sl_traces::error( "No file ".
	                    "system has been defined with the role ".
			    "< %s > for the node < %s >",
			    "- sl_node::checkInstall - ".__LINE__,
			    ( $key, $nodeName ) )
	    if ( $mandatoryFilesys{$key} == 0 );
      }
   }
   # Check that the DATA NFS file systems belongs to the 
   # 'shared' file system
   if ( defined $mandatoryFileSysLst{$sl_filesys::FILESYS_ROLE{'SHARED'}} ) {
      my $sharedFileSysName = 
         $mandatoryFileSysLst{$sl_filesys::FILESYS_ROLE{'SHARED'}}->getName;
      foreach $role ( $sl_filesys::FILESYS_ROLE{'DATA'}, 
                      $sl_filesys::FILESYS_ROLE{'SERVICES'} ) {
         my $hostFileSys = $mandatoryFileSysLst{$role}->getHostFileSys;
	 &sl_traces::error( "The ".
	    "file system < %s > of the node < %s > ".
	    "does not belong to a ".
	    "< %s > file system.".
	    "\nIt belongs to the file system < %s >", 
	    "- sl_node::checkInstall - ".__LINE__,
	    ( $role, $nodeName,
	      $sl_filesys::FILESYS_ROLE{'SHARED'}, $hostFileSys ) )
	       unless ( $hostFileSys eq $sharedFileSysName );
      }
   }
   # Check that meta-slice size does not exceed underlying physical slice size
   foreach $sliceName ( keys %metaPhysSliceLst ) {
      my $sliceSize = $sliceSizeLst{$sliceName};
      my $physSlice = $metaPhysSliceLst{$sliceName};
      my $physSliceSize = $sliceSizeLst{$physSlice};
      &sl_traces::error( "The size of the meta slice < %s > ".
	 "is greater than the underlying physical slice < %s >".
	 "\nPhysical slice size : %s".
	 "\nMeta slice size     : %s", 
	 "- sl_node::checkInstall - ".__LINE__,
	 ( $sliceName, $physSlice, $sliceSize, $physSliceSize ) )
	    unless ( ${sliceSize} <= ${physSliceSize} );   
   }
   # Check that sodt-slice size sum does not exceed underlying physical 
   # slice size
   my %sizeSliceTotLst = ();
   foreach $sliceName ( keys %softPhysSliceLst ) {
      my $sliceSize = $sliceSizeLst{$sliceName};
      my $physSlice = $softPhysSliceLst{$sliceName};
      my $physSliceSize = $sliceSizeLst{$physSlice};
      $sizeSliceTotLst{$physSlice} += $sliceSize;
      &sl_traces::error( "The size of the physical slice < %s > ".
	 "is less than the sum of the associated soft partitons size".
	 "\nPhysical slice size          : %s".
	 "\nCurrent soft slices size sum : %s".
	 "\nCurrent soft slice           : %s", 
	 "- sl_node::checkInstall - ".__LINE__,
	 ( $physSlice, $physSliceSize, 
	   $sizeSliceTotLst{$physSlice}, $sliceName ) )      
         if ( $sizeSliceTotLst{$physSlice} > $physSliceSize );
   }
}

# ------------------------------------------------------------------------------
#
sub isHidden {
   my $class = shift;
   my $enumValue = shift;
   &sl_traces::trace( 2, "- sl_node::isHidden - Testing ". 
                      "< ${enumValue} > hidden value" );
   return $NODE_HIDDEN_VALUES{$enumValue};
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   my $nodeId = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_node::iD - Node iD is < ${nodeId} >" );
   return $nodeId;
}

# ------------------------------------------------------------------------------

return 1;
