#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_group.pm 1.12     02/10/31 SMI"
# ------------------------------------------------------------------------------

BEGIN { 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_package.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_service.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_filesys.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_node.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_utils.pl"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_jumpstart.pl"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
}

# ------------------------------------------------------------------------------

package sl_group;

# ID fields separator
my $ID_SEP = '@';

my $GROUP_IDENT = "ident.txt";    # group ident file

# Software installation scripts
my $PKG_SCRIPT = "pkginstall.sh";
my $PRE_PAT_SCRIPT = "patchinstall.pre.sh";
my $POST_PAT_SCRIPT = "patchinstall.post.sh";
my $SOFT_SCRIPT = "softinstall.sh";
my $SOFT_INSTALL_LAUNCHER = "softinstall.launcher.sh";

# Header used to trigger debug mode in install-scripts
my $SCRIPT_DEBUG_HEADER = "if [ \$SMCT_DEBUG ]\n".
                          "then\n".
			  "   REDIRECT=\"\"\n".
			  "else\n".
                          "   REDIRECT=\" > /dev/null 2>&1\"\n".
			  "fi\n\n".
			  "trace() {\n".
                	  "   DATE=`date '+INFO: - %m/%d/%y - %H:%M:%S -'`\n".
                	  "   echo \$DATE\" \"\$MSG\n".
                	  "}\n\n".
			  "error() {\n".
                	  "   DATE=`date '+ERROR: - %m/%d/%y - %H:%M:%S -'`\n".
                	  "   echo \$DATE\" \"\$MSG\n".
                	  "}\n\n";
		   
# Define mount points used by jumpstart finish install script
my $ROOT = "/a";                   # Root directory at install time
my $EXPORT_TAG = "<-EXPORT->";     # SL export directory mount point
my $SOFTREPO_TAG = "<-SOFTREPO->"; # User software repo. mount point
my $SOLARIS_TAG = "<-SOLARIS->";   # Solaris packages repo. mount point
my $SOLARIS_EA_TAG = "<-SOLARIS_EA->"; # Solaris EA packages repo. mount point
my $WORKING_TAG = "\\\$WORKING_DIR";

my $NODE_TEMPLATE = "template";

# group object attributes
my @GROUP_ATTR = qw( Id
		     Os
		     Arch
		     Type
                     Name
		     Class
		     ServiceLst
		     HostGroup
		     EmbeddedGroupLst
		     PkgLst
		     PatLst
		     SoftLst
		     ConfigLst
		     NodeLst );

%GROUP_DEF_COMP_MASK = ( 'Id' => 0,
			 'Os' => 1,
			 'Arch' => 1,
			 'Type' => 1,
                	 'Name' => 1,
			 'Class' => 1,
			 'ServiceLst' => 1,
			 'HostGroup' => 1,
			 'EmbeddedGroupLst' => 1,
			 'PkgLst' => 0,
			 'PatLst' => 0,
			 'SoftLst' => 0,
			 'ConfigLst' => 0,
			 'NodeLst' => 1 );

# ------------------------------------------------------------------------------
#
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} )  && 
        ( defined $attr{'Name'} ) && 
	( defined $attr{'Os'} ) && 
	( defined $attr{'Arch'} ) &&
	( defined $attr{'Type'} ) && 
	( defined $attr{'NodeLst'} ) &&
	( defined $attr{'ServiceLst'} ) ) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      $self->{'Os'} = $attr{'Os'}; 
      $self->{'Arch'} = $attr{'Arch'}; 
      $self->{'Type'} = $attr{'Type'}; 
      $self->{'NodeLst'} = $attr{'NodeLst'}; 
      $self->{'ServiceLst'} = $attr{'ServiceLst'}; 
      # Set optional attributes      
      $self->{'PkgLst'} = $attr{'PkgLst'} 
         if ( defined $attr{'PkgLst'} );
      $self->{'PatLst'} = $attr{'PatLst'} 
         if ( defined $attr{'PatLst'} );         
      $self->{'SoftLst'} = $attr{'SoftLst'} 
         if ( defined $attr{'SoftLst'} );         
      $self->{'HostGroup'} = $attr{'HostGroup'} 
         if ( defined $attr{'HostGroup'} );
      $self->{'EmbeddedGroupLst'} = $attr{'EmbeddedGroupLst'} 
         if ( defined $attr{'EmbeddedGroupLst'} );
      $self->{'Class'} = $attr{'Class'} 
         if ( defined $attr{'Class'} );
      $self->{'ConfigLst'} = $attr{'ConfigLst'} 
         if ( defined $attr{'ConfigLst'} );
   }
   elsif ( defined $attr{'GroupObject'} ) {
      $self->_construct( $attr{'GroupObject'} );
   }
   else {
      &sl_traces::error( "Unable to create %s object. ".
                         "Invalid creation method", 
		         "- sl_group::new - ".__LINE__,
			 ( "sl_group" ) );
   }
   &sl_traces::trace( 2, "- sl_group::new - Creating group object ".
                "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $groupObject ) = @_;
   &sl_traces::trace( 3, "- sl_group::_construct - Building group object" );
   my $group = sl_iterator->new( 'Object' => $groupObject );
   $self->{'Id'} = $group->extract;
   $self->{'Os'} = $group->extract;
   $self->{'Arch'} = $group->extract;;
   $self->{'Type'} = $group->extract;
   $self->{'Name'} = $group->extract;
   $self->{'Class'} = $group->extract;
   @{$self->{'ServiceLst'}} = 
      $group->extractObjectArray( "sl_service", "ServiceObject" );
   $self->{'HostGroup'} = $group->extract;
   @{$self->{'EmbeddedGroupLst'}} = $group->extractScalarArray;
   @{$self->{'PkgLst'}} = $group->extractScalarArray;
   # Restore package objects
   @pkgLst = ();
   foreach  ( @{$self->{'PkgLst'}} ) {
      $pkg = sl_package->new( 'Id' => $_ );
      $pkgInst = $main::SOFTREP->lookUp( $pkg );
      if ( defined $pkgInst ) {
	 push( @pkgLst, $pkgInst );
      }
      else {
	 &sl_traces::error( "Cannot find %s ".
	                    "< %s > in the software repository", 
			    "- sl_group:_construct - ".__LINE__,
			    ( "package", $_ ) );
      }

   }
   # Replace package-id list by package object list
   @{$self->{'PkgLst'}} = @pkgLst;
   @{$self->{'PatLst'}} = $group->extractScalarArray;
   # Restore patch objects
   @patLst = ();
   foreach  ( @{$self->{'PatLst'}} ) {
      $pat = sl_patch->new( 'Id' => $_ );
      $patInst = $main::SOFTREP->lookUp( $pat );
      if ( defined $patInst ) {
	 push( @patLst, $patInst );
      }
      else {
	 &sl_traces::error( "Cannot find %s ".
	                    "< %s > in the software repository", 
			    "- sl_group:_construct - ".__LINE__,
			    ( "patch", $_ ) );
      }
   }
   # Replace patch-id list by patch object list
   @{$self->{'PatLst'}} = @patLst;
   @{$self->{'SoftLst'}} = $group->extractScalarArray;
   # Restore software objects
   @softLst = ();
   foreach  ( @{$self->{'SoftLst'}} ) {
      $soft = sl_software->new( 'Id' => $_ );
      $softInst = $main::SOFTREP->lookUp( $soft );
      if ( defined $softInst ) {
	 push( @softLst, $softInst );
      }
      else {
         $soft->display;
	 &sl_traces::error( "Cannot find %s ".
	                    "< %s > in the software repository", 
			    "- sl_group:_construct - ".__LINE__,
			    ( "software", $_ )  );
      }
   }
   # Replace software-id list by software object list
   @{$self->{'SoftLst'}} = @softLst;
   @{$self->{'ConfigLst'}} =
      $group->extractObjectArray( "sl_config", "ConfigObject" );
   @{$self->{'NodeLst'}} =
      $group->extractObjectArray( "sl_node", "NodeObject" );
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::serialize - Serializing group object" );
   my $groupObject = sl_iterator->new( 'Object' => "" );
   foreach $attr ( @GROUP_ATTR ) {
      if ( $attr eq 'PatLst' ) {
         # build patch id list
	 my @patchIdLst = ();
	 foreach $pat ( @{$self->{'PatLst'}} ) {
	    push( @patchIdLst, $pat->iD );
	 }
	 $groupObject->serializeScalarArray( @patchIdLst );
      }
      elsif ( $attr eq 'PkgLst' ) {
         # build package id list
	 my @pkgIdLst = ();
	 foreach $pkg ( @{$self->{'PkgLst'}} ) {
	    push( @pkgIdLst, $pkg->iD );
	 }
	 $groupObject->serializeScalarArray( @pkgIdLst );
      }	 
      elsif ( $attr eq 'SoftLst' ) {
         # build software id list
	 my @softIdLst = ();
	 foreach $soft ( @{$self->{'SoftLst'}} ) {
	    push( @softIdLst, $soft->iD );
	 }
	 $groupObject->serializeScalarArray( @softIdLst );
      }	 
      elsif ( $attr eq 'EmbeddedGroupLst' ) {
	 $groupObject->serializeScalarArray(  @{$self->{'EmbeddedGroupLst'}} );
      }
      elsif ( $attr eq 'ServiceLst' ) {
         $groupObject->serializeObjectArray( @{$self->{'ServiceLst'}} );
      }
      elsif ( $attr eq 'ConfigLst' ) {
         $groupObject->serializeObjectArray( @{$self->{'ConfigLst'}} );
      }	 
      elsif ( $attr eq 'NodeLst' ) {
         $groupObject->serializeObjectArray( @{$self->{'NodeLst'}} );
      }	 
      else {
	 $groupObject->serialize( $self->{$attr} );
      }
   }
   return $groupObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_group::display - Displaying group" );
   print STDOUT "\nGroup\n";
   print STDOUT "-----\n";
   foreach $attr ( @GROUP_ATTR ) {
      if ( defined $self->{$attr} ) {
         if ( $attr eq 'PkgLst' ) {
	    print STDOUT "$attr = ";
	    foreach $pkg ( @{$self->{'PkgLst'}} ) {
	       print STDOUT $pkg->iD." ";
	    }
	    print STDOUT "\n";
	 }
	 elsif ( $attr eq 'PatLst' ) {
	    print STDOUT "$attr = ";
	    foreach $pat ( @{$self->{'PatLst'}} ) {
	       print STDOUT $pat->iD." ";
	    }
	    print STDOUT "\n";
	 }
	 elsif ( $attr eq 'SoftLst' ) {
	    print STDOUT "$attr = ";
	    foreach $soft ( @{$self->{'SoftLst'}} ) {
	       print STDOUT $soft->iD." ";
	    }
	    print STDOUT "\n";
	 }
	 elsif ( $attr eq 'EmbeddedGroupLst' ) {
	    print STDOUT "$attr = ";
	    foreach $groupOid ( @{$self->{'EmbeddedGroupLst'}} ) {
	       print STDOUT $groupOid." ";
	    }
	    print STDOUT "\n";
	 }
	 elsif ( $attr eq 'ConfigLst' ) {
	    print STDOUT "$attr =\n";
	    foreach $config ( @{$self->{'ConfigLst'}} ) {
	       $config->display;
	    }
	 }
	 elsif ( $attr eq 'NodeLst' ) {
	    print STDOUT "$attr =\n";
	    foreach $node ( @{$self->{'NodeLst'}} ) {
	       $node->display;
	    }
	 }
	 elsif ( $attr eq 'ServiceLst' ) {
	    print STDOUT "$attr =\n\n";
	    foreach $service ( @{$self->{'ServiceLst'}} ) {
	       $service->display;
	    }
	 }
	 else {
	    print STDOUT "$attr = ";
	    print STDOUT "$self->{$attr}";
	    print STDOUT "\n";
	 }
      }
      else {
         print STDOUT "$attr =\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub compare {
   my $class = shift;
   my ( $group1, $group2 ) = @_;
   my $group1Name = ( defined $group1 ) ? $group1->getName:"<undefined>";
   my $group2Name = ( defined $group2 ) ? $group2->getName:"<undefined>";
   &sl_traces::trace( 2, "- sl_group::compare - Comparing nodes group ".
                     "< ${group1Name} > and < ${group2Name} >" );
   # String holding differences if any
   my $diffStr = undef;
   # Extract the differences if any
   foreach $attr ( keys %GROUP_DEF_COMP_MASK ) {
      my $diffStr1 = "Comparing nodesGroup  ${group1Name}.";
      my $diffStr2 = "       to nodesGroup  ${group2Name}.";
      if ( $GROUP_DEF_COMP_MASK{$attr} ) {
         if ( &sl_utils::testAttribute( $group1->{$attr} ) and 
	      &sl_utils::testAttribute( $group2->{$attr} ) ) {
            if ( $attr eq 'NodeLst' ) {
	       # Sort lists before comparison
	       $group1->setNodeLst( 
	          &sl_utils::sortListByName( $group1->getNodeLst ) );
	       $group2->setNodeLst( 
		  &sl_utils::sortListByName( $group2->getNodeLst ) );
               # Look for differences
	       @group1NodeLst = $group1->getNodeLst;
	       @group2NodeLst = $group2->getNodeLst;
	       for ($i=0; $i < scalar @group1NodeLst; $i++) {
		  $diff = sl_node->compare(
		             $group1NodeLst[$i], $group2NodeLst[$i] );
		  $diffStr .= $diffStr1."NodeLst[$i]\n".
		              $diffStr2."NodeLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
            elsif ( $attr eq 'EmbeddedGroupLst' ) {
               # Look for differences
	       @group1GroupLst = sort $group1->getEmbeddedGroupLst;
	       @group2GroupLst = sort $group2->getEmbeddedGroupLst;
	       for ($i=0; $i < scalar @group1GroupLst; $i++) {
		  if ( $group1GroupLst[$i] ne $group2GroupLst[$i] ) {
		     $diffStr .= 
		        $diffStr1.$attr."[$i]=".$group1GroupLst[$i]."\n".
		        $diffStr2.$attr."[$i]=".$group2GroupLst[$i]."\n";
	          }
	       }
	    }
            elsif ( $attr eq 'ServiceLst' ) {
	       # Sort lists before comparison
	       $group1->setServiceLst( 
	          &sl_utils::sortListByName( $group1->getServiceLst ) );
	       $group2->setServiceLst( 
	          &sl_utils::sortListByName( $group2->getServiceLst ) );
               # Look for differences
	       @group1ServiceLst = $group1->getServiceLst;
	       @group2ServiceLst = $group2->getServiceLst;
	       for ($i=0; $i < scalar @group1ServiceLst; $i++) {
		  $diff = sl_service->compare( 
		           $group1ServiceLst[$i], $group2ServiceLst[$i] );
		  $diffStr .= $diffStr1."ServiceLst[$i]\n".
		              $diffStr2."ServiceLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
            elsif ( $attr eq 'PkgLst' ) {
	       # Sort lists before comparison
	       $group1->setPkgLst( 
	          &sl_utils::sortListByName( $group1->getPkgLst ) );
	       $group2->setPkgLst( 
		  &sl_utils::sortListByName( $group2->setPkgLst ) );
               # Look for differences
	       @group1PkgLst = $group1->getPkgLst;
	       @group2PkgLst = $group2->getPkgLst;
	       for ($i=0; $i < scalar @group1PkgLst; $i++) {
		  $diff = sl_package->compare( 
		           $group1PkgLst[$i], $group2PkgLst[$i] );
		  $diffStr .= $diffStr1."PkgLst[$i]\n".
		              $diffStr2."PkgLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
            elsif ( $attr eq 'PatLst' ) {
	       # Sort lists before comparison
	       $group1->setPatLst( 
	          &sl_utils::sortListByName( $group1->getPatLst ) );
	       $group2->setPatLst( 
	          &sl_utils::sortListByName( $group2->getPatLst ) );
               # Look for differences
	       @group1PatLst = $group1->getPatLst;
	       @group2PatLst = $group2->getPatLst;
	       for ($i=0; $i < scalar @group1PatLst; $i++) {
		  $diff = sl_patch->compare( 
		           $group1PatLst[$i], $group2PatLst[$i] );
		  $diffStr .= $diffStr1."PatLst[$i]\n".
		              $diffStr2."PatLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
            elsif ( $attr eq 'SoftLst' ) {
	       # Sort lists before comparison
	       $group1->setSoftLst( 
	          &sl_utils::sortListByName( $group1->getSoftLst ) );
	       $group2->setSoftLst( 
	          &sl_utils::sortListByName( $group2->getSoftLst ) );
               # Look for differences
	       @group1SoftLst = $group1->getSoftLst;
	       @group2SoftLst = $group2->getSoftLst;
	       for ($i=0; $i < scalar @group1SoftLst; $i++) {
		  $diff = sl_software->compare( 
		           $group1SoftLst[$i], $group2SoftLst[$i] );
		  $diffStr .= $diffStr1."SoftLst[$i]\n".
		              $diffStr2."SoftLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
	    else {
	       if ( $group1->{$attr} ne $group2->{$attr} ) {
        	  $diffStr1 .= "${attr}=".$group1->{$attr}."\n";
		  $diffStr2 .= "${attr}=".$group2->{$attr}."\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }
            }
	 }
         elsif ( ! &sl_utils::testAttribute( $group1->{$attr} ) and
	           &sl_utils::testAttribute( $group2->{$attr} ) )  {
            if ( ( $attr eq 'Id' ) or
		 ( $attr eq 'Os' ) or
		 ( $attr eq 'Arch' ) or
		 ( $attr eq 'Type' ) or
                 ( $attr eq 'Name' ) or
		 ( $attr eq 'Class' ) or
		 ( $attr eq 'HostGroup' ) ) {
	       $diffStr1 .= "${attr}= <undefined>\n";
	       $diffStr2 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
	    }
	    else {
	       if ( scalar @{$group2->{$attr}} > 0 ) {
		  $diffStr1 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
		}       
	    }
         }
         elsif ( ! &sl_utils::testAttribute( $group2->{$attr} ) and
	           &sl_utils::testAttribute( $group1->{$attr} ) ) {
            if ( ( $attr eq 'Id' ) or
		 ( $attr eq 'Os' ) or
		 ( $attr eq 'Arch' ) or
		 ( $attr eq 'Type' ) or
                 ( $attr eq 'Name' ) or
		 ( $attr eq 'Class' ) or
		 ( $attr eq 'HostGroup' ) ) {
               $diffStr2 .= "${attr}= <undefined>\n";
	       $diffStr1 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
 	    }
	    else {
	       if ( scalar @{$group1->{$attr}} > 0 ) {
		  $diffStr2 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
		}       
	    }
	 }
      }
   }
   return $diffStr;
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getName - Getting group name" );
   return $self->{'Name'};
}

# ------------------------------------------------------------------------------
#
sub getArch {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getArch - Getting group arch" );
   return $self->{'Arch'};
}

# ------------------------------------------------------------------------------
#
sub getOs {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getOs - Getting group os" );
   return $self->{'Os'};
}

# ------------------------------------------------------------------------------
#
sub getType {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getType - Getting group type" );
   return $self->{'Type'};
}

# ------------------------------------------------------------------------------
#
sub getHostGroup {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getHostGroup - Getting group host" );
   return $self->{'HostGroup'};
}

# ------------------------------------------------------------------------------
#
sub setHostGroup {
   my $self = shift;
   my ( $hostGroup ) = @_;
   &sl_traces::trace( 2, "- sl_group::setHostGroup - Setting group host" );
   $self->{'HostGroup'} = $hostGroup;
}

# ------------------------------------------------------------------------------
#
sub getClass {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getClass - Getting group Class" );
   return $self->{'Class'};
}

# ------------------------------------------------------------------------------
#
sub setClass {
   my $self = shift;
   my ( $class ) = @_;
   &sl_traces::trace( 2, "- sl_group::setClass - Setting group class" );
   $self->{'Class'} = $class;
}

# ------------------------------------------------------------------------------
#
sub setEmbeddedGroupObjectLst {
   my $self = shift;
   my ( @embeddedGroupObjectLst ) = @_;
   &sl_traces::trace( 2, "- sl_group::setEmbeddedGroupObjectLst - Setting ".
                      "< $self->{'Name'} > embedded group object list" );
   @{$self->{'EmbeddedGroupObjectLst'}} =  @embeddedGroupObjectLst;
}

# ------------------------------------------------------------------------------
#
sub getEmbeddedGroupObjectLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getEmbeddedGroupObjectLst - Getting ".
                      "< $self->{'Name'} > embedded group object list" );
   if ( defined @{$self->{'EmbeddedGroupObjectLst'}} ) {
      return @{$self->{'EmbeddedGroupObjectLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub setNodeLst {
   my $self = shift;
   my @nodeLst = @_;
   if ( defined $nodeLst[0] ) {
      &sl_traces::trace( 2, "- sl_group::setNodeLst - Setting node(s) to the ".
                         "group < $self->{'Name'} >" );
      @{$self->{'NodeLst'}} = @nodeLst;
   }
   else {
      &sl_traces::error( "%s is undefined", 
                         "sl_group::setNodeLst - ".__LINE__,
			 ( "Node(s) list" ) );
   }
}

# ------------------------------------------------------------------------------
#
sub setConfigLst {
   my $self = shift;
   my ( $reset, @configLst ) = @_;
   if ( defined $configLst[0] ) {
      &sl_traces::trace( 2, "- sl_group::setConfigLst - Setting config to the ".
                         "group < $self->{'Name'} >" );
      if ( defined $reset ) {
         @{$self->{'ConfigLst'}} = @configLst;
      }
      else {
         push( @{$self->{'ConfigLst'}}, @configLst );
      }
   }
   else {
      &sl_traces::error( "%s is undefined", 
                         "sl_group::setConfigLst - ".__LINE__,
			 ( "Config list" ) );
   }
}

# ------------------------------------------------------------------------------
#
sub setServiceLst {
   my $self = shift;
   my ( @serviceLst ) = @_;
   &sl_traces::trace( 2, "- sl_group::setServiceLst - Setting ".
                      "< $self->{'Name'} > service list" );
   if ( defined $serviceLst[0] ) {
      @{$self->{'ServiceLst'}} = @serviceLst;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_group:setServiceLst - ".__LINE__,
			 ( "Service list" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub setPkgLst {
   my $self = shift;
   my @pkgLst = @_;
   my $found = 0;
   if ( defined $pkgLst[0] ) {
      &sl_traces::trace( 2, "- sl_group::setPkgLst - Set package(s) to group ".
                         "< $self->{'Name'} >" );
      if ( defined @{$self->{'PkgLst'}} ) {
         # Look if the packages already exists in the node list
	 foreach $pkg ( @pkgLst ) {
	    $pkgId = $pkg->iD;
	    $found = 0;
	    foreach $pkgCur ( @{$self->{'PkgLst'}} ) {
               if ( $pkgId eq $pkgCur->iD ) {
		  $pkgCur = $pkg;
		  $found = 1;
		  last;
	       }
	    }
	    if ( ! $found ) { push( @{$self->{'PkgLst'}}, $pkg ); }
	 }
      }
      else {
         @{$self->{'PkgLst'}} = @pkgLst;
      }
   }
   else {
      &sl_traces::error( "%s is undefined", 
                         "sl_group::setPkgLst - ".__LINE__,
			 ( "Package(s) list" ) );
   }   
}

# ------------------------------------------------------------------------------
#
sub setSoftLst {
   my $self = shift;
   my @softLst = @_;
   my $found = 0;
   if ( defined $softLst[0] ) {
      &sl_traces::trace( 2, "- sl_group::setSoftLst - Set software to group ".
                         "< $self->{'Name'} >" );
      if ( defined @{$self->{'SoftLst'}} ) {
         # Look if the software already exists in the list
	 foreach $soft ( @softLst ) {
	    $softId = $soft->iD;
	    $found = 0;
	    foreach $softCur ( @{$self->{'SoftLst'}} ) {
               if ( $softId eq $softCur->iD ) {
	          #&sl_traces::warning( 1, "sl_group::setSoftLst - Software ".
		  #   "< $pkgId > is already defined for the group ".
		  #   "< $self->{'Name'} >\nInstance parameters ".
		  #   "will be overloaded" );
                  # Replace old patch by the new one
		  $softCur = $soft;
		  $found = 1;
		  last;
	       }
	    }
	    if ( ! $found ) { push( @{$self->{'SoftLst'}}, $soft); }
	 }
      }
      else {
         @{$self->{'SoftLst'}} = @softLst;
      }
   }
   else {
      &sl_traces::error( "%s is undefined", 
                         "sl_group::setSoftLst - ".__LINE__,
			 ( "Software list" ) );
   }
}

# ------------------------------------------------------------------------------
#
sub setPatLst {
   my $self = shift;
   my @patLst = @_;
   my $found = 0;
   if ( defined $patLst[0] ) {
      &sl_traces::trace( 2, "- sl_group::setPatLst - Set Patch(es) to group ".
                         "< $self->{'Name'} >" );
      if ( defined @{$self->{'PatLst'}} ) {
         # Look if the patch already exists in the node list
	 foreach $pat ( @patLst ) {
	    $found = 0;
	    $patId = $pat->iD;
	    foreach $patCur ( @{$self->{'PatLst'}} ) {
               if ( $patId eq $patCur->iD ) {
	          #&sl_traces::warning( 1, "sl_group::setPatLst - Patch ".
		  #   "< $patId > is already defined for the group ".
		  #   "< $self->{'Name'} >\nInstance parameters ".
		  #   "will be overloaded" );
                  # Replace old patch by the new one
		  $patCur = $pat;
		  $found = 1;
		  last;
	       }
	    }
	    if ( ! $found ) { push( @{$self->{'PatLst'}}, $pat ); }
	 }
      }
      else {
         @{$self->{'PatLst'}} = @patLst;
      }
   }
   else {
      &sl_traces::error( "%s is undefined", 
                         "sl_group::setPatLst - ".__LINE__,
			 ( "Patch(es) list" ) );
   }
}

# ------------------------------------------------------------------------------
#
sub getPkgLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getPkgLst - Getting group ".
                "< $self->{'Name'} > packages list" );
   if ( defined $self->{'PkgLst'} ) {
      return @{$self->{'PkgLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getPatLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getPatLst - Getting group ".
                "< $self->{'Name'} > patches list" );
   if ( defined $self->{'PatLst'} ) {
      return @{$self->{'PatLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getSoftLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getSoftLst - Getting group ".
                      "< $self->{'Name'} > software list" );
   if ( defined $self->{'SoftLst'} ) {
      return @{$self->{'SoftLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getNodeLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getNodeLst - Getting group ".
                "< $self->{'Name'} > NodeLst" );
   if ( defined $self->{'NodeLst'} ) {
      return @{$self->{'NodeLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getServiceLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getServiceLst - Getting group ".
                "< $self->{'Name'} > ServiceLst" );
   if ( defined $self->{'ServiceLst'} ) {
      return @{$self->{'ServiceLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getConfigLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getConfigLst - Getting group ".
                "< $self->{'Name'} > ConfigLst" );
   if ( defined $self->{'ConfigLst'} ) {
      return @{$self->{'ConfigLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getEmbeddedGroupLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_group::getEmbeddedGroupLst - Getting group ".
                "< $self->{'Name'} > EmbeddedGroupLst" );
   if ( defined $self->{'EmbeddedGroupLst'} ) {
      return @{$self->{'EmbeddedGroupLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub isMasterEligible {
   my $self = shift;
   &sl_traces::trace( 3, "- sl_group::isMasterEligible - Determining group Type for ".
                "< $self->{'Name'} >" );
   if ( $self->{'Type'} eq ${sl_node::NODE_TYPE{'MASTER_ELIGIBLE'}} ) {
      return 1;
   }
   else {
      return 0;
   }
}

# ------------------------------------------------------------------------------
#
sub isDataless {
   my $self = shift;
   &sl_traces::trace( 3, "- sl_group::isDataless - Determining group Type for ".
                     "< $self->{'Name'} >" );
   if ( $self->{'Type'} eq ${sl_node::NODE_TYPE{'DATALESS'}} ) {
      return 1;
   }
   else {
      return 0;
   }
}

# ------------------------------------------------------------------------------
#
sub _hasSharedSoftware {
   my $self = shift;
   my $groupName = $self->getName;
   my $status = 0;
   my $pkgName = undef;
   my $patName = undef;
   &sl_traces::trace( 2, "- sl_group::_hasSharedSoftware - Looking at ".
                      "< ${groupName} > software" );
   # Check node group packages
   foreach $pkg ( $self->getPkgLst ) {
      if ( $pkg->isShared ) {
         $status = 1;
	 $pkgName = $pkg->getAbbr;
	 last;
      }
   }
   # Control that only MEN group have shared software
   if ( $status ) {
      &sl_traces::error( "Dataless nodes ".
                	 "group < %s > cannot have shared software ".
			 "< %s >", 
			 "- sl_group::_hasSharedSoftware - ".__LINE__,
			 ( $groupName ,$pkgName ) ) 
	 if ( $self->isDataless );
   }
   # Check node group patches
   foreach $pat ( $self->getPatLst ) {
      if ( $pat->isShared ) {
         $status = 1;
	 $patName = $pat->getName;
	 last;
      }
   }
   # Control that only MEN group have shared software
   if ( $status ) {
      &sl_traces::error( "Dataless nodes ".
                	 "group < %s > cannot have shared software ".
			 "< %s >", 
			 "- sl_group::_hasSharedSoftware - ".__LINE__,
			 ( $groupName ,$patName ) ) 
	 if ( $self->isDataless );
   }
   return $status;
}

# ------------------------------------------------------------------------------
#
sub _hasEmbeddedSoftware {
   my $self = shift;
   my $groupName = $self->getName;
   my $status = 0;
   &sl_traces::trace( 2, "- sl_group::_hasEmbeddedSoftware - Looking at ".
                      "< ${groupName} > embedded group" );
   my @embeddedGroupLst = $self->getEmbeddedGroupLst;
   if ( scalar @embeddedGroupLst > 0 ) {
      # check that the group can embed another group
      &sl_traces::error( "The NME nodes group ".
                         "< %s > cannot embed another nodes group", 
			 "- sl_group:_hasEmbeddedSoftware - ".__LINE__,
			 ( $groupName ) )
         unless ( $self->isMasterEligible );
      $status = 1;
   }
   else {
      if ( defined $self->getHostGroup ) {
	 # Check that the embedded group is neither ME nor dataless
	 &sl_traces::error( "Embedded group ".
                            "< %s > can be neither ME nor dataless", 
			    "- sl_group:_hasEmbeddedSoftware - ".__LINE__,
			    ( $groupName ) )
            if ( ( $self->isMasterEligible ) or (  $self->isDataless ) );
      }
      else {
         # Check that a NME diskless group is refrenced by a ME group
	 &sl_traces::error( "NME diskless nodes group < %s > ".
	                    "must be referenced by a ME nodes group", 
			    "- sl_group:_hasEmbeddedSoftware - ".__LINE__,
			    ( $groupName ) )
            unless ( ( $self->isMasterEligible ) or (  $self->isDataless ) );
      }	 
   }
   if ( $self->isMasterEligible ) {
      # check that the group has exactly 2 nodes if it is ME
      &sl_traces::error( "The ME nodes group ".
                         "< %s > must have 2 nodes exactly", 
			 "- sl_group:_hasEmbeddedSoftware - ".__LINE__,
			 ( $groupName )  )
         unless ( scalar @{$self->{'NodeLst'}} == 2 );
   }
   return $status;
}

# ------------------------------------------------------------------------------
#
sub _checkServices {
   my $self = shift;
   my $groupName = $self->getName;
   &sl_traces::trace( 2, "- sl_group::_checkServices - Checking group ".
                      "< ${groupName} > services" );
   my @serviceLst = $self->getServiceLst;
   foreach $serviceFamilyValue ( keys %sl_service::SERVICE_FAMILY_VALUE ) {
      my $serviceCount = 0;
      my $servicesStr = "";
      foreach $service ( @serviceLst ) {
         my $serviceFamily = $service->getFamily;
	 if ( $serviceFamily eq $serviceFamilyValue ) {
	   $serviceCount++;
	   $servicesStr .= " ".$service->getName;
	 }
      }
      &sl_traces::error( "Only one %s service can be ".
                         "configured for the nodes group < %s >".
			 "\nConfigured %s services ".
			 "%s",
			 "- sl_group:_checkServices -".__LINE__,
			 ( $serviceFamilyValue, $groupName, 
			   $serviceFamilyValue, $servicesStr )  )
	 unless ( $serviceCount <= 1 );
   }
}

# ------------------------------------------------------------------------------
#
sub _checkInstall {
   my $self = shift;
   my ( @shelfLst ) = @_;
   my $groupName = $self->getName;
   my @nodeLst = $self->getNodeLst;
   &sl_traces::trace( 2, "- sl_group::_checkInstall - Checking group ".
                      "< ${groupName} > installation parameters" );
   # Look if shared software must be installed for the group
   my $hasSharedSoftware = $self->_hasSharedSoftware;
   # Look if the group embed another group
   my $hasEmbeddedSoftware = $self->_hasEmbeddedSoftware;
   # Check group services list
   $self->_checkServices;
   # Take first node as reference node
   my $nodeRef = $nodeLst[0];
   my $nodeRefName = $nodeRef->getName;
   my @nodeRefDiskLst = $nodeRef->getDiskLst( @shelfLst );
   my $nodeRefDiskCount = scalar @nodeRefDiskLst;
   # Control disk slices and file system for each node of the group
   foreach $node ( @nodeLst ) {
      $node->checkInstall( $hasSharedSoftware, $hasEmbeddedSoftware, @shelfLst );
      $nodeName = $node->getName;
      @nodeDiskLst = $node->getDiskLst( @shelfLst );
      $nodeDiskCount = scalar @nodeDiskLst;
      # Compare node disk count to node ref disk count
      &sl_traces::error( "Each Node of the group < %s > must have ".
         "same disk count.".
	 "\nNode < %s > has < %s > disk(s) defined".
	 "\nNode < %s > has < %s > disk(s) defined",
	 "- sl_group:_checkInstall - ".__LINE__,
	 ( $groupName, $nodeRefName, $nodeRefDiskCount, 
	   $nodeName, $nodeDiskCount ) )
	 unless ( $nodeDiskCount == $nodeRefDiskCount );
      # Compare nodes disk layout
      foreach $diskRef ( @nodeRefDiskLst ) {
         $diskRefName = $diskRef->getName;
	 $diff = undef;
         # Look for an equivalent disk into the nodeRef disk list
	 foreach $disk ( @nodeDiskLst ) {
            my $diskName = $disk->getName;
	    $diff = sl_disk->compare( $diskRef, $disk );
	    unless ( defined $diff ) {
	       # Disk are identical remove found disk from the list
	       shift @nodeDiskLst;
	       last;
	    }
	 }   
	 &sl_traces::error( "Nodes of the group < %s > ".
	    "have different disk layout".
	    "\nNode  < %s >  disk < %s > has no ".
	    "equivalent among any disk of the node < %s >".
	    "\n%s", "- sl_group:_checkInstall - ".__LINE__,
	    ( $groupName, $nodeRefName, $diskRefName, $nodeName, $diff ) )
	    if ( defined $diff );
      }
   }
}

# ------------------------------------------------------------------------------
#
sub _softInstall {
   my $self = shift;
   my ( $softDir, $clusterDir, $groupPrefix ) = @_;
   my $groupName = $self->getName;
   &sl_traces::trace( 3, "- sl_group::_softInstall - Creating group ".
                      "< ${groupName} > installation software data" );
   # Create software install script
   my $softScript = "${softDir}/$SOFT_SCRIPT";
   open ( SOFTINST, ">".$softScript ) ||
      &sl_traces::error( "- sl_group::_softInstall - Cannot create ".
	                 "< ${softScript} >", __LINE__ );
   # Copy the generic installation script launcher to the jumpstart dir
   my $launcher = "${softDir}/${SOFT_INSTALL_LAUNCHER}";
   my $command = "$ENV{'SMCT_CP'} ".
                 "$ENV{'NHAS_PROD_DIR'}/nhsmct/bin/${SOFT_INSTALL_LAUNCHER} ".
                 "${launcher};";
   &sl_utils::execCmd( $command );
   my @softLst = $self->getSoftLst;
   foreach $soft ( @softLst ) {
      my $softId = $soft->iD;
      my $softName = $soft->getName;
      my $softFile = $soft->getFile;
      # Name of the software to install
      my $distrib = "${SOFTREPO_TAG}/${softId}/${softFile}";
      # Name of the installation script
      my $installScript = $soft->getInstallScript;
      if ( defined $installScript ) {
	 $installScriptName = `$ENV{'SMCT_BASENAME'} $installScript`;
	 chop( $installScriptName );
	 $installScript = "${softDir}/${installScriptName}";
	 # Copy the software installation script to the Jumpstart dir
	 # The installation is a kind of instance parameter. That's why it is
	 # copied in the SWLREP area, not in the SOFTREP area
	 my $installScriptPath = $soft->getInstallScriptPath;
	 $command = "$ENV{'SMCT_CP'} ${installScriptPath} ".
	            "${installScript};";
	 &sl_utils::execCmd( $command );
	 # Create the software install script launcher
	 # Replace local path by nfs path
	 my $clusterId=`$ENV{'SMCT_BASENAME'} ${clusterDir}`;
	 chop( $clusterId );
	 $installScript =~ s,${clusterDir},$EXPORT_TAG/${clusterId},;
	 $launcher =~ s,${clusterDir},$EXPORT_TAG/${clusterId},;
	 print SOFTINST "/bin/sh ${launcher} ${distrib} ${installScript}\n";
      }
   }
   close SOFTINST;
}

# ------------------------------------------------------------------------------
#
sub _pkgInstall {
   my $self = shift;
   my ( $pkgDir, $clusterDir, $groupPrefix, $swlVersion ) = @_;
   my $groupName = $self->getName;
   &sl_traces::trace( 3, "- sl_group::_patInstall - Creating group ".
                      "< ${groupName} > installation package data" );
   # Create packages install script
   my $pkgScript = "${pkgDir}/$PKG_SCRIPT";
   open ( PKGINST, ">".$pkgScript ) ||
      &sl_traces::error( "Cannot create < %s >", 
			 "- sl_group::_pkgInstall - ".__LINE__,
			 ( $pkgScript ) );
   print PKGINST $SCRIPT_DEBUG_HEADER;
   my @pkgLst = $self->getPkgLst;
   foreach $pkg ( @pkgLst ) {
      my $pkgId = $pkg->iD;
      my $pkgAbbr = $pkg->getAbbr;
      my $pkgMode = $pkg->getMode;
      my $pkgBase = $pkg->getBase;
      my $pkgProduct = $pkg->getProduct;
      my $pkgResp = $pkg->getResp;
      my $pkgAdmin = $pkg->getAdmin;
      # Replace SWL tag in installation base dir (Note base dir is always 
      # defined since it is initialized in the sl_package module)
      $pkgBase =~ s/${sl_filesys::SWL_TAG}/${swlVersion}/g;
      my $command = undef;
      my $rootDir = undef;
      my $baseDir = undef;
      # Define admin destination file
      my $admin = "${pkgDir}/admin.${pkgId}";
      # Define response destination file
      my $response = "${pkgDir}/response.${pkgId}";
      if ( $pkgMode eq $sl_package::PACKAGE_MODE{'LOCAL'} ) {
	 if ( ( $self->isMasterEligible ) or ( $self->isDataless ) ) {
	    # For diskfull nodes install on local file system
	    $rootDir = "${ROOT}";
	 }
	 else {
	    # For diskless diskless nodes install on an exported 
	    # file system get file systems mount points
	    $hostGroup = $self->getHostGroup;
	    $exportRoot = &sl_registry::getValue( $hostGroup, "EXPORT_ROOT" );
	    $rootDir = "${ROOT}${exportRoot}/${groupPrefix}/root/".
		       "${NODE_TEMPLATE}";
	 }
	 if ( $pkgProduct eq $sl_software::SOFTWARE_PRODUCT{'SOLARIS'} ) {
	    # Take default install dir for Solaris packages
	    $command = "echo basedir=default >> ${admin};";
	 }
	 else {
	    # Take specific admin file or default Base dir
	    if ( defined $pkgAdmin ) {
               # The admin file is a package instance parameter. That's
	       # why it is copied in the SWLREP area, not in the SOFTREP area
	       $adminPath = $pkg->getAdminPath;
	       $command = "$ENV{'SMCT_CP'} ${adminPath} ".
	                  "${admin};";
	    }
	    else {
	       # Take default value set at package creation time
	       $baseDir = $pkgBase;
	       $command = "echo basedir=${baseDir} >> ${admin};";
	    }
	 }
      }
      elsif ( $pkgMode eq $sl_package::PACKAGE_MODE{'SHARED'} ) {
	 if ( $self->isMasterEligible ) {
	    # For ME nodes install on the 'shared' file system
	    $sharedRoot = 
	       &sl_registry::getValue( $groupName, "SHARED_ROOT" );
	    $databaseRoot = 
	       &sl_registry::getValue( $groupName, "DATABASE_ROOT" );
	    # Package relocation directory
	    $rootDir = "${ROOT}${sharedRoot}/${databaseRoot}";
	    # base dir is replaced if an admin file is defined
	    if ( defined $pkgAdmin ) {
               # The admin file is a package instance parameter. That's
	       # why it is copied in the SWLREP area, not in the SOFTREP area
	       $adminPath = $pkg->getAdminPath;
	       $command = "$ENV{'SMCT_CP'} ${adminPath} ".
	                  "${admin};";
            }
	    else {
	       # take pkg base dir
	       $baseDir = $pkgBase;
	       $command = "echo basedir=${baseDir} >> ${admin};";
	    }
	 }
	 elsif ( ! $self->isDataless ) {
	    # NME diskless nodes shares packages in the usr dir
	    # For diskless diskless nodes install on an exported 
	    # file system get file systems mount points
	    $hostGroup = $self->getHostGroup;
	    $exportRoot = &sl_registry::getValue( $hostGroup, "EXPORT_ROOT" );
	    $rootDir = "${ROOT}${exportRoot}/${groupPrefix}";
	    if ( $pkgProduct eq $sl_software::SOFTWARE_PRODUCT{'SOLARIS'} ) {
	       # Take default install dir for Solaris packages
	       $command = "echo basedir=default >> ${admin};";
	    }
	    else {
	       # Take specific admin file or default Base dir
	       if ( defined $pkgAdmin ) {
        	  # The response file is a package instance paramter. That's
		  # why it is copied in the SWLREP area, not in the SOFTREP
		  # area
		  $adminPath = $pkg->getAdminPath;
		  $command = "$ENV{'SMCT_CP'} ${adminPath} ".
	                     "${admin};";
	       }
	       else {
		  # Take default value set at package creation time
		  $baseDir = $pkgBase;
		  $command = "echo basedir=${baseDir} >> ${admin};";
	       }
	    }
	    # Do not control packages dependencies since the package is
	    # installed in an environment where the pkg database is not
	    # complete
	    $command .= "echo idepend=nocheck >> ${admin};";
	 }
         else {
	    # dataless nodes cannot export shared packages
	    # (normally checked before this point)
	    &sl_traces::error( "Invalid ".
	                       "%s install mode < %s > for dataless ".
			       "group < %s >", __LINE__,
			       ( "package", $pkgMode, $groupName ) );	  
	 }
      }
      # Add nocheck value to action and conflict admin parameters
      if ( defined $command ) {
	 $command .= "echo action=nocheck >> ${admin};".
		    "echo conflict=nocheck >> ${admin};";
      }
      else {
	 $command = "echo action=nocheck >> ${admin};".
		    "echo conflict=nocheck >> ${admin};";
      }
      # Inhibit automatic mail send by pkgadd
      $command .= "echo mail= >> ${admin};";
      # Generate the pkg admin file
      if ( defined $command ) {
	 &sl_utils::execCmd( $command );
      }
      # Get pkg response file if any
      if ( defined $pkgResp ) {
         # The response file is a package instance parameter. That's why it is
	 # copied in the SWLREP area, not in the SOFTREP area
	 my $respPath = $pkg->getRespPath;
	 $command = "$ENV{'SMCT_CP'} ${respPath} ${response};";
	 &sl_utils::execCmd( $command );
      }
      # Create packages installation scripts
      # Replace local path by nfs path
      my $clusterId=`$ENV{'SMCT_BASENAME'} ${clusterDir}`;
      chomp( $clusterId );
      $admin =~ s,${clusterDir},$EXPORT_TAG/${clusterId},;
      $response =~ s,${clusterDir},$EXPORT_TAG/${clusterId},;
      # Adapt distrib. dir according to pkg type
      if ( $pkg->getProduct eq $sl_software::SOFTWARE_PRODUCT{'SOLARIS'} ) {
	 $distrib = $SOLARIS_TAG;
      }
      elsif ( $pkg->getProduct eq $sl_software::SOFTWARE_PRODUCT{'SOLARIS_EA'} ) {
         my $pkgPath = $pkg->getPath;
	 $pkgPath = &sl_utils::replaceVar( $pkgPath );
	 my $solarisEA = &sl_jumpstart::getSolarisEA;
	 ( $prePath, $subPath ) = split( $solarisEA, $pkgPath );
	 $distrib = "${SOLARIS_EA_TAG}${subPath}";
      }
      else {
	 $distrib = "${SOFTREPO_TAG}/${pkgId}";
      }
      # Calculate path root-dir/base-dir to create it on the target
      # before adding the package (for reloc pkgs only)
      if ( defined $baseDir ) {
	 $targetBaseDir = "${rootDir}${baseDir}";
	 print PKGINST "[ -d ${targetBaseDir} ] || ";
	 print PKGINST "/usr/bin/mkdir -p ${targetBaseDir}\n";
      }
      print PKGINST "MSG=\"Adding package < ${pkgAbbr} >\"\n".
                    "trace\n";
      print PKGINST "eval $ENV{'SMCT_PKGADD'} -a ${admin} -R ${rootDir} ";
      print PKGINST "-r ${response} " if ( defined $pkgResp );
      print PKGINST "-d ${distrib} ${pkgAbbr} \$REDIRECT\n";
      print PKGINST "if [ \$? != 0 ]\n".
	            "then\n".
	            "   MSG=\"Error adding package < ${pkgAbbr} >\"\n".
	            "   error\n".
	            "fi\n";
   }
   close PKGINST;
}

# ------------------------------------------------------------------------------
#
sub _patInstall {
   my $self = shift;
   my ( $patDir, $groupPrefix ) = @_;
   my $prePatScript = "${patDir}/$PRE_PAT_SCRIPT";
   my $postPatScript = "${patDir}/$POST_PAT_SCRIPT";
   my $groupName = $self->getName;
   &sl_traces::trace( 3, "- sl_group::_patInstall - Creating group ".
                      "< ${groupName} > installation patch data" );
   # Create patches install scripts
   open ( PREPATINST, ">".$prePatScript ) ||
      &sl_traces::error( "Cannot create < %s >", 
			 "- sl_group::_patInstall - ".__LINE__,
			 ( $prePatScript ) );
   open ( POSTPATINST, ">".$postPatScript ) ||
      &sl_traces::error( "Cannot create < %s >", 
			 "- sl_group::_patInstall - ".__LINE__,
			 ( $postPatScript ) );
   print PREPATINST $SCRIPT_DEBUG_HEADER;
   print POSTPATINST $SCRIPT_DEBUG_HEADER;
   my @patLst = $self->getPatLst;
   foreach $pat ( @patLst ) {
      my $patId = $pat->iD;
      my $patName = $pat->getName;
      my $command = undef;
      my $rootDir = undef;
      if ( ( $self->isMasterEligible ) or ( $self->isDataless ) ) {
	 if ( $pat->isShared ) {
	    &sl_traces::error( "Invalid ".
	                       "%s install mode < %s > for dataless ".
			       "group < %s >", __LINE__,
			       ( "patch", "SHARED", $groupName  ) )
             unless ( $self->isMasterEligible );
	    
	    my $sharedRoot = 
	       &sl_registry::getValue( $groupName, "SHARED_ROOT" );
	    my $databaseRoot = 
	       &sl_registry::getValue( $groupName, "DATABASE_ROOT" );
	    # Patch relocation directory
	    $rootDir = "${ROOT}${sharedRoot}/${databaseRoot}";	    
	 }
	 else {
	    $rootDir = "${ROOT}";
	 }
      }
      else {
	 # For diskless nodes install on an exported 
	 # file system get files systems mount points
	 $hostGroup = $self->getHostGroup;
	 $exportRoot = &sl_registry::getValue( $hostGroup, "EXPORT_ROOT" );
	 if ( $pat->isShared ) {
	    # Install on diskless /usr shared file system if needed
	    $rootDir = 
	       "${ROOT}${exportRoot}/${groupPrefix}";
	 }
	 else {
	    $rootDir = 
	       "${ROOT}${exportRoot}/${groupPrefix}/root/${NODE_TEMPLATE}";
	 }
      }
      # Create patches installation scripts
      if ( $pat->{'Type'} eq $sl_software::SOFTWARE_TYPE{'PRE_PATCH'} ) {
	 print PREPATINST "MSG=\"Adding patch < ${patName} >\"\n".
	                  "trace\n";
	 print PREPATINST "eval $ENV{'SMCT_PATCHADD'} -R ${rootDir} ".
	        	  "${SOFTREPO_TAG}/${patId}/${patName} \$REDIRECT\n";
         print PREPATINST "if [ \$? != 0 ]\n".
	                  "then\n".
	                  "   MSG=\"Error adding patch < ${patName} >\"\n".
	                  "   error\n".
	                  "fi\n";
	 
      }
      elsif ( $pat->{'Type'} eq $sl_software::SOFTWARE_TYPE{'POST_PATCH'} ) {
	 print POSTPATINST "MSG=\"Adding patch < ${patName} >\"\n".
	                   "trace\n";
	 print POSTPATINST "eval $ENV{'SMCT_PATCHADD'} -R ${rootDir} ".
	                   "${SOFTREPO_TAG}/${patId}/${patName} \$REDIRECT\n";
         print POSTPATINST "if [ \$? != 0 ]\n".
	                   "then\n".
	                   "   MSG=\"Error adding patch < ${patName} >\"\n".
	                   "   error\n".
	                   "fi\n";
      }

   }
   close PREPATINST;
   close POSTPATINST;
}

# ------------------------------------------------------------------------------
#
sub _softwareInstall {
   my $self = shift;
   my ( $groupDir, $clusterDir, $swlVersion ) = @_;
   my $groupName = $self->getName;
   my $groupArch = ${sl_board::ARCH2DIR{$self->getArch}};
   my $groupOs = ${sl_node::OS2DIR{$self->getOs}};
   my $groupPrefix = "${swlVersion}/${groupArch}/${groupOs}/${groupName}";
   &sl_traces::trace( 3, "- sl_group::_softwareInstall - Creating group ".
                      "< ${groupName} > software data" );
   # Create Jumpstart environment for the group
   my $jumpstartDir = "${groupDir}/${sl_jumpstart::JUMPSTART_INSTALL_DIR}";
   my $pkgDir = "${groupDir}/${sl_jumpstart::JUMPSTART_INSTALL_PKG_DIR}";
   my $patDir = "${groupDir}/${sl_jumpstart::JUMPSTART_INSTALL_PAT_DIR}";
   my $softDir = "${groupDir}/${sl_jumpstart::JUMPSTART_INSTALL_SW_DIR}";
   ( -d $pkgDir ) || &sl_utils::createDir( $pkgDir );
   ( -d $patDir ) || &sl_utils::createDir( $patDir );
   ( -d $softDir ) || &sl_utils::createDir( $softDir );
   # Process software lists
   $self->_pkgInstall( $pkgDir, $clusterDir, $groupPrefix, $swlVersion );
   $self->_patInstall( $patDir, $groupPrefix );
   $self->_softInstall( $softDir, $clusterDir, $groupPrefix );
}

# ------------------------------------------------------------------------------
#
sub configure {
   $self = shift;
   my ( $clusterDir ) = @_;
   &sl_traces::info( 2, "Configuring nodes group < $self->{'Name'} >" );
   my $groupName = $self->getName; 
   my $groupDir = "${clusterDir}/${groupName}";
   ( -d $groupDir ) || &sl_utils::createDir( $groupDir );
   my @configLst = $self->getConfigLst;
   # Get and install user defined config. files
   my $remDir = $groupDir;
   $remDir =~ s,${groupDir},${WORKING_TAG},;
   foreach $config ( @configLst ) {
      $config->fetch( $groupDir );
   }
   # Sort the config scripts of the nodes group and create the .order file
   sl_config->createOrderedConfigLst( $groupDir, $remDir, @configLst )
      if ( scalar @configLst > 0 );
}

# ------------------------------------------------------------------------------
#
sub deploy {
   $self = shift;
   my ( $clusterDir, $domain, $swlVersion, 
        $swlRepDir, $clusterName, @shelfLst ) = @_;
   &sl_traces::info( 2, "Configuring nodes group < %s >",
                     ( $self->{'Name'} ) );
   my $domainId = $domain->getId;
   my $groupName = $self->getName; 
   my $groupDir = "${clusterDir}/${groupName}";
   ( -d $groupDir ) || &sl_utils::createDir( $groupDir );
   my $groupRepDir = "${swlRepDir}/${groupName}";
   my $registryName = "${groupRepDir}/${sl_registry::REGISTRY_NAME}";
   # Load group registry
   &sl_registry::import( $groupName, $registryName );
   # Getting node independent config. script (if any)
   my @groupServiceLst = $self->getServiceLst;
   my $servicesDir = "${groupDir}/services";
   my $remDir = $servicesDir;
   $remDir =~ s,${groupDir}/services,${WORKING_TAG},;
   foreach $service ( @groupServiceLst ) {
      my @configLst = $service->getConfigLst;
      foreach $config ( @configLst ) {
	if ( defined $config ) {
           $config->fetch( $servicesDir );
           sl_config->createOrderedConfigLst( 
	      $servicesDir, $remDir, ( $config ) )
	}
      }
      # Create node specific configuration
      my @nodeLst = $self->getNodeLst;
      foreach $node ( @nodeLst ) {
	 # Create node configuration files
	 $node->deploy( $service, $groupDir, $domain, 
	                $swlVersion, $clusterName, @shelfLst );
      }
   }
   # Create the Jumpstart profile for deployment of nodes group 
   my $jumpstartDir = "${groupDir}/${sl_jumpstart::JUMPSTART_DEPLOY_DIR}";
   if ( ( $self->isMasterEligible ) or ( $self->isDataless ) ) {
      &sl_jumpstart::configure( 
         ${sl_jumpstart::JUMPSTART_TARGET{'CLUSTER'}},
	 $jumpstartDir, $self, $swlVersion, $clusterName, @shelfLst );
   }
}

# ------------------------------------------------------------------------------
#
sub install {
   my $self = shift;
   my ( $installDir, $swlVersion, $clusterName, @shelfLst ) = @_;
   &sl_traces::info( 2, "Installing nodes group < %s >",
                     ( $self->{'Name'} ) );
   my $name = $self->getName;
   my $groupDir = "${installDir}/${name}";
   my @nodeLst = $self->getNodeLst;
   # Install each node of the group and check the result
   foreach $node ( @nodeLst ) {
      $node->install( $groupDir );
   }
   $self->_checkInstall( @shelfLst );
   # Create the Jumpstart profile for diskfull group 
   my $jumpstartDir = "${groupDir}/${sl_jumpstart::JUMPSTART_INSTALL_DIR}";
   if ( ( $self->isMasterEligible ) or ( $self->isDataless ) ) {
      &sl_jumpstart::configure( 
         ${sl_jumpstart::JUMPSTART_TARGET{'PROTOTYPE'}},
	 $jumpstartDir, $self, $swlVersion, $clusterName, @shelfLst );
   }
   # Generate the packages and patches installation scripts
   $self->_softwareInstall( 
      $groupDir, $installDir, $swlVersion, $clusterName, @shelfLst );
   # Generate ident group file
   my $type = $self->getType;
   my $arch = ${sl_board::ARCH2DIR{$self->getArch}};
   my $os = ${sl_node::OS2DIR{$self->getOs}};
   my $class = $self->getClass;
   my $ident = "${groupDir}/${GROUP_IDENT}";
   my $command = "echo NAME ${name} >> ${ident};".
                 "echo TYPE ${type} >> ${ident};".
                 "echo ARCH ${arch} >> ${ident};".
                 "echo CLASS ${class} >> ${ident};".
		 "echo OS ${os} >> ${ident};";
   my @embeddedGroupLst = $self->getEmbeddedGroupLst;
   foreach $group ( @embeddedGroupLst ) {
      $command = $command."echo GROUP ${group} >> ${ident};";
   }
   &sl_utils::execCmd( $command );
   &sl_registry::export( $name, $groupDir, ${sl_registry::REGISTRY_NAME} );
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   my $groupId = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_group::iD - Group iD is < ${groupId} >" );
   return $groupId;
}

# ------------------------------------------------------------------------------

return 1;
