#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_drawer.pm 1.10     02/10/31 SMI"
# ------------------------------------------------------------------------------

BEGIN {
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_package.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
}

# ------------------------------------------------------------------------------
package sl_drawer;

%DRAWER_TYPE = ( 'CT400' => "CT400",
                 'CT800' => "CT800",
		 'CT410' => "CT410",
                 'CT810' => "CT810",
                 'NETRA_20' => "NETRA_20",
		 'T1_200' => "T1_200",
		 'T1_105' => "T1_105",
		 'GENERIC' => "GENERIC",
		 'SUN_STOREDGE' => "SUN_STOREDGE" );

# -----------------------------------------------------------------------------
# Drawer Hidden enum values

%DRAWER_HIDDEN_VALUES = ( 
	       'CT400' => 1,
               'CT800' => 1,
	       'CT410' => 0,
               'CT810' => 0,
               'NETRA_20' => 0,
	       'T1_200' => 0,
	       'T1_105' => 0,
	       'GENERIC' => 1,
	       'SUN_STOREDGE' => 1 );

# ID fields separator
my $ID_SEP = '@';

# Drawer object attributes
my @DRAWER_ATTR = qw( Id
		      Name
		      Type
		      BoardLst
		      DiskLst  );

%DRAWER_DEF_COMP_MASK = ( 'Id' => 0,
                	  'Name' => 1,
			  'BoardLst' => 1,
			  'DiskLst' => 1 );

# ------------------------------------------------------------------------------
#
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} ) &&
        ( defined $attr{'Name'} ) &&
        ( defined $attr{'Type'} ) ) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      $self->{'Type'} = $attr{'Type'};
      # Set optional attribute
      @{$self->{'BoardLst'}} = @{$attr{'BoardLst'}}
         if defined ( @{$attr{'BoardLst'}} );
      @{$self->{'DiskLst'}} = @{$attr{'DiskLst'}} 
         if defined ( @{$attr{'DiskLst'}} );
   }
   elsif ( defined $attr{'DrawerObject'} ) {
      $self->_construct( $attr{'DrawerObject'} );
   }
   else {
      &sl_traces::error( "Unable to create %s object. ".
                         "Invalid creation method", 
			 "- sl_drawer::new - ".__LINE__,
			 ( "sl_drawer" ) );
   }
   &sl_traces::trace( 2, "- sl_drawer::new - Creating drawer object ".
                "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $drawerObject ) = @_;
   &sl_traces::trace( 3, "- sl_drawer::_construct - Building drawer object" );
   my $drawer = sl_iterator->new( 'Object' => $drawerObject );
   $self->{'Id'} = $drawer->extract;
   $self->{'Name'} = $drawer->extract;
   $self->{'Type'} = $drawer->extract;
   @{$self->{'BoardLst'}} =
      $drawer->extractObjectArray( "sl_board", "BoardObject" );
   @{$self->{'DiskLst'}} =
      $drawer->extractObjectArray( "sl_disk", "DiskObject" );
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_drawer::serialize - Serializing drawer object" );
   my $drawerObject = sl_iterator->new( 'Object' => "" );
   foreach $attr ( @DRAWER_ATTR ) {
      if ( $attr eq 'BoardLst' ) {
         $drawerObject->serializeObjectArray( @{$self->{'BoardLst'}} );
      }
      elsif ( $attr eq 'DiskLst' ) {
         $drawerObject->serializeObjectArray( @{$self->{'DiskLst'}} );
      }
      else {
         $drawerObject->serialize( $self->{$attr} );
      }
   }
   return $drawerObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_drawer::display - Displaying drawer" );
   print STDOUT "\nDrawer\n";
   print STDOUT "------\n";
   foreach ( @DRAWER_ATTR ) {
      if ( defined $self->{$_} ) {
	 if ( $_ eq 'DiskLst' ) {
            print STDOUT "$_ =\n";
            foreach $disk (@{$self->{'DiskLst'}}) {
	       $disk->display( $brief );
	    }
	 }
	 elsif ( $_ eq 'BoardLst' ) {
            print STDOUT "$_ =\n";
            foreach $board ( @{$self->{'BoardLst'}} ) {
	       $board->display( $brief );
	    }
	 }
	 else {
            print STDOUT "$_ = ";
            print STDOUT "$self->{$_}";
            print STDOUT "\n";
	 }
      }
      else {
         print STDOUT "$_ =\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub compare {
   my $class = shift;
   my ( $drawer1, $drawer2 ) = @_;
   my $drawer1Name = ( defined $drawer1 ) ? $drawer1->getName:"<undefined>";
   my $drawer2Name = ( defined $drawer2 ) ? $drawer2->getName:"<undefined>";
   &sl_traces::trace( 2, "- sl_drawer::compare - Comparing drawers ".
                     "< ${drawer1Name} > and < ${drawer2Name} >" );
   # String holding differences if any
   my $diffStr = undef;
   # Extract the differences if any
   foreach $attr ( keys %DRAWER_DEF_COMP_MASK ) {
      $diffStr1 = "Comparing drawer : ${drawer1Name}.";
      $diffStr2 = "       to drawer : ${drawer2Name}.";
      if ( $DRAWER_DEF_COMP_MASK{$attr} ) {
         if ( &sl_utils::testAttribute( $drawer1->{$attr} ) and 
	      &sl_utils::testAttribute( $drawer2->{$attr} ) ) {
            if ( $attr eq 'BoardLst' ) {
	       # Sort the lists for comparison
	       $drawer1->setBoardLst( 
	          &sl_utils::sortListByName( $drawer1->getBoardLst ) );
	       $drawer2->setBoardLst( 
	          &sl_utils::sortListByName( $drawer2->getBoardLst ) );
               # Look for differences
	       @drawer1BoardLst = $drawer1->getBoardLst;
	       @drawer2BoardLst = $drawer2->getBoardLst;
	       for ($i=0; $i < scalar @drawer1BoardLst; $i++) {
		  $diff = sl_board->compare( 
		           $drawer1BoardLst[$i], $drawer2BoardLst[$i] );
		  $diffStr .= $diffStr1."BoardLst[$i]\n".
		              $diffStr2."BoardLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
            elsif ( $attr eq 'DiskLst' ) {
	       # Sort the lists for comparison
	       $drawer1->setDiskLst( 
	          &sl_utils::sortListByName( $drawer1->getDiskLst ) );
	       $drawer2->setDiskLst( 
	          &sl_utils::sortListByName( $drawer2->getDiskLst ) );
               # Look for different switches
	       @drawer1DiskLst = $drawer1->getDiskLst;
	       @drawer2DiskLst = $drawer2->getDiskLst;
	       for ($i=0; $i < scalar @drawer1DiskLst; $i++) {
		  $diff = sl_disk->compare( 
		           $drawer1DiskLst[$i], $drawer2DiskLst[$i] );
		  $diffStr .= $diffStr1."DiskLst[$i]\n".
		              $diffStr2."DiskLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
	    else {
	       if ( $drawer1->{$attr} ne $drawer2->{$attr} ) {
        	  $diffStr1 .= "${attr}=".$drawer1->{$attr}."\n";
		  $diffStr2 .= "${attr}=".$drawer2->{$attr}."\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }
            }
	 }
         elsif ( ! &sl_utils::testAttribute( $drawer1->{$attr} ) and
	           &sl_utils::testAttribute( $drawer2->{$attr} ) )  {
            unless ( ( $attr eq 'BoardLst' ) or 
	             ( $attr eq 'DiskLst' ) ) {
	       $diffStr1 .= "${attr}= <undefined>\n";
	       $diffStr2 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
	    }
	    else {
	       if ( scalar @{$drawer2->{$attr}} > 0 ) {
		  $diffStr1 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
		}       
	    }
         }
         elsif ( ! &sl_utils::testAttribute( $drawer2->{$attr} ) and
	           &sl_utils::testAttribute( $drawer1->{$attr} ) ) {
            unless ( ( $attr eq 'BoardLst' ) or 
	             ( $attr eq 'DiskLst' ) ) {
               $diffStr2 .= "${attr}= <undefined>\n";
	       $diffStr1 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
 	    }
	    else {
	       if ( scalar @{$drawer1->{$attr}} > 0 ) {
		  $diffStr2 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
		}       
	    }
	 }
      }
   }
   return $diffStr;
}

# ------------------------------------------------------------------------------
#
sub getDiskLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_drawer::getDiskLst - Getting ".
                      "< $self->{'Name'} > disk list" );
   if ( defined $self->{'DiskLst'} ) {
      return @{$self->{'DiskLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getBoardLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_drawer::getBoardLst - Getting ".
                      "< $self->{'Name'} > board list" );
   if ( defined $self->{'BoardLst'} ) {
      return @{$self->{'BoardLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub setBoardLst {
   my $self = shift;
   my ( @boardLst ) = @_;
   &sl_traces::trace( 2, "- sl_drawer::setBoardLst - Setting ".
                      "< $self->{'Name'} > board list" );
   if ( defined $boardLst[0] ) {
      @{$self->{'BoardLst'}} = @boardLst;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_drawer::setBoardLst - ".__LINE__,
			 ( "Board list" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub setDiskLst {
   my $self = shift;
   my ( @diskLst ) = @_;
   &sl_traces::trace( 2, "- sl_drawer::setDiskLst - Setting ".
                      "< $self->{'Name'} > disk list" );
   if ( defined $diskLst[0] ) {
      @{$self->{'DiskLst'}} = @diskLst;
   }
#   else {
#      &sl_traces::error( "%s is undefined",
#                         "- sl_drawer::setDiskLst - ".__LINE__,
#			 ( "Disk list" ) );  
#   }
}

# ------------------------------------------------------------------------------
#
sub getDisk {
   my $self = shift;
   my ( $diskOid ) = @_;
   &sl_traces::trace( 2, "- sl_drawer::getDisk - Getting disk ".
                "< ${diskOid} >" );
   my @diskLst = $self->getDiskLst;
   foreach $disk ( @diskLst ) {
      return $disk if ( $disk->getName eq $diskOid ); 
   }
   return undef;
}

# ------------------------------------------------------------------------------
#
sub getBoard {
   my $self = shift;
   my ( $boardOid ) = @_;
   &sl_traces::trace( 2, "- sl_drawer::getBoard - Getting board ".
                "< ${boardOid} >" );
   foreach $board ( @{$self->{'BoardLst'}} ) {
      return $board if ( $board->getName eq $boardOid ); 
   }
   return undef;
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_drawer::getName - Getting ".
                      "< $self->{'Name'} > Name attribute" );
   return $self->{'Name'};
}

# ------------------------------------------------------------------------------
#
sub isHidden {
   my $class = shift;
   my $enumValue = shift;
   &sl_traces::trace( 2, "- sl_drawer::isHidden - Testing ". 
                      "< ${enumValue} > hidden value" );
   return $DRAWER_HIDDEN_VALUES{$enumValue};
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   my $drawerId = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_drawer::iD - Drawer iD is < ${drawerId} >" );
   return $drawerId;
}

# ------------------------------------------------------------------------------

return 1;
