#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_disk.pm 1.10     02/10/31 SMI"
# ------------------------------------------------------------------------------


BEGIN { 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_slice.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
}

# ------------------------------------------------------------------------------

package sl_disk;

# ID fields separator
my $ID_SEP = '@';

%DISK_TYPE = ( 'SCSI' => "SCSI",
               'IDE' => "IDE",
	       'FC' => "FC" );
	       
%DISK_HIDDEN_VALUES = ( 
               'SCSI' => 0,
               'IDE' => 0,
	       'FC' => 0 );

# disk object attributes
my @DISK_ATTR = qw( Id
                    Name
		    Type
		    Size
		    Device
		    SliceLst );
		    
%DISK_DEF_COMP_MASK = ( 'Id' => 0,
                        'Name' => 0,
			'Type' => 0,
			'Size' => 1,
			'Device' => 0,
		        'SliceLst' => 1 );


# ------------------------------------------------------------------------------
# 
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} ) && 
        ( defined $attr{'Name'} ) &&
        ( defined $attr{'Type'} ) && 
        ( defined $attr{'Size'} ) && 
        ( defined $attr{'Device'} ) && 
	( defined $attr{'SliceLst'} ) ) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      $self->{'Type'} = $attr{'Type'};
      $self->{'Size'} = $attr{'Size'};
      $self->{'Device'} = $attr{'Device'};
      $self->{'SliceLst'} = $attr{'SliceLst'};
   }
   elsif ( defined $attr{'DiskObject'} ) {
      $self->_construct( $attr{'DiskObject'} );
   }
   else {
      &sl_traces::error( "Unable to create %s object. ".
                         "Invalid creation method", 
			 "- sl_disk::new - ".__LINE__,
			 ( "sl_disk" ) );
   }
   &sl_traces::trace( 2, "- sl_disk::new - Creating disk object ".
                "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $diskObject ) = @_;
   &sl_traces::trace( 3, "- sl_disk::_construct - Building disk object" );
   my $disk = sl_iterator->new( 'Object' => $diskObject );
   $self->{'Id'} = $disk->extract;
   $self->{'Name'} = $disk->extract;
   $self->{'Type'} = $disk->extract;
   $self->{'Size'} = $disk->extract;
   $self->{'Device'} = $disk->extract;
   @{$self->{'SliceLst'}} =
      $disk->extractObjectArray( "sl_slice", "SliceObject" );
}

# ------------------------------------------------------------------------------
#
sub compare {
   my $class = shift;
   my ( $disk1, $disk2 ) = @_;
   my $disk1Name = ( defined $disk1 ) ? $disk1->getName:"<undefined>";
   my $disk2Name = ( defined $disk2 ) ? $disk2->getName:"<undefined>";
   &sl_traces::trace( 2, "- sl_disk::compare - Comparing disks ".
                     "< ${disk1Name} > and < ${disk2Name} >" );
   # String holding differences if any
   my $diffStr = undef;
   # Extract the differences if any
   foreach $attr ( keys %DISK_DEF_COMP_MASK ) {
      $diffStr1 = "Comparing disk : ${disk1Name}.";
      $diffStr2 = "       to disk : ${disk2Name}.";
      if ( $DISK_DEF_COMP_MASK{$attr} ) {
         if ( &sl_utils::testAttribute( $disk1->{$attr} ) and 
	      &sl_utils::testAttribute( $disk2->{$attr} ) ) {
            if ( $attr eq 'SliceLst' ) {
	       # Sort the disks slices lists for comparison
	       $disk1->_sortSliceLst;
	       $disk2->_sortSliceLst;
               # Look for different slices
	       @disk1SliceLst = $disk1->getSliceLst;
	       @disk2SliceLst = $disk2->getSliceLst;
	       for ($i=0; $i < scalar @disk1SliceLst; $i++) {
		  $diff = sl_slice->compare( 
		           $disk1SliceLst[$i], $disk2SliceLst[$i] );
		  $diffStr .= $diffStr1."SliceLst[$i]\n".
		              $diffStr2."SliceLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
	    else {
	       if ( $disk1->{$attr} ne $disk2->{$attr} ) {
        	  $diffStr1 .= "${attr}=".$disk1->{$attr}."\n";
		  $diffStr2 .= "${attr}=".$disk2->{$attr}."\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }
            }
	 }
         elsif ( ! &sl_utils::testAttribute( $disk1->{$attr} ) and
	           &sl_utils::testAttribute( $disk2->{$attr} ) )  {
               $diffStr1 .= "${attr}= <undefined>\n";
	       $diffStr2 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
         }
         elsif ( ! &sl_utils::testAttribute( $disk2->{$attr} ) and
	           &sl_utils::testAttribute( $disk1->{$attr} ) ) {
               $diffStr2 .= "${attr}= <undefined>\n";
	       $diffStr1 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
	 }
      }
   }
   return $diffStr;
}

# ------------------------------------------------------------------------------
#
sub _sortSliceLst {
   my $self = shift;
   my $diskName = $self->getName;
   &sl_traces::trace( 2, "- sl_disk::_sortSliceLst - Sorting disk ".
                      "< ${diskName} > slices list" );
   my @sliceLst = $self->getSliceLst;
   my %sortedSliceLst = ();
   foreach $slice ( @sliceLst ) {
      my $number = "";
      if ( $slice->isMeta ) {
         $number .= "d";
      }
      else {
         $number .= "s";
      }
      $number .= $slice->getNumber;
      unless ( defined $sortedSliceLst{$number} ) {
         $sortedSliceLst{$number} = $slice;
      }
      else {
         &sl_traces::error( "The slice number ".
	                    "< %s > is multiple defined for the disk ".
			    "< %s >", 
			    "- sl_disk::_sortSliceLst - ".__LINE__,
			    ( $number, $diskName ) );
      }
   }
   # Build sorted list ( $key is the slice number )
   @sliceLst = ();
   foreach $key ( sort keys %sortedSliceLst ) {
      push( @sliceLst, $sortedSliceLst{$key} );
   }
   $self->setSliceLst( @sliceLst );
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_disk::serialize - Serializing disk object" );
   my $diskObject = sl_iterator->new( 'Object' => "" );
   foreach $attr ( @DISK_ATTR ) {
      if ( $attr eq 'SliceLst' ) {
         $diskObject->serializeObjectArray(  @{$self->{'SliceLst'}} );
      }	 
      else {         
	 $diskObject->serialize( $self->{$attr} );
      }
   }
   return $diskObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_disk::display - Displaying disk" );
   print STDOUT "\nDisk\n";
   print STDOUT "----\n";
   foreach ( @DISK_ATTR ) {
      if ( defined $self->{$_} ) {
	 if ( $_ eq 'SliceLst' ) {
            print STDOUT "$_ =\n";
            foreach $slice (@{$self->{'SliceLst'}}) {
	       $slice->display( $brief );
	    }
	 }
	 else {
	    print STDOUT "$_ = ";
	    print STDOUT "$self->{$_}";
	    print STDOUT "\n";
	 }
      }
      else {
	 print STDOUT "$_ =\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub getSliceLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_disk::getSliceLst - Getting < $self->{'Name'} > ".
                      "slice list" );
   if ( defined $self->{'SliceLst'} ) {
      return @{$self->{'SliceLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getSize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_disk::getSize - Getting < $self->{'Name'} > ".
                      "Size" );
   return $self->{'Size'};
}

# ------------------------------------------------------------------------------
#
sub getType {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_disk::getType - Getting < $self->{'Name'} > ".
                      "Type" );
   return $self->{'Type'};
}

# ------------------------------------------------------------------------------ 
#
sub isIDE {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_disk::isIDE - Testing < $self->{'Name'} > ".
                      "disk type" );
   my $diskType = $self->getType;
   if ( $diskType eq $DISK_TYPE{'IDE'} ) {
      return 1;
   }
   else {
      return 0;
   }
}

# ------------------------------------------------------------------------------
#
sub setSliceLst {
   my $self = shift;
   my ( @sliceLst ) = @_;
   &sl_traces::trace( 2, "- sl_disk::setSliceLst - Setting < $self->{'Name'} > ".
                      "slice list" );
   if ( defined $sliceLst[0] ) {
      @{$self->{'SliceLst'}} = @sliceLst;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_disk::setSliceLst - ".__LINE__,
			 ( "Slice list" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub setName {
   my $self = shift;
   my ( $name ) = @_;
   &sl_traces::trace( 2, "- sl_disk::setName - Setting < $self->{'Name'} > ".
                      "name" );
   if ( defined $name ) {
      $self->{'Name'} = $name;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_disk::setName - ".__LINE__,
			 ( "Name" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub setId {
   my $self = shift;
   my ( $id ) = @_;
   &sl_traces::trace( 2, "- sl_disk::setId - Setting < $self->{'Name'} > ".
                      "Id" );
   if ( defined $id ) {
      $self->{'Id'} = $id;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_disk::setId - ".__LINE__,
			 ( "Id" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_disk::getName - Getting < $self->{'Name'} > ".
                      "name" );
   return $self->{'Name'};
}

# ------------------------------------------------------------------------------
#
sub getDevice {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_disk::getDevice - Getting < $self->{'Name'} > ".
                     "device" );
   return $self->{'Device'};
}

# ------------------------------------------------------------------------------
#
sub isHidden {
   my $class = shift;
   my $enumValue = shift;
   &sl_traces::trace( 2, "- sl_disk::isHidden - Testing ". 
                      "< ${enumValue} > hidden value" );
   return $DISK_HIDDEN_VALUES{$enumValue};
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   my $diskId = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_disk::iD - Disk iD is < ${diskId} >" );
   return $diskId;
}

# ------------------------------------------------------------------------------

return 1;
