#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_config.pm 1.8     02/10/31 SMI"
# ------------------------------------------------------------------------------

BEGIN { 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_file.pm"; 
}

# ------------------------------------------------------------------------------

package sl_config;

# ID fields separator
my $ID_SEP = '@';

# config object attributes
my @CONFIG_ATTR = qw( Id
                      Name
		      Next
		      FileLst );

my $CONFIG_DIR = "config";

# Installation scripts
my $CONFIG_PRE_INSTALL_SCRIPT = "confinstall.pre.sh";
my $CONFIG_POST_INSTALL_SCRIPT = "confinstall.post";
my $CONFIG_DATA_INSTALL_SCRIPT = "confinstall.sh";

# ------------------------------------------------------------------------------
#
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} )  && 
        ( defined $attr{'Name'} ) && 
	( defined $attr{'FileLst'} ) ) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      $self->{'FileLst'} = $attr{'FileLst'};
      # Set optional attributes
      $self->{'Next'} = $attr{'Next'}
         if ( defined $attr{'Next'} );     
   }
   elsif ( defined $attr{'ConfigObject'} ) {
      $self->_construct( $attr{'ConfigObject'} );
   }
   else {
      &sl_traces::error( "Unable to create %s object. ".
                         "Invalid creation method",
			 "- sl_config::new - ".__LINE__,
			 ( "sl_config" ) );
   }
   &sl_traces::trace( 2, "- sl_config::new - Creating config object ".
                      "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $configObject ) = @_;
   &sl_traces::trace( 3, "- sl_config::_construct - Building config object" );
   my $config = sl_iterator->new( 'Object' => $configObject );
   $self->{'Id'} = $config->extract;
   $self->{'Name'} = $config->extract;
   $self->{'Next'} = $config->extract;
   @{$self->{'FileLst'}} =
      $config->extractObjectArray( "sl_file", "FileObject" );
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_config::serialize - Serializing config ".
                      "object" );
   my $configObject = sl_iterator->new( 'Object' =>  "" );
   foreach $attr ( @CONFIG_ATTR ) {
      if ( $attr eq "FileLst" ) {
         $configObject->serializeObjectArray( @{$self->{'FileLst'}} );
      }
      else {
	 $configObject->serialize( $self->{$attr} );
      }
   }
   return $configObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_config::display - Displaying config" );
   print STDOUT "\nConfig\n";
   print STDOUT "------\n";
   foreach ( @CONFIG_ATTR ) {
      if ( defined $self->{$_} ) {
         if ( $_ eq "FileLst" ) {
	    print STDOUT "$_ =\n\n";
	    foreach $file ( @{$self->{$_}} ) {
	       $file->display( $brief );
	    }
	 }
	 else {
	    print STDOUT "$_ = ";
	    print STDOUT "$self->{$_}";
	    print STDOUT "\n";
	 }
      }
      else {
	 print STDOUT "$_ = ";
         print STDOUT "\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub fetch {
   my $self = shift;
   my ( $destDir ) = @_;  
   &sl_traces::trace( 2, "- sl_config::fetch - Fetching config files for ".
                      "< $self->{'Name'} >" );
   my @fileLst = $self->getFileLst;
   my $configName = $self->getName;
   foreach $file ( @fileLst ) {
      $file->fetch( "$destDir/${CONFIG_DIR}/${configName}" );
   }
}

# ------------------------------------------------------------------------------
#
sub _install {
   my $self = shift;
   my ( $destDir, $remDir ) = @_;  
   &sl_traces::trace( 3, "- sl_config::install - Installing config files for ".
                      "< $self->{'Name'} >" );
   my $configName = $self->getName;
   my @fileLst = $self->getFileLst;
   foreach $file ( @fileLst ) {
      my $confFile = $file->getFile;
      my $command = undef;
      ( -d $destDir ) || &sl_utils::createDir( $destDir );
      if ( $file->isPreScript ) {
	 # Append command to the end of the install script
	 my $installScript = "${destDir}/${CONFIG_PRE_INSTALL_SCRIPT}";
	 my $configScript = "${remDir}/${confFile}";
	 $command = "echo /bin/sh \"${configScript}\" >> ${installScript}";
	 &sl_utils::execCmd( $command );
      }
      elsif ( $file->isPostScript ) {
	 # Append command to the end of the install script
	 my $runLevel = $file->getRunLevel;
	 my $installScript = 
	    "${destDir}/${CONFIG_POST_INSTALL_SCRIPT}.${runLevel}.sh";
	 my $configScript = "${remDir}/${confFile}";
	 $command = "echo /bin/sh \"${configScript}\" >> ${installScript}";
	 &sl_utils::execCmd( $command );
      }
      elsif ( $file->isData ) {
	 # There is nothing to do for data files 
	 # (the fetch method puts the file in the appropriate directory)
      }
   }
}

# ------------------------------------------------------------------------------
#
sub createOrderedConfigLst {
   my $class = shift;
   my ( $destDir, $remDir, @configLst ) = @_;
   &sl_traces::trace( 2, "- sl_config::createOrderedConfigLst - Creating ".
                      "ordered config script list" );
   # Extract pre and post config scripts sub-lists
   my @preConfigLst = ();
   my @postConfigLst = ();
   foreach $config ( @configLst ) {
      my $configName = $config->getName;
      my @fileLst = $config->getFileLst;
      my $configType = undef;
      foreach $file ( @fileLst ) {
         my $fileType = $file->getType;
         if ( defined $configType ) {
	    &sl_traces::error( "Files of the config element < %s > ".
	                       "cannot have different types",
	                       "- sl_config::createOrderedConfigLst -".
			       __LINE__, ( $configName )  )
               unless ( ( $configType eq $fileType ) or 
	                ( $fileType eq $sl_file::FILE_TYPE{'DATA'} ) );
	 }
	 else {
	    # Skip data files when determining config type
            $configType = $fileType
	       unless ( $fileType eq $sl_file::FILE_TYPE{'DATA'} );
	 }
      }
      if ( $configType eq $sl_file::FILE_TYPE{'PRE_SCRIPT'} ) {
	 push( @preConfigLst, $config );
      }
      elsif ( $configType eq $sl_file::FILE_TYPE{'POST_SCRIPT'} ) { 
	 push( @postConfigLst, $config );
      }
      elsif ( $configType eq $sl_file::FILE_TYPE{'DATA'} ) {
         # The data files dont need to be ordered
      }
   }
   my %confTypeLst = ();
   @{$confTypeLst{$sl_file::FILE_TYPE{'PRE_SCRIPT'}}} = @preConfigLst;
   @{$confTypeLst{$sl_file::FILE_TYPE{'POST_SCRIPT'}}} = @postConfigLst;
   foreach $key ( keys %confTypeLst ) {
      @configLst = @{$confTypeLst{$key}};
      next if ( scalar @configLst == 0 );
      my %configTab = ();
      # Look for last config element and perform some checks
      foreach $config ( @configLst ) {
	 my $next = $config->getNext;
	 $next = "LAST" unless ( defined $next );
	 unless ( defined $configTab{$next} ) {
            my $configName = $config->getName;
            &sl_traces::error( "The config element < %s > cannot ".
	                       "reference itself",
	                       "- sl_config::createOrderedConfigLst -".
			       __LINE__, ( $configName ) )
	       if ( $configName eq $next );
            $configTab{$next} = $config;
	 }
	 else {
            my $elt1 = $configTab{$next}->getName;
	    my $elt2 = $config->getName;
            &sl_traces::error( "The config element < %s > is referenced ".
	                       "more than one time:".
			       "\nConfig element : %s".
			       "\nConfig element : %s",
	                       "- sl_config::createOrderedConfigLst -".
			       __LINE__, ( $next, $elt1, $elt2 ) );
	 }
      }
      &sl_traces::error( "Infinite config element chain",
	        	 "- sl_config::createOrderedConfigLst -".__LINE__ )
	 unless ( defined $configTab{'LAST'} );
      # Classify the config elements
      my @altConfigLst = ();
      my $currentConfig = $configTab{'LAST'}->getName;
      push( @altConfigLst, $configTab{'LAST'} );
      if ( scalar @configLst > 1 ) {
	 my $maxCount = 1;
	 my $count = scalar @configLst;
	 while ( $count > 0 ) {
	    $maxCount *= $count;
	    $count--;
	 }
	 $maxCount += 1;
	 $count = 0;
	 while ( scalar @configLst > 0 ) {
	    $count++;
	    my $config = shift @configLst;
	    my $nextConfig = $config->getNext;
	    &sl_traces::error( "Circular dependency or broken chain has been ".
	                       "detected.\nCheck config file types and config ".
			       "element dependencies.".
			       "\nLast config element found : %s",
			    "- sl_config::createOrderedConfigLst -".
			    __LINE__, ( $currentConfig ) )
              if ( $count >= $maxCount );
	    # Look for config element element that reference the current one
	    if ( defined $nextConfig ) {
               if ( $nextConfig eq $currentConfig ) {
		  my $configName = $config->getName;
        	  push( @altConfigLst, $config );
		  $currentConfig = $configName;
	       }
	       else {
        	  push( @configLst, $config );
	       }
	    }
	 }
      }
      # Create ordered config install scripts 
      while ( scalar @altConfigLst > 0 ) {
	 my $config = pop @altConfigLst;
	 my $configName = $config->getName;
	 $config->_install( $destDir, "${remDir}/${CONFIG_DIR}/${configName}" );
      }
   }
}

# ------------------------------------------------------------------------------
#
sub getFileLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_config::getFileLst - Getting FileLst attribute ".
                      "of < $self->{'Name'} >" );
   if ( defined $self->{'FileLst'} ) {
      return @{$self->{'FileLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub setFileLst {
   my $self = shift;
   my @fileLst = @_;
   if ( defined $fileLst[0] ) {
      &sl_traces::trace( 2, "- sl_config::setFileLst - Setting file(s) to the ".
                         "config < $self->{'Name'} >" );
      @{$self->{'FileLst'}} = @fileLst;
   }
   else {
      &sl_traces::error( "%s is undefined", 
                         "- sl_config::setFileLst -".__LINE__,
			 ( "File(s) list" ) );
   }
}

# ------------------------------------------------------------------------------
#
sub getNext {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_config::getNext - Getting Next ".
                      "attribute of < $self->{'Name'} >" );
   if ( defined $self->{'Next'} ) {
      return $self->{'Next'};
   }
   else {
      return undef;
   }
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_config::getName - Getting Name attribute of ".
                "< $self->{'Name'} >" );
   if ( defined $self->{'Name'} ) {
      return $self->{'Name'};
   }
   else {
      return undef;
   }
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   my $configId = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_config::iD - config iD is < ${configId} >" );
   return $configId;
}

# ------------------------------------------------------------------------------

return 1;
