#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)fl_jumpstart.pl 1.11     02/10/30 SMI"
# ------------------------------------------------------------------------------

my @DISKLESS_DIR_LIST = ( "bin","etc","lib","sbin","var","dev","home",
                          "platform","xfn", "kernel","net", "devices", "opt" );	    		    	    

my @EXCLUDED_FILES_LIST = ( "etc/hostname.*" );

#-------------------------------------------------------------------------------
#
# Configure jumpstart environment for prototype 
#
#-------------------------------------------------------------------------------
sub jumpstart_configure {
   my ( $iplink ) = @_;
   my $server_ip = $iplink->{'server_ip'};
   my $proto_ip  = $iplink->{'node_ip'};
   my $server_name = $iplink->{'server'};
   my $proto_name  = $iplink->{'prototype'};
   my $proto_netmask = $iplink->{'netmask'};
   my $proto_ether = $iplink->{'node_ether'};
   # Control and modify /etc/hosts on installation host
   &add_host( $proto_ip, $proto_name  );
   &add_host( $server_ip, $server_name );   
   # Control end modify /etc/ethers on installation host
   if ( defined( $proto_ether ) ) {
      &add_ether ( $proto_ether, $proto_name );
   } else {
      $proto_ether = &getetherbyname ( $proto_name );
      &error (
         "No Ether address found for host \"%s\" in \"/etc/ethers\"",
	 ( $proto_name ) )
         unless ($proto_ether);
      &error (
         "Duplicate Ethernet address \"%s\" in \"/etc/ethers\"",
	 ( $proto_ether ) ) 
	 if ( &duplicated_ether( $proto_ether ) );
   }
}

#-------------------------------------------------------------------------------
#
# Create jumpstart environment for prototype 
#
#-------------------------------------------------------------------------------
sub jumpstart_create {
   my ( $nodeId, $jumpstart, $solaris, $slbt, 
        $iplink, $flash, $group ) = @_;
  
   # Table holding data that configure the Jumpstart files templates
   my %dict = ();
   
   # Networking data between install-server and prototype machine
   my $server_ip = $iplink->{'server_ip'};
   my $proto_ip  = $iplink->{'node_ip'};
   my $server_name = $iplink->{'server'};
   my $proto_name  = $iplink->{'prototype'};
   my $proto_netmask = $iplink->{'netmask'};
   my $proto_ether = $iplink->{'node_ether'};
   my $jumpstart_dir = "$jumpstart/$server_name/$proto_name";
   my $groupName = $group->getName;
   my $nodeIdLst = $group->getNodeLst;
   
   my $jumpstart_config_dir = ( $INSTAL ) ?
      "$slbt->{'CLUSTER_DIR'}/${groupName}/jumpstart/install":
      "$slbt->{'CLUSTER_DIR'}/${groupName}/jumpstart/deploy";
   
   $dict{'SERVER_IP'} = $server_ip;
   $dict{'PROTO_NAME'} = $proto_name;
   $dict{'PROTO_IP'} = $proto_ip;
   $dict{'NETMASK'} = $proto_netmask;

   # Files systems to install in the diskless root file system
   $dict{'DISKLESS_SUBTREE'} = "\"@DISKLESS_DIR_LIST\"";

   # File to remove from the diskless root file system
   $dict{'EXCLUDED_FILES_LIST'} = "\"@EXCLUDED_FILES_LIST\"";

   $dict{'FLASH_PATH'} = "";  
   if ( defined( $flash ) ) {
      $dict{'FLASH_PATH'} = $flash->{'PATH'};
   }

   $dict{'NODE_ID_LST'} = $nodeIdLst;

   $dict{'NODE_ID'} = "";  
   if ( defined( $nodeId ) ) {
      $dict{'NODE_ID'} = $nodeId;
   }
   
   # Remote NFS parameters used during Jumpstart finish-install script
   $dict{'SOFTREPO'} = $slbt->{'softrepo'};
   $dict{'EXPORT'} = $slbt->{'path'};
   $dict{'SOLARIS'} = $solaris->getDist;
   $dict{'SOLARIS_EA'} = $solaris->getEADist;

   # SWL id
   $dict{'SWL_ID'} = $slbt->{'swloadid'};

   # System ident
   $dict{'SYSTEM_IDENT'} = fl_flash->getSignature( $slbt, $group, $iplink );

   # Test the  jumpstart directory tree, create it if it does not exist
   &control_dir_rxw_or_create ( "$jumpstart_dir" );

   # Create Jumpstart config files from templates
   &create_sysidcfg ( $jumpstart_config_dir, $jumpstart_dir, %dict );
   &create_begin ( $jumpstart_config_dir, $jumpstart_dir, %dict );
   &create_finish ( $group, $slbt, $jumpstart_dir,
                    $jumpstart_config_dir, $solaris->{'SOLARIS_MEDIA'}, %dict );
   &create_profile ( $jumpstart_config_dir, $jumpstart_dir, %dict );
   &create_and_test_rules ( $jumpstart_config_dir, $solaris,
                            $jumpstart_dir, %dict );   
   
   # Purge pre-existant boot client, ignore errors silently
   &exec_cmd( "$solaris->{'rm_install_client'} $proto_name > /dev/null 2>&1" );
   
   # Kill arpd so that it will be restarted by add_install_client 
   # with the new conf
   $cmd = "/usr/bin/pkill in.rarpd";
   &exec_cmd( $cmd ); # do not test error code => arp may be already down 
   
   # Add boot client
   my $platform = $group->getClass;
   $cmd = "$solaris->{'add_install_client'}";
   $cmd .= " -i \"$proto_ip\"";
   $cmd .= " -e \"$proto_ether\"";
   $cmd .= " -s \"$server_ip:$solaris->{'SOLARIS_MEDIA'}\"";
   $cmd .= " -c \"$server_ip:$jumpstart_dir\"";
   $cmd .= " -p \"$server_ip:$jumpstart_dir\"";
   $cmd .= " -n :none";
   $cmd .= " $proto_name $platform";   
   &exec_cmd( $cmd ) || &error( "Execution of \"$cmd\" failed" ); 
   
   # Remplace install server name by server IP address in /etc/bootparams
   $cmd="/usr/bin/perl -p -i.bak";
   $cmd .= " -e 'if ( /^$proto_name/ ) {";
   $cmd .= " s/^$proto_name\\s+[^:]+:/$proto_name root=$server_ip:/;";
   $cmd .= " }'";
   $cmd .= " /etc/bootparams";
   &exec_cmd( $cmd ) || &error( "Can't modify \"/etc/bootparams\" file" );
      
   # Tell user that he can perform a jumpstart
   &info( 1, "You can now proceed to the installation of \"%s\" ".
             "by booting using: \"boot net - install\"", ( $proto_name ) );
}

#-------------------------------------------------------------------------------
#
# Create Sysidcfg file
#
#-------------------------------------------------------------------------------
sub create_sysidcfg {
   my ( $jumpstart_config_dir, $jumpstart_dir, %dict ) = @_;
   &info( 2, "Creating %s file", ( "sysidcfg" ) );
   &replace_in_file ( 
      "$jumpstart_config_dir/sysidcfg", "$jumpstart_dir/sysidcfg", %dict );       
}

#-------------------------------------------------------------------------------
#
# Create begin file
#
#-------------------------------------------------------------------------------
sub create_begin {
   my ($jumpstart_config_dir, $jumpstart_dir, %dict) = @_;
   &info( 2, "Creating %s file", ( "begin" ) );
   &replace_in_file ( 
      "$jumpstart_config_dir/begin", "$jumpstart_dir/begin", %dict);       
}

#-------------------------------------------------------------------------------
#
# Create jumpstart profile file
#
#-------------------------------------------------------------------------------
sub create_profile {
   my ( $jumpstart_config_dir, $jumpstart_dir, %dict) = @_;      
   my $profile = "$jumpstart_config_dir/profile";
   &info( 2, "Creating %s file", ( "profile" ) );
   &replace_in_file ( 
      "$jumpstart_config_dir/profile", "$jumpstart_dir/profile", %dict );
}

#------------------------------------------------------------------------------- 
#
# Create jumpstart rules file
#
#-------------------------------------------------------------------------------
sub create_and_test_rules {
   my ( $jumpstart_config_dir, $solaris, $jumpstart_dir, %dict ) = @_;
   &info( 2, "Creating %s file", ( "rules" ) );
   &replace_in_file ( 
      "$jumpstart_config_dir/rules", "$jumpstart_dir/rules", %dict );   
   open ( CHECK, "cd $jumpstart_dir; $solaris->{'check'} |" ) || 
   &error ( "Unable to check \"rules\" file" ); 
   while ( <CHECK> ) {
      chop();
      &trace ( 1, $_ );
   }
   close ( CHECK ) || &error( "Error found while checking \"rules\" file" );
}

#-------------------------------------------------------------------------------
#
# Create jumpstart finish.sh file
#
#-------------------------------------------------------------------------------
sub create_finish {
   my ( $group, $slbt, $jumpstart_dir, 
        $jumpstart_config_dir, $solaris_dir, %dict ) = @_;
   my $groupArch = $group->getArch;
   my $groupOs = $group->getOs;
   my $groupName = $group->getName;
   &info( 2, "Creating %s file", ( "finish" ) );
   my $clusterDir = `/usr/bin/basename $slbt->{'CLUSTER_DIR'}`;
   chomp( $clusterDir );

   # Exported file system root (exists only if embedded groups exist for
   # a ME group)
   $dict{'EXPORT_ROOT'} = "\"\"";
   $dict{'SHARED_SW_ROOT'} = "\"\"";
   $dict{'EMBEDDED_GROUP_LIST'}= "\"\"";

   # Identify the group under installation
   $dict{'MAIN_GROUP'} = "${clusterDir}/$groupName";
   $dict{'MAIN_GROUP_TYPE'} = $group->getType;
   if ( $group->isME ) {
      # Get exported file system root (exists only if embedded groups exist
      # for the ME group)
      $dict{'EXPORT_ROOT'} = $group->getExportRoot
        if ( defined  $group->getExportRoot );
      # Get the list of groups to install
      @groupLst = $group->getGroupLst;
      $dict{'EMBEDDED_GROUP_LIST'} = "";
      foreach $group ( @groupLst ) {
	 $dict{'EMBEDDED_GROUP_LIST'} = 
            $dict{'EMBEDDED_GROUP_LIST'}." ${clusterDir}/$group"; 
      }
      $dict{'EMBEDDED_GROUP_LIST'}="\"$dict{'EMBEDDED_GROUP_LIST'}\"";
      # Get the root of the shared sw database
      $dict{'SHARED_SW_ROOT'} = $group->getSharedSwRoot;
   }

   # Generate finish.sh script from template
   &replace_in_file ( 
      "$jumpstart_config_dir/finish", "$jumpstart_dir/finish", %dict );    
}

return 1;
