#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)fl_flash.pm 1.9     02/10/31 SMI"
# ------------------------------------------------------------------------------


BEGIN {
  require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/fl_filesys.pl";
  require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/fl_traces.pl";
}


# ------------------------------------------------------------------------------

package fl_flash;

# Flash User keywords
$X_SWL_ID = "X-NHAS-SWL-ID";
$X_SWL_NAME = "X-NHAS-SWL-NAME";
$X_NODES_GROUP_NAME = "X-NHAS-NODE-GROUP-NAME";
$X_FLASH_FORMAT = "X-NHAS-FLASH-FORMAT";
$X_SMCT_VERSION = "X-NHAS-SMCT-VERSION";
$X_SOE_RELEASE = "X-NHAS-SOE-RELEASE";

# Version of the current Flash archive format used by the SMCT
$FLASH_FORMAT = "1.0";

# Parameters for remote accessibility
my $PING_RETRY = 5;

# System ident parameters
my $IDENT_SEP = "\x01";
my $IDENT_FILE = "ident.nhsmct";  
my $REM_IDENT_FILE = "/var/opt/SUNWcgha/nhsmct/${IDENT_FILE}";  
   
# Remote commands (on the prototype machine)  
my @REMOTE_BINARIES=();
my $RMOUNT = $ENV{'SMCT_MOUNT'}; @REMOTE_BINARIES=(@REMOTE_BINARIES, $RMOUNT);
my $RUMOUNT = $ENV{'SMCT_UMOUNT'}; @REMOTE_BINARIES=(@REMOTE_BINARIES,$RUMOUNT);
my $RFLAR= $ENV{'SMCT_FLARCREATE'}; @REMOTE_BINARIES=(@REMOTE_BINARIES,$FLARCREATE);

# Working directory for temporary files
my $TMP_DIR = $ENV{'SMCT_TMP_DIR'};

# Directory to exclude from the group section
@EXCLUDE_DIR = ( "jumpstart" );

# ------------------------------------------------------------------------------

sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;

   $self->{'PATH'}  = $attr{'PATH'};
   $self->{'SOLARIS_DIST'}  = $attr{'SOLARIS_DIST'};
   
   &main::control_file_is_readable( $self->{'PATH'} );
 
   my $flarCmd = $self->{'SOLARIS_DIST'}->{'flar'};
   open (FLASHINFO, "${flarCmd} -i $self->{'PATH'} |" ) ||
      &main::error("Unable to retrieve flash archive info");
   while ( ( $key, @valueLst ) = split /=/,<FLASHINFO> ) {
     my $value = "";
     my $nbFields = scalar @valueLst;
     my $i = 0;
     foreach $val ( @valueLst ) {
        $i++;
        chomp( $val );
	$value .= $val;
	$value .= "=" unless ( $i == $nbFields );
     }
     $self->{ $key} = $value;     
   }
   # Initialize w/ default value if header is uncomplete
   $self->{$X_SWL_ID} = "<undefined>" 
      unless ( defined $self->{$X_SWL_ID} );
   $self->{$X_SWL_NAME} = "<undefined>" 
	 unless ( defined $self->{$X_SWL_NAME} );
   $self->{$X_NODES_GROUP_NAME} = "<undefined>" 
	 unless ( defined $self->{$X_NODES_GROUP_NAME} );
   $self->{$X_FLASH_FORMAT} = "<undefined>" 
	 unless ( defined $self->{$X_FLASH_FORMAT} );
   $self->{$X_SMCT_VERSION} = "<undefined>" 
	 unless ( defined $self->{$X_SMCT_VERSION} );
   $self->{$X_SOE_RELEASE} = "<undefined>" 
	 unless ( defined $self->{$X_SOE_RELEASE} );
   close(FLASHINFO);
   &main::trace( 2, "- fl_flash::new - Creating flash object ".
                 "< $self->{'PATH'} >" );    
   return $self;
}

#-------------------------------------------------------------------------------
#
# Add a section to an existing Flash archive
#
#-------------------------------------------------------------------------------
sub addSections {
   my $self = shift;
   my ( @sectionLst ) = @_;
   my $sectionNameOpt = "";
   my $sectionDir = $ENV{'SMCT_TMP_DIR'};
   my $splitSectionDir = "${sectionDir}/split";
   my $flashArchive = $self->{'PATH'};
   my $flashFile = `${main::BASENAME} ${flashArchive}`; chop( $flashFile );
   my $tmpFlashArchive = "${sectionDir}/${flashFile}";
   my $flashDir = `${main::DIRNAME} ${flashArchive}`; chop( $flashDir );
   my $solaris = $self->{'SOLARIS_DIST'};
   my $flarCmd = $solaris->{'flar'};
   
   &main::trace( 2, "- fl_flash::addSection - Adding section(s) to Flash ".
                 "archive < $self->{'PATH'} >" );

   &main::info( 1, "Adding sections to the Flash archive ... ".
                "(this operation takes some time; please, wait)" );

   foreach $section ( @sectionLst ) {
      $sectionName = `${main::BASENAME} ${section}`;
      chop( $sectionName );
      $sectionNameOpt .= " -u ".$sectionName;
   }
   
   &main::trace( 2, "- fl_flash::addSection - Splitting Flash ".
                 "archive < ${flashArchive} > into < ${splitSectionDir} >" );
   # split the archive and keep identification/archive/cookie sections
   # (the others will be either created or replaced by new ones)
   my $cmd = "${main::MKDIR} -p ${splitSectionDir};".
             "${flarCmd} split -d ${splitSectionDir} ${flashArchive};".
	     "${main::MV} ${splitSectionDir}/identification ${sectionDir};".
	     "${main::MV} ${splitSectionDir}/archive ${sectionDir};". 
	     "${main::MV} ${splitSectionDir}/cookie ${sectionDir};"; 
   &main::exec_cmd( "${cmd}" ) ||
      &main::error( "Cannot split Flash archive %s", ( $flashArchive ) );
   
   &main::trace( 2, "- fl_flash::addSection - Combining Flash ".
                 "archive sections into < ${tmpFlashArchive} >" );
   $cmd = "${flarCmd} combine -d ${sectionDir} ".
          "${sectionNameOpt} ${tmpFlashArchive}";
   &main::exec_cmd( "${cmd}" ) ||
      &main::error( "Cannot add sections %s to %s", 
                    ( $sectionNameOpt, $tmpFlashArchive ) );

   # Replace the old Flash archive by the new one
   $cmd = "${main::MV} ${tmpFlashArchive} ${flashArchive}";
   &main::exec_cmd( "${cmd}" ) ||
      &main::error( "Cannot move %s to %s",
                    ( $tmpFlashArchive, $flashArchive ) );
}
   
#-------------------------------------------------------------------------------
#
# Prepare a Flash archive section from a file
#
#-------------------------------------------------------------------------------
sub prepareSectionFile {
   my $class = shift;
   my ( $sectionFile, $outputDir, $remote ) = @_;
   my $section = `${main::BASENAME} ${sectionFile}`; chop( $section );
   my $compressedFile = "${TMP_DIR}/${section}.gz";
   &main::trace( 2, "- fl_flash::prepareSectionFile - Preparing flash section ".
                 "< $sectionFile >" );
   
   $section =~ /(.+)\.(.+)/;
   my $sectionName = $1;
   &main::info( 1, "Creating Flash archive section \"%s\" ", ( $sectionName ) );
   my $base64File = "${outputDir}/${section}.gz.uu";

   if ( -e $sectionFile ) {
      # Compressed the section file using gzip
      if ( defined $remote ) {
	 my $cmd = "${main::CP} ${sectionFile} ${TMP_DIR}";
	 &main::exec_cmd( $cmd ) ||
	    &main::error( "Cannot %s file %s using:\n%s",
	                  ( "copy", $sectionFile, $cmd ) );      
      }
      $cmd =  "${main::GZIP} --best ${TMP_DIR}/${section}";
      &main::exec_cmd( $cmd ) ||
	 &main::error( "Cannot %s file %s using:\n%s",
	               ( "compress", $sectionFile, $cmd ) );   
      # Encode the compressed file using base64 coding algorythm
      $cmd = "${main::UUENCODE} ${compressedFile} ${section}.gz > ".
             "${base64File}";
      &main::exec_cmd( $cmd ) ||
	 &main::error( "Cannot %s file %s using:\n%s",
	               ( "encode", $compressedFile, $cmd ) );   
   }
   else {
      # Create a fake file to avoid useless traces at install time
      $cmd = "/usr/bin/touch ${base64File}";
      &main::exec_cmd( $cmd ) ||
	 &main::error( "Cannot %s file %s using:\n%s",
	               ( "create", $base64File, $cmd ) );   
   }
   return $base64File;
}

#-------------------------------------------------------------------------------
#
# Prepare a Flash archive section from a files tree
#
#-------------------------------------------------------------------------------
sub prepareSectionDir {
   my $class = shift;
   my ( $section, $outputDir, $sectionFile, @excludeDirLst ) = @_;
   my $archiveFile = "${outputDir}/${sectionFile}.tar";
   &main::trace( 2, "- fl_flash::prepareSectionDir - Preparing flash section ".
                 "< $section >" );
   my $cmd = "";
   my $excludeFile = "$ENV{'SMCT_TMP_DIR'}/exclude.txt";
   my $sectionName = `${main::BASENAME} ${section}`; chop( $sectionName );
   my $sectionDir = `${main::DIRNAME} ${section}`; chop( $sectionDir );
   if ( defined $excludeDirLst[0] ) {
      foreach $dir ( @excludeDirLst ) {
         $cmd .= "echo ${sectionName}/${dir} >> ${excludeFile};";
      }
      &main::exec_cmd( $cmd ) ||
	 &main::error( "Cannot %s file %s using:\n$cmd",
	               ( "create tar exclude", $excludeFile, $cmd ) ); 	 
   }
   # Create a file archive for the directory given as parameter
   if ( -e $section ) {
      $cmd = ( defined $excludeDirLst[0] ) ?
	 "cd ${sectionDir}; ".
	 "${main::TAR} cfX ${archiveFile} ${excludeFile} ${sectionName}":
	 "cd ${sectionDir}; ".
	 "${main::TAR} cf ${archiveFile} ${sectionName}";
   &main::exec_cmd( $cmd ) ||
      &main::error( "Cannot %s file %s using:\n%s",
	            ( "create archive", $archiveFile, $cmd ) ); 
   }
   # Compress and encode the archive
   return $class->prepareSectionFile( $archiveFile, $outputDir );  
}
   
#-------------------------------------------------------------------------------
#
# Create a flash archive directly on the prototype machine
#
#-------------------------------------------------------------------------------
sub create_remote_flash_archive {
   my $class = shift;
   my ( $flashArchive, $slbt, $iplink, $group, $solaris ) = @_;

   my $flarCmd = "unset ENV; ${main::KSH} +o noclobber ${RFLAR} ";
   my $flashFile = `${main::BASENAME} ${flashArchive}`; chop( $flashFile );
   my $workingDir = $ENV{'SMCT_TMP_DIR'};
   my $tmpFlashArchive = "${workingDir}/${flashFile}";
   my $remFlashDir = "/mnt/flash";
   my $proto_ip  = $iplink->{'node_ip'};
   my $server_ip = $iplink->{'server_ip'};
   my $soeRelease = $solaris->getRelease;
   
   #       
   # Prepare SWL descriptor section parameters
   #
   my $section = $slbt->createSections;
   my $flarSection = `${main::BASENAME} ${section}`; chop( $flarSection );

   #
   # Test if proto is rechable
   #	   
   &main::exec_cmd( "${main::PING} ${proto_ip} ${PING_RETRY} > /dev/null" ) ||
      &main::error( "Cannot ping prototype machine %s thru interface", 
                    ( $proto_ip ) );
   &main::remote_exec( ${proto_ip},"${main::CAT} ${REM_IDENT_FILE} " );

   #
   # Share (host) and mount (proto) Flash repository
   #
   &main::remote_exec ( ${proto_ip}, "${RMOUNT} -o retry=0 ".
                        "${server_ip}:${workingDir} ${remFlashDir}" ) || 
      &main::error( 
         "Unable to NFS mount \"%s\" from master system".
         "\n Check that the /mnt directory is not already used as mount point ".
	 "on the master-system".
         "\n(execute #/usr/sbin/umount /mnt on the master-system)",
	 ( $workingDir ) ); 

   # Copy system ident file
   &main::remote_exec( 
      ${proto_ip},"${main::CP} ${REM_IDENT_FILE} ${remFlashDir}" );
   # Extract signature from file
   my $remSignature = `${main::CAT} $ENV{'SMCT_TMP_DIR'}/${IDENT_FILE}`;
   chomp( $remSignature );
   $remSignature =~ s/${IDENT_SEP}/ /g;
   # Get current signature
   my $signature = $class->getSignature( $slbt, $group, $iplink );
   $signature =~ s/${IDENT_SEP}/ /g;
   # Check that signature are equivalent or raise an error
   unless ( $remSignature eq $signature ) {
      &main::remote_exec ( $proto_ip, "$RUMOUNT ${remFlashDir}" ) ||
	 &main::error ( "Unable to NFS umount \"%s\" from master system", 
			( "${server_ip}:${workingDir}" ) );
      &main::error(
	 "The system you want to Flash does not correspond to the parameters ".
	 "you gave".
	 "\nRemote system expected : %s".
	 "\nActual remote system   : %s",
	 ( $signature, $remSignature ) );
   }
   
   #
   # Create Flash archive 
   #
   &main::info( 1, "Launching Flash archive creation ... ".
                "(this operation takes some time; please, wait)" );
   my $swlId = $slbt->{'swloadid'};
   my $swlName = $slbt->{'NAME'};
   my $groupName = $group->getName;
   my $flashName = "FLASH-${swlName}-${swlId}-${groupName}";
   my $cmd =
      "${flarCmd} -c -n ${flashName} -d ${remFlashDir} -u ${flarSection} ".
      "-U \"${X_SWL_ID}=${swlId}\" ".
      "-U \"${X_SWL_NAME}=${swlName}\" ".
      "-U \"${X_NODES_GROUP_NAME}=${groupName}\" ".
      "-U \"${X_SMCT_VERSION}=$ENV{'SMCT_VER'}\" ".
      "-U \"${X_FLASH_FORMAT}=${FLASH_FORMAT}\" ".
      "-U \"${X_SOE_RELEASE}=${soeRelease}\" ".
      "${remFlashDir}/${flashFile}"; 

   &main::remote_exec ( $proto_ip, $cmd ) || 
      &main::error ( "Creation of Flash archive failed on prototype machine" );

   #
   # umount and unshare Flash repository 
   #
   &main::remote_exec ( $proto_ip, "$RUMOUNT ${remFlashDir}" ) ||
      &main::error ( "Unable to NFS umount \"%s\" from master system ",
		     ( "${server_ip}:${workingDir}" ) );

   # Move the Flash archive to the destination directory
   $cmd = "${main::MV} ${tmpFlashArchive} ${flashArchive}";
   &main::exec_cmd( "${cmd}" ) ||
      &main::error( "Cannot move %s to %s",
                    ( $tmpFlashArchive, $flashArchive ) );
}

# ------------------------------------------------------------------------------

sub getSignature {
   my $class = shift;
   my ( $slbt, $group, $iplink,  ) = @_;
   my $signature = $slbt->getSignature.${IDENT_SEP};
   $signature .= $group->getSignature.${IDENT_SEP};
   $signature .= $iplink->{'node_ether'};
   return $signature;
}

# ------------------------------------------------------------------------------

sub compare {
   my $self = shift;
   my ( $slbt, $group, $solaris ) = @_;
   my $diff = "";
   my $groupName = $group->getName;
   my $swlId = $slbt->{'swloadid'};
   my $swlName = $slbt->{'NAME'};
   my $soeRelease = $solaris->getRelease;
   $diff .= "\t${X_SWL_ID}".
            "\n\t\tCurrent value  : $self->{$X_SWL_ID} ".
            "\n\t\tExpected value : ${swlId}\n"
      unless ( $self->{$X_SWL_ID} eq ${swlId} );
   $diff .= "\t${X_SWL_NAME}".
            "\n\t\tCurrent value  : $self->{$X_SWL_NAME} ".
            "\n\t\tExpected value : ${swlName}\n"
      unless ( $self->{$X_SWL_NAME} eq ${swlName} );
   $diff .= "\t${X_NODES_GROUP_NAME}".
            "\n\t\tCurrent value  : $self->{$X_NODES_GROUP_NAME} ".
            "\n\t\tExpected value : ${groupName}\n"
      unless ( $self->{$X_NODES_GROUP_NAME} eq ${groupName} );
   $diff .= "\t${X_FLASH_FORMAT}".
            "\n\t\tCurrent value  : $self->{$X_FLASH_FORMAT} ".
            "\n\t\tExpected value : ${FLASH_FORMAT}\n"
      unless ( $self->{$X_FLASH_FORMAT} eq ${FLASH_FORMAT} );
   $diff .= "\t${X_SMCT_VERSION}".
            "\n\t\tCurrent value  : $self->{$X_SMCT_VERSION} ".
            "\n\t\tExpected value : $ENV{'SMCT_VER'}\n"
      unless ( $self->{$X_SMCT_VERSION} eq $ENV{'SMCT_VER'} ); 
   $diff .= "\t${X_SOE_RELEASE}".
            "\n\t\tCurrent value  : $self->{$X_SOE_RELEASE} ".
            "\n\t\tExpected value : ${soeRelease}\n"
      unless ( $self->{$X_SOE_RELEASE} eq $soeRelease ); 
      
   ( $diff = undef ) if ( $diff eq "" );
   return $diff;
}

# ------------------------------------------------------------------------------

return 1;   
