#!/usr/bin/perl
# ------------------------------------------------------------------------------
#  ident "@(#)common.pl 1.32 03/04/18 SMI"
# ------------------------------------------------------------------------------
#
#  Copyright 2001-2003 Sun Microsystems, Inc. All rights reserved.
#
# ------------------------------------------------------------------------------

package COMMON;

use variables;

require "$ENV{NHINSTALL_LIB}/tools/lib/conf.pl";

#-----------------------------------------------------------
#
#  init
#
#-----------------------------------------------------------

sub init {

  if (defined($ENV{NH_LOG_FILE}) && ($ENV{NH_LOG_FILE} ne "")) {
    $LOGFILE_NAME=$ENV{NH_LOG_FILE} ;
    $IN_LOGFILE = TRUE ;
  }
}

#-----------------------------------------------------------
#
#  date
#
#-----------------------------------------------------------

sub getdate {
  my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime;
  $date = sprintf("%02.02d:%02.02d:%02.02d", $hour, $min, $sec) ;
  return $date ;
}


#===============================================================================
#
#  OUTPUT MANAGEMENT FUNCTIONS
#
#===============================================================================

#-----------------------------------------------------------
#
#   general output
#
#-----------------------------------------------------------

# stdin

sub print_log {
  ($msg) = @_ ;

  print $LOG "$msg" ;
  if ($IN_LOGFILE eq TRUE) {
    # open, write and close to be sure all is ok even if there is a crash later
    if (!open(LOGFILE, ">>$LOGFILE_NAME")) {
      my $date = getdate() ;
      print("$date WARNING: Can't append log file $LOGFILE_NAME: logfile option disabled\n") ;
      $IN_LOGFILE = FALSE;
      return ;
    }
    print LOGFILE "$msg";
    close(LOGFILE);
  }
}

# stderr

sub print_err {
  ($msg) = @_ ;

  print $LOGERR "$msg" ;
  if ($IN_LOGFILE eq TRUE) {
    # open, write and close to be sure all is ok even if there is a crash later
    if (!open(LOGFILE, ">>$LOGFILE_NAME")) {
      my $date = getdate() ;
      print("$date WARNING: Can't append log file $LOGFILE_NAME: logfile option disabled\n") ;
      $IN_LOGFILE = FALSE;
      return ;
    }
    print LOGFILE "$msg";
    close(LOGFILE);
  }
}

#-----------------------------------------------------------
#
#  debug
#
#  1: name of debug variable to be tested
#  2: string to be displayed
#
#-----------------------------------------------------------

sub print_debug {
  my ($debug, $msg) = @_ ;

  if ($$debug ne "TRUE") {
    return ;
  }

  my $string = "  DEBUG: $msg\n";
  print_log($string);
}

#-----------------------------------------------------------
#
#  display stage description message
#
#-----------------------------------------------------------

sub print_stage {
  my ($comment) = @_ ;

  my $date = getdate() ;
  print_log("$date $comment\n") ;
}

#-----------------------------------------------------------
#
#  display stage description message with node reference
#
#-----------------------------------------------------------

sub print_stage_node {
  my ($comment) = @_ ;

  my $nodename = get_external_name($NODE_ID, "") ;
  my $date = getdate() ;
  print_log("$date $comment (on node $nodename)\n") ;
}

#-----------------------------------------------------------
#
#  display recovery description message
#
#-----------------------------------------------------------

sub print_recovery_node {
  my ($comment) = @_ ;

  my $nodename = get_external_name($NODE_ID, "") ;
  my $date = getdate() ;
  print_log("$date Recovery after failure: $comment (on node $nodename)\n") ;
}

#-----------------------------------------------------------
#
#  display action description message
#
#-----------------------------------------------------------

sub print_action {
  my ($comment) = @_ ;

  my $date = getdate() ;
  print_log("$date   - $comment\n") ;
}

#-----------------------------------------------------------
#
#  display sub-action description message
#
#-----------------------------------------------------------

sub print_subaction {
  my ($comment) = @_ ;

  my $date = getdate() ;
  print_log("$date    . $comment\n") ;
}

#-----------------------------------------------------------
#
# error displays an error message and exit
#
# 1: error message
#
#-----------------------------------------------------------

sub error {
   my ($errorinfo) = @_;

   my $date = getdate() ;
   print_err("$date Error: $errorinfo\n") ;
   exit 1;
}

#------------------------------------------------------------
#
# warning displays a message
#
#   1: message
#
#------------------------------------------------------------

sub warning {
   my ($warninginfo) = @_;

   my $date = getdate() ;
   print_log("$date Warning: $warninginfo\n") ;
}

#===============================================================================
#
#  DIRECTORY AND FILE MANAGEMENT
#
#===============================================================================

#------------------------------------------------------------
#
# Directory and file status
#
#------------------------------------------------------------

sub control_dir_is_readable {
   my ($directory) = @_;
   &error("Directory \"$directory\" does not exist or is not readable") 
   unless ( -d $directory && -x $directory  && -r $directory );   
}

sub control_dir_rxw_or_create {
   my ($directory) = @_;
   
   if ( -d $directory ) {
      &error("\"$directory\" directory exist but user don't have \"rwx\" rights on it") 
      unless ( -r $directory && -w $directory && -x $directory );
   } else {
      mkdir($directory,0777) || &error("Can not create \"$directory\" directory" );
   }
}

sub control_file_is_readable {
   my ($file) = @_;
   &error("File \"$file\" does not exist or is not readable") 
   unless ( -f $file && -r $file );
}

sub control_file_is_executable {
   my ($file) = @_;
   &error("File \"$file\" does not exist or is not executable") 
   unless ( -f $file && -x $file );
}


#===============================================================================
#
# COMMAND EXECUTION
#
#===============================================================================

#------------------------------------------------------------
#
#  prepare a command containing "`" by adding a "\" before
#
#  1: remote node
#  2: command
#
#------------------------------------------------------------

sub remote_exec_bis {
   my ($client, $command ) = @_ ;

   remote_exec($client, $command) ;
 }

#------------------------------------------------------------
#
#  execute a remote command and return the status result
#  if an error occurs, the output is displayed
#
#  1: remote node
#  2: command
#
#------------------------------------------------------------
sub remote_exec {
   my ($client, $command ) = @_ ;
   my $status = "OK";
   my $local_statusfile = "$WORKING_DIR/install-status.$$";
   my $remote_statusfile = "/tmp/install-status.$$";
   
   #
   # Initialize status on client side
   #
   open (LSTATUS,">$local_statusfile") || return 0;
   print LSTATUS "$status\n" ;
   close(LSTATUS);
      
   unless ( &exec_cmd( "$RCP $local_statusfile $client:$remote_statusfile" ) ) {  
        unlink($local_statusfile);
        &error ("Unable to initialize remote command status on client \"$client\" (remote node may be unreachable)");
   }
   
   unlink($local_statusfile);
      
   print_debug("DEBUG_CMD", "$RSH -n $client \"$command\"");

   #
   # Execute  remote command
   #   
   
   my $output="";
   open ( REMOTEXEC, " $RSH -n $client \"$command || echo \$? > $remote_statusfile \" 2>&1|" )
     || error("Can not fork command: $RSH -n $client \"$command\"");
   while (<REMOTEXEC>) {
      chop();
      print_debug("DEBUG_CMD", "  $_");

      # capture outputs to displayed if an error occurs
      $output="$output  $_\n" ;

   }
   close (REMOTEXEC) || error ("Failure when executing: $RSH -n $client \"$command\" 2>&1 (remote node may be unreachable)"); 
   
   #
   # Retrieve execution status on client side
   #   
   unless ( &exec_cmd( "$RCP $client:$remote_statusfile $local_statusfile" ) ) {
     unlink($local_statusfile);
     &error ("Unable to retrieve remote command status on client \"$client\"");
   }
   
   unless ( open (LSTATUS,$local_statusfile) ) {
      unlink($local_statusfile);
      return 0;
   }
   chop($status=<LSTATUS>);
   close(LSTATUS);
   unlink($local_statusfile);
   
   unless ( $status eq "OK" ) {
     print_log("---------------------------------------------------------\n") ;
     &warning ( "remote command \"$command\" returned status: \"$status\" on client \"$client\""
	      );
     print_log($output) ;
     print_log("---------------------------------------------------------\n") ;
     return 0;
   }
	
}

#------------------------------------------------------------
#
#  execute a remote command without displaying remote errors
#  (error only if command can't be issued)
#
#  1: remote node
#  2: command
#
#------------------------------------------------------------

sub remote_exec_ignore {
   my ($client, $command ) = @_ ;

   #
   # Execute  remote command
   #   
   
   print_debug("DEBUG_CMD", "$RSH -n $client \"$command\"");

   my $output="";
   open ( REMOTEXEC, " $RSH -n $client \"$command\" 2>&1|" )
     || error("Can not fork command: $RSH -n $client \"$command\" 2>$1") ;
   while (<REMOTEXEC>) {
      chop();
      print_debug("DEBUG_CMD", "  $_");
   }
   close (REMOTEXEC) || error ("Failure when executing: $RSH -n $client \"$command\" 2>&1 (remote node may be unreachable)");   
}

#------------------------------------------------------------
#
#  execute a local command and display the output:
#  . if there is an error
#  . if tracing option is set
#
#  1: command
#
#------------------------------------------------------------

sub exec_cmd {
    my ($command) = @_;

    print_debug("DEBUG_CMD", "$command");

    @result = qx/$command 2>&1/;
  
    # in case of error, display output and exit
    if ( $? ne 0 ) {
      $status = $? / 256 ;
      print_log("-----------------------------------------------------------------------------\n");
      print_log("Error when executing: $command (status=$status)\n") ;
      foreach $line (@result)
	{
	  print_log("  $line\n") ;
	}
      print_log("-----------------------------------------------------------------------------\n");
       exit 1;
    }

   # if it's OK, display if tracing is required

    foreach $line (@result)
      {
	print_debug("DEBUG_CMD", "  $line");
      }
  
    return 1;
}

#------------------------------------------------------------
#
#  execute a local command and ignore the error:
#  if tracing option is set, outputs are displayed
#  1: command
#
#------------------------------------------------------------

sub exec_cmd_ignore {
    my ($command) = @_;

    print_debug("DEBUG_CMD", "$command");
    @result = qx/$command 2>&1/;

   # if it's OK, display if tracing is required

    foreach $line (@result)
      {
	print_debug("DEBUG_CMD", "  $line");
      }
  
    return 1;
}

#-------------------------------------------------------------------------------
#
# remote_mount
#
# . dismount the mounting point (to prevent errors if it is already mounted)
# . mount the remote directory on the mounting point
#
# 1: node IP
# 2: exported directory to mount
# 3: mounting point
#
#-------------------------------------------------------------------------------

sub remote_mount
  {
    my ($node_ip, $exported, $mountpoint) = @_ ;

    remote_exec_ignore($node_ip, "$RUMOUNT $mountpoint") ;
    remote_exec ($node_ip, "$RMOUNT -o retry=0 -f nfs $SERVER_IP:$exported $mountpoint" ) || 
      error ( "Unable to NFS mount $exported" );
  }

#==============================================================================
#
#    SHARE
#
#==============================================================================

#------------------------------------------------------------------------------
#
#  share a directory on the installation server
#
#------------------------------------------------------------------------------

sub share {

  my ($directory) = @_;

  my @result;
  my @result_parent;
  my $i;
  my $found;
  my $command = "$SHARE -F nfs -o ro,anon=0 $directory";
  
  control_dir_is_readable($directory);
  
  
  print_debug("DEBUG_CMD", "$command");
  
  @result = qx/$command 2>&1/;
  
  # check if the error is because the parent directory is already shared
  # then check the share option of the parent (anon=0 must be set)
  if ( $? ne 0 ) {
    if ($result[0] =~ m/parent-directory \((.*)\) already shared/) {
      my $parent = $1;
      # retrieve the share information
      @result_parent = qx/$SHARE/;
      $found = 0;
      for ($i = 0 ; $i < scalar(@result_parent) ; $i++) {
	#look for the parent: 1 = directory name, 2 = option
	if ($result_parent[$i] 
	    =~ /^[^\s\t]*[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/) {
	  if ($1 eq $parent) {
	    if ($2 !~ m/anon=0/) {
	      error("Parent directory $parent must be shared with the \"anon=0\" option");
	      exit 1;
	    } else {
	      $found = 1;
	    }
	  }
	}
      
      }
      if (! $found) {
	error("Can't retrieve information about $parent");
      }
    } else {
      # real error
      $status = $? / 256 ;
      print_log("-----------------------------------------------------------------------------\n");
      print_log("Error when executing: $command (status=$status)\n") ;
      foreach $line (@result)
	{
	  print_log("  $line\n") ;
	}
      print_log("-----------------------------------------------------------------------------\n");
      exit 1;
    }
  }
  
  # if it's OK, display if tracing is required
  
  foreach $line (@result)
    {
      print_debug("DEBUG_CMD", "  $line");
    }
  
  return 1;
  
}

#===============================================================================
#
#    MOUNT/UMOUNT
#
#===============================================================================

#-------------------------------------------------------------------------------
#
# mount required mounting points after a men boot
# the current status of these mounting points is unknown (after a recovery)
# they are unmounted (without taking into account errors) and re-mounted
#
#-------------------------------------------------------------------------------
sub mount_men {

   my ($node_id) = @_;

   my $node_ip = get_external_ip($node_id);
   
    # ignore errors
   remote_exec_ignore($node_ip, "$RUMOUNT $SLICE_SUNWCGHA");

   if ($CHECK_MOUNT_ERROR ne "TRUE") {
     
     # this is the recovery
          
     remote_exec_ignore($node_ip, "$RMOUNT $SLICE_SUNWCGHA");
     
   } else {
     
     # initial case, errors are displayed
     
     remote_exec($node_ip, "$RMOUNT $SLICE_SUNWCGHA");
   }
 }

#-------------------------------------------------------------------------------
#
# mount required mounting points
# the current status of these mounting points is unknown (after a recovery)
# they are unmounted (without taking into account errors) and re-mounted
#
#-------------------------------------------------------------------------------
sub mount_nmen {

   my ($node_id) = @_;

   my $node_ip = get_external_ip($node_id);

   my $i;
   my $exec_fct;
   
   # all the replicated slice are unmounted (ignore errors)
   for $i (0..$#SLICE_LIST) {
     if ($SLICE_LIST[$i]{bitmap} ne "") {
       # ignore errors
	 remote_exec_ignore($node_ip, "$RUMOUNT $SLICE_LIST[$i]{mounting_point}") ;
     }
   }

   # reboot has been partial so mounting point are not present

   # ignore errors when called from recovery mechanism (they may be still
   # used so error "busy" may occur)
   if ($CHECK_MOUNT_ERROR ne "TRUE") {
     $exec_fct = "remote_exec_ignore";
   } else {
     $exec_fct = "remote_exec";
   }

   # if SVM is used, /export must be mounted on the physical because
   # of a smosservices bug (not needed when using DCSS)
   # so do the same for all the replicated slices as some of them can be
   # something like /export/root if the /export has been split

   for $i (0..$#SLICE_LIST) {
     if ($SLICE_LIST[$i]{bitmap} ne "") {
       # need to be mounted
       if (($USE_SVM eq "YES") and ($USE_DCSS eq "NO")) {

         my $devname = "/dev/dsk/$SLICE_LIST[$i]{name}";

         &$exec_fct( $node_ip, "$RMOUNT $devname $SLICE_LIST[$i]{mounting_point}") ||
           error ( "Unable to mount $devname on $SLICE_LIST[$i]{mounting_point}");

     } else {

	     &$exec_fct($node_ip, "$RMOUNT $SLICE_LIST[$i]{mounting_point}");
       }
     }
   }
}

#===============================================================================
#
#    REMOTE FILE HANDLING
#
#===============================================================================

#------------------------------------------------------------
#
# test if a remote file exists
#
#------------------------------------------------------------

sub exist_file {
    my ($node_ip, $remote_file) = @_;

    my $remote_backup_file = $remote_file . ".bak" ;

    $cmd = "if [ -f $remote_file ] ; then echo OK ; else echo KO ; fi";
    print_debug(DEBUG_CMD, "rsh -n $node_ip \"$cmd\"");

    chomp($result = qx/rsh -n $node_ip \"$cmd\"/);
    print_debug(DEBUG_CMD, "  $result");

    if ($result eq "OK") {
      return "TRUE";
    }
    return "FALSE";
}

#------------------------------------------------------------
#
# get a file from a remote host
#
#------------------------------------------------------------

sub get_file {
    my ($node_ip, $remote_file, $local_file) = @_;

    exec_cmd("$RCP $node_ip:$remote_file $local_file") ||
      error("Can not get $node_ip:$remote_file > $local_file");

    if ($DEBUG_FILE eq "TRUE") {
      print_debug(DEBUG_FILE, "---------------------------------------------");
      print_debug(DEBUG_FILE, "GET: CONTENT OF $node_ip:$remote_file");
      print_debug(DEBUG_FILE, "---------------------------------------------");
      system("cat $local_file");
      print_debug(DEBUG_FILE, "---------------------------------------------");
    }
}

#------------------------------------------------------------
#
# copy a file to a remote host
#
#------------------------------------------------------------

sub put_file {
    my ($node_ip, $local_file, $remote_file) = @_;

    if ($DEBUG_FILE eq "TRUE") {
      print_debug(DEBUG_FILE, "---------------------------------------------");
      print_debug(DEBUG_FILE, "PUT: CONTENT OF $node_ip:$remote_file");
      print_debug(DEBUG_FILE, "---------------------------------------------");
      system("cat $local_file");
      print_debug(DEBUG_FILE, "---------------------------------------------");
    }

    exec_cmd("$RCP $local_file $node_ip:$remote_file ") ||
      error("Can not put $local_file > $node_ip:$remote_file");
}

#------------------------------------------------------------
#
# transfer a file from a remote host to another remote host
#
#------------------------------------------------------------

sub transfer_file {
    my ($node_ip_src, $node_ip_dst, $remote_file) = @_;

    my $tmp_file    = "$WORKING_DIR/trans.tmp";

    exec_cmd("$RCP $node_ip_src:$remote_file $tmp_file") ||
       error("Unable to copy $node_ip_src:$remote_file to $tmp_file");
    exec_cmd("$RCP $tmp_file $node_ip_dst:$remote_file") ||
       error("Unable to copy $tmp_file to $node_ip_dst:$remote_file");
}

#------------------------------------------------------------
#
# create a backup file on the remote host
#
# store an indicator to be associated to memorize if the file
# exists
#
#------------------------------------------------------------

sub save_file {
    my ($node_ip, $remote_file, $tag) = @_;

    my $remote_backup_file = $remote_file . ".bak" ;
    $cmd = "if [ -f $remote_file ] ; then echo OK ; else echo KO ; fi";
    chomp($result = qx/rsh -n $node_ip \"$cmd\"/);

    if ($result eq "OK") {
      # file exists
      SEQUENCER::set_data($tag, "EXIST");
      remote_exec($node_ip, "$RCOPY $remote_file $remote_backup_file") ||
	error("Can not save $remote_file > $remote_backup_file on $node_ip");
      print_debug(DEBUG_FILE, "File $remote_file saved into $remote_backup_file on $node_ip");
    } else {
      SEQUENCER::set_data($tag, "NOT_EXIST");
      print_debug(DEBUG_FILE, "File $remote_file does not exist on $node_ip");
    }
}

#------------------------------------------------------------
#
# restore a file from its backup on the remote host
#
# CAUTION !!!
#
# if the file didn't exist when saving, the new file
# is deleted to restore the previous state
#
#------------------------------------------------------------

sub restore_file {
    my ($node_ip, $remote_file, $tag) = @_;

    my $remote_backup_file = $remote_file . ".bak" ;
    
    my $state = SEQUENCER::get_data($tag);
    if ($state eq "") {
      error("Restore_file: Unknown state for file $remote_file, tag $tag");
    }

    if ($state eq "EXIST") {
      remote_exec($node_ip, "$RCOPY $remote_backup_file $remote_file") ||
	error("Can not restore $remote_backup_file > $remote_file on $node_ip");
      print_debug(DEBUG_FILE, "File $remote_file restored from $remote_backup_file on $node_ip");
    } else {
      remote_exec($node_ip, "$RRM -f $remote_file") ||
      print_debug(DEBUG_FILE, "File $remote_file deleted when Restoring on $node_ip");
    }
}

#------------------------------------------------------------
#
# replacing a string by another in a file
#
#------------------------------------------------------------

sub replace_in_file {

  my ($local_file, $old_string, $new_string) = @_;
  my $tmp_file = "$WORKING_DIR/replace.tmp";

  open(LOCAL_FILE, $local_file) 
    || COMMON::error("Unable to open $local_file");
  
  open(TMP_FILE,">$tmp_file") ||
    COMMON::error("Unable to create $tmp_file");
   
   while ( <LOCAL_FILE> ) {	
     chop();
     s/$old_string/$new_string/g;
     print TMP_FILE "$_\n";
   }
   close(LOCAL_FILE);
   close(TMP_FILE);

  exec_cmd("$CP $tmp_file $local_file");
  unlink($tmp_file);
}

#===============================================================================
#
#    PACKAGES MANAGEMENT ROUTINES
#
#===============================================================================


#-------------------------------------------------------------------------------
#
#  clean up when a package installation fails
#
#  1: admin file
#  2: answers file
#  3: node ip address
#
#-------------------------------------------------------------------------------

sub exit_install_pkg {
  
  my ($admin_file, $answers_file, $node_ip, $package, $dir, $subdir) = @_;

  unlink($admin_file);
  if ($answers_file ne "") {
    unlink($answers_file) ;
  }
  remote_exec ( $node_ip, "$RUMOUNT /mnt" );
  error("Unable to install packages $package (from $dir/$subdir)" );
}

#-------------------------------------------------------------------------------
#
#  remove a package from a MEN (errors are ignored)
#
#  1: Node ID
#  2: Package list
#  3: installed on local (LOCAL) or shared (SHARED) directory
#     or USR_SPECIFIC (/usr for diskless)
#
#-------------------------------------------------------------------------------

sub remove_men_pkg {

  my ($node_id, $package_list, $option) = @_;

  my $node_ip = get_external_ip($node_id);
  my $admin_file  = "$WORKING_DIR/admin";
  
  #
  #  configure settings depending installation is shared or not
  #
  my $basedir = "" ;
  my $optcmd  = "" ;
  my $adj     = "" ;
  
  if ($option eq "LOCAL") {

    $basedir = "default" ;
    $optcmd  = "" ;
    $adj     = "local" ;

  } elsif ($option eq "SHARED") {

    $basedir = "/$OPT_SUBDIR" ;
    $optcmd  = "-R $SHARED_FS" ;
    $adj     = "shared" ;

  } elsif ( $option eq "USR_SPECIFIC") {

    $loc = $SERVICE_USR_PATH;
    $basedir = $SERVICE_USR_BASEDIR;
    $optcmd  = "-R $loc" ;
    $adj     = "usr-specific" ;

  } else {
    error("Invalid option ($option) for MEN package removing") ;
  }
  
  print_action("Removing $adj packages: $package_list ...") ;

  # 
  # Create a pkgadd admin file
  #
  open(ADMIN, ">$admin_file") ||
    &error("Unable to create $admin_file");

   print ADMIN <<"FEO";
	
mail=
instance=overwrite
partial=nocheck
runlevel=quit
idepend=quit
rdepend=nocheck
space=quit
setuid=nocheck
conflict=nocheck
action=nocheck
basedir=$basedir

FEO


  close(ADMIN);
  
  #
  # copy the admin file on the remote node
  #
  exec_cmd( "$RCP $admin_file $node_ip:/tmp/admin" ) ||
    &error ( "Unable to copy $admin_file on $node_ip:/tmp/admin") ;
  
  #
  # remove the packages in the reverse order
  #
  # do not display error
  my @list = split(" ",$package_list) ;
  for (my $i = (scalar(@list) - 1) ; $i >= 0 ; $i--) {
    # remove the ending part of package name like SUNWefcx.u
    # (the package is known as SUNWefcx after installation)
    my $pkg = $list[$i];
    my $pos = index($pkg, ".");
    if ($pos > -1) {
      $pkg = substr($pkg, 0, $pos);
    }
    remote_exec_ignore ( $node_ip, "echo y | $RPKGRM $optcmd -a /tmp/admin $pkg" ) ;
  }

  #
  # remove the admin file from the remote node
  #
  remote_exec ($node_ip, "$RM /tmp/admin" ) ;

  # remove the local adminfile
  unlink ($admin_file) ;
}

#-------------------------------------------------------------------------------
#
# Install a package on a MEN 
#
# 1: Node ID
# 2: Package list
# 3: installed on 
#    . LOCAL:         /opt/SUNWcgha                 (basedir=default)
#    . SHARED:        /SUNWcgha/export/services     (basedir=/ha_v1/opt)
#    . USR_SPECIFIC:  /export/Solaris_x             (basedir=/usr_sparc.all
# 4: answers file (null string if not required)
#
# if exists, the answers file is deleted before exiting !
#
#-------------------------------------------------------------------------------

sub install_men_pkg {

  my ($node_id, $root_dir, $package_subdir, $package_list, $option, $answers_file) = @_;

  my $node_ip     = get_external_ip($node_id);
  my $admin_file  = "$WORKING_DIR/admin";
  
  #
  #  configure settings
  #
  my $basedir = "default" ;
  my $dir     = "/opt/SUNWcgha" ;
  my $optcmd  = "" ;
  my $adj     = "" ;
  
  if ($option eq "LOCAL") {

    $basedir = "default" ;
    $dir     = "/opt/SUNWcgha" ;
    $optcmd  = "" ;
    $adj     = "local" ;

  } elsif ($option eq "SHARED") {

    $basedir = "/$OPT_SUBDIR" ;
    $dir     = "$SHARED_FS/$OPT_SUBDIR" ;
    $optcmd  = "-M -R $SHARED_FS" ;
    $adj     = "shared" ;

  } elsif ( $option eq "USR_SPECIFIC") {

    $loc = $SERVICE_USR_PATH;
    $basedir = $SERVICE_USR_BASEDIR;
    $dir     = "" ;
    $optcmd  = "-M -R $loc" ;
    $adj     = "usr-specific" ;

  } else {

    error("Invalid option ($option) for MEN package installation") ;
  }

  print_action("Installing $adj packages: $package_list ...") ;

  #
  # Test the product directory tree
  #
  &control_dir_is_readable ("$root_dir/$package_subdir");
  
  #
  # mount ROOT_DIR
  #   
  remote_mount($node_ip, $root_dir, "/mnt") ;
  
  # 
  # Create a pkgadd admin file
  #
  open(ADMIN, ">$admin_file") ||
    &error("Unable to create $admin_file");

   print ADMIN <<"FEO";
	
mail=
instance=overwrite
partial=nocheck
runlevel=quit
idepend=nocheck
rdepend=nocheck
space=quit
setuid=nocheck
conflict=nocheck
action=nocheck
basedir=$basedir

FEO


  close(ADMIN);
  
  #
  # copy the admin file on the remote node
  #
  exec_cmd( "$RCP $admin_file $node_ip:/tmp/admin" ) ||
    error ( "Unable to copy $admin_file on $node_ip:/tmp/admin") ;
  
  #
  # copy the answers file on the remote node
  #
  if ($answers_file ne "") {
    exec_cmd( "$RCP $answers_file $node_ip:/tmp/answers" ) ||
      error ( "Unable to copy $answers_file on $node_ip:/tmp/answers") ;
  }

  #
  # Remotely create the install directory in case it is needed by BASE_DIR
  #
  if ($dir ne "") {
    remote_exec ($node_ip, "$RMKDIR -p $dir" ) ||
      error("Unable to create $dir directory needed by BASE_DIR package installation setup");
  }

  #
  # Install packages remotely
  #
  if ($answers_file eq "") {
    &remote_exec ( "$node_ip", "$RPKGADD -n -a /tmp/admin -d /mnt/$package_subdir $optcmd $package_list" ) ||
      &exit_install_pkg($admin_file, $answers_file, $node_ip, $package_list, $root_dir, $package_subdir);
  } else {
    &remote_exec ( "$node_ip", "$RPKGADD -n -a /tmp/admin -r /tmp/answers -d /mnt/$package_subdir $optcmd $package_list" ) ||
      &exit_install_pkg($admin_file, $answers_file, $node_ip, $package_list, $root_dir, $package_subdir);
  }
  
  #
  # remove the admin file from the remote node
  #
  &remote_exec ( "$node_ip", "$RM /tmp/admin" ) ||
    &exit_install_pkg($admin_file, $answers_file, $node_ip, $package_list, $root_dir, $package_subdir);
  
  #
  # remove the answers file from the remote node
  #
  if ($answers_file ne "") {
    &remote_exec ( "$node_ip", "$RM /tmp/answers" ) ||
      &exit_install_pkg($admin_file, $answers_file, $node_ip, $package_list, $root_dir, $package_subdir);
  }

  #
  # remove previously created pkgadd admin and answers file
  #
  unlink($admin_file);
  if ($answers_file ne "") {
    unlink($answers_file);
  }

  #
  # umount ROOT_DIR
  #   
  &remote_exec ( "$node_ip", "$RUMOUNT /mnt" ) ||
    &error ( "Unable to NFS umount /mnt" );
}

#-------------------------------------------------------------------------------
#
# Remove packages for all NMENs on a MEN
#
# 1: Node ID
# 2: Package list
#
#-------------------------------------------------------------------------------

sub remove_nmen_pkg {

  my ($nodeid, $package_list) = @_;

  my $node_ip     = get_external_ip($nodeid);
  my $admin_file  = "$WORKING_DIR/admin";
  
  print_action("Remove packages for diskless nodes: $package_list ...") ;

  # 
  # Create a pkgrm admin file
  #
  open(ADMIN, ">$admin_file") ||
    &error("Unable to create $admin_file");
  
  $basedir="default" ;

   print ADMIN <<"FEO";
	
mail=
instance=overwrite
partial=nocheck
runlevel=quit
idepend=quit
rdepend=nocheck
space=quit
setuid=nocheck
conflict=nocheck
action=nocheck
basedir=$basedir

FEO


  close(ADMIN);
  
  #
  # copy the admin file on the remote node
  #
  &exec_cmd( "$RCP $admin_file $node_ip:/tmp/admin" ) ||
    &error ( "Unable to copy $admin_file on $node_ip:/tmp/admin") ;

  #
  # remove packages for all NMEN concerned by the installation
  #
  for my $node (0..$#NODE_LIST) {

    if (($NODE_LIST[$node]{type} eq "DISKLESS") && ($NODE_LIST[$node]{concerned} eq "TRUE")) {

      my $node_name_template = build_node_name($NODE_LIST[$node]{id}, "");
      my $node_root = "/export/root/$node_name_template";

      print_subaction("for $node_name_template ...") ;
      #
      # remove the packages in the reverse order
      #
      # do not display error
      my @list = split(" ",$package_list) ;
      for (my $i = (scalar(@list) - 1) ; $i >= 0 ; $i--) {
	# remove the ending part of package name like SUNWefcx.u
	# (the package is known as SUNWefcx after installation)
	my $pkg = $list[$i];
	my $pos = index($pkg, ".");
	if ($pos > -1) {
	  $pkg = substr($pkg, 0, $pos);
	}
	remote_exec_ignore ( "$node_ip", "echo y | $RPKGRM -a /tmp/admin -R $node_root $pkg" ) ;
      }
    }
  }
  
  #
  # remove the admin file from the remote node
  #
  remote_exec ( "$node_ip", "$RM /tmp/admin" ) ;
  
  # remove previously created pkgadd admin file
  #
  unlink($admin_file);	
}

#-------------------------------------------------------------------------------
#
# Install a package for all NMENs on a MEN
#
# 1: Node ID
# 2: Package list
# 3: answers file (null string if not required)
#
# if exists, the answers file is deleted before exiting !
#
#-------------------------------------------------------------------------------

sub install_nmen_pkg {

  my ($node_id, $root_dir, $package_subdir, $package_list, $answers_file) = @_;

  my $node_ip     = get_external_ip($node_id);
  my $admin_file  = "$WORKING_DIR/admin";
  
  #
  # Test the product directory tree
  #
  &control_dir_is_readable ("$root_dir/$package_subdir");
  
  print_action("Installing packages for diskless nodes: $package_list ...") ;

  #
  # mount ROOT_DIR
  #   
  remote_mount($node_ip, $root_dir, "/mnt") ;
  
  # 
  # Create a pkgadd admin file
  #
  open(ADMIN, ">$admin_file") ||
    &error("Unable to create $admin_file");
  
  $basedir="default" ;

   print ADMIN <<"FEO";
	
mail=
instance=overwrite
partial=nocheck
runlevel=quit
idepend=nocheck
rdepend=nocheck
space=quit
setuid=nocheck
conflict=nocheck
action=nocheck
basedir=$basedir

FEO


  close(ADMIN);
  
  #
  # copy the admin file on the remote node
  #
  &exec_cmd( "$RCP $admin_file $node_ip:/tmp/admin" ) ||
    &error ( "Unable to copy $admin_file on $node_ip:/tmp/admin") ;
  
  #
  # copy the answers file on the remote node
  #
  if ($answers_file ne "") {
    &exec_cmd( "$RCP $answers_file $node_ip:/tmp/answers" ) ||
	  &error ( "Unable to copy $answers_file on $node_ip:/tmp/answers") ;
  }

  #
  # Install packages on NMEN concerned by the installation
  #
  foreach $node (0..$#NODE_LIST) {
    if (($NODE_LIST[$node]{type} eq "DISKLESS") && ($NODE_LIST[$node]{concerned} eq "TRUE")) {
      $node_name_template = build_node_name($NODE_LIST[$node]{id}, "");
      $node_root = "/export/root/$node_name_template";

      #
      # Remotely create the install directory in case it is needed by BASE_DIR
      #
      &remote_exec ( "$node_ip", "$RMKDIR -p $node_root/opt/SUNWcgha " ) ||
	&error("Unable to create $node_root/opt/SUNWcgha directory needed by BASE_DIR package installation setup");	
      print_subaction("for $node_name_template ...") ;
      #
      # Install packages remotely
      #
      if ($answers_file eq "") {
	&remote_exec ( "$node_ip", "$RPKGADD -n -a /tmp/admin -d /mnt/$package_subdir -M -R $node_root $package_list" ) ||
	  &exit_install_pkg($admin_file, $answers_file, $node_ip, $package_list, $root_dir, $package_subdir);
      } else {
	&remote_exec ( "$node_ip", "$RPKGADD -n -a /tmp/admin -r /tmp/answers -d /mnt/$package_subdir -M -R $node_root $package_list" ) ||
	  &exit_install_pkg($admin_file, $answers_file, $node_ip, $package_list, $root_dir, $package_subdir);
      }
    }
  } 

  
  #
  # remove the admin file from the remote node
  #
  &remote_exec ( "$node_ip", "$RM /tmp/admin" ) ||
    &exit_install_pkg($admin_file, $answers_file, $node_ip, $package_list, $root_dir, $package_subdir);
  
  #
  # remove the answers file from the remote node
  #
  if ($answers_file ne "") {
    &remote_exec ( "$node_ip", "$RM /tmp/answers" ) ||
      &exit_install_pkg($admin_file, $answers_file, $node_ip, $package_list, $root_dir, $package_subdir);
  }

  #
  # remove previously created pkgadd admin and answers file
  #
  unlink($admin_file);
  if ($answers_file ne "") {
    unlink($answers_file);
  }

  #
  # umount ROOT_DIR
  #   
  &remote_exec ( "$node_ip", "$RUMOUNT /mnt" ) ||
    &error ( "Unable to NFS umount /mnt" );	
}

#===============================================================================
#
#    PATCH MANAGEMENT ROUTINES
#
#===============================================================================

#-------------------------------------------------------------------------------
#
# Create the prerequisites for patching shared packages
#
#-------------------------------------------------------------------------------

sub shared_patch_preparation {
  my $dir  = "var/sadm/system/admin" ;
  my $file = "INST_RELEASE" ;

  my $node_ip   = get_external_ip($NODE_ID);

  print_action("Creating prerequisites for patching shared packages ...") ;

  #
  # Remotely create directory
  #
  remote_exec( $node_ip, "$RMKDIR -p $DIR_SUNWCGHA_EXPORTED/services/$dir" ) ||
    error ( "Unable to create $DIR_SUNWCGHA_EXPORTED/services/$dir" );

  #
  # Remotely copy file
  #
  remote_exec($node_ip, "$RCOPY /$dir/$file $DIR_SUNWCGHA_EXPORTED/services/$dir/$file") ||
    error ( "Unable to copy $file on $DIR_SUNWCGHA_EXPORTED/services/$dir/$file" );
}

#-------------------------------------------------------------------------------
#
#  clean up when a patch installation fails
#
#  1: node ip address
#
#-------------------------------------------------------------------------------

sub exit_install_patch {

  my ($node_ip, $patch_id, $dir, $subdir) = @_;

  &remote_exec ( "$node_ip", "$RUMOUNT /mnt" );
  &error("Unable to install patch $patch_id (from $dir/$subdir)" );
}

#-------------------------------------------------------------------------------
#
# remove patch on a MEN
# 
#-------------------------------------------------------------------------------

sub remove_men_patch {

  my ($node_id, $patch, $option) = @_;

  my $node_ip     = get_external_ip($node_id);
  my $optcmd ;
  my $adj ;

  my $patchid = $patch;
  if (substr($patch,0,1) eq "T") {
    # it's a temporary patch, remove the T to have the patch ID
    $patchid = substr($patch,1,length($patch)-1);
  }

  if ($option eq "LOCAL") {
    $optcmd = "" ;
    $adj = "local" ;
  } elsif ($option eq "SHARED") {
    $optcmd = "-R $SHARED_FS" ;
    $adj = "shared" ;
  } else {
    error("Invalid option ($option) for MEN patch installation") ;
  }
  print_action("Removing $adj patch: $patch ...") ;	
		
  #
  # Remove patch remotely
  #
  my @list = split(" ",$patch) ;
  for (my $i = (scalar(@list) - 1) ; $i >= 0 ; $i--) {
    remote_exec_ignore ( "$node_ip", "$RPATCHRM $optcmd $list[$i]" ) ;
  }
}

#-------------------------------------------------------------------------------
#
# install patch on a MEN
# 
#-------------------------------------------------------------------------------

sub install_men_patch {

  my ($node_id, $root_dir, $patch_subdir, $patch, $option) = @_;

  my $node_ip     = get_external_ip($node_id);
  my $optcmd ;
  my $adj ;

  if ($option eq "LOCAL") {
    $optcmd = "" ;
    $adj = "local" ;
  } elsif ($option eq "SHARED") {
    $optcmd = "-R $SHARED_FS" ;
    $adj = "shared" ;
  } else {
    error("Invalid option ($option) for MEN patch installation") ;
  }

  print_action("Installing $adj patch: $patch ...") ;

  #
  # Test the product directory tree
  #
  &control_dir_is_readable ("$root_dir/$patch_subdir");

  #
  # mount ROOT_DIR
  #   
  remote_mount($node_ip, $root_dir, "/mnt") ;
		
  #
  # Add patch remotely
  #
  &remote_exec ( "$node_ip", "$RPATCHADD -M /mnt/$patch_subdir $optcmd $patch" ) ||
    &exit_install_patch($node_ip, $patch, $root_dir, $patch_subdir);
      
  #
  # umount ROOT_DIR
  #   
  &remote_exec ( "$node_ip", "$RUMOUNT /mnt" ) || 
    &error ( "Unable to NFS umount /mnt" );
}

#-------------------------------------------------------------------------------
#
# remove a patch for all NMENs on a MEN
# 
#-------------------------------------------------------------------------------

sub remove_nmen_patch {

  my ($node_id, $patch, $where) = @_;

  my $node_ip     = get_external_ip($node_id);
   
  print_action("Removing patch for diskless nodes: $patch ...") ;
		
  if (($where ne "SOLARIS") && ($where ne "DISKLESS")) {
    error("internal error: install_nmen_patch location: $where (SOLARIS or DISKLESS required)");
  }
  
  my $patchid = $patch;
  if (substr($patch,0,1) eq "T") {
    # it's a temporary patch, remove the T to have the patch ID
    $patchid = substr($patch,1,length($patch)-1);
  }

  if ($where eq "SOLARIS") {
    #
    # Remove patch from Solaris environment remotely
    #
    print_subaction("for $TARGET_OS service ...") ;
    my @list = split(" ",$patch) ;
    for (my $i = (scalar(@list) - 1) ; $i >= 0 ; $i--) {
      remote_exec_ignore ( "$node_ip", "$RPATCHRM -S $TARGET_OS $list[$i]" ) ;
    }
  } else {

    #
    # Remove patches from diskless nodes remotely
    #
    for my $node (0..$#NODE_LIST) {
      if (($NODE_LIST[$node]{type} eq "DISKLESS") && ($NODE_LIST[$node]{concerned} eq "TRUE")) {
	$node_name_template = build_node_name($NODE_LIST[$node]{id}, "");
	$node_root = "/export/root/$node_name_template";
	
	print_subaction("for $node_name_template ...") ;
	
	my @list = split(" ",$patch) ;
	for (my $i = (scalar(@list) - 1) ; $i >= 0 ; $i--) {
	  remote_exec_ignore ( "$node_ip", "$RPATCHRM -R $node_root $list[$i]" ) ;
	}
      }
    }
  }
}

#-------------------------------------------------------------------------------
#
# install patch for all NMENs on a MEN
# 
#-------------------------------------------------------------------------------

sub install_nmen_patch {

  my ($node_id, $root_dir, $patch_subdir, $patch, $where) = @_;

  my $node_ip     = get_external_ip($node_id);
   
  print_action("Installing patch for diskless nodes: $patch ...") ;

  #
  # Test the product directory tree
  #
  &control_dir_is_readable ("$root_dir/$patch_subdir");

  #
  # mount ROOT_DIR
  #
  remote_mount($node_ip, $root_dir, "/mnt") ;

  if (($where ne "SOLARIS") && ($where ne "DISKLESS")) {
    error("internal error: install_nmen_patch location: $where (SOLARIS or DISKLESS required)");
  }

  if ($where eq "SOLARIS") {

    #
    # Add patch on Solaris environment remotely
    #
    print_subaction("for $TARGET_OS service ...") ;
    if ($USE_DCSS eq "YES") {
      remote_exec ( "$node_ip", "$RNETRAOS patch -i -f -M /mnt/$patch_subdir -P $patch $TARGET_OS" ) ||
	exit_install_patch($node_ip, $patch, $root_dir, $patch_subdir);
    } else {
      remote_exec ( "$node_ip", "$RPATCHADD -M /mnt/$patch_subdir -S $TARGET_OS $patch" ) ||
	exit_install_patch($node_ip, $patch, $root_dir, $patch_subdir);
    }

  } else {

    #
    # Install patches on diskless nodes remotely
    #
    for my $node (0..$#NODE_LIST) {
      if (($NODE_LIST[$node]{type} eq "DISKLESS") && ($NODE_LIST[$node]{concerned} eq "TRUE")) {
	$node_name_template = build_node_name($NODE_LIST[$node]{id}, "");
	$node_root = "/export/root/$node_name_template";
	
	print_subaction("for $node_name_template ...") ;
	
	&remote_exec ( "$node_ip", "$RPATCHADD -R $node_root -M /mnt/$patch_subdir $patch" ) ||
	  &exit_install_patch($node_ip, $patch, $root_dir, $patch_subdir);
      }
    }
  }
   
  #
  # umount ROOT_DIR
  #   
  &remote_exec ( "$node_ip", "$RUMOUNT /mnt" ) || 
    &error ( "Unable to NFS umount /mnt" );
}

#===============================================================================
#
# HANDLING OF NODE NAMES AND RELATED FILES
#
#===============================================================================

#-------------------------------------------------------------------------------
#
# build the server name
#
#-------------------------------------------------------------------------------

sub build_server_name {
 
  return "sar-c${CLUSTER}-n${SERVER_NODE}" ;
}

#-------------------------------------------------------------------------------
#
# look for a node by its nodeid
#
#-------------------------------------------------------------------------------

sub getnode
  {
    my ($node_id) = @_ ;

    for my $i (0..$#NODE_LIST) {
      if ($NODE_LIST[$i]{id} eq $node_id) {
	return $i;
      }
    }

    error("Internal error: getnode: $node_id can't be reached\n");
  }

#-------------------------------------------------------------------------------
#
# build the node name depending on the node ID
#
#-------------------------------------------------------------------------------

sub build_node_name {
 
  my ($nodeid, $interface) = @_ ;

  my $name = "";

  if ($nodeid eq $MASTER_ID) {
    $name = "master" ;
  } elsif ($nodeid eq $VICEMASTER_ID) {
    $name = "vice-master" ;
  } else {
    my $hostname = $NODE_LIST[getnode($nodeid)]{name};
    
    if (($hostname ne "") && ($hostname ne "-")) {
      $name = $hostname ;
    } else {
      if (($nodeid eq $NODE_LIST[$MEN1]{id}) || ($nodeid eq $NODE_LIST[$MEN2]{id})) {
	$name = "MEN-C${CLUSTER}-N${nodeid}" ;
      } else {
	$name = "NMEN-C${CLUSTER}-N${nodeid}" ; 
      }
    }
  }

  my $post = "" ;
  if ($interface eq "") {
    $post = "" ;
  }
  if ($interface eq "NIC0") {
    $post = $POST_NIC0 ;
  }
  if ($interface eq "NIC1") {
    $post = $POST_NIC1 ;
  }
  if ($interface eq "CGTP") {
    $post = $POST_CGTP ;
  }

  $name = $name . $post ;

  return $name ;
}

#-------------------------------------------------------------------------------
#
# build the cluster IP address depending on the node ID
#
#-------------------------------------------------------------------------------

sub build_node_ip {
 
  my ($node_id, $interface) = @_ ;

  # default is NIC0 (for external IP)

  if ($node_id == $MASTER_ID) {
    if ($interface eq "NIC0") {
      return computeIp($MASTER_ID, 
		       $CLUSTER_BIT_NIC0_NET, 
		       $CLUSTER_BIT_NETMASK);
    }
    if ($interface eq "NIC1") {
      return computeIp($MASTER_ID, 
		       $CLUSTER_BIT_NIC1_NET, 
		       $CLUSTER_BIT_NETMASK);
    }
    if ($interface eq "CGTP") {
      return computeIp($MASTER_ID, 
		       $CLUSTER_BIT_CGTP_NET, 
		       $CLUSTER_BIT_NETMASK);
    }
    return computeIp($MASTER_ID, 
		     $CLUSTER_BIT_NIC0_NET, 
		     $CLUSTER_BIT_NETMASK);
  }

  my $idx=getnode($node_id);

  if ($interface eq "NIC0") {
    return $NODE_LIST[$idx]{nic0Ip};
  }
  if ($interface eq "NIC1") {
    return $NODE_LIST[$idx]{nic1Ip};
  }
  if ($interface eq "CGTP") {
    return $NODE_LIST[$idx]{cgtpIp};
  }

  return $NODE_LIST[$idx]{nic0Ip};

}

#-------------------------------------------------------------------------------
#
# get the external IP address of a MEN node
#
#-------------------------------------------------------------------------------

sub get_external_ip {

  my ($node_id) = @_;

  if ($USE_PUBLIC_NETWORK eq "YES") {
    return $NODE_LIST[getnode($node_id)]{pubIp};
  } else {
    return build_node_ip($node_id, "");
  }

}

#-------------------------------------------------------------------------------
#
# get the external name of a MEN node
#
#-------------------------------------------------------------------------------

sub get_external_name {

  ($node_id) = @_;

  if ($USE_PUBLIC_NETWORK eq "YES") {
    return $NODE_LIST[getnode($node_id)]{pubName};
  } else {
    return build_node_name($node_id, "");
  }

}

#------------------------------------------------------------------------------
#
# updating a netmask file
#
#------------------------------------------------------------------------------

sub add_netmask {
  my ( $netmasks_file, $network, $mask ) = @_ ;
  my $count=0;
  my $read_network;
  my $read_mask;
  
  $count = 0;   
  if ( -f "$netmasks_file" ) {
    open (NETMASKS, $netmasks_file ) 
      || COMMON::error("Unable to open $netmasks_file");
    
    while ( <NETMASKS> ) {      
      chop();
      s/#.*//;
      ($read_network, $read_mask) = split (/[\s\t]+/, $_) ;
      next unless (defined($read_network));
      next unless (defined($read_mask));
      
      if ("$read_network" eq "$network") {
	if ("$read_mask" ne "$mask") {
	  COMMON::error("Ip \"$network\" does not match \"$mask\" as first hostname in $netmasks_file") 
	}
	$count++;
	last;
      }
    }
    close (NETMASKS);
  }

  if ( $count == 0) {
    open (NETMASKS, ">>$netmasks_file" ) || COMMON::error("Unable to update $netmasks_file");
    print  NETMASKS "$network $mask\n";
    close (NETMASKS);
  }   
}

#-------------------------------------------------------------------------------
#
# updating an host entry
#
# . look for the entry based on the IP address
# . when found, add each alias that are not yet defined and keep the already
#   alias and entries
#
#-------------------------------------------------------------------------------

sub update_hosts_entry {

  my ($hosts_file, $host_ip, $alias_list) = @_;
  my $found_host = FALSE ;
  my $found_alias = FALSE;

  #
  # look for the entry
  #

  open(HOSTS, "<$hosts_file");
  open(HOSTS_TMP, ">$hosts_file.tmp");
  while (<HOSTS>) {
    chomp();
    s/#.*//;

    # is it a valid host entry ?
    $line = $_;
    if ($line =~ m/([0-9]*.[0-9]*.[0-9]*.[0-9]*)[\s\t]*(.*)/) {

      # $1: IP addresse
      # $2: aliases list

      # look for the searched one
      if ($1 eq $host_ip) {
	$found_host = TRUE;

	# check for all required aliases and add the missing ones

	foreach my $expected_alias (split(/[\s\t]+/,$alias_list)) {
	  $found_alias = FALSE;

	  foreach my $current_alias (split(/[\s\t]+/,$2)) {
	    if ($expected_alias eq $current_alias) {
	      $found_alias = TRUE;
	    }
	  }

	  # add the alias when not found
	  if ($found_alias eq FALSE) {
	    $line .= " $expected_alias";
	  }
	}
      }
      # copy the host entry (the same entry or the update one)
      print HOSTS_TMP "$line\n";
    }
  }

  if ($found_host eq FALSE) {
    # add the entry
    print HOSTS_TMP "$host_ip $alias_list\n";
  }

  close (HOSTS);
  close (HOSTS_TMP);

  # move the temporary file on the final one
  exec_cmd ("$MV $hosts_file.tmp $hosts_file");
  }

#-------------------------------------------------------------------------------
#
#  add alias
#
#  add an alias in the string if it's not already defined
#
#-------------------------------------------------------------------------------

sub add_alias {

  my ($alias_ref, $value) = @_;

  foreach $elem (split(' ', $$alias_ref)) {
    if ($elem eq $value) {
      return ;
    }
  }

  $$alias_ref = join " ", $$alias_ref, $value;
}

#-------------------------------------------------------------------------------
#
#  add one entry in the host file
#
#-------------------------------------------------------------------------------

sub add_hosts_file_entry {

  my ($hosts_file, $node, $current_node_id) = @_;
    
  my $node_id = $NODE_LIST[$node]{id};

  my $loghost= "loghost" ;
  my $alias_list = "";
    
  my $node_name = build_node_name($node_id, "") ;

  my $node_ip_if0;
  my $node_ip_if1;
  my $node_ip_cgtp0;
      
  my $node_name_if0;   
  my $node_name_if1;
  my $node_name_cgtp;

  $node_ip_if0   = COMMON::build_node_ip($node_id, "NIC0");
  $node_name_if0 = build_node_name($node_id, "NIC0") ;

  if ($USE_CGTP eq "YES") {
    $node_ip_if1   = COMMON::build_node_ip($node_id, "NIC1");
    $node_ip_cgtp0 = COMMON::build_node_ip($node_id, "CGTP");
    $node_name_if1 = build_node_name($node_id, "NIC1") ;
    $node_name_cgtp0= build_node_name($node_id, "CGTP") ;
  }


  #
  # for diskless, a alias with the generic name (without postfix) 
  # must be created for the nic0 interface (must be the first)
  # to avoid a failure while smdiskless is executing
  #

  my $domain = SEQUENCER::get_data("DOMAIN");
  
  if ($node_id eq $current_node_id) {
    
    $alias_list = "";
    add_alias(\$alias_list, "$node_name");
    add_alias(\$alias_list, "$node_name.$domain");
    add_alias(\$alias_list, "$node_name_if0");
    add_alias(\$alias_list, "$node_name_if0.$domain") ;
    if ($NODE_LIST[$node]{type} eq "DISKLESS") {
      # add "loghost" for diskless node because /etc/hosts created
      # for MEN, it's already defined because /etc/hosts only updated
      add_alias(\$alias_list, "loghost") ;
    }
    
    update_hosts_entry($hosts_file, $node_ip_if0, $alias_list);
    
    if ($USE_CGTP eq "YES") {
      $alias_list = "";
      add_alias(\$alias_list, "$node_name_if1");
      add_alias(\$alias_list, "$node_name_if1.$domain") ;
      update_hosts_entry($hosts_file, $node_ip_if1 , $alias_list);
      
      $alias_list = "";
      add_alias(\$alias_list, "$node_name_cgtp0");
      add_alias(\$alias_list, "$node_name_cgtp0.$domain");
      update_hosts_entry($hosts_file, $node_ip_cgtp0, $alias_list);
    }
  } else {
    
    $alias_list = "";
    add_alias(\$alias_list, "$node_name");
    add_alias(\$alias_list, "$node_name_if0");
    update_hosts_entry($hosts_file, $node_ip_if0, $alias_list);
    
    if ($USE_CGTP eq "YES") {
      $alias_list = "$node_name_if1" ;
      update_hosts_entry($hosts_file, $node_ip_if1, $alias_list);
      $alias_list = "$node_name_cgtp0" ;
      update_hosts_entry($hosts_file, $node_ip_cgtp0, $alias_list);
    }
  }

  if ($USE_PUBLIC_NETWORK eq "YES") {
    $alias_list = "";
    add_alias(\$alias_list, "$NODE_LIST[$node]{pubName}");
    update_hosts_entry($hosts_file, $NODE_LIST[$node]{pubIp}, $alias_list);
  }
}

#-------------------------------------------------------------------------------
#
#  update the local hosts file before transferring (for MEN)
#
#-------------------------------------------------------------------------------

sub update_hosts_file {
  
  my ($hosts_file, $current_node_id) = @_ ;
  
  my $master_name_if0;
  my $master_name_if1;
  my $master_name_cgtp0;
  my $master_ip_if0;
  my $master_ip_if1;
  my $master_ip_cgtp0;
  
  $master_name_if0 = build_node_name($MASTER_ID, "NIC0");
  $master_ip_if0 = build_node_ip($MASTER_ID, "NIC0");

  if ($USE_CGTP eq "YES") {
    $master_name_if1 = build_node_name($MASTER_ID, "NIC1");
    $master_name_cgtp0 = build_node_name($MASTER_ID, "CGTP");
    $master_ip_if1 = build_node_ip($MASTER_ID, "NIC1");
    $master_ip_cgtp0 =  build_node_ip($MASTER_ID, "CGTP");
  }
    
    for my $node (0..$#NODE_LIST) {
      add_hosts_file_entry($hosts_file, $node, $current_node_id);
    }
    
    update_hosts_entry($hosts_file, $master_ip_if0, "$master_name_if0") ;
    if ($USE_CGTP eq "YES") {
      update_hosts_entry($hosts_file, $master_ip_if1, "$master_name_if1") ;
      update_hosts_entry($hosts_file, $master_ip_cgtp0, "$master_name_cgtp0");
    }

    if ($USE_EXTERNAL_ACCESS eq "YES") {
      # remove the netmask size if present
      my $ip = $EXTERNAL_IP;
      my $pos = index($ip, "/");
      if ($pos > -1) {
	$ip = substr($ip, 0, $pos);
      }
      update_hosts_entry($hosts_file, $ip, $EXTERNAL_HOSTNAME);
    }
    
  }

#-------------------------------------------------------------------------------
#
#  create the local hosts file before transferring (for diskless)
#
#-------------------------------------------------------------------------------

sub create_hosts_file
  {
    
    my ($hosts_file, $current_node_id) = @_ ;
    
    open (HOSTS, ">$hosts_file" ) || &error("Unable to open $hosts_file");
    print HOSTS "127.0.0.1 localhost\n" ;
    close (HOSTS);
    update_hosts_file($hosts_file, $current_node_id);
  }

#===============================================================================
#
#  ARGUMENT PROCESSING
#
#===============================================================================

#-------------------------------------------------------------------------------
#
#  check if an argument is in a list
#
#  proces the list of variables to be updated and check if the argument refers
#  to one of these variables
#
#-------------------------------------------------------------------------------

sub check_arg {

  ($list, $arg) = @_ ;

  foreach $variable (@$list) {
    if (($ARGLST{$variable}[$ARGSHORT] eq $arg) || ($ARGLST{$variable}[$ARGLONG] eq $arg)) {
      return $variable ;
    }
  }
  return "" ;
}

#-------------------------------------------------------------------------------
#
# init and check prerequisites
#
#-------------------------------------------------------------------------------

sub init_and_check {

  ($required) = @_ ;

  # check that required arguments are present
  foreach $variable (@$required) {
    if (! defined($$variable) || ($$variable eq "")) {
      print_log("$variable is required\n") ;
      exit 1 ;
    }
  }
}

#-------------------------------------------------------------------------------
#
# get a checksum
#
#-------------------------------------------------------------------------------

sub get_checksum {

  my ($filename) = @_;

  my $result = qx/$SUM $filename/;
  if ($? ne 0) {
    error("Can not compute checksum for $filename by using $SUM");
  }
  my ($checksum) = split(/[\t\s]+/, $result);

  return $checksum;
}

#-------------------------------------------------------------------------------
#
# get the node id of the vice master
#
#-------------------------------------------------------------------------------

sub get_id {

  my ($role) = @_;
  my $node_ip = build_node_ip($MEN1_ID,"");
  
  my $command = "$RSH -n $node_ip $CMMADM -c $role";
  print_debug("DEBUG_CMD", "$command");

  @result = qx/$command/;

  foreach $line (@result) {
    chomp($line);
    print_debug("DEBUG_CMD", "  $line");
    if ($line =~ m/node_id[\s\t]*=[\s\t]([0-9]*)/) {
      print_debug("DEBUG_CMD", "    node id ($role): $1");
      return $1;
    }
  }

  error("Cannot determine node id for node with role=$role");
}

#===============================================================================
#
#  IP address handling
#
#===============================================================================

#------------------------------------------------------------------------------
#
#  convertDottedToBitmask
#
#    convert a dotted form to a bit mask
#
#------------------------------------------------------------------------------

sub convertDottedToBitmask {

  my ($dotted) = @_;

  my $mask = 0;
  foreach $value (split('\.', $dotted)) {
    $mask = $mask * 256 + $value;
  }

  return $mask;
}

#------------------------------------------------------------------------------
#
#  convertBitmaskToDotted
#
#    convert a bit mask to a dotted form
#
#------------------------------------------------------------------------------

sub convertBitmaskToDotted {

  my ($bitmask) = @_;

  my $dotted = "";
  for (my $i=0; $i < 4; $i++) {
    if ($i == 0) {
      $dotted = sprintf("%d", $bitmask % 256);
    } else {
      $dotted = sprintf("%d.%s", $bitmask % 256, $dotted)
    }
    $bitmask = $bitmask / 256;
  }

  return $dotted;
}

#------------------------------------------------------------------------------
#
#  computeIp
#
#    compute the IP address from:
#    . the node id
#    . the subnetwork (bit mask)
#    . the netmask (bit mask)
#
#------------------------------------------------------------------------------

sub computeIp {

  my ($nodeId, $subnet, $netmask) = @_;

  my $ip = convertBitmaskToDotted(($subnet & $netmask) + $nodeId);
  
  return $ip;
}

#------------------------------------------------------------------------------
#
#  computeBroadcast
#
#    compute the default broadcast address from:
#    . the subnetwork (bit mask)
#    . the netmask (bit mask)
#
#------------------------------------------------------------------------------

sub computeBroadcast {

  my ($subnet, $netmask) = @_;

  my $hostid = 0xffffffff ^ $netmask;

  my $ip = convertBitmaskToDotted(($subnet & $netmask) + $hostid);
  
  return $ip;
}

{
}
