#!/usr/bin/perl
# ------------------------------------------------------------------------------
#  ident "@(#)addon.pl 1.12 03/04/24 SMI"
# ------------------------------------------------------------------------------
#
#  Copyright 2001-2003 Sun Microsystems, Inc. All rights reserved.
#
# ------------------------------------------------------------------------------
package ADDON;

use variables;


#
# What and when ?
#
#
# Packages and patches installed during a call to this script depend on:
# . the phase assigned to the package or the patch.
# . the scope of the package
# . the current phase
#
# Herefafter actions performed

#
# When ?
#   depends on the current phase and the required phase
#
#    current phase                     !        phase to install
#                                      !   on MEN    !   on diskless
# -------------------------------------------------------------------------
#   I_MEN (during Solaris on MEN)      !     I       !       -
#   S_MEN (after Solaris on MEN)       !     S       !       -
#   F_MEN (after FS on MEN)            !     F       !       -
#   S_NMEN (after Solaris on diskless) !     x       !       I+S if men #1
#   F_NMEN (after FS on diskless)      !     D       !       F or D if men #1

# 
# Where ?
#   depends on the scope
#
# Scope LOCAL: on / for MEN and diskless (root file systtem for each diskless)
#
# Scope USR_SPECIFIC: =LOCAL for MEN and on /export/os.. or diskless (only for men #1)
#
# Scope SHARED: only for men #1: on the shared directory (not for S_MEN: not already created).
#


my $FOR_MEN="";
my $FOR_NMEN="";
my $WHERE="";

my $SCOPE; # to_be_processed can change the scope defined by the user
my $CHANGE; # indicate that a change has been made and a warning must be displayed
my $DIR;          # directory where the package/patch is really located
my $SUBDIR;       # subdirectory where the package/patch is really located
my $MANAGE_SHARE; # must share/unshare command be executed

#-----------------------------------------------------------
#
#   used for testing (retrieve local module variables)
#
#-----------------------------------------------------------

sub get_info {
  my ($for_men, $for_nmen, $where, $scope, $change) = @_;

  $$for_men = $FOR_MEN;
  $$for_nmen = $FOR_NMEN;
  $$where = $WHERE;
  $$scope = $SCOPE;
  $$change = $CHANGE;
}

#-----------------------------------------------------------
#
#   determine if the patch must be installed
#
#-----------------------------------------------------------

sub to_be_processed  {
    my ($type, $phase, $men, $nmen, $scope, $dir, $subdir) = @_;

    $SCOPE = $scope;
    $CHANGE = "NO";

    $FOR_MEN = "FALSE" ;
    $FOR_NMEN = "FALSE" ;

    if (($type ne "PACKAGE") && ($type ne "PATCH")) {
      COMMON::error("Internal error: to_be_processed: invalid type $type");
    }

    $DIR = $dir;
    $SUBDIR = $subdir;
    $MANAGE_SHARE = "TRUE";
    
    if ($dir eq "NHAS") {
      $MANAGE_SHARE = "FALSE";
      $DIR = $ROOT_DIR;
      if ($subdir eq "") {
	if ($type eq "PACKAGE") {
	  $SUBDIR = $PACKAGE_SUBDIR;
	} else {
	  $SUBDIR = $PATCH_SUBDIR;
	}
      }
    } elsif ($dir eq "SOLARIS") {
      $MANAGE_SHARE = "FALSE";
      $DIR = $SOLARIS_DIR;
      if ($subdir eq "") {
	if ($type eq "PACKAGE") {
	  $SUBDIR = $SOLARIS_PACKAGE_SUBDIR;
	} else {
	  $SUBDIR = $SOLARIS_PATCH_SUBDIR;
	}
      }
    } elsif (($dir eq $SOLARIS_DIR) || ($dir eq $ROOT_DIR)) {
      $MANAGE_SHARE = "FALSE";
    }

    $WHERE = "";

    my $installed = "FALSE";

    # determine if it's a good phase for MEN
    my $good_phase_for_men = "FALSE";
    if ( 
        (($phase eq "I") && ($PHASE eq "I_MEN"))
        || (($phase eq "S") && ($PHASE eq "S_MEN"))
        || (($phase eq "F") && ($PHASE eq "F_MEN"))
        || (($phase eq "D") && ($PHASE eq "F_NMEN"))
       ) {
      $good_phase_for_men = "TRUE";
    }

    # determine if it's a good phase for diskless (note: no I_NMEN phase!)
    my $good_phase_for_diskless = "FALSE";
    if (
        (($phase eq "I") && ($PHASE eq "S_NMEN"))
     || (($phase eq "S") && ($PHASE eq "S_NMEN"))
	 || ((($phase eq "F") || ($phase eq "D")) && ($PHASE eq "F_NMEN"))
       ) {
      $good_phase_for_diskless = "TRUE";
    }
    
    # scope: SHARED
    if ($scope eq "SHARED") {
      if (($good_phase_for_men eq "TRUE") && ($PHASE ne "I_MEN") && ($PHASE ne "S_MEN") && ($NODE_ID eq $MEN1_ID)) {
	$FOR_MEN = "TRUE";
	return "TRUE" ;
      } 
      return "FALSE";
    }

    # scope: USR_SPECIFIC
    if ($scope eq "USR_SPECIFIC") {
      if (($good_phase_for_men eq "TRUE") && ($men eq "Y")) {
	$FOR_MEN = "TRUE";
	# in that case, the scope is changed to LOCAL (/usr is on root file system)
	$SCOPE = "LOCAL";
	# the warning is not shown in this case
#	$CHANGE = "YES";
	$installed = "TRUE" ;
      } 
      # to be installed only at the first installation and not when adding a
      # diskless
      if (($good_phase_for_diskless eq "TRUE") && ($nmen eq "Y") && ($NODE_ID eq $MEN1_ID) && ($INSTALL_TYPE eq "CLUSTER")) {
	# be aware that the install_men_xxx is called (historical reason ...)
	$WHERE = "SOLARIS";
	$FOR_NMEN = "TRUE";
	$installed = "TRUE";
      }
      return $installed;
    }

    # scope: LOCAL (test for both MEN and NMEN)
    if (($good_phase_for_men eq "TRUE") && ($men eq "Y")) {
      $FOR_MEN = "TRUE" ;
      $installed = "TRUE";
    }
    if (($good_phase_for_diskless eq "TRUE") && ($nmen eq "Y") && ($NODE_ID eq $MEN1_ID)) {
      $WHERE = "DISKLESS" ;
      $FOR_NMEN = "TRUE";
      $installed = "TRUE";
      return "TRUE";
    }

    return $installed;
}

#-----------------------------------------------------------
# PACKAGES
#-----------------------------------------------------------

#-----------------------------------------------------------
#
#   installing a list of packages
#
#-----------------------------------------------------------

sub install_packages
  {
    for (my $i = 0 ; $i < scalar(@PACKAGE_LIST) ; $i++) {
      if (to_be_processed("PACKAGE",
			  $PACKAGE_LIST[$i]{phase}, $PACKAGE_LIST[$i]{men},
			  $PACKAGE_LIST[$i]{nmen}, $PACKAGE_LIST[$i]{scope},
			  $PACKAGE_LIST[$i]{dir}, $PACKAGE_LIST[$i]{subdir}) 
	  eq "TRUE") 
	{
	  if ($CHANGE eq "YES") {
	    COMMON::warning("scope $PACKAGE_LIST[$i]{scope} changed to $SCOPE for $PACKAGE_LIST[$i]{reference} when installing on MEN\n");
	  }

	  if ($MANAGE_SHARE eq "TRUE") {
	    COMMON::share($DIR);
	  }

	  if ($FOR_MEN eq "TRUE") {
	    COMMON::install_men_pkg($NODE_ID, $DIR, $SUBDIR,
				    $PACKAGE_LIST[$i]{reference}, $SCOPE, "") ;
	  } 
	  if ($FOR_NMEN eq "TRUE") {
	    if ($SCOPE eq "USR_SPECIFIC") {
	      COMMON::install_men_pkg($NODE_ID, $DIR, $SUBDIR,
				      $PACKAGE_LIST[$i]{reference}, $SCOPE, "") ;
	    } else {
	      COMMON::install_nmen_pkg($NODE_ID, $DIR, $SUBDIR,
				       $PACKAGE_LIST[$i]{reference}, "") ;
	    }
	  }
# do not unshare: can be used by another user
#	  if ($MANAGE_SHARE eq "TRUE") {
#	    COMMON::exec_cmd("$UNSHARE $DIR");
#	  }
	}
    }
  }

#-----------------------------------------------------------
#
#   removing a list of packages
#
#-----------------------------------------------------------

sub remove_packages
  {
    # removed in thre reverse order
    for (my $i = scalar(@PACKAGE_LIST) - 1 ; $i >= 0 ; $i--) {
      if (to_be_processed("PACKAGE",
			  $PACKAGE_LIST[$i]{phase}, $PACKAGE_LIST[$i]{men},
			  $PACKAGE_LIST[$i]{nmen}, $PACKAGE_LIST[$i]{scope},
			  $PACKAGE_LIST[$i]{dir}, $PACKAGE_LIST[$i]{subdir})
	  eq "TRUE") 
	{
	  if ($FOR_MEN eq "TRUE") {
	    COMMON::remove_men_pkg($NODE_ID, $PACKAGE_LIST[$i]{reference}, $SCOPE) ;
	  } 
	  if ($FOR_NMEN eq "TRUE") {
	    if ($SCOPE eq "USR_SPECIFIC") {
	      COMMON::remove_men_pkg($NODE_ID, $PACKAGE_LIST[$i]{reference}, $SCOPE) ;
	    } else {
	      COMMON::remove_nmen_pkg($NODE_ID, $PACKAGE_LIST[$i]{reference}) ;
	    }
	  }
	}
    }
  }

#-----------------------------------------------------------
# PATCHES
#-----------------------------------------------------------

#-----------------------------------------------------------
#
#   installing a list of patches
#
#-----------------------------------------------------------

sub install_patches
  {
    for ($i = 0 ; $i < scalar(@PATCH_LIST) ; $i++) {
      if (to_be_processed("PATCH",
			  $PATCH_LIST[$i]{phase}, $PATCH_LIST[$i]{men},
			  $PATCH_LIST[$i]{nmen}, $PATCH_LIST[$i]{scope},
			  $PATCH_LIST[$i]{dir}, $PATCH_LIST[$i]{subdir}) 
	  eq "TRUE") 
	{
	  if ($CHANGE eq "YES") {
	    COMMON::warning("scope $PATCH_LIST[$i]{scope} changed to $SCOPE for $PATCH_LIST[$i]{reference} when installing on MEN");
	  }
	  if ($MANAGE_SHARE eq "TRUE") {
	    COMMON::share($DIR);
	  }

	  if ($FOR_MEN eq "TRUE") {
	    COMMON::install_men_patch($NODE_ID, $DIR, $SUBDIR,
				      $PATCH_LIST[$i]{reference}, $SCOPE) ;
	  } 
	  if ($FOR_NMEN eq "TRUE") {
	    COMMON::install_nmen_patch($NODE_ID, $DIR, $SUBDIR,
				       $PATCH_LIST[$i]{reference}, $WHERE) ;
	  }
# do not unshare: can be used by another user
#	  if ($MANAGE_SHARE eq "TRUE") {
#	    COMMON::exec_cmd("$UNSHARE $DIR");
#	  }
	}
    }
  }

#-----------------------------------------------------------
#
#   removing a list of patches
#
#-----------------------------------------------------------

sub remove_patches
  {
    # removed in thre reverse order
    for (my $i = scalar(@PATCH_LIST) - 1 ; $i >= 0 ; $i--) {
      if (to_be_processed("PATCH",
			  $PATCH_LIST[$i]{phase}, $PATCH_LIST[$i]{men},
			  $PATCH_LIST[$i]{nmen}, $PATCH_LIST[$i]{scope},
			  $PATCH_LIST[$i]{dir}, $PATCH_LIST[$i]{subdir})
	  eq "TRUE") 
	{
	  if ($FOR_MEN eq "TRUE") {
	    COMMON::remove_men_patch($NODE_ID, $PATCH_LIST[$i]{reference}, $SCOPE) ;
	  } 
	  if ($FOR_NMEN eq "TRUE") {
	    COMMON::remove_nmen_patch($NODE_ID, $PATCH_LIST[$i]{reference}, $WHERE) ;
	  }
	}
    }
  }

#-----------------------------------------------------------
#
#   install add-on
#
#-----------------------------------------------------------

sub install_addon
  {
    install_packages();
    install_patches();
  }

#-----------------------------------------------------------
#
#   remove add-on
#
#-----------------------------------------------------------

sub remove_addon
  {  
    remove_patches();
    remove_packages();
  }


#-----------------------------------------------------------
#
#  general function
#
#-----------------------------------------------------------

sub process_addon {

  ($PHASE, $RECOVERY) = @_;
  
  if ($DO_ADDON eq FALSE) {
    return;
  }
  
  my $name="for unknown phase";
 SWITCH: {
    if ($PHASE eq "S_MEN") {
      $name = "after Solaris installation on master-eligible nodes" ;
      last SWITCH;
    }
    if ($PHASE eq "S_NMEN") {
      $name = "after Solaris installation on diskless nodes" ;
      last SWITCH;
    }
    if ($PHASE eq "F_MEN") {
      $name = "after Foundation Services installation on master-eligible nodes" ;
      last SWITCH;
    }
    if ($PHASE eq "F_NMEN") {
      $name = "after Foundation Services installation on diskless nodes" ;
      last SWITCH;
    }
    if ($PHASE eq "A_MEN") {
      $name = "after Application Services installation on master-eligible nodes" ;
      last SWITCH;
    }
    if ($PHASE eq "A_NMEN") {
      $name = "after Application Services installation on diskless nodes" ;
      last SWITCH;
    }
  }
  
  if ($RECOVERY ne "") {
    COMMON::print_recovery_node("Add-on $name: clean-up") ;
    remove_addon();
    
  } else {
    
    COMMON::print_stage_node("Add-on $name") ;
    install_addon();
  }
}

#-----------------------------------------------------------
#
# functions for each phases
#
#-----------------------------------------------------------

sub men_addon_solaris {
  
  ($NODE_ID) = @_;
  
  process_addon("S_MEN", "");
}

sub recovery_men_addon_solaris {
  
  ($NODE_ID) = @_;
  
  process_addon("S_MEN", "YES");
}

#-----------------------------------------------------------

sub nmen_addon_solaris {
  
  ($NODE_ID) = @_;
  
  process_addon("S_NMEN", "");
}

sub recovery_nmen_addon_solaris {
  
  ($NODE_ID) = @_;
  
  process_addon("S_NMEN", "YES");
}

#-----------------------------------------------------------

sub men_addon_fs {
  
  ($NODE_ID) = @_;
  
  process_addon("F_MEN", "");
}

sub recovery_men_addon_fs {
  
  ($NODE_ID) = @_;
  
  process_addon("F_MEN", "YES");
}

#-----------------------------------------------------------

sub nmen_addon_fs {
  
  ($NODE_ID) = @_;
  
  process_addon("F_NMEN", "");
}

sub recovery_nmen_addon_fs {
  
  ($NODE_ID) = @_;
  
  process_addon("F_NMEN", "YES");
}
#------------------------------------------------------------------------------
#
#  Update the finish.sh file with the creation of the admin file
#
#------------------------------------------------------------------------------

sub update_finish_admin {

	print FINISH  <<"FEO";

# create admin file
$RECHO "mail=" > /a/tmp/admin
$RECHO "instance=overwrite" >> /a/tmp/admin
$RECHO "partial=nocheck" >> /a/tmp/admin
$RECHO "runlevel=quit" >> /a/tmp/admin
$RECHO "idepend=nocheck" >> /a/tmp/admin
$RECHO "rdepend=nocheck" >> /a/tmp/admin
$RECHO "space=quit" >> /a/tmp/admin
$RECHO "setuid=nocheck" >> /a/tmp/admin
$RECHO "conflict=nocheck" >> /a/tmp/admin
$RECHO "action=nocheck" >> /a/tmp/admin
$RECHO "basedir=default" >> /a/tmp/admin

FEO
}

#------------------------------------------------------------------------------
#
#  Update the finish.sh file to install packages and patches during
#  the Solaris installation
#
#------------------------------------------------------------------------------

sub update_finish {

  my ($finish, $mountpoint) = @_;

  my $first = "TRUE";
  my $i;

  if (! $DO_ADDON) {
    return;
  }

  $PHASE = "I_MEN";
  COMMON::print_stage_node("Add-on during Solaris installation");

  open (FINISH, ">>$finish") || 
    COMMON::error("Unable to update $finish file") ;

  # PACKAGES

  for (my $i = 0 ; $i < scalar(@PACKAGE_LIST) ; $i++) {
    if (to_be_processed("PACKAGE",
			$PACKAGE_LIST[$i]{phase}, $PACKAGE_LIST[$i]{men},
			$PACKAGE_LIST[$i]{nmen}, $PACKAGE_LIST[$i]{scope},
			$PACKAGE_LIST[$i]{dir}, $PACKAGE_LIST[$i]{subdir}) 
        eq "TRUE") {

      my $reference = $PACKAGE_LIST[$i]{reference};

      if ($CHANGE eq "YES") {
	COMMON::warning("scope $PACKAGE_LIST[$i]{scope} changed to $SCOPE for $reference when installing on MEN\n");
      }
      
      # if it's the first element, open the file
      # and create the admin file
      if ($first eq "TRUE") {
        update_finish_admin();
        print FINISH "$RMKDIR -p $mountpoint\n";
        $first = "FALSE";
      }
            
      COMMON::print_action("Adding commands for package $reference in finish.sh");
      # export the directory on the installation server
	  if ($MANAGE_SHARE eq "TRUE") {
        COMMON::share($DIR);
      }
      
      print FINISH "#\n";
      print FINISH "# package $reference\n";
      print FINISH "#\n";
      print FINISH "$RMOUNT $SERVER_IP:/$DIR $mountpoint\n";
      print FINISH "$RPKGADD -M -d $mountpoint/$SUBDIR -R /a -a /a/tmp/admin $reference\n";
      print FINISH "$RUMOUNT $mountpoint\n";
    } # if
  } #for

  # PATCHES

  for (my $i = 0 ; $i < scalar(@PATCH_LIST) ; $i++) {
    if (to_be_processed("PATCH",
			$PATCH_LIST[$i]{phase}, $PATCH_LIST[$i]{men},
			$PATCH_LIST[$i]{nmen}, $PATCH_LIST[$i]{scope},
			$PATCH_LIST[$i]{dir}, $PATCH_LIST[$i]{subdir}) 
        eq "TRUE") {

      my $reference = $PATCH_LIST[$i]{reference};

      if ($CHANGE eq "YES") {
        COMMON::warning("scope $PATCH_LIST[$i]{scope} changed to $SCOPE for $reference when installing on MEN\n");
      }
      
      # if it's the first element, open the file
      # and create the admin file
      if ($first eq "TRUE") {
        update_finish_admin();
        print FINISH "$RMKDIR -p $mountpoint\n";
        $first = "FALSE";
      }
      
      
      COMMON::print_action("Adding commands for patch $reference in finish.sh");
      # export the directory on the installation server
	  if ($MANAGE_SHARE eq "TRUE") {
        COMMON::share($DIR);
      }

      print FINISH "#\n";
      print FINISH "# patch $reference\n";
      print FINISH "#\n";
      print FINISH "$RMOUNT $SERVER_IP:/$DIR $mountpoint\n";
      print FINISH "$RPATCHADD -M $mountpoint/$SUBDIR -R /a $reference\n";
      print FINISH "$RUMOUNT $mountpoint\n";
    } # if
  } # for

  if ($first eq "FALSE") {
    # at least on element: mounting point has been created
      print FINISH "$RRMDIR $mountpoint\n";
    }
  close(FINISH);
}

{
}




