#!/usr/bin/perl
# ------------------------------------------------------------------------------
#   ident "@(#)nmen_diskless.pl 1.13 03/04/18 SMI"
# ------------------------------------------------------------------------------
#
#  Copyright 2001-2003 Sun Microsystems, Inc. All rights reserved.
#
# ------------------------------------------------------------------------------

package NMEN_DISKLESS;

use variables;

my @IP_LIST;

#------------------------------------------------------------------------------
#
#  INIT action
#
#------------------------------------------------------------------------------

sub save_files {

  ($NODE_ID) = @_;

  if ($NODE_ID ne $MEN1_ID) {
    return ;
  }

  my $node_ip = COMMON::get_external_ip($NODE_ID);

  COMMON::print_stage_node("Saving /etc/ethers and /etc/hosts");
  COMMON::save_file($node_ip, "/etc/ethers", "ETC_ETHERS_$NODE_ID");
  COMMON::save_file($node_ip, "/etc/hosts",  "ETC_HOSTS_$NODE_ID");
}

#------------------------------------------------------------------------------
#
#  INIT action
#
#------------------------------------------------------------------------------

sub restore_files {

  ($NODE_ID) = @_;

  if ($NODE_ID ne $MEN1_ID) {
    return ;
  }

  my $node_ip = COMMON::get_external_ip($NODE_ID);

  COMMON::print_stage_node("Restoring /etc/ethers and /etc/hosts");
  COMMON::restore_file($node_ip, "/etc/ethers", "ETC_ETHERS_$NODE_ID");
  COMMON::restore_file($node_ip, "/etc/hosts",  "ETC_HOSTS_$NODE_ID");
}
#------------------------------------------------------------------------------
#
# get the list of IP addresses defined by "ifconfig -a"
#
#------------------------------------------------------------------------------

sub get_defined_ip {

  my ($node_ip) = @_;

  my $remote_file = "/tmp/ifconfig.out";
  my $local_file  = "$WORKING_DIR/ifconfig.out";

  COMMON::remote_exec($node_ip, "ifconfig -a > $remote_file");
  COMMON::get_file($node_ip, $remote_file, $local_file);

  # extract all the IP address from the output
  open(FI,  $local_file);
  while (<FI>) {
    chomp($_);
    if ($_ =~ m/inet ([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*)/o) {
      push @IP_LIST, $1;
    }
  }

  close(FI);
}

#------------------------------------------------------------------------------
#
# update /etc/hosts with the undefined IP address
#
#------------------------------------------------------------------------------

sub define_ip {

  my ($node_ip) = @_;
  my $ip;

  my $local_hosts = "$WORKING_DIR/hosts.tmp";

  # hostname with an underscore is not authorized: no possible conflict
  # with an already defined one

  COMMON::get_file($node_ip, "/etc/hosts", $local_hosts);

  for ($i = 0; $i < scalar(@IP_LIST) ; $i++) {
    $ip = $IP_LIST[$i];
    # look for the exact IP address (not starting with the same value)
    $result = qx/$GREP \"$ip\[^0-9\]\" $local_hosts/;
    if ($result eq "") {
      qx/$ECHO $ip _dummy$i >> $local_hosts/;
    }
  }

  COMMON::put_file($node_ip, $local_hosts, "/etc/hosts");

}


#------------------------------------------------------------------------------
#
# set the vice-master for a smdiskless environment
#
# - copy /etc/ethers from master to this node
# - complete /etc/bootparmas with diskless entries
#
#------------------------------------------------------------------------------

sub men2_smdiskless_env {

   my ($men1_node_id, $men2_node_id) = @_;

   my $vice_master_name = COMMON::build_node_name($men2_node_id, "");
   my $master_name= COMMON::build_node_name($men1_node_id, "");

   my $men2_ip = COMMON::get_external_ip($men2_node_id);
   my $men1_ip = COMMON::get_external_ip($men1_node_id);

   my $local_bootparams ="$WORKING_DIR/bootparams.tmp";
   my $remote_bootparams = "/etc/bootparams";

   # get localy from men1 node the /etc/bootparams file and update it to have 
   # second node id in it.

   COMMON::get_file($men1_ip, "$remote_bootparams", "$local_bootparams");
   COMMON::replace_in_file($local_bootparams, "$master_name", "$vice_master_name");
      
   #
   # remote copy bootparams to second node
   #
   
   COMMON::put_file($men2_ip, $local_bootparams, $remote_bootparams);
   
}

#-------------------------------------------------------------------------------
#
# common part for each diskless
#
#-------------------------------------------------------------------------------

sub common_per_diskless {

  my ($proto_ip, $diskless_id) = @_;

  my $setting_file="/etc/default/init" ;

  my $node_name = COMMON::build_node_name($diskless_id, "");
  
  # we copy the /etc/default/init to set the timezone and other variables
  COMMON::remote_exec ( "$proto_ip", "$RCOPY $setting_file /export/root/${node_name}${setting_file}" ) || 
    COMMON::error ( "Unable to cp $setting_file into /export/root/${node_name}${setting_file}" );
  
}

#-------------------------------------------------------------------------------
#
# Declare Solaris diskless client on MEN1
#
#-------------------------------------------------------------------------------

sub smdiskless {

   my ($proto_node_id) = @_;
   my $proto_ip   = COMMON::get_external_ip($proto_node_id);
	
   my $node_ip_template;
   my $node_eth_template;
   my $node_eth;
   my $node_name_template;
   my $cmd;

   # to solve the smdiskless problem (error if an interface has its IP
   # address not defined in the /etc/hosts file
   # -> detect all this IP addresses and define them temporarly
   get_defined_ip($proto_ip);
   define_ip($proto_ip);

   #
   # We issue the smdiskless for all diskless nodes
   #
   for my $node (0..$#NODE_LIST) {
     
     my $node_id = $NODE_LIST[$node]{id};

     $node_ip_template = COMMON::build_node_ip($node_id, "NIC0");
     $node_eth = $NODE_LIST[$node]{eth0};
     if ($node_eth eq "") {
       # MAC address not configured: DHCP mode is dynamic but argument is required for smdiskless
       # depends on the node rank to have a different MAC address per diskless
       $hexa = sprintf("%2.2x", $node_id);
       $node_eth = "00:00:00:00:00:$hexa" ;
     }

     if (($NODE_LIST[$node]{type} eq "DISKLESS") && ($NODE_LIST[$node]{concerned} eq "TRUE")) {
       $node_name_template = COMMON::build_node_name($node_id, "");

       COMMON::print_subaction("adding $node_name_template") ;

       my $timezone = SEQUENCER::get_data("TIMEZONE");

       $cmd = "$RSMDISKLESS add -p $PASSWORD -- -i ${node_ip_template} -e $node_eth -n ${node_name_template} -x os=sparc.${ARCH}.$TARGET_OS -x passwd=$PASSWORD -x tz=$timezone -x locale=$LOCALE";													
       COMMON::remote_exec ( "$proto_ip", $cmd) ||
	 COMMON::error( " Cannot remotely execute $cmd" );

       common_per_diskless($proto_ip, $node_id);
     }
   }
}

#-------------------------------------------------------------------------------
#
# Clean up the diskless environment when using smdiskless
#
#-------------------------------------------------------------------------------

sub smdiskless_cleanup {

   my ($node_id) = @_;
   my $node_ip   = COMMON::get_external_ip($node_id);

   #
   # remove root and swap directory for each diksless
   #
   for $node (0..$#NODE_LIST) {
     if (($NODE_LIST[$node]{type} eq "DISKLESS") && ($NODE_LIST[$node]{concerned} eq "TRUE")) {

       $node_name_template = COMMON::build_node_name($NODE_LIST[$node]{id}, "");

       COMMON::print_subaction("cleaning $node_name_template diskless environment") ;
       COMMON::remote_exec ( "$node_ip", "$RRM -Rf /export/root/$node_name_template") ;
       COMMON::remote_exec ( "$node_ip", "$RRM -Rf /export/swap/$node_name_template") ;
		    	
     }
   }
}

#-------------------------------------------------------------------------------
#
# Declare Solaris diskless client on MEN1 by using netradc
#
#-------------------------------------------------------------------------------

sub netradc {

   my ($proto_node_id) = @_;
   my $proto_ip   = COMMON::get_external_ip($proto_node_id);
	
   my $node_name_template;
   my $cmd;

   my $root_client_root = "/export/root";
   my $root_client_swap = "/export/swap";

   
   # create parent dir
   COMMON::remote_exec ( "$proto_ip", "$RMKDIR -p $root_client_root") ||
     COMMON::error( " Cannot remotely create $root_client_root on $proto_ip" );
   COMMON::remote_exec ( "$proto_ip", "$RMKDIR -p $root_client_swap") ||
     COMMON::error( " Cannot remotely create $root_client_swap on $proto_ip" );

   #
   # We issue the smdiskless for all diskless nodes
   #
   for my $node (0..$#NODE_LIST) {
     
     my $node_id = $NODE_LIST[$node]{id};

     if (($NODE_LIST[$node]{type} eq "DISKLESS") && ($NODE_LIST[$node]{concerned} eq "TRUE")) {
       $node_name_template = COMMON::build_node_name($node_id, "");

       COMMON::print_subaction("adding $node_name_template") ;

       my $client_root = "$root_client_root/$node_name_template";
       my $client_swap = "$root_client_swap/$node_name_template";

       my $timezone = SEQUENCER::get_data("TIMEZONE");

       $cmd = "$RNETRADC add -S $TARGET_OS -p $ARCH -r $client_root -w $client_swap -P $PASSWORD -l $LOCALE -t $timezone $node_name_template";

       COMMON::remote_exec ( "$proto_ip", $cmd) ||
	 COMMON::error( " Cannot remotely execute $cmd" );

       common_per_diskless($proto_ip, $node_id);
     }
   }
}

#------------------------------------------------------------------------------
#
# set the vice-master for a netradc environment
#
#------------------------------------------------------------------------------

sub men2_netradc_env {

   my ($men1_node_id, $men2_node_id) = @_;

   my $vice_master_name = COMMON::build_node_name($men2_node_id, "");
   my $master_name= COMMON::build_node_name($men1_node_id, "");

   my $men2_ip = COMMON::get_external_ip($men2_node_id);
   my $men1_ip =      COMMON::get_external_ip($men1_node_id);

   my $dcclients_dir = "/var/sadm/system/admin/dcclients";

   # get localy from master node the /etc/bootparams file and update it to have 
   # second node id in it.
   
   for my $node (0..$#NODE_LIST) {
     
     my $node_id = $NODE_LIST[$node]{id};

     if (($NODE_LIST[$node]{type} eq "DISKLESS") && ($NODE_LIST[$node]{concerned} eq "TRUE")) {
       $node_name_template = COMMON::build_node_name($node_id, "");

       COMMON::print_subaction("copying client description for $node_name_template") ;
       COMMON::transfer_file($men1_ip, $men2_ip, "$dcclients_dir/$node_name_template");
     }
   }   
}

#-------------------------------------------------------------------------------
#
# Clean up the diskless environment when using netradc
#
#-------------------------------------------------------------------------------

sub netradc_cleanup {

   my ($node_id) = @_;
   my $node_ip   = COMMON::get_external_ip($node_id);

   # remove clients descriptions
   COMMON::remote_exec ( "$node_ip", "$RRM -Rf /var/sadm/system/admin/dcclients/*") ;

   #
   # remove root and swap directory for each diksless
   #
   for $node (0..$#NODE_LIST) {
     if (($NODE_LIST[$node]{type} eq "DISKLESS") && ($NODE_LIST[$node]{concerned} eq "TRUE")) {

       $node_name_template = COMMON::build_node_name($NODE_LIST[$node]{id}, "");

       COMMON::print_subaction("cleaning $node_name_template diskless environment") ;
       COMMON::remote_exec ( "$node_ip", "$RRM -Rf /export/root/$node_name_template") ;
       COMMON::remote_exec ( "$node_ip", "$RRM -Rf /export/swap/$node_name_template") ;
		    	
     }
   }
}

#-------------------------------------------------------------------------------
#
#  Installation of environment per diskless
#
#-------------------------------------------------------------------------------

sub nmen_diskless {

  ($NODE_ID) = @_;

  COMMON::init_and_check([]);

  if  ( $NODE_ID eq $MEN1_ID ) {
    COMMON::print_stage_node("Diskless environment creation") ;

    if ($USE_DCSS eq "YES") {
      netradc($NODE_ID);
    } else {
      smdiskless($NODE_ID);
    }

  } else {
    
    # Complete the diskless environment on the second eligible master node
    # copy /etc/ethers from MEN1
    my $name = COMMON::build_node_name($MEN2_ID, "");
    COMMON::print_stage_node("Completing the diskless environment");
    if ($USE_DCSS eq "YES") {
      men2_netradc_env($MEN1_ID, $NODE_ID);
    } else {
      men2_smdiskless_env($MEN1_ID, $NODE_ID);
    }

  }
}

#-------------------------------------------------------------------------------
#
#  recovery after failure
#
#-------------------------------------------------------------------------------

sub recovery_nmen_diskless {

  ($NODE_ID) = @_;

  COMMON::init_and_check([]);

  if ($NODE_ID eq $MEN1_ID) {

    COMMON::print_recovery_node("Diskless environment clean-up") ;
    if ($USE_DCSS eq "YES") {
      netradc_cleanup($NODE_ID);
    } else {
      smdiskless_cleanup ($NODE_ID) ;
    }
    
  }
}

{
}










