#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_parser.pl 1.14     02/10/31 SMI"
# ------------------------------------------------------------------------------

BEGIN {
   # Cluster model objects
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_cluster.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_group.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_node.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_package.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_software.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_patch.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_domain.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_service.pm";
   # Machine model objects
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_shelf.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_switch.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_port.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_drawer.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_board.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_nic.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_disk.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_slice.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_filesys.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_filesys_nfs.pm";
   # Network model objects
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_ip.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_network.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_router.pm";
   # Config model objects
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_config.pm";
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_file.pm";
   #
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_jumpstart.pl";
}

# ------------------------------------------------------------------------------

package sl_parser;

# -----------------------------------------------------------------------------
# Configuration file name
#
$CLUSTER_CONF = "models/cluster.conf";
$MACHINE_CONF = "models/machine.conf";
$NETWORK_CONF = "models/network.conf";
#
# -----------------------------------------------------------------------------
# Configuration data separator
#
my $CONF_SEP = "\x20";
#
# -----------------------------------------------------------------------------
# Configuration block tags
#
my %BLOCK_TAG = ( 'ELEMENT' => "ELEMENT",
                  'INVOLVE' => "INVOLVE",
	          'BELONG_TO' => "BELONG_TO",
		  'CONTAIN' => "CONTAIN",
		  'USE' => "USE",
		  'RUN' => "RUN",
		  'INCLUDE' => "INCLUDE",
		  'MANAGE' => "MANAGE",
		  'MAP' => "MAP",
		  'CONNECT' => "CONNECT" );
#
# -----------------------------------------------------------------------------
# Cluster model keywords
#
my %CLUSTER_KEYWORDS = ();
@{$CLUSTER_KEYWORDS{'ELEMENT'}} = qw ( cluster );
@{$CLUSTER_KEYWORDS{'INVOLVE'}} = qw ( nodeGroup shelf domain );
@{$CLUSTER_KEYWORDS{'MANDATORY'}} = qw ( cluster nodeGroup shelf domain );
#
my %DOMAIN_KEYWORDS = ();
@{$DOMAIN_KEYWORDS{'ELEMENT'}} = qw ( domain id );
@{$DOMAIN_KEYWORDS{'INVOLVE'}} = qw ( network router );
@{$DOMAIN_KEYWORDS{'USE'}} = qw ( ip );
@{$DOMAIN_KEYWORDS{'MANDATORY'}} = qw ( domain );
#
my %NODE_KEYWORDS = ();
@{$NODE_KEYWORDS{'ELEMENT'}} = qw ( node id );
@{$NODE_KEYWORDS{'USE'}} = qw ( board disk );
@{$NODE_KEYWORDS{'MANDATORY'}} = qw ( node id board );
#
my %GROUP_KEYWORDS = ();
@{$GROUP_KEYWORDS{'ELEMENT'}} = qw ( nodeGroup type os arch );
@{$GROUP_KEYWORDS{'INCLUDE'}} = qw ( nodeGroup node );
@{$GROUP_KEYWORDS{'RUN'}} = qw ( service );
@{$GROUP_KEYWORDS{'MANDATORY'}} = qw ( nodeGroup node type os arch service );
#
my %SOFTWARE_KEYWORDS = ();
@{$SOFTWARE_KEYWORDS{'ELEMENT'}} = qw ( software type name version 
                                        installScript );
@{$SOFTWARE_KEYWORDS{'MANDATORY'}} = qw ( software type ); 
#
my %PACKAGE_KEYWORDS = ();
@{$PACKAGE_KEYWORDS{'ELEMENT'}} = qw ( software mode type base name  
                                      adminFile responseFile version );
@{$PACKAGE_KEYWORDS{'MANDATORY'}} = qw ( software type mode ); 
#
my %PATCH_KEYWORDS = ();
@{$PATCH_KEYWORDS{'ELEMENT'}} = qw ( software type mode name version );
@{$PATCH_KEYWORDS{'MANDATORY'}} = qw ( software type mode ); 
#
# -----------------------------------------------------------------------------
# Machine model keywords
#
my %SHELF_KEYWORDS = ();
@{$SHELF_KEYWORDS{'ELEMENT'}} = qw ( shelf );
@{$SHELF_KEYWORDS{'CONTAIN'}} = qw ( drawer switch );
@{$SHELF_KEYWORDS{'MANDATORY'}}= qw ( shelf drawer );
#
my %DRAWER_KEYWORDS = ();
@{$DRAWER_KEYWORDS{'ELEMENT'}} = qw ( drawer type );
@{$DRAWER_KEYWORDS{'CONTAIN'}} = qw ( board disk );
@{$DRAWER_KEYWORDS{'MANDATORY'}} = qw ( drawer type );
#
my %SWITCH_KEYWORDS = ();
@{$SWITCH_KEYWORDS{'ELEMENT'}} = qw ( switch type );
@{$SWITCH_KEYWORDS{'CONTAIN'}} = qw ( port );
@{$SWITCH_KEYWORDS{'USE'}} = qw ( ip );
@{$SWITCH_KEYWORDS{'MANDATORY'}} = qw ( switch type );
#
my %PORT_KEYWORDS = ();
@{$PORT_KEYWORDS{'ELEMENT'}} = qw ( port number nic );
@{$PORT_KEYWORDS{'CONNECT'}} = qw ( nic );
@{$PORT_KEYWORDS{'MANDATORY'}} = qw ( port number nic );
#
my %BOARD_KEYWORDS = ();
@{$BOARD_KEYWORDS{'ELEMENT'}} = qw ( board type arch nic class vendorType
                                     clientId );
@{$BOARD_KEYWORDS{'USE'}} = qw ( nic );
@{$BOARD_KEYWORDS{'MANDATORY'}} = qw ( board arch type );
#
my %NIC_KEYWORDS = ();
@{$NIC_KEYWORDS{'ELEMENT'}} = qw ( nic device type address 
                                   class role );
@{$NIC_KEYWORDS{'USE'}} = qw ( ip );
@{$NIC_KEYWORDS{'MANDATORY'}} = qw ( nic device type class role );
#
my %DISK_KEYWORDS = ();
@{$DISK_KEYWORDS{'ELEMENT'}} = qw ( disk device size type );
@{$DISK_KEYWORDS{'CONTAIN'}} = qw ( slice );
@{$DISK_KEYWORDS{'MANDATORY'}} =  qw ( disk device type size slice );
#
my %SLICE_KEYWORDS = ();
@{$SLICE_KEYWORDS{'ELEMENT'}} = qw ( slice number type size 
                                     role rawDev blockDev );
@{$SLICE_KEYWORDS{'USE'}} = qw ( slice );
@{$SLICE_KEYWORDS{'MAP'}} = qw ( filesys );
@{$SLICE_KEYWORDS{'MANAGE'}} = qw ( replicatedSlice );
@{$SLICE_KEYWORDS{'MANDATORY'}} = qw ( slice number type size rawDev blockDev );
#
my %FILESYS_KEYWORDS = ();
@{$FILESYS_KEYWORDS{'ELEMENT'}} = qw ( filesys role type fsck size
                                       mntPt remMntPt mntBt mntOpt );
@{$FILESYS_KEYWORDS{'CONTAIN'}} = qw ( exportedFileSys );
@{$FILESYS_KEYWORDS{'MANDATORY'}} = qw ( filesys role type size );
#
my %FILESYS_NFS_KEYWORDS = ();
@{$FILESYS_NFS_KEYWORDS{'ELEMENT'}} = qw ( filesys role type mntPt 
                                           mntBt remMntPt mntOpt );
@{$FILESYS_NFS_KEYWORDS{'MANDATORY'}} = qw ( filesys role type 
                                             mntPt remMntPt );
#
# -----------------------------------------------------------------------------
# Network model keywords
#
my %NETWORK_KEYWORDS = ();
@{$NETWORK_KEYWORDS{'ELEMENT'}} = qw ( network type number netmask );
@{$NETWORK_KEYWORDS{'MANDATORY'}} = qw ( network type number netmask );
#
my %IP_KEYWORDS = ();
@{$IP_KEYWORDS{'ELEMENT'}} = qw ( ip address type alias );
@{$IP_KEYWORDS{'BELONG_TO'}} = qw ( network );
@{$IP_KEYWORDS{'MANDATORY'}} = qw ( ip address network type );
#
my %ROUTER_KEYWORDS = ();
@{$ROUTER_KEYWORDS{'ELEMENT'}} = qw ( router ip );
@{$ROUTER_KEYWORDS{'USE'}} = qw ( ip );
@{$ROUTER_KEYWORDS{'MANDATORY'}} = qw ( router ip );

#
# -----------------------------------------------------------------------------
# Nodes groups & services config model keywords
#
my %CONFIG_KEYWORDS = ();
@{$CONFIG_KEYWORDS{'ELEMENT'}} = qw ( config next );
@{$CONFIG_KEYWORDS{'INVOLVE'}} = qw ( file );
@{$CONFIG_KEYWORDS{'MANDATORY'}} = qw ( config file );
#
my %FILE_KEYWORDS = ();
@{$FILE_KEYWORDS{'ELEMENT'}} = qw ( file type location runLevel );
@{$FILE_KEYWORDS{'MANDATORY'}} = qw ( file type location );
#
# -----------------------------------------------------------------------------
# Array of Objects elaborated from configuration data
#
my @CLUSTER_OBJECTS = ();
my @SHELF_OBJECTS = ();
my @DRAWER_OBJECTS = ();
my @SWITCH_OBJECTS = ();
my @PORT_OBJECTS = ();
my @NODE_OBJECTS = ();
my @ALT_NODE_OBJECTS = ();
my @BOARD_OBJECTS = ();
my @NIC_OBJECTS = ();
my @DISK_OBJECTS = ();
my @SLICE_OBJECTS = ();
my @FILESYS_OBJECTS = ();
my @FILESYS_NFS_OBJECTS = ();
my @PACKAGE_OBJECTS = ();
my @PATCH_OBJECTS = ();
my @SOFTWARE_OBJECTS = ();
my @GROUP_OBJECTS = ();
my @NETWORK_OBJECTS = ();
my @DOMAIN_OBJECTS = ();
my @IP_OBJECTS = ();
my @ROUTER_OBJECTS = ();
my @CONFIG_OBJECTS = ();
my @FILE_OBJECTS = ();
#
# -----------------------------------------------------------------------------
# Object ID counter used for dynamic ID allocation
#
my $OID_COUNTER = 1;
#
# -----------------------------------------------------------------------------
# Hash table holding object IDs for unicity checks. Used at configuration
# load-time to detect redundant configuration objects (see _createXxx routines).
#
my %CONFIGURATION_OBJECTS = ();
#
# -----------------------------------------------------------------------------
# Hash table holding mac addresses for unicity checks. Used at configuration
# load-time to detect redundant configuration objects (see _createXxx routines).
#
my %ADDRESSES = ();
#
# -----------------------------------------------------------------------------
# Hash tables holding object IDs for cross-reference checks. Must be used for
# containment relationships checks (see _resolveXxxDep routines).
#
my %CONTAINER_OBJECTS = ();
# Additional hash for the double cross-reference 
# ( e.g. board referenced by the node and also referenced by the drawer )
my %ALT_CONTAINER_OBJECTS = ();
#
# -----------------------------------------------------------------------------
# Global variables for syntaxic error messages
#
my $LINE_COUNT = 0;
my %ELEM_STARTING_LINE = ();
my $CURRENT_ELEM = 0;
my $CURRENT_BLOCK = "";
my $CURRENT_RAW = "";
my $CONFIG_FILE = "";

# ------------------------------------------------------------------------------
# Tag used in auto name generation (e.g. for service config. files) from
# generic config file templates
#
my $AUTO_NAME_TAG = "<-AUTO_NAME->";
my $FAIL_FLAG = "fail_tolerant";

# ------------------------------------------------------------------------------
#
sub _checkKeyword {
   my ( $value, $configString, @valueLst ) = @_;
   my $validValues = undef;
   &sl_traces::trace( 3, "- sl_parser::_checkKeyword - Checking < ${value} >" );
   foreach $val ( @valueLst ) {
     return 1 if ( $value eq $val );
     if ( defined $validValues ) {
        $validValues = $validValues." | ".$val;
     }
     else {
        $validValues = $val;
     }
   }
   &sl_traces::error( "Configuration data ".
                 "< ${value} > is invalid".
		 "\nValid values are < $validValues >".
		 "\nCurrent element string is < ${CURRENT_RAW} >".
		 "\nCurrent block is < ${CURRENT_BLOCK} >".
		 "\nCurrent block string is < ${configString} >".
		 "\nCurrent starting element line is ".
		 "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		 "\nCurrent file is < ${CONFIG_FILE} >" , 
		 "- sl_parser::_checkKeyword - ".__LINE__ );
}

# ------------------------------------------------------------------------------
#
sub _checkEnum {
   my ( $configData, $configValue, $class, @valuesList ) = @_;
   my $validValues = undef;
   &sl_traces::trace( 3, "- sl_parser::_checkEnum - Checking value of ".
                      "< ${configData} >" );
   return 1 unless ( defined $configValue );
   foreach $value ( @valuesList ) {
     return 1 if ( $configValue eq $value );
     if ( defined $validValues ) {
        $validValues = $validValues." | ".$value
	   unless $class->isHidden( $value );
     }
     else {
        $validValues = $value
	   unless $class->isHidden( $value );
     }
   }
   &sl_traces::error( "Configuration data ".
                 "< ${configData} > has an invalid value < $configValue >".
		 "\nValid values are < $validValues >".
		 "\nCurrent element string is < ${CURRENT_RAW} >".
		 "\nCurrent block is < ${CURRENT_BLOCK} >".
		 "\nCurrent starting element line is ".
		 "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		 "\nCurrent file is < ${CONFIG_FILE} >" , 
		 "- sl_parser::_checkEnum - ".__LINE__ );
}

# ------------------------------------------------------------------------------
#
sub _checkInt {
   my ( $item, @idLst ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_checkInt - Checking < $item > value" );
   foreach $id ( @idLst ) {
      if ( $id =~ /[^0-9]/ ) {
	 &sl_traces::error( "Integer value < $id > ".
        	 "from configuration data < $item > must be numeric".
		 "\nCurrent element string is < ${CURRENT_RAW} >".
		 "\nCurrent block is < ${CURRENT_BLOCK} >".
		 "\nCurrent starting element line is ".
		 "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		 "\nCurrent file is < ${CONFIG_FILE} >" , 
		 "- sl_parser::_checkInt - ".__LINE__ );
      }
   }
}

# ------------------------------------------------------------------------------
#
sub _checkRange {
   my ( $item, $itemMin, $itemMax, @itemLst ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_checkRange - Checking < $item > value" );
   &_checkInt( $item, @itemLst );
   foreach ( @itemLst ) {
      if ( ( $_ < $itemMin ) or ( $_ > $itemMax ) ) {
	 &sl_traces::error( "Enumeration value < ${_} > ".
        	 "from configuration data < $item > is out of range.".
		 "\nMinimum value is < ${itemMin} >".
		 "\nMaximum value is < ${itemMax} >".
		 "\nCurrent element string is < ${CURRENT_RAW} >".
		 "\nCurrent block is < ${CURRENT_BLOCK} >".
		 "\nCurrent starting element line is ".
		 "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		 "\nCurrent file is < ${CONFIG_FILE} >" , 
		 "- sl_parser::_checkRange - ".__LINE__ );
      }
   }
}

# ------------------------------------------------------------------------------
#
sub _prepareConfigFile {
   my ( $file, $autoValue ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_prepareConfigFile - Preparing ".
                      "configuration file < ${file} > with < ${autoValue} >" );
   my $preparedFile = `$ENV{'SMCT_BASENAME'} $file`; 
   chop( $preparedFile );
   $preparedFile = "${preparedFile}.${autoValue}";
   my $command = "$ENV{'SMCT_CAT'} ${file} | ".
                 "$ENV{'SMCT_SED'} -e 's,${AUTO_NAME_TAG},${autoValue},g'".
                 "> $ENV{'SMCT_TMP_DIR'}/${preparedFile}";
   &sl_utils::execCmd( $command );
   return $preparedFile;
}

# ------------------------------------------------------------------------------
#
sub _lookUpByName {
   my ( $name, $item, $type, @list ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_lookUpByName - Looking for ".
                      "< ${item} = ${name} >" );
   foreach ( @list ) {
      my $objName = $_->getName;	
      if ( $_->{'Name'} eq $name ) {
         $oid = $_->iD;
	 # Increment the references to the object
	 &_refObject( $_->getName );
         return $_;
      }
   }
   &sl_traces::error( "Cannot retrieve ".
                      "< $item = $name > from < $type > elements",
		      "- sl_parser::_lookUpByName - ".__LINE__ )
      unless ( $type eq $FAIL_FLAG );
   return undef;   
}

# ------------------------------------------------------------------------------
#
sub _sortParamLst {
   my ( $elementType, $elementName, @lst ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_sortParamLst - sorting ".
                      " list of < ${elementType} > parameters" );
   # Look for $elementName param in the list
   my @startLst = ();
   my @endLst = ();
   foreach $param ( @lst ) {
      if ( $param eq $elementName ) {
         push( @startLst, $param );
      }
      else {
         push( @endLst, $param );
      }
   }
   return ( @startLst, @endLst );
}

# ------------------------------------------------------------------------------
#
sub _checkMandatoryParams {
   my ( $configRaw, %paramLst ) = @_;
   my %classKeywords = %{$paramLst{'KEYWORDS'}};
   my @mandatoryParamLst = @{$classKeywords{'MANDATORY'}};
   my $notFound = "";
   &sl_traces::trace( 3, "- sl_parser::_checkMandatoryParams - ".
                "Checking mandatory parameters for\n< ${configRaw} >" );
   $configRaw =~ 
      /ELEMENT${CONF_SEP}([^${CONF_SEP}]+)${CONF_SEP}([^${CONF_SEP}]+)/;
   my $elementType = $1;
   my $elementName = $2;
   foreach $param ( @mandatoryParamLst ) {
      if ( defined $paramLst{$param} ) {
         @lst = @{$paramLst{$param}};
         if ( scalar @lst > 0 ) {
	    if ( ( $elementType eq $param ) and ( scalar @lst > 1 ) ) {
	       @{$paramLst{$param}} = 
	          &_sortParamLst( $elementType, $elementName, @lst );
	    }
	    next;
	 }
      }
      $notFound = $notFound." ".$param;
   }
   if ( $notFound ne "" ) {
      &sl_traces::error( "Mandatory ".
                 "parameter(s) < ${notFound} > is/are not defined". 
		 "\nCurrent element string is < ${CURRENT_RAW} >".
		 "\nCurrent block is < ${CURRENT_BLOCK} >".
		 "\nCurrent starting element line is ".
		 "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		 "\nCurrent file is < ${CONFIG_FILE} >" , 
		 "- sl_parser::_checkMandatoryParams - ".__LINE__ );
   }
}

# ------------------------------------------------------------------------------
# 
sub _match {
   my ( $item, $keyword, @itemLst  ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_match - Looking for keyword / value pair ".
                "< ${keyword} - ${item} >" );
   foreach ( @itemLst ) {
      return 1 if ( $item eq $_ );
   }
   return 0;
}

# ------------------------------------------------------------------------------
# 
sub _setValue {
   my ( $item, $keyword, $configRaw, @{itemLst}  ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_setValue - Setting keyword / value pair ".
                "< ${keyword} - ${item} >" );
   if ( &_match( $item, $keyword, @{itemLst} ) ) {
      &sl_traces::error( "The keyword / value pair ".
                 "< ${keyword} - ${item} > is already defined\n".
		 "in config row < ${configRaw} >".
		 "\nCurrent element string is < ${CURRENT_RAW} >".
		 "\nCurrent block is < ${CURRENT_BLOCK} >".
		 "\nCurrent starting element line is ".
		 "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		 "\nCurrent file is < ${CONFIG_FILE} >" , 
		 "- sl_parser::_setValue- ".__LINE__ );
   }
   else {
      return $item;
   }
}

# ------------------------------------------------------------------------------
# 
sub _split {
   my ( $configRaw, @keywordLst ) = @_;
   my %configVal = ();
   # Suppress Block delimiter from the config data
   foreach $tag ( keys %BLOCK_TAG ) {
      $configRaw =~ s/${tag}${CONF_SEP}//g;
   }
   my @itemLst = split( $CONF_SEP, $configRaw );
   my $keyword = undef;
   &sl_traces::trace( 3, "- sl_parser::_split - Getting keyword/value pairs from\n".
                "< $configRaw >" );
   foreach $item ( @itemLst ) {
      if ( defined $keyword ) {
         # Manage each keyword value(s) as a list
	 push( @{$configVal{$keyword}}, 
	   &_setValue( $item, $keyword, $configRaw, @{$configVal{$keyword}} ) );
	 # Reset keyword for next key/value pair
	 $keyword = undef;
      }
      else {
         # A keyword is expected
         $keyword = $item if ( &_checkKeyword( $item, $configRaw, @keywordLst ) );
      }
   }
   return %configVal;
}

# ------------------------------------------------------------------------------
#
sub _compress {
   my ( @configData ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_compress - Compressing configuration ".
                      "data" );
   my @compressedData = ();
   my $buffer = "";
   my $elem = 0;
   foreach $configRaw ( @configData ) {
      $LINE_COUNT++;
      # Normalize config data sring
      chomp( $configRaw );
      $configRaw =~ s/\n//g;
      $configRaw =~ s/^\s*//;
      $configRaw =~ s/\s*$//; 
      $configRaw =~ s/\s+/${CONF_SEP}/g;
      # Skip comments
      next if ( $configRaw =~ /^#/ );
      next if ( $configRaw eq "" );
      # Look for block delimiters
      if ( $configRaw =~ /^$BLOCK_TAG{'ELEMENT'}${CONF_SEP}/ ) {
	 $ELEM_STARTING_LINE{$CURRENT_ELEM} = $LINE_COUNT;
	 $CURRENT_ELEM++;
         if ( ( $elem ) or ( $buffer ne "" ) ) {
	    # Store compressed element
 	    push( @compressedData, $buffer );
	    $elem = 0;
	 }
	 else {
            # Start compression of a new config element
	    $elem = 1;
	 }
	 $buffer = $configRaw;
      }
      else {
         $buffer .= $CONF_SEP.$configRaw;
      }
   }
   # Store last element of the file (if any)
   if ( ( $elem ) or ( $buffer ne "" ) ) {
      # Store compressed element
      push( @compressedData, $buffer );
   }
   return @compressedData;
}

# ------------------------------------------------------------------------------
#
sub _extractBlocksData {
   my ( $configRaw ) = @_;
   my $currentBlock = "";
   my %blocksData = ();
   my $newBlock = 0;
   &sl_traces::trace( 3, "- sl_parser::_extractBlocksData - Extracting blocks ".
                      "from configuration data\n${configRaw}" );
   my @configData = split( $CONF_SEP, $configRaw );
   foreach $data ( @configData ) {
      $newBlock = 0;
      foreach $tag ( keys %BLOCK_TAG ) {
         if ( $data eq $tag ) {
            # Select a block when a block delimiter is found
	    $currentBlock = $data;
            $newBlock = 1;
	    $blocksData{$currentBlock} = "" 
	       unless ( defined $blocksData{$currentBlock} );
	    last;
	 }
      } 
      unless ( $newBlock ) {
	 # Grab the data of the block and store it
	 $blocksData{$currentBlock} .= $data.$CONF_SEP;
      }
   }
   return %blocksData;
}

# ------------------------------------------------------------------------------
#
sub _extract {
   my ( @configData ) = @_;
   my %configVal = ();
   my %classKeywords = ();
   my %blocksData = ();
   &sl_traces::trace( 3, "- sl_parser::_extract - Extracting configuration ".
                      "data" );
   my @compressedData = &_compress( @configData );
   $CURRENT_ELEM = 0;
   foreach $configRaw ( @compressedData ) {
      $CURRENT_RAW = $configRaw;
      %configVal = ();
      # Extract the config element of each block for further processing
      %blocksData = &_extractBlocksData( $configRaw );
      # Parse config data string
      %configVal = ();
      %classKeywords = ();
      if ( $configRaw =~ /^ELEMENT cluster${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %CLUSTER_KEYWORDS;
	 $configVal{'CLASS'} = "sl_cluster";
	 $configVal{'ELEMENT_TYPE'} = "cluster";
      }
      elsif ( $configRaw =~ /^ELEMENT shelf${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %SHELF_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_shelf";
	 $configVal{'ELEMENT_TYPE'} = "shelf";
      }
      elsif ( $configRaw =~ /^ELEMENT drawer${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %DRAWER_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_drawer";
	 $configVal{'ELEMENT_TYPE'} = "drawer";
      }
      elsif ( $configRaw =~ /^ELEMENT node${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %NODE_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_node";
	 $configVal{'ELEMENT_TYPE'} = "node";
      }
      elsif ( $configRaw =~ /^ELEMENT board${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %BOARD_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_board";
	 $configVal{'ELEMENT_TYPE'} = "board";
      }
      elsif ( $configRaw =~ /^ELEMENT nic${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %NIC_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_nic";
	 $configVal{'ELEMENT_TYPE'} = "nic";
      }
      elsif ( $configRaw =~ /^ELEMENT switch${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %SWITCH_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_switch";
	 $configVal{'ELEMENT_TYPE'} = "switch";
      }
      elsif ( $configRaw =~ /^ELEMENT port${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %PORT_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_port";
	 $configVal{'ELEMENT_TYPE'} = "port";
      }
      elsif ( $configRaw =~ /^ELEMENT disk${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %DISK_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_disk";
	 $configVal{'ELEMENT_TYPE'} = "disk";
      }
      elsif ( $configRaw =~ /^ELEMENT filesys${CONF_SEP}/ ) {
         # Extract the filesys type to select the appropriate object class
	 $configRaw =~ /.+type${CONF_SEP}(\S+)${CONF_SEP}/;
	 my $fileSysType = $1 if ( defined $1 );
	 if ( defined $fileSysType ) {
	    if ( $fileSysType eq ${sl_filesys::FILESYS_TYPE{'NFS'}} ) {
	       %{$configVal{'KEYWORDS'}} = %FILESYS_NFS_KEYWORDS;
	       $configVal{'CLASS'} =  "sl_filesys_nfs";
	       $configVal{'ELEMENT_TYPE'} = "filesys";
	    }
	    else {
	       %{$configVal{'KEYWORDS'}} = %FILESYS_KEYWORDS;
	       $configVal{'CLASS'} =  "sl_filesys";
	       $configVal{'ELEMENT_TYPE'} = "filesys";
	    }
	 }
	 else {
            &sl_traces::error( "filesys type is ".
	                       "not defined < $configRaw >".
			       "\nCurrent line is < ${LINE_COUNT} >".
		               "\nCurrent file is < ${CONFIG_FILE} >", 
			       "- sl_parser::_extract - ".__LINE__ );
            
	 }
      }
      elsif ( $configRaw =~ /^ELEMENT slice${CONF_SEP}/ ) {
         $configRaw =~ /^ELEMENT slice${CONF_SEP}/;
	 %{$configVal{'KEYWORDS'}} = %SLICE_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_slice";
	 $configVal{'ELEMENT_TYPE'} = "slice";
      }
      elsif ( $configRaw =~ /^ELEMENT software${CONF_SEP}/ ) {
         # Extract the software type to select the appropriate object class
	 $configRaw =~ /.+type${CONF_SEP}(\S+)/;
	 my $softwareType = $1 if ( defined $1 );
	 if ( defined $softwareType ) {
	    if ( $softwareType eq $sl_software::SOFTWARE_TYPE{'PACKAGE'} ) {
	       %{$configVal{'KEYWORDS'}} = %PACKAGE_KEYWORDS;
	       $configVal{'CLASS'} =  "sl_package";
	       $configVal{'ELEMENT_TYPE'} = "software";
	    }
	    elsif ( 
	    ( $softwareType eq $sl_software::SOFTWARE_TYPE{'PRE_PATCH'} ) or
	    ( $softwareType eq $sl_software::SOFTWARE_TYPE{'POST_PATCH'} ) ) {
	       %{$configVal{'KEYWORDS'}} = %PATCH_KEYWORDS;
	       $configVal{'CLASS'} =  "sl_patch";
	       $configVal{'ELEMENT_TYPE'} = "software";
	    }
	    else {
	       %{$configVal{'KEYWORDS'}} = %SOFTWARE_KEYWORDS;
	       $configVal{'CLASS'} =  "sl_software";
	       $configVal{'ELEMENT_TYPE'} = "software";
	    }
	 }
	 else {
            &sl_traces::error( "software type is ".
	                       "not defined < $configRaw >".
			       "\nCurrent line is < ${LINE_COUNT} >".
		               "\nCurrent file is < ${CONFIG_FILE} >", 
			       "- sl_parser::_extract - ".__LINE__ );
	 }
      }
      elsif ( $configRaw =~ /^ELEMENT nodeGroup${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %GROUP_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_group";
	 $configVal{'ELEMENT_TYPE'} = "group";
      }
      elsif ( $configRaw =~ /^ELEMENT network${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %NETWORK_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_network";
	 $configVal{'ELEMENT_TYPE'} = "network";
      }
      elsif ( $configRaw =~ /^ELEMENT domain${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %DOMAIN_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_domain";
	 $configVal{'ELEMENT_TYPE'} = "domain";
      }
      elsif ( $configRaw =~ /^ELEMENT ip${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %IP_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_ip";
	 $configVal{'ELEMENT_TYPE'} = "ip";
      }
      elsif ( $configRaw =~ /^ELEMENT router${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %ROUTER_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_router";
	 $configVal{'ELEMENT_TYPE'} = "router";
      }
      elsif ( $configRaw =~ /^ELEMENT config${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %CONFIG_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_config";
	 $configVal{'ELEMENT_TYPE'} = "config";
      }
      elsif ( $configRaw =~ /^ELEMENT file${CONF_SEP}/ ) {
	 %{$configVal{'KEYWORDS'}} = %FILE_KEYWORDS;
	 $configVal{'CLASS'} =  "sl_file";
	 $configVal{'ELEMENT_TYPE'} = "file";
      }
      else {
         # Detect unknown config data string
         &sl_traces::error( "Unknown ".
	                    "configuration pattern < $configRaw >".
			    "\nCurrent element string is < ${CURRENT_RAW} >".
			    "\nCurrent block is < ${CURRENT_BLOCK} >".
			    "\nCurrent block string is < $blocksData{$block} >".
			    "\nCurrent starting element line is ".
			    "< ${LINE_COUNT} >".
			    "\nCurrent file is < ${CONFIG_FILE} >" , 
			    "- sl_parser::_extract - ".__LINE__ );
      }
      if ( defined $configVal{"CLASS"} ) {
         %classKeywords = %{$configVal{'KEYWORDS'}};
         foreach $block ( keys %blocksData ) {
	    $CURRENT_BLOCK = $block;
	    @blockKeywords = @{$classKeywords{$block}};
	    if ( $#blockKeywords >= 0 ) {
	       %paramLst = &_split( $blocksData{$block}, @blockKeywords );
	       # Add params to main param list
	       foreach $param ( keys %paramLst ) {
		  push( @{$configVal{$param}}, @{$paramLst{$param}} );
	       }
	    }
	    else {
	       &sl_traces::error( "Element type ".
	          "< $configVal{'ELEMENT_TYPE'} > cannot include a ".
		  "< ${block} > block".
		  "\nCurrent element string is < ${CURRENT_RAW} >".
		  "\nCurrent block is < ${CURRENT_BLOCK} >".
		  "\nCurrent block string is < $blocksData{$block} >".
		  "\nCurrent starting element line is ".
		  "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		  "\nCurrent file is < ${CONFIG_FILE} >" , 
		  "- sl_parser::_extract - ".__LINE__ );   
	    }
	 }
	 $CURRENT_BLOCK = "ELEMENT";
         &_checkMandatoryParams( $configRaw, %configVal ); 
         # Instantiate objects from configuration data
         &_createObjects( %configVal );
      }
      $CURRENT_ELEM++;
   }
}

# ------------------------------------------------------------------------------
#
sub _setObject {
   my ( $objectName ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_setObject - Setting config object ".
                "< ${objectName} >" );
   if ( defined $CONFIGURATION_OBJECTS{$objectName} ) {
      &sl_traces::error( "The object ".
                    "< ${objectName} > is multiple defined".
                    "\nCurrent line is < ${LINE_COUNT} >".
		    "\nCurrent file is < ${CONFIG_FILE} >", 
		    "- sl_parser::_setObject - ".__LINE__ );
   }
   else {
      $CONFIGURATION_OBJECTS{$objectName} = 0;
   }
}

# ------------------------------------------------------------------------------
#
sub _setAddress {
   my ( $address ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_setAddress - Setting mac addresses ".
                "< $(address} > " );
   if ( defined $ADDRESSES{$address} ) {
      &sl_traces::error( "The address ".
                    "< ${address} > is multiple defined".
		    "\nCurrent file is < ${CONFIG_FILE} >", 
		    "- sl_parser::_setAddress - ".__LINE__ );
   }
   else {
      $ADDRESSES{$address} = 0;
   }
}

# ------------------------------------------------------------------------------
#
sub _refObject {
   my ( $objectName ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_refObject - Referencing config object ".
                      "< ${objectName} > " );
   unless ( defined $CONFIGURATION_OBJECTS{$objectName} ) {
      &sl_traces::error( "The referenced object ".
                    "< ${objectName} > is not defined".
		    "\nCurrent line is < ${LINE_COUNT} >".
		    "\nCurrent file is < ${CONFIG_FILE} >", 
		    "- sl_parser::_refObject - ".__LINE__ );

   }
   else {
      $CONFIGURATION_OBJECTS{$objectName}++;
   }
}

# ------------------------------------------------------------------------------
#
sub _setRef {
   my ( $refoid, $oid, $alt ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_setRef - Setting referenced object ".
                "< ${refoid} > to object < ${oid} >" );
   $oid =~ /(.+)@(.+)/;
   $objectName = $1;
   unless ( defined $alt ) {
      if ( defined $CONTAINER_OBJECTS{$refoid} ) {
         $refoid =~ /(.+)@(.+)/;
	 $refObjectName = $1;
	 $CONTAINER_OBJECTS{$refoid} =~ /(.+)@(.+)/;
	 $contObjectName = $1;
	 &sl_traces::error( "The object ".
                       "< ${refObjectName} > is already referenced by the ".
		       "object < ${contObjectName} >\n".
		       "It cannot be referenced by the object ".
		       "< ${objectName} >", 
		       "- sl_parser::_setRef - ".__LINE__ );
      }
      else {
	 $CONTAINER_OBJECTS{$refoid} = $oid;
      }
   }
   else {
      if ( defined $ALT_CONTAINER_OBJECTS{$refoid} ) {
         $refoid =~ /(.+)@(.+)/;
	 $refObjectName = $1;
	 $ALT_CONTAINER_OBJECTS{$refoid} =~ /(.+)@(.+)/;
	 $contObjectName = $1;
	 &sl_traces::error( "The object ".
                       "< ${refObjectName} > is already referenced by the ".
		       "object < ${contObjectName} >\n".
		       "It cannot be referenced by the object ".
		       "< ${objectName} >", 
		       "- sl_parser::_setRef - ".__LINE__ );
      }
      else {
         $ALT_CONTAINER_OBJECTS{$refoid} = $oid;
      }
   }
}

# ------------------------------------------------------------------------------
#
sub _checkObjects {
   &sl_traces::trace( 3, "- sl_parser::_checkObjects - Checking configured ".
                      "objects " );
   # Check unused configured objects
   foreach $object ( sort keys %CONFIGURATION_OBJECTS ) {
      if ( $CONFIGURATION_OBJECTS{$object} eq 0 ) {
         #$_ =~ /(.+)@(.+)/;
	 &sl_traces::warning( 1, "The ".
                       "object < ${object} > is not referenced by the ".
		       "other configured objects" );
      }
   }
}

# ------------------------------------------------------------------------------
#
sub _createObjects {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createObjects - Creating objects from ".
                "configuration data" );
   # Create object(s) according to the class
   if ( $configVal{'CLASS'} eq "sl_cluster" ) {
      push( @CLUSTER_OBJECTS, &_createCluster( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_shelf" ) {
      push( @SHELF_OBJECTS, &_createShelf( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_drawer" ) {
      push( @DRAWER_OBJECTS, &_createDrawer( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_node" ) {
      push( @NODE_OBJECTS, &_createNode( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_board" ) {
      push( @BOARD_OBJECTS, &_createBoard( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_nic" ) {
      push( @NIC_OBJECTS, &_createNic( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_package" ) {
      push( @PACKAGE_OBJECTS, &_createPackage( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_patch" ) {
      push( @PATCH_OBJECTS, &_createPatch( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_software" ) {
      push( @SOFTWARE_OBJECTS, &_createSoftware( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_disk" ) {
      push( @DISK_OBJECTS, &_createDisk( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_slice" ) {
      push( @SLICE_OBJECTS, &_createSlice( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_filesys" ) {
      push( @FILESYS_OBJECTS, &_createFileSys( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_filesys_nfs" ) {
      push( @FILESYS_NFS_OBJECTS, &_createFileSysNfs( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_switch" ) {
      push( @SWITCH_OBJECTS, &_createSwitch( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_port" ) {
      push( @PORT_OBJECTS, &_createPort( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_group" ) {
      push( @GROUP_OBJECTS, &_createGroup( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_network" ) {
      push( @NETWORK_OBJECTS, &_createNetwork( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_domain" ) {
      push( @DOMAIN_OBJECTS, &_createDomain( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_ip" ) {
      push( @IP_OBJECTS, &_createIp( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_router" ) {
      push( @ROUTER_OBJECTS, &_createRouter( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_config" ) {
      push( @CONFIG_OBJECTS, &_createConfig( %configVal ) );
   }
   elsif ( $configVal{'CLASS'} eq "sl_file" ) {
      push( @FILE_OBJECTS, &_createFile( %configVal ) );
   }
   else {
      &sl_traces::error( "- sl_parser::_createObjects - Unknown object class ".
                         "< $configVal{'CLASS'} >", __LINE__ );
   }
}

# ------------------------------------------------------------------------------
#
sub _createFile {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createFile - Creating ".
                     "file object" );
   &_checkEnum( "@{$configVal{'file'}}[0](type)", 
                @{$configVal{'type'}}[0],
		"sl_file",
		values %{sl_file::FILE_TYPE} );
   &_checkEnum( "@{$configVal{'file'}}[0](runLevel)", 
                @{$configVal{'runLevel'}}[0], 
		"sl_file",
		values %{sl_file::FILE_RUN_LEVEL} );
   my %fileAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => @{$configVal{'file'}}[0],
      'Type' => @{$configVal{'type'}}[0],
      'RunLevel' => @{$configVal{'runLevel'}}[0],
      'Uri' => @{$configVal{'location'}}[0] );
   my $file = sl_file->new( %fileAttr );
   &_setObject( $file->getName );
   return $file;
}

# ------------------------------------------------------------------------------
#
sub _createConfig {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createConfig - Creating ".
                     "config object" );
   my %configAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => @{$configVal{'config'}}[0],
      'Next' => @{$configVal{'next'}}[0],
      'FileLst' => $configVal{'file'} );
   my $config = sl_config->new( %configAttr );
   &_setObject( $config->getName );
   return $config;
}

# ------------------------------------------------------------------------------
#
sub _createGroup {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createGroup - Creating group object" );
   &_checkEnum( "@{$configVal{'nodeGroup'}}[0](os)", 
                 @{$configVal{'os'}}[0], 
                 "sl_node",
                 ( values %{sl_node::NODE_OS} ) );
   &_checkEnum( "@{$configVal{'nodeGroup'}}[0](type)", 
                 @{$configVal{'type'}}[0], 
                 "sl_node",
                 ( values %{sl_node::NODE_TYPE} ) );
   &_checkEnum( "@{$configVal{'nodeGroup'}}[0](arch)", 
                 @{$configVal{'arch'}}[0], 
                 "sl_board",
                 ( values %{sl_board::BOARD_ARCH} ) );
   foreach $serviceName ( @{$configVal{'service'}} ) {
      &_checkEnum( "@{$configVal{'nodeGroup'}}[0](service)", 
                   $serviceName, 
		   "sl_service",
		   values %{sl_service::SERVICE_NAME} );
   }
   # Extract the main nodes group from the nodes group list
   # The remainders are the embedded nodes groups
   my $nodesGroup = shift @{$configVal{'nodeGroup'}};
   my %groupAttr = (
      'Id' => $OID_COUNTER++,
      'EmbeddedGroupLst' => $configVal{'nodeGroup'},
      'ServiceLst' => $configVal{'service'},
      'Name' => $nodesGroup,
      'NodeLst' => $configVal{'node'},
      'Os' => @{$configVal{'os'}}[0], 
      'Arch' => @{$configVal{'arch'}}[0], 
      'Type' => @{$configVal{'type'}}[0] );
   my $group = sl_group->new( %groupAttr );
   &_setObject( $group->getName );
   return $group;
}

# ------------------------------------------------------------------------------
#
sub _createDomain {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createDomain - Creating domain object" );
   &_checkInt( "@{$configVal{'domain'}}[0](id)", @{$configVal{'id'}}[0] );
   &_checkRange( "@{$configVal{'domain'}}[0](id)",
                 ${sl_domain::DOMAIN_ID_MIN},
                 ${sl_domain::DOMAIN_ID_MAX},
   		 @{$configVal{'id'}}[0] );
   my %domainAttr = (
      'Id' => @{$configVal{'id'}}[0],
      'IpAddressLst' => $configVal{'ip'},
      'NetworkLst' => $configVal{'network'},
      'RouterLst' => $configVal{'router'},
      'Name' => @{$configVal{'domain'}}[0] );
   my $domain = sl_domain->new( %domainAttr );
   &_setObject( $domain->getName );
   return $domain;
}

# ------------------------------------------------------------------------------
#
sub _createRouter {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createRouter - Creating router object" );
   my %routerAttr = (
      'Id' => $OID_COUNTER++,
      'IpAddressLst' => $configVal{'ip'},
      'Name' => @{$configVal{'router'}}[0] );
   my $router = sl_router->new( %routerAttr );
   &_setObject( $router->getName );
   return $router;
}

# ------------------------------------------------------------------------------
#
sub _createNetwork {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createNetwork - Creating ".
                     "network object" );
   &_checkEnum( "@{$configVal{'network'}}[0](type)", 
                @{$configVal{'type'}}[0], 
                "sl_network",
		values %{sl_network::NETWORK_TYPE} );
   sl_ip->checkNetworkNumber(
      "@{$configVal{'network'}}[0](number)", @{$configVal{'number'}}[0] );
   sl_ip->checkNetMask( 
      "@{$configVal{'network'}}[0](netmask)", @{$configVal{'netmask'}}[0] );
   my %networkAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => @{$configVal{'network'}}[0],
      'Type' => @{$configVal{'type'}}[0],
      'Number' => @{$configVal{'number'}}[0],
      'NetMask' => @{$configVal{'netmask'}}[0] );
   my $network = sl_network->new( %networkAttr );
   &_setObject( $network->getName );
   return $network;
}

# ------------------------------------------------------------------------------
#
sub _createIp {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createIp - Creating ip object" );
   &_checkEnum( "@{$configVal{'ip'}}[0](type)", 
               @{$configVal{'type'}}[0],
	       "sl_ip",
	       values %{sl_ip::IP_TYPE} );
   sl_ip->checkIpAddress(
      "@{$configVal{'ip'}}[0](address)", @{$configVal{'address'}}[0] );
   my %ipAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => @{$configVal{'ip'}}[0],
      'Address' => @{$configVal{'address'}}[0],
      'Type' => @{$configVal{'type'}}[0],
      'AliasLst' => $configVal{'alias'},
      'Network' => @{$configVal{'network'}}[0] );
   my $ip = sl_ip->new( %ipAttr );
   &_setObject( $ip->getName );
   my $address = $ip->getAddress;
   &_setAddress( $address ) if ( defined $address );
   return $ip;
}

# ------------------------------------------------------------------------------
#
sub _createSwitch {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createSwitch - Creating switch object" );
   &_checkEnum( "@{$configVal{'switch'}}[0](type)", 
               @{$configVal{'type'}}[0],
	       "sl_switch",
	       values %{sl_switch::SWITCH_TYPE} );
   my %switchAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => @{$configVal{'switch'}}[0],
      'Type' => @{$configVal{'type'}}[0],
      'IpAddressLst' => $configVal{'ip'},
      'PortLst' => $configVal{'port'} );
   my $switch = sl_switch->new( %switchAttr );
   &_setObject( $switch->getName );
   return $switch;
}

# ------------------------------------------------------------------------------
#
sub _createPort {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createPort - Creating port object" );
   &_checkInt( "@{$configVal{'port'}}[0](number)", @{$configVal{'number'}}[0] );
   my %portAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => @{$configVal{'port'}}[0],
      'Number' => @{$configVal{'number'}}[0],
      'Nic' => @{$configVal{'nic'}}[0] );
   my $port = sl_port->new( %portAttr );
   &_setObject( $port->getName );
   return $port;
}

# ------------------------------------------------------------------------------
#
sub _createFileSys {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createFileSys - Creating filesys object" );
   &_checkInt( "@{$configVal{'filesys'}}[0](fsck)", @{$configVal{'fsck'}}[0] )
      if ( defined @{$configVal{'fsck'}}[0] );
   &_checkEnum( "@{$configVal{'filesys'}}[0](type)", @{$configVal{'type'}}[0], 
                "sl_filesys",
		( values %{sl_filesys::FILESYS_TYPE} ) );
   &_checkEnum( "@{$configVal{'filesys'}}[0](mntBt)", 
                @{$configVal{'mntBt'}}[0], 
                "sl_filesys",
		( values %{sl_filesys::FILESYS_MNTBT} ) );
   &_checkEnum( "@{$configVal{'filesys'}}[0](role)", 
                @{$configVal{'role'}}[0], 
                "sl_filesys",
		( values %{sl_filesys::FILESYS_ROLE} ) );
   &_checkInt( "@{$configVal{'filesys'}}[0](size)", @{$configVal{'size'}}[0] );
   my %fileSysAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => @{$configVal{'filesys'}}[0],
      'Type' => @{$configVal{'type'}}[0],
      'Size' => @{$configVal{'size'}}[0],
      'Role' => @{$configVal{'role'}}[0],
      'Fsck' => @{$configVal{'fsck'}}[0],
      'MntOpt' => @{$configVal{'mntOpt'}}[0],
      'MntBoot' => @{$configVal{'mntBt'}}[0],
      'MntPoint' => @{$configVal{'mntPt'}}[0],
      'ExportedFileSysLst' => $configVal{'exportedFileSys'} );
   my $fileSys = sl_filesys->new( %fileSysAttr );
   &_setObject( $fileSys->getName );
   return $fileSys;
}

# ------------------------------------------------------------------------------
#
sub _createFileSysNfs {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createFileSysNfs - Creating ".
                      "filesys_nfs object" );
   &_checkEnum( "@{$configVal{'filesys'}}[0](type)", @{$configVal{'type'}}[0], 
                "sl_filesys",
		( values %{sl_filesys::FILESYS_TYPE} ) );
   &_checkEnum( "@{$configVal{'filesys'}}[0](mntBt)", 
                @{$configVal{'mntBt'}}[0], 
                "sl_filesys",
		( values %{sl_filesys::FILESYS_MNTBT} ) );
   &_checkEnum( "@{$configVal{'filesys'}}[0](role)", 
                @{$configVal{'role'}}[0], 
                "sl_filesys",
		( values %{sl_filesys::FILESYS_ROLE} ) );
   my %fileSysNfsAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => @{$configVal{'filesys'}}[0],
      'Type' => @{$configVal{'type'}}[0],
      'Role' => @{$configVal{'role'}}[0],
      'MntOpt' => @{$configVal{'mntOpt'}}[0],
      'MntBoot' => @{$configVal{'mntBt'}}[0],
      'MntPoint' => @{$configVal{'mntPt'}}[0],
      'RemMntPoint' => @{$configVal{'remMntPt'}}[0] );
   my $fileSysNfs = sl_filesys_nfs->new( %fileSysNfsAttr );
   &_setObject( $fileSysNfs->getName );
   return $fileSysNfs;
}

# ------------------------------------------------------------------------------
#
sub _createSlice {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createSlice - Creating slice object" );
   &_checkInt( "@{$configVal{'slice'}}[0](size)", @{$configVal{'size'}}[0] );
   # Control slice type
   &_checkEnum( "@{$configVal{'slice'}}[0](type)", @{$configVal{'type'}}[0],
                "sl_slice",
                 values %{sl_slice::SLICE_TYPE} );
   # Control slice number
   my $sliceNumberMin =
       ( @{$configVal{'type'}}[0] eq $sl_slice::SLICE_TYPE{'PHYS'} ) ?
       $sl_slice::SLICE_PHYS_NUMBER_MIN:
       $sl_slice::SLICE_LOG_NUMBER_MIN;
   my $sliceNumberMax =
       ( @{$configVal{'type'}}[0] eq $sl_slice::SLICE_TYPE{'PHYS'} ) ?
       $sl_slice::SLICE_PHYS_NUMBER_MAX:
       $sl_slice::SLICE_LOG_NUMBER_MAX;
   &_checkRange( "@{$configVal{'slice'}}[0](number)",
                 $sliceNumberMin,
                 $sliceNumberMax,
   		 @{$configVal{'number'}}[0] );
   if ( ( @{$configVal{'type'}}[0] eq $sl_slice::SLICE_TYPE{'PHYS'} ) and 
      ( @{$configVal{'number'}}[0] == $sl_slice::SLICE_PHYS_NUMBER_FORBIDDEN ) ) {
      &sl_traces::error( "- sl_parser::_createSlice - Slice number ".
                 "< ${sl_slice::SLICE_PHYS_NUMBER_FORBIDDEN} > cannot be used". 
		 "\nCurrent element string is < ${CURRENT_RAW} >".
		 "\nCurrent block is < ${CURRENT_BLOCK} >".
		 "\nCurrent starting element line is ".
		 "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		 "\nCurrent file is < ${CONFIG_FILE} >" , 
		 "- sl_parser::_createSlice - ".__LINE__ );
   }
   # Control slice role
   &_checkEnum( "@{$configVal{'slice'}}[0](role)", @{$configVal{'role'}}[0],
                "sl_slice",
                 values %{sl_slice::SLICE_ROLE} );
   # Extract slice, phys-slice and rep-slice from slice list
   my $sliceName = @{$configVal{'slice'}}[0];
   my $physSlice = @{$configVal{'slice'}}[1];
   my $repSlice = @{$configVal{'replicatedSlice'}}[0];
   # Check that a replicated slice is configured for a bitmap slice
   if ( defined @{$configVal{'role'}}[0] ) {
      if ( @{$configVal{'role'}}[0] eq ${sl_slice::SLICE_ROLE{'BITMAP_SNDR'}} ) {
         if ( ! defined $repSlice ) {
	    &sl_traces::error( "Bitmap slice must ".
                    "have a replicated slice configured". 
		    "\nCurrent element string is < ${CURRENT_RAW} >".
		    "\nCurrent block is < ${CURRENT_BLOCK} >".
		    "\nCurrent starting element line is ".
		    "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		    "\nCurrent file is < ${CONFIG_FILE} >" , 
		    "- sl_parser::_createSlice - ".__LINE__ );
         }
      }
      if ( @{$configVal{'role'}}[0] eq ${sl_slice::SLICE_ROLE{'REPLICATED'}} ) {
	 # Check that a replicated slice has a filesystem defined 
	 &sl_traces::error( "Replicated slices must have ".
        	    "a file system configured".
   		    "\nCurrent element string is < ${CURRENT_RAW} >".
		    "\nCurrent block is < ${CURRENT_BLOCK} >".
		    "\nCurrent starting element line is ".
		    "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
		    "\nCurrent file is < ${CONFIG_FILE} >" , 
		    "- sl_parser::_createSlice - ".__LINE__ )
	    unless ( defined @{$configVal{'filesys'}}[0] );	
	 # Check that a replicated slice has no rep-slice defined 
	 &sl_traces::error( "Replicated slices cannot have ".
        	    "a mirrored slice configured".
		    "\nCurrent element string is < ${CURRENT_RAW} >".
		    "\nCurrent block is < ${CURRENT_BLOCK} >".
		    "\nCurrent starting element line is ".
		    "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".	   
		    "\nCurrent file is < ${CONFIG_FILE} >" , 
		    "- sl_parser::_createSlice - ".__LINE__ )
	    if ( defined $repSlice );
      } 
   }
   # Create slice object
   my %sliceAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => $sliceName,
      'Type' => @{$configVal{'type'}}[0],
      'Size' => @{$configVal{'size'}}[0],
      'RawDev' => @{$configVal{'rawDev'}}[0],
      'BlockDev' => @{$configVal{'blockDev'}}[0],
      'RepSlice' => $repSlice,
      'PhysSlice' => $physSlice,
      'Number' => @{$configVal{'number'}}[0],
      'Role' => @{$configVal{'role'}}[0],
      'FileSys' => @{$configVal{'filesys'}}[0] );
   my $slice = sl_slice->new( %sliceAttr );
   &_setObject( $slice->getName );
   return $slice;
}

# ------------------------------------------------------------------------------
#
sub _createDisk {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createDisk - Creating disk object" );
   &_checkInt( "@{$configVal{'disk'}}[0](size)", @{$configVal{'size'}}[0] );
   &_checkEnum( "@{$configVal{'disk'}}[0](type)", 
                 @{$configVal{'type'}}[0], 
                 "sl_disk",
                 ( values %{sl_disk::DISK_TYPE} ) );

   my %diskAttr = (
      'Id' => $OID_COUNTER++,
      'Name' => @{$configVal{'disk'}}[0],
      'Type' => @{$configVal{'type'}}[0],
      'Size' => @{$configVal{'size'}}[0],
      'Device' => @{$configVal{'device'}}[0],
      'SliceLst' => $configVal{'slice'} );
   my $disk =  sl_disk->new( %diskAttr );
   &_setObject( $disk->getName );
   return $disk;
}

# ------------------------------------------------------------------------------
#
sub _createPackage {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createPackage - Creating package ".
                      "object" );
   my %packageAttr = ();
   # Control attributes values
   &_checkEnum( "@{$configVal{'software'}}[0](type)", 
                 @{$configVal{'type'}}[0], 
                 "sl_software",
                 ( values %{sl_software::SOFTWARE_TYPE} ) );
   &_checkEnum( "@{$configVal{'software'}}[0](mode)", 
                 @{$configVal{'mode'}}[0],
                 "sl_package",
                 ( values %{sl_package::PACKAGE_MODE} ) );
   # Set mandatory attributes
   $packageAttr{'Uri'} = @{$configVal{'software'}}[0];
   $packageAttr{'Type'} = @{$configVal{'type'}}[0];
   $packageAttr{'Mode'} = @{$configVal{'mode'}}[0];
   # Set optional attributes
   $packageAttr{'Base'} = @{$configVal{'base'}}[0]; 
   $packageAttr{'Admin'} = @{$configVal{'adminFile'}}[0];
   $packageAttr{'Resp'} = @{$configVal{'responseFile'}}[0];
   return sl_package->new( %packageAttr );
}

# ------------------------------------------------------------------------------
#
sub _createPatch {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createPatch - Creating patch ".
                      "object" );
   my %patchAttr = ();
   # Control attributes values
   &_checkEnum( "@{$configVal{'software'}}[0](type)", 
                 @{$configVal{'type'}}[0], 
                 "sl_software",
                 ( values %{sl_software::SOFTWARE_TYPE} ) );
   &_checkEnum( "@{$configVal{'software'}}[0](mode)", 
                 @{$configVal{'mode'}}[0],
                 "sl_patch",
                 ( values %{sl_patch::PATCH_MODE} ) );
   # Set mandatory attributes
   $patchAttr{'Uri'} = @{$configVal{'software'}}[0];
   $patchAttr{'Type'} = @{$configVal{'type'}}[0];
   $patchAttr{'Mode'} = @{$configVal{'mode'}}[0];
   # Set optional attributes
   $patchAttr{'Name'} = @{$configVal{'name'}}[0];
   $patchAttr{'Version'} = @{$configVal{'version'}}[0];
   return sl_patch->new( %patchAttr );
}

# ------------------------------------------------------------------------------
#
sub _createSoftware {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createSoftware - Creating software ".
                      "object" );
   my %softwareAttr = ();
   # Control attributes values
   &_checkEnum( "@{$configVal{'software'}}[0](type)", 
                 @{$configVal{'type'}}[0], 
                 "sl_software",
                 ( values %{sl_software::SOFTWARE_TYPE} ) );
   # Set mandatory attributes
   $softwareAttr{'Uri'} = @{$configVal{'software'}}[0];
   $softwareAttr{'Type'} = @{$configVal{'type'}}[0];
   $softwareAttr{'Name'} = @{$configVal{'name'}}[0];
   $softwareAttr{'Version'} = @{$configVal{'version'}}[0];
   $softwareAttr{'InstallScript'} = @{$configVal{'installScript'}}[0];
   return sl_software->new( %softwareAttr );
}

# ------------------------------------------------------------------------------
#
sub _createDrawer {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createDrawer - Creating drawer object" );
   &_checkEnum( "@{$configVal{'drawer'}}[0](type)", 
                 @{$configVal{'type'}}[0],
		 "sl_drawer",
                 ( values %{sl_drawer::DRAWER_TYPE} ) );
   my %drawerAttr = (
      'Type' => @{$configVal{'type'}}[0],
      'Name' => @{$configVal{'drawer'}}[0],
      'Id' => $OID_COUNTER++,
      'BoardLst' => $configVal{'board'},
      'DiskLst' => $configVal{'disk'} );
   my $drawer =  sl_drawer->new( %drawerAttr );
   &_setObject( $drawer->getName );
   return $drawer;
}

# ------------------------------------------------------------------------------
#
sub _createBoard {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createBoard - Creating board object" );
   &_checkEnum( "@{$configVal{'board'}}[0](arch)", 
                 @{$configVal{'arch'}}[0],
		 "sl_board",
                 ( values %{sl_board::BOARD_ARCH} ) );
   &_checkEnum( "@{$configVal{'board'}}[0](type)",
                 @{$configVal{'type'}}[0],
		 "sl_board",
                 ( values %{sl_board::BOARD_TYPE} ) );
   my $boardClass = undef;
   my $vendorType = undef;
   if ( @{$configVal{'type'}}[0] eq $sl_board::BOARD_TYPE{'GENERIC'} ) {
      # Check that the class and the vendorType parameters are defined
      &sl_traces::error( "class and vendorType parameters must be defined ".
         "when board type is < GENERIC >".
	 "\nCurrent element string is < ${CURRENT_RAW} >".
	 "\nCurrent block is < ${CURRENT_BLOCK} >".
	 "\nCurrent starting element line is ".
	 "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
	 "\nCurrent file is < ${CONFIG_FILE} >" ,
	 "- sl_parser::_createBoard - ".__LINE__ )
	 unless ( ( defined @{$configVal{'class'}}[0] ) and
	          ( defined @{$configVal{'vendorType'}}[0] ) );
      $boardClass = @{$configVal{'class'}}[0];
      $vendorType = @{$configVal{'vendorType'}}[0];		  
      &_checkEnum( "@{$configVal{'board'}}[0](class)", 
                    $boardClass,
		    "sl_board",
                    ( values %{sl_board::BOARD_CLASS} ) );
   }
   else {
      # Load predefined class and vendor type values
      my $boardConf = "hardware/@{$configVal{'type'}}[0].conf";
      my $configFile = &sl_utils::selectFile( $boardConf, undef, "mandatory" );
      &sl_registry::import( $boardConf, $configFile );
      $boardClass = &sl_registry::getValue( $boardConf, "class" );
      $vendorType = &sl_registry::getValue( $boardConf, "vendor" );
      &sl_traces::error( "Cannot find class and vendorType predefined ".
         "values for board type < @{$configVal{'type'}}[0] >".
	 "\nCurrent element string is < ${CURRENT_RAW} >".
	 "\nCurrent block is < ${CURRENT_BLOCK} >".
	 "\nCurrent starting element line is ".
	 "< $ELEM_STARTING_LINE{$CURRENT_ELEM} >".
	 "\nCurrent file is < ${CONFIG_FILE} >" ,
	 "- sl_parser::_createBoard - ".__LINE__ )
	 unless ( ( defined $boardClass ) and ( defined $vendorType ) );
      &_checkEnum( "@{$configVal{'board'}}[0](class)", 
                    $boardClass,
		    "sl_board",
                    ( $sl_board::BOARD_CLASS{@{$configVal{'type'}}[0]} ) );
   }
   my %boardAttr = (
      'Type' => @{$configVal{'type'}}[0],
      'Name' => @{$configVal{'board'}}[0],
      'Arch' => @{$configVal{'arch'}}[0],
      'Class' => $boardClass,
      'VendorType' => $vendorType,
      'SlotId' => @{$configVal{'clientId'}}[0],
      'Id' => $OID_COUNTER++,
      'NicLst' => $configVal{'nic'} );
   my $board =  sl_board->new( %boardAttr );
   &_setObject( $board->getName );
   return $board;
}

# ------------------------------------------------------------------------------
#
sub _createNic {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createNic - Creating nic object" );
   &_checkEnum( "@{$configVal{'nic'}}[0](type)", 
                 @{$configVal{'type'}}[0],
		 "sl_nic",
                 ( values %{sl_nic::NIC_TYPE} ) );
   &_checkEnum( "@{$configVal{'nic'}}[0](class)", 
                 @{$configVal{'class'}}[0],
		 "sl_nic",
                 ( values %{sl_nic::NIC_CLASS} ) );
   &_checkEnum( "@{$configVal{'nic'}}[0](role)", 
                 @{$configVal{'role'}}[0],
		 "sl_nic",
                 ( values %{sl_nic::NIC_ROLE} ) );
   if ( ( @{$configVal{'type'}}[0] eq ${sl_nic::NIC_TYPE{'PHYSICAL'}} ) and
        ( defined @{$configVal{'address'}}[0] ) ) {
      sl_ip->checkMacAddress( 
	 "@{$configVal{'nic'}}[0](address)", @{$configVal{'address'}}[0] );
   }
   my %nicAttr = (
      'Type' => @{$configVal{'type'}}[0],
      'Name' => @{$configVal{'nic'}}[0],
      'Id' => $OID_COUNTER++,
      'Class' => @{$configVal{'class'}}[0],
      'Role' => @{$configVal{'role'}}[0],
      'Device' => @{$configVal{'device'}}[0],
      'IpAddress' => @{$configVal{'ip'}}[0],
      'Address' => @{$configVal{'address'}}[0] );
   my $nic = sl_nic->new( %nicAttr );
   &_setObject( $nic->getName );
   # Check for mac address unicity
   my $mac = $nic->getAddress;
   &_setAddress( $mac ) if ( defined $mac );
   return $nic;
}

# ------------------------------------------------------------------------------
#
sub _createNode {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createNode - Creating node object" );
   &_checkInt( "@{$configVal{'node'}}[0](id)", @{$configVal{'id'}}[0] );
   &_checkRange( "@{$configVal{'node'}}[0](id)",
                 ${sl_node::NODE_ID_MIN},
                 ${sl_node::NODE_ID_MAX},
   		 @{$configVal{'id'}}[0] );
   my %nodeAttr = (
      'Board' => @{$configVal{'board'}}[0],
      'DiskLst' => $configVal{'disk'},
      'Name' => @{$configVal{'node'}}[0],
      'Id' => @{$configVal{'id'}}[0] );
   my $node = sl_node->new( %nodeAttr );
   &_setObject( $node->getName );
   return $node;
}

# ------------------------------------------------------------------------------
#
sub _createCluster {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createCluster - Creating cluster object" );
   my %clusterAttr = ( 
      'Name' => @{$configVal{'cluster'}}[0],
      'Id' => $OID_COUNTER++,
      'ShelfLst' => $configVal{'shelf'},
      'Domain' => @{$configVal{'domain'}}[0],
      'GroupLst' => $configVal{'nodeGroup'} );
   return sl_cluster->new( %clusterAttr );
}

# ------------------------------------------------------------------------------
#
sub _createShelf {
   my ( %configVal ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_createShelf - Creating shelf object" );
   my %shelfAttr = ( 
      'Name' => @{$configVal{'shelf'}}[0],
      'Id' => $OID_COUNTER++,
      'SwitchLst' => $configVal{'switch'},
      'DrawerLst' => $configVal{'drawer'} );
   my $shelf = sl_shelf->new( %shelfAttr );
   &_setObject( $shelf->getName );
   return $shelf;
}

# ------------------------------------------------------------------------------
#
sub _load {
   my ( $configFile, $template, $mandatory ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_load - Loading configuration file ".
                      "< ${configFile} >" );
     # Look for the config. file
   $configFile = &sl_utils::selectFile( $configFile, $template, $mandatory );
   if ( defined $configFile ) {
      &sl_traces::trace( 3, "- sl_parser::_load - Loading configuration data ".
                         "from < ${configFile} >" );
      open( CONFIG , $configFile ) ||
	 &sl_traces::error( "- sl_parser::_load - Cannot open ".
                            "< ${configFile} >", __LINE__ );
      # Read config data from file
      my @configData = <CONFIG>;
      # Initialize Global variables
      $LINE_COUNT = 0;
      $CONFIG_FILE = $configFile;
      %ELEM_STARTING_LINE = ();
      $CURRENT_ELEM = 0;
      $CURRENT_BLOCK = "";
      $CURRENT_RAW = "";
      # Release the config file
      close( CONFIG );
      # Extract and check keyword/value config data pairs
      &_extract( @configData );
   }
   return $configFile;
}

# ------------------------------------------------------------------------------
#
sub load {
   my ( @configFile ) = @_;
   foreach $file ( @configFile ) {
      &sl_traces::trace( 2, "- sl_parser::load - Loading configuration  ".
                	"data from < ${file} >" );
      # look for configuration file
      &_load( $file, undef, "mandatory" );
   }
   &_resolveDep;
   &_checkObjects;
   #$CLUSTER_OBJECTS[0]->display;
}

# ------------------------------------------------------------------------------
#
sub getObjects {
   my ( $class, @objectIndLst ) = @_;
   my @objectLst = ();
   &sl_traces::trace( 2, "- sl_parser::getObjects - Getting < $class > object(s)" );
   if ( defined $objectIndLst[0] ) {
      foreach ( @objectIndLst ) {
	 if ( $class eq 'sl_cluster' ) {
	    push( @objectLst, $CLUSTER_OBJECTS[$_] );
	 }
	 elsif ( $class eq 'sl_shelf' ) {
	    push( @objectLst, $SHELF_OBJECTS[$_] );
	 }
	 elsif ( $class eq "sl_drawer" ) {
	    push( @objectLst, $DRAWER_OBJECTS[$_] );
	 }
	 elsif ( $class eq "sl_node" ) {
	    push( @objectLst, $NODE_OBJECTS[$_] );
	 }
	 elsif ( $class eq "sl_board" ) {
	    push( @objectLst, $BOARD_OBJECTS[$_] );
	 }
	 elsif ( $class eq "sl_package" ) {
	    push( @objectLst, $PACKAGE_OBJECTS[$_] );
	 }
	 elsif ( $class eq "sl_disk" ) {
	    push( @objectLst, $DISK_OBJECTS[$_] );
	 }
	 elsif ( $class eq "sl_slice" ) {
	    push( @objectLst, $SLICE_OBJECTS[$_] );
	 }
	 elsif ( $class eq "sl_filesys" ) {
	    push( @objectLst, $FILESYS_OBJECTS[$_] );
	 }
	 elsif ( $class eq "sl_switch" ) {
	    push( @objectLst, $SWITCH_OBJECTS[$_] );
	 }
	 else {
	    &sl_traces::error( "- sl_parser::getObject - Unknown object class ".
                	  "< ${class} >", __LINE__ );
	 }
      }
   }
   else {
      if ( $class eq 'sl_cluster' ) {
	 push( @objectLst, @CLUSTER_OBJECTS );
      }
      elsif ( $class eq 'sl_shelf' ) {
	 push( @objectLst, @SHELF_OBJECTS );
      }
      elsif ( $class eq "sl_drawer" ) {
	 push( @objectLst, @DRAWER_OBJECTS );
      }
      elsif ( $class eq "sl_node" ) {
	 push( @objectLst, @NODE_OBJECTS );
      }
      elsif ( $class eq "sl_board" ) {
	 push( @objectLst, @BOARD_OBJECTS );
      }
      elsif ( $class eq "sl_package" ) {
	 push( @objectLst, @PACKAGE_OBJECTS );
      }
      elsif ( $class eq "sl_disk" ) {
	 push( @objectLst, @DISK_OBJECTS );
      }
      elsif ( $class eq "sl_slice" ) {
	 push( @objectLst, @SLICE_OBJECTS );
      }
      elsif ( $class eq "sl_filesys" ) {
	 push( @objectLst, @FILESYS_OBJECTS );
      }
      elsif ( $class eq "sl_switch" ) {
	 push( @objectLst, @SWITCH_OBJECTS );
      }
      elsif ( $class eq "sl_network" ) {
	 push( @objectLst, @NETWORK_OBJECTS );
      }
      else {
	 &sl_traces::error( "- sl_parser::getObject - Unknown object class ".
                       "< ${class} >", __LINE__ );
      }
   }
   return @objectLst;   
}

# ------------------------------------------------------------------------------
#
sub _resolveDep {
   &sl_traces::trace( 3, "- sl_parser::_resolveDep - Resolving objects ".
                "dependencies" );
   # Resolve dependencies
   foreach ( @IP_OBJECTS ) {
      &_resolveIpDep( $_ );
   }
   foreach ( @NIC_OBJECTS ) {
      &_resolveNicDep( $_ );
   }
   foreach ( @ROUTER_OBJECTS ) {
      &_resolveRouterDep( $_ );
   }
   foreach ( @FILESYS_OBJECTS ) {
      &_resolveFileSysDep( $_ );
   }
   foreach ( @SLICE_OBJECTS ) {
      &_resolveSliceDep( $_ );
   }
   foreach ( @DISK_OBJECTS ) {
      &_resolveDiskDep( $_ );
   }
   foreach ( @BOARD_OBJECTS ) {
      &_resolveBoardDep( $_ );
   }
   foreach ( @DRAWER_OBJECTS ) {
      &_resolveDrawerDep( $_ );
   }
   foreach ( @SHELF_OBJECTS ) {
      &_resolveShelfDep( $_ );
   }
   foreach ( @PORT_OBJECTS ) {
      &_resolvePortDep( $_ );
   }
   foreach ( @SWITCH_OBJECTS ) {
      &_resolveSwitchDep( $_ );
   }
   # !!! Group dependencies must be resolved 
   # before nodes dependencies !!!
   foreach ( @GROUP_OBJECTS ) {
      &_resolveGroupDep( $_ );
      # Build a new object nodes list once the group
      # specialization of the nodes has been done according
      # to the group characteristics
      push( @ALT_NODE_OBJECTS, $_->getNodeLst ); 
   }
   foreach ( @ALT_NODE_OBJECTS ) {
      &_resolveNodeDep( $_ );
   }
   foreach ( @DOMAIN_OBJECTS ) {
      &_resolveDomainDep( $_ );
   }
   foreach ( @CLUSTER_OBJECTS ) {
      &_resolveClusterDep( $_ );
   }
}

# ------------------------------------------------------------------------------
#
sub _resolveGroupDep {
   my ( $group ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveGroupDep - Resolving group ".
                "< $group->{'Name'} > dependencies" );
   # Resolve nodes list dependencies
   my @nodeLst = $group->getNodeLst;
   my $arch = $group->getArch;
   my $os = $group->getOs;
   my $groupType = $group->getType;
   my $groupName = $group->getName;
   my @nodesGroupLst = ();
   foreach $nodeName ( @nodeLst ) {
      $node = &_lookUpByName( $nodeName, "$group->{'Name'}(node)", 
                              "node", @NODE_OBJECTS );
      &_setRef( $node->iD, $group->iD );
      # Grab parameters from the existing node
      # Add the parameters inherited from the group
      %nodeAttr = ( 
	 'Arch' => $arch,
         'Os' => $os,
	 'Type' => $groupType,
         'Id' => $node->getId,
         'Name' => $node->getName,
	 'Board' => $node->getBoard,
	 'GroupName' => $groupName );
      @{$nodeAttr{'DiskLst'}} = $node->getDiskOidLst;
      # Specialize the node according to the running OS
      if ( $nodeAttr{'Os'} eq ${sl_node::NODE_OS{'SOLARIS'}} ) {
	 $newNode = sl_node_solaris->new( %nodeAttr );
         push( @nodesGroupLst, $newNode );
      }
      else {
         &sl_traces::error( "- sl_parser::_resolveGroupDep - Os < $os > ".
	                    "is not supported", __LINE__ );
      }
   }
   # Check that all the boards of a group have same class and vendor type
   my $nodeNameRef = $nodesGroupLst[0]->getName;
   my $boardNameRef = $nodesGroupLst[0]->getBoard;
   my $nodeBoardRef = &_lookUpByName( $boardNameRef, "${nodeNameRef}(board)", 
                                      "board", @BOARD_OBJECTS );
   my $boardClassRef = $nodeBoardRef->getClass;
   my $boardTypeRef = $nodeBoardRef->getType;
   for ( $i=1; $i < scalar @nodesGroupLst; $i++) {
      $nodeName = $nodesGroupLst[$i]->getName;
      $boardName = $nodesGroupLst[$i]->getBoard;
      $nodeBoard = &_lookUpByName( $boardName, "${nodeName}(board)", 
                                   "board", @BOARD_OBJECTS );
      $boardClass = $nodeBoard->getClass;
      $boardType = $nodeBoard->getType;
      &sl_traces::error( "Boards of the group < ${groupName} > must have ".
         "same type and class parameters".
	 "\nReference board < ${boardNameRef} >".
	 "\nReference board type  = ${boardTypeRef}".     
	 "\nReference board class = ${boardClassRef}".     
	 "\nCurrent board < ${boardName} >".
	 "\nCurrent board type  = ${boardType}".     
	 "\nCurrent board class = ${boardClass}",     
	 "- sl_parser::_resolveGroupDep -".__LINE__ )
	 unless ( ( $boardClassRef eq $boardClass ) and 
	          ( $boardTypeRef eq $boardType ) );    
   }
   # Replace the group node list by the new one
   $group->setNodeLst( @nodesGroupLst );
   # Add the class and vendor type parameter to the group definition
   $group->setClass( $boardClassRef );
   # Resolve service dependency
   my @serviceLst = ();
   my $solarisName = &sl_jumpstart::getSolarisName;
   foreach $serviceName ( @{$group->{'ServiceLst'}} ) {
      # Create new service objects
      my %serviceAttr = (
	 'Id' => $OID_COUNTER++,
	 'Name' => $serviceName );
      my $service = sl_service->new( %serviceAttr );
      my $serviceType = $service->getType;
      my $groupServiceType = $sl_node::NODE_SERVICE_TYPE{$groupType};
      &sl_traces::error( "Services and nodes group ".
         "must have same type".
	 "\nNodes group              : ${groupName}".
	 "\nNodes group service type : ${groupServiceType}".
	 "\nService                  : ${serviceName}".
	 "\nService type             : ${serviceType}",    
	 "- sl_parser::_resolveGroupDep -".__LINE__ )
         unless ( 
	    ( $serviceType eq $groupServiceType ) or
	    ( $serviceType eq $sl_service::SERVICE_TYPE_VALUE{'GENERIC'} ) );
      &_setRef( $service->iD, $group->iD );
      if ( $main::OPERATION eq $main::DEPLOY ) {
	 # Load service config if defined
	 my $serviceConfig = "config.${serviceName}.${arch}.${os}";
	 my $altServiceConfig = "${solarisName}/${serviceConfig}";
	 $serviceConfig = "services/${serviceConfig}";
	 $altServiceConfig = "services/${altServiceConfig}";
	 $serviceConfig = 
            &sl_utils::selectFile( $serviceConfig, $altServiceConfig );
	 if ( defined $serviceConfig ) {
	    # Since the same file is used to configure multiple times the same
	    # service the tag <-AUTO_NAME-> must be replaced each time the file 
	    # is loaded to obtain distinct configuration element names before
	    # loading
	    my $refName = $OID_COUNTER++;
            my $preparedConfig = &_prepareConfigFile( $serviceConfig, $refName );
            &_load( $preparedConfig );
	    # Look for config object (service name = config name + 
	    # PRE or POST_RL3 or POST_RL2)
	    my @configLst = ();
	    for $configType ( "PRE", "POST_RL2", "POST_RL3" ) {
	       my $config = 
		  &_lookUpByName( "${serviceName}_${configType}_${refName}", 
	                	  "$service->{'Name'}(config)", 
				  "$FAIL_FLAG", @CONFIG_OBJECTS ); 
	       # Resolve config file dependencies
	       if ( defined $config ) {
		  &_resolveConfigDep( $config );
		  # Add Config data to the service
		  push( @configLst, $config );
	       }
            }
	    $service->setConfigLst( @configLst );
	 }
      }
      push( @serviceLst, $service );
   }
   # Add implicit GENERIC service (used to trigger the IP/File-system/SWL
   # configuration methods)
   my $genericService = sl_service->new( 'Name' => "GENERIC" , 
	                                 'Id' => $OID_COUNTER++ );
   push( @serviceLst, $genericService );
   @{$group->{'ServiceLst'}} = @serviceLst;
   if ( $main::OPERATION eq $main::CONFIG ) {
      # Load optional User defined config. files and scripts
      # Reset config object list to capture only nodes group config data
      @CONFIG_OBJECTS = ();
      @FILE_OBJECTS = ();
      my $groupConfigFile = "config.${groupName}";
      my $altgroupConfigFile = "${solarisName}/${groupConfigFile}";
      $groupConfigFile = "models/${groupConfigFile}";
      $altgroupConfigFile = "models/${altgroupConfigFile}";
      &_load( $groupConfigFile, $altgroupConfigFile );
      # Resolve config file dependencies
      foreach $config ( @CONFIG_OBJECTS ) {
	 &_resolveConfigDep( $config );
	 &_refObject( $config->getName );
      }
      # Add config data to the nodes group definition
      $group->setConfigLst( undef, @CONFIG_OBJECTS ) 
         if ( scalar @CONFIG_OBJECTS > 0 );
   }
   if ( $main::OPERATION eq $main::CREATE ) {
      # Load NHAS packages according to the service type, group arch and group os
      # Reset soft. object lists before loading new soft. config data
      @PACKAGE_OBJECTS = ();
      @PATCH_OBJECTS = ();
      @SOFTWARE_OBJECTS = ();
      foreach $service ( @serviceLst ) {
	 if ( $service->hasSoftware ) {
            my @subServiceLst = $service->getServiceLst;
	    foreach $subService ( @subServiceLst ) {
	       my $tmpService = sl_service->new( 'Name' => $subService , 
	                                	 'Id' => $OID_COUNTER++ );
	       if ( $tmpService->hasSoftware ) {
		  my $groupTypeSoftConf = "${subService}.${arch}.${os}";
        	  my $altGroupTypeSoftConf = "${solarisName}/${groupTypeSoftConf}";
        	  $groupTypeSoftConf = "services/${groupTypeSoftConf}";
        	  $altGroupTypeSoftConf = "services/${altGroupTypeSoftConf}";
		  &_load( $groupTypeSoftConf, $altGroupTypeSoftConf, "mandatory" );
	       }
	    }
	 }
      }
      # Load Hardware dependent packages
      my @nodeLst = $group->getNodeLst;
      foreach $node ( @nodeLst ) {
         my $boardName = $node->getBoard;
         my $board = &_lookUpByName( 
                     $boardName, "$node->{'Name'}(board)", 
                     "board", @BOARD_OBJECTS );
	 my $boardType = $board->getType;
	 my $boardTypeSoftConf = "${boardType}.${arch}.${os}";
	 my $altBoardTypeSoftConf = "${solarisName}/${boardTypeSoftConf}";
	 $boardTypeSoftConf = "hardware/${boardTypeSoftConf}";
	 $altBoardTypeSoftConf = "hardware/${altBoardTypeSoftConf}";
	 &_load( $boardTypeSoftConf, $altBoardTypeSoftConf, undef );
      }
      # Load optional group add-on packages if any
      my $name = $group->getName;
      my $groupAddOnSoftConf = "${name}.${arch}.${os}";
      my $altGroupAddOnSoftConf = "${solarisName}/${groupAddOnSoftConf}";
      $groupAddOnSoftConf = "services/$groupAddOnSoftConf";
      $altGroupAddOnSoftConf = "services/$altGroupAddOnSoftConf";
      &_load( $groupAddOnSoftConf, $altGroupAddOnSoftConf );
      # Add package/patch/soft. to group
      $group->setPkgLst( @PACKAGE_OBJECTS ) if ( scalar @PACKAGE_OBJECTS > 0 );
      $group->setPatLst( @PATCH_OBJECTS ) if ( scalar @PATCH_OBJECTS > 0 );
      $group->setSoftLst( @SOFTWARE_OBJECTS ) if ( scalar @SOFTWARE_OBJECTS > 0 );
   }
   # Look for groups referenced by the current
   my $hostGroupName = $group->getName;
   my @groupLst = ();
   foreach $groupName ( @{$group->{'EmbeddedGroupLst'}} ) {
      my $refGroup = &_lookUpByName( 
                     $groupName, "$group->{'Name'}(embeddedGroup)", 
                     "embeddedGroup", @GROUP_OBJECTS );
      # Make reference to the host group for each embedded group
      $refGroup->setHostGroup( $hostGroupName );
      push( @groupLst, $refGroup->getName );
      &_setRef( $refGroup->iD, $group->iD, "alt" );
   }
   @{$group->{'EmbeddedGroupLst'}} = @groupLst;
   #$group->display;
}

# ------------------------------------------------------------------------------
#
sub _resolveNodeDep {
   my ( $node ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveNodeDep - Resolving node ".
                "< $node->{'Name'} > dependencies" );
   # Look for board object referenced by the node
   if ( defined $node->{'Board'} ) {
      my $board = &_lookUpByName( $node->{'Board'}, "$node->{'Name'}(board)", 
                        	  "board", @BOARD_OBJECTS );
      $node->{'Board'} = $board->getName;
      &_setRef( $board->iD, $node->iD, "alt" );
      # Check that the node arch is the same than the board arch
      $boardArch = $board->getArch;
      $boardName = $board->getName;
      $nodeArch = $node->getArch;
      $nodeName = $node->getName;
      &sl_traces::error( " - sl_parser::_resolveNodeDep - The node ".
                	 "< ${nodeName} > and the board < $boardName > ".
			 "\nmust have the same ARCH parameter".
			 "\nnode arch = < ${nodeArch} > ".
			 "board arch = < ${boardArch} >", __LINE__ )
	 unless ( $boardArch eq $nodeArch );
      # Retrieve the drawer associated to the board and associate the node
      # to the drawer node list (indirect retrieval)
      my $drawerOid = $CONTAINER_OBJECTS{$board->iD};
      # Add the drawer ref. to the node
      $node->setDrawer( $drawerOid );
      # Set the node as a referenced object
      &_refObject( $node->getName );
   }
   # Look for disk object referenced by the node
   @diskLst = ();
   foreach $diskName ( @{$node->{'DiskLst'}} ) {
      $disk = &_lookUpByName( $diskName, "$node->{'Name'}(disk)", 
                              "disk", @DISK_OBJECTS );
      push( @diskLst, $disk->getName );
   }
   @{$node->{'DiskLst'}} = @diskLst;
}

# ------------------------------------------------------------------------------
#
sub _resolvePortDep {
   my ( $port ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolvePortDep - Resolving port ".
                "< $port->{'Name'} > dependencies" );
   # Look for nic object referenced by the port
   $nicName = $port->getNic;
   $nic = &_lookUpByName( $nicName, "$port->{'Name'}(nic)", 
	                    "nic", @NIC_OBJECTS );
   $port->{'Nic'} = $nic->iD;
   # Use alternate cross-references table because the main table is already
   # used for board/nic dependencies
   &_setRef( $nic->iD, $port->iD, "alt" );
}

# ------------------------------------------------------------------------------
#
sub _resolveSwitchDep {
   my ( $switch ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveSwitchDep - Resolving switch ".
                "< $switch->{'Name'} > dependencies" );
   # Look for board objects referenced by the nic
   my @portLst = ();
   foreach $portName ( @{$switch->{'PortLst'}} ) {
       $port = &_lookUpByName( $portName, "$switch->{'Name'}(port)", 
	                        "port", @PORT_OBJECTS );
       push( @portLst, $port );
   }
   @{$switch->{'PortLst'}} = @portLst;
   # Look for ip objects referenced by the switch
   my @ipLst = ();
   foreach $ipName ( @{$switch->{'IpAddressLst'}} ) {
       $ip = &_lookUpByName( 
          $ipName, "$switch->{'Name'}(ip)", "ip", @IP_OBJECTS );
       push( @ipLst, $ip );
       &_setRef( $ip->iD, $switch->iD );
   }
   @{$switch->{'IpAddressLst'}} = @ipLst;
}

# ------------------------------------------------------------------------------
#
sub _resolveRouterDep {
   my ( $router ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveRouterDep - Resolving router ".
                "< $router->{'Name'} > dependencies" );
   # Look for ip objects referenced by the router
   my @ipLst = ();
   foreach $ipName ( @{$router->{'IpAddressLst'}} ) {
       $ip = &_lookUpByName( 
          $ipName, "$router->{'Name'}(ip)", "ip", @IP_OBJECTS );
       push( @ipLst, $ip );
       &_setRef( $ip->iD, $router->iD );
   }
   @{$router->{'IpAddressLst'}} = @ipLst;
}

# ------------------------------------------------------------------------------
#
sub _resolveConfigDep {
   my ( $config ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveConfigDep - Resolving config ".
                "< $config->{'Name'} > dependencies" );
   # Look for file objects referenced by the config element
   my @fileLst = ();
   foreach $fileName ( @{$config->{'FileLst'}} ) {
      $file = &_lookUpByName( 
          $fileName, "$config->{'Name'}(file)", "file", @FILE_OBJECTS );
       push( @fileLst, $file );
       &_setRef( $file->iD, $config->iD );
   }
   $config->setFileLst( @fileLst );
}

# ------------------------------------------------------------------------------
#
sub _resolveFileSysDep {
   my ( $filesys ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveFileSysDep - Resolving filesys ".
                      "< $filesys->{'Name'} > dependencies" );
   # Look for filesys_nfs objects referenced by the filesys
   my @fileSysLst = ();
   my @expFileSysLst = $filesys->getExportedFileSysLst;
   foreach $filesysName ( @expFileSysLst ) {
       $filesysNfs = &_lookUpByName( $filesysName, 
                                   "$filesys->{'Name'}(ExportedFileSysLst)",
				   "ExportedFileSysLst", @FILESYS_NFS_OBJECTS );
       $filesysNfs->setHostFileSys( $filesys->getName );
       #$filesysNfs->display;
       push( @fileSysLst, $filesysNfs );
       &_setRef( $filesysNfs->iD, $filesys->iD );
   }
   @{$filesys->{'ExportedFileSysLst'}} = @fileSysLst;
   #$filesys->display;
}

# ------------------------------------------------------------------------------
#
sub _resolveIpDep {
   my ( $ip ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveIpDep - Resolving ip ".
                      "< $ip->{'Name'} > dependencies" );
   # Look for network objects referenced by the ip
   my $networkName = $ip->{'Network'};
   my $network = &_lookUpByName( $networkName, "$ip->{'Name'}(network)", 
	                         "network", @NETWORK_OBJECTS );

   my $networkNumber = $network->getNumber;
   $ip->compareNetwork( $networkNumber );
   $ip->{'Network'} = $network->iD;
}

# ------------------------------------------------------------------------------
#
sub _resolveNicDep {
   my ( $nic ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveNicDep - Resolving nic ".
                "< $nic->{'Name'} > dependencies" );
   # Look for nic objects referenced by the nic
   my $ipName = $nic->{'IpAddress'};
   $ip = &_lookUpByName( $ipName, "$nic->{'Name'}(ip)", "ip", @IP_OBJECTS );
   &_setRef( $ip->iD, $nic->iD ) 
      unless ( $ip->isFloating );
   $nic->{'IpAddress'} = $ip;
}

# ------------------------------------------------------------------------------
#
sub _resolveSliceDep {
   my ( $slice ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveSliceDep - Resolving slice ".
                "< $slice->{'Name'} > dependencies" );
   # Look for filesys objects referenced by the slice
   my $filesSysName = $slice->{'FileSys'};
   if ( defined $filesSysName ) {
      $slice->{'FileSys'} = &_lookUpByName( 
                               $filesSysName, "$slice->{'Name'}(filesys)", 
	                       "filesys", @FILESYS_OBJECTS );
   }
   # Check that the mirrored slice exists (if any)
   if ( defined $slice->{'RepSlice'} ) {
      my $sliceName = $slice->{'RepSlice'};
      my $repSlice = &_lookUpByName( 
                            $sliceName, "$slice->{'Name'}(RepSlice)", 
	                    "slice", @SLICE_OBJECTS );
      # Leave the slice name as is: lookUp call just for dependency check
   }
}

# ------------------------------------------------------------------------------
#
sub _resolveDiskDep {
   my ( $disk ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveDiskDep - Resolving disk ".
                "< $disk->{'Name'} > dependencies" );
   # Look for slice objects referenced by the disk
   my @sliceLst = ();
   foreach $sliceName ( @{$disk->{'SliceLst'}} ) {
      $slice = &_lookUpByName( 
         $sliceName, "$disk->{'Name'}(slice)", "slice", @SLICE_OBJECTS ); 
      push( @sliceLst, $slice );
   }
   @{$disk->{'SliceLst'}} = @sliceLst;
}

# ------------------------------------------------------------------------------
#
sub _resolveBoardDep {
   my ( $board ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveBoardDep - Resolving board ".
                      "< $board->{'Name'} > dependencies" );
   # Look for nic objects referenced by the board
   my @nicLst = ();
   foreach $nicName ( @{$board->{'NicLst'}} ) {
      $nic = &_lookUpByName( $nicName, "$board->{'Name'}(nic)", 
                             "nic", @NIC_OBJECTS ); 
      push( @nicLst, $nic );
        &_setRef( $nic->iD, $board->iD );
   }
   @{$board->{'NicLst'}} = @nicLst;
}

# ------------------------------------------------------------------------------
#
sub _resolveDrawerDep {
   my ( $drawer ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveDrawerDep - Resolving drawer ".
                "< $drawer->{'Name'} > dependencies" );
   # Look for board objects referenced by the drawer
   my @boardLst = ();
   foreach $boardName ( @{$drawer->{'BoardLst'}} ) {
      $board = &_lookUpByName( 
         $boardName, "$drawer->{'Name'}(board)", "board", @BOARD_OBJECTS ); 
      push( @boardLst, $board );
        &_setRef( $board->iD, $drawer->iD );
   }
   @{$drawer->{'BoardLst'}} = @boardLst;
   # Look for disk objects referenced by the drawer
   my @diskLst = ();
   foreach $diskName ( @{$drawer->{'DiskLst'}} ) {
      $disk = &_lookUpByName( 
         $diskName, "$drawer->{'Name'}(disk)", "disk", @DISK_OBJECTS );
      push( @diskLst, $disk );
        &_setRef( $disk->iD, $drawer->iD );
   }
   @{$drawer->{'DiskLst'}} = @diskLst;
}

# ------------------------------------------------------------------------------
#
sub _resolveShelfDep {
   my ( $shelf ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveShelfDep - Resolving shelf ".
                "< $shelf->{'Name'} > dependencies" );
   # Look for drawer objects referenced by the shelf
   my @drawerLst = ();
   foreach $drawerName ( @{$shelf->{'DrawerLst'}} ) {
      $drawer = &_lookUpByName( 
         $drawerName, "$shelf->{'Name'}(drawer)", "drawer", @DRAWER_OBJECTS );
      push( @drawerLst, $drawer  );
        &_setRef( $drawer->iD, $shelf->iD );
   }
   @{$shelf->{'DrawerLst'}} = @drawerLst;
   # Look for switch objects referenced by the shelf
   my @switchLst = ();
   foreach $switchName ( @{$shelf->{'SwitchLst'}} ) {
      $switch = &_lookUpByName( 
         $switchName, "$shelf->{'Name'}(switch)", "switch", @SWITCH_OBJECTS ); 
      push( @switchLst, $switch );
        &_setRef( $switch->iD, $shelf->iD );
   }
   @{$shelf->{'SwitchLst'}} = @switchLst;
}

# ------------------------------------------------------------------------------
#
sub _resolveDomainDep {
   my ( $domain ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveDomainDep - Resolving domain ".
                "< $domain->{'Name'} > dependencies" );
   # Look for ip objects referenced by the domain
   my @ipLst = ();
   foreach $ipName ( @{$domain->{'IpAddressLst'}} ) {
       $ip = &_lookUpByName( $ipName, "$domain->{'Name'}(ip)", 
	                     "ip", @IP_OBJECTS );
       &_setRef( $ip->iD, $domain->iD );
       push( @ipLst, $ip );
   }
   @{$domain->{'IpAddressLst'}} = @ipLst;
   # Look for network objects referenced by the domain
   my @networkLst = ();
   foreach $networkName ( @{$domain->{'NetworkLst'}} ) {
      $network = &_lookUpByName( $networkName, "$domain->{'Name'}(network)", 
                                 "network", @NETWORK_OBJECTS ); 
      push( @networkLst, $network );
      &_setRef( $network->iD, $domain->iD );
   }
   @{$domain->{'NetworkLst'}} = @networkLst;
   # Look for router objects referenced by the domain
   my @routerLst = ();
   foreach $routerName ( @{$domain->{'RouterLst'}} ) {
      $router = &_lookUpByName( $routerName, "$domain->{'Name'}(router)", 
                                "router", @ROUTER_OBJECTS ); 
      push( @routerLst, $router );
      &_setRef( $router->iD, $domain->iD );
   }
   @{$domain->{'RouterLst'}} = @routerLst;
}

# ------------------------------------------------------------------------------
#
sub _resolveClusterDep {
   my ( $cluster ) = @_;
   &sl_traces::trace( 3, "- sl_parser::_resolveClusterDep - Resolving cluster ".
                "< $cluster->{'Name'} > dependencies" );
   # Look for shelf object referenced by the cluster
   my @shelfLst = ();
   foreach $shelfName ( @{$cluster->{'ShelfLst'}} ) {
      $shelf = &_lookUpByName( $shelfName, "$cluster->{'Name'}(shelf)",
                               "shelf", @SHELF_OBJECTS );
      push( @shelfLst, $shelf );
      &_setRef( $shelf->iD, $cluster->iD );
   }
   @{$cluster->{'ShelfLst'}} = @shelfLst;
   # Look for domain object referenced by the cluster
   my $domainName = $cluster->{'Domain'};
   my $domain = &_lookUpByName( 
      $domainName, "$cluster->{'Name'}(domain)", "domain", @DOMAIN_OBJECTS );
   &_setRef( $domain->iD, $cluster->iD );
   $cluster->{'Domain'} = $domain;
   # Look for group object referenced by the cluster
   my @groupLst = ();
   foreach $groupName ( @{$cluster->{'GroupLst'}} ) {
      $group = &_lookUpByName( $groupName, "$cluster->{'Name'}(nodeGroup)",
                               "nodeGroup", @GROUP_OBJECTS );
      push( @groupLst, $group );
      &_setRef( $group->iD, $cluster->iD  );
   }
   # Set group of node to cluster definition (logical cluster def)
   @{$cluster->{'GroupLst'}} = @groupLst;
}

# ------------------------------------------------------------------------------

return 1;
