#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_network.pm 1.10     02/11/19 SMI"
# ------------------------------------------------------------------------------

BEGIN { 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_ip.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
}

# ------------------------------------------------------------------------------

package sl_network;

# Network Type constants
%NETWORK_TYPE = ( 'IPV4' => "IPV4" );

# Network enum hidden values
%NETWORK_HIDDEN_VALUES = ( 'IPV4' => 0 );

# ID fields separator
my $ID_SEP = '@';

# Network object attributes
my @NETWORK_ATTR = qw( Id
                       Name
		       Type
		       Number
		       NetMask );
     
# ------------------------------------------------------------------------------
#
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} )  && 
        ( defined $attr{'Name'} ) && 
        ( defined $attr{'Type'} ) && 
        ( defined $attr{'Number'} ) && 
        ( defined $attr{'NetMask'} ) ) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      $self->{'Type'} = $attr{'Type'};
      $self->{'Number'} = $attr{'Number'};
      $self->{'NetMask'} = $attr{'NetMask'};
   }
   elsif ( defined $attr{'NetworkObject'} ) {
      $self->_construct( $attr{'NetworkObject'} );
   }
   else {
      &sl_traces::error( "Unable to create network object.".
                         " Invalid creation method",
			 "- sl_network::new - ".__LINE__,
			 ( "sl_network" ) );
   }
   &sl_traces::trace( 2, "- sl_network::new - Creating network object ".
                     "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $networkObject ) = @_;
   &sl_traces::trace( 3, "- sl_network::_construct - Building network object" );
   my $network = sl_iterator->new( 'Object' => $networkObject );
   $self->{'Id'} = $network->extract;
   $self->{'Name'} = $network->extract;
   $self->{'Type'} = $network->extract;
   $self->{'Number'} = $network->extract;
   $self->{'NetMask'} = $network->extract;
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_network::serialize - Serializing ".
                     "network object" );
   my $networkObject = sl_iterator->new( 'Object' =>  "" );
   foreach $attr ( @NETWORK_ATTR ) {
      $networkObject->serialize( $self->{$attr} );
   }
   return $networkObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_network::display - Displaying network" );
   print STDOUT "\nNetwork\n";
   print STDOUT "-------\n";
   foreach ( @NETWORK_ATTR ) {
      if ( defined $self->{$_} ) {
	 print STDOUT "$_ = ";
	 print STDOUT "$self->{$_}";
	 print STDOUT "\n";
      }
      else {
         print STDOUT "$_ =\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_network::getName - Getting network Name" );
   return $self->{'Name'};
}

# ------------------------------------------------------------------------------
#
sub getNumber {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_network::getNumber - Getting network Number" );
   return $self->{'Number'};
}

# ------------------------------------------------------------------------------
#
sub getNetMask {
   my $self = shift;
   my ( $prefixLength ) = @_;
   &sl_traces::trace( 2, "- sl_network::getNetMask - Getting network NetMask" );
   if ( defined $prefixLength ) {
      my $netMask = 0;
      my @maskTab = split( '\.', $self->{'NetMask'} );
      foreach ( @maskTab ) {
         $netMask += sl_ip->getPrefixLength( $_ );
      }
      return $netMask;
   }
   else {
      return $self->{'NetMask'};
   }
}

# ------------------------------------------------------------------------------
#
sub getBroadCast {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_network::getBroadCast - Getting network ".
                      "BroadCast" );
   my @broadCastTab = ();
   my $netmask = $self->getNetMask;
   my $number = $self->getNumber;
   my @splittedNumber = split( '\.', $number );
   my @splittedNetmask = split( '\.', $netmask );
   for ( my $i=0; $i<4; $i++ ) {
      my $numByte = $splittedNumber[$i];
      my $netByte = $splittedNetmask[$i];
      my $broadCastByte = $numByte | ( $netByte ^ 0xFF );
      push( @broadCastTab, $broadCastByte );
   }
   # Build complete broadcast mask
   my $broadCast = join( '.', @broadCastTab );
   return $broadCast;
}

# ------------------------------------------------------------------------------
#
sub getType {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_network::getType - Getting network Type" );
   return $self->{'Type'};
}

# ------------------------------------------------------------------------------
#
sub deploy {
   my $self = shift;
   my ( $destDir ) = @_;
   &sl_traces::trace( 2, "- sl_network::deploy - Configuring network ".
                     "< $self->{'Name'} >" );
   # Add floating addresses to temporary hosts file   
   sl_ip->addNetMask( $destDir, $self->getNumber, $self->getNetMask );
}

# ------------------------------------------------------------------------------
#
sub isHidden {
   my $class = shift;
   my $enumValue = shift;
   &sl_traces::trace( 2, "- sl_network::isHidden - Testing ". 
                      "< ${enumValue} > hidden value" );
   return $NETWORK_HIDDEN_VALUES{$enumValue};
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   my $networkId = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_network::iD - Network iD is < ${networkId} >" );
   return $networkId;
}

# ------------------------------------------------------------------------------

return 1;
