#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_ip.pm 1.12     02/11/13 SMI"
# ------------------------------------------------------------------------------

BEGIN { 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
}

# ------------------------------------------------------------------------------

package sl_ip;

# Ip Type constants
%IP_TYPE = ( 'STATIC' => "STATIC",
	     'FLOATING' => "FLOATING" );

# ip hidden enum values
%IP_HIDDEN_VALUES = (
             'STATIC' => 0,
	     'FLOATING' => 0 );

# ID fields separator
my $ID_SEP = '@';

# Ip object attributes
my @IP_ATTR = qw( Id
                  Name
		  Address
		  Type
		  AliasLst
		  Network );

# ------------------------------------------------------------------------------
#
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} )  && 
        ( defined $attr{'Name'} ) && 
        ( defined $attr{'Address'} ) && 
        ( defined $attr{'Type'} ) && 
        ( defined $attr{'Network'} ) ) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      $self->{'Address'} = $attr{'Address'};
      $self->{'Network'} = $attr{'Network'};
      $self->{'Type'} = $attr{'Type'};
      # Set optional attributes
      $self->{'AliasLst'} = $attr{'AliasLst'}
         if ( defined $attr{'AliasLst'} );
   }
   elsif ( defined $attr{'IpObject'} ) {
      $self->_construct( $attr{'IpObject'} );
   }
   else {
      &sl_traces::error( "Unable to create %s object.".
                         " Invalid creation method", 
			 "- sl_ip::new - ".__LINE__,
			 ( "sl_ip" ) );
   }
   &sl_traces::trace( 2, "- sl_ip::new - Creating ip object ".
                     "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $ipObject ) = @_;
   &sl_traces::trace( 3, "- sl_ip::_construct - Building ip object" );
   my $ip = sl_iterator->new( 'Object' => $ipObject );
   $self->{'Id'} = $ip->extract;
   $self->{'Name'} = $ip->extract;
   $self->{'Address'} = $ip->extract;
   $self->{'Type'} = $ip->extract;
   @{$self->{'AliasLst'}} = $ip->extractScalarArray;
   $self->{'Network'} = $ip->extract;
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_ip::serialize - Serializing ip object" );
   my $ipObject = sl_iterator->new( 'Object' =>  "" );
   foreach $attr ( @IP_ATTR ) {
      if ( $attr eq 'AliasLst' ) {
         $ipObject->serializeScalarArray( @{$self->{'AliasLst'}} );
      }
      else {
	 $ipObject->serialize( $self->{$attr} );
      }
   }
   return $ipObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_ip::display - Displaying ip" );
   print STDOUT "Ip\n";
   print STDOUT "--\n";
   foreach ( @IP_ATTR ) {
      if ( defined $self->{$_} ) {
	 if ( $_ eq 'AliasLst' ) {
	    print STDOUT "$_ = ";
            foreach $alias ( @{$self->{'AliasLst'}} ) {
               print STDOUT $alias." ";
	    }
	    print STDOUT "\n";
	 }
	 else {
	    print STDOUT "$_ = ";
	    print STDOUT "$self->{$_}";
	    print STDOUT "\n";
	 }
      }
      else {
         print STDOUT "$_ =\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_ip::getName - Getting ip ".
                     "< $self->{'Name'} > Name" );
   return $self->{'Name'};
}

# ------------------------------------------------------------------------------
#
sub getAliasLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_ip::getAliasLst - Getting ip ".
                     "< $self->{'Name'} > AliasLst" );
   if ( defined $self->{'AliasLst'} ) {
      return @{$self->{'AliasLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getType {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_ip::getType - Getting ip ".
                     "< $self->{'Name'} > Type" );
   return $self->{'Type'};
}
# ------------------------------------------------------------------------------
#
sub getAddress {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_ip::getAddress - Getting ip ".
                     "< $self->{'Name'} > Address" );
   return $self->{'Address'};
}

# ------------------------------------------------------------------------------
#
sub getNetwork {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_ip::getNetwork - Getting ip ".
                     "< $self->{'Name'} > Network" );
   return $self->{'Network'};
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   my $ipId = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_ip::iD - Ip iD is < ${ipId} >" );
   return $ipId;
}

# ------------------------------------------------------------------------------
#
sub isHidden {
   my $class = shift;
   my $enumValue = shift;
   &sl_traces::trace( 2, "- sl_ip::isHidden - Testing ". 
                      "< ${enumValue} > hidden value" );
   return $IP_HIDDEN_VALUES{$enumValue};
}

# ------------------------------------------------------------------------------
#
sub isFloating {
   my $self = shift;
   my $type = $self->getType;
   &sl_traces::trace( 2, "- sl_ip::isFloating - Evaluating ip ".
                     "< $self->{'Name'} > type" );
   if ( $type eq $IP_TYPE{'FLOATING'} ) {
      return 1;
   }
   else {
      return 0;
   }
}

# ------------------------------------------------------------------------------
#
sub checkMacAddress {
   my $class = shift;
   my ( $item, $mac ) = @_;
   my ( $m1, $m2, $m3, $m4, $m5, $m6, $trash ) = split( ":", $mac );
   &sl_traces::trace( 2, "- sl_ip::checkMacAddress - Checking < ${item} > MAC ".
                "address < ${mac} >" );
   foreach $m ( ( $m1, $m2, $m3, $m4, $m5, $m6 ) ) {
      if ( ( ! defined $m ) or 
           ( $m =~ /[^0-9a-fA-F]/ ) or
	   ( defined $trash ) or
	   ( hex($m) > 255 ) ) {
	 &sl_traces::error( "Invalid %s address < %s > for < %s >", 
		            "- sl_ip::checkMacAddress -".__LINE__,
		            ( "MAC", $mac, $item ) );
      }
   }
}

# ------------------------------------------------------------------------------
#
sub checkIpAddress {
   my $class = shift;
   my ( $item, $ip ) = @_;
   my ( $m1, $m2, $m3, $m4, $trash ) = split(  '\.', $ip );
   &sl_traces::trace( 2, "- sl_ip::checkIpAddress - Checking < ${item} > IP ".
                "address < ${ip} >" );
   foreach $m ( ( $m1, $m2, $m3, $m4 ) ) {
      if ( ( ! defined $m ) or 
           ( $m !~ /[0-9]/ ) or
	   ( defined $trash ) or
	   ( $m > 255 ) ) {
	 &sl_traces::error( "Invalid %s address < %s > for < %s >", 
		            "- sl_ip::checkMacAddress -".__LINE__,
		            ( "IP", $ip, $item ) );
      }
   }
}

# ------------------------------------------------------------------------------
#
sub splitAddress {
   my $class = shift;
   my $address = shift;
   &sl_traces::trace( 2, "- sl_ip::splitAddress - Slitting < ${address} > " );
   my ( $m1, $m2, $m3, $m4, $trash ) = split(  '\.', $address );
   return ( $m1, $m2, $m3, $m4 );
}

# ------------------------------------------------------------------------------
#
sub collapseMacAddress {
   my $class = shift;
   my ( $macAddress ) = shift;
   &sl_traces::trace( 2, "- sl_ip::collapseMacAddress - Collapsing ".
                      "mac address < $macAddress > " );
   my @macTab = split( ':', $macAddress );
   my $colAddress = "";
   foreach ( @macTab ) {
      foreach $digit ( 
         ( "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", 
	   "A", "B", "C", "D", "E", "F" ) ) {
	 $_ = "0${digit}" if ( uc($_) eq $digit );
      }
      $colAddress .= $_;
   }
   return uc( $colAddress );
}

# ------------------------------------------------------------------------------
#
sub getPrefixLength {
   my $class = shift;
   my ( $item ) = shift;
   &sl_traces::trace( 2, "- sl_ip::getPrefixLength - Getting < $item > ".
                     "prefix length" );
   my $str = unpack( "B32", pack("N", $item) );
   $str =~ s/^0+(?=\d)//; 
   my @strTab = split( "", $str );
   my $prefixLength = 0;
   foreach ( @strTab ) { 
      last if ( $_ == 0 );
      $prefixLength++ if ( $_ == 1 );
   }
   return $prefixLength;
}

# ------------------------------------------------------------------------------
#
sub checkNetworkNumber {
   my $class = shift;
   my ( $item, $number ) = @_;
   my ( $m1, $m2, $m3, $m4, $trash ) = split( '\.', $number );
   &sl_traces::trace( 2, "- sl_ip::checkNetworkNumber - Checking < ${item} > ".
                "network number < ${number} >" );
   if ( ( defined $m4 ) && ( $m4 ne "0" ) ) {
      &sl_traces::error( "Invalid network number < %s > for < %s >\n".
			 "Must end with 0", 
			 "- sl_ip::checkNetworkNumber -".__LINE__,
			 ( $number, $item ) );
   }
   else {
      sl_ip->checkIpAddress( $item, $number );
   }
}

# ------------------------------------------------------------------------------
#
sub checkNetMask {
   my $class = shift;
   my ( $item, $netmask ) = @_;
   &sl_traces::trace( 2, "- sl_ip::checkNetMask - Checking < ${item} > ".
                     "netmask < ${netmask} >" );
   sl_ip->checkIpAddress( $item, $netmask );
}

# ------------------------------------------------------------------------------
#
sub compareNetwork {
   my $self = shift;
   my ( $netNumber ) = @_;
   my $address = $self->getAddress;
   &sl_traces::trace( 2, "- sl_ip::compareNetwork - Comparing ".
                     "< ${netNumber} > to < ${address} >" );
   my $hostName = $self->getName;
   my @mLst = split(  '\.', $netNumber );
   my @nLst = split(  '\.', $address );
   my $nNet = undef;
   # Check equality btw network-number and ip-address network part
   for ( my $i=0; $i < 3; $i++ ) {
      if ( $mLst[$i] ne "0" ) {
         if ( $mLst[$i] ne $nLst[$i] ) {
	    &sl_traces::error( "Network number < %s > is not compatible ".
                               "with ip address < %s >".
			       "\nIp address < %s >",
			       " - sl_ip::compareNetwork - ".__LINE__,
			       ( $netNumber, $address, $hostName ) );
         }
      }
   }
}

# ------------------------------------------------------------------------------
#
sub makeHostsHeader {
   my $class = shift;
   my ( $destDir ) = @_;
   my $hosts = "${destDir}/hosts";
   &sl_traces::trace( 2, "- sl_ip::makeHostsHeader - Creating ".
                "< ${hosts}> file header" );
   open ( HOSTS, ">>".$hosts ) ||
      &sl_traces::error( "Cannot open < %s >", 
                	 "- sl_ip::makeHostsHeader - ".__LINE__,
			 ( $hosts ) );
   print HOSTS "#\n";
   print HOSTS "# Internet host table\n";
   print HOSTS "#\n";
   print HOSTS "127.0.0.1\tlocalhost loghost\n";
   close HOSTS;
}

# ------------------------------------------------------------------------------
#
sub addToHosts {
   my $class = shift;
   my ( $destDir, $ipAddress ) = @_;
   my $hosts = "${destDir}/hosts";
   &sl_traces::trace( 2, "- sl_ip::addToHosts - Adding hosts to < ${hosts} > ".
                      "file" );
   open( HOSTS, ">>".$hosts ) ||
      &sl_traces::error( "Cannot open < %s >", 
                         "- sl_ip::makeHostsHeader - ".__LINE__,
		         ( $hosts ) );
   my @aliasLst = $ipAddress->getAliasLst;
   print HOSTS $ipAddress->getAddress." ";
   print HOSTS $ipAddress->getName." ";
   foreach $alias ( @aliasLst ) {
      print HOSTS $alias." ";
   }
   print HOSTS "\n";
   close HOSTS;
}

# ------------------------------------------------------------------------------
#
sub makeDefaultRouter {
   my $class = shift;
   my ( $destDir, @ipLst ) = @_;
   my $defaultRouter = "${destDir}/defaultrouter";
   &sl_traces::trace( 2, "- sl_ip::makeDefaultRouter - Creating ".
                      "< ${defaultRouter} > file" );
   open ( ROUTER, ">".$defaultRouter ) ||
      &sl_traces::error( "Cannot open < %s >", 
			 "- sl_ip::makeDefaultRouter - ".__LINE__,
			 ( $defaultRouter ) );
   foreach $ip ( @ipLst ) {
      print ROUTER $ip->getName. "\n";
   }
   close ROUTER;
}

# ------------------------------------------------------------------------------
#
sub makeNotRouter {
   my $class = shift;
   my ( $destDir ) = @_;
   my $notRouter = "${destDir}/notrouter";
   &sl_traces::trace( 2, "- sl_ip::makeNotRouter - Creating ".
                      "< ${notRouter} > file" );
   open ( ROUTER, ">".$notRouter ) ||
      &sl_traces::error( "Cannot open < %s >", 
			 "- sl_ip::makeNotRouter - ".__LINE__,
			 ( $notRouter ) );
   close ROUTER;
}

# ------------------------------------------------------------------------------
#
sub makeReconfigure {
   my $class = shift;
   my ( $destDir ) = @_;
   my $reconfigure = "${destDir}/reconfigure";
   &sl_traces::trace( 2, "- sl_ip::makeReconfigure - Creating ".
                      "< ${reconfigure} > file" );
   open ( RECONF, ">".$reconfigure ) ||
      &sl_traces::error( "Cannot open < %s >", 
			 "- sl_ip::makeReconfigure - ".__LINE__,
			 ( $reconfigure ) );
   close RECONF;
}

# ------------------------------------------------------------------------------
#
sub makeNodeName {
   my $class = shift;
   my ( $destDir, $name ) = @_;
   my $nodeName = "${destDir}/nodename";
   &sl_traces::trace( 2, "- sl_ip::makeNodeName - Creating < ${nodeName} > file" );
   open ( NODE, ">".$nodeName ) ||
      &sl_traces::error( "Cannot open < %s >", 
			 "- sl_ip::makeNodeName - ".__LINE__,
			 ( $nodeName ) );
   print NODE $name."\n";
   close NODE;
}

# ------------------------------------------------------------------------------
#
sub makeHostName {
   my $class = shift;
   my ( $destDir, $nodeType, @nicLst ) = @_;
   &sl_traces::trace( 2, "- sl_ip::makeHostName - Creating hostname.xxx files" );
   foreach $nic ( @nicLst ) {
      $nicName = $nic->getDevice;
      $nicRole = $nic->getRole;
      next if ( $nicRole eq $sl_nic::NIC_ROLE{'NSM'} );
      next if ( ( $nicRole eq $sl_nic::NIC_ROLE{'CGTP'} ) and
                ( $nodeType eq $sl_node::NODE_TYPE{'DISKLESS'} ) );
      $hostName = "${destDir}/hostname.${nicName}";
      $hosts = undef;
      $ipAddress = $nic->getIpAddress;
      $hosts = $ipAddress->getName;
      unless ( defined $hosts ) {
	 &sl_traces::error( "Cannot retrieve ip address for nic < %s} >", 
			    "- sl_ip::makeHostName - ".__LINE__,
			    ( $nicName ) );
      }
      open ( HOST, ">".$hostName ) ||
	 &sl_traces::error( "Cannot open < %s >", 
			    "- sl_ip::makeHostName - ".__LINE__,
			    ( $hostName ) );
      print HOST $hosts."\n";
      close HOST;
   }
}

# ------------------------------------------------------------------------------
#
sub addNetMask {
   my $class = shift;
   my ( $destDir, $number, $netMask ) = @_;
   my $netMasks = "${destDir}/netmasks";
   &sl_traces::trace( 2, "- sl_ip::addNetMask - Creating < ${netMasks} > file" );
   open ( NET, ">>".$netMasks ) ||
	 &sl_traces::error( "Cannot open < %s >", 
			    "- sl_ip::makeNetmasks - ".__LINE__,
			    ( $netMasks ) );
   print NET "${number} ${netMask}\n";
   close NET;
}

# ------------------------------------------------------------------------------
#
sub getCgtpAddr {
   my $class = shift;
   my ( @nicLst ) = @_;
   &sl_traces::trace( 2, "- sl_ip::getCgtpAddr - Looking for cgtp address" );
   my $cgtpNic = undef;
   foreach $nic ( @nicLst ) {
      if ( $nic->getRole eq ${sl_nic::NIC_ROLE{'CGTP'}} ) {
         $cgtpNic = $nic;
	 last;
      }
   }
   if ( defined $cgtpNic ) {
      return $cgtpNic->getIpAddress;
   }
   else {
      &sl_traces::error( "Cannot find %s", 
                         "- sl_ip::getCgtpAddr - ".__LINE__,
			 ( "CGTP address" ) );
   }
   
}

# ------------------------------------------------------------------------------
#
sub getHostId {
   my $class = shift;
   my ( $ipAddress, $netmask ) = @_;
   &sl_traces::trace( 2, "- sl_ip::getHostId - Extracting host-id from ".
                      "ip address < ${ipAddress} >" );
   my @splittedAddr = $class->splitAddress( $ipAddress->getAddress );
   my @splittedMask = $class->splitAddress( $netmask );
   my @splittedHostId = ();
   for ( my $i = 0; $i < scalar @splittedAddr; $i++ ) {
      $splittedHostId[$i] = 
         $splittedAddr[$i] & ( $splittedMask[$i] ^ 0xFF );
   }
   return join( ".", @splittedHostId );
}

# ------------------------------------------------------------------------------
#
sub hostToInt {
   my $class = shift;
   my $hostId = shift;
   &sl_traces::trace( 2, "- sl_ip::hostToInt - Calculating node-id from ".
                      "host id < ${hostId} >" );
   my @splittedHostId =  $class->splitAddress( $hostId );
   my $intHostId = 0;
   for (my $i = 0; $i < 4; $i++ ) {
      $intHostId += ( 16**(3 - $i) ) * $splittedHostId[$i];
   }
   return $intHostId;
}

# ------------------------------------------------------------------------------
#
sub configure {
   my $class = shift;
   my ( $node, $nodeDir, $drawer ) = @_;
   my $nodeName = $node->getName;
   my $nodeType = $node->getType;
   &sl_traces::trace( 2, "- sl_node_solaris::configure - Making node ".
                      "< ${nodeName} > IP configuration files" );
   # Determine etc directory
   my $etcDir = "${nodeDir}/etc"; 
   ( -d $etcDir ) || &sl_utils::createDir( $etcDir );
   # Determine the board associated to the node
   my $board = $drawer->getBoard( $node->getBoard );
   my @nicLst = $board->getNicLst;
   # Look for primary network IF
   foreach $nic ( @nicLst ) {
      $role = $nic->getRole;
      if ( $role eq ${sl_nic::NIC_ROLE{'CGTP'}} ) {
         # Look for ip address associated to the primary nic
	 $ipAddress = $nic->getIpAddress;
	 $nodeName = $ipAddress->getName;
	 last;
      }
   }
   # Create network characteristics files:
   sl_ip->makeNodeName( $etcDir, $nodeName );
   sl_ip->makeNotRouter( $etcDir );
   sl_ip->makeReconfigure( $etcDir );
   sl_ip->makeHostName( $etcDir, $nodeType, @nicLst );
   sl_ip->makeHostsHeader( $etcDir );
   # The hosts file body is fed each time a node is configured. Its content is
   # appened to node specific hosts file once each node has been configured
   # (see static method sl_node->configure)
   foreach $nic ( @nicLst ) {
      sl_ip->addToHosts( $ENV{'SMCT_TMP_DIR'}, $nic->getIpAddress );
   }
}

# ------------------------------------------------------------------------------

return 1;
