#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_filesys.pm 1.9     02/11/13 SMI"
# ------------------------------------------------------------------------------

BEGIN { 
   require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/sl_iterator.pm"; 
}

# ------------------------------------------------------------------------------

package sl_filesys;

# filesys MntBt constants
%FILESYS_MNTBT = ( 'MNTBT_YES' => "yes", 
                   'MNTBT_NO' => "no" );

# filesys Type constants
%FILESYS_TYPE = ( 'UFS' => "ufs", 
                  'NFS' => "nfs",
		  'SWAP' => "swap" );

# filesys Type constants
%FILESYS_ROLE = ( 'ROOT' => "root", 
                  'EXPORT' => "export",
		  'SWAP' => "swap",
		  'SHARED' => "shared",
		  'SERVICES' => "services",
		  'DATA' => "data",
		  'DATABASE' => "database",
		  'USER' => "user" );

# filesys hidden enum values
%FILESYS_HIDDEN_VALUES = (
		  'yes' => 0, 
		  'no' => 0,
		  'ufs' => 0, 
		  'nfs' => 0,
		  'swap' => 0,
		  'root' => 0, 
		  'export' => 0,
		  'swap' => 0,
		  'shared' => 0,
		  'services' => 0,
		  'data' => 0,
		  'database' => 0,
		  'user' => 0 );
		  
# Tag used to parametrize the filesys mount points
$SWL_TAG = "<--SWLID-->";

# Default value for vfstab fields
$DEFAULT = "-";

# ID fields separator
my $ID_SEP = '@';

# filesys object attributes
my @FILESYS_ATTR = qw( Id
                       Name
		       Type
		       Role
		       Size
		       Fsck
		       RemMntPoint
		       MntPoint
		       ExportedFileSysLst
		       MntBoot
		       MntOpt );

%FILESYS_DEF_COMP_MASK = ( 'Id' => 0,
                	   'Name' => 0,
			   'Type' => 1,
			   'Role' => 1,
			   'Size' => 1,
		           'Fsck' => 1,
		           'RemMntPoint' => 1,
		           'MntPoint' => 1,
		           'ExportedFileSysLst' => 1,
		           'MntBoot' => 1,
		           'MntOpt' => 1 );


# ------------------------------------------------------------------------------
#
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} )  && 
        ( defined $attr{'Name'} ) && 
        ( defined $attr{'Type'} ) && 
	( defined $attr{'Role'} ) && 
	( defined $attr{'Size'} ) ) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      $self->{'Type'} = $attr{'Type'};
      $self->{'Role'} = $attr{'Role'};
      $self->{'Size'} = $attr{'Size'};
      $self->{'MntBoot'} = $attr{'MntBoot'};
      # Set optional attributes
      $self->{'MntOpt'} = $attr{'MntOpt'}
         if ( defined $attr{'MntOpt'} );
      $self->{'MntPoint'} = $attr{'MntPoint'}
         if ( defined $attr{'MntPoint'} );
      $self->{'RemMntPoint'} = $attr{'RemMntPoint'}
         if ( defined $attr{'RemMntPoint'} );
      $self->{'ExportedFileSysLst'} = $attr{'ExportedFileSysLst'}
         if ( defined $attr{'ExportedFileSysLst'} );
      $self->{'Fsck'} = $attr{'Fsck'}
         if ( defined $attr{'Fsck'} );      
   }
   elsif ( defined $attr{'FileSysObject'} ) {
      $self->_construct( $attr{'FileSysObject'} );
   }
   else {
      &sl_traces::error( "Unable to create %s object. ".
                         "Invalid creation method",
		         "- sl_filesys::new - ".__LINE__,
			 ( "sl_filesys" ) );
   }
   &sl_traces::trace( 2, "- sl_filesys::new - Creating filesys object ".
                "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $filesysObject ) = @_;
   &sl_traces::trace( 3, "- sl_filesys::_construct - Building filesys object" );
   my $fileSys = sl_iterator->new( 'Object' => $filesysObject );
   $self->{'Id'} = $fileSys->extract;
   $self->{'Name'} = $fileSys->extract;
   $self->{'Type'} = $fileSys->extract;
   $self->{'Role'} = $fileSys->extract;
   $self->{'Size'} = $fileSys->extract;
   $self->{'Fsck'} = $fileSys->extract;
   $self->{'RemMntPoint'} = $fileSys->extract;
   $self->{'MntPoint'} = $fileSys->extract;
   @{$self->{'ExportedFileSysLst'}} =
      $fileSys->extractObjectArray( "sl_filesys_nfs", "FileSysNfsObject" );
   $self->{'MntBoot'} = $fileSys->extract;
   $self->{'MntOpt'} = $fileSys->extract;
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::serialize - Serializing filesys ".
                      "object" );
   my $filesysObject = sl_iterator->new( 'Object' => "" );
   foreach $attr ( @FILESYS_ATTR ) {
      if ( $attr eq "ExportedFileSysLst" ) {
         $filesysObject->serializeObjectArray( @{$self->{'ExportedFileSysLst'}} );
      }
      else {
	 $filesysObject->serialize( $self->{$attr} );
      }
   }
   return $filesysObject->get;
}

# ------------------------------------------------------------------------------
#
sub compare {
   my $class = shift;
   my ( $filesys1, $filesys2 ) = @_;
   my $filesys1Name = ( defined $filesys1 ) ? $filesys1->getName:"<undefined>";
   my $filesys2Name = ( defined $filesys2 ) ? $filesys2->getName:"<undefined>";
   # String holding differences if any
   my $diffStr = undef;
   &sl_traces::trace( 2, "- sl_filesys::compare - Comparing filesys ".
                     "< ${filesys1Name} > and < ${filesys2Name} >" );
   # Extract the differences
   foreach $attr ( keys %FILESYS_DEF_COMP_MASK ) {
      $diffStr1 = "Comparing filesys : ${filesys1Name}.";
      $diffStr2 = "       to filesys : ${filesys2Name}.";
      if ( $FILESYS_DEF_COMP_MASK{$attr} ) {
         if ( &sl_utils::testAttribute( $filesys1->{$attr} ) and 
	      &sl_utils::testAttribute( $filesys2->{$attr} ) ) {
	    if ( $attr eq 'ExportedFileSysLst' ) { 
	       #$diffStr .= $diff if ( defined $diff );
	    }
	    else {
	       if ( $filesys1->{$attr} ne $filesys2->{$attr} ) {
        	  $diffStr1 .= "${attr}=".$filesys1->{$attr}."\n";
		  $diffStr2 .= "${attr}=".$filesys2->{$attr}."\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }
	    }
	 }
         elsif ( ! &sl_utils::testAttribute( $filesys1->{$attr} ) and
	           &sl_utils::testAttribute( $filesys2->{$attr} ) )  {
            unless ( $attr eq 'ExportedFileSysLst' ) {
	       $diffStr1 .= "${attr}= <undefined>\n";
	       $diffStr2 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
	    }
	    else {
	       if ( scalar @{$filesys2->{$attr}} > 0 ) {
		  $diffStr1 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
		}       
	    }
         }
         elsif ( ! &sl_utils::testAttribute( $filesys2->{$attr} ) and
	           &sl_utils::testAttribute( $filesys1->{$attr} ) ) {
            unless ( $attr eq 'ExportedFileSysLst' ) {
               $diffStr2 .= "${attr}= <undefined>\n";
	       $diffStr1 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
 	    }
	    else {
	       if ( scalar @{$filesys1->{$attr}} > 0 ) {
		  $diffStr2 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }       
	    }
         }
      }
   }
   return $diffStr;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_filesys::display - Displaying filesys" );
   print STDOUT "\nFileSys\n";
   print STDOUT "-------\n";
   foreach ( @FILESYS_ATTR ) {
      if ( defined $self->{$_} ) {
         if ( $_ eq "ExportedFileSysLst" ) {
	    print STDOUT "$_ =\n\n";
	    foreach $fileSys ( @{$self->{$_}} ) {
	       $fileSys->display( $brief );
	    }
	    print STDOUT "\n";
	 }
	 else {
	    print STDOUT "$_ = ";
	    print STDOUT "$self->{$_}";
	    print STDOUT "\n";
	 }
      }
      else {
	 print STDOUT "$_ = ";
         print STDOUT "\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub getMntPoint {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::getMntPoint - Getting MntPoint attribute of ".
                "< $self->{'Name'} >" );
   if ( defined $self->{'MntPoint'} ) {
      return $self->{'MntPoint'};
   }
   else {
      return "-";
   }
}

# ------------------------------------------------------------------------------
#
sub getExportedFileSysLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::getExportedFileSysLst - ".
                      "Getting < $self->{'Name'} > ExportedFileSysLst" );
   if ( defined $self->{'ExportedFileSysLst'} ) {
      return @{$self->{'ExportedFileSysLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getType {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::getType - Getting Type attribute of ".
                "< $self->{'Name'} >" );
   if ( defined $self->{'Type'} ) {
      return $self->{'Type'};
   }
   else {
      return "-";
   }
}

# ------------------------------------------------------------------------------
#
sub getSize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::getSize - Getting Size attribute of ".
                "< $self->{'Name'} >" );
   return $self->{'Size'};
}

# ------------------------------------------------------------------------------
#
sub getMntOpt {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::getMntOpt - Getting MntOpt attribute of ".
                "< $self->{'Name'} >" );
   if ( defined $self->{'MntOpt'} ) {
      return $self->{'MntOpt'};
   }
   else {
      return "-";
   }
}

# ------------------------------------------------------------------------------
#
sub getMntBoot {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::getMntBoot - Getting MntBoot attribute of ".
                "< $self->{'Name'} >" );
   if ( defined $self->{'MntBoot'} ) {
      return $self->{'MntBoot'};
   }
   else {
      return "-";
   }
}

# ------------------------------------------------------------------------------
#
sub getFsck {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::getFsck - Getting Fsck attribute of ".
                "< $self->{'Name'} >" );
   if ( defined $self->{'Fsck'} ) {
      return $self->{'Fsck'};
   }
   else {
      return "-";
   }
}

# ------------------------------------------------------------------------------
#
sub getRole {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::getRole - Getting Role attribute of ".
                "< $self->{'Name'} >" );
   if ( defined $self->{'Role'} ) {
      return $self->{'Role'};
   }
   else {
      return undef;
   }
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::getName - Getting Name attribute of ".
                "< $self->{'Name'} >" );
   if ( defined $self->{'Name'} ) {
      return $self->{'Name'};
   }
   else {
      return undef;
   }
}

# ------------------------------------------------------------------------------
#
sub setName {
   my $self = shift;
   my ( $name ) = @_;
   &sl_traces::trace( 2, "- sl_filesys::setName - Setting < $self->{'Name'} > ".
                      "name" );
   if ( defined $name ) {
      $self->{'Name'} = $name;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_filesys::setName - ".__LINE__,
			 ( "Name" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub setId {
   my $self = shift;
   my ( $id ) = @_;
   &sl_traces::trace( 2, "- sl_filesys::setId - Setting < $self->{'Name'} > ".
                      "Id" );
   if ( defined $id ) {
      $self->{'Id'} = $id;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_filesys::setId - ".__LINE__,
			 ( "Id" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub isNFS {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_filesys::isNFS - Determining filesys Type for ".
                      "< $self->{'Name'} >" );
   if ( $self->{'Type'} eq $FILESYS_TYPE{'NFS'} ) {
      return 1;
   }
   else {
      return 0;
   }
}

# ------------------------------------------------------------------------------
#
sub isHidden {
   my $class = shift;
   my $enumValue = shift;
   &sl_traces::trace( 2, "- sl_filesys::isHidden - Testing ". 
                      "< ${enumValue} > hidden value" );
   return $FILESYS_HIDDEN_VALUES{$enumValue};
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   my $fileSysId = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_filesys::iD - Filesys iD is < ${fileSysId} >" );
   return $fileSysId;
}

# ------------------------------------------------------------------------------
#
sub configure {
   my $class = shift;
   my ( $node, $nodeDir, $swlVersion, $clusterName, $drawer ) = @_;
   my $nodeName = $node->getName;
   &sl_traces::trace( 2, "- sl_filesys::configure - Creating node ".
                      "< ${nodeName} > FS configuration" );
   my $vfstab = "${nodeDir}/etc/vfstab";
   # Create invariant vfstab part
   my $command = 
         "echo \"#device		device		mount		FS	fsck	mount	mount\"".
	 " > ${vfstab};".
         "echo \"#to mount	to fsck		point		type	pass	at boot	options\"".
	 " >> ${vfstab};".
         "echo \"#\"".
	 " >> ${vfstab};".
         "echo \"#/dev/dsk/c1d0s2 /dev/rdsk/c1d0s2 /usr		ufs	1	yes	-\"".
	 " >> ${vfstab};".
         "echo \"/proc                        -  /proc     proc  -  no  -\"".
	 " >> ${vfstab};".
         "echo \"fd                           -  /dev/fd   fd    -  no  -\"".
	 " >> ${vfstab};".
         "echo \"swap                         -  /tmp      tmpfs -  yes -\"".
	 " >> ${vfstab};";
   # Read file system info from the cluster registry
   my $exportRoot = 
      &sl_registry::getValue( $clusterName, "EXPORT_ROOT" );
   my $sharedRoot = 
      &sl_registry::getValue( $clusterName, "SHARED_ROOT" );
   my $servicesRoot = 
      &sl_registry::getValue( $clusterName, "SERVICES_ROOT" );
   my $servicesRemote = 
      &sl_registry::getValue( $clusterName, "SERVICES_REMOTE" );
   my $servicesMntOpt = 
      &sl_registry::getValue( $clusterName, "SERVICES_MNT_OPT" );
   my $dataRoot = 
      &sl_registry::getValue( $clusterName, "DATA_ROOT" );
   my $dataRemote = 
      &sl_registry::getValue( $clusterName, "DATA_REMOTE" );
   my $dataMntOpt = 
      &sl_registry::getValue( $clusterName, "DATA_MNT_OPT" );
   my $databaseRoot = 
      &sl_registry::getValue( $clusterName, "DATABASE_ROOT" );
   my $databaseRemote = 
      &sl_registry::getValue( $clusterName, "DATABASE_REMOTE" );
   my $databaseMntOpt = 
      &sl_registry::getValue( $clusterName, "DATABASE_MNT_OPT" );
   # Return the name of the master node
   my $cgtpMaster = $sl_domain::DOMAIN_MASTER_NAME;
   if ( ( $node->isMasterEligible ) or ( $node->isDataless ) ) {
      foreach $diskOid ( @{$node->{'DiskLst'}} ) {
         my $disk = $drawer->getDisk( $diskOid );
	 my $diskName = $disk->getDevice;
	 foreach $slice ( $disk->getSliceLst ) {
	    my $rawDev = $slice->getRawDev;
	    my $blkDev = $slice->getBlockDev;
	    my $sliceNumber = $slice->getNumber;
	    my $fileSys = $slice->getFileSys;
	    my $fullSliceName = "";
	    if ( ( $slice->isMeta ) or
	         ( $slice->isSoft ) ) {
	       $fullSliceName = "d".${sliceNumber};
	    }
	    else {
	       $fullSliceName = ${diskName}."s".${sliceNumber};
	    }
	    if ( defined $fileSys ) {
	       my $fsType = $fileSys->getType;
	       my $mntPt = $fileSys->getMntPoint;
	       my $mntOpt = $fileSys->getMntOpt;
	       my $mntBoot = $fileSys->getMntBoot;
	       my $fsck = $fileSys->getFsck;
	       my $role = $fileSys->getRole;
	       if ( $fsck ne ${sl_filesys::DEFAULT} ) {
	          # if FSCK pass is not set do not provide raw device
		  $command = $command. 
	             "echo \"${blkDev}/${fullSliceName} ".
		     "${rawDev}/${fullSliceName} ${mntPt} ".
                     "${fsType} ${fsck} ${mntBoot} ${mntOpt}\">> ${vfstab};";
               }
	       else {
		  $command = $command. 
	             "echo \"${blkDev}/${fullSliceName} ".
		     "- ${mntPt} ".
                     "${fsType} ${fsck} ${mntBoot} ${mntOpt}\">> ${vfstab};";
	       }
               # Process NFS file systems
	       my @exportedFileSysLst = $fileSys->getExportedFileSysLst;
	       foreach $expFileSys ( @exportedFileSysLst ) {
	          my $mntOpt = $expFileSys->getMntOpt;
		  my $mntPt = $expFileSys->getMntPoint;
                  # Replace swl tag by swlVersion value
		  $mntPt =~ s/${sl_filesys::SWL_TAG}/${swlVersion}/g;
		  my $remMntPt = $expFileSys->getRemMntPoint;
		  my $nfsMntPt =  "${sharedRoot}/${mntPt}";
        	  # Create mount points directories for nfs mount
		  ( -d "${nodeDir}/$nfsMntPt" ) || 
		     &sl_utils::createDir( "${nodeDir}/$nfsMntPt" );
        	  # Add nfs filesystems
        	  $command = $command.
                     "echo \"${cgtpMaster}:${nfsMntPt} - ".
		     "${remMntPt} nfs - no ${mntOpt}\" >> ${vfstab};";
	       }
	    }
	 }
      }
   }
   else {
      # Process diskless nodes
      my $groupName = $node->getGroupName;
      my $nodeArch = $sl_board::ARCH2DIR{$node->getArch};
      my $nodeOs = $sl_node::OS2DIR{$node->getOs};
      my $nodeId = $node->getId;
      # NFS mount points used by the diskless nodes
      my $nfsRemote = "${sharedRoot}/${dataRoot}";
      my $nfsServices = "${sharedRoot}/${servicesRoot}";
      my $nfsLdap = "${sharedRoot}/${databaseRoot}"
         if ( defined ${databaseRoot} );
      my $nfsRoot = "${exportRoot}/${swlVersion}/${nodeArch}/${nodeOs}/".
		    "${groupName}/root/${nodeId}";
      my $nfsSwap = "${exportRoot}/${swlVersion}/${nodeArch}/${nodeOs}/".
		    "${groupName}/swap/${nodeId}/swap";
      my $nfsUsr = "${exportRoot}/${swlVersion}/${nodeArch}/${nodeOs}/".
	   	   "${groupName}/usr";
      # Create vfstab
      $command = $command.
         "echo \"${cgtpMaster}:${nfsRoot} -  /         nfs   -  -   rw\"".
	 ">> ${vfstab};".
         "echo \"${cgtpMaster}:${nfsSwap} -  /dev/swap nfs   -  -   -\"".
	 " >> ${vfstab};".
         "echo \"/dev/swap                    -  -         swap  -  -  -\"".
	 " >> ${vfstab};".
         "echo \"${cgtpMaster}:${nfsUsr}  -  /usr      nfs   -  -   ro\"".
	 " >> ${vfstab};".
         "echo \"${cgtpMaster}:${nfsRemote} - ${dataRemote} ".
	 "nfs - yes ${dataMntOpt}\" >> ${vfstab};".
         "echo \"${cgtpMaster}:${nfsServices} - ${servicesRemote} ".
	 "nfs - yes ${servicesMntOpt}\" >> ${vfstab};";
      $command .= "echo \"${cgtpMaster}:${nfsLdap} - ${databaseRemote} ".
	          "nfs - yes ${databaseMntOpt}\" >> ${vfstab};"
         if ( defined ${databaseRoot} );
   }
   if ( $node->isDataless ) {
      # Terminate vfstab generation for the DLN
      # Process NFS mount points used by the dataless nodes
      my $nfsRemote = "${sharedRoot}/${dataRoot}";
      my $nfsServices = "${sharedRoot}/${servicesRoot}";
      my $nfsLdap = "${sharedRoot}/${databaseRoot}" 
         if ( defined ${databaseRoot} );
      # Create vfstab
      $command = $command.
         "echo \"${cgtpMaster}:${nfsRemote} - ${dataRemote} ".
	 "nfs - yes ${dataMntOpt}\" >> ${vfstab};".
         "echo \"${cgtpMaster}:${nfsServices} - ${servicesRemote} ".
	 "nfs - yes ${servicesMntOpt}\" >> ${vfstab};";
      $command .= "echo \"${cgtpMaster}:${nfsLdap} - ${databaseRemote} ".
	          "nfs - yes ${databaseMntOpt}\" >> ${vfstab};"
         if ( defined ${databaseRoot} );  
   }
   # Create the remote mount point directories for DLN and diskless nodes
   # Create data mnt point directory
   $servicesDir = "${nodeDir}/${servicesRemote}";
   ( -d $servicesDir ) || &sl_utils::createDir( $servicesDir );
   # Create services mnt point directory
   $remoteDir = "${nodeDir}/${dataRemote}";
   ( -d $remoteDir ) || &sl_utils::createDir( $remoteDir );
   if ( defined ${databaseRoot} ) {
      # Create ldap mnt point directory
      $ldapDir = "${nodeDir}/${databaseRemote}";
      ( -d $ldapDir ) || &sl_utils::createDir( $ldapDir );
   }
   &sl_utils::execCmd( $command );
}

# ------------------------------------------------------------------------------

return 1;
