#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_domain.pm 1.9     02/11/13 SMI"
# ------------------------------------------------------------------------------

BEGIN { 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
}

package sl_domain;

# ID fields separator
my $ID_SEP = '@';

# Domain object attributes
my @DOMAIN_ATTR = qw( Id
                      Name
                      IpAddressLst
		      NetworkLst
		      RouterLst );
		      
# Domain master ip address name
$DOMAIN_MASTER_NAME = "master-cgtp";

# Domain ID range
$DOMAIN_ID_MIN = 1;
$DOMAIN_ID_MAX = 255;
     
%DOMAIN_DEF_COMP_MASK = ( 'Id' => 0,
                          'IpAddressLst' => 0,
		          'NetworkLst' => 0,
		          'RouterLst' => 0,
                          'Name' => 1 );

# ------------------------------------------------------------------------------
#
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( defined $attr{'Name'} ) {
      # Set mandatory attributes
      $self->{'Name'} = $attr{'Name'};
      # Set optional attributes
      $self->{'IpAddressLst'} = $attr{'IpAddressLst'}
         if ( defined $attr{'IpAddressLst'} );
      $self->{'NetworkLst'} = $attr{'NetworkLst'}
         if ( defined $attr{'NetworkLst'} );
      $self->{'RouterLst'} = $attr{'RouterLst'}
         if ( defined $attr{'RouterLst'} );
      $self->{'Id'} = $attr{'Id'}
         if ( defined $attr{'Id'} );
   }
   elsif ( defined $attr{'DomainObject'} ) {
      $self->_construct( $attr{'DomainObject'} );
   }
   else {
      &sl_traces::error( "Unable to create %s object.".
                         " Invalid creation method", 
			 "- sl_domain::new - ".__LINE__,
			 ( "sl_domain" ) );
   }
   &sl_traces::trace( 2, "- sl_domain::new - Creating domain object ".
                     "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $domainObject ) = @_;
   &sl_traces::trace( 3, "- sl_domain::_construct - Building domain object" );
   my $domain = sl_iterator->new( 'Object' => $domainObject );
   $self->{'Id'} = $domain->extract;
   $self->{'Name'} = $domain->extract;
   @{$self->{'IpAddressLst'}} =
      $domain->extractObjectArray( "sl_ip", "IpObject" );
   @{$self->{'NetworkLst'}} =
      $domain->extractObjectArray( "sl_network", "NetworkObject" );
   @{$self->{'RouterLst'}} =
      $domain->extractObjectArray( "sl_router", "RouterObject" );
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_domain::serialize - Serializing ".
                     "domain object" );
   my $domainObject =  sl_iterator->new( 'Object' => "" );
   foreach $attr ( @DOMAIN_ATTR ) {
      if ( $attr eq 'IpAddressLst' ) {
         $domainObject->serializeObjectArray( @{$self->{'IpAddressLst'}} );
      } 
      elsif ( $attr eq 'NetworkLst' ) {
         $domainObject->serializeObjectArray( @{$self->{'NetworkLst'}} );
      } 
      elsif ( $attr eq 'RouterLst' ) {
         $domainObject->serializeObjectArray( @{$self->{'RouterLst'}} );
      } 
      else {
         $domainObject->serialize( $self->{$attr} ); 
      }
   }
   return $domainObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_domain::display - Displaying domain" );
   print STDOUT "Domain\n";
   print STDOUT "------\n";
   foreach ( @DOMAIN_ATTR ) {
      if ( defined $self->{$_} ) {
         if ( $_ eq 'IpAddressLst' ) {
	    print STDOUT "$_ = \n\n";
	    foreach $ip ( @{$self->{'IpAddressLst'}} ) {
	       $ip->display( $brief );
	    }
	 }
         elsif ( $_ eq 'NetworkLst' ) {
	    print STDOUT "$_ = \n\n";
	    foreach $network ( @{$self->{'NetworkLst'}} ) {
	       $network->display( $brief );
	    }
	 }
         elsif ( $_ eq 'RouterLst' ) {
	    print STDOUT "$_ = \n\n";
	    foreach $router ( @{$self->{'RouterLst'}} ) {
	       $router->display( $brief );
	    }
	 }
	 else {
	    print STDOUT "$_ = ";
	    print STDOUT "$self->{$_}";
	    print STDOUT "\n";
         }
      }
      else {
         print STDOUT "$_ =\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub compare {
   my $class = shift;
   my ( $domain1, $domain2 ) = @_;
   my $domain1Name = ( defined $domain1 ) ? $domain1->getName:"<undefined>";
   my $domain2Name = ( defined $domain2 ) ? $domain2->getName:"<undefined>";
   &sl_traces::trace( 2, "- sl_domain::compare - Comparing domain ".
                     "< ${domain1Name} > and < ${domain2Name} >" );
   # String holding differences if any
   my $diffStr = undef;
   # Extract the differences if any
   foreach $attr ( keys %DOMAIN_DEF_COMP_MASK ) {
      $diffStr1 = "Comparing domain : ${domain1Name}.";
      $diffStr2 = "       to domain : ${domain2Name}.";
      if ( $DOMAIN_DEF_COMP_MASK{$attr} ) {
         if ( &sl_utils::testAttribute( $domain1->{$attr} ) and 
	      &sl_utils::testAttribute( $domain2->{$attr} ) and
	      ( $domain1->{$attr} ne $domain2->{$attr} ) ) {
            $diffStr1 .= "${attr}=".$domain1->{$attr}."\n";
	    $diffStr2 .= "${attr}=".$domain2->{$attr}."\n";
            $diffStr .= $diffStr1.$diffStr2;
	 }
         elsif ( ! &sl_utils::testAttribute( $domain1->{$attr} ) and
	           &sl_utils::testAttribute( $domain2->{$attr} ) )  {
               $diffStr1 .= "${attr}= <undefined>\n";
	       $diffStr2 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
         }
         elsif ( ! &sl_utils::testAttribute( $domain2->{$attr} ) and
	           &sl_utils::testAttribute( $domain1->{$attr} ) ) {
               $diffStr2 .= "${attr}= <undefined>\n";
	       $diffStr1 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
	 }
      }
   }
   return $diffStr;
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_domain::getName - Getting domain Name" );
   return $self->{'Name'};
}

# ------------------------------------------------------------------------------
#
sub getIpAddressLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_domain::getIpAddressLst - Getting domain ".
                     "IpAddressLst" );
   return @{$self->{'IpAddressLst'}};
}

# ------------------------------------------------------------------------------
#
sub getNetworkLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_domain::getNetworkLst - Getting domain ".
                     "NetworkLst" );
   return @{$self->{'NetworkLst'}};
}

# ------------------------------------------------------------------------------
#
sub getNetwork {
   my $self = shift;
   my ( $networkOid ) = @_;
   &sl_traces::trace( 2, "- sl_domain::getNetwork - Getting domain ".
                     "Network < ${networkOid} >" );
   my $network = undef;
   my @networkLst = $self->getNetworkLst; 
   foreach $net ( @networkLst ) {
      if ( $net->iD eq $networkOid ) {
         $network = $net;
      }
   }
   return $network;
}

# ------------------------------------------------------------------------------
#
sub getRouterLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_domain::getRouterLst - Getting domain ".
                     "RouterLst" );
   return @{$self->{'RouterLst'}};
}

# ------------------------------------------------------------------------------
#
sub getMasterIpAddress {
   my $self = shift;
   my $masterIp = undef;
   &sl_traces::trace( 2, "- sl_domain::getRouterLst - Getting domain ".
                     "master floating address" );
   my @floatingAddrLst = $self->getIpAddressLst;
   foreach $ip ( @floatingAddrLst ) {
      my $ipName = $ip->getName;
      if ( $ipName eq $DOMAIN_MASTER_NAME ) {
         $masterIp = $ip;
	 last;
      }
   }
   &sl_traces::error( "Master floating address < %s > is not defined",
                      "- sl_domain::getMasterIpAddress - ".__LINE__ ,
		      ( "${DOMAIN_MASTER_NAME}" ) )
      unless ( defined $masterIp );
   return $masterIp;
}

# ------------------------------------------------------------------------------
#
sub deploy {
   my $self = shift;
   my ( $destDir ) = @_;
   &sl_traces::trace( 2, "- sl_domain::deploy - Configuring domain ".
                     "< $self->{'Name'} >" );
   # Add floating addresses to hosts file 
   my @ipAddressLst = $self->getIpAddressLst;
   foreach $ipAddress ( @ipAddressLst ) {
      sl_ip->addToHosts( $destDir, $ipAddress );
   }
   # Configure routers
   my @routerLst = $self->getRouterLst;
   foreach $router ( @routerLst ) {
      $router->deploy( $destDir );
   }
   # Configure networks
   my @networkLst = $self->getNetworkLst;
   foreach $network ( @networkLst ) {
      $network->deploy( $destDir );
   }
}

# ------------------------------------------------------------------------------
#
sub getId {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_domain::getId - Getting domain ".
                     " < $self->{'Name'} > id" );
   return $self->{'Id'};
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   my $domainId = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_domain::iD - Domain iD is < ${domainId} >" );
   return $domainId;
}

# ------------------------------------------------------------------------------

return 1;
