#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)sl_cluster.pm 1.10     02/10/31 SMI"
# ------------------------------------------------------------------------------

BEGIN { 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_node.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_node_solaris.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_group.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_shelf.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_network.pm"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_utils.pl"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_registry.pl"; 
   require "$ENV{'NHAS_PROD_DIR'}/nhsmct/lib/sl_iterator.pm"; 
}

# ------------------------------------------------------------------------------

package sl_cluster;

# ID fields separator
my $ID_SEP = '@';

# ID min and max values
$ID_MIN = 0;
$ID_MAX = 255;

#  node object attributes
my @CLUSTER_ATTR = qw( Id
                       Name
		       ShelfLst
		       Domain
		       GroupLst );

%CLUSTER_DEF_COMP_MASK = ( 'Id' => 0,
                	   'Name' => 1,
			   'ShelfLst' => 1,
			   'Domain' => 1,
			   'GroupLst' => 1 );

# ------------------------------------------------------------------------------
#
sub new {
   my $class = shift;
   my %attr = @_;
   my $self = {};
   bless $self, $class;
   if ( ( defined $attr{'Id'} ) &&
	( defined $attr{'Name'} ) &&
	( defined $attr{'ShelfLst'} ) &&
	( defined $attr{'Domain'} )) {
      # Set mandatory attributes
      $self->{'Id'} = $attr{'Id'};
      $self->{'Name'} = $attr{'Name'};
      $self->{'ShelfLst'} = $attr{'ShelfLst'};
      $self->{'Domain'} = $attr{'Domain'};
      # Set optional attributes
      $self->{'GroupLst'} = $attr{'GroupLst'}
         if ( defined $attr{'GroupLst'} );
   }
   elsif ( defined $attr{'ClusterObject'} ) {
      $self->_construct( $attr{'ClusterObject'} );
   }
   else {
      &sl_traces::error( "Unable to create %s object. ".
                         "Invalid creation method", 
		         "- sl_cluster::new - ".__LINE__,
			 ( "sl_cluster" ) );
   }
   &sl_traces::trace( 2, "- sl_cluster::new - Creating cluster object ".
                "< $self->{'Name'} > Id < $self->{'Id'} >" );
   return $self;
}

# ------------------------------------------------------------------------------
#
sub _construct {
   my $self = shift;
   my ( $clusterObject ) = @_;
   &sl_traces::trace( 3, "- sl_cluster::_construct - Building cluster object" );
   my $cluster = sl_iterator->new( 'Object' => $clusterObject );
   $self->{'Id'} = $cluster->extract;
   $self->{'Name'} = $cluster->extract;
   @{$self->{'ShelfLst'}} =
      $cluster->extractObjectArray( "sl_shelf", "ShelfObject" );
   $self->{'Domain'} = 
      $cluster->extractObject( "sl_domain", "DomainObject" );
   @{$self->{'GroupLst'}} =
      $cluster->extractObjectArray( "sl_group", "GroupObject" );
}

# ------------------------------------------------------------------------------
#
sub serialize {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_cluster::serialize - Serializing cluster object" );
   my $clusterObject = sl_iterator->new( 'Object' =>  "" );
   foreach $attr ( @CLUSTER_ATTR ) {
      if ( $attr eq 'GroupLst' ) {
         $clusterObject->serializeObjectArray( @{$self->{'GroupLst'}} );
      }
      elsif ( $attr eq 'ShelfLst' ) {
         $clusterObject->serializeObjectArray( @{$self->{'ShelfLst'}} );
      }
      elsif ( $attr eq 'Domain' ) {
         $clusterObject->serializeObject( $self->{'Domain'} );
      }
      else {
         $clusterObject->serialize( $self->{$attr} );
      }
   }
   return $clusterObject->get;
}

# ------------------------------------------------------------------------------
#
sub display {
   my $self = shift;
   my ( $brief ) = @_;
   &sl_traces::trace( 2, "- sl_cluster::display - Displaying cluster" );
   print STDOUT "\nCluster\n";
   print STDOUT "-------\n";
   foreach ( @CLUSTER_ATTR ) {
      if ( $_ eq 'GroupLst' ) {
         print STDOUT "$_ = \n";
         foreach $group (@{$self->{'GroupLst'}}) {
	    $group->display( $brief );
	 }
      }
      elsif ( $_ eq 'ShelfLst' ) {
         print STDOUT "$_ =\n";
         foreach $shelf (@{$self->{'ShelfLst'}}) {
            $shelf->display( $brief );
         }
      }
      elsif ( $_ eq 'Domain' ) {
         print STDOUT "$_ =\n\n";
         $self->{$_}->display( $brief );
      }
      else {
         print STDOUT "$_ = ";
         print STDOUT "$self->{$_}" if ( defined $self->{$_} );
         print STDOUT "\n";
      }
   }
   print STDOUT "\n";
}

# ------------------------------------------------------------------------------
#
sub getBoard {
   my $self = shift;
   my ( $boardOid ) = @_;
   my @shelfLst = $self->getShelfLst;
   &sl_traces::trace( 2, "- sl_cluster::getBoard - Getting < ${boardOid} > ".
                      "board" );
   my $board = undef;
   SHELF:
   foreach $shelf ( @shelfLst ) {
      my @drawerLst = $shelf->getDrawerLst;
      foreach $drawer ( @drawerLst ) {
         $board = $drawer->getBoard( $boardOid );
	 last SHELF if defined $board;
      }
   }
   return $board;
}

# ------------------------------------------------------------------------------
#
sub getNodeLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_cluster::getNodeLst - Getting < $self->{'Name'} > ".
                      "nodes list" );
   my @nodeLst = ();
   foreach $group ( @{$self->{'GroupLst'}} ) {
      @groupNodeLst = $group->getNodeLst;
      # Retrieve board objects
      foreach $node ( @groupNodeLst ) {
         my $boardOid = $node->getBoard;
	 my $board = $self->getBoard( $boardOid );
	 $node->setBoardObject( $board );
      }
      push( @nodeLst, @groupNodeLst ) if ( defined @groupNodeLst );
   }
   return @nodeLst;
}

# ------------------------------------------------------------------------------
#
sub getGroupLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_cluster::getGroupLst - Getting < $self->{'Name'} > ".
                      "group list" );
   if ( defined @{$self->{'GroupLst'}} ) {
      my @groupLst = @{$self->{'GroupLst'}};
      foreach $group ( @groupLst ) {
         my @groupObjectLst = ();
         foreach $embeddedGroup ( $group->getEmbeddedGroupLst ) {
	    foreach $grp ( @groupLst ) {
	       push( @groupObjectLst, $grp )
	          if ( $grp->getName eq $embeddedGroup );
	    }
	 }
	 $group->setEmbeddedGroupObjectLst( @groupObjectLst )
	    if ( scalar @groupObjectLst > 0 );
      }
      $self->setGroupLst( @groupLst );
      return @groupLst;
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getDomain {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_cluster::getDomain - Getting ".
                     "< $self->{'Name'} > domain" );
   return $self->{'Domain'};
}

# ------------------------------------------------------------------------------
#
sub getName {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_cluster::getName - Getting ".
                     "< $self->{'Name'} > name" );
   return $self->{'Name'};
}

# ------------------------------------------------------------------------------
#
sub getShelfLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_cluster::getShelfLst - Getting ".
                     "< $self->{'Name'} > ShelfLst" );
   if ( defined @{$self->{'ShelfLst'}} ) {
      return @{$self->{'ShelfLst'}};
   }
   else {
      return ();
   }
}

# ------------------------------------------------------------------------------
#
sub getSoftLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_cluster::getSoftLst - Getting cluster ".
                      "< $self->{'Name'} > software list" );
   my %softLst = ();
   # Get software lists
   my @nodeLst = $self->getNodeLst;
   foreach $group ( @{$self->{'GroupLst'}} ) {
      # Get node software lists
      foreach ( $group->getPkgLst, $group->getPatLst, $group->getSoftLst ) {
         $softLst{$_->iD} = $_;
      }
   }
   return %softLst;
}

# ------------------------------------------------------------------------------
#
sub setGroupLst {
   my $self = shift;
   my ( @groupLst ) = @_;
   &sl_traces::trace( 2, "- sl_cluster::setGroupLst - Setting ".
                      "< $self->{'Name'} > group list" );
   if ( defined $groupLst[0] ) {
      @{$self->{'GroupLst'}} = @groupLst;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_cluster::setGroupLst - ".__LINE__,
			 ( "Nodes group list" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub setShelfLst {
   my $self = shift;
   my ( @shelfLst ) = @_;
   &sl_traces::trace( 2, "- sl_cluster::setShelfLst - Setting ".
                      "< $self->{'Name'} > shelf list" );
   if ( defined $shelfLst[0] ) {
      @{$self->{'ShelfLst'}} = @shelfLst;
   }
   else {
      &sl_traces::error( "%s is undefined",
                         "- sl_cluster::setShelfLst - ".__LINE__,
			 ( "Shelf list" ) );  
   }
}

# ------------------------------------------------------------------------------
#
sub compare {
   my $class = shift;
   my ( $cluster1, $cluster2 ) = @_;
   my $cluster1Name = ( defined $cluster1 ) ? $cluster1->getName:"<undefined>";
   my $cluster2Name = ( defined $cluster2 ) ? $cluster2->getName:"<undefined>";
   &sl_traces::trace( 2, "- sl_cluster::compare - Comparing clusters ".
                     "< ${cluster1Name} > and < ${cluster2Name} >" );
   # String holding differences if any
   my $diffStr = undef;
   # Extract the differences if any
   foreach $attr ( keys %CLUSTER_DEF_COMP_MASK ) {
      $diffStr1 = "Comparing cluster : ${cluster1Name}.";
      $diffStr2 = "       to cluster : ${cluster2Name}.";
      if ( $CLUSTER_DEF_COMP_MASK{$attr} ) {
         if ( &sl_utils::testAttribute( $cluster1->{$attr} ) and 
	      &sl_utils::testAttribute( $cluster2->{$attr} ) ) {
            if ( $attr eq 'GroupLst' ) {
	       # Sort the clusters lists for comparison
	       $cluster1->setGroupLst( 
		  &sl_utils::sortListByName( $cluster1->getGroupLst ) );
	       $cluster2->setGroupLst( 
		  &sl_utils::sortListByName( $cluster2->getGroupLst ) );
               # Look for different groups
	       @cluster1GroupLst = $cluster1->getGroupLst;
	       @cluster2GroupLst = $cluster2->getGroupLst;
	       for ($i=0; $i < scalar @cluster1GroupLst; $i++) {
		  $diff = sl_group->compare( 
		           $cluster1GroupLst[$i], $cluster2GroupLst[$i] );
		  $diffStr .= $diffStr1."GroupLst[$i]\n".
		              $diffStr2."GroupLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
	    elsif ( $attr eq 'ShelfLst' ) {
	       # Sort the clusters lists for comparison
	       $cluster1->setShelfLst( 
		  &sl_utils::sortListByName( $cluster1->getShelfLst ) );
	       $cluster2->setShelfLst( 
		  &sl_utils::sortListByName( $cluster2->getShelfLst ) );
               # Look for different shelves
	       @cluster1ShelfLst = $cluster1->getShelfLst;
	       @cluster2ShelfLst = $cluster2->getShelfLst;
	       for ($i=0; $i < scalar @cluster1ShelfLst; $i++) {
		  $diff = sl_shelf->compare( 
		           $cluster1ShelfLst[$i], $cluster2ShelfLst[$i] );
		  $diffStr .= $diffStr1."ShelfLst[$i]\n".
		              $diffStr2."ShelfLst[$i]\n".$diff 
			         if ( defined $diff );
	       }
	    }
	    elsif ( $attr eq 'Domain' ) {
               # Look for different shelves
	       $cluster1Domain = $cluster1->getDomain;
	       $cluster2Domain = $cluster2->getDomain;
	       $diff = sl_domain->compare($cluster1Domain, $cluster2Domain );
	       $diffStr .= $diffStr1."Domain\n".
		           $diffStr2."Domain\n".$diff 
			      if ( defined $diff );
	    }
	    else {
	       if ( $cluster1->{$attr} ne $cluster2->{$attr} ) {
        	  $diffStr1 .= "${attr}=".$cluster1->{$attr}."\n";
		  $diffStr2 .= "${attr}=".$cluster2->{$attr}."\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }
            }
	 }
         elsif ( ! &sl_utils::testAttribute( $cluster1->{$attr} ) and
	           &sl_utils::testAttribute( $cluster2->{$attr} ) )  {
            unless ( ( $attr eq 'GroupLst' ) or
	             ( $attr eq 'ShelfLst' ) ) {
	       $diffStr1 .= "${attr}= <undefined>\n";
	       $diffStr2 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
	    }
	    else {
	       if ( scalar @{$cluster2->{$attr}} > 0 ) {
		  $diffStr1 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
		}       
	    }
         }
         elsif ( ! &sl_utils::testAttribute( $cluster2->{$attr} ) and
	           &sl_utils::testAttribute( $cluster1->{$attr} ) ) {
            unless ( ( $attr eq 'GroupLst' ) or
	             ( $attr eq 'ShelfLst' ) ) {
               $diffStr2 .= "${attr}= <undefined>\n";
	       $diffStr1 .= "${attr}= <defined>\n";
               $diffStr .= $diffStr1.$diffStr2;
 	    }
	    else {
	       if ( scalar @{$cluster1->{$attr}} > 0 ) {
		  $diffStr2 .= "${attr}= <undefined>\n";
		  $diffStr2 .= "${attr}= <defined>\n";
        	  $diffStr .= $diffStr1.$diffStr2;
	       }       
	    }
	 }
      }
   }
   return $diffStr;
}

# ------------------------------------------------------------------------------
#
sub _checkGroupLst {
   my $self = shift;
   &sl_traces::trace( 2, "- sl_cluster::_checkGroupLst - Checking cluster ".
                     "< $self->{'Name'} > group list" );
   my @groupLst = $self->getGroupLst;
   my $masterGroupFound = 0;
   foreach $group ( @groupLst ) {
      $masterGroupFound++ if $group->isMasterEligible;
   }
   &sl_traces::error( "< %s > master eligible nodes group(s) ".
                      "is (are) defined for the cluster < %s >",
		      "- sl_cluster::_checkGroupLst -".__LINE__,
		      ( $masterGroupFound, $self->{'Name'} ) )
      unless ( $masterGroupFound == 1 );
}

# ------------------------------------------------------------------------------
#
sub configure {
   my $self = shift;
   my ( $swlVersion, $swlId ) = @_;
   &sl_traces::info( 2, "Configuring cluster < $self->{'Name'} >" );
   my $clusterName = $self->getName;
   my $swlDir = "$ENV{'SMCT_TMP_DIR'}/${swlId}";
   my $clusterDir = "${swlDir}/${clusterName}";
   my @groupLst = $self->getGroupLst;
   # Configure the cluster nodes group
   foreach $group ( @groupLst ) {
      $group->configure( $clusterDir );
   }
}

# ------------------------------------------------------------------------------
#
sub deploy {
   my $self = shift;
   my ( $swlVersion, $swlId ) = @_;
   &sl_traces::info( 2, "%s cluster < %s >",
                     ( "Configuring", $self->{'Name'} ) );
   my $clusterName = $self->getName;
   my $domain = $self->getDomain;
   my $domainId = $domain->getId;
   my @shelfLst = $self->getShelfLst;
   my $swlDir = "$ENV{'SMCT_TMP_DIR'}/${swlId}";
   my $swlRepDir = "$ENV{'SMCT_SWLREP_DIR'}/${swlId}/${clusterName}";
   my $clusterDir = "${swlDir}/${clusterName}";
   &sl_traces::trace( 2, "- sl_cluster::deploy - Configuring cluster ".
                      "< $self->{'Name'} >" );
   # Import cluster registry
   &sl_registry::import(
      $clusterName, "${swlRepDir}/${sl_registry::REGISTRY_NAME}" );
   # Check CMM configuration
   my @nodeLst = $self->getNodeLst;
   &sl_cmm::checkConfig( $domain, @nodeLst );
   # Generate cluster wide configuration
   $domain->deploy( $ENV{'SMCT_TMP_DIR'} );
   foreach $shelf ( @shelfLst ) {
      $shelf->deploy( $ENV{'SMCT_TMP_DIR'} );
   }   
   my @groupLst = $self->getGroupLst;
   # Configure the cluster group of nodes
   foreach $group ( @groupLst ) {
      $group->deploy( 
         $clusterDir, $domain, $swlVersion, 
	 $swlRepDir, $clusterName, @shelfLst );
   }
   # Finalize group-wide configuration (cluster_nodes_table + hosts files)
   sl_node->deploy( $clusterDir, $swlVersion, $clusterName, @groupLst );
}

# ------------------------------------------------------------------------------
#
sub install {
   my $self = shift;
   my ( $installDir, $swlVersion ) = @_;
   &sl_traces::info( 2, "%s cluster < %s >",
                     ( "Installing", $self->{'Name'} ) );
   my @groupLst = $self->getGroupLst;
   my @shelfLst = $self->getShelfLst;
   my $clusterName = $self->getName;
   my $clusterDir = "${installDir}/${clusterName}";
   $self->_checkGroupLst;
   foreach $group ( @groupLst ) {
      $group->install( $clusterDir, $swlVersion, $clusterName, @shelfLst );
   }
   # Store registry
   &sl_registry::export( 
      $clusterName, $clusterDir, ${sl_registry::REGISTRY_NAME} );
}

# ------------------------------------------------------------------------------
#
sub update {
   my $self = shift;
   my ( $cluster, $operation ) = @_;
   &sl_traces::trace( 2, "- sl_cluster::update - Updating cluster ".
                       "< $self->{'Name'} >" );
   my @groupLst = &sl_utils::sortListByName( $cluster->getGroupLst );
   my @refGroupLst = &sl_utils::sortListByName( $self->getGroupLst );
   if ( $operation eq $main::CONFIG ) {
      # Update group ConfigLst attribute
      for ( my $i =0; $i < scalar @groupLst; $i++ ) {
         my @configLst = $groupLst[$i]->getConfigLst;
	 $refGroupLst[$i]->setConfigLst( "reset", @configLst )
	    if ( defined $configLst[0] );
      }
   }
   elsif ( $operation eq $main::DEPLOY ) {
      # Update group ServiceLst attribute
      for ( my $i =0; $i < scalar @groupLst; $i++ ) {
         my @serviceLst = $groupLst[$i]->getServiceLst;
	 $refGroupLst[$i]->setServiceLst( @serviceLst )
	    if ( defined $serviceLst[0] );
      }
   }
   else {
      &sl_traces::error( "Wrong update operation < %s >",
                         "- sl_cluster::update -".__LINE__,
			 ( $operation )  );
   }
   $self->setGroupLst( @refGroupLst );
}

# ------------------------------------------------------------------------------
#
sub iD {
   my $self = shift;
   $id = $self->{'Name'}.$ID_SEP.$self->{'Id'};
   &sl_traces::trace( 3, "- sl_cluster::iD - Cluster iD is < ${id} >" );
   return $id;
}

# ------------------------------------------------------------------------------

return 1;
